/* flac_tag.c - 2003/12/27 */
/*
 *  EasyTAG - Tag editor for MP3 and Ogg Vorbis files
 *  Copyright (C) 2001-2003  Jerome Couderc <j.couderc@ifrance.com>
 *  Copyright (C) 2003       Pavel Minayev <thalion@front.ru>
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

#include <config.h>

#ifdef ENABLE_FLAC

#include <gtk/gtk.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <errno.h>
#include <FLAC/metadata.h>
#include <unistd.h>

#include "easytag.h"
#include "flac_tag.h"
#include "vcedit.h"
#include "et_core.h"
#include "id3_tag.h"
#include "misc.h"
#include "setting.h"
#include "charset.h"
#include "i18n.h"


/***************
 * Declaration *
 ***************/

#define MULTIFIELD_SEPARATOR " - "

/* FLAC uses Ogg Vorbis comments
 * Ogg Vorbis fields names :
 *  - TITLE        : Track name
 *  - VERSION      : The version field may be used to differentiate multiple version of the same track title in a single collection. (e.g. remix info)
 *  - ALBUM        : The collection name to which this track belongs
 *  - TRACKNUMBER  : The track number of this piece if part of a specific larger collection or album
 *  - ARTIST       : Track performer
 *  - ORGANIZATION : Name of the organization producing the track (i.e. the 'record label')
 *  - DESCRIPTION  : A short text description of the contents
 *  - GENRE        : A short text indication of music genre
 *  - DATE         : Date the track was recorded
 *  - LOCATION     : Location where track was recorded
 *  - COPYRIGHT    : Copyright information
 *  - ISRC         : ISRC number for the track; see the ISRC intro page for more information on ISRC numbers.
 *
 * Field names should not be 'internationalized'; this is a concession to simplicity
 * not an attempt to exclude the majority of the world that doesn't speak English.
 * Field *contents*, however, are represented in UTF-8 to allow easy representation
 * of any language.
 */



/**************
 * Prototypes *
 **************/
gboolean Flac_Tag_Write_File (FILE *file_in, gchar *filename_in, vcedit_state *state);


/*************
 * Functions *
 *************/

/*
 * Read tag data from a FLAC file.
 * Note:
 *  - if field is found but contains no info (strlen(str)==0), we don't read it
 */
gboolean Flac_Tag_Read_File_Tag (gchar *filename, File_Tag *FileTag)
{
    FLAC__Metadata_SimpleIterator               *iter;
    FLAC__StreamMetadata                        *vc_block;
    FLAC__StreamMetadata_VorbisComment          *vc;
    FLAC__StreamMetadata_VorbisComment_Entry    *field;
    gchar                                       *field_value;
    gchar                                       *string = NULL;
    gchar                                       *string1 = NULL;
    gint                                        field_num;
    gint                                        field_len;
    guint                                       i;


    if (!filename || !FileTag)
        return FALSE;

    flac_error_msg = NULL;
    
    iter = FLAC__metadata_simple_iterator_new();
    if ( iter == NULL || !FLAC__metadata_simple_iterator_init(iter, filename, true, false) )
    {
        if ( iter == NULL )
        {
            flac_error_msg = FLAC__Metadata_SimpleIteratorStatusString[FLAC__METADATA_SIMPLE_ITERATOR_STATUS_MEMORY_ALLOCATION_ERROR];
        }else
        {
            flac_error_msg = FLAC__Metadata_SimpleIteratorStatusString[FLAC__metadata_simple_iterator_status(iter)];
            FLAC__metadata_simple_iterator_delete(iter);
        }
        
        g_print(_("ERROR while opening file: '%s' as FLAC (%s).\n\a"),filename,flac_error_msg);
        return FALSE;
    }

    /* libFLAC is able to detect (and skip) ID3v2 tags by itself */
    
    /* Find the VORBIS_COMMENT block */
    while ( FLAC__metadata_simple_iterator_get_block_type(iter) != FLAC__METADATA_TYPE_VORBIS_COMMENT )
    {
        if ( !FLAC__metadata_simple_iterator_next(iter) )
        {
            /* End of metadata: comment block not found, nothing to read */
            FLAC__metadata_simple_iterator_delete(iter);
            return TRUE;
        }
    }

    /* Get comments from block */
    vc_block = FLAC__metadata_simple_iterator_get_block(iter);
    vc = &vc_block->data.vorbis_comment;

    /*********
     * Title *
     *********/
    field_num = 0;
    while ( (field_num = FLAC__metadata_object_vorbiscomment_find_entry_from(vc_block,field_num,"title")) >= 0 )
    {
        /* Extract field value */
        field = &vc->comments[field_num++];
        field_value = memchr(field->entry, '=', field->length);
      
        if ( field_value && strlen(field_value)>0 )
        {
            field_value++;
            field_len = field->length - (field_value - (gchar*) field->entry);
            field_value = g_strndup(field_value, field_len);
            string = convert_from_utf8(field_value);
            //Strip_String(string);
            if (FileTag->title==NULL)
                FileTag->title = g_strdup(string);
            else
                FileTag->title = g_strconcat(FileTag->title,MULTIFIELD_SEPARATOR,string,NULL);
            g_free(string);
            g_free(field_value);
        }                
    }

    /**********
     * Artist *
     **********/
    field_num = 0;
    while ( (field_num = FLAC__metadata_object_vorbiscomment_find_entry_from(vc_block,field_num,"artist")) >= 0 )
    {
        /* Extract field value */
        field = &vc->comments[field_num++];
        field_value = memchr(field->entry, '=', field->length);
      
        if ( field_value && strlen(field_value)>0 )
        {
            field_value++;
            field_len = field->length - (field_value - (gchar*) field->entry);
            field_value = g_strndup(field_value, field_len);
            string = convert_from_utf8(field_value);
            //Strip_String(string);
            if (FileTag->artist==NULL)
                FileTag->artist = g_strdup(string);
            else
                FileTag->artist = g_strconcat(FileTag->artist,MULTIFIELD_SEPARATOR,string,NULL);
            g_free(string);
            g_free(field_value);
        }                
    }

    /*********
     * Album *
     *********/
    field_num = 0;
    while ( (field_num = FLAC__metadata_object_vorbiscomment_find_entry_from(vc_block,field_num,"album")) >= 0 )
    {
        /* Extract field value */
        field = &vc->comments[field_num++];
        field_value = memchr(field->entry, '=', field->length);
      
        if ( field_value && strlen(field_value)>0 )
        {
            field_value++;
            field_len = field->length - (field_value - (gchar*) field->entry);
            field_value = g_strndup(field_value, field_len);
            string = convert_from_utf8(field_value);
            //Strip_String(string);
            if (FileTag->album==NULL)
                FileTag->album = g_strdup(string);
            else
                FileTag->album = g_strconcat(FileTag->album,MULTIFIELD_SEPARATOR,string,NULL);
            g_free(string);
            g_free(field_value);
        }                
    }

    /********
     * Year *
     ********/
    if ( (field_num = FLAC__metadata_object_vorbiscomment_find_entry_from(vc_block,0,"date")) >= 0 )
    {
        /* Extract field value */
        field = &vc->comments[field_num];
        field_value = memchr(field->entry, '=', field->length);
        
        if ( field_value && strlen(field_value)>0 )
        {
            field_value++;
            field_len = field->length - (field_value - (gchar*) field->entry);
            field_value = g_strndup(field_value, field_len);
            string = convert_from_utf8(field_value);
            //Strip_String(string);
            FileTag->year = g_strdup(string);
            g_free(string);
            g_free(field_value);
        }            
    }

    /*************************
     * Track and Total Track *
     *************************/
    if ( (field_num = FLAC__metadata_object_vorbiscomment_find_entry_from(vc_block,0,"tracknumber")) >= 0 )
    {
        /* Extract field value */
        field = &vc->comments[field_num];
        field_value = memchr(field->entry, '=', field->length);
        
        if ( field_value && strlen(field_value)>0 )
        {
            field_value++;
            field_len = field->length - (field_value - (gchar*) field->entry);
            field_value = g_strndup(field_value, field_len);
            string = convert_from_utf8(field_value);
            //Strip_String(string);
            string1 = strchr(string,'/');
            if (NUMBER_TRACK_FORMATED)
            {
                if (string1)
                {
                    FileTag->track_total = g_strdup_printf("%.*d",NUMBER_TRACK_FORMATED_SPIN_BUTTON,atoi(string1+1));
                    *string1 = '\0';
                }
                FileTag->track = g_strdup_printf("%.*d",NUMBER_TRACK_FORMATED_SPIN_BUTTON,atoi(string));
            }else
            {
                if (string1)
                {
                    FileTag->track_total = g_strdup(string1+1);
                    *string1 = '\0';
                }
                FileTag->track = g_strdup(string);
            }
            g_free(string);
            g_free(field_value);
        }            
    }

    /*********
     * Genre *
     *********/
    field_num = 0;
    while ( (field_num = FLAC__metadata_object_vorbiscomment_find_entry_from(vc_block,field_num,"genre")) >= 0 )
    {
        /* Extract field value */
        field = &vc->comments[field_num++];
        field_value = memchr(field->entry, '=', field->length);
        
        if ( field_value && strlen(field_value)>0 )
        {
            field_value++;
            field_len = field->length - (field_value - (gchar*) field->entry);
            field_value = g_strndup(field_value, field_len);
            string = convert_from_utf8(field_value);
            //Strip_String(string);
            if (FileTag->genre==NULL)
                FileTag->genre = g_strdup(string);
            else
                FileTag->genre = g_strconcat(FileTag->genre,MULTIFIELD_SEPARATOR,string,NULL);
            g_free(string);
            g_free(field_value);
        }                
    }

    /***********
     * Comment *
     ***********/
    field_num = 0;
    while ( (field_num = FLAC__metadata_object_vorbiscomment_find_entry_from(vc_block,field_num,"description")) >= 0 )
    {
        /* Extract field value */
        field = &vc->comments[field_num++];
        field_value = memchr(field->entry, '=', field->length);
        
        if ( field_value && strlen(field_value)>0 )
        {
            field_value++;
            field_len = field->length - (field_value - (gchar*) field->entry);
            field_value = g_strndup(field_value, field_len);
            string = convert_from_utf8(field_value);
            //Strip_String(string);
            if (FileTag->comment==NULL)
                FileTag->comment = g_strdup(string);
            else
                FileTag->comment = g_strconcat(FileTag->comment,MULTIFIELD_SEPARATOR,string,NULL);
            g_free(string);
            g_free(field_value);
        }                
    }

    /***************************
     * Save unsupported fields *
     ***************************/
    for (i=0;i<(guint)vc->num_comments;i++)
    {
        field = &vc->comments[i];
        if ( strncasecmp(field->entry,"title=",       MIN(6,  field->length)) != 0
          && strncasecmp(field->entry,"artist=",      MIN(7,  field->length)) != 0
          && strncasecmp(field->entry,"album=",       MIN(6,  field->length)) != 0
          && strncasecmp(field->entry,"date=",        MIN(5,  field->length)) != 0
          && strncasecmp(field->entry,"tracknumber=", MIN(12, field->length)) != 0
          && strncasecmp(field->entry,"genre=",       MIN(6,  field->length)) != 0
          && strncasecmp(field->entry,"description=", MIN(12, field->length)) != 0 )
        {
            //g_print("custom %*s\n", field->length, field->entry);
            FileTag->other = g_list_append(FileTag->other,g_strndup(field->entry, field->length));
        }
    }

    FLAC__metadata_object_delete(vc_block);
    FLAC__metadata_simple_iterator_delete(iter);

    /* If no FLAC vorbis tag found : we try to get the ID3 tag if it exists (will be deleted when writing the tag */
    if ( FileTag->title       == NULL
      && FileTag->artist      == NULL
      && FileTag->album       == NULL
      && FileTag->year        == NULL
      && FileTag->track       == NULL
      && FileTag->track_total == NULL
      && FileTag->genre       == NULL
      && FileTag->comment     == NULL)
    {
        gint rc;
        
        rc = Id3tag_Read_File_Tag(filename,FileTag);
        
        // If an ID3 tag has been found, we mark the file as unsaved to rewritte a flac tag
        if ( FileTag->title       != NULL
          || FileTag->artist      != NULL
          || FileTag->album       != NULL
          || FileTag->year        != NULL
          || FileTag->track       != NULL
          || FileTag->track_total != NULL
          || FileTag->genre       != NULL
          || FileTag->comment     != NULL)
        {
            FileTag->saved = FALSE;
        }
        
        return rc;
    }
    
    return TRUE;
}


gboolean Flac_Tag_Write_File_Tag (ET_File *ETFile)
{
    File_Tag                                    *FileTag;
    gchar                                       *filename_in;
    FLAC__Metadata_SimpleIterator               *iter;
    FLAC__StreamMetadata                        *vc_block;
    FLAC__StreamMetadata_VorbisComment_Entry    field;
    FLAC__bool                                  write_ok;
    gchar                                       *string, *string1;
    GList                                       *list;


    if (!ETFile || !ETFile->FileTag)
        return FALSE;

    FileTag     = (File_Tag *)ETFile->FileTag->data;
    filename_in = ((File_Name *)ETFile->FileNameCur->data)->value;
    flac_error_msg = NULL;

    /* libFLAC is able to detect (and skip) ID3v2 tags by itself */

    iter = FLAC__metadata_simple_iterator_new();
    if ( iter == NULL || !FLAC__metadata_simple_iterator_init(iter,filename_in,false,false) )
    {
        if ( iter == NULL )
        {
            flac_error_msg = FLAC__Metadata_SimpleIteratorStatusString[FLAC__METADATA_SIMPLE_ITERATOR_STATUS_MEMORY_ALLOCATION_ERROR];
        }else
        {
            flac_error_msg = FLAC__Metadata_SimpleIteratorStatusString[FLAC__metadata_simple_iterator_status(iter)];
            FLAC__metadata_simple_iterator_delete(iter);
        }
        
        g_print(_("ERROR while opening file: '%s' as FLAC (%s).\n\a"),filename_in,flac_error_msg);
        return FALSE;
    }
    
    vc_block = FLAC__metadata_object_new(FLAC__METADATA_TYPE_VORBIS_COMMENT);

    /*********
     * Title *
     *********/
    if ( FileTag->title )
    {
        string  = g_strconcat("title=",FileTag->title,NULL);
        string1 = convert_to_utf8(string);
        field.entry = string1;
        field.length = strlen(string1);
        FLAC__metadata_object_vorbiscomment_insert_comment(vc_block,vc_block->data.vorbis_comment.num_comments,field,true);
        g_free(string);
        g_free(string1);
    }

    /**********
     * Artist *
     **********/
    if ( FileTag->artist )
    {
        string  = g_strconcat("artist=",FileTag->artist,NULL);
        string1 = convert_to_utf8(string);
        field.entry = string1;
        field.length = strlen(string1);
        FLAC__metadata_object_vorbiscomment_insert_comment(vc_block,vc_block->data.vorbis_comment.num_comments,field,true);
        g_free(string);
        g_free(string1);
    }

    /*********
     * Album *
     *********/
    if ( FileTag->album )
    {
        string  = g_strconcat("album=",FileTag->album,NULL);
        string1 = convert_to_utf8(string);
        field.entry = string1;
        field.length = strlen(string1);
        FLAC__metadata_object_vorbiscomment_insert_comment(vc_block,vc_block->data.vorbis_comment.num_comments,field,true);
        g_free(string);
        g_free(string1);
    }

    /********
     * Year *
     ********/
    if ( FileTag->year )
    {
        string  = g_strconcat("date=",FileTag->year,NULL);
        string1 = convert_to_utf8(string);
        field.entry = string1;
        field.length = strlen(string1);
        FLAC__metadata_object_vorbiscomment_insert_comment(vc_block,vc_block->data.vorbis_comment.num_comments,field,true);
        g_free(string);
        g_free(string1);
    }

    /*************************
     * Track and Total Track *
     *************************/
    if ( FileTag->track )
    {
        if ( FileTag->track_total && strlen(FileTag->track_total)>0 )
            string = g_strconcat("tracknumber=",FileTag->track,"/",FileTag->track_total,NULL);
        else
            string = g_strconcat("tracknumber=",FileTag->track,NULL);
        string1 = convert_to_utf8(string);
        field.entry = string1;
        field.length = strlen(string1);
        FLAC__metadata_object_vorbiscomment_insert_comment(vc_block,vc_block->data.vorbis_comment.num_comments,field,true);
        g_free(string);
        g_free(string1);
    }

    /*********
     * Genre *
     *********/
    if ( FileTag->genre )
    {
        string  = g_strconcat("genre=",FileTag->genre,NULL);
        string1 = convert_to_utf8(string);
        field.entry = string1;
        field.length = strlen(string1);
        FLAC__metadata_object_vorbiscomment_insert_comment(vc_block,vc_block->data.vorbis_comment.num_comments,field,true);
        g_free(string);
        g_free(string1);
    }

    /***********
     * Comment *
     ***********/
    // We write the comment using the "both" format
    if ( FileTag->comment )
    {
        string = g_strconcat("description=",FileTag->comment,NULL);
        string1 = convert_to_utf8(string);
        field.entry = string1;
        field.length = strlen(string1);
        FLAC__metadata_object_vorbiscomment_insert_comment(vc_block,vc_block->data.vorbis_comment.num_comments,field,true);
        g_free(string);
        g_free(string1);
    }

    /**************************
     * Set unsupported fields *
     **************************/
    list = FileTag->other;
    while (list)
    {
        if (list->data)
        {
            string1 = (gchar*)list->data;
            field.entry = string1;
            field.length = strlen(string1);
            FLAC__metadata_object_vorbiscomment_insert_comment(vc_block,vc_block->data.vorbis_comment.num_comments,field,true);
        }            
        list = list->next;
    }
    
    /* Find the VORBIS_COMMENT block */
    while ( FLAC__metadata_simple_iterator_get_block_type(iter) != FLAC__METADATA_TYPE_VORBIS_COMMENT )
    {
        if ( !FLAC__metadata_simple_iterator_next(iter) )
            break;
    }

    /* Write tag */
    if ( FLAC__metadata_simple_iterator_get_block_type(iter) != FLAC__METADATA_TYPE_VORBIS_COMMENT )
    {
        /* End of metadata: no comment block, so insert one */
        write_ok = FLAC__metadata_simple_iterator_insert_block_after(iter,vc_block,true);
    }
    else
    {
        write_ok = FLAC__metadata_simple_iterator_set_block(iter,vc_block,true);
    }        

    if ( !write_ok )
    {
        flac_error_msg = FLAC__Metadata_SimpleIteratorStatusString[FLAC__metadata_simple_iterator_status(iter)];
        g_print(_("ERROR: Failed to write comments to file '%s' (%s).\n"),filename_in,flac_error_msg);
        FLAC__metadata_simple_iterator_delete(iter);
        FLAC__metadata_object_delete(vc_block);
        return FALSE;
    }else
    {
        g_print(_("Written tag of '%s'\n"),g_basename(filename_in));
    }

    FLAC__metadata_simple_iterator_delete(iter);
    FLAC__metadata_object_delete(vc_block);
    
    // FIX ME! : Remove the ID3 tag if found....

    return TRUE;
}


#endif /* ENABLE_FLAC */
