/* prefs.c - 2000/05/06 */
/*
 *  EasyTAG - Tag editor for MP3 and Ogg Vorbis files
 *  Copyright (C) 2000-2003  Jerome Couderc <easytag@gmail.com>
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

#include <config.h>

#include <gtk/gtk.h>
#include <errno.h>
#include <stdlib.h>
#include <gdk/gdkkeysyms.h>
#include <gdk/gdk.h>
#include <glib/gi18n-lib.h>
#include <stdio.h>
#include <string.h>
#include <sys/stat.h>
#include <unistd.h>

#include "prefs.h"
#include "setting.h"
#include "msgbox.h"
#include "bar.h"
#include "misc.h"
#include "scan.h"
#include "easytag.h"
#include "browser.h"
#include "cddb.h"
#include "charset.h"


/**************
 * Prototypes *
 **************/
/* Options window */
void OptionsWindow_Quit          (void);
void OptionsWindow_Ok_Button     (void);
void OptionsWindow_Save_Button   (void);
void OptionsWindow_Cancel_Button (void);
gboolean OptionsWindow_Key_Press (GtkWidget *window, GdkEvent *event);
gint Check_Config                (void);

void Set_Main_Window_Size_Toggled      (void);
void Get_Main_Window_Size              (void);
void Set_Pane_Handle_Position1_Toggled (void);
void Get_Pane_Handle_Position1         (void);
void Set_Pane_Handle_Position2_Toggled (void);
void Get_Pane_Handle_Position2         (void);
void Set_Pane_Handle_Position3_Toggled (void);
void Get_Pane_Handle_Position3         (void);

void Scanner_Window_Position_Check_Button_Toggled (void);
void Get_Scanner_Window_Position                  (void);

void Get_Cddb_Window_Size                  (void);
void Get_Cddb_Pane_Handle_Position         (void);
void Set_Cddb_Window_Size_Toggled          (void);
void Set_Cddb_Pane_Handle_Position_Toggled (void);

void Set_Default_Comment_Check_Button_Toggled  (void);
void Number_Track_Formated_Toggled             (void);
void Number_Track_Formated_Spin_Button_Changed (GtkObject *Label, GtkObject *SpinButton);
void Use_ISO88591_Character_Set_Translation_Toggled (void);
void Use_Character_Set_Translation_Toggled     (void);
void Scanner_Convert_Check_Button_Toggled_1    (GtkObject *object_rec, GtkObject *object_emi);
void Cddb_Use_Proxy_Toggled                    (void);

void DefaultPathToMp3_Combo_Add_String (void);

/* Browser */
static void Open_File_Selection_Window (GtkWidget *entry, gchar *title, GtkFileChooserAction action);
void        File_Selection_Window_For_File      (GtkWidget *entry);
void        File_Selection_Window_For_Directory (GtkWidget *entry);


/*************
 * Functions *
 *************/
void Init_OptionsWindow (void)
{
    OptionsWindow = (GtkWidget *)NULL;
}

/*
 * The window for options
 */
void Open_OptionsWindow (void)
{
    GtkWidget *OptionsVBox;
    GtkWidget *ButtonBox;
    GtkWidget *Button;
    GtkWidget *Label;
    GtkWidget *Frame;
    GtkWidget *Table;
    GtkWidget *VBox, *vbox;
    GtkWidget *HBox, *hbox;
    GtkWidget *Separator;
    GtkWidget *EventBox;
    GtkTooltips *Tips;
    gchar temp[MAX_STRING_LEN];
    gchar *msg;
    gchar *path_utf8;

    /* Check if already opened */
    if (OptionsWindow)
    {
        gdk_window_show(OptionsWindow->window);
        return;
    }

    /* The window */
    OptionsWindow = gtk_window_new(GTK_WINDOW_TOPLEVEL);
    /* Config */
    gtk_window_set_position(GTK_WINDOW(OptionsWindow),GTK_WIN_POS_CENTER);
    gtk_window_set_transient_for(GTK_WINDOW(OptionsWindow),GTK_WINDOW(MainWindow));
    gtk_container_set_border_width(GTK_CONTAINER(OptionsWindow), 5);
    /* Title */
    gtk_window_set_title(GTK_WINDOW(OptionsWindow),_("Preferences..."));

    /* Signals connection */
    g_signal_connect(G_OBJECT(OptionsWindow),"destroy", G_CALLBACK(OptionsWindow_Quit),NULL);
    g_signal_connect(G_OBJECT(OptionsWindow),"delete_event",G_CALLBACK(OptionsWindow_Quit),NULL);
    g_signal_connect(G_OBJECT(OptionsWindow),"key_press_event",
                     G_CALLBACK(OptionsWindow_Key_Press),NULL);

    Tips = gtk_tooltips_new_1();

     /* Options */
     /* The vbox */
    OptionsVBox = gtk_vbox_new(FALSE,0);
    gtk_box_set_spacing (GTK_BOX(OptionsVBox),5);
    gtk_container_add(GTK_CONTAINER(OptionsWindow),OptionsVBox);

     /* Options NoteBook */
    OptionsNoteBook = gtk_notebook_new();
    gtk_notebook_popup_enable(GTK_NOTEBOOK(OptionsNoteBook));
    gtk_box_pack_start(GTK_BOX(OptionsVBox),OptionsNoteBook,TRUE,TRUE,0);



    /*
     * Browser
     */
    Label = gtk_label_new(_("Browser"));
    Frame = gtk_frame_new(_("Browser"));
    gtk_notebook_append_page(GTK_NOTEBOOK(OptionsNoteBook),Frame,Label);
    gtk_container_set_border_width(GTK_CONTAINER(Frame), 5);

    VBox = gtk_vbox_new(FALSE,4);
    gtk_container_add(GTK_CONTAINER(Frame),VBox);
    gtk_container_set_border_width(GTK_CONTAINER(VBox), 4);


    /* File Browser frame */
    Frame = gtk_frame_new(_("File Browser"));
    gtk_box_pack_start(GTK_BOX(VBox),Frame,FALSE,FALSE,0);
    vbox = gtk_vbox_new(FALSE,0);
    gtk_container_add(GTK_CONTAINER(Frame),vbox);
    gtk_container_set_border_width(GTK_CONTAINER(vbox), 2);

    /* Browse subdirectories */
    BrowseSubdir = gtk_check_button_new_with_label(_("Search subdirectories"));
    gtk_box_pack_start(GTK_BOX(vbox),BrowseSubdir,FALSE,FALSE,0);
    gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(BrowseSubdir),BROWSE_SUBDIR);
    gtk_tooltips_set_tip(Tips,BrowseSubdir,_("Search subdirectories for files when reading "
        "a directory into the tree."),NULL);

    /* Browse hidden subdirectories */
    BrowseHiddendir = gtk_check_button_new_with_label(_("Search hidden directories"));
    gtk_box_pack_start(GTK_BOX(vbox),BrowseHiddendir,FALSE,FALSE,0);
    gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(BrowseHiddendir),BROWSE_HIDDEN_DIR);
    gtk_tooltips_set_tip(Tips,BrowseHiddendir,_("Search hidden directories for files "
        "(directories starting by a '.')."),NULL);

    /* Load directory on startup */
    // Check box
    LoadOnStartup = gtk_check_button_new_with_label(_("Load on startup the directory"));
    gtk_box_pack_start(GTK_BOX(vbox),LoadOnStartup,FALSE,FALSE,0);
    gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(LoadOnStartup),LOAD_ON_STARTUP);
    gtk_tooltips_set_tip(Tips,LoadOnStartup,_("Automatically search files, when EasyTAG starts, "
        "into the following directory. Note that this path may be overriden by the parameter "
        "passed to easytag (easytag /path_to/mp3_files)."),NULL);

    HBox = gtk_hbox_new(FALSE,4);
    gtk_box_pack_start(GTK_BOX(vbox),HBox,FALSE,FALSE,2);
    gtk_container_set_border_width(GTK_CONTAINER(HBox), 2);

    // Combo
    if (DefaultPathModel != NULL)
        gtk_list_store_clear(DefaultPathModel);
    else
        DefaultPathModel = gtk_list_store_new(MISC_COMBO_COUNT, G_TYPE_STRING);

    DefaultPathToMp3 = gtk_combo_box_entry_new_with_model(GTK_TREE_MODEL(DefaultPathModel), MISC_COMBO_TEXT);
    gtk_box_pack_start(GTK_BOX(HBox),DefaultPathToMp3,TRUE,TRUE,0);
    gtk_widget_set_size_request(DefaultPathToMp3, 400, -1);
    gtk_tooltips_set_tip(Tips,GTK_BIN(DefaultPathToMp3)->child,_("Specify the directory where "
        "your files are located. This path will be loaded when EasyTAG starts without parameter."),NULL);
    g_signal_connect(G_OBJECT(GTK_ENTRY(GTK_BIN(DefaultPathToMp3)->child)),"activate",G_CALLBACK(DefaultPathToMp3_Combo_Add_String),NULL);
    //g_signal_connect(G_OBJECT(GTK_ENTRY(GTK_BIN(DefaultPathToMp3)->child)),"focus_out_event",G_CALLBACK(DefaultPathToMp3_Combo_Add_String),NULL);

    // History list
    Load_Default_Path_To_MP3_List(DefaultPathModel, MISC_COMBO_TEXT);

    // If default path hasn't been added already, add it now..
    path_utf8 = filename_to_display(DEFAULT_PATH_TO_MP3);
    Add_String_To_Combo_List(DefaultPathModel, path_utf8);
    gtk_entry_set_text(GTK_ENTRY(GTK_BIN(DefaultPathToMp3)->child), path_utf8);
    g_free(path_utf8);

    // Button browse
    Button = Create_Button_With_Pixmap(BUTTON_BROWSE);
    gtk_box_pack_start(GTK_BOX(HBox),Button,FALSE,FALSE,0);
    g_signal_connect_swapped(G_OBJECT(Button),"clicked",
                             G_CALLBACK(File_Selection_Window_For_Directory),G_OBJECT(GTK_BIN(DefaultPathToMp3)->child));

    /* Open the node to show subdirectories */
    OpenSelectedBrowserNode = gtk_check_button_new_with_label(_("Show sub-directories when selecting "
        "a directory"));
    gtk_box_pack_start(GTK_BOX(vbox),OpenSelectedBrowserNode,FALSE,FALSE,0);
    gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(OpenSelectedBrowserNode),OPEN_SELECTED_BROWSER_NODE);
    gtk_tooltips_set_tip(Tips,OpenSelectedBrowserNode,_("This expands the selected node into the file "
        "browser to display the sub-directories."),NULL);

    /*
     * User Interface
     */
    Label = gtk_label_new (_("User Interface"));
    Frame = gtk_frame_new (_("User Interface"));
    gtk_notebook_append_page (GTK_NOTEBOOK(OptionsNoteBook),Frame,Label);
    gtk_container_set_border_width(GTK_CONTAINER(Frame), 5);

    VBox = gtk_vbox_new(FALSE,2);
    gtk_container_add(GTK_CONTAINER(Frame),VBox);
    gtk_container_set_border_width(GTK_CONTAINER(VBox), 2);

    Table = gtk_table_new(6,3,FALSE);
    gtk_box_pack_start(GTK_BOX(VBox),Table,FALSE,FALSE,0);
    gtk_table_set_row_spacings(GTK_TABLE(Table),2);
    gtk_table_set_col_spacings(GTK_TABLE(Table),4);

    // Main window size
    SetMainWindowSize = gtk_check_button_new_with_label(_("Set main window size"));
    gtk_table_attach(GTK_TABLE(Table),SetMainWindowSize,0,1,0,1,GTK_FILL,GTK_FILL,0,0);
    Label = gtk_label_new (_("Width :"));
    gtk_table_attach(GTK_TABLE(Table),Label,1,2,0,1,GTK_FILL,GTK_FILL,0,0);
    MainWindowWidth = gtk_entry_new();
    gtk_entry_set_max_length(GTK_ENTRY(MainWindowWidth), 4);
    gtk_table_attach(GTK_TABLE(Table),MainWindowWidth,2,3,0,1,GTK_FILL,GTK_FILL,0,0);
    Label = gtk_label_new (_("Height :"));
    gtk_table_attach(GTK_TABLE(Table),Label,3,4,0,1,GTK_FILL,GTK_FILL,0,0);
    MainWindowHeight = gtk_entry_new();
    gtk_entry_set_max_length(GTK_ENTRY(MainWindowHeight), 4);
    gtk_table_attach(GTK_TABLE(Table),MainWindowHeight,4,5,0,1,GTK_FILL,GTK_FILL,0,0);
    MainWindowButton = Create_Button_With_Icon_And_Label("easytag-grab",_(" Get current size "));
    gtk_table_attach(GTK_TABLE(Table),MainWindowButton,5,6,0,1,GTK_FILL,GTK_FILL,0,0);
    g_signal_connect(G_OBJECT(MainWindowButton),"clicked", G_CALLBACK(Get_Main_Window_Size),NULL);
    gtk_widget_set_size_request(MainWindowHeight, 40, -1);
    gtk_widget_set_size_request(MainWindowWidth, 40, -1);
    gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(SetMainWindowSize),SET_MAIN_WINDOW_SIZE);
    sprintf(temp,"%i",MAIN_WINDOW_HEIGHT);
    gtk_entry_set_text(GTK_ENTRY(MainWindowHeight),temp);
    sprintf(temp,"%i",MAIN_WINDOW_WIDTH);
    gtk_entry_set_text(GTK_ENTRY(MainWindowWidth),temp);
    g_signal_connect(G_OBJECT(MainWindowHeight),"insert_text",G_CALLBACK(Insert_Only_Digit),NULL);
    g_signal_connect(G_OBJECT(MainWindowWidth),"insert_text",G_CALLBACK(Insert_Only_Digit),NULL);
    /* Placed at the end of the block cause we must have all widgets built */
    g_signal_connect(G_OBJECT(SetMainWindowSize),"toggled",G_CALLBACK(Set_Main_Window_Size_Toggled),NULL);
    Set_Main_Window_Size_Toggled();
    msg = g_strdup_printf(_("Specify the default size for the main window. You must restart the "
        "program to apply the new size. For your display, the max values are: width=%d and height=%d."),
        gdk_screen_width(),gdk_screen_height());
    gtk_tooltips_set_tip(Tips,SetMainWindowSize,msg,NULL);
    gtk_tooltips_set_tip(Tips,MainWindowWidth,_("Width: set -1 for automatic size"),NULL);
    gtk_tooltips_set_tip(Tips,MainWindowHeight,_("Heigth: set -1 for automatic size"),NULL);
    g_free(msg);

    /* Pane handle position 1 */
    SetPaneHandlePosition1 = gtk_check_button_new_with_label(_("Set vertical right pane handle position"));
    gtk_table_attach(GTK_TABLE(Table),SetPaneHandlePosition1,0,2,1,2,GTK_FILL,GTK_FILL,0,0);
    Label = gtk_label_new (_("Position :"));
    gtk_table_attach(GTK_TABLE(Table),Label,2,4,1,2,GTK_FILL,GTK_FILL,0,0);
    gtk_misc_set_alignment(GTK_MISC(Label),1,0.5);
    PaneHandlePosition1 = gtk_entry_new();
    gtk_entry_set_max_length(GTK_ENTRY(PaneHandlePosition1), 4);
    gtk_table_attach(GTK_TABLE(Table),PaneHandlePosition1,4,5,1,2,GTK_FILL,GTK_FILL,0,0);
    sprintf(temp,"%i",PANE_HANDLE_POSITION1);
    gtk_entry_set_text(GTK_ENTRY(PaneHandlePosition1),temp);
    g_signal_connect(G_OBJECT(PaneHandlePosition1),"insert_text",G_CALLBACK(Insert_Only_Digit),NULL);
    PaneHandleButton1 = Create_Button_With_Icon_And_Label("easytag-grab",_(" Get current position "));
    gtk_table_attach(GTK_TABLE(Table),PaneHandleButton1,5,6,1,2,GTK_FILL,GTK_FILL,0,0);
    g_signal_connect(G_OBJECT(PaneHandleButton1),"clicked", G_CALLBACK(Get_Pane_Handle_Position1),NULL);
    gtk_widget_set_size_request(PaneHandlePosition1, 40, -1);
    gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(SetPaneHandlePosition1),SET_PANE_HANDLE_POSITION1);
    gtk_tooltips_set_tip(Tips,SetPaneHandlePosition1,_("Specify the default position"
        " for the pane handle located between the browser frame and the tag frame. "
        "Changes will be applied the next start."),NULL);
    gtk_tooltips_set_tip(Tips,PaneHandlePosition1,_("Position: set -1 for automatic positioning"),NULL);
    g_signal_connect(G_OBJECT(SetPaneHandlePosition1),"toggled",
        G_CALLBACK(Set_Pane_Handle_Position1_Toggled),NULL);
    Set_Pane_Handle_Position1_Toggled();

    /* Pane handle position 2 */
    SetPaneHandlePosition2 = gtk_check_button_new_with_label(_("Set vertical left pane handle position"));
    gtk_table_attach(GTK_TABLE(Table),SetPaneHandlePosition2,0,2,2,3,GTK_FILL,GTK_FILL,0,0);
    Label = gtk_label_new (_("Position :"));
    gtk_misc_set_alignment(GTK_MISC(Label),1,0.5);
    gtk_table_attach(GTK_TABLE(Table),Label,2,4,2,3,GTK_FILL,GTK_FILL,0,0);
    PaneHandlePosition2 = gtk_entry_new();
    gtk_entry_set_max_length(GTK_ENTRY(PaneHandlePosition2), 4);
    gtk_table_attach(GTK_TABLE(Table),PaneHandlePosition2,4,5,2,3,GTK_FILL,GTK_FILL,0,0);
    sprintf(temp,"%i",PANE_HANDLE_POSITION2);
    gtk_entry_set_text(GTK_ENTRY(PaneHandlePosition2),temp);
    g_signal_connect(G_OBJECT(PaneHandlePosition2),"insert_text",G_CALLBACK(Insert_Only_Digit),NULL);
    PaneHandleButton2 = Create_Button_With_Icon_And_Label("easytag-grab",_(" Get current position "));
    gtk_table_attach(GTK_TABLE(Table),PaneHandleButton2,5,6,2,3,GTK_FILL,GTK_FILL,0,0);
    g_signal_connect(G_OBJECT(PaneHandleButton2),"clicked",G_CALLBACK(Get_Pane_Handle_Position2),NULL);
    gtk_widget_set_size_request(PaneHandlePosition2, 40, -1);
    gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(SetPaneHandlePosition2),SET_PANE_HANDLE_POSITION2);
    gtk_tooltips_set_tip(Tips,SetPaneHandlePosition2,_("Specify the default position for the pane handle "
        "located between the browser tree and the list of files. Changes will be applied the next start."),NULL);
    gtk_tooltips_set_tip(Tips,PaneHandlePosition2,_("Position: set -1 for automatic positioning"),NULL);
    g_signal_connect(G_OBJECT(SetPaneHandlePosition2),"toggled",
        G_CALLBACK(Set_Pane_Handle_Position2_Toggled),NULL);
    Set_Pane_Handle_Position2_Toggled();

    /* Pane handle position 3 */
    SetPaneHandlePosition3 = gtk_check_button_new_with_label(_("Set horizontal pane handle position"));
    gtk_table_attach(GTK_TABLE(Table),SetPaneHandlePosition3,0,2,3,4,GTK_FILL,GTK_FILL,0,0);
    Label = gtk_label_new (_("Position :"));
    gtk_misc_set_alignment(GTK_MISC(Label),1,0.5);
    gtk_table_attach(GTK_TABLE(Table),Label,2,4,3,4,GTK_FILL,GTK_FILL,0,0);
    PaneHandlePosition3 = gtk_entry_new();
    gtk_entry_set_max_length(GTK_ENTRY(PaneHandlePosition3), 4);
    gtk_table_attach(GTK_TABLE(Table),PaneHandlePosition3,4,5,3,4,GTK_FILL,GTK_FILL,0,0);
    sprintf(temp,"%i",PANE_HANDLE_POSITION3);
    gtk_entry_set_text(GTK_ENTRY(PaneHandlePosition3),temp);
    g_signal_connect(GTK_OBJECT(PaneHandlePosition3),"insert_text",G_CALLBACK(Insert_Only_Digit),NULL);
    PaneHandleButton3 = Create_Button_With_Icon_And_Label("easytag-grab",_(" Get current position "));
    gtk_table_attach(GTK_TABLE(Table),PaneHandleButton3,5,6,3,4,GTK_FILL,GTK_FILL,0,0);
    g_signal_connect(G_OBJECT(PaneHandleButton3),"clicked",G_CALLBACK(Get_Pane_Handle_Position3),NULL);
    gtk_widget_set_size_request(PaneHandlePosition3,40,-1);
    gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(SetPaneHandlePosition3),SET_PANE_HANDLE_POSITION3);
    gtk_tooltips_set_tip(Tips,SetPaneHandlePosition2,_("Specify the default position for the pane handle "
        "located between the artist list and album list. Changes will be applied the next start."),NULL);
    gtk_tooltips_set_tip(Tips,PaneHandlePosition3,_("Position: set -1 for automatic positioning"),NULL);
    g_signal_connect(G_OBJECT(SetPaneHandlePosition3),"toggled",
        G_CALLBACK(Set_Pane_Handle_Position3_Toggled),NULL);
    Set_Pane_Handle_Position3_Toggled();

    /* Separator line */
    Separator = gtk_hseparator_new();
    gtk_box_pack_start(GTK_BOX(VBox),Separator,FALSE,FALSE,0);

    ShowHeaderInfos = gtk_check_button_new_with_label(_("Show header informations of file"));
    gtk_box_pack_start(GTK_BOX(VBox),ShowHeaderInfos,FALSE,FALSE,0);
    gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(ShowHeaderInfos),SHOW_HEADER_INFO);
    gtk_tooltips_set_tip(Tips,ShowHeaderInfos,_("If activated, informations about the file as "
        "the bitrate, the time, the size, will be displayed under the filename entry."),NULL);

    /* Separator line */
    Separator = gtk_hseparator_new();
    gtk_box_pack_start(GTK_BOX(VBox),Separator,FALSE,FALSE,0);

    /* Set yellow color to tooltips */
    UseColoredTooltips = gtk_check_button_new_with_label(_("Set yellow background for tooltips"));
    gtk_box_pack_start(GTK_BOX(VBox),UseColoredTooltips,FALSE,FALSE,0);
    gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(UseColoredTooltips),USE_COLORED_TOOLTIPS);
    gtk_tooltips_set_tip(Tips,UseColoredTooltips,_("If activated, the background of tooltips will "
        "be yellow colored, overriding your window manager."),NULL);

    /* Set "new" Gtk+-2.0ish black/bold style for changes items */
    UseNewListStyle = gtk_check_button_new_with_label(_("Set new style for displaying list items"));
    gtk_box_pack_start(GTK_BOX(VBox), UseNewListStyle, FALSE, FALSE, 0);
    gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(UseNewListStyle), USE_NEW_LIST_STYLE);
    gtk_tooltips_set_tip(Tips, UseNewListStyle, _("If activated, items in list will be styled "
        "black and bold instead of red and gray (example : for changed and not saved files)."), NULL);



    /*
     * Misc
     */
    Label = gtk_label_new (_("Misc"));
    Frame = gtk_frame_new (_("Misc"));
    gtk_notebook_append_page (GTK_NOTEBOOK(OptionsNoteBook),Frame,Label);
    gtk_container_set_border_width(GTK_CONTAINER(Frame), 5);

    VBox = gtk_vbox_new(FALSE,4);
    gtk_container_add(GTK_CONTAINER(Frame),VBox);
    gtk_container_set_border_width(GTK_CONTAINER(VBox), 4);


    /* File(name) Options */
    Frame = gtk_frame_new (_("File Options"));
    gtk_box_pack_start(GTK_BOX(VBox),Frame,FALSE,FALSE,0);
    vbox = gtk_vbox_new(FALSE,2);
    gtk_container_add(GTK_CONTAINER(Frame),vbox);
    gtk_container_set_border_width(GTK_CONTAINER(vbox), 2);

    ReplaceIllegalCharactersInFilename = gtk_check_button_new_with_label(_("Replace illegal characters in filename (for Windows and CD-Rom)"));
    gtk_box_pack_start(GTK_BOX(vbox),ReplaceIllegalCharactersInFilename,FALSE,FALSE,0);
    gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(ReplaceIllegalCharactersInFilename),REPLACE_ILLEGAL_CHARACTERS_IN_FILENAME);
    gtk_tooltips_set_tip(Tips,ReplaceIllegalCharactersInFilename,_("Convert illegal characters for "
        "FAT32/16 and ISO9660 + Joliet filesystems ('\\', ':', ';', '*', '?', '\"', '<', '>', '|') "
        "of the filename to avoid problem when renaming the file. This is usefull when renaming the "
        "file from the tag with the scanner."),NULL);

    hbox = gtk_hbox_new(FALSE,2);
    gtk_box_pack_start(GTK_BOX(vbox),hbox,FALSE,FALSE,0);
    gtk_container_set_border_width(GTK_CONTAINER(hbox), 2);
    /* Extension case */
    Label = gtk_label_new(_("Convert filename extension to :"));
    gtk_box_pack_start(GTK_BOX(hbox),Label,FALSE,FALSE,0);
    
    FilenameExtensionLowerCase = gtk_radio_button_new_with_label(NULL,_("Lower Case"));
    gtk_box_pack_start(GTK_BOX(hbox),FilenameExtensionLowerCase,FALSE,FALSE,2);
    gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(FilenameExtensionLowerCase),FILENAME_EXTENSION_LOWER_CASE);
    gtk_tooltips_set_tip(Tips,FilenameExtensionLowerCase,_("For example, the extension will be converted to '.mp3'"),NULL);

    FilenameExtensionUpperCase = gtk_radio_button_new_with_label(
        gtk_radio_button_get_group(GTK_RADIO_BUTTON(FilenameExtensionLowerCase)),_("Upper Case"));
    gtk_box_pack_start(GTK_BOX(hbox),FilenameExtensionUpperCase,FALSE,FALSE,2);
    gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(FilenameExtensionUpperCase),FILENAME_EXTENSION_UPPER_CASE);
    gtk_tooltips_set_tip(Tips,FilenameExtensionUpperCase,_("For example, the extension will be converted to '.MP3'"),NULL);

    FilenameExtensionNoChange = gtk_radio_button_new_with_label(
        gtk_radio_button_get_group(GTK_RADIO_BUTTON(FilenameExtensionLowerCase)),_("No Change"));
    gtk_box_pack_start(GTK_BOX(hbox),FilenameExtensionNoChange,FALSE,FALSE,2);
    gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(FilenameExtensionNoChange),FILENAME_EXTENSION_NO_CHANGE);
    gtk_tooltips_set_tip(Tips,FilenameExtensionNoChange,_("The extension will not be converted"),NULL);

    /* Preserve modification time */
    PreserveModificationTime = gtk_check_button_new_with_label(_("Preserve modification time of the file"));
    gtk_box_pack_start(GTK_BOX(vbox),PreserveModificationTime,FALSE,FALSE,0);
    gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(PreserveModificationTime),PRESERVE_MODIFICATION_TIME);
    gtk_tooltips_set_tip(Tips,PreserveModificationTime,_("Preserve the modification time "
        "(in file properties) when saving the file."),NULL);


    /* Tag Options */
    Frame = gtk_frame_new (_("Tag Options"));
    gtk_box_pack_start(GTK_BOX(VBox),Frame,FALSE,FALSE,0);
    vbox = gtk_vbox_new(FALSE,2);
    gtk_container_add(GTK_CONTAINER(Frame),vbox);
    gtk_container_set_border_width(GTK_CONTAINER(vbox), 2);

    DateAutoCompletion = gtk_check_button_new_with_label(_("Auto completion of date if not complete"));
    gtk_box_pack_start(GTK_BOX(vbox),DateAutoCompletion,FALSE,FALSE,0);
    gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(DateAutoCompletion),DATE_AUTO_COMPLETION);
    gtk_tooltips_set_tip(Tips,DateAutoCompletion,_("Try to complete the year field if you enter "
        "only the last numerals of the date (for instance, if the current year is 2005: "
        "5 => 2005, 4 => 2004, 6 => 1996, 95 => 1995, ...)."),NULL);

    hbox = gtk_hbox_new(FALSE,0);
    gtk_box_pack_start(GTK_BOX(vbox),hbox,FALSE,FALSE,0);

    NumberTrackFormated = gtk_check_button_new_with_label(_("Write the track field with the following number of digits :"));
    //gtk_box_pack_start(GTK_BOX(vbox),NumberTrackFormated,FALSE,FALSE,0);
    gtk_box_pack_start(GTK_BOX(hbox),NumberTrackFormated,FALSE,FALSE,0);
    gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(NumberTrackFormated),NUMBER_TRACK_FORMATED);
    gtk_tooltips_set_tip(Tips,NumberTrackFormated,_("If activated, the track field is written using "
        "the number '0' as padding to obtain a number with 'n' digits (Ex. with two digits : '05', "
        "'09', '10',...). Else it keeps the 'raw' track value."),NULL);

    NumberTrackFormatedSpinButton = gtk_spin_button_new((GtkAdjustment *)gtk_adjustment_new(2.0,2.0,6.0,1.0,1.0,1.0),1.0,0);
    gtk_box_pack_start(GTK_BOX(hbox),NumberTrackFormatedSpinButton,FALSE,FALSE,0);
    gtk_spin_button_set_value(GTK_SPIN_BUTTON(NumberTrackFormatedSpinButton),(gfloat)NUMBER_TRACK_FORMATED_SPIN_BUTTON);
    g_signal_connect(G_OBJECT(NumberTrackFormated),"toggled",G_CALLBACK(Number_Track_Formated_Toggled),NULL);
    g_signal_emit_by_name(G_OBJECT(NumberTrackFormated),"toggled");

    Label = gtk_label_new("");
    gtk_box_pack_start(GTK_BOX(hbox),Label,FALSE,FALSE,4);
    g_signal_connect_swapped(G_OBJECT(NumberTrackFormatedSpinButton),"changed",G_CALLBACK(Number_Track_Formated_Spin_Button_Changed),G_OBJECT(Label));
    g_signal_emit_by_name(G_OBJECT(NumberTrackFormatedSpinButton),"changed",NULL);

    OggTagWriteXmmsComment = gtk_check_button_new_with_label(_("Ogg Vorbis Files : Write also the comment to the XMMS format"));
    gtk_box_pack_start(GTK_BOX(vbox),OggTagWriteXmmsComment,FALSE,FALSE,0);
    gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(OggTagWriteXmmsComment),OGG_TAG_WRITE_XMMS_COMMENT);
    gtk_tooltips_set_tip(Tips,OggTagWriteXmmsComment,_("XMMS doesn't make use of the right way to "
        "identify a comment in Ogg Vorbis files as other apps do. In fact, this field is usually labeled "
        "with 'comment=', whereas XMMS uses only `='. Please, uncheck this option if you don't want "
        "other apps to complain about an unknown field. Comments won't be shown in XMMS, though."),NULL);


    /* Sorting List Options */
    Frame = gtk_frame_new (_("Sorting List Options"));
    gtk_box_pack_start(GTK_BOX(VBox),Frame,FALSE,FALSE,0);
    vbox = gtk_vbox_new(FALSE,2);
    gtk_container_add(GTK_CONTAINER(Frame),vbox);
    gtk_container_set_border_width(GTK_CONTAINER(vbox), 2);

    hbox = gtk_hbox_new(FALSE,2);
    gtk_box_pack_start(GTK_BOX(vbox),hbox,FALSE,FALSE,0);
    gtk_container_set_border_width(GTK_CONTAINER(hbox), 2);
    /* Sorting method */
    Label = gtk_label_new(_("Sort the file list by :"));
    gtk_box_pack_start(GTK_BOX(hbox),Label,FALSE,FALSE,0);

    EventBox = gtk_event_box_new();
    SortingFileCombo = gtk_combo_box_new_text();
    gtk_container_add(GTK_CONTAINER(EventBox),SortingFileCombo);
    gtk_box_pack_start(GTK_BOX(hbox),EventBox,FALSE,FALSE,2);
    gtk_widget_set_size_request(GTK_WIDGET(SortingFileCombo), 260, -1);
    gtk_combo_box_set_wrap_width(GTK_COMBO_BOX(SortingFileCombo),2); // Two columns

    // Items of option menu
    gtk_combo_box_append_text(GTK_COMBO_BOX(SortingFileCombo), _("Ascending file name"));
    gtk_combo_box_append_text(GTK_COMBO_BOX(SortingFileCombo), _("Descending file name"));
    gtk_combo_box_append_text(GTK_COMBO_BOX(SortingFileCombo), _("Ascending track number"));
    gtk_combo_box_append_text(GTK_COMBO_BOX(SortingFileCombo), _("Descending track number"));
    gtk_combo_box_append_text(GTK_COMBO_BOX(SortingFileCombo), _("Ascending creation date"));
    gtk_combo_box_append_text(GTK_COMBO_BOX(SortingFileCombo), _("Descending creation date"));
    gtk_combo_box_append_text(GTK_COMBO_BOX(SortingFileCombo), _("Ascending title"));
    gtk_combo_box_append_text(GTK_COMBO_BOX(SortingFileCombo), _("Descending title"));
    gtk_combo_box_append_text(GTK_COMBO_BOX(SortingFileCombo), _("Ascending artist"));
    gtk_combo_box_append_text(GTK_COMBO_BOX(SortingFileCombo), _("Descending artist"));
    gtk_combo_box_append_text(GTK_COMBO_BOX(SortingFileCombo), _("Ascending album"));
    gtk_combo_box_append_text(GTK_COMBO_BOX(SortingFileCombo), _("Descending album"));
    gtk_combo_box_append_text(GTK_COMBO_BOX(SortingFileCombo), _("Ascending year"));
    gtk_combo_box_append_text(GTK_COMBO_BOX(SortingFileCombo), _("Descending year"));
    gtk_combo_box_append_text(GTK_COMBO_BOX(SortingFileCombo), _("Ascending genre"));
    gtk_combo_box_append_text(GTK_COMBO_BOX(SortingFileCombo), _("Descending genre"));
    gtk_combo_box_append_text(GTK_COMBO_BOX(SortingFileCombo), _("Ascending comment"));
    gtk_combo_box_append_text(GTK_COMBO_BOX(SortingFileCombo), _("Descending comment"));

    gtk_combo_box_set_active(GTK_COMBO_BOX(SortingFileCombo), SORTING_FILE_MODE);
    gtk_tooltips_set_tip(Tips,EventBox,_("Select the type of file sorting "
        "when loading a directory."),NULL);

    SortingFileCaseSensitive = gtk_check_button_new_with_label(_("Case sensitive"));
    gtk_box_pack_start(GTK_BOX(hbox),SortingFileCaseSensitive,FALSE,FALSE,0);
    gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(SortingFileCaseSensitive),SORTING_FILE_CASE_SENSITIVE);
    gtk_tooltips_set_tip(Tips,SortingFileCaseSensitive,_("If activated, the sorting of "
        "the list will be dependent on the case."),NULL);

    /* Message Dialog Position */
    Frame = gtk_frame_new (_("Message Dialog Position"));
    gtk_box_pack_start(GTK_BOX(VBox),Frame,FALSE,FALSE,0);
    hbox = gtk_hbox_new(TRUE,4);
    gtk_container_add(GTK_CONTAINER(Frame),hbox);
    gtk_container_set_border_width(GTK_CONTAINER(hbox), 4);

    MessageBoxPositionNone = gtk_radio_button_new_with_label(NULL,_("No particular position"));
    gtk_box_pack_start(GTK_BOX(hbox),MessageBoxPositionNone,FALSE,FALSE,0);
    gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(MessageBoxPositionNone),MESSAGE_BOX_POSITION_NONE);
    gtk_tooltips_set_tip(Tips,MessageBoxPositionNone,_("Let the Window Manager to place the windows."),NULL);

    MessageBoxPositionCenter = gtk_radio_button_new_with_label(
        gtk_radio_button_get_group(GTK_RADIO_BUTTON(MessageBoxPositionNone)),_("Center of the screen"));
    gtk_box_pack_start(GTK_BOX(hbox),MessageBoxPositionCenter,FALSE,FALSE,0);
    gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(MessageBoxPositionCenter),MESSAGE_BOX_POSITION_CENTER);
    gtk_tooltips_set_tip(Tips,MessageBoxPositionCenter,_("Windows should be placed in the center "
        "of the screen."),NULL);

    MessageBoxPositionMouse = gtk_radio_button_new_with_label(
        gtk_radio_button_get_group(GTK_RADIO_BUTTON(MessageBoxPositionNone)),_("Mouse position"));
    gtk_box_pack_start(GTK_BOX(hbox),MessageBoxPositionMouse,FALSE,FALSE,0);
    gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(MessageBoxPositionMouse),MESSAGE_BOX_POSITION_MOUSE);
    gtk_tooltips_set_tip(Tips,MessageBoxPositionMouse,_("Windows should be placed at the current "
        "mouse position."),NULL);

    /* File Player */
    Frame = gtk_frame_new (_("File Audio Player"));
    gtk_box_pack_start(GTK_BOX(VBox),Frame,FALSE,FALSE,0);

    // Player name with params
    if (FilePlayerModel == NULL)
        FilePlayerModel = gtk_list_store_new(MISC_COMBO_COUNT, G_TYPE_STRING);
    else
        gtk_list_store_clear(FilePlayerModel);

    hbox = gtk_hbox_new(FALSE,4);
    gtk_container_add(GTK_CONTAINER(Frame),hbox);
    gtk_container_set_border_width(GTK_CONTAINER(hbox), 4);
    Label = gtk_label_new (_("Player to run :"));
    gtk_box_pack_start(GTK_BOX(hbox),Label,FALSE,FALSE,0);
    FilePlayerCombo = gtk_combo_box_entry_new_with_model(GTK_TREE_MODEL(FilePlayerModel), MISC_COMBO_TEXT);
    gtk_widget_set_size_request(GTK_WIDGET(FilePlayerCombo), 300, -1);
    gtk_box_pack_start(GTK_BOX(hbox),FilePlayerCombo,FALSE,FALSE,0);
    gtk_tooltips_set_tip(Tips,GTK_BIN(FilePlayerCombo)->child,_("Enter the program used to "
        "play the files. Some arguments can be passed for the program (as 'xmms -p') before "
        "to receive files as other arguments."),NULL);
    // History List
    Load_Audio_File_Player_List(FilePlayerModel, MISC_COMBO_TEXT);
    Add_String_To_Combo_List(FilePlayerModel, AUDIO_FILE_PLAYER);
    gtk_entry_set_text(GTK_ENTRY(GTK_BIN(FilePlayerCombo)->child), AUDIO_FILE_PLAYER);

    // Button browse
    Button = Create_Button_With_Pixmap(BUTTON_BROWSE);
    gtk_box_pack_start(GTK_BOX(hbox),Button,FALSE,FALSE,0);
    g_signal_connect_swapped(G_OBJECT(Button),"clicked",
                             G_CALLBACK(File_Selection_Window_For_File), G_OBJECT(GTK_BIN(FilePlayerCombo)->child));



    /*
     * Tag Settings
     */
    Label = gtk_label_new (_("Tag Settings"));
    Frame = gtk_frame_new (_("Tag Settings"));
    gtk_notebook_append_page (GTK_NOTEBOOK(OptionsNoteBook),Frame,Label);
    gtk_container_set_border_width(GTK_CONTAINER(Frame),5);

    VBox = gtk_vbox_new(FALSE,2);
    gtk_container_add(GTK_CONTAINER(Frame),VBox);
    gtk_container_set_border_width(GTK_CONTAINER(VBox),4);


    /* Tag Rules frame */
    Frame = gtk_frame_new (_("ID3 Tag Rules"));
    gtk_box_pack_start(GTK_BOX(VBox),Frame,FALSE,FALSE,0);
    vbox = gtk_vbox_new(FALSE,2);
    gtk_container_add(GTK_CONTAINER(Frame),vbox);
    gtk_container_set_border_width(GTK_CONTAINER(vbox),2);
    /* Write ID3v1 tag */
    WriteId3v1Tag = gtk_check_button_new_with_label(_("Write ID3v1.x tag"));
    gtk_box_pack_start(GTK_BOX(vbox),WriteId3v1Tag,FALSE,FALSE,0);
    gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(WriteId3v1Tag),WRITE_ID3V1_TAG);
    gtk_tooltips_set_tip(Tips,WriteId3v1Tag,_("If activated, an ID3v1 tag will be added or "
        "updated at the end of the MP3 files. Else it will be stripped."),NULL);
    /* Write ID3v2 tag */
    WriteId3v2Tag = gtk_check_button_new_with_label(_("Write ID3v2 tag"));
    gtk_box_pack_start(GTK_BOX(vbox),WriteId3v2Tag,FALSE,FALSE,0);
    gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(WriteId3v2Tag),WRITE_ID3V2_TAG);
    gtk_tooltips_set_tip(Tips,WriteId3v2Tag,_("If activated, an ID3v2 tag will be added or "
        "updated at the beginning of the MP3 files. Else it will be stripped."),NULL);
    /* Write ID3 tags in FLAC files */
    WriteId3TagsInFlacFiles = gtk_check_button_new_with_label(_("Write ID3 tags in FLAC files with FLAC tag"));
    gtk_box_pack_start(GTK_BOX(vbox),WriteId3TagsInFlacFiles,FALSE,FALSE,0);
    gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(WriteId3TagsInFlacFiles),WRITE_ID3_TAGS_IN_FLAC_FILE);
    gtk_tooltips_set_tip(Tips,WriteId3TagsInFlacFiles,_("If activated, ID3 tags will be "
        "also added in the FLAC file (according the two rules above, plus the FLAC tag). "
        "Else ID3 tags will be stripped."),NULL);
    /* Use padding */
    UseId3v2Padding = gtk_check_button_new_with_label(_("Use padding"));
    gtk_box_pack_start(GTK_BOX(vbox),UseId3v2Padding,FALSE,FALSE,0);
    gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(UseId3v2Padding),USE_ID3V2_PADDING);
    gtk_tooltips_set_tip(Tips,UseId3v2Padding,_("Set padding in the ID3v2 tags for faster "
        "writing during next changes in the tag."),NULL);
    /* Strip tag when fields (managed by EasyTAG) are empty */
    StripTagWhenEmptyFields = gtk_check_button_new_with_label(_("Strip tags if all fields are set to blank"));
    gtk_box_pack_start(GTK_BOX(vbox),StripTagWhenEmptyFields,FALSE,FALSE,0);
    gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(StripTagWhenEmptyFields),STRIP_TAG_WHEN_EMPTY_FIELDS);
    gtk_tooltips_set_tip(Tips,StripTagWhenEmptyFields,_("As ID3v2 tags may contain other data than "
        "Title, Artist, Album, Year, Track, Genre or Comment (as an attached picture, lyrics, ...), "
        "this option allows you to strip the whole tag when these seven standard data fields have "
        "been set to blank."),NULL);
    /* Convert old ID3v2 tag version */
    ConvertOldId3v2TagVersion = gtk_check_button_new_with_label(_("Automatically convert old ID3v2 tag versions"));
    gtk_box_pack_start(GTK_BOX(vbox),ConvertOldId3v2TagVersion,FALSE,FALSE,0);
    gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(ConvertOldId3v2TagVersion),CONVERT_OLD_ID3V2_TAG_VERSION);
    gtk_tooltips_set_tip(Tips,ConvertOldId3v2TagVersion,_("If activated, an old ID3v2 tag version (as "
        "ID3v2.2) will be updated to the ID3v2.3 version."),NULL);


    /* Character Set */
    Frame = gtk_frame_new (_("Character Set for ID3 tags"));
    gtk_box_pack_start(GTK_BOX(VBox),Frame,FALSE,FALSE,0);

    vbox = gtk_vbox_new(FALSE,4);
    gtk_container_add(GTK_CONTAINER(Frame),vbox);
    gtk_container_set_border_width(GTK_CONTAINER(vbox),2);

    UseISO88591CharacterSetTranslation = gtk_check_button_new_with_label(_("Use standard ISO-8859-1 character set for ID3 tags (recommended)"));
    gtk_box_pack_start(GTK_BOX(vbox),UseISO88591CharacterSetTranslation,FALSE,FALSE,0);
    gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(UseISO88591CharacterSetTranslation),USE_ISO_8859_1_CHARACTER_SET_TRANSLATION);
    gtk_tooltips_set_tip(Tips,UseISO88591CharacterSetTranslation,_("If activated, the "
        "ID3 tag will be written with the ISO-8859-1 character set. Note that ID3v2.3 "
        "tags support only ISO-8859-1 and UTF16."),NULL);
    g_signal_connect_after(G_OBJECT(UseISO88591CharacterSetTranslation),"toggled",
                           G_CALLBACK(Use_ISO88591_Character_Set_Translation_Toggled),NULL);

    UseCharacterSetTranslation = gtk_check_button_new_with_label(_("Use this character set translation for ID3 tags :"));
    gtk_box_pack_start(GTK_BOX(vbox),UseCharacterSetTranslation,FALSE,FALSE,0);
    gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(UseCharacterSetTranslation),USE_CHARACTER_SET_TRANSLATION);
    gtk_tooltips_set_tip(Tips,UseCharacterSetTranslation,_("If activated, this "
        "character set translation will be used for reading and writing ID3 "
        "tag (for mp3 and mp2 files)."),NULL);
    g_signal_connect_after(G_OBJECT(UseCharacterSetTranslation),"toggled",
                           G_CALLBACK(Use_Character_Set_Translation_Toggled),NULL);

    // Combo boxes...
    hbox = gtk_hbox_new(FALSE,4);
    gtk_container_add(GTK_CONTAINER(vbox),hbox);
    Label = gtk_label_new(_("Convert character set from :"));
    gtk_box_pack_start(GTK_BOX(hbox),Label,FALSE,FALSE,0);

    EventBox = gtk_event_box_new();
    FileReadingCharacterSetCombo = gtk_combo_box_new_text();
    gtk_container_add(GTK_CONTAINER(EventBox),FileReadingCharacterSetCombo);
    gtk_box_pack_start(GTK_BOX(hbox),EventBox,TRUE,TRUE,0);
    gtk_tooltips_set_tip(Tips,EventBox,_("Character set used to read tag data "
        "in the file."),NULL);

    Label = gtk_label_new (_("to :"));
    gtk_box_pack_start(GTK_BOX(hbox),Label,FALSE,FALSE,0);
    EventBox = gtk_event_box_new();
    FileWritingCharacterSetCombo = gtk_combo_box_new_text();
    gtk_container_add(GTK_CONTAINER(EventBox),FileWritingCharacterSetCombo);
    gtk_box_pack_start(GTK_BOX(hbox),EventBox,TRUE,TRUE,0);
    gtk_tooltips_set_tip(Tips,EventBox,_("Character set used to write the tag "
        "data in the file."),NULL);
    Charset_Populate_Combobox(GTK_COMBO_BOX(FileReadingCharacterSetCombo), FILE_READING_CHARACTER_SET);
    Charset_Populate_Combobox(GTK_COMBO_BOX(FileWritingCharacterSetCombo), FILE_WRITING_CHARACTER_SET);
    //gtk_widget_set_usize(FileCharacterSetEntry,80,-1);
    //gtk_widget_set_usize(UserCharacterSetEntry,80,-1);
    Use_Character_Set_Translation_Toggled();
    Use_ISO88591_Character_Set_Translation_Toggled();



    /*
     * Scanner
     */
    Label = gtk_label_new (_("Scanner"));
    Frame = gtk_frame_new (_("Scanner"));
    gtk_notebook_append_page (GTK_NOTEBOOK(OptionsNoteBook),Frame,Label);
    gtk_container_set_border_width(GTK_CONTAINER(Frame), 5);

    /* Save the number of the page. Asked in Scanner window */
    OptionsNoteBook_Scanner_Page_Num = gtk_notebook_page_num(GTK_NOTEBOOK(OptionsNoteBook),Frame);

    VBox = gtk_vbox_new(FALSE,2);
    gtk_container_add(GTK_CONTAINER(Frame),VBox);
    gtk_container_set_border_width(GTK_CONTAINER(VBox), 4);

    /* Character conversion for the 'Fill Tag' scanner (=> FTS...) */
    Frame = gtk_frame_new (_("Fill Tag Scanner - Character Conversion"));
    gtk_box_pack_start(GTK_BOX(VBox),Frame,FALSE,FALSE,0);
    vbox = gtk_vbox_new(FALSE,2);
    gtk_container_add(GTK_CONTAINER(Frame),vbox);
    gtk_container_set_border_width(GTK_CONTAINER(vbox), 2);

    FTSConvertUnderscoreAndP20IntoSpace = gtk_check_button_new_with_label(_("Convert underscore "
        "character '_' and string '%20' to space ' '"));
    FTSConvertSpaceIntoUnderscore = gtk_check_button_new_with_label(_("Convert space ' ' to underscore '_'"));
    gtk_box_pack_start(GTK_BOX(vbox),FTSConvertUnderscoreAndP20IntoSpace,FALSE,FALSE,0);
    gtk_box_pack_start(GTK_BOX(vbox),FTSConvertSpaceIntoUnderscore,      FALSE,FALSE,0);
    g_signal_connect_swapped(G_OBJECT(FTSConvertUnderscoreAndP20IntoSpace),"toggled",
        G_CALLBACK(Scanner_Convert_Check_Button_Toggled_1),G_OBJECT(FTSConvertSpaceIntoUnderscore));
    g_signal_connect_swapped(G_OBJECT(FTSConvertSpaceIntoUnderscore),"toggled",
        G_CALLBACK(Scanner_Convert_Check_Button_Toggled_1),G_OBJECT(FTSConvertUnderscoreAndP20IntoSpace));

    gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(FTSConvertUnderscoreAndP20IntoSpace),
        FTS_CONVERT_UNDERSCORE_AND_P20_INTO_SPACE);
    gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(FTSConvertSpaceIntoUnderscore),
        FTS_CONVERT_SPACE_INTO_UNDERSCORE);
    gtk_tooltips_set_tip(Tips,FTSConvertUnderscoreAndP20IntoSpace,_("If activated, this conversion "
        "will be used when applying a mask from the scanner for tags."),NULL);
    gtk_tooltips_set_tip(Tips,FTSConvertSpaceIntoUnderscore,_("If activated, this conversion "
        "will be used when applying a mask from the scanner for tags."),NULL);

    /* Character conversion for the 'Rename File' scanner (=> RFS...) */
    Frame = gtk_frame_new (_("Rename File Scanner - Character Conversion"));
    gtk_box_pack_start(GTK_BOX(VBox),Frame,FALSE,FALSE,0);
    vbox = gtk_vbox_new(FALSE,2);
    gtk_container_add(GTK_CONTAINER(Frame),vbox);
    gtk_container_set_border_width(GTK_CONTAINER(vbox),2);

    RFSConvertUnderscoreAndP20IntoSpace = gtk_check_button_new_with_label(_("Convert underscore "
        "character '_' and string '%20' to space ' '"));
    RFSConvertSpaceIntoUnderscore = gtk_check_button_new_with_label(_("Convert space ' ' to underscore '_'"));
    gtk_box_pack_start(GTK_BOX(vbox),RFSConvertUnderscoreAndP20IntoSpace,FALSE,FALSE,0);
    gtk_box_pack_start(GTK_BOX(vbox),RFSConvertSpaceIntoUnderscore,      FALSE,FALSE,0);
    g_signal_connect_swapped(G_OBJECT(RFSConvertUnderscoreAndP20IntoSpace),"toggled",
                             G_CALLBACK(Scanner_Convert_Check_Button_Toggled_1),G_OBJECT(RFSConvertSpaceIntoUnderscore));
    g_signal_connect_swapped(G_OBJECT(RFSConvertSpaceIntoUnderscore),"toggled",
                             G_CALLBACK(Scanner_Convert_Check_Button_Toggled_1),G_OBJECT(RFSConvertUnderscoreAndP20IntoSpace));

    gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(RFSConvertUnderscoreAndP20IntoSpace),
        RFS_CONVERT_UNDERSCORE_AND_P20_INTO_SPACE);
    gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(RFSConvertSpaceIntoUnderscore),
        RFS_CONVERT_SPACE_INTO_UNDERSCORE);
    gtk_tooltips_set_tip(Tips,RFSConvertUnderscoreAndP20IntoSpace,_("If activated, this conversion "
        "will be used when applying a mask from the scanner for filenames."),NULL);
    gtk_tooltips_set_tip(Tips,RFSConvertSpaceIntoUnderscore,_("If activated, this conversion "
        "will be used when applying a mask from the scanner for filenames."),NULL);

    /* Character conversion for the 'Process Fields' scanner (=> PFS...) */
    Frame = gtk_frame_new (_("Process Fields Scanner - Character Conversion"));
    gtk_box_pack_start(GTK_BOX(VBox),Frame,FALSE,FALSE,0);
    vbox = gtk_vbox_new(FALSE,2);
    gtk_container_add(GTK_CONTAINER(Frame),vbox);
    gtk_container_set_border_width(GTK_CONTAINER(vbox),2);
    
    // Don't convert some words like to, feat. first letter uppercase.
    PFSDontUpperSomeWords = gtk_check_button_new_with_label(_("Don't uppercase "
        "first letter of words for some prepositions and articles."));
    gtk_box_pack_start(GTK_BOX(vbox),PFSDontUpperSomeWords, FALSE, FALSE, 0);
    gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(PFSDontUpperSomeWords), PFS_DONT_UPPER_SOME_WORDS);
    gtk_tooltips_set_tip(Tips, PFSDontUpperSomeWords, _("Don't convert first "
        "letter of the words like prepositions, articles and words like feat., "
        "when using the scanner 'First letter uppercase of each word' (for "
        "example, you will obtain 'Text in an Entry' instead of 'Text In An Entry')."), NULL);

    /* Properties of the scanner window */
    Frame = gtk_frame_new (_("Scanner Window"));
    gtk_box_pack_start(GTK_BOX(VBox),Frame,FALSE,FALSE,0);
    vbox = gtk_vbox_new(FALSE,2);
    gtk_container_add(GTK_CONTAINER(Frame),vbox);
    gtk_container_set_border_width(GTK_CONTAINER(vbox), 2);

    OpenScannerWindowOnStartup = gtk_check_button_new_with_label(_("Open the Scanner Window on startup"));
    gtk_box_pack_start(GTK_BOX(vbox),OpenScannerWindowOnStartup,FALSE,FALSE,0);
    gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(OpenScannerWindowOnStartup),OPEN_SCANNER_WINDOW_ON_STARTUP);
    gtk_tooltips_set_tip(Tips,OpenScannerWindowOnStartup,_("Activate this option to open automatically "
        "the scanner window when EasyTAG starts."),NULL);

    ScannerWindowOnTop = gtk_check_button_new_with_label(_("Scanner window always on top"));
    gtk_box_pack_start(GTK_BOX(vbox),ScannerWindowOnTop,FALSE,FALSE,0);
    gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(ScannerWindowOnTop),SCANNER_WINDOW_ON_TOP);
    gtk_tooltips_set_tip(Tips,ScannerWindowOnTop,_("If activated, the window which contains the masks "
                        "will stay always over the main window."),NULL);

    // Scanner Window Position
    hbox = gtk_hbox_new(FALSE,4);
    gtk_box_pack_start(GTK_BOX(vbox),hbox,FALSE,FALSE,0);

    SetScannerWindowPosition = gtk_check_button_new_with_label(_("Set scanner window position"));
    gtk_box_pack_start(GTK_BOX(hbox),SetScannerWindowPosition,FALSE,FALSE,0);
    gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(SetScannerWindowPosition),SET_SCANNER_WINDOW_POSITION);
    msg = g_strdup_printf(_("If activated, the scanner window will appear at the specified "
                "coordinates relative to the main window. Use it if you want to keep the same "
                "position of the scanner window each times you open it. If deactivated, the "
                "window manager determines the position."));
    gtk_tooltips_set_tip(Tips,SetScannerWindowPosition,msg,NULL);
    g_free(msg);

    Label = gtk_label_new (_("X :"));
    gtk_box_pack_start(GTK_BOX(hbox),Label,FALSE,FALSE,0);
    ScannerWindowX = gtk_entry_new();
    gtk_entry_set_max_length(GTK_ENTRY(ScannerWindowX), 5);
    gtk_box_pack_start(GTK_BOX(hbox),ScannerWindowX,FALSE,FALSE,0);
    gtk_widget_set_size_request(ScannerWindowX, 40, -1);
    Label = gtk_label_new (_("Y :"));
    gtk_box_pack_start(GTK_BOX(hbox),Label,FALSE,FALSE,0);
    ScannerWindowY = gtk_entry_new();
    gtk_entry_set_max_length(GTK_ENTRY(ScannerWindowY), 5);
    gtk_box_pack_start(GTK_BOX(hbox),ScannerWindowY,FALSE,FALSE,0);
    gtk_widget_set_size_request(ScannerWindowY, 40, -1);
    ScannerWindowButton = Create_Button_With_Icon_And_Label("easytag-grab",_(" Get current position "));
    gtk_box_pack_start(GTK_BOX(hbox),ScannerWindowButton,FALSE,FALSE,0);
    g_signal_connect(G_OBJECT(ScannerWindowButton),"clicked",
        G_CALLBACK(Get_Scanner_Window_Position),NULL);
    /* Load values */
    sprintf(temp,"%i",SCANNER_WINDOW_X);
    gtk_entry_set_text(GTK_ENTRY(ScannerWindowX),temp);
    sprintf(temp,"%i",SCANNER_WINDOW_Y);
    gtk_entry_set_text(GTK_ENTRY(ScannerWindowY),temp);
    g_signal_connect(G_OBJECT(ScannerWindowX),"insert_text",G_CALLBACK(Insert_Only_Digit),NULL);
    g_signal_connect(G_OBJECT(ScannerWindowY),"insert_text",G_CALLBACK(Insert_Only_Digit),NULL);
    // Placed at the end of the block because we must have all widgets built
    g_signal_connect(G_OBJECT(SetScannerWindowPosition),"toggled",
            G_CALLBACK(Scanner_Window_Position_Check_Button_Toggled),NULL);
    Scanner_Window_Position_Check_Button_Toggled();


    /* Other options */
    Frame = gtk_frame_new (_("Fields"));
    gtk_box_pack_start(GTK_BOX(VBox),Frame,FALSE,FALSE,0);
    vbox = gtk_vbox_new(FALSE,2);
    gtk_container_add(GTK_CONTAINER(Frame),vbox);
    gtk_container_set_border_width(GTK_CONTAINER(vbox), 2);

    // Overwrite text into tag fields
    OverwriteTagField = gtk_check_button_new_with_label(_("Overwrite fields when scanning tag"));
    gtk_box_pack_start(GTK_BOX(vbox),OverwriteTagField,FALSE,FALSE,0);
    gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(OverwriteTagField),OVERWRITE_TAG_FIELD);
    gtk_tooltips_set_tip(Tips,OverwriteTagField,_("If activated, the scanner will replace existing text "
        "in fields by the new one. If deactivated, only blank fields of the tag will be filled."),NULL);

    // Set a default comment text or CRC-32 checksum
    if (!DefaultCommentModel)
        DefaultCommentModel = gtk_list_store_new(MISC_COMBO_COUNT, G_TYPE_STRING);
    else
        gtk_list_store_clear(DefaultCommentModel);

    hbox = gtk_hbox_new(FALSE,2);
    gtk_box_pack_start(GTK_BOX(vbox),hbox,FALSE,FALSE,0);
    SetDefaultComment = gtk_check_button_new_with_label(_("Set this text as default comment :"));
    gtk_box_pack_start(GTK_BOX(hbox),SetDefaultComment,FALSE,FALSE,0);
    gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(SetDefaultComment),SET_DEFAULT_COMMENT);
    gtk_tooltips_set_tip(Tips,SetDefaultComment,_("Activate this option if you want to put the "
        "following string into the comment field when using the 'Fill Tag' scanner."),NULL);
    DefaultComment = gtk_combo_box_entry_new_with_model(GTK_TREE_MODEL(DefaultCommentModel), MISC_COMBO_TEXT);
    gtk_box_pack_start(GTK_BOX(hbox),DefaultComment,FALSE,FALSE,0);
    gtk_widget_set_size_request(GTK_WIDGET(DefaultComment), 250, -1);
    g_signal_connect(G_OBJECT(SetDefaultComment),"toggled",
            G_CALLBACK(Set_Default_Comment_Check_Button_Toggled),NULL);
    if (DEFAULT_COMMENT==NULL)
        gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(SetDefaultComment),FALSE);
    Set_Default_Comment_Check_Button_Toggled();
    /* History list */
    Load_Default_Tag_Comment_Text_List(DefaultCommentModel, MISC_COMBO_TEXT);
    Add_String_To_Combo_List(DefaultCommentModel, DEFAULT_COMMENT);
    gtk_entry_set_text(GTK_ENTRY(GTK_BIN(DefaultComment)->child), DEFAULT_COMMENT);

    // CRC32 comment
    Crc32Comment = gtk_check_button_new_with_label(_("Use CRC32 as the default "
        "comment (for files with ID3 tags only)."));
    gtk_box_pack_start(GTK_BOX(vbox),Crc32Comment,FALSE,FALSE,0);
    gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(Crc32Comment),SET_CRC32_COMMENT);
    gtk_tooltips_set_tip(Tips,Crc32Comment,_("Calculates the CRC-32 value of the file "
        "and writes it into the comment field when using the 'Fill Tag' scanner."),NULL);
    g_signal_connect_swapped(G_OBJECT(SetDefaultComment), "toggled",
        G_CALLBACK(Scanner_Convert_Check_Button_Toggled_1),G_OBJECT(Crc32Comment));
    g_signal_connect_swapped(G_OBJECT(Crc32Comment), "toggled",
        G_CALLBACK(Scanner_Convert_Check_Button_Toggled_1),G_OBJECT(SetDefaultComment));


    /*
     * CDDB
     */
    Label = gtk_label_new (_("CD Data Base"));
    Frame = gtk_frame_new (_("CD Data Base"));
    gtk_notebook_append_page (GTK_NOTEBOOK(OptionsNoteBook),Frame,Label);
    gtk_container_set_border_width(GTK_CONTAINER(Frame), 5);

    VBox = gtk_vbox_new(FALSE,2);
    gtk_container_add(GTK_CONTAINER(Frame),VBox);
    gtk_container_set_border_width(GTK_CONTAINER(VBox), 2);

    // CDDB Server Settings
    Frame = gtk_frame_new (_("Server Settings"));
    gtk_box_pack_start(GTK_BOX(VBox),Frame,FALSE,FALSE,0);
    vbox = gtk_vbox_new(FALSE,2);
    gtk_container_add(GTK_CONTAINER(Frame),vbox);
    gtk_container_set_border_width(GTK_CONTAINER(vbox), 4);

    hbox = gtk_hbox_new(FALSE,4);
    gtk_container_add(GTK_CONTAINER(vbox),hbox);
    Label = gtk_label_new(_("Name :"));
    gtk_box_pack_start(GTK_BOX(hbox),Label,FALSE,FALSE,2);
    CddbServerName = gtk_combo_box_entry_new_text();
    gtk_box_pack_start(GTK_BOX(hbox),CddbServerName,FALSE,FALSE,0);
    gtk_combo_box_append_text(GTK_COMBO_BOX(CddbServerName), "freedb.freedb.org");
    gtk_combo_box_append_text(GTK_COMBO_BOX(CddbServerName), "at.freedb.org");
    gtk_combo_box_append_text(GTK_COMBO_BOX(CddbServerName), "au.freedb.org");
    gtk_combo_box_append_text(GTK_COMBO_BOX(CddbServerName), "ca.freedb.org");
    gtk_combo_box_append_text(GTK_COMBO_BOX(CddbServerName), "ca2.freedb.org");
    gtk_combo_box_append_text(GTK_COMBO_BOX(CddbServerName), "de.freedb.org");
    gtk_combo_box_append_text(GTK_COMBO_BOX(CddbServerName), "es.freedb.org");
    gtk_combo_box_append_text(GTK_COMBO_BOX(CddbServerName), "fi.freedb.org");
    gtk_combo_box_append_text(GTK_COMBO_BOX(CddbServerName), "ru.freedb.org");
    gtk_combo_box_append_text(GTK_COMBO_BOX(CddbServerName), "uk.freedb.org");
    gtk_combo_box_append_text(GTK_COMBO_BOX(CddbServerName), "us.freedb.org");
    gtk_entry_set_text(GTK_ENTRY(GTK_BIN(CddbServerName)->child),CDDB_SERVER_NAME);

    Label = gtk_label_new (_("Port :"));
    gtk_box_pack_start(GTK_BOX(hbox),Label,FALSE,FALSE,2);
    CddbServerPort = gtk_entry_new();
    gtk_widget_set_size_request(GTK_WIDGET(CddbServerPort), 45, -1);
    gtk_entry_set_max_length(GTK_ENTRY(CddbServerPort),5);
    gtk_box_pack_start(GTK_BOX(hbox),CddbServerPort,FALSE,FALSE,0);
    sprintf(temp,"%i",CDDB_SERVER_PORT);
    gtk_entry_set_text(GTK_ENTRY(CddbServerPort),temp);
    g_signal_connect(G_OBJECT(CddbServerPort),"insert_text",G_CALLBACK(Insert_Only_Digit),NULL);

    Label = gtk_label_new (_("CGI Path :"));
    gtk_box_pack_start(GTK_BOX(hbox),Label,FALSE,FALSE,2);
    CddbServerCgiPath = gtk_entry_new();
    gtk_box_pack_start(GTK_BOX(hbox),CddbServerCgiPath,TRUE,TRUE,0);
    gtk_entry_set_text(GTK_ENTRY(CddbServerCgiPath) ,CDDB_SERVER_CGI_PATH);

    // CDDB Proxy Settings
    Frame = gtk_frame_new (_("Proxy Settings"));
    gtk_box_pack_start(GTK_BOX(VBox),Frame,FALSE,FALSE,0);

    Table = gtk_table_new(4,4,FALSE);
    gtk_container_add(GTK_CONTAINER(Frame),Table);
    gtk_table_set_row_spacings(GTK_TABLE(Table),2);
    gtk_table_set_col_spacings(GTK_TABLE(Table),4);
    gtk_container_set_border_width(GTK_CONTAINER(Table), 2);

    CddbUseProxy = gtk_check_button_new_with_label(_("Use a proxy"));
    gtk_table_attach(GTK_TABLE(Table),CddbUseProxy,0,4,0,1,GTK_FILL,GTK_FILL,0,0);
    gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(CddbUseProxy),CDDB_USE_PROXY);
    gtk_tooltips_set_tip(Tips,CddbUseProxy,_("Set active the settings of the proxy server."),NULL);

    Label = gtk_label_new(_("Host Name :"));
    gtk_table_attach(GTK_TABLE(Table),Label,0,1,1,2,GTK_FILL,GTK_FILL,0,0);
    gtk_misc_set_alignment(GTK_MISC(Label),1,0.5);
    CddbProxyName = gtk_entry_new();
    gtk_table_attach(GTK_TABLE(Table),CddbProxyName,1,2,1,2,GTK_FILL,GTK_FILL,0,0);
    if (CDDB_PROXY_NAME)
        gtk_entry_set_text(GTK_ENTRY(CddbProxyName),CDDB_PROXY_NAME);
    gtk_tooltips_set_tip(Tips,CddbProxyName,_("Name of the proxy server."),NULL);
    Label = gtk_label_new (_("Port :"));
    gtk_table_attach(GTK_TABLE(Table),Label,2,3,1,2,GTK_FILL,GTK_FILL,0,0);
    gtk_misc_set_alignment(GTK_MISC(Label),1,0.5);
    CddbProxyPort = gtk_entry_new();
    gtk_widget_set_size_request(GTK_WIDGET(CddbProxyPort), 45, -1);
    gtk_entry_set_max_length(GTK_ENTRY(CddbProxyPort),5);
    gtk_table_attach(GTK_TABLE(Table),CddbProxyPort,3,4,1,2,GTK_FILL,GTK_FILL,0,0);
    gtk_tooltips_set_tip(Tips,CddbProxyPort,_("Port of the proxy server."),NULL);
    sprintf(temp,"%i",CDDB_PROXY_PORT);
    gtk_entry_set_text(GTK_ENTRY(CddbProxyPort),temp);
    g_signal_connect(G_OBJECT(CddbProxyPort),"insert_text",G_CALLBACK(Insert_Only_Digit),NULL);
    g_signal_connect(G_OBJECT(CddbUseProxy),"toggled",G_CALLBACK(Cddb_Use_Proxy_Toggled),NULL);
    Label = gtk_label_new(_("User Name :"));
    gtk_table_attach(GTK_TABLE(Table),Label,0,1,2,3,GTK_FILL,GTK_FILL,0,0);
    gtk_misc_set_alignment(GTK_MISC(Label),1,0.5);
    CddbProxyUserName = gtk_entry_new();
    if (CDDB_PROXY_USER_NAME)
        gtk_entry_set_text(GTK_ENTRY(CddbProxyUserName),CDDB_PROXY_USER_NAME);
    gtk_table_attach(GTK_TABLE(Table),CddbProxyUserName,1,2,2,3,GTK_FILL,GTK_FILL,0,0);
    gtk_tooltips_set_tip(Tips,CddbProxyUserName,_("Name of user for the the proxy server."),NULL);
//    Label = gtk_label_new(_("(Not available!)"));
//    gtk_table_attach(GTK_TABLE(Table),Label,2,5,2,3,GTK_FILL,GTK_FILL,0,0);
    Label = gtk_label_new(_("User Password :"));
    gtk_table_attach(GTK_TABLE(Table),Label,0,1,3,4,GTK_FILL,GTK_FILL,0,0);
    gtk_misc_set_alignment(GTK_MISC(Label),1,0.5);
    CddbProxyUserPassword = gtk_entry_new();
    if (CDDB_PROXY_USER_PASSWORD)
        gtk_entry_set_text(GTK_ENTRY(CddbProxyUserPassword),CDDB_PROXY_USER_PASSWORD);
    gtk_table_attach(GTK_TABLE(Table),CddbProxyUserPassword,1,2,3,4,GTK_FILL,GTK_FILL,0,0);
    gtk_entry_set_visibility(GTK_ENTRY(CddbProxyUserPassword),FALSE);
    gtk_tooltips_set_tip(Tips,CddbProxyUserPassword,_("Password of user for the the proxy server."),NULL);
//    Label = gtk_label_new(_("(Not available!)"));
//    gtk_table_attach(GTK_TABLE(Table),Label,2,5,3,4,GTK_FILL,GTK_FILL,0,0);
    Cddb_Use_Proxy_Toggled();


    // CDDB Window
    Frame = gtk_frame_new (_("CD Data Base Window"));
    gtk_box_pack_start(GTK_BOX(VBox),Frame,FALSE,FALSE,0);

    Table = gtk_table_new(6,3,FALSE);
    gtk_container_add(GTK_CONTAINER(Frame),Table);
    gtk_table_set_row_spacings(GTK_TABLE(Table),2);
    gtk_table_set_col_spacings(GTK_TABLE(Table),4);
    gtk_container_set_border_width(GTK_CONTAINER(Table), 2);

    // Window size
    SetCddbWindowSize = gtk_check_button_new_with_label(_("Set CD Data Base window size"));
    gtk_table_attach(GTK_TABLE(Table),SetCddbWindowSize,0,1,0,1,GTK_FILL,GTK_FILL,0,0);
    Label = gtk_label_new (_("Width :"));
    gtk_table_attach(GTK_TABLE(Table),Label,1,2,0,1,GTK_FILL,GTK_FILL,0,0);
    CddbWindowWidth = gtk_entry_new();
    gtk_entry_set_max_length(GTK_ENTRY(CddbWindowWidth), 4);
    gtk_table_attach(GTK_TABLE(Table),CddbWindowWidth,2,3,0,1,GTK_FILL,GTK_FILL,0,0);
    gtk_widget_set_size_request(CddbWindowWidth, 40, -1);
    Label = gtk_label_new (_("Height :"));
    gtk_table_attach(GTK_TABLE(Table),Label,3,4,0,1,GTK_FILL,GTK_FILL,0,0);
    CddbWindowHeight = gtk_entry_new();
    gtk_entry_set_max_length(GTK_ENTRY(CddbWindowHeight), 4);
    gtk_table_attach(GTK_TABLE(Table),CddbWindowHeight,4,5,0,1,GTK_FILL,GTK_FILL,0,0);
    gtk_widget_set_size_request(CddbWindowHeight, 40, -1);
    CddbWindowButton = Create_Button_With_Icon_And_Label("easytag-grab",_(" Get current size "));
    gtk_table_attach(GTK_TABLE(Table),CddbWindowButton,5,6,0,1,GTK_FILL,GTK_FILL,0,0);
    g_signal_connect(G_OBJECT(CddbWindowButton),"clicked",G_CALLBACK(Get_Cddb_Window_Size),NULL);
    gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(SetCddbWindowSize),SET_CDDB_WINDOW_SIZE);
    sprintf(temp,"%i",CDDB_WINDOW_HEIGHT);
    gtk_entry_set_text(GTK_ENTRY(CddbWindowHeight),temp);
    sprintf(temp,"%i",CDDB_WINDOW_WIDTH);
    gtk_entry_set_text(GTK_ENTRY(CddbWindowWidth),temp);
    g_signal_connect(G_OBJECT(CddbWindowHeight),"insert_text",G_CALLBACK(Insert_Only_Digit),NULL);
    g_signal_connect(G_OBJECT(CddbWindowWidth),"insert_text",G_CALLBACK(Insert_Only_Digit),NULL);
    // Placed at the end of the block cause we must have all widgets built
    g_signal_connect(G_OBJECT(SetCddbWindowSize),"toggled",G_CALLBACK(Set_Cddb_Window_Size_Toggled),NULL);
    Set_Cddb_Window_Size_Toggled();
    gtk_tooltips_set_tip(Tips,SetCddbWindowSize,_("Specify the default size for the CDDB window."),NULL);
    gtk_tooltips_set_tip(Tips,CddbWindowWidth,_("Width: set -1 for automatic size"),NULL);
    gtk_tooltips_set_tip(Tips,CddbWindowHeight,_("Heigth: set -1 for automatic size"),NULL);

    // Pane handle position
    SetCddbPaneHandlePosition = gtk_check_button_new_with_label(_("Set pane handle position"));
    gtk_table_attach(GTK_TABLE(Table),SetCddbPaneHandlePosition,0,2,1,2,GTK_FILL,GTK_FILL,0,0);
    Label = gtk_label_new (_("Position :"));
    gtk_table_attach(GTK_TABLE(Table),Label,2,4,1,2,GTK_FILL,GTK_FILL,0,0);
    gtk_misc_set_alignment(GTK_MISC(Label),1,0.5);
    CddbPaneHandlePosition = gtk_entry_new();
    gtk_entry_set_max_length(GTK_ENTRY(CddbPaneHandlePosition), 4);
    gtk_table_attach(GTK_TABLE(Table),CddbPaneHandlePosition,4,5,1,2,GTK_FILL,GTK_FILL,0,0);
    sprintf(temp,"%i",CDDB_PANE_HANDLE_POSITION);
    gtk_entry_set_text(GTK_ENTRY(CddbPaneHandlePosition),temp);
    g_signal_connect(G_OBJECT(CddbPaneHandlePosition),"insert_text",G_CALLBACK(Insert_Only_Digit),NULL);
    gtk_widget_set_size_request(CddbPaneHandlePosition, 40, -1);
    CddbPaneHandleButton = Create_Button_With_Icon_And_Label("easytag-grab",_(" Get current position "));
    gtk_table_attach(GTK_TABLE(Table),CddbPaneHandleButton,5,6,1,2,GTK_FILL,GTK_FILL,0,0);
    g_signal_connect (G_OBJECT(CddbPaneHandleButton),"clicked", G_CALLBACK(Get_Cddb_Pane_Handle_Position),NULL);
    gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(SetCddbPaneHandlePosition),SET_CDDB_PANE_HANDLE_POSITION);
    gtk_tooltips_set_tip(Tips,SetCddbPaneHandlePosition,_("Specify the default position for the pane "
        "handle in the CDDB window."),NULL);
    gtk_tooltips_set_tip(Tips,CddbPaneHandlePosition,_("Position: set -1 for automatic positioning"),NULL);
    g_signal_connect(G_OBJECT(SetCddbPaneHandlePosition),"toggled",G_CALLBACK(Set_Cddb_Pane_Handle_Position_Toggled),NULL);
    Set_Cddb_Pane_Handle_Position_Toggled();

    // Track Name list (CDDB results)
    Frame = gtk_frame_new (_("Track Name List"));
    gtk_box_pack_start(GTK_BOX(VBox),Frame,FALSE,FALSE,0);

    vbox = gtk_vbox_new(FALSE,2);
    gtk_container_add(GTK_CONTAINER(Frame),vbox);
    gtk_container_set_border_width(GTK_CONTAINER(vbox), 4);

    CddbFollowFile = gtk_check_button_new_with_label(_("Select corresponding audio "
        "file (according position or DLM if activated below)"));
    gtk_box_pack_start(GTK_BOX(vbox),CddbFollowFile,FALSE,FALSE,2);
    gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(CddbFollowFile),CDDB_FOLLOW_FILE);
    gtk_tooltips_set_tip(Tips,CddbFollowFile,_("If activated, when selecting a "
        "line in the list of tracks name, the corresponding audio file in the "
        "main list will be also selected."),NULL);

    // Check box to use DLM
    CddbUseDLM = gtk_check_button_new_with_label(_("Use the Levenshtein algorithm "
        "(DLM) to match lines (using title) with audio files (using filename)"));
    gtk_box_pack_start(GTK_BOX(vbox),CddbUseDLM,FALSE,FALSE,2);
    gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(CddbUseDLM),CDDB_USE_DLM);
    gtk_tooltips_set_tip(Tips,CddbUseDLM,_("When activating this option, the "
        "Levenshtein algorithm (DLM : Damerau-Levenshtein Metric) will be used "
        "to match the CDDB title against every file name in the current folder, "
        "and to select the best match. This will be used when selecting the "
        "corresponding audio file, or applying cddb results, instead of using "
        "directly the position order."),NULL);


    /*
     * Confirmation
     */
    Label = gtk_label_new (_("Confirmation"));
    Frame = gtk_frame_new (_("Confirmation"));
    gtk_notebook_append_page (GTK_NOTEBOOK(OptionsNoteBook),Frame,Label);
    gtk_container_set_border_width(GTK_CONTAINER(Frame), 5);

    VBox = gtk_vbox_new(FALSE,2);
    gtk_container_add(GTK_CONTAINER(Frame),VBox);
    gtk_container_set_border_width(GTK_CONTAINER(VBox), 2);

    ConfirmBeforeExit = gtk_check_button_new_with_label(_("Confirm exit from program"));
    gtk_box_pack_start(GTK_BOX(VBox),ConfirmBeforeExit,FALSE,FALSE,0);
    gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(ConfirmBeforeExit),CONFIRM_BEFORE_EXIT);
    gtk_tooltips_set_tip(Tips,ConfirmBeforeExit,_("If activated, opens a dialog box to ask "
        "confirmation before exiting the program."),NULL);

    ConfirmWriteTag = gtk_check_button_new_with_label(_("Confirm writing of file tag"));
    gtk_box_pack_start(GTK_BOX(VBox),ConfirmWriteTag,FALSE,FALSE,0);
    gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(ConfirmWriteTag),CONFIRM_WRITE_TAG);

    ConfirmRenameFile = gtk_check_button_new_with_label(_("Confirm renaming of file"));
    gtk_box_pack_start(GTK_BOX(VBox),ConfirmRenameFile,FALSE,FALSE,0);
    gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(ConfirmRenameFile),CONFIRM_RENAME_FILE);

    ConfirmDeleteFile = gtk_check_button_new_with_label(_("Confirm deleting of file"));
    gtk_box_pack_start(GTK_BOX(VBox),ConfirmDeleteFile,FALSE,FALSE,0);
    gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(ConfirmDeleteFile),CONFIRM_DELETE_FILE);

    ConfirmWritePlayList = gtk_check_button_new_with_label(_("Confirm writing of playlist"));
    gtk_box_pack_start(GTK_BOX(VBox),ConfirmWritePlayList,FALSE,FALSE,0);
    gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(ConfirmWritePlayList),CONFIRM_WRITE_PLAYLIST);


    /* Separator line */
    Separator = gtk_hseparator_new();
    gtk_box_pack_start(GTK_BOX(VBox),Separator,FALSE,FALSE,0);

    /* Save configuration */
    SaveConfigBeforeExit = gtk_check_button_new_with_label(_("Save the configuration before "
        "exiting the program"));
    gtk_box_pack_start(GTK_BOX(VBox),SaveConfigBeforeExit,FALSE,FALSE,0);
    gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(SaveConfigBeforeExit),SAVE_CONFIG_BEFORE_EXIT);
    gtk_tooltips_set_tip(Tips,SaveConfigBeforeExit,_("If activated, the configuration will be "
        "automatically saved before exiting the program."),NULL);



    /* 
     * Buttons box of Option Window
     */
    ButtonBox = gtk_hbutton_box_new ();
    gtk_box_pack_start(GTK_BOX(OptionsVBox), ButtonBox, FALSE, FALSE, 4);

    gtk_button_box_set_layout (GTK_BUTTON_BOX (ButtonBox), GTK_BUTTONBOX_END);
    gtk_box_set_spacing (GTK_BOX(ButtonBox), 15);


    /* Cancel Button */
    Button = Create_Button_With_Pixmap(BUTTON_CANCEL);
    gtk_container_add(GTK_CONTAINER(ButtonBox), Button);
    g_signal_connect(G_OBJECT(Button),"clicked", G_CALLBACK(OptionsWindow_Cancel_Button),NULL);
    GTK_WIDGET_SET_FLAGS(Button, GTK_CAN_DEFAULT);
    gtk_widget_grab_default (Button);
    gtk_tooltips_set_tip(Tips,Button,_("Close this window without saving"),NULL);

    /* Save Button */
    Button = Create_Button_With_Pixmap(BUTTON_SAVE);
    gtk_container_add(GTK_CONTAINER(ButtonBox), Button);
    g_signal_connect(G_OBJECT(Button),"clicked", G_CALLBACK(OptionsWindow_Save_Button),NULL);
    GTK_WIDGET_SET_FLAGS(Button, GTK_CAN_DEFAULT);
    gtk_tooltips_set_tip(Tips,Button,_("Save changes and close this window"),NULL);

    /* Apply Button */
    Button = Create_Button_With_Pixmap(BUTTON_APPLY);
    gtk_container_add(GTK_CONTAINER(ButtonBox),Button);
    g_signal_connect(G_OBJECT(Button),"clicked",G_CALLBACK(OptionsWindow_Ok_Button),NULL);
    GTK_WIDGET_SET_FLAGS(Button, GTK_CAN_DEFAULT);
    gtk_tooltips_set_tip(Tips,Button,_("Apply changes (but don't save) and close this window"),NULL);


    /* Show all in the options window */
    gtk_widget_show_all(OptionsWindow);

    /* Load the default page */
    gtk_notebook_set_current_page(GTK_NOTEBOOK(OptionsNoteBook), OPTIONS_NOTEBOOK_PAGE);
}


void Set_Main_Window_Size_Toggled (void)
{
    gtk_widget_set_sensitive(MainWindowWidth, GTK_TOGGLE_BUTTON(SetMainWindowSize)->active);
    gtk_widget_set_sensitive(MainWindowHeight,GTK_TOGGLE_BUTTON(SetMainWindowSize)->active);
    gtk_widget_set_sensitive(MainWindowButton,GTK_TOGGLE_BUTTON(SetMainWindowSize)->active);
}

void Set_Pane_Handle_Position1_Toggled (void)
{
    gtk_widget_set_sensitive(PaneHandlePosition1,GTK_TOGGLE_BUTTON(SetPaneHandlePosition1)->active);
    gtk_widget_set_sensitive(PaneHandleButton1,  GTK_TOGGLE_BUTTON(SetPaneHandlePosition1)->active);
}

void Set_Pane_Handle_Position2_Toggled (void)
{
    gtk_widget_set_sensitive(PaneHandlePosition2,GTK_TOGGLE_BUTTON(SetPaneHandlePosition2)->active);
    gtk_widget_set_sensitive(PaneHandleButton2,  GTK_TOGGLE_BUTTON(SetPaneHandlePosition2)->active);
}

void Set_Pane_Handle_Position3_Toggled (void)
{
    gtk_widget_set_sensitive(PaneHandlePosition3,GTK_TOGGLE_BUTTON(SetPaneHandlePosition3)->active);
    gtk_widget_set_sensitive(PaneHandleButton3,  GTK_TOGGLE_BUTTON(SetPaneHandlePosition3)->active);
}

void Scanner_Window_Position_Check_Button_Toggled (void)
{
    gtk_widget_set_sensitive(ScannerWindowX,     GTK_TOGGLE_BUTTON(SetScannerWindowPosition)->active);
    gtk_widget_set_sensitive(ScannerWindowY,     GTK_TOGGLE_BUTTON(SetScannerWindowPosition)->active);
    gtk_widget_set_sensitive(ScannerWindowButton,GTK_TOGGLE_BUTTON(SetScannerWindowPosition)->active);
}

void Set_Default_Comment_Check_Button_Toggled (void)
{
    gtk_widget_set_sensitive(DefaultComment,GTK_TOGGLE_BUTTON(SetDefaultComment)->active);
}

void Number_Track_Formated_Toggled (void)
{
    gtk_widget_set_sensitive(NumberTrackFormatedSpinButton,GTK_TOGGLE_BUTTON(NumberTrackFormated)->active);
    // To update the example...
    g_signal_emit_by_name(G_OBJECT(NumberTrackFormatedSpinButton),"changed",NULL);
}

void Number_Track_Formated_Spin_Button_Changed (GtkObject *Label, GtkObject *SpinButton)
{
    gchar *tmp;
    gint val;

    if (GTK_TOGGLE_BUTTON(NumberTrackFormated)->active)
        val = gtk_spin_button_get_value_as_int(GTK_SPIN_BUTTON(SpinButton));
    else
        val = 1;

    // For translators : be aware to NOT translate '%.*d' in this string
    tmp = g_strdup_printf(_("(Example : %.*d_-_Track_name_1.mp3)"),val,1);

    gtk_label_set_text(GTK_LABEL(Label),tmp);
    g_free(tmp);
}

void Use_ISO88591_Character_Set_Translation_Toggled (void)
{
    gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(UseCharacterSetTranslation),!GTK_TOGGLE_BUTTON(UseISO88591CharacterSetTranslation)->active);
}

void Use_Character_Set_Translation_Toggled (void)
{
    gtk_widget_set_sensitive(FileReadingCharacterSetCombo,GTK_TOGGLE_BUTTON(UseCharacterSetTranslation)->active);
    gtk_widget_set_sensitive(FileWritingCharacterSetCombo,GTK_TOGGLE_BUTTON(UseCharacterSetTranslation)->active);

    gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(UseISO88591CharacterSetTranslation),!GTK_TOGGLE_BUTTON(UseCharacterSetTranslation)->active);
}

void Cddb_Use_Proxy_Toggled (void)
{
    gtk_widget_set_sensitive(CddbProxyName,GTK_TOGGLE_BUTTON(CddbUseProxy)->active);
    gtk_widget_set_sensitive(CddbProxyPort,GTK_TOGGLE_BUTTON(CddbUseProxy)->active);
    gtk_widget_set_sensitive(CddbProxyUserName,GTK_TOGGLE_BUTTON(CddbUseProxy)->active);
    gtk_widget_set_sensitive(CddbProxyUserPassword,GTK_TOGGLE_BUTTON(CddbUseProxy)->active);
}

void Set_Cddb_Window_Size_Toggled (void)
{
    gtk_widget_set_sensitive(CddbWindowWidth, GTK_TOGGLE_BUTTON(SetCddbWindowSize)->active);
    gtk_widget_set_sensitive(CddbWindowHeight,GTK_TOGGLE_BUTTON(SetCddbWindowSize)->active);
    gtk_widget_set_sensitive(CddbWindowButton,GTK_TOGGLE_BUTTON(SetCddbWindowSize)->active);
}

void Set_Cddb_Pane_Handle_Position_Toggled (void)
{
    gtk_widget_set_sensitive(CddbPaneHandlePosition,GTK_TOGGLE_BUTTON(SetCddbPaneHandlePosition)->active);
    gtk_widget_set_sensitive(CddbPaneHandleButton,  GTK_TOGGLE_BUTTON(SetCddbPaneHandlePosition)->active);
}

/* Callback from Open_OptionsWindow */
gboolean OptionsWindow_Key_Press (GtkWidget *window, GdkEvent *event)
{
    GdkEventKey *kevent;

    if (event && event->type == GDK_KEY_PRESS)
    {
        kevent = (GdkEventKey *)event;
        switch(kevent->keyval)
        {
            case GDK_Escape:    
            {
                OptionsWindow_Quit();
                break;
            }
        }
    }
    return FALSE;
}

/* Callback from Open_OptionsWindow */
void OptionsWindow_Ok_Button(void)
{
    if (!Check_Config()) return;
    Add_String_To_Combo_List(DefaultPathModel, (gchar*) gtk_entry_get_text(GTK_ENTRY(GTK_BIN(DefaultPathToMp3)->child)));
    Add_String_To_Combo_List(FilePlayerModel, (gchar*) gtk_entry_get_text(GTK_ENTRY(GTK_BIN(FilePlayerCombo)->child)));
    Add_String_To_Combo_List(DefaultCommentModel, (gchar*) gtk_entry_get_text(GTK_ENTRY(GTK_BIN(DefaultComment)->child)));
    Apply_Changes();
    OptionsWindow_Quit();
    Statusbar_Message(_("Changes applied"),TRUE);
}

/* Callback from Open_OptionsWindow */
void OptionsWindow_Save_Button(void)
{
    if (!Check_Config()) return;
    Add_String_To_Combo_List(DefaultPathModel, (gchar*) gtk_entry_get_text(GTK_ENTRY(GTK_BIN(DefaultPathToMp3)->child)));
    Add_String_To_Combo_List(FilePlayerModel, (gchar*) gtk_entry_get_text(GTK_ENTRY(GTK_BIN(FilePlayerCombo)->child)));
    Add_String_To_Combo_List(DefaultCommentModel, (gchar*) gtk_entry_get_text(GTK_ENTRY(GTK_BIN(DefaultComment)->child)));
    Save_Changes();
    OptionsWindow_Quit();
    Statusbar_Message(_("Configuration saved"),TRUE);
}

/* Callback from Open_OptionsWindow */
void OptionsWindow_Cancel_Button(void)
{
    OptionsWindow_Quit();
    Statusbar_Message(_("Configuration unchanged"),TRUE);
}

/* Callback from Open_OptionsWindow */
void OptionsWindow_Quit(void)
{
    if (OptionsWindow)
    {
        /* Save combobox history lists before exit */
        Save_Default_Path_To_MP3_List(DefaultPathModel, MISC_COMBO_TEXT);
        Save_Default_Tag_Comment_Text_List(DefaultCommentModel, MISC_COMBO_TEXT);
        Save_Audio_File_Player_List(FilePlayerModel, MISC_COMBO_TEXT);

        /* Now quit */
        gtk_widget_destroy(OptionsWindow);
        OptionsWindow = (GtkWidget *)NULL;
        gtk_widget_set_sensitive(MainWindow, TRUE);
    }
}


/*
 * Check_Config: Check if config informations are correct
 * dsd: Check this... going from utf8 to raw is dodgy stuff
 *
 * Problem noted : if a character is escaped (like : 'C\351line DION') in
 *                 gtk_file_chooser it will converted to UTF-8. So after, there
 *                 is a problem to convert it in the right system encoding to be
 *                 passed to stat(), and it can find the directory.
 * exemple :
 *  - initial file on system                        : C\351line DION - D'eux (1995)
 *  - converted to UTF-8 (path_utf8)                : Céline DION - D'eux (1995)
 *  - try to convert to system encoding (path_real) : ?????
 */
gint Check_DefaultPathToMp3 (void)
{
    gchar *path_utf8;
    gchar *path_real;
    struct stat stbuf;

    path_utf8 = g_strdup(gtk_entry_get_text(GTK_ENTRY(GTK_BIN(DefaultPathToMp3)->child)));
    if (!path_utf8 || g_utf8_strlen(path_utf8, -1) < 1)
    {
        g_free(path_utf8);
        return 1;
    }

    path_real = filename_from_display(path_utf8);

    if ( stat(path_real,&stbuf)==0 && S_ISDIR(stbuf.st_mode) )
    {
        g_free(path_real);
        g_free(path_utf8);
        return 1;    /* Path is good */
    }else
    {
        gchar *msg = g_strdup_printf(_(" The selected path for 'Default path to "
            "files' isn't valid!\n'%s'\n(%s) "),path_utf8,
            (stat(path_real,&stbuf)==0)?_("Not a directory"):g_strerror(errno) );
        GtkWidget *msgbox = msg_box_new (_("Confirm..."),msg,GTK_STOCK_DIALOG_ERROR,BUTTON_OK,0);
        msg_box_hide_check_button(MSG_BOX(msgbox));
        msg_box_run(MSG_BOX(msgbox));
        gtk_widget_destroy(msgbox);
        g_free(msg);
        g_free(path_real);
        g_free(path_utf8);
        return 0;
    }
}
gint Check_Size (void)
{
    if ( atoi(gtk_entry_get_text(GTK_ENTRY(MainWindowHeight))) > gdk_screen_height() )
    {
        gchar *msg = g_strdup_printf(_(" Window height (%d) is bigger than screen height (%d)! "),
                atoi(gtk_entry_get_text(GTK_ENTRY(MainWindowHeight))),gdk_screen_height());
        GtkWidget *msgbox = msg_box_new (_("Size Error..."),msg,GTK_STOCK_DIALOG_ERROR,BUTTON_OK,0);
        msg_box_hide_check_button(MSG_BOX(msgbox));
        msg_box_run(MSG_BOX(msgbox));
        gtk_widget_destroy(msgbox);
        g_free(msg);
        return 0;
    }
    if ( atoi(gtk_entry_get_text(GTK_ENTRY(MainWindowWidth))) > gdk_screen_width() )
    {
        gchar *msg = g_strdup_printf(_(" Window width (%d) is bigger than screen width (%d)! "),
                atoi(gtk_entry_get_text(GTK_ENTRY(MainWindowWidth))),gdk_screen_width());
        GtkWidget *msgbox = msg_box_new (_("Size Error..."),msg,GTK_STOCK_DIALOG_ERROR,BUTTON_OK,0);
        msg_box_hide_check_button(MSG_BOX(msgbox));
        msg_box_run(MSG_BOX(msgbox));
        gtk_widget_destroy(msgbox);
        g_free(msg);
        return 0;
    }
    /* Size is good */
    return 1;
}

/*
 * The character set conversion is used for ID3 tag. UTF-8 is used to display.
 *  - reading_character is converted to UTF-8
 *  - writing_character is converted from UTF-8
 */
gint Check_CharacterSetTranslation (void)
{
    gchar *temp;
    gchar *reading_character;
    gchar *writing_character;

    temp = Get_Active_Combo_Box_Item(GTK_COMBO_BOX(FileReadingCharacterSetCombo));
    reading_character = Charset_Get_Name_From_Title(temp);
    g_free(temp);

    temp = Get_Active_Combo_Box_Item(GTK_COMBO_BOX(FileWritingCharacterSetCombo));
    writing_character = Charset_Get_Name_From_Title(temp);
    g_free(temp);

    // Check conversion when reading file
    if ( GTK_TOGGLE_BUTTON(UseCharacterSetTranslation)->active
    && (test_conversion_charset(reading_character,"UTF-8")!=TRUE) )
    {
        gchar *msg = g_strdup_printf(_("The character set translation from '%s'\n"
                                       "to '%s' isn't supported!"),reading_character,"UTF-8");
        GtkWidget *msgbox = msg_box_new (_("Error..."),msg,GTK_STOCK_DIALOG_ERROR,BUTTON_OK,0);
        msg_box_hide_check_button(MSG_BOX(msgbox));
        msg_box_run(MSG_BOX(msgbox));
        gtk_widget_destroy(msgbox);
        g_free(msg);
        gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(UseCharacterSetTranslation),FALSE);
        return 0;
    }
    // Check conversion when writing file
    if ( GTK_TOGGLE_BUTTON(UseCharacterSetTranslation)->active
    && (test_conversion_charset("UTF-8",writing_character)!=TRUE) )
    {
        gchar *msg = g_strdup_printf(_("The character set translation from '%s'\n"
                                       "to '%s' isn't supported!"),"UTF-8",writing_character);
        GtkWidget *msgbox = msg_box_new (_("Error..."),msg,GTK_STOCK_DIALOG_ERROR,BUTTON_OK,0);
        msg_box_hide_check_button(MSG_BOX(msgbox));
        msg_box_run(MSG_BOX(msgbox));
        gtk_widget_destroy(msgbox);
        g_free(msg);
        gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(UseCharacterSetTranslation),FALSE);
        return 0;
    }
    return 1;
}

gint Check_DefaultComment (void)
{
    const gchar *file;

    file = gtk_entry_get_text(GTK_ENTRY(GTK_BIN(DefaultComment)->child));
    if (!file || g_utf8_strlen(file, -1) < 1)
        gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(SetDefaultComment),FALSE);

    return 1;    /* A blank entry is ignored */
}

gint Check_Config (void)
{
    if ( Check_DefaultPathToMp3() 
    && Check_Size() 
    && Check_CharacterSetTranslation() 
    && Check_DefaultComment() )
        return 1;    /* No problem detected */
    else
        return 0;    /* Oups! */
}



void File_Selection_Window_For_File (GtkWidget *entry)
{
    Open_File_Selection_Window(entry, _("Select directory..."), GTK_FILE_CHOOSER_ACTION_OPEN);
}

void File_Selection_Window_For_Directory (GtkWidget *entry)
{
    Open_File_Selection_Window(entry, _("Select file..."), GTK_FILE_CHOOSER_ACTION_SELECT_FOLDER);
}

/*
 * Open the file selection window and saves the selected file path into entry
 */
static void Open_File_Selection_Window (GtkWidget *entry, gchar *title, GtkFileChooserAction action)
{
    gchar *tmp;
    gchar *filename, *filename_utf8;
    GtkWidget *FileSelectionWindow;
    GtkWindow *parent_window = NULL;

    parent_window = (GtkWindow*) gtk_widget_get_toplevel(entry);
    if (!GTK_WIDGET_TOPLEVEL(parent_window))
    {
        g_warning("Could not get parent window\n");
        return;
    }

    FileSelectionWindow = gtk_file_chooser_dialog_new(title, parent_window, action,
                                                      GTK_STOCK_CANCEL, GTK_RESPONSE_CANCEL,
                                                      GTK_STOCK_OPEN, GTK_RESPONSE_ACCEPT,
                                                      NULL);
    // Set initial directory
    tmp = (gchar*) gtk_entry_get_text(GTK_ENTRY(entry));
    if (tmp && *tmp)
    {
        if (!gtk_file_chooser_set_filename(GTK_FILE_CHOOSER(FileSelectionWindow),tmp))
            gtk_file_chooser_set_current_folder(GTK_FILE_CHOOSER(FileSelectionWindow),tmp);
    }


    if (gtk_dialog_run(GTK_DIALOG(FileSelectionWindow)) == GTK_RESPONSE_ACCEPT)
    {
        filename = gtk_file_chooser_get_filename(GTK_FILE_CHOOSER(FileSelectionWindow));
        filename_utf8 = filename_to_display(filename);
        gtk_entry_set_text(GTK_ENTRY(entry),filename_utf8);
        g_free(filename);
        g_free(filename_utf8);
    }
    gtk_widget_destroy(FileSelectionWindow);
}

/* 
 * Get current size of main window and complete "window size" fields
 */
void Get_Main_Window_Size (void)
{
    gint height;
    gint width;
    gchar temp[MAX_STRING_LEN];

    gdk_drawable_get_size(MainWindow->window,&width,&height);
    sprintf(temp,"%i",height);
    gtk_entry_set_text(GTK_ENTRY(MainWindowHeight),temp);
    sprintf(temp,"%i",width);
    gtk_entry_set_text(GTK_ENTRY(MainWindowWidth),temp);
}



/* 
 * Get current position of pane handle
 */
void Get_Pane_Handle_Position1 (void)
{
    gchar temp[MAX_STRING_LEN];

    sprintf(temp,"%i",GTK_PANED(MainWindowHPaned)->child1_size);
    gtk_entry_set_text(GTK_ENTRY(PaneHandlePosition1),temp);
}
void Get_Pane_Handle_Position2 (void)
{
    gchar temp[MAX_STRING_LEN];

    sprintf(temp,"%i",GTK_PANED(BrowserHPaned)->child1_size);
    gtk_entry_set_text(GTK_ENTRY(PaneHandlePosition2),temp);
}
void Get_Pane_Handle_Position3 (void)
{
    gchar temp[MAX_STRING_LEN];

    sprintf(temp,"%i",GTK_PANED(ArtistAlbumVPaned)->child1_size);
    gtk_entry_set_text(GTK_ENTRY(PaneHandlePosition3),temp);
}



/* 
 * Get current position of scanner window with regards to main 
 * window and complete "window position" fields.
 */
void Get_Scanner_Window_Position (void)
{
    gint x,x1;
    gint y,y1;
    gchar temp[MAX_STRING_LEN];

    if (!MainWindow) return;
    if (!ScannerWindow)
    {
        GtkWidget *msgbox = msg_box_new (_("Error..."),_("The scanner window must be open to get its current position!"),GTK_STOCK_DIALOG_ERROR,BUTTON_OK,0);
        msg_box_hide_check_button(MSG_BOX(msgbox));
        msg_box_run(MSG_BOX(msgbox));
        gtk_widget_destroy(msgbox);
    }else
    {
        gdk_window_get_position(MainWindow->window,&x,&y);
        gdk_window_get_position(ScannerWindow->window,&x1,&y1);
        sprintf(temp,"%i",x1-x);
        gtk_entry_set_text(GTK_ENTRY(ScannerWindowX),temp);
        sprintf(temp,"%i",y1-y);
        gtk_entry_set_text(GTK_ENTRY(ScannerWindowY),temp);
    }
}



/*
 * Manage Check buttons into Scanner tab: conversion group
 * This reproduces "something" like the behaviour of radio buttons with check buttons
 */
void Scanner_Convert_Check_Button_Toggled_1 (GtkObject *object_rec, GtkObject *object_emi)
{
    if (!object_rec && !object_emi) return;

    if (GTK_TOGGLE_BUTTON(object_emi)->active == TRUE)
        gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(object_rec),
                        !GTK_TOGGLE_BUTTON(object_emi)->active);

}


void Get_Cddb_Window_Size (void)
{
    gint height;
    gint width;
    gchar temp[MAX_STRING_LEN];

    if (!CddbWindow)
    {
        GtkWidget *msgbox = msg_box_new (_("Error..."),_("The CDDB window must be "
            "open to get its current position!"),GTK_STOCK_DIALOG_ERROR,BUTTON_OK,0);
        msg_box_hide_check_button(MSG_BOX(msgbox));
        msg_box_run(MSG_BOX(msgbox));
        gtk_widget_destroy(msgbox);
        return;
    }else
    {
        gdk_drawable_get_size(CddbWindow->window,&width,&height);
        sprintf(temp,"%i",height);
        gtk_entry_set_text(GTK_ENTRY(CddbWindowHeight),temp);
        sprintf(temp,"%i",width);
        gtk_entry_set_text(GTK_ENTRY(CddbWindowWidth),temp);
    }
}

void Get_Cddb_Pane_Handle_Position (void)
{
    gchar temp[MAX_STRING_LEN];

    if (!CddbWindow)
    {
        GtkWidget *msgbox = msg_box_new (_("Error..."),_("The CDDB window must be "
            "open to get its current position!"),GTK_STOCK_DIALOG_ERROR,BUTTON_OK,0);
        msg_box_hide_check_button(MSG_BOX(msgbox));
        msg_box_run(MSG_BOX(msgbox));
        gtk_widget_destroy(msgbox);
        return;
    }else
    {
        sprintf(temp,"%i",GTK_PANED(CddbWindowHPaned)->child1_size);
        gtk_entry_set_text(GTK_ENTRY(CddbPaneHandlePosition),temp);
    }
}


void DefaultPathToMp3_Combo_Add_String (void)
{
    const gchar *path;

    path = gtk_entry_get_text(GTK_ENTRY(GTK_BIN(DefaultPathToMp3)->child));
    Add_String_To_Combo_List(GTK_LIST_STORE(DefaultPathModel), (gchar *)path);
}
