package EBox::OpenVPN::Daemon;
# package: Parent class for the distinct types of OpenVPN daemons
use strict;
use warnings;

use base qw(EBox::OpenVPN::ModulePartition EBox::NetworkObserver);

use File::Basename;

sub new
{
    my ($class, $name, $daemonPrefix, $openvpnModule) = @_;
    
    my $confKeysBase = "$daemonPrefix/$name";
    if (!$openvpnModule->dir_exists($confKeysBase) ) {
	throw EBox::Exceptions::Internal("Tried to instantiate a daemon with a name not found in module configuration: $name");
    }


    my $self = $class->SUPER::new($confKeysBase, $openvpnModule);
    $self->{name} = $name;
      
    bless $self, $class;

    return $self;
}


sub _openvpnModule
{
    my ($self) = @_;
    return $self->fullModule();
}

#
# Method: init
#
#   Must be implimented by child to do initalisation stuff
#
#  Parametes:
#        $name        - the name of the daemon
#        @params      - initialization parameters, may be different for each daemon type
#
sub init
{
  throw EBox::Exceptions::NotImplemented();
}



sub name
{
    my ($self) = @_;
    return $self->{name};
}


sub  ifaceNumber
{
  my ($self) = @_;
  return $self->getConfInt('iface_number');
}

#
# Method: iface
#
#   gets the iface device name used for this daemon
#
# Returns:
#    - the device name
sub iface
{
  my ($self) = @_;

  my $ifaceType = $self->ifaceType();
  my $number    = $self->ifaceNumber();
  return "$ifaceType$number";
} 


#
# Method: ifaceType
#
#   gets the iface device type used for this daemon
#
# Returns:
#    - the device type
sub ifaceType
{
  my ($self) = @_;
  return 'tap';
}

sub user
{
    my ($self) = @_;
    return $self->_openvpnModule->user();
}


sub group
{
    my ($self) = @_;
    return $self->_openvpnModule->group();
}

sub dh
{
    my ($self) = @_;
    return $self->_openvpnModule->dh();
}

#
# Method: confFile
#
#   gets the configuration file for this daemon
#
#  Parameters:
#     confDir - directory where the configuration file will be stored
#
# Returns:
#    - the path of the configuration file
sub confFile
{
    my ($self, $confDir) = @_;
    my $confFile = $self->name() . '.conf';
    my $confFilePath = defined $confDir ? "$confDir/$confFile" : $confFile;

    return $confFilePath;
}

#
# Method: writeConfFile
#
#   writes the daemon's configuration file
#
#  Parameters:
#     confDir - directory where the configuration file will be stored
#
sub writeConfFile
{
    my ($self, $confDir) = @_;

    my $confFilePath   = $self->confFile($confDir);
    my $templatePath   = $self->confFileTemplate();
    my $templateParams = $self->confFileParams();

    my $defaults     = {
	uid  => 0,
	gid  => 0,
	mode => '0400',
    };


    EBox::GConfModule->writeConfFile($confFilePath, $templatePath, $templateParams, $defaults);
}


#
# Method: confFileTemplate
#
#   Abstract method. Must return the configuration file template
#
# Returns:
#   the mason path of the configuration file template
sub confFileTemplate
{
  throw EBox::Exceptions::NotImplemented();
}


#
# Method: delete
#
#   This method is called to make the Daemon delete itself
#
#  Warning:
#    discard and do NOT use again the daemon object or any instance of it remaining after calling this method
sub delete
{
  my ($self) = @_;

  # daemon's files will not longer needed in the next restart
  $self->_openvpnModule->deleteOnRestart($self->daemonFiles);

  # delete itself..
  my $daemonDir = $self->confKeysBase();
  $self->_openvpnModule->delete_dir($daemonDir);

  # invalidate object instance
  $_[0] = undef;
}

#
# Method: daemonFiles
#
#    Gets a list with the files and directories generated by the daemon. Paths will be relative to OpenVPN's configuration directory. Directories contents are not included
#
#    This is a default implementation, specifics daemon may want to override this to include their 
#      additional files
#
# Returns: 
#  a list with each path as string
sub daemonFiles
{
  my ($self) = @_;

  my $confDir  = $self->_openvpnModule->confDir();
  my $confFile = basename $self->confFile($confDir);

  return ($confFile);
}



#
# Method: confFileTemplate
#
#   Abstract method. Must return the configuration file template
#
# Returns:
#   a reference to the parameter's list
sub confFileParams
{
  throw EBox::Exceptions::NotImplemented();
}

#
# Method: ripDaemon
#
#   Abstract method. Must return the configuration file template
#
# Returns:
#   undef if no ripDaemon is needed by the openvpn's daemon
#   if the ripDaemons is needed it must return a hash refrence with the
#   following keys:
#       iface        - the iface where the rip daemon needs to be active
#       redistribute - wether the daemon wants to redistribute routes or not
sub ripDaemon
{
  throw EBox::Exceptions::NotImplemented();
}




sub running
{
    my ($self) = @_;
    my $bin = $self->_openvpnModule->openvpnBin;
    my $name = $self->name;

    `/usr/bin/pgrep -f $bin.*$name`;

    return ($? == 0) ? 1 : 0;
}


#
# Method: summary
#
#   Abstract method. May be implemented by any daemos which wants his summary section
#
# Returns:
#     the summary data as list; the first element will be the title of the summary 
#     section and the following pairs will be usd to build EBox::Summary::Value objects
sub summary
{
  my ($self) = @_;
  return ();
}

1;
