/*
 *  ---------
 * |.**> <**.|  CardContact
 * |*       *|  Software & System Consulting
 * |*       *|  Minden, Germany
 * |**> <**|  Copyright (c) 1999. All rights reserved
 *  --------- 
 *
 * See file LICENSE for details on licensing
 *
 * Abstract :       Main API interface according to MKT specification
 *
 * Author :         Frank Thater (FTH)
 *
 * Last modified:   04/04/2000
 *
 *****************************************************************************/

#include <stdio.h>
#include <stdlib.h>

#include "defines.h"
#include "ctapi.h"
#include "sercom.h"
#include "eco5000.h"
#include "ctbcs.h"
#include "ecotools.h"



/* Handle up to 8 ECO readers */

struct eco5000_t *ecoTable[MAX_READER] = {NULL, NULL, NULL, NULL,
					  NULL, NULL, NULL, NULL};



/*
 * Locate matching card terminal number in table of active readers.
 *
 */

static int LookupReader(unsigned short ctn)

{
    int i;

    for (i = 0; i < MAX_READER; i++) {
	if (ecoTable[i] && (ecoTable[i]->ctn == ctn)) {
	     return i;
	}
    }
    return -1;
}



/*
 * Initialise the interface to the card reader attached
 * to the port number specified in <pn>
 *
 */

char LINKAGE CT_init (unsigned short ctn, unsigned short pn)

{
    int rc;
    HANDLE fh;
    char device[20];
    struct eco5000_t *ctx;

    rc = LookupReader(ctn);

    if (rc >= 0)
	return ERR_CT;

#ifdef WIN32
    sprintf(device, "COM%d", pn);
#else
    if (pn >= 64)
	strcpy(device, "/dev/smartcard");
    else
	sprintf(device, "/dev/ttyS%d", pn);
#endif

#ifdef DEBUG
    printf("Opening reader on %s\n", device);
#endif

    rc = rs232Open(&fh, device, 9600, 'E', 8, 1, 200);

    if (rc < 0) {
        return ERR_CT;        /* Could not allocate port */
    }

    for (rc = 0; (rc < MAX_READER) && ecoTable[rc]; rc++);

    if (rc == MAX_READER)
	return ERR_MEMORY;

    ctx = malloc(sizeof(struct eco5000_t));

    if (!ctx)
	return ERR_MEMORY;

    ctx->ctn = ctn;
    ctx->fh = fh;
    ctx->CTModFunc = NULL;
    ctx->RXStatus = OFF;
    ctx->Indirect = FALSE;
    ctx->Baud = 9600;
    ctx->Data.card = NULL;
    ctx->Data.t0 = NULL;
    ctx->Data.t1 = NULL;

    if (getFirmware (ctx) <= 0) {
        free(ctx);
        return ERR_CT;
    }
    
    ecoTable[rc] = ctx;

#ifdef DEBUG
    if(strcmp(ctx->Firmware, ECO_V201) == 0)
	printf("\nOld Firmware detected!\n");
    else
	printf("\nNew Firmware detected!\n");

    printf("\nECO 5000 initialized, Firmware Version %s\n\n", ctx->Firmware);
#endif
  
    return OK;
}



/*
 * Close the interface and free all allocated resources
 *
 */

char LINKAGE CT_close(unsigned short ctn)

{
    int rc;
    struct eco5000_t *ctx;
    struct memorycard_t *pt;

    rc = LookupReader(ctn);

    if (rc < 0)
        return ERR_CT;
  
    ctx = ecoTable[rc];

    if (!ctx)
        return ERR_CT;

    if (ctx->Data.card != NULL) {
      
	pt = ctx->Data.card;
      
#ifdef DEBUG
	printf("\nCleaning up handling structure ...\n");
#endif
      
	if (pt != NULL)
	    free(pt);
    }

    rs232Mode(ctx->fh, 9600, 'E', 8, 1, 100);
    rs232Close(ctx->fh);

    free(ctx);
    ecoTable[rc] = NULL;

    return OK;
}            



/*
 * Pass a command to the reader driver and receive the response
 *
 */

char LINKAGE CT_data(unsigned short ctn, unsigned char *dad, unsigned char *sad,
                     unsigned short lc, unsigned char *cmd, unsigned short *lr,
                     unsigned char *rsp )

{
    int rc;
    unsigned int ilr;
    struct eco5000_t *ctx;
 
    rc = LookupReader(ctn);

    if (rc < 0)
        return ERR_CT;
  
    ctx = ecoTable[rc];
 
    if (!ctx)
        return ERR_CT;

    ilr = (int)*lr;                     /* Overcome problem with lr size     */

    rc = 0;
    if (*dad == 1) {
        *sad = 1;                       /* Source Reader    */
        *dad = 2;                       /* Destination Host */

        /*******************/
        /* CT-BCS Commands */
        /*******************/

        if(cmd[0] == 0x20)  {
            switch(cmd[1])  {
                case 0x12:              /* Request ICC                       */
                    rc = RequestICC(ctx, lc, cmd, &ilr, rsp);
                    break; 

                case 0x11:              /* Resets the Card/Terminal returns ATR */
                    if (cmd[2]) {
                        rc = ResetCard(ctx, lc, cmd, &ilr, rsp);
                    } else {
                        rc = ResetTerminal(ctx, &ilr, rsp);
                    }
                    break;

                case 0x13:              /* Get Status - Gets reader status   */
                    rc = GetStatus(ctx, cmd, &ilr, rsp);
                    break;

                case 0x15:              /* Eject ICC - Deactivates Reader    */
                    rc = EjectICC(ctx, lc, cmd, &ilr, rsp);
                    break;
            
                default:   /* Wrong instruction for CTAPI */
                    rsp[0] = HIGH(WRONG_INSTRUCTION);
                    rsp[1] = LOW(WRONG_INSTRUCTION);
                    ilr = 2;
            }
        }
        else   /* Wrong class for CTAPI */  
        {
	    rsp[0] = HIGH(CLASS_NOT_SUPPORTED);
	    rsp[1] = LOW(CLASS_NOT_SUPPORTED);
	    ilr = 2;
        }
    } else if (*dad == 0) {             /* This command goes to the card     */
    
        // Don't get confused here this is for the return saying
        // the source was the card and the destination the host

        *sad = 0;  /* Source Smartcard */
        *dad = 2;  /* Destination Host */

        if (*ctx->CTModFunc)
            rc =  (*ctx->CTModFunc)(ctx, lc, cmd, &ilr, rsp);
        else {
          
            *sad = 1;   /* Shows that response comes from CTAPI */
            *dad = 2;
            rc = 0;
            rsp[0] = HIGH(COMMUNICATION_NOT_POSSIBLE);
            rsp[1] = LOW(COMMUNICATION_NOT_POSSIBLE);
            ilr = 2;
        }
    } else {
        rc = ERR_INVALID;              /* Invalid SAD/DAD Address */
        ilr = 0;
    }

    *lr = ilr;
    return rc;
}










