/*============================================================================
 *  Dfinitions des fonctions
 *   associes  la structure `ecs_champ_t' dcrivant un champ
 *   et propres aux champs auxiliaires de type "attribut"
 *============================================================================*/

/*
  This file is part of the Code_Saturne Preprocessor, element of the
  Code_Saturne CFD tool.

  Copyright (C) 1999-2007 EDF S.A., France

  contact: saturne-support@edf.fr

  The Code_Saturne Preprocessor is free software; you can redistribute it
  and/or modify it under the terms of the GNU General Public License
  as published by the Free Software Foundation; either version 2 of
  the License, or (at your option) any later version.

  The Code_Saturne Preprocessor is distributed in the hope that it will be
  useful, but WITHOUT ANY WARRANTY; without even the implied warranty
  of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.

  You should have received a copy of the GNU General Public License
  along with the Code_Saturne Preprocessor; if not, write to the
  Free Software Foundation, Inc.,
  51 Franklin St, Fifth Floor,
  Boston, MA  02110-1301  USA
*/


/*============================================================================
 *                                 Visibilit
 *============================================================================*/

/*----------------------------------------------------------------------------
 *  Fichiers `include' librairie standard C
 *----------------------------------------------------------------------------*/

#include <assert.h>
#include <string.h> /* strlen() */


/*----------------------------------------------------------------------------
 *  Fichiers `include' systme ou BFT
 *----------------------------------------------------------------------------*/

#include <bft_mem.h>


/*----------------------------------------------------------------------------
 *  Fichiers `include' visibles du  paquetage global "Utilitaire"
 *----------------------------------------------------------------------------*/

#include "ecs_chaine_glob.h"
#include "ecs_def.h"
#include "ecs_tab.h"


/*----------------------------------------------------------------------------
 *  Fichiers `include' visibles des paquetages visibles
 *----------------------------------------------------------------------------*/

#include "ecs_param_perio_glob.h"
#include "ecs_vec_def_perio.h"
#include "ecs_vec_int.h"
#include "ecs_vec_int_att.h"


/*----------------------------------------------------------------------------
 *  Fichiers `include' visibles du  paquetage courant
 *----------------------------------------------------------------------------*/

#include "ecs_champ.h"
#include "ecs_champ_vec_int.h"
#include "ecs_champ_vec_real.h"
#include "ecs_descr_chaine.h"
#include "ecs_descr.h"
#include "ecs_famille_chaine.h"


/*----------------------------------------------------------------------------
 *  Fichier  `include' du  paquetage courant associ au fichier courant
 *----------------------------------------------------------------------------*/

#include "ecs_champ_att.h"


/*----------------------------------------------------------------------------
 *  Fichiers `include' prives   du  paquetage courant
 *----------------------------------------------------------------------------*/

#include "ecs_champ_priv.h"


/*============================================================================
 *                       Prototypes de fonctions prives
 *============================================================================*/

/*----------------------------------------------------------------------------
 *  Fonction qui cre une liste d'lments ayant pour valeur de l'attribut
 *   la valeur donne en argument
 *----------------------------------------------------------------------------*/

static ecs_tab_int_t ecs_loc_champ_att__cree_liste
(
 ecs_champ_t *const champ_att,
 ecs_int_t          valeur_att
) ;


/*----------------------------------------------------------------------------
 *  Fonction qui renvoie, parmi les descripteurs du champ donn,
 *   le numro de descripteur correspondant
 *    la liste d'entiers,  la liste de rels et  la chane de caractres
 *   donnes
 *
 *  Si aucun descripteur correspondant n'a t trouve,
 *   la fonction renvoie DESCR_NUM_NUL
 *----------------------------------------------------------------------------*/

static int ecs_loc_champ_att__ret_numdescr
(
 const ecs_champ_t *const champ,
 ECS_DESCR_TYP_E          descr_typ_e,
       int                descr_ide,
 const char        *const descr_nom
) ;


/*============================================================================
 *                             Fonctions publiques
 *============================================================================*/

/*----------------------------------------------------------------------------
 *  Fonction qui assemble un champ donn dans un champ rcepteur donn
 *
 *  Si les champs  assembler ont le mme nom, le nom donn au champ rcepteur
 *   sera ce mme nom, et on passera en argument NULL au nom  donner
 *  Sinon, on donnera en argument le nouveau nom du champ rcepteur
 *
 *  L'assemblage consiste  :
 *  - regrouper sous la mme position les valeurs des 2 champs
 *    (cela suppose donc que les 2 champs ont le mme nombre de positions)
 *  - assembler les membres des descripteurs des 2 champs
 *    Les descripteurs des 2 champs peuvent tre  `NULL'
 *    et si le descripteur du champ rcepteur est  `NULL',
 *          le descripteur du champ assembl est celui du champ  assembler
 *
 *  Le champ  assembler est dtruit apres assemblage
 *----------------------------------------------------------------------------*/

void ecs_champ_att__assemble
(
       ecs_champ_t    *const champ_recept,
       ecs_champ_t    *      champ_assemb,
 const char           *      nom
)
{

  ecs_vec_int_t  *vec_recept ;
  ecs_vec_int_t  *vec_assemb ;

  ecs_descr_t * descr_tete_loc ;


  /*xxxxxxxxxxxxxxxxxxxxxxxxxxx Instructions xxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxx*/


  assert(champ_recept != NULL) ;
  assert(champ_assemb != NULL) ;


  if (nom != NULL) {

    BFT_FREE(champ_recept->nom) ;

    BFT_MALLOC(champ_recept->nom, strlen(nom) + 1, char) ;
    strcpy(champ_recept->nom, nom) ;

  }
  else {

    assert(strcmp(champ_recept->nom, champ_assemb->nom) == 0) ;

  }


  /* Assemblage des valeurs */

  vec_recept = ecs_champ__initialise_vec_int(champ_recept) ;
  vec_assemb = ecs_champ__initialise_vec_int(champ_assemb) ;

  ecs_vec_int_att__assemble(vec_recept,
                            vec_assemb) ;


  ecs_champ__libere_vec_int(champ_assemb,
                            vec_assemb) ;

  ecs_champ__transfere_vec_int(champ_recept,
                               vec_recept) ;


  /* Ajout des descripteurs du champ  assembler au champ rcepteur */

  descr_tete_loc = ecs_descr_chaine__copie(champ_assemb->descr) ;

  ecs_descr_chaine__ajoute(&champ_recept->descr,
                           descr_tete_loc) ;


  ecs_champ__detruit(champ_assemb) ;

}


/*----------------------------------------------------------------------------
 *  Fonction ralisant l'hritage des valeurs d'un attribut d'un lment
 *   sur les sous-lments issus de sa dcomposition
 *----------------------------------------------------------------------------*/

ecs_champ_t * ecs_champ_att__herite
(
 ecs_champ_t      *const champ_att_elt,
 ecs_champ_t      *const champ_elt_def_sselt,
 size_t                  nbr_sselt
)
{

  ecs_champ_t *this_champ_att_sselt ;


  ecs_vec_int_t  *vec_att_elt ;       /* Un attribut */
  ecs_vec_int_t  *vec_att_sselt ;
  ecs_vec_int_t  *vec_elt_def_sselt ;

  ecs_tab_int_t   vect_transf_num_descr ;


  /*xxxxxxxxxxxxxxxxxxxxxxxxxxx Instructions xxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxx*/


  vec_elt_def_sselt = ecs_champ__initialise_vec_int(champ_elt_def_sselt) ;


  /* Construction, pour les lments,                      */
  /*  du vecteur `ecs_vec_int_t' associ au champ attribut */
  /*-------------------------------------------------------*/

  vec_att_elt = ecs_champ__initialise_vec_int(champ_att_elt) ;


  /* Allocation et initialisation pour les sous-lments, */
  /*  des vecteurs `ecs_vec_int_t' associ au champ attribut  */
  /*------------------------------------------------------*/

  vec_att_sselt = ecs_vec_int__herite(vec_att_elt,
                                      vec_elt_def_sselt,
                                      nbr_sselt) ;


  ecs_champ__libere_vec_int(champ_elt_def_sselt,
                            vec_elt_def_sselt) ;


  if (vec_att_sselt != NULL) {


    if (champ_att_elt->descr != NULL) {

      const ecs_int_t nbr_descr_old
        = ecs_descr_chaine__ret_nbr(champ_att_elt->descr) ;

      /* Renumrotation des numros de descripteur */
      /*-------------------------------------------*/

      vect_transf_num_descr = ecs_vec_int_att__renum_descr(vec_att_sselt,
                                                           ECS_DESCR_NUM_NUL,
                                                           nbr_descr_old) ;

    }


    /* Allocation et initialisation du champ "attribut" */
    /*  de l'entit de maillage des sous-lments       */
    /*--------------------------------------------------*/

    this_champ_att_sselt = ecs_champ__init_avec_vec_int(vec_att_sselt,
                                                        champ_att_elt->nom) ;

    this_champ_att_sselt->statut_e  = champ_att_elt->statut_e ;

   if (champ_att_elt->descr != NULL) {

     /* Renumrotation des numros de descripteur */
     /*-------------------------------------------*/

     this_champ_att_sselt->descr
       = ecs_descr_chaine__renumerote(champ_att_elt->descr,
                                      vect_transf_num_descr ) ;

     BFT_FREE(vect_transf_num_descr.val) ;

   }

  }
  else {

    this_champ_att_sselt = NULL ;

  }


  ecs_champ__libere_vec_int(champ_att_elt,
                            vec_att_elt) ;



  return this_champ_att_sselt ;


}


/*----------------------------------------------------------------------------
 *  Fonction ralisant la transformation d'un champ
 *   en fusionnant les proprits de ses lments
 *   qui sont identiquement transformes par le vecteur de transformation donn
 *----------------------------------------------------------------------------*/

void ecs_champ_att__fusionne
(
       ecs_champ_t    *const this_champ_att,
       size_t                nbr_elt_new,
 const ecs_tab_int_t         vect_transf
)
{

  ecs_vec_int_t *vec_int_att ;


  /*xxxxxxxxxxxxxxxxxxxxxxxxxxx Instructions xxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxx*/


  vec_int_att = ecs_champ__initialise_vec_int(this_champ_att) ;


  ecs_vec_int_att__fusionne(vec_int_att,
                            nbr_elt_new,
                            vect_transf ) ;

  ecs_champ__transfere_vec_int(this_champ_att,
                               vec_int_att) ;


}


/*----------------------------------------------------------------------------
 *  Fonction qui renvoie la liste des numros de famille des lments
 *
 *  Pour les lments de famille 0 ou n'ayant pas de famille, on leur
 *   attribue le numro de famille par dfaut
 *----------------------------------------------------------------------------*/

ecs_tab_int_t ecs_champ_att__fam_elt
(
 ecs_champ_t       *const this_champ_fam,
 ecs_tab_int_t     *const tab_nbr_elt_fam
)
{

  ecs_tab_int_t  tab_fam_liste_elt ;

  ecs_vec_int_t *vec_int_fam ;


  /*xxxxxxxxxxxxxxxxxxxxxxxxxxx Instructions xxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxx*/


  vec_int_fam = ecs_champ__initialise_vec_int(this_champ_fam) ;


  tab_fam_liste_elt = ecs_vec_int_att__fam_elt(vec_int_fam,
                                               tab_nbr_elt_fam) ;

  ecs_champ__transfere_vec_int(this_champ_fam,
                               vec_int_fam) ;


  return tab_fam_liste_elt ;


}


/*----------------------------------------------------------------------------
 * Fonction qui construit la liste des lments  portant des numros de
 *  familles marqus dans le tableau indic_famille
 *----------------------------------------------------------------------------*/

ecs_tab_int_t  ecs_champ_att__liste_elt_fam
(
       ecs_champ_t     *const champ_famille,
 const ecs_tab_int_t   *const liste_filtre,
 const ecs_tab_bool_t  *const indic_famille
)
{
  ecs_tab_int_t   liste_elt_select ;

  ecs_vec_int_t * vec_int_famille ;


  /*xxxxxxxxxxxxxxxxxxxxxxxxxxx Instructions xxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxx*/


  assert(champ_famille != NULL) ;


  vec_int_famille = ecs_champ__initialise_vec_int(champ_famille) ;

  liste_elt_select = ecs_vec_int_att__liste_elt_fam(vec_int_famille,
                                                    liste_filtre,
                                                    indic_famille) ;

  ecs_champ__libere_vec_int(champ_famille,
                            vec_int_famille) ;


  return liste_elt_select ;

}


/*----------------------------------------------------------------------------
 *  Fonction qui renvoie un tableau donnant pour chaque valeur du champ
 *   le nombre d'lments ayant cette valeur
 *----------------------------------------------------------------------------*/

ecs_int_t * ecs_champ_att__ret_nbr_elt_val
(
 ecs_champ_t      *const champ_att,
 size_t                  nbr_val_att
)
{

  ecs_int_t   * cpt_elt_par_val ;

  ecs_vec_int_t * vec_int_att ;


  /*xxxxxxxxxxxxxxxxxxxxxxxxxxx Instructions xxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxx*/


  assert(champ_att != NULL) ;


  vec_int_att = ecs_champ__initialise_vec_int(champ_att) ;

  cpt_elt_par_val = ecs_vec_int_att__cpt_elt_val(vec_int_att,
                                                 nbr_val_att) ;

  ecs_champ__libere_vec_int(champ_att,
                            vec_int_att) ;


  return cpt_elt_par_val ;


}


/*----------------------------------------------------------------------------
 *  Fonction qui construit un nouveau champ attribut  partir
 *   d'un champ donn qui contient la dfinition d'lments
 *                                 en fonction de nouveaux lments
 *  L'attribut cre donne la rfrence de l'lment d'origine
 *   auquel est associ le nouvel lment
 *----------------------------------------------------------------------------*/

ecs_champ_t * ecs_champ_att__filialise
(
 ecs_champ_t  *const champ_def,
 size_t              nbr_sselt
)
{

  ecs_champ_t * champ_att ;


  /*xxxxxxxxxxxxxxxxxxxxxxxxxxx Instructions xxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxx*/


  champ_att = ecs_champ__inverse(champ_def,
                                 nbr_sselt) ;

  BFT_FREE(champ_att->nom) ;
  BFT_MALLOC(champ_att->nom, strlen(ECS_CHAMP_NOM_FILIATION) + 1, char) ;
  strcpy(champ_att->nom, ECS_CHAMP_NOM_FILIATION) ;

  champ_att->descr = NULL ;

  champ_att->l_champ_sui = NULL ;

  champ_att->statut_e  = ECS_CHAMP_STATUT_HERITABLE ;


  return champ_att ;


}


/*----------------------------------------------------------------------------
 *  Fonction qui remplit un tableau de boolens
 *   Le boolen correspondant  un lment du champ est  `ECS_TRUE'
 *    si l'lment est slectionn par la description donne :
 *    - le numro du descripteur
 *    - le nom    du descripteur
 *----------------------------------------------------------------------------*/

ecs_int_t ecs_champ_att__cree_masque
(
       ecs_tab_bool_t  *const bool_elt_select,
 const ecs_tab_int_t   *const liste_filtre,
       ecs_champ_t     *const champ_select,
 const ecs_int_t              descr_ide,
 const char            *const descr_nom
)
{

  int               indice_att ;
  size_t            nbr_elt_select ;

  size_t            ielt ;

  ECS_DESCR_TYP_E descr_typ_e ;

  ecs_tab_int_t     liste_elt = {0, NULL} ;


  /*xxxxxxxxxxxxxxxxxxxxxxxxxxx Instructions xxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxx*/


  if      (strcmp(champ_select->nom, ECS_CHAMP_NOM_COULEUR) == 0)
    descr_typ_e = ECS_DESCR_TYP_COULEUR ;
  else if (strcmp(champ_select->nom, ECS_CHAMP_NOM_GROUPE ) == 0)
    descr_typ_e = ECS_DESCR_TYP_GROUPE ;
  else
    assert(strcmp(champ_select->nom, ECS_CHAMP_NOM_COULEUR) == 0 ||
           strcmp(champ_select->nom, ECS_CHAMP_NOM_GROUPE ) == 0   ) ;


  indice_att = ecs_loc_champ_att__ret_numdescr(champ_select,
                                               descr_typ_e,
                                               descr_ide,
                                               descr_nom) ;

  if (indice_att != ECS_DESCR_NUM_NUL) {

    liste_elt = ecs_loc_champ_att__cree_liste(champ_select,
                                              indice_att) ;

  }


  nbr_elt_select = 0 ;

  if (liste_elt.nbr != 0) {

    if (liste_filtre == NULL) {

      for (ielt = 0 ; ielt < liste_elt.nbr ; ielt++)
        bool_elt_select->val[liste_elt.val[ielt] - 1] = ECS_TRUE ;

      nbr_elt_select += liste_elt.nbr ;

    }
    else {

      ecs_int_t ind_val = 0 ;
      size_t ind_filtre = 0 ;

      for (ielt = 0 ; ielt < liste_elt.nbr ; ielt++) {

        ind_val = liste_elt.val[ielt] - 1 ;

        while (   ind_filtre < liste_filtre->nbr - 1
               && liste_filtre->val[ind_filtre] < ind_val)
          ind_filtre++ ;

        if (liste_filtre->val[ind_filtre] == ind_val) {

          bool_elt_select->val[ind_val] = ECS_TRUE ;

          nbr_elt_select += 1 ;

        }

      }

    }

    BFT_FREE(liste_elt.val) ;

  }
  /* else : rien  faire */


  return nbr_elt_select ;


}


/*----------------------------------------------------------------------------
 *  Fonction qui construit les familles  partir
 *   de la liste chane de tous les champs de type "attribut"
 *   pour toutes des entits ; ces familles sont ajoutes  la liste
 *   chane fournie en argument ;
 *
 *  Elle renvoie les champs "famille" par entit
 *
 *  Elle dtermine aussi :
 *   - le nombre de familles
 *   - les nombres d'lments par famille
 *----------------------------------------------------------------------------*/

ecs_champ_t * ecs_champ_att__construit_fam
(
 ecs_champ_t        *      champ_att_tete,
 ecs_famille_t    * *const vect_fam_tete,
 size_t                    nbr_elt_ent,
 int                       num_fam_deb,
 int                *const nbr_fam,
 ecs_int_t        * *const cpt_elt_fam
)
{

  ecs_int_t         ifam ;
  ecs_int_t       * nbr_descr_fam ;
  ecs_int_t       * cpt_elt_ent_fam ;
  ecs_int_t     * * def_fam_descr ;

  ecs_champ_t       * champ_att_unifie ;
  ecs_champ_t       * ptr_champ_att ;

  ecs_tab_int_t       tab_idem ;
  ecs_tab_int_t       tab_renum_elt ;

  ecs_vec_int_t     * vec_int_att_unifie ;
  ecs_vec_int_t     * vec_int_fam ;

  ecs_champ_t       * champ_fam ;

  /*xxxxxxxxxxxxxxxxxxxxxxxxxxx Instructions xxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxx*/


  assert(champ_att_tete != NULL) ;


  champ_att_unifie = ecs_champ__copie(champ_att_tete) ;

  BFT_FREE(champ_att_unifie->nom) ;
  champ_att_unifie->l_champ_sui = NULL ;


  for (ptr_champ_att  = champ_att_tete->l_champ_sui ;
       ptr_champ_att != NULL ;
       ptr_champ_att  = ptr_champ_att->l_champ_sui) {


    /* Concatnation du champ "attribut" traite */
    /*  au champ "attribut" unificateur         */
    /*------------------------------------------*/

    ecs_champ__concatene(champ_att_unifie,
                         ptr_champ_att) ;

  }


  /* Tri des valeurs du champ unifie */
  /*---------------------------------*/

  vec_int_att_unifie = ecs_champ__initialise_vec_int(champ_att_unifie) ;


  tab_renum_elt = ecs_vec_int__trie_val(&vec_int_att_unifie) ;


  /* Tous les lments ayant des listes de numros de descripteurs d'attribut */
  /*  identiques appartiennent  la mme famille                              */
  /*  (famille dfinie par la liste de descripteurs d'attribut)               */


  tab_idem = ecs_vec_int__compare_val_pos(vec_int_att_unifie) ;


  vec_int_fam = ecs_vec_int__attribue_fam(vec_int_att_unifie,
                                          tab_idem,
                                          &def_fam_descr,
                                          &nbr_descr_fam,
                                          num_fam_deb,
                                          nbr_fam) ;


  BFT_FREE(tab_idem.val) ;


  ecs_champ__transfere_vec_int(champ_att_unifie,
                               vec_int_att_unifie) ;


  /* Retour  la numrotation initiale des lments pour les familles */

  ecs_vec_int__transforme_pos(vec_int_fam,
                              tab_renum_elt.nbr,
                              tab_renum_elt) ;


  BFT_FREE(tab_renum_elt.val) ;


  /*-------------------------------------------------------------------------*/
  /* Construction des familles  partir :                                    */
  /*  - du vecteur indexe donnant pour chaque numro de famille              */
  /*     la liste des numros d'attributs                                    */
  /*  - de la liste chane de l'ensemble des descripteurs d'attribut        */
  /*-------------------------------------------------------------------------*/

  *vect_fam_tete = ecs_famille_chaine__cree(def_fam_descr,
                                            nbr_descr_fam,
                                            num_fam_deb,
                                            *nbr_fam,
                                            champ_att_unifie->descr) ;


  ecs_champ__detruit(champ_att_unifie) ;


  /*-----------------------------------*/
  /* Construction des champs "famille" */
  /*-----------------------------------*/

  vec_int_fam = ecs_vec_int__prolonge_val_pos(vec_int_fam,
                                              0,
                                              nbr_elt_ent) ;


  /* Comptabilisation des nombres d'lments par numro de famille */
  /*---------------------------------------------------------------*/

  BFT_MALLOC(*cpt_elt_fam, *nbr_fam + num_fam_deb - 1, ecs_int_t) ;

  cpt_elt_ent_fam = ecs_vec_int_att__cpt_elt_val(vec_int_fam,
                                                 *nbr_fam + num_fam_deb - 1) ;

  for (ifam = 0 ; ifam < *nbr_fam + num_fam_deb - 1 ; ifam++)
    ((*cpt_elt_fam)[ifam]) = cpt_elt_ent_fam[ifam] ;


  BFT_FREE(cpt_elt_ent_fam) ;

  for (ifam = 0 ; ifam < *nbr_fam ; ifam++)
    BFT_FREE(def_fam_descr[ifam]) ;
  BFT_FREE(def_fam_descr) ;
  BFT_FREE(nbr_descr_fam) ;


  if (vec_int_fam != NULL) {

    champ_fam = ecs_champ__init_avec_vec_int(vec_int_fam,
                                             ECS_CHAMP_NOM_FAMILLE) ;

    champ_fam->statut_e  = ECS_CHAMP_STATUT_HERITABLE ;

  }
  else {

    champ_fam = NULL ;

  }


  return champ_fam ;

}


/*----------------------------------------------------------------------------
 *  Fonction qui cre les champs "couleur" et "groupe"  partir
 *   du champ "famille" et de la liste chane des familles
 *----------------------------------------------------------------------------*/

void ecs_champ_att__cree_att_fam
(
 ecs_champ_t      *const champ_famille,
 ecs_famille_t    *const famille,
 ecs_champ_t    * *const champ_couleur,
 ecs_champ_t    * *const champ_groupe
)
{

  size_t        ifam ;
  int           num_fam_max ;
  ecs_int_t     nbr_couleur ;
  ecs_int_t     nbr_groupe ;
  ecs_int_t     nbr_max_att_fam ;

  ecs_descr_t   * descr_tete_couleur ;
  ecs_descr_t   * descr_tete_groupe ;

  ecs_tab_int_t   tab_fam ;
  ecs_tab_int_t * tab_couleur_fam ;
  ecs_tab_int_t * tab_groupe_fam ;

  ecs_vec_int_t * vec_couleur ;
  ecs_vec_int_t * vec_groupe ;
  ecs_vec_int_t * vec_fam_couleur ;
  ecs_vec_int_t * vec_fam_groupe ;
  ecs_vec_int_t * vec_famille ;

  /*xxxxxxxxxxxxxxxxxxxxxxxxxxx Instructions xxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxx*/


  assert(champ_famille != NULL) ;
  assert(famille       != NULL) ;


  /* Cration de la liste des numros de famille rfrencs */
  /*  dans le champ "famille"                               */
  /*--------------------------------------------------------*/

  tab_fam = ecs_champ__ret_reference(champ_famille) ;

  num_fam_max = 0 ;
  for (ifam = 0 ; ifam < tab_fam.nbr ; ifam++) {
    if (tab_fam.val[ifam] > num_fam_max)
      num_fam_max = tab_fam.val[ifam] ;
  }

  BFT_MALLOC(tab_couleur_fam, num_fam_max, ecs_tab_int_t) ;
  BFT_MALLOC(tab_groupe_fam , num_fam_max, ecs_tab_int_t) ;

  for (ifam = 0 ; ifam < (size_t)num_fam_max ; ifam++) {
    tab_couleur_fam[ifam].val = NULL ;
    tab_couleur_fam[ifam].nbr = 0 ;
    tab_groupe_fam [ifam].val = NULL ;
    tab_groupe_fam [ifam].nbr = 0 ;
  }


  /* Cration des 2 listes chanes de descripteurs */
  /*  pour les champs "couleur" et "groupe"         */
  /*------------------------------------------------*/

  ecs_famille_chaine__cree_descr(famille,
                                 tab_fam,
                                 &descr_tete_couleur,
                                 &descr_tete_groupe,
                                 tab_couleur_fam,
                                 tab_groupe_fam,
                                 &nbr_max_att_fam) ;

  if (tab_fam.nbr != 0)
    BFT_FREE(tab_fam.val) ;


  vec_famille = ecs_champ__initialise_vec_int(champ_famille) ;


  nbr_couleur = 0 ;
  for (ifam = 0 ; ifam < (size_t)num_fam_max ; ifam++)
    nbr_couleur += tab_couleur_fam[ifam].nbr ;

  if (nbr_couleur != 0) {

    /* Cration du champ "couleur" */
    /*-----------------------------*/

    vec_fam_couleur = ecs_vec_int__transforme_bi_tab(tab_couleur_fam,
                                                     num_fam_max,
                                                     nbr_couleur) ;

    vec_couleur = ecs_vec_int__ret_remplace_ref_att(vec_famille,
                                                    vec_fam_couleur) ;

    ecs_vec_int__detruit(vec_fam_couleur) ;


    *champ_couleur = ecs_champ__init_avec_vec_int(vec_couleur,
                                                  ECS_CHAMP_NOM_COULEUR) ;

    (*champ_couleur)->descr = descr_tete_couleur ;
    (*champ_couleur)->statut_e = ECS_CHAMP_STATUT_HERITABLE ;


    for (ifam = 0 ; ifam < (size_t)num_fam_max ; ifam++)
      if (tab_couleur_fam[ifam].val != NULL)
        BFT_FREE(tab_couleur_fam[ifam].val) ;

  }
  else {

    (*champ_couleur) = NULL ;

  }


  nbr_groupe = 0 ;
  for (ifam = 0 ; ifam < (size_t)num_fam_max ; ifam++)
    nbr_groupe += tab_groupe_fam[ifam].nbr ;

  if (nbr_groupe != 0) {

    /* Cration du champ "groupe" */
    /*----------------------------*/

    vec_fam_groupe  = ecs_vec_int__transforme_bi_tab(tab_groupe_fam,
                                                     num_fam_max,
                                                     nbr_groupe) ;

    vec_groupe = ecs_vec_int__ret_remplace_ref(vec_famille,
                                               vec_fam_groupe) ;

    ecs_vec_int__detruit(vec_fam_groupe) ;


    *champ_groupe = ecs_champ__init_avec_vec_int(vec_groupe,
                                                 ECS_CHAMP_NOM_GROUPE) ;

    (*champ_groupe)->descr = descr_tete_groupe ;
    (*champ_groupe)->statut_e = ECS_CHAMP_STATUT_HERITABLE ;


    for (ifam = 0 ; ifam < (size_t)num_fam_max ; ifam++)
      if (tab_groupe_fam[ifam].val != NULL)
        BFT_FREE(tab_groupe_fam[ifam].val) ;

  }
  else {

    (*champ_groupe) = NULL ;

  }


  BFT_FREE(tab_couleur_fam) ;
  BFT_FREE(tab_groupe_fam) ;

  ecs_champ__libere_vec_int(champ_famille,
                            vec_famille) ;


}


/*----------------------------------------------------------------------------
 *  Fonction qui met  jour la renumrotation
 *  "lments initiaux -> lments priodiques" en fonction d'un champ
 *  "lments prcdents -> nouveaux lments" en cas de modifications
 *  successives de la dfinition des lments.
 *  Valable pour les champs attributs de statut ECS_CHAMP_STATUT_REF_ELT
 *  On modifie  la fois 'val_tab' et on dcale 'pos_tab'.
 *----------------------------------------------------------------------------*/

void ecs_champ_att__renum_perio
(
 ecs_champ_t  *const champ_elt_perio,
 ecs_champ_t  *const champ_elt_old_new,
 size_t              nbr_elt_new
)
{

  ecs_vec_int_t  * vec_elt_perio ;
  ecs_vec_int_t  * vec_elt_old_new ;


  /*xxxxxxxxxxxxxxxxxxxxxxxxxxx Instructions xxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxx*/


  assert(champ_elt_perio != NULL) ;
  assert(champ_elt_old_new != NULL) ;


  vec_elt_perio = ecs_champ__initialise_vec_int(champ_elt_perio) ;
  vec_elt_old_new = ecs_champ__initialise_vec_int(champ_elt_old_new) ;


  ecs_vec_def__maj_elt_perio(vec_elt_perio,
                             vec_elt_old_new,
                             nbr_elt_new) ;


  ecs_champ__libere_vec_int(champ_elt_old_new,
                            vec_elt_old_new) ;

  ecs_champ__transfere_vec_int(champ_elt_perio,
                               vec_elt_perio) ;


}


/*----------------------------------------------------------------------------
 *  Fonction qui met  jour le tableau renvoy par
 * ecs_champ_def__typ_fac_cel() en fonction d'une connectivits ajoute par
 * une priodicit.
 *----------------------------------------------------------------------------*/

void ecs_champ_att__typ_fac_perio
(
 ecs_champ_t    * champ_fac_perio,
 ecs_tab_int_t  * typ_fac
)
{

  ecs_vec_int_t  * vec_fac_perio ;


  /*xxxxxxxxxxxxxxxxxxxxxxxxxxx Instructions xxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxx*/


  assert(champ_fac_perio != NULL) ;
  assert(typ_fac != NULL) ;


  vec_fac_perio = ecs_champ__initialise_vec_int(champ_fac_perio) ;


  ecs_vec_def__typ_fac_perio(vec_fac_perio,
                             typ_fac) ;


  ecs_champ__libere_vec_int(champ_fac_perio,
                            vec_fac_perio) ;

}


/*============================================================================
 *                              Fonctions prives
 *============================================================================*/

/*----------------------------------------------------------------------------
 *  Fonction qui cree une liste d'lments ayant pour valeur de l'attribut
 *   la valeur donne en argument
 *----------------------------------------------------------------------------*/

static ecs_tab_int_t ecs_loc_champ_att__cree_liste
(
 ecs_champ_t *const champ_att,
 ecs_int_t          indice_att
)
{

  ecs_vec_int_t * vec_int_att ;

  ecs_tab_int_t   liste_elt = {0, NULL} ;


  /*xxxxxxxxxxxxxxxxxxxxxxxxxxx Instructions xxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxx*/


  vec_int_att = ecs_champ__initialise_vec_int(champ_att) ;

  liste_elt = ecs_vec_int_att__cree_liste(vec_int_att,
                                          indice_att) ;

  ecs_champ__libere_vec_int(champ_att,
                            vec_int_att) ;


  return liste_elt ;


}


/*----------------------------------------------------------------------------
 *  Fonction qui renvoie, parmi les descripteurs du champ donn,
 *   le numro de descripteur correspondant
 *    l'identificateur et au nom donns
 *
 *  Si aucun descripteur correspondant n'a t trouv,
 *   la fonction renvoie ECS_DESCR_NUM_NUL
 *----------------------------------------------------------------------------*/

static int ecs_loc_champ_att__ret_numdescr
(
 const ecs_champ_t   *const champ,
 ECS_DESCR_TYP_E            descr_typ_e,
       int                  descr_ide,
 const char          *const descr_nom
)
{

  ecs_int_t     num_descr ;

  ecs_descr_t   * descr_rech ;


  /*xxxxxxxxxxxxxxxxxxxxxxxxxxx Instructions xxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxx*/


  descr_rech = ecs_descr__cree(descr_typ_e,
                               descr_ide,
                               descr_nom) ;

  num_descr = ecs_descr_chaine__trouve_num(champ->descr,
                                           descr_rech) ;

  ecs_descr__detruit(descr_rech) ;


  return num_descr ;

}

