/*============================================================================
 *  Dfinitions des fonctions de base
 *   associes  une liste chane de structures `ecs_famille_t' dcrivant
 *   une famille
 *============================================================================*/

/*
  This file is part of the Code_Saturne Preprocessor, element of the
  Code_Saturne CFD tool.

  Copyright (C) 1999-2007 EDF S.A., France

  contact: saturne-support@edf.fr

  The Code_Saturne Preprocessor is free software; you can redistribute it
  and/or modify it under the terms of the GNU General Public License
  as published by the Free Software Foundation; either version 2 of
  the License, or (at your option) any later version.

  The Code_Saturne Preprocessor is distributed in the hope that it will be
  useful, but WITHOUT ANY WARRANTY; without even the implied warranty
  of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.

  You should have received a copy of the GNU General Public License
  along with the Code_Saturne Preprocessor; if not, write to the
  Free Software Foundation, Inc.,
  51 Franklin St, Fifth Floor,
  Boston, MA  02110-1301  USA
*/


/*============================================================================
 *                                 Visibilit
 *============================================================================*/

/*----------------------------------------------------------------------------
 *  Fichiers `include' librairie standard C ou BFT
 *----------------------------------------------------------------------------*/

#include <assert.h>
#include <string.h> /* strlen() */

#include <bft_file.h>
#include <bft_mem.h>
#include <bft_printf.h>


/*----------------------------------------------------------------------------
 *  Fichiers `include' visibles du  paquetage global "Utilitaire"
 *----------------------------------------------------------------------------*/

#include "ecs_def.h"
#include "ecs_fic.h"
#include "ecs_tab.h"


/*----------------------------------------------------------------------------
 *  Fichiers `include' visibles des paquetages visibles
 *----------------------------------------------------------------------------*/

#include "ecs_descr.h"
#include "ecs_descr_chaine.h"


/*----------------------------------------------------------------------------
 *  Fichiers `include' visibles du  paquetage courant
 *----------------------------------------------------------------------------*/

#include "ecs_famille.h"


/*----------------------------------------------------------------------------
 *  Fichier  `include' du  paquetage courant associe au fichier courant
 *----------------------------------------------------------------------------*/

#include "ecs_famille_chaine.h"


/*----------------------------------------------------------------------------
 *  Fichiers `include' prives   du  paquetage courant
 *----------------------------------------------------------------------------*/

#include "ecs_famille_priv.h"


/*============================================================================
 *                       Prototypes de fonctions prives
 *============================================================================*/

/*----------------------------------------------------------------------------
 *  Fonction qui dtermine la liste des attributs rfrencs par une
 *   liste chane de familles dont la tte est donne, ainsi que
 *   la liste des familles correspondant  chaque attribut
 *
 *  Le tableau optionnel nbr_elt_fam permet de limiter ventuellement
 *   la liste des attributs  ceux qu sont effectivement rfrencs
 *----------------------------------------------------------------------------*/

static void ecs_loc_famille_chaine__fam_att
(
 const ecs_famille_t      * fam_tete,
 const ecs_int_t          * nbr_elt_fam,
       ecs_tab_int_t      * tab_descr,
       ecs_tab_int_t    * * tab_fam_descr,
       ecs_descr_t    * * * liste_ref_descr
) ;


/*============================================================================
 *                             Fonctions publiques
 *============================================================================*/


/*----------------------------------------------------------------------------
 *  Fonction qui cre une liste chane de familles  partir :
 *   - des dfinitions de chaque famille en fonction des numros de descripteur
 *   - de la liste chane des descripteurs
 *  La fonction renvoie la tte de la liste chane
 *----------------------------------------------------------------------------*/

ecs_famille_t * ecs_famille_chaine__cree
(
       ecs_int_t   * *const def_fam_descr,  /* Dfinition des familles        */
 const ecs_int_t     *const nbr_descr_fam,  /* Nombre de descripteurs par fam */
 const ecs_int_t            num_fam_deb,    /* Premier numro de famille      */
 const ecs_int_t            nbr_fam,        /* Nombre de familles             */
       ecs_descr_t   *const descr_tete      /* Liste chane des descripteurs */
)
{

  ecs_int_t   ifam ;
  ecs_int_t   idescr ;

  ecs_descr_t  * descr_fam ;
  ecs_descr_t  * descr_tete_fam ;
  ecs_descr_t  * ptr_descr ;

  ecs_famille_t   * fam  ;
  ecs_famille_t   * fam_tete ;


  /*xxxxxxxxxxxxxxxxxxxxxxxxxxx Instructions xxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxx*/


  fam_tete = NULL ;


  for (ifam = 0 ; ifam < nbr_fam ; ifam++) {


    descr_tete_fam = NULL ;


    for (idescr = 0 ; idescr < nbr_descr_fam[ifam] ; idescr++) {

      ptr_descr = ecs_descr_chaine__cherche_num(descr_tete,
                                                def_fam_descr[ifam][idescr]);

      assert(ptr_descr != NULL) ;

      descr_fam = ecs_descr__copie(ptr_descr) ;

      ecs_descr_chaine__ajoute(&descr_tete_fam,
                               descr_fam) ;

    }


    fam = ecs_famille__cree(ifam + num_fam_deb, descr_tete_fam) ;

    ecs_famille_chaine__ajoute(&fam_tete,
                               fam) ;


  }


  return fam_tete ;

}


/*----------------------------------------------------------------------------
 *  Fonction librant la portion d'une liste chane de familles
 *    partir d'un noeud dont le pointeur est donn en argument.
 *  Le noeud est  NULL au retour de la fonction
 *----------------------------------------------------------------------------*/

void ecs_famille_chaine__detruit
(
 ecs_famille_t * * this_fam_noeud
)
{

  /*xxxxxxxxxxxxxxxxxxxxxxxxxxx Instructions xxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxx*/


  if (*this_fam_noeud != NULL) {

    ecs_famille_chaine__detruit(&(*this_fam_noeud)->l_famille_sui) ;

    *this_fam_noeud = ecs_famille__detruit(*this_fam_noeud) ;

  }


}


/*----------------------------------------------------------------------------
 *  Fonction imprimant  partir d'un noeud `ecs_famille_t' donn
 *   une liste chane de champs
 *   sur le flux dcrit par la structure `bft_file_t'
 *----------------------------------------------------------------------------*/

void ecs_famille_chaine__imprime
(
 const ecs_famille_t  *const this_fam_noeud,
       ecs_int_t             imp_col,
       bft_file_t     *const fic_imp
)
{


#define ECS_FCT_IMP_FAMILLE_NOEUD         "famille"


  /*xxxxxxxxxxxxxxxxxxxxxxxxxxx Instructions xxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxx*/


  if (this_fam_noeud != NULL) {

    ecs_famille_chaine__imprime(this_fam_noeud->l_famille_sui,
                                imp_col,
                                fic_imp) ;

    ecs_fic__imprime_ptr(fic_imp, imp_col, ECS_FCT_IMP_FAMILLE_NOEUD,
                         (const void *)this_fam_noeud) ;

    ecs_famille__imprime(this_fam_noeud,
                         imp_col,
                         fic_imp) ;

  }


}


/*----------------------------------------------------------------------------
 *  Fonction qui renvoie la taille en octets
 *   d'une chane de structures `ecs_famille_t'
 *----------------------------------------------------------------------------*/

float ecs_famille_chaine__ret_taille
(
 const ecs_famille_t *const this_fam_noeud
)
{

  float         taille ;


  /*xxxxxxxxxxxxxxxxxxxxxxxxxxx Instructions xxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxx*/


  taille = 0. ;


  if (this_fam_noeud != NULL) {

    taille += ecs_famille_chaine__ret_taille(this_fam_noeud->l_famille_sui) ;

    taille += ecs_famille__ret_taille(this_fam_noeud) ;

  }


  return taille ;


}


/*----------------------------------------------------------------------------
 *  Fonction qui ajoute  la fin d'une liste chane de familles
 *   rceptrice dont la tte est donne,
 *   une liste chane de familles  concatner dont la tte est donne
 *----------------------------------------------------------------------------*/

void ecs_famille_chaine__ajoute
(
 ecs_famille_t *      * this_fam_tete,
 ecs_famille_t *const   fam_concat_tete
)
{

  ecs_famille_t * loc_fam_prec ;
  ecs_famille_t * ptr_fam  ;


  /*xxxxxxxxxxxxxxxxxxxxxxxxxxx Instructions xxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxx*/


  assert(this_fam_tete != NULL) ;


  if (*this_fam_tete != NULL) {


    /* On va  la fin de la chane rceptrice */

    for (ptr_fam = *this_fam_tete ;
         ptr_fam != NULL ;
         ptr_fam = ptr_fam->l_famille_sui  )
      loc_fam_prec = ptr_fam ;


    /* On ajoute le lien avec le dbut de la chane  concatner */

    loc_fam_prec->l_famille_sui = fam_concat_tete ;


  }
  else {


    *this_fam_tete = fam_concat_tete ;

  }


}


/*----------------------------------------------------------------------------
 *  Fonction qui affiche la dfinition de la famille de numro donn
 *    partir de la liste chane des familles dont la tte est donne
 *----------------------------------------------------------------------------*/

void ecs_famille_chaine__affiche
(
 const ecs_int_t            num_fam,
       ecs_famille_t *const fam_tete
)
{

  ecs_int_t     ifam ;

  ecs_famille_t * ptr_fam ;


  /*xxxxxxxxxxxxxxxxxxxxxxxxxxx Instructions xxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxx*/


  assert(fam_tete != NULL) ;


  ifam = 1 ;
  ptr_fam = fam_tete ;
  while (ptr_fam != NULL && ifam++ != num_fam)
    ptr_fam = ptr_fam->l_famille_sui ;

  assert(ptr_fam != NULL) ;

  ecs_famille__affiche(ptr_fam) ;


}


/*----------------------------------------------------------------------------
 *  Fonction qui renvoie le nombre de familles
 *   de la liste chane des familles dont la tte est donne
 *----------------------------------------------------------------------------*/

ecs_int_t ecs_famille_chaine__ret_nbr
(
 const ecs_famille_t *const fam_tete
)
{

  ecs_int_t     nbr_fam ;

  const ecs_famille_t * ptr_fam ;


  /*xxxxxxxxxxxxxxxxxxxxxxxxxxx Instructions xxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxx*/


  /* Dtermination du nombre de familles de la liste chane */
  /*---------------------------------------------------------*/

  nbr_fam = 0 ;

  for (ptr_fam  = fam_tete ;
       ptr_fam != NULL ;
       ptr_fam  = ptr_fam->l_famille_sui) {

    nbr_fam++ ;

  }


  return nbr_fam ;


}


/*----------------------------------------------------------------------------
 *  Fonction qui copie une liste chane de familles
 *   dont la tte est donne
 *----------------------------------------------------------------------------*/

ecs_famille_t * ecs_famille_chaine__copie
(
 ecs_famille_t * famille_tete
)
{

  ecs_famille_t * famille_copie ;
  ecs_famille_t * famille_tete_copie ;
  ecs_famille_t * ptr_famille ;


  /*xxxxxxxxxxxxxxxxxxxxxxxxxxx Instructions xxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxx*/


  famille_tete_copie = NULL ;

  for (ptr_famille  = famille_tete ;
       ptr_famille != NULL ;
       ptr_famille  = ptr_famille->l_famille_sui   ) {

    famille_copie = ecs_famille__copie(ptr_famille) ;

    ecs_famille_chaine__ajoute(&famille_tete_copie,
                               famille_copie) ;

  }


  return famille_tete_copie ;


}


/*----------------------------------------------------------------------------
 *  Fonction qui renvoie pour chaque numro de famille
 *   le nombre et la liste des identificateurs de type couleur
 *   des descripteurs de la famille
 *----------------------------------------------------------------------------*/

ecs_tab_int_t * ecs_famille_chaine__ret_ide
(
 ecs_famille_t   *const fam_tete
)
{

  ecs_int_t     ifam ;
  ecs_int_t     nbr_fam ;

  ecs_famille_t * ptr_fam ;

  ecs_tab_int_t * tab_propr_fam ;


  /*xxxxxxxxxxxxxxxxxxxxxxxxxxx Instructions xxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxx*/


  assert(fam_tete    != NULL) ;


  /* Dtermination du nombre de familles de la liste chane */
  /*---------------------------------------------------------*/

  nbr_fam = ecs_famille_chaine__ret_nbr(fam_tete) ;


  /* Dtermination des couleurs pour chaque famille de la liste chane */
  /*--------------------------------------------------------------------*/

  BFT_MALLOC(tab_propr_fam, nbr_fam, ecs_tab_int_t) ;


  for (ptr_fam  = fam_tete, ifam = 0 ;
       ptr_fam != NULL ;
       ptr_fam  = ptr_fam->l_famille_sui, ifam++) {

    tab_propr_fam[ifam] = ecs_famille__ret_ide(ptr_fam) ;

  }

  return tab_propr_fam ;

}


/*----------------------------------------------------------------------------
 *  Fonction qui renvoie pour chaque numro de famille
 *   le nombre et une liste de pointeurs sur les noms des identificateurs
 *   de type groupe des descripteurs de la famille
 *----------------------------------------------------------------------------*/

ecs_tab_char_t * ecs_famille_chaine__ret_nom
(
 ecs_famille_t   *const fam_tete
)
{

  ecs_int_t     ifam ;
  ecs_int_t     nbr_fam ;

  ecs_famille_t  * ptr_fam ;

  ecs_tab_char_t * tab_propr_fam ;


  /*xxxxxxxxxxxxxxxxxxxxxxxxxxx Instructions xxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxx*/


  assert(fam_tete    != NULL) ;


  /* Dtermination du nombre de familles de la liste chane */
  /*---------------------------------------------------------*/

  nbr_fam = ecs_famille_chaine__ret_nbr(fam_tete) ;


  /* Dtermination des groupes pour chaque famille de la liste chane */
  /*-------------------------------------------------------------------*/

  BFT_MALLOC(tab_propr_fam, nbr_fam, ecs_tab_char_t) ;


  for (ptr_fam  = fam_tete, ifam = 0 ;
       ptr_fam != NULL ;
       ptr_fam  = ptr_fam->l_famille_sui, ifam++) {

    tab_propr_fam[ifam] = ecs_famille__ret_nom(ptr_fam) ;

  }

  return tab_propr_fam ;

}


/*----------------------------------------------------------------------------
 *  Fonction qui affiche par attribut (couleur ou groupe),
 *   les numros des familles auxquelles l'attribut appartient
 *    partir d'une liste chane de familles dont la tte est donne
 *----------------------------------------------------------------------------*/

void ecs_famille_chaine__aff_fam_att
(
 ecs_famille_t *const fam_tete
)
{

  size_t               idescr ;
  size_t               ifam ;

  ecs_tab_int_t        tab_descr ;
  ecs_tab_int_t      * tab_fam_descr ;
  ecs_descr_t      * * liste_ref_descr ;


  /*xxxxxxxxxxxxxxxxxxxxxxxxxxx Instructions xxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxx*/


  ecs_loc_famille_chaine__fam_att(fam_tete,
                                  NULL,
                                  &tab_descr,
                                  &tab_fam_descr,
                                  &liste_ref_descr) ;


  /* Affichage des attributs tris en fonction des */
  /* familles auxquelles ils appartiennent         */
  /*-----------------------------------------------*/

  for (idescr = 0 ; idescr < tab_descr.nbr ; idescr++) {

    ecs_descr__affiche(liste_ref_descr[idescr],
                       0) ;

    for (ifam = 0 ; ifam < tab_fam_descr[idescr].nbr ; ifam++) {

      bft_printf("  %*s%s %" ECS_FORMAT_ecs_int_t "\n",
                 strlen(_("Color")) + 1, "", _("Family"),
                 tab_fam_descr[idescr].val[ifam]) ;

    }

  }


  /* Libration mmoire */

  BFT_FREE(liste_ref_descr) ;

  for (idescr = 0 ; idescr < tab_descr.nbr ; idescr++)
    BFT_FREE(tab_fam_descr[idescr].val) ;
  BFT_FREE(tab_fam_descr) ;

  BFT_FREE(tab_descr.val) ;

}


/*----------------------------------------------------------------------------
 *  Fonction qui construit les 2 listes chanes de descripteurs de type
 *   "couleur" et "groupe"
 *   pour chaque numro de famille contenu dans le tableau donn
 *   et  partir de la liste chane des familles
 *
 *  Cette fonction dtermine aussi les 2 tableaux donnant pour chaque famille
 *   respectivement la liste des numros de couleurs associs  la famille
 *               et la liste des numros de groupes  associs  la famille
 *----------------------------------------------------------------------------*/

void ecs_famille_chaine__cree_descr
(
 ecs_famille_t   *const famille,
 ecs_tab_int_t          tab_fam,
 ecs_descr_t   * *const descr_tete_couleur,
 ecs_descr_t   * *const descr_tete_groupe,
 ecs_tab_int_t   *const tab_couleur_fam,
 ecs_tab_int_t   *const tab_groupe_fam,
 int             *const nbr_max_att_fam
)
{

  size_t        ifam ;
  size_t        icoul ;
  size_t        igrp ;
  ecs_int_t     ind_fam ;

  ecs_descr_t   * descr_tete_couleur_fam ;
  ecs_descr_t   * descr_tete_groupe_fam ;

  ecs_famille_t * ptr_fam ;

  ecs_tab_int_t   tab_renum_descr ;


  /*xxxxxxxxxxxxxxxxxxxxxxxxxxx Instructions xxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxx*/


  assert(famille != NULL) ;


  *nbr_max_att_fam = 0 ;
  *descr_tete_couleur = NULL ;
  *descr_tete_groupe  = NULL ;


  for (ifam = 0 ; ifam < tab_fam.nbr ; ifam++) {

    ind_fam = tab_fam.val[ifam] - 1 ;

    ptr_fam = famille ;
    while (ptr_fam != NULL && ECS_ABS(ptr_fam->num) != tab_fam.val[ifam])
      ptr_fam = ptr_fam->l_famille_sui ;

    assert(ptr_fam != NULL) ;

    ecs_famille__cree_descr(ptr_fam,
                            &descr_tete_couleur_fam,
                            &descr_tete_groupe_fam) ;

    if (descr_tete_couleur_fam != NULL) {

      tab_renum_descr = ecs_descr_chaine__concatene(descr_tete_couleur,
                                                    &descr_tete_couleur_fam) ;

      tab_couleur_fam[ind_fam].nbr = tab_renum_descr.nbr ;
      BFT_MALLOC(tab_couleur_fam[ind_fam].val,
                 tab_couleur_fam[ind_fam].nbr, ecs_int_t) ;

      for (icoul = 0 ; icoul < tab_renum_descr.nbr ; icoul++)
        tab_couleur_fam[ind_fam].val[icoul] = tab_renum_descr.val[icoul] + 1 ;

      BFT_FREE(tab_renum_descr.val) ;

      *nbr_max_att_fam = ECS_MAX(*nbr_max_att_fam,
                                 (int)(tab_renum_descr.nbr)) ;

    }
    else {

      tab_couleur_fam[ind_fam].val = NULL ;
      tab_couleur_fam[ind_fam].nbr = 0 ;

    }


    if (descr_tete_groupe_fam != NULL) {

      tab_renum_descr = ecs_descr_chaine__concatene(descr_tete_groupe,
                                                    &descr_tete_groupe_fam) ;

      tab_groupe_fam[ind_fam].nbr = tab_renum_descr.nbr ;
      BFT_MALLOC(tab_groupe_fam[ind_fam].val,
                 tab_groupe_fam[ind_fam].nbr, ecs_int_t) ;

      for (igrp = 0 ; igrp < tab_renum_descr.nbr ; igrp++)
        tab_groupe_fam[ind_fam].val[igrp  ] = tab_renum_descr.val[igrp] + 1 ;

      BFT_FREE(tab_renum_descr.val) ;

      *nbr_max_att_fam = ECS_MAX(*nbr_max_att_fam,
                                 (int)(tab_renum_descr.nbr)) ;

    }
    else {

      tab_groupe_fam[ind_fam].val = NULL ;
      tab_groupe_fam[ind_fam].nbr = 0 ;

    }

  }


}


/*----------------------------------------------------------------------------
 *  Fonction qui renvoie un tableau indiquant pour chaque numro de
 *   famille si cette famille contient l'un des attributs (couleur ou groupe)
 *   fournis en argument.
 *
 *  La libration du tableau est  la charge du code appelant
 *----------------------------------------------------------------------------*/

ecs_tab_bool_t ecs_famille_chaine__indic_fam_att
(
 const ecs_famille_t   *const fam_tete,
 const ecs_tab_int_t          liste_couleur,
 const ecs_tab_char_t         liste_groupe
)
{

  size_t               ifam ;
  ecs_int_t            num_fam_max ;

  const ecs_famille_t   * ptr_fam ;

  ecs_tab_bool_t    tab_select ;


  /*xxxxxxxxxxxxxxxxxxxxxxxxxxx Instructions xxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxx*/


  /* Dtermination du numro de famille maximal de la liste chane */
  /*----------------------------------------------------------------*/

  num_fam_max = 0 ;

  for (ptr_fam  = fam_tete ;
       ptr_fam != NULL ;
       ptr_fam  = ptr_fam->l_famille_sui) {

    if (ptr_fam->num > num_fam_max)
      num_fam_max = ptr_fam->num ;

  }

  tab_select.nbr = num_fam_max + 1 ;
  BFT_MALLOC(tab_select.val, tab_select.nbr, ecs_bool_t) ;

  for (ifam = 0 ; ifam < tab_select.nbr ; ifam++)
    tab_select.val[ifam] = ECS_FALSE ;


  /* Marquage des familles correspondant aux critres de slection */
  /*---------------------------------------------------------------*/

  for (ptr_fam  = fam_tete ;
       ptr_fam != NULL ;
       ptr_fam  = ptr_fam->l_famille_sui) {

    tab_select.val[ptr_fam->num] = ecs_descr_chaine__select(ptr_fam->descr,
                                                            liste_couleur,
                                                            liste_groupe) ;

  }

  return tab_select ;

}


/*----------------------------------------------------------------------------
 *  Fonction qui construit un tableau de description d'attributs
 *   dcrits par une liste chane de familles, avec pour chacun la
 *   liste des lments portant cet attribut ( partir du numro de
 *   famille associ  chaque lment et donn par tab_fam_ent)
 *
 *  Les arguments prefixe_coul et prefixe_grp permettent de dterminer les
 *   chanes de caractres prcdent ventuellement les numros de couleur ou
 *   noms de groupe dans le tableau rsultant tab_nom_descr.
 *  L'argument "grouper_ident" indique si l'on doit combiner les attributs
 *   appartenant exactement aux mmes familles.
 *
 *  La libration des tableaux tab_nom_desc et tab_lst_desc est  la
 *   charge de la fonction utilisatrice.
 *----------------------------------------------------------------------------*/

void ecs_famille_chaine__att_fam_elt
(
 const ecs_famille_t     * fam_tete,
 const ecs_tab_int_t       tab_fam_elt,
 const char              * prefixe_coul,
 const char              * prefixe_grp,
       ecs_bool_t          grouper_ident,
       ecs_tab_char_t    * tab_nom_descr,
       ecs_tab_int_t   * * tab_lst_descr
)
{

  size_t               ielt ;
  int                  ifam ;
  int                  idescr ;
  int                  idescr_cmp ;
  int                  num_fam_loc ;
  int                  num_fam_max ;
  ecs_int_t          * nbr_elt_fam ;
  ecs_int_t        * * lst_elt_fam ;
  ecs_int_t          * nbr_elt_descr ;
  ecs_int_t        * * lst_elt_descr ;

  int                * descr_equiv ;

  ecs_tab_int_t        tab_descr ;
  ecs_tab_int_t        tab_elt_descr ;
  ecs_tab_int_t        tab_renum ;
  ecs_tab_int_t      * tab_fam_descr ;
  ecs_descr_t      * * liste_ref_descr ;

  const char           *_prefixe_coul ;
  const char           *_prefixe_grp ;

  const ecs_famille_t  * ptr_fam ;

  const char           chaine_vide[] = "" ;

  /*xxxxxxxxxxxxxxxxxxxxxxxxxxx Instructions xxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxx*/


  if (tab_fam_elt.nbr == 0)
    return ;

  /* Si les prfixes ne sont pas fournis, on en prvoit un par dfaut */

  if (prefixe_coul != NULL)
    _prefixe_coul = prefixe_coul ;
  else
    _prefixe_coul = chaine_vide ;

  if (prefixe_grp != NULL)
    _prefixe_grp = prefixe_grp ;
  else
    _prefixe_grp = chaine_vide ;


  /* Dtermination du numro de famille maximal de la liste chane */

  num_fam_max = 0 ;

  for (ptr_fam  = fam_tete ;
       ptr_fam != NULL ;
       ptr_fam  = ptr_fam->l_famille_sui) {

    if (ptr_fam->num > num_fam_max)
      num_fam_max = ptr_fam->num ;

  }


  /* Cration des listes d'lments associs  chaque famille */
  /*----------------------------------------------------------*/

  BFT_MALLOC(nbr_elt_fam, num_fam_max + 1, ecs_int_t) ;
  BFT_MALLOC(lst_elt_fam, num_fam_max + 1, ecs_int_t *) ;

  /* Comptage et allocation */

  for (ifam = 0 ; ifam <= num_fam_max ; ifam++) {
    nbr_elt_fam[ifam] = 0 ;
    lst_elt_fam[ifam] = NULL ;
  }

  for (ielt = 0 ; ielt < tab_fam_elt.nbr ; ielt++)
    nbr_elt_fam[tab_fam_elt.val[ielt]] += 1 ;

  for (ifam = 0 ; ifam <= num_fam_max ; ifam++) {
    if (nbr_elt_fam[ifam] > 0)
      BFT_MALLOC(lst_elt_fam[ifam], nbr_elt_fam[ifam], ecs_int_t) ;
  }


  /* Dtermination des attributs et familles associes */
  /*---------------------------------------------------*/

  ecs_loc_famille_chaine__fam_att(fam_tete,
                                  nbr_elt_fam,
                                  &tab_descr,
                                  &tab_fam_descr,
                                  &liste_ref_descr) ;


  /* Construction des listes */

  for (ifam = 0 ; ifam <= num_fam_max ; ifam++) {
    nbr_elt_fam[ifam] = 0 ;
  }

  for (ielt = 0 ; ielt < tab_fam_elt.nbr ; ielt++) {
    num_fam_loc = tab_fam_elt.val[ielt] ;
    lst_elt_fam[num_fam_loc][nbr_elt_fam[num_fam_loc]] = ielt ;
    nbr_elt_fam[num_fam_loc] += 1 ;
  }


  /* Dtermination des descripteurs quivalents */
  /*--------------------------------------------*/

  BFT_MALLOC(descr_equiv, tab_descr.nbr, int) ;

  for (idescr = 0 ; idescr < (int)(tab_descr.nbr) ; idescr++)
    descr_equiv[idescr] = -1 ;

  if (grouper_ident == ECS_TRUE) {

    for (idescr = 0 ; idescr < (int)(tab_descr.nbr) ; idescr++) {
      for (idescr_cmp = idescr +1 ;
           idescr_cmp < (int)(tab_descr.nbr) ;
           idescr_cmp++) {

        ecs_tab_int_t tab_fam     = tab_fam_descr[tab_descr.val[idescr]] ;
        ecs_tab_int_t tab_fam_cmp = tab_fam_descr[tab_descr.val[idescr_cmp]] ;

        if (tab_fam.nbr == tab_fam_cmp.nbr) {

          for (ifam = 0 ; ifam < (int)(tab_fam.nbr) ; ifam++) {
            if (tab_fam.val[ifam] != tab_fam_cmp.val[ifam])
              break ;
          }
          if (ifam == (int)(tab_fam.nbr)) {
            descr_equiv[idescr] = idescr_cmp ;
            break ; /* sortie de la boucle sur idescr_cmp, ainsi chaque
                       descripteur ayant un quivalent pointe sur
                       l'quivalent suivant (sorte de liste chane) */
          }

        } /* fin si le nombre de familles associ est le mme */

      } /* fin de la boucle sur idescr_cmp */
    } /* fin de la boucle sur les descripteurs */

  }


  /* Cration des listes associes aux descripteurs */
  /*------------------------------------------------*/

  BFT_MALLOC(nbr_elt_descr, tab_descr.nbr, ecs_int_t) ;
  BFT_MALLOC(lst_elt_descr, tab_descr.nbr, ecs_int_t *) ;

  for (idescr = 0 ; idescr < (int)(tab_descr.nbr) ; idescr++) {

    ecs_tab_int_t tab_fam = tab_fam_descr[tab_descr.val[idescr]] ;

    /* En cas d'quivalence, on utilisera la liste du descripteur
       en fin de liste d'quivalence */

    if (descr_equiv[idescr] > -1) {
      nbr_elt_descr[idescr] = 0 ;
      lst_elt_descr[idescr] = NULL ;
      continue ;
    }

    /* Sans quivalence ou pour le dernier descripteur quivalent,
       on construit la liste */

    nbr_elt_descr[idescr] = 0 ;
    for (ifam = 0 ; ifam < (int)(tab_fam.nbr) ; ifam++) {
      num_fam_loc = tab_fam.val[ifam] ;
      nbr_elt_descr[idescr] += nbr_elt_fam[num_fam_loc] ;
    }

    BFT_MALLOC(lst_elt_descr[idescr], nbr_elt_descr[idescr], ecs_int_t) ;

    nbr_elt_descr[idescr] = 0 ;
    for (ifam = 0 ; ifam < (int)(tab_fam.nbr) ; ifam++) {
      num_fam_loc = tab_fam.val[ifam] ;
      for (ielt = 0 ; ielt < (size_t)(nbr_elt_fam[num_fam_loc]) ; ielt++) {
        lst_elt_descr[idescr][nbr_elt_descr[idescr] + ielt]
          = lst_elt_fam[num_fam_loc][ielt] ;
      }
      nbr_elt_descr[idescr] += nbr_elt_fam[num_fam_loc] ;
    }

  }


  /* Prparation des valeurs de retour */
  /*-----------------------------------*/

  tab_nom_descr->nbr = tab_descr.nbr ;
  BFT_MALLOC(tab_nom_descr->val, tab_nom_descr->nbr, char *) ;
  BFT_MALLOC(*tab_lst_descr, tab_nom_descr->nbr, ecs_tab_int_t) ;

  for (idescr = 0 ; idescr < (int)(tab_descr.nbr) ; idescr++) {

    ecs_descr_t *descr = liste_ref_descr[tab_descr.val[idescr]] ;

    switch (ecs_descr__ret_typ(descr)) {

    case ECS_DESCR_TYP_COULEUR:
      {
        const ecs_int_t ide = ecs_descr__ret_ide(descr) ;

        ecs_int_t ide_div = ECS_ABS(ide) ;
        ecs_int_t l_str_ide = 0 ;

        while (ide_div > 0) {
          l_str_ide++ ;
          ide_div /= 10 ;
        }
        if (ide <= 0)
          l_str_ide++ ;

        BFT_MALLOC(tab_nom_descr->val[idescr],
                   strlen(_prefixe_coul) + l_str_ide + 1,
                   char) ;
        sprintf(tab_nom_descr->val[idescr], "%s%d", _prefixe_coul, ide) ;
      }
      break ;

    case ECS_DESCR_TYP_GROUPE:
      {
        const char *nom_groupe = ecs_descr__ret_nom(descr) ;

        BFT_MALLOC(tab_nom_descr->val[idescr],
                   strlen(_prefixe_grp) + strlen(nom_groupe) + 1, char) ;
        sprintf(tab_nom_descr->val[idescr], "%s%s", _prefixe_grp, nom_groupe) ;
      }
      break ;

    }

    for (idescr_cmp = 0 ; idescr_cmp < idescr ; idescr_cmp++) {
      if (descr_equiv[idescr_cmp] == idescr) {
        BFT_REALLOC(tab_nom_descr->val[idescr_cmp],
                    (strlen(tab_nom_descr->val[idescr_cmp])
                     + strlen(tab_nom_descr->val[idescr])
                     + strlen(", ") + 1),
                    char) ;
        strcat(tab_nom_descr->val[idescr_cmp], ", ") ;
        strcat(tab_nom_descr->val[idescr_cmp], tab_nom_descr->val[idescr]) ;
        BFT_FREE(tab_nom_descr->val[idescr]) ;
        tab_nom_descr->val[idescr] = tab_nom_descr->val[idescr_cmp] ;
        tab_nom_descr->val[idescr_cmp] = NULL ;
      }
    }

    /* Tri des tableaux et affectation  la liste finale */

    tab_elt_descr.nbr = nbr_elt_descr[idescr] ;
    tab_elt_descr.val = lst_elt_descr[idescr] ;

    tab_renum.nbr = nbr_elt_descr[idescr] ;
    BFT_MALLOC(tab_renum.val, tab_renum.nbr, ecs_int_t) ;

    *(*tab_lst_descr + idescr) = ecs_tab_int__trie_et_renvoie(tab_elt_descr,
                                                              tab_renum) ;

    tab_renum.nbr = 0 ;
    BFT_FREE(tab_renum.val) ;

    tab_elt_descr.nbr = 0 ;
    BFT_FREE(tab_elt_descr.val) ;

  }

  BFT_FREE(nbr_elt_descr) ;
  BFT_FREE(lst_elt_descr) ;


  /* Ajout d'un descripteur correspondant au descripteur vide */

  if (nbr_elt_fam[0] > 0) {

    tab_nom_descr->nbr += 1 ;
    BFT_REALLOC(tab_nom_descr->val, tab_nom_descr->nbr, char *) ;
    tab_nom_descr->val[tab_nom_descr->nbr - 1] = NULL ;

    BFT_REALLOC(*tab_lst_descr, tab_nom_descr->nbr, ecs_tab_int_t) ;
    (*tab_lst_descr + tab_nom_descr->nbr - 1)->nbr = nbr_elt_fam[0] ;
    (*tab_lst_descr + tab_nom_descr->nbr - 1)->val = lst_elt_fam[0] ;
    lst_elt_fam[0] = NULL ; /* proprit du tableau transfre */

  }

  /* Libration mmoire */

  for (ifam = 0 ; ifam <= num_fam_max ; ifam++) {
    if (lst_elt_fam[ifam] != NULL)
      BFT_FREE(lst_elt_fam[ifam]) ;
  }

  BFT_FREE(lst_elt_fam) ;
  BFT_FREE(nbr_elt_fam) ;

  BFT_FREE(descr_equiv) ;

  BFT_FREE(liste_ref_descr) ;

  for (idescr = 0 ; idescr < (int)(tab_descr.nbr) ; idescr++)
    BFT_FREE(tab_fam_descr[idescr].val) ;
  BFT_FREE(tab_fam_descr) ;

  BFT_FREE(tab_descr.val) ;

}


/*============================================================================
 *                              Fonctions prives
 *============================================================================*/

/*----------------------------------------------------------------------------
 *  Fonction qui dtermine la liste des attributs rfrencs par une
 *   liste chane de familles dont la tte est donne, ainsi que
 *   la liste des familles correspondant  chaque attribut
 *
 *  Le tableau optionnel nbr_elt_fam permet de limiter ventuellement
 *   la liste des attributs  ceux qu sont effectivement rfrencs
 *----------------------------------------------------------------------------*/

static void ecs_loc_famille_chaine__fam_att
(
 const ecs_famille_t      * fam_tete,
 const ecs_int_t          * nbr_elt_fam,
       ecs_tab_int_t      * tab_descr,
       ecs_tab_int_t    * * tab_fam_descr,
       ecs_descr_t    * * * liste_ref_descr
)
{

  ECS_DESCR_TYP_E    descr_typ_e ;

  ecs_int_t            cpt_descr ;
  ecs_int_t            cpt_fam ;
  ecs_int_t            icoul ;
  ecs_int_t            idescr ;
  ecs_int_t            ifam ;
  ecs_int_t            igrp ;
  ecs_int_t            nbr_fam ;
  ecs_int_t            nbr_max_descr ;
  ecs_int_t            nbr_ref_descr ;
  ecs_int_t          * liste_nbr_descr_fam ;
  ecs_int_t          * liste_num_descr_coul ;
  ecs_int_t          * liste_num_descr_grp ;

  ecs_descr_t      * * * liste_ref_descr_fam ;

  ecs_tab_int_t          tab_couleur ;
  ecs_tab_int_t          tab_groupe  ;
  ecs_tab_int_t          tab_renum_coul  ;
  ecs_tab_int_t          tab_renum_grp ;

  const ecs_famille_t  * ptr_fam ;

  ecs_tab_int_t          _tab_descr ;
  ecs_tab_int_t        * _tab_fam_descr ;
  ecs_descr_t        * * _liste_ref_descr ;


  /*xxxxxxxxxxxxxxxxxxxxxxxxxxx Instructions xxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxx*/


  /* Dtermination du nombre de familles de la liste chane */
  /*---------------------------------------------------------*/

  for (ptr_fam  = fam_tete, cpt_fam = 0 ;
       ptr_fam != NULL ;
       ptr_fam  = ptr_fam->l_famille_sui) {

    if (nbr_elt_fam != NULL && nbr_elt_fam[ptr_fam->num] == 0)
      continue ;

    cpt_fam++ ;

  }

  nbr_fam = cpt_fam ;


  BFT_MALLOC(liste_ref_descr_fam, nbr_fam, ecs_descr_t * *) ;
  BFT_MALLOC(liste_nbr_descr_fam, nbr_fam, ecs_int_t    ) ;


  nbr_max_descr = 0 ;

  for (ptr_fam  = fam_tete, cpt_fam = 0 ;
       ptr_fam != NULL ;
       ptr_fam  = ptr_fam->l_famille_sui) {

    if (nbr_elt_fam != NULL && nbr_elt_fam[ptr_fam->num] == 0)
      continue ;

    liste_ref_descr_fam[cpt_fam]
      = ecs_descr_chaine__ret_ref(ptr_fam->descr,
                                  &liste_nbr_descr_fam[cpt_fam]) ;

    nbr_max_descr += liste_nbr_descr_fam[cpt_fam] ;

    cpt_fam++ ;

  }


  /* Construction du tableau donnant pour chaque descripteur */
  /*  les numros de famille auxquelles il appartient        */
  /*---------------------------------------------------------*/

  BFT_MALLOC(_tab_fam_descr, nbr_max_descr, ecs_tab_int_t) ;
  _liste_ref_descr = NULL ;
  nbr_ref_descr = 0 ;


  for (ptr_fam  = fam_tete, cpt_fam = 0 ;
       ptr_fam != NULL ;
       ptr_fam  = ptr_fam->l_famille_sui) {

    if (nbr_elt_fam != NULL && nbr_elt_fam[ptr_fam->num] == 0)
      continue ;

    for (idescr = 0 ; idescr < liste_nbr_descr_fam[cpt_fam] ; idescr++) {

      cpt_descr = 0 ;
      while (   cpt_descr < nbr_ref_descr
             && (   ecs_descr__compare(_liste_ref_descr[cpt_descr],
                                       liste_ref_descr_fam[cpt_fam][idescr])
                 == ECS_FALSE))
        cpt_descr++ ;

      if (cpt_descr == nbr_ref_descr) {

        /* Le descripteur n'a pas dj t rencontre */
        /* On le rajoute  la liste */

        BFT_REALLOC(_liste_ref_descr, nbr_ref_descr + 1, ecs_descr_t *) ;

        _liste_ref_descr[cpt_descr] = liste_ref_descr_fam[cpt_fam][idescr] ;

        nbr_ref_descr++ ;

        /* On stocke le numro de la famille  laquelle il appartient */

        _tab_fam_descr[cpt_descr].nbr = 1 ;
        BFT_MALLOC(_tab_fam_descr[cpt_descr].val,
                   _tab_fam_descr[cpt_descr].nbr, ecs_int_t) ;

        _tab_fam_descr[cpt_descr].val[0] = ptr_fam->num ;

      }
      else {

        /* Le descripteur a dj t rencontre */
        /* On rajoute le numro de famille */

        BFT_REALLOC(_tab_fam_descr[cpt_descr].val,
                    _tab_fam_descr[cpt_descr].nbr + 1, ecs_int_t) ;

        _tab_fam_descr[cpt_descr].val[_tab_fam_descr[cpt_descr].nbr]
          = ptr_fam->num ;

        _tab_fam_descr[cpt_descr].nbr++ ;

      }

    } /* Fin : boucle sur les descripteurs de la famille */

    cpt_fam++ ;

  } /* Fin : boucle sur les familles */


  for (ifam = 0 ; ifam < nbr_fam ; ifam++)
    BFT_FREE(liste_ref_descr_fam[ifam]) ;
  BFT_FREE(liste_ref_descr_fam) ;
  BFT_FREE(liste_nbr_descr_fam) ;


  /* Sparation des descripteurs de type "couleur" */
  /*         et des descripteurs de type "groupe"  */
  /*-----------------------------------------------*/

  tab_couleur.nbr = 0 ;
  tab_groupe.nbr  = 0 ;

  BFT_MALLOC(tab_couleur.val,      nbr_ref_descr, ecs_int_t) ;
  BFT_MALLOC(liste_num_descr_coul, nbr_ref_descr, ecs_int_t) ;
  BFT_MALLOC(tab_groupe.val,       nbr_ref_descr, ecs_int_t) ;
  BFT_MALLOC(liste_num_descr_grp,  nbr_ref_descr, ecs_int_t) ;

  for (idescr = 0 ; idescr < nbr_ref_descr ; idescr++) {

    descr_typ_e = ecs_descr__ret_typ(_liste_ref_descr[idescr]) ;

    switch (descr_typ_e) {

    case ECS_DESCR_TYP_COULEUR:

      tab_couleur.val[tab_couleur.nbr]
        = ecs_descr__ret_ide(_liste_ref_descr[idescr]) ;
      liste_num_descr_coul[tab_couleur.nbr++] = idescr ;

      break ;

    case ECS_DESCR_TYP_GROUPE:

      tab_groupe.val[tab_groupe.nbr]
        = ecs_descr__ret_ide(_liste_ref_descr[idescr]) ;
      liste_num_descr_grp[tab_groupe.nbr++] = idescr ;

      break ;

    default:

      assert(descr_typ_e == ECS_DESCR_TYP_COULEUR ||
             descr_typ_e == ECS_DESCR_TYP_GROUPE    ) ;

    }

  }

  if (tab_couleur.nbr != 0)
    BFT_REALLOC(tab_couleur.val, tab_couleur.nbr, ecs_int_t) ;
  else
    BFT_FREE(tab_couleur.val) ;

  if (tab_groupe.nbr != 0)
    BFT_REALLOC(tab_groupe.val, tab_groupe.nbr, ecs_int_t) ;
  else
    BFT_FREE(tab_groupe.val) ;


  /* Ordination des couleurs et groupes par leur numro */
  /*----------------------------------------------------*/

  assert(nbr_ref_descr = tab_couleur.nbr + tab_groupe.nbr) ;

  _tab_descr.nbr = nbr_ref_descr ;
  BFT_MALLOC(_tab_descr.val, _tab_descr.nbr, ecs_int_t) ;

  for (idescr = 0 ; idescr < (int)(_tab_descr.nbr) ; idescr++)
    _tab_descr.val[idescr] = -1 ;

  idescr = 0 ;


  if (tab_couleur.nbr != 0) {

    tab_renum_coul.nbr = tab_couleur.nbr ;
    BFT_MALLOC(tab_renum_coul.val, tab_renum_coul.nbr, ecs_int_t) ;
    for (icoul = 0 ; icoul < (int)(tab_renum_coul.nbr) ; icoul++)
      tab_renum_coul.val[icoul] = icoul ;

    ecs_tab_int__trie(tab_couleur,
                      tab_renum_coul) ;

    BFT_FREE(tab_couleur.val) ;

    for (icoul = 0 ; icoul < (int)(tab_renum_coul.nbr) ; icoul++)
      _tab_descr.val[idescr++]
        = liste_num_descr_coul[tab_renum_coul.val[icoul]] ;

    BFT_FREE(tab_renum_coul.val) ;

  }

  if (tab_groupe.nbr != 0) {

    tab_renum_grp.nbr = tab_groupe.nbr ;
    BFT_MALLOC(tab_renum_grp.val, tab_renum_grp.nbr, ecs_int_t) ;
    for (igrp = 0 ; igrp < (int)(tab_renum_grp.nbr) ; igrp++)
      tab_renum_grp.val[igrp] = igrp ;

    ecs_tab_int__trie(tab_groupe,
                      tab_renum_grp) ;

    BFT_FREE(tab_groupe.val) ;

    for (igrp = 0 ; igrp < (int)(tab_renum_grp.nbr) ; igrp++)
      _tab_descr.val[idescr++]
        = liste_num_descr_grp[tab_renum_grp.val[igrp]] ;

    BFT_FREE(tab_renum_grp.val) ;

  }

  BFT_FREE(liste_num_descr_coul) ;
  BFT_FREE(liste_num_descr_grp) ;

  assert(idescr = _tab_descr.nbr) ;


  /* Valeurs de retour */
  /*-------------------*/

  *tab_descr       = _tab_descr ;
  *tab_fam_descr   = _tab_fam_descr ;
  *liste_ref_descr = _liste_ref_descr ;

}

