/*============================================================================
 *  Définitions des fonctions
 *   associées à la structure `ecs_champ_t' décrivant un champ
 *   et liées à la structure `ecs_vec_int_t'
 *============================================================================*/

/*
  This file is part of the Code_Saturne Preprocessor, element of the
  Code_Saturne CFD tool.

  Copyright (C) 1999-2007 EDF S.A., France

  contact: saturne-support@edf.fr

  The Code_Saturne Preprocessor is free software; you can redistribute it
  and/or modify it under the terms of the GNU General Public License
  as published by the Free Software Foundation; either version 2 of
  the License, or (at your option) any later version.

  The Code_Saturne Preprocessor is distributed in the hope that it will be
  useful, but WITHOUT ANY WARRANTY; without even the implied warranty
  of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.

  You should have received a copy of the GNU General Public License
  along with the Code_Saturne Preprocessor; if not, write to the
  Free Software Foundation, Inc.,
  51 Franklin St, Fifth Floor,
  Boston, MA  02110-1301  USA
*/


/*----------------------------------------------------------------------------
 *  Fichiers `include' librairie standard C ou BFT
 *----------------------------------------------------------------------------*/

#include <assert.h>
#include <string.h> /* strlen(), strcpy() */


#include <bft_mem.h>


/*----------------------------------------------------------------------------
 *  Fichiers `include' visibles du  paquetage global "Utilitaire"
 *----------------------------------------------------------------------------*/

#include "ecs_chaine_glob.h"
#include "ecs_def.h"
#include "ecs_tab.h"


/*----------------------------------------------------------------------------
 *  Fichiers `include' visibles des paquetages visibles
 *----------------------------------------------------------------------------*/

#include "ecs_vec_int.h"
#include "ecs_vec_int_tri.h"


/*----------------------------------------------------------------------------
 *  Fichiers `include' visibles du  paquetage courant
 *----------------------------------------------------------------------------*/

#include "ecs_champ.h"

/*----------------------------------------------------------------------------
 *  Fichier  `include' du  paquetage courant associé au fichier courant
 *----------------------------------------------------------------------------*/

#include "ecs_champ_vec_int.h"


/*----------------------------------------------------------------------------
 *  Fichiers `include' privés   du  paquetage courant
 *----------------------------------------------------------------------------*/

#include "ecs_champ_priv.h"



/*============================================================================
 *                       Prototypes de fonctions privées
 *============================================================================*/


/*============================================================================
 *                             Fonctions publiques
 *============================================================================*/

/*----------------------------------------------------------------------------
 *  Fonction construisant un vecteur `ecs_vec_int_t' associé au champ donné
 *
 *  Si les tables sont REGLEES, les tableaux sont construits
 *
 *  Cette fonction doit être utilisée conjointement avec :
 *  - soit `ecs_champ__transfere_vec_int()'
 *                               si le contenu du vecteur   a changé
 *  - soit `ecs_champ__libere_vec_int()'
 *                               si le contenu du vecteur n'a pas été modifié
 *  Ceci afin de reconstuire, si nécessaire, les tables REGLEES
 *----------------------------------------------------------------------------*/

ecs_vec_int_t * ecs_champ__initialise_vec_int
(
 ecs_champ_t *this_champ
)
{

  ecs_vec_int_t *vec_int_loc ;

  size_t       pos_nbr ;  /* Nombre   de positions */
  ecs_size_t  *pos_tab ;  /* Tableau des positions */
  ecs_int_t   *val_tab ;  /* Tableau des valeurs */


  /*xxxxxxxxxxxxxxxxxxxxxxxxxxx Instructions xxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxx*/


  assert(this_champ != NULL) ;


  assert(ecs_champ__ret_val_typ(this_champ) == ECS_TYPE_ecs_int_t) ;


  /* Récupération des références des tableaux positions et valeurs   */
  /*  des tables contenues par le champ                              */
  /* (pour une table REGLEE, le tableau correspondant est developpé) */
  /*-----------------------------------------------------------------*/

  pos_nbr =              this_champ->nbr_elt + 1 ;
  pos_tab =              ecs_champ__ret_pos_tab(this_champ) ;
  val_tab = (ecs_int_t *)(this_champ->val_tab) ;


  /* Affectation des valeurs recupérées au vecteur `ecs_vec_int_t' crée */
  /*--------------------------------------------------------------------*/

  vec_int_loc = ecs_vec_int__initialise(pos_nbr,
                                        pos_tab,
                                        val_tab) ;

  return vec_int_loc ;


}


/*----------------------------------------------------------------------------
 *  Fonction qui transfère le contenu d'un vecteur `ecs_vec_int_t'
 *   dans les tables d'un champ donné
 *
 *  Les tables sont transformées en REGLES si possible
 *
 *  La structure `ecs_vec_int_t' est libérée
 *----------------------------------------------------------------------------*/

void ecs_champ__transfere_vec_int
(
 ecs_champ_t    *this_champ,  /* <-> Champ dans lequel on transfère `vec_int' */
 ecs_vec_int_t  *vec_int      /* <-> Vecteur `ecs_vec_int_t' à transférer     */
)
{

  size_t       pos_nbr ;  /* Nombre   de positions */
  ecs_size_t  *pos_tab ;  /* Tableau des positions */
  size_t       val_nbr ;  /* Nombre   de valeurs */
  ecs_int_t   *val_tab ;  /* Tableau des valeurs */


  /*xxxxxxxxxxxxxxxxxxxxxxxxxxx Instructions xxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxx*/


  assert(this_champ->typ_val == ECS_TYPE_ecs_int_t) ;


  pos_nbr = ecs_vec_int__ret_pos_nbr(vec_int) ;
  pos_tab = ecs_vec_int__ret_pos_tab(vec_int) ;
  val_nbr = ecs_vec_int__ret_val_nbr(vec_int) ;
  val_tab = ecs_vec_int__ret_val_tab(vec_int) ;


  if (pos_tab == NULL) {
    /* Les positions forment une REGLE de pas unité */
    this_champ->pos_pas = 1 ;
  }
  else {
    this_champ->pos_pas = 0 ;
  }

  this_champ->nbr_elt = pos_nbr - 1 ;

  this_champ->pos_tab = pos_tab ;
  ecs_champ__pos_en_regle(this_champ) ;

  this_champ->val_tab = val_tab ;

  ecs_vec_int__libere(vec_int) ;

}


/*----------------------------------------------------------------------------
 *  Fonction qui libère le tableau d'une table REGLEE qui a été
 *   précédemment developpé par l'appel de `ecs_champ__initialise_vec_int()'
 *  et qui détruit le vecteur `ecs_vec_int_t' qui a été
 *   précédemment crée      par l'appel de `ecs_champ__initialise_vec_int()'
 *----------------------------------------------------------------------------*/

void ecs_champ__libere_vec_int
(
 ecs_champ_t    * this_champ,                 /* <-> Champ associé au vecteur */
 ecs_vec_int_t  * vec_int                     /* <-> Vecteur à détruire       */
)
{

  /*xxxxxxxxxxxxxxxxxxxxxxxxxxx Instructions xxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxx*/


  ecs_champ__libere_pos_tab(this_champ,
                            ecs_vec_int__ret_pos_tab(vec_int)) ;

  ecs_vec_int__libere(vec_int) ;

}


/*----------------------------------------------------------------------------
 *  Fonction qui crée un champ à partir
 *   du contenu d'un vecteur `ecs_vec_int_t'
 *
 *  Seuls le nom du champ et les tables du champ sont remplis
 *   (les tables sont transformées en REGLES si possible)
 *
 *  Les autres membres de la structure prennent leur valeur d'initialisation
 *
 *  La structure `ecs_vec_int_t' est libérée
 *----------------------------------------------------------------------------*/

ecs_champ_t * ecs_champ__init_avec_vec_int
(
       ecs_vec_int_t  *      vec_int,            /* --> Vecteur à transférer */
 const char           *const nom
)
{

  ecs_champ_t  *this_champ ;

  size_t        pos_nbr ;  /* Nombre   de positions */
  ecs_size_t   *pos_tab ;  /* Tableau des positions */
  ecs_int_t    *val_tab ;  /* Tableau des valeurs */


  /*xxxxxxxxxxxxxxxxxxxxxxxxxxx Instructions xxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxx*/


  /* Récupération des valeurs du vecteur `ecs_vec_int_t' */

  pos_nbr = ecs_vec_int__ret_pos_nbr(vec_int) ;
  pos_tab = ecs_vec_int__ret_pos_tab(vec_int) ;
  val_tab = ecs_vec_int__ret_val_tab(vec_int) ;


  /* Libération du vecteur indexé */
  /*------------------------------*/

  ecs_vec_int__libere(vec_int) ;


  /*------------------------------------------------*/
  /* Création de la structure de type `ecs_champ_t' */
  /*------------------------------------------------*/

  BFT_MALLOC(this_champ, 1, ecs_champ_t);


  this_champ->nbr_elt = pos_nbr - 1 ;

  this_champ->typ_val = ECS_TYPE_ecs_int_t ;


  /* Allocation et affectation du nom du champ */
  /*-------------------------------------------*/

  if (nom != NULL) {
    BFT_MALLOC(this_champ->nom, strlen(nom) + 1, char) ;
    strcpy(this_champ->nom, nom) ;
  }
  else {
    this_champ->nom = NULL ;
  }


  /* Initialisation de la référence au descripteur de champ */
  /*--------------------------------------------------------*/

  this_champ->descr       = NULL ;


  /* Initialisation du lien pour les listes chaînées */
  /* des champs auxiliaires                          */
  /*-------------------------------------------------*/

  this_champ->l_champ_sui = NULL ;


  /* Statut du champ dans une transformation */
  /*  de type découpage ou recollement       */
  /*-----------------------------------------*/

  this_champ->statut_e    = ECS_CHAMP_STATUT_INDEFINI ;


  /* Affectation de la table des positions des valeurs */
  /*---------------------------------------------------*/

  if (pos_tab == NULL) {
    /* Les positions forment une REGLE de pas unité */
    this_champ->pos_pas = 1 ;
  }
  else {
    this_champ->pos_pas = 0 ;
  }

  this_champ->pos_tab = pos_tab ;
  ecs_champ__pos_en_regle(this_champ) ;


  /* Affectation de la table des valeurs */
  /*-------------------------------------*/

  this_champ->val_tab = val_tab ;


  return this_champ ;

}


/*----------------------------------------------------------------------------
 *  Fonction qui incrémente les valeurs d'un champ donné
 *   d'une constante donnée
 *----------------------------------------------------------------------------*/

void ecs_champ__incremente_val
(
       ecs_champ_t  * this_champ,
 const ecs_int_t      increment
)
{

  ecs_vec_int_t *vec_int ;


  /*xxxxxxxxxxxxxxxxxxxxxxxxxxx Instructions xxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxx*/


  vec_int = ecs_champ__initialise_vec_int(this_champ) ;

  ecs_vec_int__incremente_val_sgn(vec_int,
                                  increment) ;

  ecs_champ__transfere_vec_int(this_champ,
                               vec_int) ;


}


/*----------------------------------------------------------------------------
 *  Fonction réalisant la transformation d'un champ
 *   en appliquant directement le vecteur de transformation donné
 *   sur ses valeurs
 *----------------------------------------------------------------------------*/

void ecs_champ__transforme_val
(
       ecs_champ_t    *const this_champ,
       size_t                nbr_elt_new,
 const ecs_tab_int_t         vect_transf
)
{

  ecs_vec_int_t *vec_int ;


  /*xxxxxxxxxxxxxxxxxxxxxxxxxxx Instructions xxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxx*/


  vec_int = ecs_champ__initialise_vec_int(this_champ) ;


  ecs_vec_int__transforme_val(vec_int,
                              nbr_elt_new,
                              vect_transf) ;

  ecs_champ__transfere_vec_int(this_champ,
                               vec_int) ;


}


/*----------------------------------------------------------------------------
 *  Fonction réalisant la transformation d'un champ
 *   en appliquant directement le vecteur de transformation donné
 *   sur ses positions
 *----------------------------------------------------------------------------*/

void ecs_champ__transforme_pos
(
       ecs_champ_t    *const this_champ,
       size_t                nbr_elt_ref,
 const ecs_tab_int_t         vect_transf
)
{

  ecs_vec_int_t *vec_int ;


  /*xxxxxxxxxxxxxxxxxxxxxxxxxxx Instructions xxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxx*/


  vec_int = ecs_champ__initialise_vec_int(this_champ) ;


  ecs_vec_int__transforme_pos(vec_int,
                              nbr_elt_ref,
                              vect_transf) ;

  ecs_champ__transfere_vec_int(this_champ,
                               vec_int) ;


}


/*----------------------------------------------------------------------------
 *  Fonction réalisant la transformation d'un champ
 *   en appliquant directement le vecteur de transformation donné
 *   sur les valeurs associées à ses éléments
 *----------------------------------------------------------------------------*/

void ecs_champ__renumerote
(
       ecs_champ_t    *const this_champ,
 const ecs_tab_int_t         vect_transf
)
{

  ecs_vec_int_t *vec_int ;


  /*xxxxxxxxxxxxxxxxxxxxxxxxxxx Instructions xxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxx*/


  vec_int = ecs_champ__initialise_vec_int(this_champ) ;


  ecs_vec_int__renumerote(vec_int,
                          vect_transf) ;

  ecs_champ__transfere_vec_int(this_champ,
                               vec_int) ;


}


void ecs_champ__renumerote_et_sgn
(
       ecs_champ_t    *const this_champ,
 const ecs_tab_int_t         vect_transf,
 const ecs_tab_int_t         signe_elt
)
{

  ecs_vec_int_t *vec_int ;


  /*xxxxxxxxxxxxxxxxxxxxxxxxxxx Instructions xxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxx*/


  vec_int = ecs_champ__initialise_vec_int(this_champ) ;


  ecs_vec_int__renumerote_et_sgn(vec_int,
                                 vect_transf,
                                 signe_elt) ;

  ecs_champ__transfere_vec_int(this_champ,
                               vec_int) ;


}


/*----------------------------------------------------------------------------
 *  Fonction qui retourne le tableau des valeurs du champ donné,
 *   dimensionné au nombre de références distinctes du champ,
 *   (valeurs qui correspondent à des références)
 *----------------------------------------------------------------------------*/

ecs_tab_int_t ecs_champ__ret_reference
(
 ecs_champ_t  *const this_champ
)
{

  ecs_tab_int_t   tab_ref ;

  ecs_vec_int_t * vec_int ;


  /*xxxxxxxxxxxxxxxxxxxxxxxxxxx Instructions xxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxx*/


  vec_int = ecs_champ__initialise_vec_int(this_champ) ;


  tab_ref = ecs_vec_int__ret_reference(vec_int) ;


  ecs_champ__libere_vec_int(this_champ,
                            vec_int) ;


  return tab_ref ;


}


/*----------------------------------------------------------------------------
 *  Fonction qui remplace les références à des éléments
 *  en des références à d'autres éléments liés aux premiers
 *  par un champ de type "définition"
 *----------------------------------------------------------------------------*/

void ecs_champ__remplace_ref
(
 ecs_champ_t *const champ_rep,
 ecs_champ_t *const champ_def
)
{

  ecs_vec_int_t * vec_rep ;
  ecs_vec_int_t * vec_def ;


  /*xxxxxxxxxxxxxxxxxxxxxxxxxxx Instructions xxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxx*/


  vec_rep = ecs_champ__initialise_vec_int(champ_rep) ;
  vec_def = ecs_champ__initialise_vec_int(champ_def) ;


  ecs_vec_int__remplace_ref(vec_rep,
                            vec_def) ;


  ecs_champ__libere_vec_int(champ_def,
                            vec_def) ;

  ecs_champ__transfere_vec_int(champ_rep,
                               vec_rep) ;


}


/*----------------------------------------------------------------------------
 *  Fonction qui construit la  table de connectivité inverse
 *  "sous-éléments -> éléments"
 *----------------------------------------------------------------------------*/

ecs_champ_t * ecs_champ__inverse
(
 ecs_champ_t  *const champ_def_elt,
 size_t              nbr_sselt
)
{

  ecs_champ_t   * champ_sselt_def_elt ;

  ecs_vec_int_t * vec_def_elt ;
  ecs_vec_int_t * vec_sselt_def_elt ;


  /*xxxxxxxxxxxxxxxxxxxxxxxxxxx Instructions xxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxx*/


  assert(champ_def_elt != NULL) ;


  vec_def_elt = ecs_champ__initialise_vec_int(champ_def_elt) ;


  vec_sselt_def_elt = ecs_vec_int__inverse(vec_def_elt,
                                           nbr_sselt) ;


  ecs_champ__libere_vec_int(champ_def_elt,
                            vec_def_elt) ;

  champ_sselt_def_elt = ecs_champ__init_avec_vec_int(vec_sselt_def_elt,
                                                     ECS_CHAMP_NOM_DEFINIT) ;

  champ_sselt_def_elt->statut_e = ECS_CHAMP_STATUT_INDEFINI ;


  return  champ_sselt_def_elt ;


}


/*============================================================================
 *                              Fonctions privées
 *============================================================================*/
