/*
 * Eee Applet 
 * Copyright (C) 2008 Olivier Rolland <billl@users.sourceforge.net>
 *
 *  This library is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU General Public
 *  License as published by the Free Software Foundation; either
 *  version 2 of the License, or (at your option) any later version.
 *
 *  This library is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *  General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public
 *  License along with this library; if not, write to the Free
 *  Software Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

#ifdef HAVE_CONFIG_H
#include "config.h"
#endif

#include <string.h>
#include <stdlib.h>
#include <unistd.h>
#include <fcntl.h>

#include <glib/gi18n.h>

#include <gtk/gtkmain.h>
#include <gtk/gtkicontheme.h>
#include <gtk/gtkaboutdialog.h>
#include <gtk/gtkstatusicon.h>
#include <gtk/gtkuimanager.h>
#include <gtk/gtkactiongroup.h>
#include <gtk/gtkradioaction.h>
#include <gtk/gtktoggleaction.h>
#include <gtk/gtkstock.h>

#include <gconf/gconf-client.h>

#include <gksu.h>

#ifdef HAVE_HOTKEYS
#include "eee-monitor-hotkeys.h"
#endif

#include "eee-monitor-procfs.h"
#include "eee-procfs.h"
#include "eee-gconf.h"
#include "eee-pref.h"

#define EEE_TYPE_APPLET            (eee_applet_get_type ())
#define EEE_APPLET(obj)            (G_TYPE_CHECK_INSTANCE_CAST ((obj), EEE_TYPE_APPLET, EeeApplet))
#define EEE_APPLET_CLASS(klass)    (G_TYPE_CHECK_CLASS_CAST((klass), EEE_TYPE_APPLET, EeeAppletClass))
#define EEE_IS_APPLET(obj)         (G_TYPE_CHECK_INSTANCE_TYPE ((obj), EEE_TYPE_APPLET))
#define EEE_IS_APPLET_CLASS(klass) (G_TYPE_CHECK_CLASS_TYPE ((klass), EEE_TYPE_APPLET))
#define EEE_APPLET_GET_CLASS(obj)  (G_TYPE_INSTANCE_GET_CLASS ((obj), EEE_TYPE_APPLET, EeeAppletClass))

#define EEE_APPLET_ICON "pixmaps/eee-applet.png"

typedef struct
{
  GObject parent_instance;

  GtkStatusIcon *icon;
  GdkPixbuf *pixbuf;

  GtkWidget *popup;

#ifdef HAVE_HOTKEYS
  EeeMonitor *hotkeys;
#endif
  EeeMonitor *procfs;

  GtkAction *wlan_action;
  GtkAction *cardr_action;
  GtkAction *camera_action;
  GtkAction *clock_actions[3];
  GtkAction *fan_actions[6];
} EeeApplet;

typedef struct
{
  GObjectClass parent_class;
} EeeAppletClass;

static GType eee_applet_get_type (void) G_GNUC_CONST;
static void  eee_applet_dispose  (GObject *object);

GConfClient *client;

static const gchar *ui_popup =
"<ui>"
"  <popup name='EeePopup' action='EeePopupActions'>"
"    <placeholder name='AsusItems'/>"
"    <placeholder name='EeeItems'/>"
"    <separator />"
"    <menuitem action='Preferences'/>"
"    <separator />"
"    <menuitem action='About'/>"
"  </popup>"
"</ui>";

static const gchar *ui_asus =
"<ui>"
"  <popup name='EeePopup' action='EeePopupActions'>"
"    <placeholder name='AsusItems'>"
"      <menuitem action='Camera'/>"
"      <menuitem action='Wlan'/>"
"      <menuitem action='Cardr'/>"
"    </placeholder>"
"  </popup>"
"</ui>";

static const gchar *ui_eee =
"<ui>"
"  <popup name='EeePopup' action='EeePopupActions'>"
"    <placeholder name='EeeItems'>"
"      <menu action='FanMenu'>"
"        <menuitem action='FanAutomatic'/>"
"        <menuitem action='FanOff'/>"
"        <menuitem action='FanNormal'/>"
"        <menuitem action='FanFast'/>"
"        <menuitem action='FanFaster'/>"
"        <menuitem action='FanFastest'/>"
"      </menu>"
"      <menu action='ClockMenu'>"
"        <menuitem action='ClockNormal'/>"
"        <menuitem action='ClockHigher'/>"
"        <menuitem action='ClockHighest'/>"
"      </menu>"
"    </placeholder>"
"  </popup>"
"</ui>";

static const GtkActionEntry entries[] =
{
  { "ClockMenu",     NULL,                  N_("Overclock"), NULL },
  { "FanMenu",       NULL,                  N_("Fan"),       NULL },
  { "Preferences",   GTK_STOCK_PREFERENCES, NULL,            NULL },
  { "About",         GTK_STOCK_ABOUT,       NULL,            NULL }
};

static const GtkToggleActionEntry toggle_entries[] =
{
  { "Camera",    NULL, N_("Enable camera"),       NULL, NULL, NULL, FALSE },
  { "Wlan",      NULL, N_("Enable wlan"),         NULL, NULL, NULL, FALSE },
  { "Cardr",     NULL, N_("Enable card reeader"), NULL, NULL, NULL, FALSE },
};

static const GtkRadioActionEntry clock_entries[] =
{
  { "ClockNormal",  NULL, N_("Normal"),  NULL, NULL, 0 },
  { "ClockHigher",  NULL, N_("Higher"),  NULL, NULL, 1 },
  { "ClockHighest", NULL, N_("Highest"), NULL, NULL, 2 }
};

static const GtkRadioActionEntry fan_entries[] =
{
  { "FanAutomatic", NULL, N_("Automatic"), NULL, NULL, 0 },
  { "FanOff",       NULL, N_("Off"),       NULL, NULL, 1 },
  { "FanNormal",    NULL, N_("Normal"),    NULL, NULL, 2 },
  { "FanFast",      NULL, N_("Fast"),      NULL, NULL, 3 },
  { "FanFaster",    NULL, N_("Faster"),    NULL, NULL, 4 },
  { "FanFastest",   NULL, N_("Fastest"),   NULL, NULL, 5 }
};

static void
eee_applet_write_string (const gchar *filename, const gchar *data, guint len)
{
  gint fd;

  fd = open (filename, O_WRONLY);
  if (fd > 0)
  {
    write (fd, data, len);
    write (fd, "\n", 1);
    close (fd);
  }
  else
  {
    gchar *cmd;

    cmd = g_strdup_printf ("echo \"%s\" > \"%s\"", data, filename);
    gksu_su (cmd, NULL);
    g_free (cmd);
  }
}

static void
eee_applet_write_int (const gchar *filename, gint value)
{
  gchar *buffer;

  buffer = g_strdup_printf ("%d", value);
  eee_applet_write_string (filename, buffer, strlen (buffer));
  g_free (buffer);
}

static gboolean
eee_applet_write_fsb (gint speed)
{
  gchar *cmd;

  cmd = g_strdup_printf (DATADIR "/eee-applet/eee-fsb.sh %d", speed);
  gksu_su (cmd, NULL);
  g_free (cmd);

  return FALSE;
}

static void
eee_applet_pref_activate_cb (EeeApplet *applet)
{
  static GtkWidget *dialog = NULL;

  if (!dialog)
  {
    dialog = eee_pref_dialog_new ();
    gtk_window_set_position (GTK_WINDOW (dialog), GTK_WIN_POS_CENTER);
    gtk_window_set_gravity (GTK_WINDOW (dialog), GDK_GRAVITY_CENTER);

    g_signal_connect (dialog, "response",
        G_CALLBACK (gtk_widget_destroy), NULL);
    g_signal_connect (dialog, "destroy",
        G_CALLBACK (gtk_widget_destroyed), &dialog);
  }

  gtk_window_present (GTK_WINDOW (dialog));
}

static gboolean
eee_applet_size_changed_cb (EeeApplet *applet, gint size)
{
  GdkPixbuf *pixbuf;

  pixbuf = gdk_pixbuf_scale_simple (applet->pixbuf, size, size, GDK_INTERP_HYPER);
  gtk_status_icon_set_from_pixbuf (applet->icon, pixbuf);
  g_object_unref (pixbuf);

  return TRUE;
}

static void
eee_applet_popup_menu_cb (EeeApplet *applet, guint button, guint activate_time)
{
  gtk_menu_popup (GTK_MENU (applet->popup), NULL, NULL,
      gtk_status_icon_position_menu, applet->icon, button, activate_time);
}

static void
eee_applet_camera_activate_cb (EeeApplet *applet, GtkAction *action)
{
  gboolean active;

  active = gtk_toggle_action_get_active (GTK_TOGGLE_ACTION (action));
  eee_applet_write_int (EEE_PROC_CAMERA, active ? 1 : 0);
}

static void
eee_applet_cardr_activate_cb (EeeApplet *applet, GtkAction *action)
{
  gboolean active;

  active = gtk_toggle_action_get_active (GTK_TOGGLE_ACTION (action));
  eee_applet_write_int (EEE_PROC_CARDR, active ? 1 : 0);
}

static void
eee_applet_wlan_activate_cb (EeeApplet *applet, GtkAction *action)
{
  gboolean active;

  active = gtk_toggle_action_get_active (GTK_TOGGLE_ACTION (action));
  eee_applet_write_int (EEE_PROC_WLAN, active ? 1 : 0);
}

static void
eee_applet_clock_changed_cb (EeeApplet *applet, GtkAction *current, GtkAction *action)
{
  static const guint clock[] = { 70, 85, 100 };
  guint i, old_clock = 70;

  for (i = 0; i < G_N_ELEMENTS (clock_entries); i++)
    if (current == applet->clock_actions[i])
      break;

  if (applet->procfs)
    old_clock = eee_monitor_procfs_get_fsb_speed (EEE_MONITOR_PROCFS (applet->procfs));

  if (clock[i] != old_clock)
  {
    gconf_client_set_int (client, EEE_GCONF_CLOCK, i, NULL);

    if (clock[i] - old_clock == 30)
    {
      eee_applet_write_fsb (85);
      g_timeout_add (1000, (GSourceFunc) eee_applet_write_fsb, GINT_TO_POINTER (100));
    }
    else if (old_clock - clock[i] == 30)
    {
      eee_applet_write_fsb (85);
      g_timeout_add (1000, (GSourceFunc) eee_applet_write_fsb, GINT_TO_POINTER (70));
    }
    else
      eee_applet_write_fsb (clock[i]);
  }
}

static void
eee_applet_fan_changed_cb (EeeApplet *applet, GtkAction *current, GtkAction *action)
{
  static const guint speed[] = { 0, 0, 40, 65, 80, 100 };
  guint i;

  for (i = 0; i < G_N_ELEMENTS (fan_entries); i++)
    if (current == applet->fan_actions[i])
      break;

  gconf_client_set_int (client, EEE_GCONF_FAN, i, NULL);

  if (i == 0)
    eee_applet_write_int (EEE_PROC_FAN_MANUAL, 0);
  else
  {
    eee_applet_write_int (EEE_PROC_FAN_MANUAL, 1);
    eee_applet_write_int (EEE_PROC_FAN_SPEED, speed[i]);
  }
}

static void
eee_applet_preferences_activate_cb (EeeApplet *applet, GtkAction *action)
{
}

static void
eee_applet_about_activate_cb (EeeApplet *applet, GtkAction *action)
{
  const gchar *authors[] = 
  { 
    "Olivier Rolland <billl@users.sourceforge.net>", 
    NULL 
  };

  gchar *translator_credits = _("translator-credits");

  if (g_str_equal (translator_credits, "translator-credits"))
    translator_credits = NULL;

  gtk_show_about_dialog (NULL,
      "name", PACKAGE_NAME,
      "version", PACKAGE_VERSION,
      "comments", _("A systray applet for monitoring Eee Pc specific info"),
      "copyright", "(c) 2008 Olivier Rolland",
      "website", "http://eee-applet.sourceforge.net",
      "translator-credits", translator_credits,
      "logo", applet->pixbuf,
      "authors", authors,
      NULL);
}

static void
eee_applet_monitor_procfs_online_cb (EeeApplet *applet, EeeMonitorProcfs *monitor)
{
  gboolean online;

  online = eee_monitor_procfs_is_online (EEE_PROCFS_EEE);
#ifdef G_ENABLE_DEBUG
  g_debug ("eee: %s", online ? "online" : "offline");
#endif

  online = eee_monitor_procfs_is_online (EEE_PROCFS_ASUS);
#ifdef G_ENABLE_DEBUG
  g_debug ("asus_acpi: %s", online ? "online" : "offline");
#endif

  gtk_action_set_visible (applet->camera_action, online);
  gtk_action_set_visible (applet->cardr_action, online);
  gtk_action_set_visible (applet->wlan_action, online);
}

static void
eee_applet_monitor_procfs_changed_cb (EeeApplet *applet, EeeMonitorProcfs *monitor)
{
  gboolean active;
  gint value;

  if (eee_monitor_procfs_is_online (EEE_PROCFS_EEE))
  {
    /* FSB */
    value = eee_monitor_procfs_get_fsb_speed (monitor);

    g_signal_handlers_block_by_func (applet->clock_actions[0], eee_applet_clock_changed_cb, applet);
    if (value < 80)
      gtk_radio_action_set_current_value (GTK_RADIO_ACTION (applet->clock_actions[0]), 0);
    else if (value < 90)
      gtk_radio_action_set_current_value (GTK_RADIO_ACTION (applet->clock_actions[0]), 1);
    else
      gtk_radio_action_set_current_value (GTK_RADIO_ACTION (applet->clock_actions[0]), 2);
    g_signal_handlers_unblock_by_func (applet->clock_actions[0], eee_applet_clock_changed_cb, applet);

    /* FAN */
    active = eee_monitor_procfs_get_fan_manual (monitor);
    g_signal_handlers_block_by_func (applet->fan_actions[0], eee_applet_fan_changed_cb, applet);
    if (!active)
      gtk_radio_action_set_current_value (GTK_RADIO_ACTION (applet->fan_actions[0]), 0);
    else
    {
      value = eee_monitor_procfs_get_fan_speed (monitor);
      if (value == 0)
        gtk_radio_action_set_current_value (GTK_RADIO_ACTION (applet->fan_actions[0]), 1);
      else if (value <= 40)
        gtk_radio_action_set_current_value (GTK_RADIO_ACTION (applet->fan_actions[0]), 2);
      else if (value <= 65)
        gtk_radio_action_set_current_value (GTK_RADIO_ACTION (applet->fan_actions[0]), 3);
      else if (value <= 80)
        gtk_radio_action_set_current_value (GTK_RADIO_ACTION (applet->fan_actions[0]), 4);
      else
        gtk_radio_action_set_current_value (GTK_RADIO_ACTION (applet->fan_actions[0]), 5);

    }
    g_signal_handlers_unblock_by_func (applet->fan_actions[0], eee_applet_fan_changed_cb, applet);
  }

  if (eee_monitor_procfs_is_online (EEE_PROCFS_ASUS))
  {
    gchar *tip;

    /* Camera */
    active = eee_monitor_procfs_get_camera (monitor);
    g_signal_handlers_block_by_func (applet->camera_action, eee_applet_camera_activate_cb, applet);
    gtk_toggle_action_set_active (GTK_TOGGLE_ACTION (applet->camera_action), active);
    g_signal_handlers_unblock_by_func (applet->camera_action, eee_applet_camera_activate_cb, applet);

    /* Wlan */
    active = eee_monitor_procfs_get_wlan (monitor);
    g_signal_handlers_block_by_func (applet->wlan_action, eee_applet_wlan_activate_cb, applet);
    gtk_toggle_action_set_active (GTK_TOGGLE_ACTION (applet->wlan_action), active);
    g_signal_handlers_unblock_by_func (applet->wlan_action, eee_applet_wlan_activate_cb, applet);

    /* Card reader */
    active = eee_monitor_procfs_get_card_reader (monitor);
    g_signal_handlers_block_by_func (applet->cardr_action, eee_applet_cardr_activate_cb, applet);
    gtk_toggle_action_set_active (GTK_TOGGLE_ACTION (applet->cardr_action), active);
    g_signal_handlers_unblock_by_func (applet->cardr_action, eee_applet_cardr_activate_cb, applet);

    tip = g_strdup_printf ("%s: %d%%\n%s: %d°C\n%s: %d%%",
        _("Brightness"), eee_monitor_procfs_get_brightness (monitor) * 100 / 15,
        _("Temperature"), eee_monitor_procfs_get_temperature (monitor),
        _("Fan speed"), eee_monitor_procfs_get_fan_speed (monitor));
    gtk_status_icon_set_tooltip (applet->icon, tip);
    g_free (tip);
  }
}

static void
eee_applet_can_disable_notified (GConfClient *client, guint cnxn_id,
    GConfEntry *entry, GtkAction *action)
{
  gtk_action_set_sensitive (action,
      gconf_value_get_bool (entry->value));
}

static void
eee_applet_create_popup (EeeApplet *applet)
{
  GConfClient *client;

  GtkAction *action;
  GtkActionGroup *action_group;
  GtkUIManager *ui_manager;

  gint i;
  gboolean active;

  client = gconf_client_get_default ();
  gconf_client_add_dir (client, EEE_GCONF_ROOT, GCONF_CLIENT_PRELOAD_ONELEVEL, NULL);

  action_group = gtk_action_group_new ("PopupActions");
  gtk_action_group_set_translation_domain (action_group, GETTEXT_PACKAGE);

  gtk_action_group_add_actions (action_group, entries,
      G_N_ELEMENTS (entries), NULL);
  gtk_action_group_add_toggle_actions (action_group, toggle_entries,
      G_N_ELEMENTS (toggle_entries), NULL);
  gtk_action_group_add_radio_actions (action_group, clock_entries,
      G_N_ELEMENTS (clock_entries), 0, NULL, NULL);
  gtk_action_group_add_radio_actions (action_group, fan_entries,
      G_N_ELEMENTS (fan_entries), 0, NULL, NULL);

  ui_manager = gtk_ui_manager_new ();

  gtk_ui_manager_insert_action_group (ui_manager, action_group, 0);
  gtk_ui_manager_add_ui_from_string (ui_manager, ui_popup, -1, NULL);

  action = gtk_action_group_get_action (action_group, "Preferences");
  g_signal_connect_swapped (action, "activate",
      G_CALLBACK (eee_applet_pref_activate_cb), applet);

  /*
   * asus_acpi module
   */

  gtk_ui_manager_add_ui_from_string (ui_manager, ui_asus, -1, NULL);

  applet->camera_action = gtk_action_group_get_action (action_group, "Camera");
  gtk_action_set_sensitive (applet->camera_action,
      gconf_client_get_bool (client, EEE_GCONF_CAN_DISABLE_CAMERA, NULL));
  gconf_client_notify_add (client, EEE_GCONF_CAN_DISABLE_CAMERA,
      (GConfClientNotifyFunc) eee_applet_can_disable_notified, applet->camera_action, NULL, NULL);

  applet->cardr_action = gtk_action_group_get_action (action_group, "Cardr");
  gtk_action_set_sensitive (applet->cardr_action,
      gconf_client_get_bool (client, EEE_GCONF_CAN_DISABLE_CARDR, NULL));
  gconf_client_notify_add (client, EEE_GCONF_CAN_DISABLE_CARDR,
      (GConfClientNotifyFunc) eee_applet_can_disable_notified, applet->cardr_action, NULL, NULL);

  applet->wlan_action = gtk_action_group_get_action (action_group, "Wlan");
  gtk_action_set_sensitive (applet->wlan_action,
      gconf_client_get_bool (client, EEE_GCONF_CAN_DISABLE_WLAN, NULL));
  gconf_client_notify_add (client, EEE_GCONF_CAN_DISABLE_WLAN,
      (GConfClientNotifyFunc) eee_applet_can_disable_notified, applet->wlan_action, NULL, NULL);


  /*
   * eee module
   */

  gtk_ui_manager_add_ui_from_string (ui_manager, ui_eee, -1, NULL);
/*
  gconf_client_notify_add (client, EEE_GCONF_CAN_OVERCLOCK,
      (GConfClientNotifyFunc) eee_applet_can_disable_notified, applet->clock_menu_action, NULL, NULL);
*/
  active = gconf_client_get_bool (client, EEE_GCONF_CAN_OVERCLOCK, NULL);
  for (i = 0; i < G_N_ELEMENTS (clock_entries); i ++)
  {
    applet->clock_actions[i] = gtk_action_group_get_action (action_group, clock_entries[i].name);
    gtk_action_set_sensitive (applet->clock_actions[i], active);
  }
/*
  gconf_client_notify_add (client, EEE_GCONF_CAN_CHANGE_FAN_SPEED,
      (GConfClientNotifyFunc) eee_applet_can_disable_notified, applet->fan_menu_action, NULL, NULL);
*/
  active = gconf_client_get_bool (client, EEE_GCONF_CAN_CHANGE_FAN_SPEED, NULL);
  for (i = 0; i < G_N_ELEMENTS (fan_entries); i ++)
  {
    applet->fan_actions[i] = gtk_action_group_get_action (action_group, fan_entries[i].name);
    gtk_action_set_sensitive (applet->fan_actions[i], active);
  }

  action = gtk_action_group_get_action (action_group, "Preferences");
  g_signal_connect_swapped (action, "activate",
      G_CALLBACK (eee_applet_preferences_activate_cb), applet);

  action = gtk_action_group_get_action (action_group, "About");
  g_signal_connect_swapped (action, "activate",
      G_CALLBACK (eee_applet_about_activate_cb), applet);

  applet->popup = gtk_ui_manager_get_widget (ui_manager, "/EeePopup");
}

G_DEFINE_TYPE (EeeApplet, eee_applet, G_TYPE_OBJECT)

static void
eee_applet_init (EeeApplet *applet)
{
  gint active;

  applet->icon = gtk_status_icon_new ();
  gtk_status_icon_set_tooltip (applet->icon, "Eee Applet");

  g_signal_connect_swapped (applet->icon, "size-changed",
      G_CALLBACK (eee_applet_size_changed_cb), applet);

  g_signal_connect_swapped (applet->icon, "popup-menu",
      G_CALLBACK (eee_applet_popup_menu_cb), applet);

  applet->pixbuf = gdk_pixbuf_new_from_file (DATADIR "/" EEE_APPLET_ICON, NULL);

  eee_applet_create_popup (applet);

#ifdef HAVE_HOTKEYS
  if (gconf_client_get_bool (client, EEE_GCONF_HOTKEYS, NULL))
  {
    applet->hotkeys = eee_monitor_hotkeys_new ();
    eee_monitor_run (applet->hotkeys);
  }
#endif

  /*
   * asus_acpi module
   */

  g_signal_connect_swapped (applet->camera_action, "activate",
      G_CALLBACK (eee_applet_camera_activate_cb), applet);

  g_signal_connect_swapped (applet->cardr_action, "activate",
      G_CALLBACK (eee_applet_cardr_activate_cb), applet);

  g_signal_connect_swapped (applet->wlan_action, "activate",
      G_CALLBACK (eee_applet_wlan_activate_cb), applet);

  /*
   * eee module
   */

  g_signal_connect_swapped (applet->clock_actions[0], "changed",
      G_CALLBACK (eee_applet_clock_changed_cb), applet);

  active = gconf_client_get_int (client, EEE_GCONF_CLOCK, NULL);
  if (active >= 0 && active < G_N_ELEMENTS (clock_entries))
    gtk_radio_action_set_current_value (GTK_RADIO_ACTION (applet->clock_actions[0]), active);

  g_signal_connect_swapped (applet->fan_actions[0], "changed",
      G_CALLBACK (eee_applet_fan_changed_cb), applet);

  active = gconf_client_get_int (client, EEE_GCONF_FAN, NULL);
  if (active >= 0 && active < G_N_ELEMENTS (fan_entries))
    gtk_radio_action_set_current_value (GTK_RADIO_ACTION (applet->fan_actions[0]), active);

  applet->procfs = eee_monitor_procfs_new ();
  g_signal_connect_swapped (applet->procfs, "online",
      G_CALLBACK (eee_applet_monitor_procfs_online_cb), applet);
  eee_monitor_run (applet->procfs);

  g_signal_connect_swapped (applet->procfs, "changed",
      G_CALLBACK (eee_applet_monitor_procfs_changed_cb), applet);
}

static void
eee_applet_class_init (EeeAppletClass *klass)
{
  GObjectClass *object_class = G_OBJECT_CLASS (klass);

  object_class->dispose = eee_applet_dispose;
}

static void
eee_applet_dispose (GObject *object)
{
  EeeApplet *applet;

  applet = EEE_APPLET (object);

  if (applet->icon)
  {
    g_object_unref (applet->icon);
    applet->icon = NULL;
  }

  if (applet->pixbuf)
  {
    g_object_unref (applet->pixbuf);
    applet->pixbuf = NULL;
  }

#ifdef HAVE_HOTKEYS
  if (applet->hotkeys)
  {
    g_object_unref (applet->hotkeys);
    applet->hotkeys = NULL;
  }
#endif

  if (applet->procfs)
  {
    g_object_unref (applet->procfs);
    applet->procfs = NULL;
  }

  G_OBJECT_CLASS (eee_applet_parent_class)->finalize (object);
}

EeeApplet *
eee_applet_new (void)
{
  return g_object_new (EEE_TYPE_APPLET, NULL);
}

int
main (int argc, char *argv[])
{
  EeeApplet *applet;
  GMainLoop *loop = NULL;

#ifdef ENABLE_NLS
  bindtextdomain (GETTEXT_PACKAGE, LOCALEDIR);
  bind_textdomain_codeset (GETTEXT_PACKAGE, "UTF-8");
  textdomain (GETTEXT_PACKAGE);
#endif

  gtk_init (&argc, &argv);

  g_set_application_name (_("Eee Applet"));

  client = gconf_client_get_default ();
  gconf_client_add_dir (client, EEE_GCONF_ROOT, GCONF_CLIENT_PRELOAD_ONELEVEL, NULL);

  applet = eee_applet_new ();
  if (applet == NULL)
    return EXIT_FAILURE;

  gtk_window_set_default_icon (applet->pixbuf);

  loop = g_main_loop_new (NULL, FALSE);
  g_main_loop_run (loop);

  g_object_unref (G_OBJECT (applet));

  return EXIT_SUCCESS;
}

