/* Copyright (C) 1999 - 2001 Chris Vine

This program is distributed under the General Public Licence, version 2.
For particulars of this and relevant disclaimers see the file
COPYRIGHT distributed with the source files.

*/

// pipes.cpp

#include <iostream>
#include <cstdlib>
#include <cstring>
#include <limits.h>
#include <fcntl.h>

#include "pipes.h"

Pipe_fifo::Pipe_fifo(Fifo_mode mode): write_blocking_mode(block) {
    int file_pipes[2];
    if (pipe(file_pipes) == -1) {
        std::cerr << "Cannot create pipe" << std::endl;
	exit(PIPE_ERROR);
    }
    else {
        read_fd = file_pipes[0];
	write_fd = file_pipes[1];
	if (mode == non_block) {
	    int fdflags = fcntl(read_fd, F_GETFL);
	    fdflags |= O_NONBLOCK;
	    fcntl(read_fd, F_SETFL, fdflags);
	    read_blocking_mode = non_block;
	}
	else read_blocking_mode = block;
    }
}

Pipe_fifo::Pipe_fifo(void): read_fd(-1), write_fd(-1),
			    read_blocking_mode(block), write_blocking_mode(block) {}

int Pipe_fifo::open(Fifo_mode mode) {
    int file_pipes[2];
    int return_val = pipe(file_pipes);
    if (return_val != -1) {
        read_fd = file_pipes[0];
	write_fd = file_pipes[1];
	if (mode == non_block) {
	    int fdflags = fcntl(read_fd, F_GETFL);
	    fdflags |= O_NONBLOCK;
	    fcntl(read_fd, F_SETFL, fdflags);
	    read_blocking_mode = non_block;
	}
	else read_blocking_mode = block;
    }
    return return_val;
}

void Pipe_fifo::close(void) {
    make_writeonly();
    make_readonly();
    read_blocking_mode = block;
    write_blocking_mode = block;
}

ssize_t Pipe_fifo::read(char* buffer, size_t max_number) {
    ssize_t result;
    if (read_fd == -1) result = -1;
    else if ((result = ::read(read_fd, buffer, max_number)) < 0) result = 0;  // call unix read() in file scope
    return result;
}      

int Pipe_fifo::read(void) {
    unsigned char item;
    if (read_fd != -1 && ::read(read_fd, &item, 1) > 0) return (int)item;   // call unix read() in file scope
    else return -1;
}

ssize_t Pipe_fifo::write(const char* buffer) {
    size_t length = strlen(buffer);
    if ((write_blocking_mode == block && length > PIPE_BUF) || !write_fd) return -1;
    return write(buffer, length);
}

ssize_t Pipe_fifo::write(const char* buffer, size_t max_number) {
    ssize_t result;
    if (write_fd == -1) result = -1;
    else if ((result = ::write(write_fd, buffer, max_number)) < 0) result = 0;  // call unix write() in file scope
    return result;
}

void Pipe_fifo::make_writeonly(void) {
    if (read_fd != -1) {
        ::close(read_fd); // call unix close() in file scope
	read_fd = -1;
    }
}

void Pipe_fifo::make_readonly(void) {
    if (write_fd != -1) {
        ::close(write_fd); // call unix close() in file scope
	write_fd = -1;
    }
}

int Pipe_fifo::make_write_non_block(void) {
    if (read_blocking_mode == non_block) {
        int fdflags = fcntl(write_fd, F_GETFL);
	fdflags |= O_NONBLOCK;
	fcntl(write_fd, F_SETFL, fdflags);
	return 0;
    }
    return -1;
}

int Pipe_fifo::connect_to_stdin(void) {

  // check preconditions
  if (read_fd == -1) return -1;

  // first we need to duplicate the read file descriptor onto stdin
  ::close(0); // stdin
  dup(read_fd); // stdin will now read from the pipe
  make_writeonly();    // this will close the original read file descriptor
                       // but leave stdin unaffected
  make_readonly();     // since the pipe is unidirectional, we can also close the write fd
  
  return 0;
}

int Pipe_fifo::connect_to_stdout(void) {

  // check preconditions
  if (write_fd == -1) return -1;

  // first we need to duplicate the write file descriptor onto stdout
  ::close(1); // stdout
  dup(write_fd); // stdout will now write to the pipe
  make_readonly();     // this will close the original write file descriptor
                       // but leave stdout unaffected
  make_writeonly();    // since the pipe is unidirectional, we can also close the read fd
  
  return 0;
}

int Pipe_fifo::connect_to_stderr(void) {

  // check preconditions
  if (write_fd == -1) return -1;

  // first we need to duplicate the write file descriptor onto stderr
  ::close(2); // stderr
  dup(write_fd); // stderr will now write to the pipe
  make_readonly();     // this will close the original write file descriptor
                       // but leave stderr unaffected
  make_writeonly();    // since the pipe is unidirectional, we can also close the read fd
  
  return 0;
}
