/* Copyright (C) 2006 Chris Vine

This program is distributed under the General Public Licence, version 2.
For particulars of this and relevant disclaimers see the file
COPYING distributed with the source files.

*/

#ifndef PRINT_MANAGER_H
#define PRINT_MANAGER_H

#include "prog_defs.h"

#include <gtk/gtkversion.h>
// we only compile this file if GTK+ version is 2.10 or higher
#if GTK_CHECK_VERSION(2,10,0)

#include <string>

#include <gtk/gtkwindow.h>
#include <gtk/gtkdialog.h>
#include <gtk/gtkprinter.h>
#include <gtk/gtkprintsettings.h>
#include <gtk/gtkpagesetup.h>
#include <gtk/gtkprintjob.h>

#include <sigc++/sigc++.h>

#include "utils/window.h"
#include "utils/notifier.h"
#include "utils/mutex.h"
#include "utils/gobj_handle.h"
#include "utils/intrusive_ptr.h"

/* The PrintManager class prints a file using the GTK+ printer
   interface provided with GTK+ 2.10.0 onwards.  To obtain a
   PrintManager object, call PrintManager::create_manager() in the
   thread in which the main Glib event loop runs.
   PrintManager::set_filename(), which passes the name of the file to
   be printed, can be called in any thread.  To print a page, emit
   call PrintManager::print(), which may also be done in any thread.
   The PrintManager class will unlink() the file to be printed when it
   has been finished with unless false is passed as the second
   argument of PrintManager::create_manager().

   Once PrintManager::print() has been called, the PrintManager class
   owns a reference to itself and so manages its own lifetime - so
   once that method has been called it doesn't matter if the
   IntrusivePtr returned by PrintManager::create_manager() goes out of
   scope (however, once PrintManager::print() has been called, the
   object will not be deleted until both printing has completed or
   failed AND the IntrusivePtr returned by
   PrintManager::create_manager() has gone out of scope or has been
   reset()).
*/

namespace { // we put the functions in anonymous namespace in dialogs.cpp
            // so they are not exported at link time
namespace PrintDialogCB {
  extern "C" void print_selected(GtkDialog*, int, void*);
}
}   // anonymous namespace

class PrintDialog: public WinBase {
protected:
  virtual void on_delete_event(void);
public:
  friend void PrintDialogCB::print_selected(GtkDialog*, int, void*);

  sigc::signal0<void> accepted;
  sigc::signal0<void> rejected;
  GtkPrinter* get_printer(void) const;
  GobjHandle<GtkPrintSettings> get_settings(void) const;
  GtkPageSetup* get_page_setup(void) const;
  PrintDialog(GtkWindow* parent_p);
};

namespace { // we put the functions in anonymous namespace in dialogs.cpp
            // so they are not exported at link time
namespace PrintManagerCB {
  extern "C" void print_job_complete(GtkPrintJob*, void*, GError*);
}
}   // anonymous namespace

class PrintManger;

class PrintManager: public sigc::trackable, public IntrusiveLockCounter {
  Thread::Mutex filename_mutex;
  GtkWindow* parent_p;
  bool manage;
  std::string filename;
  PrintDialog* dialog_p;
  Notifier print_notifier;

  void show_dialog(void);
  void print_file(void);
  PrintManager(void) {}
public:
  friend void PrintManagerCB::print_job_complete(GtkPrintJob*, void*, GError*);
  static IntrusivePtr<PrintManager> create_manager(GtkWindow* parent = 0, bool manage_file = true);
  void set_filename(const char*);
  void print(void);
  ~PrintManager(void);
};

#endif // GTK_CHECK_VERSION
#endif // PRINT_MANAGER_H
