%{
/*
 *   TTKCruncher Copyright (c) 2002, 2003 Coleebris
 *   File: ttkcrunch.l
 *   $Revision: 1.6 $
 *
 *   This file is part of TTKCruncher.
 *
 *   TTKCruncher is free software; you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation; either version 2 of the License, or
 *   (at your option) any later version.
 *
 *   TTKCruncher is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *   GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with TTKCruncher; if not, write to the Free Software
 *   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 */
	
/*
** Author : Cyril Jean <cjean@coleebris.com>
** Version: 0.2
*/
	
#include <stdlib.h>
#include <string.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <unistd.h>
#include <dirent.h>

#define VERSION "0.2"
	
#define	FILE_STACK_SIZE	1024
#define F_MALLOC(ptr,size)\
        if ( (ptr = malloc(size)) == 0 )\
        {\
        fprintf(stderr,"err malloc (%s,%d)\n",__FILE__,__LINE__);\
        exit(1);\
        }\
        else\
        memset(ptr,0,size);

 char *get_file_from_fullpath(char *path);
 char *get_final_path(char *inpath, char *output_path, char *output_files_name_add, char *output_extenstion);
 char *get_ext_from_path(char *inpath);
 char *add_to_file_name(char *output_files_name_add, char *file);
 char *get_file_name_without_ext(char *inpath);
 char *concat_all(char *end_path, char *end_file, char *end_ext);
 int   show_gain_info(char *in, char *out);
 int   crunchcrunch(char *in, char* out);
 void  parse_directory(char *to_dir, char *in_ext, char *output_path, char *output_files_name_add, char *output_extenstion);
 char *add_level_to_path(char *path, char *level);
 int   create_directory_if_not(char *path);
 int max_gain, min_gain, verbose, full_size, comp_size;
%}

%option noyywrap

%x mlcomment slcomment quote dquote

%%
"/*" {
  BEGIN (mlcomment);
}

"//" {
  BEGIN (slcomment);
}

\" {
  fprintf(yyout, "%s", yytext);
  BEGIN (dquote);
}

\' {
  fprintf(yyout, "%s", yytext);
  BEGIN (quote);
}

[ \n\t]?(return|else|var|function|delete|in|new)[ \n\t]?  {
  ECHO;
}

<mlcomment>[^*]*
<mlcomment>"*"+[^*/\n]*
<mlcomment>"*"+"/" {
  BEGIN (INITIAL);
}

<slcomment>[^\n]*
<slcomment>\n {
  BEGIN (INITIAL);
}

<quote>[^\']* {
  fprintf(yyout, "%s", yytext);
}

<quote>\' {
  fprintf(yyout, "%s", yytext);
  BEGIN (INITIAL);
}

<dquote>[^"]*|[^\\"] {
  fprintf(yyout, "%s", yytext);
}

<dquote>[\\]["] {
  fprintf(yyout, "%s", yytext);
}

<dquote>["] {
  fprintf(yyout, "%s", yytext);
  BEGIN (INITIAL);
}

[\t\n]+
[ ]+
%%
int main(int ac, char **av)
{
	char usage[]="\nUsage: ttkcrunch input_files|[input_path -r [-i input_extension]] [-o output_path] [-a output_suffix] [-e output_extension] [-v] [-h]\n\n";
	char help[]="TTKCruncher Options :\n\t-o arg\tDefine output root path\n\t-a arg\tDefine suffix to add on output file name\n\t-e arg\tDefine output file extension\n\t-r\tRecursive mode\n\t-i arg\tSpecify input files extension, if not set, all files will be proceeded (only in recursive mode for now)\n\t-v\tVerbose mode.\n\t-h\tThis help\n\tAt least one option must be specified (-o, -a or -e), in order to avoid any loss of data\n\n";
	char common[]="TTKCruncher V%s (c) 2002, 2003 Coleebris\n";
	int	i, nb_files, recursive_mode, gain;
	char *output_path, *output_files_name_add, *output_extenstion, *in_ext, *dest;
	char **input_files;

	output_path = output_files_name_add = output_extenstion = in_ext = 0;
	F_MALLOC(input_files, FILE_STACK_SIZE * sizeof(input_files));
	nb_files = recursive_mode = verbose = max_gain = full_size = comp_size = 0;
	min_gain = 100;
	printf(common, VERSION);
	if (ac > 1)
    {
		/* parsage des options */
		for (i=1; i < ac; i++)
		{
			if (av[i][0] == '-')
			{
				if (av[i+1] && av[i+1][0] != '-' || av[i][1] == 'h' || av[i][1] == 'r' || av[i][1] == 'v')
				{
					/* on gere les options */
					switch (av[i][1]) 
					{
					case 'o':
						output_path = strdup(av[i+1]);
						i++;
						break;
					case 'a':
						output_files_name_add = strdup(av[i+1]);
						i++;
						break;
					case 'e':
						output_extenstion = strdup(av[i+1]);
						i++;
						break;
					case 'i':
						in_ext = strdup(av[i+1]);
						i++;
						break;
					case 'h':
						printf("%s", usage);
						printf("%s", help);
						exit(1);
					case 'r':
						recursive_mode = 1;
						break;
					case 'v':
						verbose = 1;
						break;
					default:
						printf("%s", usage);
						exit(1);
					}
				} else
				{
					fprintf(stderr, "\n%s require an argument.\nAbort.\n", av[i]);
					exit(1);
				}		      
			} else
			{
				/* on merge les fichiers */
				input_files[nb_files++] = strdup(av[i]);
			}
		}
		if (!output_path && !output_files_name_add && !output_extenstion)
		{
			fprintf(stderr, "\nSpecify at least one of -o, -a or -e options, please.\nAbort.\n");
			exit(1);
		}
		/* maintenant on peut cruncher */
		if (!recursive_mode)
		{
			for (i=0; i < nb_files; i++)
			{
				dest = get_final_path(input_files[i], output_path, output_files_name_add, output_extenstion);
				crunchcrunch(input_files[i], dest);
				gain = show_gain_info(input_files[i], dest);
				if (verbose) printf("Crunching file %s, destination : %s\t\tCompress: %i%%\n", input_files[i], dest, gain);

				if (gain > max_gain)
				{
					max_gain = gain;
				}

				if (gain < min_gain)
				{
					min_gain = gain;
				}
			}
		} else
		{
			parse_directory(input_files[0], in_ext, output_path, output_files_name_add, output_extenstion);
		}
    } else
    {
		fprintf(stderr, "%s", usage);
		exit(1);
    }
	printf("Original size : %i bytes, Crunched size: %i bytes\n", full_size, comp_size);
	printf("Average compress : %i%%, Min : %i%%, Max : %i%% \n", 100 - (comp_size * 100 / (full_size ? full_size : 1)), min_gain, max_gain);
}

void parse_directory(char *to_dir, char *in_ext, char *output_path, char *output_files_name_add, char *output_extenstion) 
{
	struct stat st;
	struct dirent *fic;
	DIR *dir;
	int fmt, gain;
	char *full_path, *dest, *ext;

	gain = 0;
	if (!(dir = opendir(to_dir)))
    {
		fprintf(stderr, "ERROR - Can't open directory %s -- Continue...\n", to_dir);
		return;
    }
	
	if (verbose) printf("Scanning %s\n", to_dir);
	
	if (output_path)
	{
		create_directory_if_not(output_path);
	}

	while (fic = readdir (dir))
	{
		if (*(fic->d_name) != '.')
		{
			full_path = add_level_to_path(to_dir, fic->d_name);

			if (lstat (full_path, &st))
			{
				fprintf(stderr, "ERROR - Can't read file %s stats -- Continue...\n",fic->d_name);
			}

			fmt = st.st_mode & S_IFMT;
			if (fmt == S_IFDIR)
			{
				if (strcmp(fic->d_name, "CVS"))
				{
					parse_directory(full_path, in_ext, add_level_to_path(output_path, fic->d_name), output_files_name_add, output_extenstion);
				}
			} else
			{
				ext = get_ext_from_path(full_path);
				if (in_ext && strcmp(ext, in_ext))
				{
					continue;
				}

				dest = get_final_path(full_path, output_path, output_files_name_add, output_extenstion);
				crunchcrunch(full_path, dest);
				gain = show_gain_info(full_path, dest);

				if (verbose) printf("Crunching file %s, destination: %s\t\tCompress: %i%%\n", full_path, dest, gain);

				if (gain > max_gain)
				{
					max_gain = gain;
				}
				if (gain < min_gain)
				{
					min_gain = gain;
				}
			}
			free(full_path);
		}
	}

	free(dir);
}

int crunchcrunch(char *in, char* out) /* flex call */
{
	if (!(yyin = fopen(in, "r")))
    {
		fprintf(stderr, "ERROR - Can't open file %s -- Ignoring file and continue...\n", in);
		return;
    }
	if (!(yyout = fopen(out, "w")))
	{
		fprintf(stderr, "ERROR - Can't open file %s -- Ignoring file and continue...\n", out);
		return;
	}
	yylex();
	fclose(yyin);
	fclose(yyout);
}

int show_gain_info(char *in, char *out) 
{
	struct stat st_in;
	struct stat st_out;
	int gain;
	int pourcent;

	if (lstat(in, &st_in))
    {
		fprintf(stderr, "ERROR - Can't read file %s stats -- Continue...\n", out);
		return 0;
    }

	if (lstat(out, &st_out))
    {
		fprintf(stderr, "ERROR - Can't read file %s stats -- Continue...\n", out);
		return 0;
    }
	
	full_size += st_in.st_size;
	comp_size += st_out.st_size;
	gain = st_in.st_size-st_out.st_size;
	pourcent = 100*st_out.st_size/(st_in.st_size != 0 ? st_in.st_size : 1);
	
	return (100-pourcent);
}

char *get_file_from_fullpath(char *path)
{
	char *tmp, *res;

	tmp = path;
	tmp += (strlen(tmp) - 1);
	while (*tmp != '/')
	{
		tmp--;
	}
	tmp++;

	F_MALLOC(res, (strlen(tmp) + 1) * sizeof(char));
	return (strcpy(res, tmp));
}

int create_directory_if_not(char *path) 
{
	if (!opendir(path))
    {
		if (verbose)  printf("Creating new directory %s...\n", path);

		if (mkdir(path, S_IWUSR | S_IXUSR | S_IRUSR))
		{
			fprintf(stderr, "ERROR - Can't create dir %s -- Continue\n", path);
		}
    }
}
  
char *get_final_path(char *inpath, char *output_path, char *output_files_name_add, char *output_extenstion)
{
	char *end_file, *end_path, *end_ext;
  
	end_file = end_path = end_ext = 0;

	if (output_path)
    {
		end_path = strdup(output_path);
    } else
	{
		end_path = strdup(inpath);
	}


	if (output_extenstion)
	{
		end_ext = strdup(output_extenstion);
	} else
	{
		end_ext = get_ext_from_path(inpath);
	}

	if (output_files_name_add)
	{
		end_file = add_to_file_name(output_files_name_add, get_file_name_without_ext(inpath));
	} else
	{
		end_file = get_file_name_without_ext(inpath);
	}
 
	return (concat_all(end_path, end_file, end_ext));
}

/* Pointer Utilities Functions
*/

char *add_level_to_path(char *path, char *level)
{
	char *res;
	
	F_MALLOC(res, (strlen(path) + strlen(level) + 3) * sizeof(char));
	res = strcat(res, path);

	if (res[strlen(res)-1] != '/')
	{
		res = strcat(res, "/");
	}

	res = strcat(res, level);
	return (res);
}

char *get_ext_from_path(char *inpath)
{
	char *tmp, *res;

	tmp = inpath;
	tmp += strlen(tmp);
	tmp--;

	while (tmp > inpath && *tmp != '.' && *tmp != '/')
	{
		tmp--;
	}

	if (tmp == inpath || *tmp == '/')
	{
		return ("");
	}

	tmp++;

	F_MALLOC(res, (strlen(tmp) + 1) * sizeof(char));
	return (strcpy(res, tmp));
}

char *add_to_file_name(char *output_files_name_add, char *file)
{
	char *tmp, *res;

	tmp = file;
	F_MALLOC(res, (strlen(tmp) + strlen(output_files_name_add) + 1) * sizeof(char));
	res = strcat(res, tmp);
	return (strcat(res, output_files_name_add));
}

char *get_file_name_without_ext(char *inpath)
{
	char *tmp, *res;
	int pos;
  
	pos = 0;
	tmp = inpath;
	tmp += strlen(tmp);
	tmp--;
	while (tmp > inpath && *tmp != '.')
    {
		if (*tmp == '/' || !*tmp)
		{
			return get_file_from_fullpath(inpath);
		}
		tmp--;
		pos++;
    }
	pos++;

	while (tmp > inpath && *tmp != '/')
	{
		tmp--;
	}

	tmp++;
	F_MALLOC(res, (strlen(tmp) + 1)* sizeof(char));
	return (strncpy(res, tmp, strlen(tmp) - pos));
}

char *concat_all(char *end_path, char *end_file, char *end_ext)
{
	char *tmp, *res;
	
	F_MALLOC(res, (strlen(end_path)+strlen(end_file)+strlen(end_ext)+3)*sizeof(char));
	res = strcat(res, end_path);

	if (res[strlen(res)-1] != '/')
	{
		res = strcat(res, "/");
	}

	res = strcat(res, end_file);
	
	if (end_ext)
	{
		res = strcat(res, ".");
		res = strcat(res, end_ext);
		free(end_ext);
	}

	free(end_path);
	free(end_file);
	return(res);
}
