/*
 
    Ejecter - Safely, easily remove external peripherals
    Copyright 2008-2009, Federico Pelloni <federico.pelloni@gmail.com>
 
    
    This file is part of Ejecter.
 
    Ejecter is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.
    
    Ejecter is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.
    
    You should have received a copy of the GNU General Public License
    along with Ejecter.  If not, see <http://www.gnu.org/licenses/>.
   
*/ 

using GLib;


namespace Ejecter {

public class Device: GLib.Object {


    public GLib.Drive drive { get; construct; }
    private GLib.List<GLib.Volume> volumes;
    private GLib.List<GLib.Mount> mounts;
    public string type;
    public string bus;
    private Gtk.Table table;
    private Gtk.Image icon;
    private Gtk.Image eject_icon;
    private Gtk.Label label;
    private Gtk.Button button;
    private bool was_removed;



    
    public Device (GLib.Drive drive) {
    
        this.drive = drive;
        
    }
                   
                   
    construct {
    
        this.volumes = new GLib.List<GLib.Volume>();
        this.mounts = new GLib.List<GLib.Mount>();
        
        this.icon = Ejecter.Alternatives.build_device_icon(this.drive);
        this.icon.show();
        
        this.label = new Gtk.Label("");
        Gtk.Style style = this.label.get_style();
        this.label.use_markup = true;
        this.label.set_alignment(0, (float) 0.5);
        this.label.modify_fg(Gtk.StateType.NORMAL, style.text[(int) Gtk.StateType.NORMAL]);
        this.update_label();
        this.label.show();
        
        this.button = new Gtk.Button();
        this.button.set_relief(Gtk.ReliefStyle.NONE);
        
        this.eject_icon = new Gtk.Image.from_icon_name("media-eject", Gtk.IconSize.MENU);
        this.eject_icon.show();
        
        this.button.add(this.eject_icon);
        this.button.clicked += this.clicked_eject;
        this.button.show();
        
        this.set_mounted(false);
        this.was_removed = false;
        
        this.drive.disconnected += this.handle_removed_drive;
        
    }
    
    
    private void update_label () {
    
        if (this.volumes.length() == 0)
        
            this.label.set_markup("<b>" + this.drive.get_name() + "</b>");
        
        else if (this.volumes.length() == 1) {
        
            this.volumes.first();
            GLib.Volume v = this.volumes.data;
         
            this.label.set_markup("<b>" + v.get_name() + "</b>\n<small>" + 
                                  this.drive.get_name() + "</small>");
        
        } else {
            
            string volumes = "";
            foreach (GLib.Volume v in this.volumes)
                if (volumes == "")
                    volumes = v.get_name();
                else
                    volumes += ", " + v.get_name();
            
            this.label.set_markup("<b>" + this.drive.get_name() + "</b>\n<small>" + volumes + "</small>");
            
        }
    
    }
        
        
    public void attach (Gtk.Table table) {
    
        uint rows = table.n_rows;
        table.n_rows++;
        
        table.attach(this.icon, 0, 1, rows, rows+1, Gtk.AttachOptions.FILL, Gtk.AttachOptions.FILL, 5, 1);
        
        table.attach(this.label, 1, 2, rows, rows+1, Gtk.AttachOptions.EXPAND|Gtk.AttachOptions.FILL, 
                     Gtk.AttachOptions.FILL, 0, 1);
                     
        table.attach(this.button, 2, 3, rows, rows+1, Gtk.AttachOptions.FILL, Gtk.AttachOptions.FILL, 3, 1);
        
        this.table = table;
        
    }
        
    
    public signal void removed ();
    public signal void ejected ();
    
    public void remove () {
    
        if (this.was_removed) return;
        
        this.table.remove(this.icon);
        this.table.remove(this.label);
        this.table.remove(this.button);
        
        this.table.n_rows--;
        
        this.icon.destroy();
        this.label.destroy();
        this.eject_icon.destroy();
        this.button.destroy();
        
        this.was_removed = true;
        this.removed();
        
    }
        
        
    public void set_mounted (bool mounted) {
        
        this.icon.set_sensitive(mounted);
        this.label.set_sensitive(mounted);
        this.button.set_sensitive(mounted);
        
    }
        
        
    private void clicked_eject (Gtk.Button button) {
        
        if (this.drive.can_eject()) {
        
            this.drive.eject(GLib.MountUnmountFlags.NONE, null, null);
            
        } else {
            
            foreach (GLib.Volume v in (GLib.List<GLib.Volume>) this.drive.get_volumes())
                v.get_mount().unmount(GLib.MountUnmountFlags.NONE, null, null);
            
        }
        
    }
    
    
    private void handle_unmounted (GLib.Mount mount) {
    
        this.mounts.remove(mount);
        
        if (this.mounts.length() == 0) {
    
            this.set_mounted(false);
            //if (this.type == "cdrom") this.remove();
            
        }
    
    }
    
    
    private void handle_removed_drive (GLib.Drive drive) {
    
        this.remove();
    
    }
    
    
    private void handle_removed_volume (GLib.Volume volume) {
    
        this.volumes.remove(volume);
        
        if (this.volumes.length() == 0)
        
            this.remove();
    
    }
    
    
    public void add_volume (GLib.Volume volume) {
    
        this.volumes.append(volume);
        this.update_label();
        volume.removed += this.handle_removed_volume;
    
    }
    
    
    public void add_mount (GLib.Mount mount) {
    
        this.mounts.append(mount);
        this.set_mounted(true);
        mount.unmounted += this.handle_unmounted;
    
    }

}

}


