# -*- coding: utf-8 -*-
# Elisa - Home multimedia server
# Copyright (C) 2006-2008 Fluendo Embedded S.L. (www.fluendo.com).
# All rights reserved.
#
# This file is available under one of two license agreements.
#
# This file is licensed under the GPL version 3.
# See "LICENSE.GPL" in the root of this distribution including a special
# exception to use Elisa with Fluendo's plugins.
#
# The GPL part of Elisa is also available under a commercial licensing
# agreement from Fluendo.
# See "LICENSE.Elisa" in the root directory of this distribution package
# for details on that license.
#
# Author: Benjamin Kampmann <benjamin@fluendo.com>

"""
Scan the resource providers, add some metadata and store it in the database
"""

from elisa.core import common
from elisa.core.utils.locale_helper import system_encoding
from elisa.core.media_uri import MediaUri

from twisted.internet import task, reactor, threads
from twisted.python.failure import Failure

from elisa.core.utils.cancellable_defer import cancellable_coiterate, \
        CancelledError

from elisa.core.utils import defer

import time, os

from elisa.core.components.resource_provider import ResourceProvider

from elisa.plugins.database.scanner_models import Statistic, ScanResource
from elisa.plugins.database.database_parser import DatabaseParser
from elisa.plugins.database.models import \
         PICTURES_SECTION, MUSIC_SECTION, VIDEO_SECTION

from elisa.plugins.base.local_resource import LocalResource

from elisa.plugins.database.database_updater import SCHEMA, DatabaseUpdaterNG

from elisa.plugins.base.models.file import DirectoryModel

try:
    import dbus
    from elisa.plugins.database.dbus_iface import DBusInterface
except ImportError:
    dbus = None

class MediaScanner(ResourceProvider):
    """
    Asynchronous working media scanner.
    
    @ivar store: the store for the data
    @type store: L{storm.twisted.storm.DeferredStore}
    """

    default_config = {"delay" : 0.1, "scan_every" : 24}
    config_doc = {
        "delay": "the delay (in seconds) between processing two files",
        "scan_every" : "the delay (in hours) between two automatic scans"
        }

    supported_uri = '^media_scanner://localhost/'

    def __init__(self):
        super(MediaScanner, self).__init__()
        self.store = None
        self._current_delay = None
        self._start_delay = None
        self._current_scan_deferred = None

        self.scan_stat = Statistic()
        self.scan_stat.uri = MediaUri('media_scanner://localhost/statistic')
        self._pending_auto_rescan = None

        self.running = False

    def _scan_directories(self):
        # ugly hack to start the scanning process for the directories the user
        # has set up in the configuration file
        directories_section = common.application.config.get_section('directories')
        if not directories_section:
            self.debug("no default directories")
            return

        for section in ('music', 'pictures', 'video'):
            paths = directories_section.get(section, [])
            for path in paths:
                if path != '*default*': # skip the default configuration thingy
                    uri = MediaUri('file://%s' % path)
                    self.debug("adding %s" % uri)
                    self.put(uri, None, section=section)

    def initialize(self):

        def set_parser(parser):
            self.parser = parser
            self.parser.store = self.store
            # start the scanner with a delay of 5 seconds
            self._start_delay = reactor.callLater(5, self._scan_directories)
            return self
 
        def set_local_resource(local_resource):
            self.local_resource = local_resource
            return self

        dfr = super(MediaScanner, self).initialize()
        store = common.application.store
        self.store = store

        dfr.addCallback(lambda x: self.create_schema())
        # FIXME: add configuration system. Passing through my configuration is
        # not exactly nice.
        dfr.addCallback(lambda x: DatabaseParser.create(self.config))
        dfr.addCallback(set_parser)
        dfr.addCallback(lambda x: LocalResource.create({}))
        dfr.addCallback(set_local_resource)
        # start dbus at the end
        dfr.addCallback(self._initialize_dbus)
        return dfr

    def _initialize_dbus(self, result=None):
        if dbus is None:
            # no dbus support
            return self

        bus = dbus.SessionBus()
        self.bus_name = dbus.service.BusName('com.fluendo.Elisa', bus)

        self.dbus_scanner = DBusInterface(self, bus,
                '/com/fluendo/Elisa/Plugins/Database/MediaScanner', self.bus_name)
        return self

    def _clean_dbus(self):
        if dbus is None:
            # no dbus support
            return

        bus = dbus.SessionBus()
        self.dbus_scanner.remove_from_connection(bus,
                '/com/fluendo/Elisa/Plugins/Database/MediaScanner')
        # BusName implements __del__, eew
        del self.bus_name

        # remove the reference cycle
        self.dbus_scanner = None


    def clean(self):
        self._clean_dbus()
        if self._current_delay and not self._current_delay.called:
            self._current_delay.cancel()

        if self._start_delay and not self._start_delay.called:
            self._start_delay.cancel()

        return super(MediaScanner, self).clean()
        
    def create_schema(self):
        upgrader = DatabaseUpdaterNG(self.store)
        return upgrader.update_db()

    def running_get(self):
        return self._running

    def running_set(self, value):
        self._running = value
        self.scan_stat.running = value
        if not value:
            self._current_scan_deferred = None

    running = property(fget=running_get, fset=running_set)

    # general Resource Provider API
    def get(self, uri, context_model=None):
        """
        If the filename is C{statistic} you receive the
        L{elisa.plugins.database.scanner_models.Statistic} for this scanner
        """
        if uri.filename == 'statistic':
            return (self.scan_stat, defer.succeed(self.scan_stat))

        return (None, defer.fail(NotImplementedError()))

    def put(self, source_uri, container_uri,
            context_model=None, section=None):
        """
        put another uri into the queue and start the scanning process if it is
        not yet running

        @ivar section:  the name of the section relevant to the content that
                        is put.
        @type section:  L{str}
        """
        # FIXME check the container uri
        # FIXME no check if this uri might already in the queue...

        if source_uri.filename != '':
            # we want to have the content, so we have to add a trailing slash
            source_uri = source_uri.join('')

        def start_scanning(result, source_uri, section):
            name = os.path.basename(os.path.dirname(source_uri.path))
            s = ScanResource(source_uri, name, section=section)
            self.scan_stat.queue.put(s)

            if not self.running:
                self._scan()

            return s.defer

        return self.parser.mark_deleted(source_uri.path).addCallback(start_scanning,
                source_uri, section)

    # internally used
    def _update_stat(self, result, model, stat):
        stat.files_scanned += 1

        # deactivated as we don't use it atm and it leaks memory
        #if isinstance(result, Failure):
        #    stat.files_failed.append( (model.uri, result) )

    def _count_files(self, path):
        path = os.path.abspath(path)

        deferred = defer.Deferred()

        def iterate(path):
            counter = 0
            try:
                for root, dirs, files in os.walk(path.encode(system_encoding())):

                    if deferred.called:
                        # we are cancelled. Stop the processing without calling
                        # the deferred again
                        return

                    # count only the files
                    counter += len(files)

                reactor.callFromThread(deferred.callback, counter)

            except Exception, exc:
                reactor.callFromThread(deferred.errback, exc)

        # push it into background
        threads.deferToThread(iterate, path)

        return deferred

    def _file_found(self, model, stat):
        dfr = self.parser.query_model(model, stat)
        dfr.addBoth(self._update_stat, model, stat)
        return dfr

    def _cleanup_deleted_files(self, result, scan_resource):
        return self.parser.delete_files(scan_resource.root_uri.path)

    def _scan(self, result=None):

        def run_next(result, scan_resource):
            scan_resource.state = ScanResource.SCANNING_DONE

            self.scan_stat.currently_scanning = None
            self.scan_stat.scanned.append(scan_resource)

            if self.scan_stat.queue.empty():
                # stop processing
                self.running = False
                self._reset_auto_rescan(True)
            else:
                reactor.callLater(0.1, self._scan)

        self.running = True

        scan_resource = self.scan_stat.queue.get_nowait()
        self.scan_stat.currently_scanning = scan_resource

        # clear stat
        scan_resource.last_scan = time.time()
        scan_resource.files_scanned = 0
        scan_resource.files_total = 0
        scan_resource.state = ScanResource.SCANNING_FS

        def set_total_files(result, scan_resource):
            scan_resource.files_total = result
            return self._scan_recursive(scan_resource)

        dfr = self._count_files(scan_resource.root_uri.path)
        dfr.addCallback(set_total_files, scan_resource)
        dfr.addCallback(self._cleanup_deleted_files, scan_resource)
        dfr.addCallback(run_next, scan_resource)

        self._current_scan_deferred = dfr

    def _rescan(self):
        self.debug("Starting to scan everything again")
        dfr = self._reschedule_scanned()
        dfr.addCallback(self._scan)
        return dfr

    def _reset_auto_rescan(self, restart=False):
        if self._pending_auto_rescan and self._pending_auto_rescan.active():
            self._pending_auto_rescan.cancel()

        rescan_every = self.config.get('scan_every', 0) * 3600
        if restart and rescan_every != 0:
            pending = reactor.callLater(rescan_every, self._rescan)
            self._pending_auto_rescan = pending

    def _reschedule_scanned(self):
        scan_stat = self.scan_stat

        scanned, scan_stat.scanned = scan_stat.scanned, [] 

        deleted_dfrs = []
        for stat in scanned:
            scan_stat.queue.put(stat)
            deleted_dfrs.append(self.parser.mark_deleted(stat.root_uri.path))

        if len(deleted_dfrs) == 0:
            return defer.succeed(None)

        return defer.DeferredList(deleted_dfrs)

    def _scan_recursive(self, scan_resource):

        def stopper(data, file):
            self.warning("%s failed: %s" % (file, data))
            return None

        def scan_children(model, dirs):
            def iterator(model, dirs):
                while len(model.files):
                    item = model.files.pop(0)
                    if isinstance(item, DirectoryModel):
                        # delay searching the subfolders for later
                        dirs.append(item.uri)
                    else:
                        dfr = self._file_found(item, scan_resource)
                        yield dfr

            return cancellable_coiterate(iterator, model, dirs)

        def scan_dirs(dirs):
            def iterator(dirs):
                while len(dirs):
                    uri = dirs.pop(0)
                    if uri.filename != '':
                        # we have a filename append the slash to get the
                        # content
                        uri = uri.join('')
                    model, dfr = self.local_resource.get(uri)
                    dfr.addCallback(scan_children, dirs)
                    dfr.addErrback(stopper, uri.path)
                    yield dfr

            return cancellable_coiterate(iterator, dirs)

        return scan_dirs([scan_resource.root_uri])
