# -*- coding: utf-8 -*-
# Elisa - Home multimedia server
# Copyright (C) 2006-2008 Fluendo Embedded S.L. (www.fluendo.com).
# All rights reserved.
#
# This file is available under one of two license agreements.
#
# This file is licensed under the GPL version 3.
# See "LICENSE.GPL" in the root of this distribution including a special
# exception to use Elisa with Fluendo's plugins.
#
# The GPL part of Elisa is also available under a commercial licensing
# agreement from Fluendo.
# See "LICENSE.Elisa" in the root directory of this distribution package
# for details on that license.
#
# Authors: Guido Amoruso <guidonte@fluendo.com>
#          Benjamin Kampmann <benjamin@fluendo.com>
#          Olivier Tilloy <olivier@fluendo.com>

from elisa.core import common
from elisa.core.media_uri import MediaUri
from elisa.core.utils.i18n import install_translation
from elisa.core.utils.cancellable_defer import CancelledError

from elisa.plugins.base.models.media import PlayableModel
from elisa.plugins.database.models import *

from elisa.plugins.poblesec.link import Link

from elisa.plugins.poblesec.section import SectionMenuViewMode
from elisa.plugins.poblesec.base.hierarchy import HierarchyController
from elisa.plugins.poblesec.base.list_switcher import ListSwitcherController
from elisa.plugins.poblesec.music_library import ArtistsViewMode, \
                                                 AlbumsViewMode, \
                                                 TracksViewMode

from elisa.plugins.poblesec.base.preview_list import \
    MenuItemPreviewListController, DoubleLineMenuItemPreviewListController
from elisa.plugins.poblesec.base.coverflow import \
    ImageWithReflectionCoverflowController
from elisa.plugins.poblesec.base.grid import GridItemGridController

from elisa.plugins.poblesec.actions import Action
from elisa.plugins.database.actions import ArtistPlayAllAction, \
                                           AlbumPlayAllAction, \
                                           TrackPlayAllAction, \
                                           TrackRemoveAction
from elisa.plugins.database.actions import ArtistShuffleAction, \
                                           AlbumShuffleAction, \
                                           TrackShuffleAction
from elisa.plugins.database.actions import AlbumAddToFavoritesAction, \
                                           TrackAddToFavoritesAction, \
                                           PhotoAddToFavoritesAction
from elisa.plugins.database.actions import ViewPhotoAlbumSlideshowAction

from elisa.plugins.favorites.models import FavoritesItem

from twisted.internet import defer, task

from storm.expr import Not

import datetime

_ = install_translation('database')

# this is totally searcher controller based
from elisa.plugins.poblesec.search_controller import SearcherEntry


def music_search_result_decorator(controller):
    searcher = 'DBSearcher'
    title = _('Local Search Results')
    icon_resource = 'elisa.plugins.poblesec.search'
    searcher_entry = SearcherEntry(searcher, title, icon_resource)
    controller.searchers.insert(0, searcher_entry)
    return defer.succeed(None)


# other decorators
def music_lib_decorator(controller):
    link = Link()
    link.controller_path = '/poblesec/database/music_library'
    link.label = _('Music Library')
    link.icon = 'elisa.plugins.poblesec.music_library'
    controller.model.append(link)

    return defer.succeed(None)


def music_lib_artists_decorator(controller):
    artists = Link()
    artists.controller_path = '/poblesec/database/music/artists'
    artists.label = _('Artists')
    artists.icon = 'elisa.plugins.poblesec.by_artist'
    controller.model.append(artists)
    return defer.succeed(None)


def music_lib_albums_decorator(controller):
    albums = Link()
    albums.controller_path = '/poblesec/database/music/albums'
    albums.label = _('Albums')
    albums.icon = 'elisa.plugins.poblesec.by_album'
    controller.model.append(albums)
    return defer.succeed(None)


def music_lib_tracks_decorator(controller):
    tracks = Link()
    tracks.controller_path = '/poblesec/database/music/tracks'
    tracks.label = _('Tracks')
    tracks.icon = 'elisa.plugins.poblesec.by_track'
    controller.model.append(tracks)
    return defer.succeed(None)


def music_lib_genres_decorator(controller):
    genres = Link()
    genres.controller_path = '/poblesec/database/music/genres'
    genres.label = _('Genres')
    genres.icon = 'elisa.plugins.poblesec.by_genre'
    controller.model.append(genres)
    return defer.succeed(None)


def music_lib_decades_decorator(controller):
    decades = Link()
    decades.controller_path = '/poblesec/database/music/decades'
    decades.label = _('Decades')
    decades.icon = 'elisa.plugins.poblesec.by_decade'
    controller.model.append(decades)
    return defer.succeed(None)


class GenericArtistsController(HierarchyController):

    artist_controller_path = '/poblesec/database/music/albums_of_artist'

    empty_label = _('There are no artists in this section')
    empty_icon = 'elisa.plugins.poblesec.by_artist'

    def initialize(self):
        dfr = super(GenericArtistsController, self).initialize()
        self.actions.extend(self.make_actions())
        return dfr

    def node_clicked(self, widget, item):
        self.cancel_deferreds(id(self))

        if isinstance(item, Action):
            def silence_canceller(failure):
                failure.trap(CancelledError)

            dfr = item.run()
            dfr.addErrback(silence_canceller)
            self.register_deferred(id(self), dfr)
        else:
            browser = self.frontend.retrieve_controllers('/poblesec/browser')[0]
            path = self.artist_controller_path
            deferred = browser.history.append_controller(path, item.name, artist=item)

    def make_actions(self):
        actions = []

        action = ArtistPlayAllAction(self)
        action.title = _('Play All')
        # FIXME: there is no two lines support in ArtistController
        action.subtitle = _('Play All Tracks By All Artists')
        actions.append(action)

        action = ArtistShuffleAction(self)
        action.title = _('Shuffle & Play')
        # FIXME: there is no two lines support in ArtistController
        action.subtitle = _('Shuffle & Play All Tracks By All Artists')
        actions.append(action)

        return actions


class ArtistsDbController(GenericArtistsController):

    def initialize(self, artists=None):
        deferred = super(ArtistsDbController, self).initialize()

        self.artists = artists

        def got_artists(artists):
            self.model.extend(artists)
            return self

        def sort_artists(result_set):
            result_set.order_by(Artist.name)
            return result_set.all()

        def get_artists(self):
            return common.application.store.find(Artist)

        def populate_model(self, artists):
            self.model.extend(artists)
            return self

        if artists is None:
            deferred.addCallback(get_artists)
            deferred.addCallback(sort_artists)
            deferred.addCallback(got_artists)
        else:
            deferred.addCallback(populate_model, artists)
        return deferred

    def clean(self):
        dfr = super(ArtistsDbController, self).clean()
        self.artists = None
        return dfr

class ArtistsDbViewMode(ArtistsViewMode):

    def get_label(self, item):
        if isinstance(item, Action):
            return defer.succeed(item.title)
        return super(ArtistsDbViewMode, self).get_label(item)

    def get_sublabel(self, item):
        if isinstance(item, Action):
            return defer.succeed(item.subtitle)
        return super(ArtistsDbViewMode, self).get_sublabel(item)

    def get_default_image(self, item):
        if isinstance(item, Action):
            return item.icon
        return super(ArtistsDbViewMode, self).get_default_image(item)

    def get_image(self, item, theme):
        if isinstance(item, Action):
            return None
        return super(ArtistsDbViewMode, self).get_image(item, theme)

    def get_preview_image(self, item, theme):
        if isinstance(item, Action):
            return None
        return super(ArtistsDbViewMode, self).get_preview_image(item, theme)


class ArtistsDbPreviewListController(ArtistsDbController, MenuItemPreviewListController):
    view_mode = ArtistsDbViewMode
    fastscroller = True

    def item_to_label(self, item):
        if isinstance(item, Action):
            return '#'
        return item.name

class ArtistsDbCoverflowController(ArtistsDbController, ImageWithReflectionCoverflowController):
    view_mode = ArtistsDbViewMode

class ArtistsDbGridController(ArtistsDbController, GridItemGridController):
    view_mode = ArtistsDbViewMode

class ArtistsDbListSwitcherController(ListSwitcherController):
    modes = [ArtistsDbPreviewListController,
             ArtistsDbCoverflowController,
             ArtistsDbGridController]
    default_mode = ArtistsDbPreviewListController


class GenericAlbumsDbController(HierarchyController):

    album_controller_path = '/poblesec/database/music/tracks'

    empty_label = _('There are no albums in this section')
    empty_icon = 'elisa.plugins.poblesec.by_album'

    def initialize(self, artist=None):
        self.artist = artist
        dfr = super(GenericAlbumsDbController, self).initialize()
        self.actions.extend(self.make_actions())
        return dfr

    def make_actions(self):
        actions = []

        if self.artist:
            action = AlbumPlayAllAction(self)
            action.title = _('Play All')
            # FIXME: there is no two lines support in AlbumController
            action.subtitle = _("Play All Tracks By '%s'") % self.artist.name
            actions.append(action)

            action = AlbumShuffleAction(self)
            action.title = _('Shuffle & Play')
            # FIXME: there is no two lines support in AlbumController
            action.subtitle = _("Shuffle & Play All Tracks By '%s'") % self.artist.name
            actions.append(action)

            action = AlbumAddToFavoritesAction(self)
            # FIXME: there is no two lines support in AlbumController
            action.untoggled_title = _('Add To Favorites')
            action.untoggled_subtitle = _("Add The Artist '%s' To Your Favorites") % self.artist.name
            action.toggled_title = _('Remove From Favorites')
            action.toggled_subtitle = _("Remove The Artist '%s' From Your Favorites") % self.artist.name
            actions.append(action)

            uri = getattr(self.artist, 'uri', None)
            if uri:
                dfr = common.application.store.find(FavoritesItem,
                                                    FavoritesItem.uri == unicode(uri),
                                                    FavoritesItem.type == u'artist')
            else:
                dfr = common.application.store.find(FavoritesItem,
                                                    FavoritesItem.foreign_id == self.artist.name,
                                                    FavoritesItem.foreign_class == u'Artist')
            dfr.addCallback(lambda rs: rs.all())
            dfr.addCallback(action.setup)
        else:
            action = AlbumPlayAllAction(self)
            action.title = _('Play All')
            # FIXME: there is no two lines support in AlbumController
            action.subtitle = _('Play All Tracks From All Albums')
            actions.append(action)

            action = AlbumShuffleAction(self)
            action.title = _('Shuffle & Play')
            # FIXME: there is no two lines support in AlbumController
            action.subtitle = _('Shuffle & Play All Tracks From All Albums')
            actions.append(action)

        return actions

    def node_clicked(self, widget, item):
        if isinstance(item, Action):
            dfr = item.run()
        elif hasattr(item, 'all_tracks'):
            # Dummy "All Tracks" album
            browser = self.frontend.retrieve_controllers('/poblesec/browser')[0]
            dfr = browser.history.append_controller('/poblesec/database/music/tracks',
                                                    item.name,
                                                    artist=item.artist)
        elif hasattr(item, 'action') and issubclass(item.action, Action):
            action = item.action(self)
            dfr = action.run(item)
        else:
            browser = self.frontend.retrieve_controllers('/poblesec/browser')[0]
            path = self.album_controller_path
            dfr = browser.history.append_controller(path,
                                                    item.name,
                                                    album=item)
        return dfr


class AlbumsOfArtistDbController(GenericAlbumsDbController):

    def initialize(self, artist=None):
        deferred = super(AlbumsOfArtistDbController, self).initialize(artist=artist)

        # Add an "All Tracks" album for a given artist
        all_tracks_album = MusicAlbum()
        all_tracks_album.name = u'All Tracks'
        # fake attribute to differentiate the 'All tracks' album from
        # the regular albums
        all_tracks_album.all_tracks = True
        # FIXME: very wrong hot plumbing because we need get_tracks()
        # to return at some other place.
        all_tracks_album.get_tracks = lambda: defer.succeed([])
        all_tracks_album.artist = self.artist # dynamic field
        self.model.append(all_tracks_album)

        def sort_albums(result_set):
            result_set.order_by(MusicAlbum.name)
            return result_set.all()

        def add_albums(albums):
            self.model.extend(albums)
            return self

        def get_albums(album_names):
            store = common.application.store
            return store.find(MusicAlbum,
                              MusicAlbum.name.is_in(album_names))

        def order_tracks(result_set):
            return result_set.values(MusicTrack.album_name)

        def get_tracks(self):
            return artist.tracks.find(MusicTrack.file_path == File.path,
                                      File.hidden == False)

        # retrieve tracks for not-database artists
        if getattr(artist, 'uri', None):
            self.model.extend(artist.albums)
        # retrieve tracks for database artists
        else:
            deferred.addCallback(get_tracks)
            deferred.addCallback(order_tracks)
            deferred.addCallback(get_albums)
            deferred.addCallback(sort_albums)
            deferred.addCallback(add_albums)

        return deferred

    def clean(self):
        dfr = super(AlbumsOfArtistDbController, self).clean()
        self.albums = None
        return dfr

class AlbumsDbController(GenericAlbumsDbController):

    def initialize(self, albums=None):
        deferred = super(AlbumsDbController, self).initialize()

        self.albums = albums

        def sort_albums(result_set):
            result_set.order_by(MusicAlbum.name)
            return result_set.all()

        def add_albums(albums):
            self.model.extend(albums)
            return self

        def populate_model(self, albums):
            self.model.extend(albums)
            return self

        # Show given albums
        if albums is not None:
            return deferred.addCallback(populate_model, albums)

        # Show all albums
        def get_albums(self):
            return common.application.store.find(MusicAlbum)

        deferred.addCallback(get_albums)
        deferred.addCallback(sort_albums)
        deferred.addCallback(add_albums)

        return deferred

    def clean(self):
        dfr = super(AlbumsDbController, self).clean()
        self.albums = None
        return dfr


class AlbumsDbViewMode(AlbumsViewMode):

    def get_label(self, item):
        if isinstance(item, Action):
            return defer.succeed(item.title)
        return super(AlbumsDbViewMode, self).get_label(item)

    def get_sublabel(self, item):
        if isinstance(item, Action):
            return defer.succeed(item.subtitle)
        return super(AlbumsDbViewMode, self).get_sublabel(item)

    def get_default_image(self, item):
        if isinstance(item, Action):
            return item.icon
        elif hasattr(item, 'all_tracks'):
            # "All Tracks" fake album
            return 'elisa.plugins.poblesec.by_track'
        return super(AlbumsDbViewMode, self).get_default_image(item)

    def get_image(self, item, theme):
        if isinstance(item, Action):
            return None
        elif hasattr(item, 'all_tracks'):
            # "All Tracks" fake album
            return None
        return super(AlbumsDbViewMode, self).get_image(item, theme)

    def get_preview_image(self, item, theme):
        if isinstance(item, Action):
            return None
        elif hasattr(item, 'all_tracks'):
            # "All Tracks" fake album
            return None
        return super(AlbumsDbViewMode, self).get_preview_image(item, theme)


class AlbumsDbPreviewListController(AlbumsDbController, DoubleLineMenuItemPreviewListController):
    view_mode = AlbumsDbViewMode
    fastscroller = True

    def item_to_label(self, item):
        if isinstance(item, Action):
            return '#'
        return item.name

class AlbumsDbCoverflowController(AlbumsDbController, ImageWithReflectionCoverflowController):
    view_mode = AlbumsDbViewMode

class AlbumsDbGridController(AlbumsDbController, GridItemGridController):
    view_mode = AlbumsDbViewMode

class AlbumsDbListSwitcherController(ListSwitcherController):
    modes = [AlbumsDbPreviewListController,
             AlbumsDbCoverflowController,
             AlbumsDbGridController]
    default_mode = AlbumsDbPreviewListController

class AlbumsOfArtistDbPreviewListController(AlbumsOfArtistDbController, MenuItemPreviewListController):
    view_mode = AlbumsDbViewMode
    fastscroller = True

    def item_to_label(self, item):
        if isinstance(item, Action):
            return '#'
        return item.name

class AlbumsOfArtistDbCoverflowController(AlbumsOfArtistDbController, ImageWithReflectionCoverflowController):
    view_mode = AlbumsDbViewMode

class AlbumsOfArtistDbGridController(AlbumsOfArtistDbController, GridItemGridController):
    view_mode = AlbumsDbViewMode

class AlbumsOfArtistDbListSwitcherController(ListSwitcherController):
    modes = [AlbumsOfArtistDbPreviewListController,
             AlbumsOfArtistDbCoverflowController,
             AlbumsOfArtistDbGridController]
    default_mode = AlbumsOfArtistDbPreviewListController


class GenericTracksDbController(HierarchyController):

    track_controller_path = '/poblesec/database/music/tracks'

    empty_label = _('There are no tracks in this section')
    empty_icon = 'elisa.plugins.poblesec.by_track'

    def initialize(self, album=None, track=None, tracks=None, artist=None):
        self.album = album
        self.track = track
        self.tracks = tracks
        self.artist = artist

        # we don't need the fastscroller, because we're going to order tracks
        # by track number
        if album:
            self.fastscroller = False

        dfr= super(GenericTracksDbController, self).initialize()

        self.actions.extend(self.make_actions())
        return dfr

    def is_empty(self):
        return super(GenericTracksDbController, self).is_empty() and self.track is None

    def item_to_label(self, item):
        if isinstance(item, Action):
            return '#'
        return item.title

    def make_actions(self):
        actions = []

        if self.album:
            action = TrackPlayAllAction(self)
            action.title = _('Play All')
            action.subtitle = _("Play All Tracks From '%s'") % self.album.name
            actions.append(action)

            action = TrackShuffleAction(self)
            action.title = _('Shuffle & Play')
            action.subtitle = _("Shuffle & Play All Tracks From '%s'") % self.album.name
            actions.append(action)

            action = TrackAddToFavoritesAction(self)
            action.untoggled_title = _('Add To Favorites')
            action.untoggled_subtitle = _("Add The Album '%s' To Your Favorites") % self.album.name
            action.toggled_title = _('Remove From Favorites')
            action.toggled_subtitle = _("Remove The Album '%s' From Your Favorites") % self.album.name
            actions.append(action)

            uri = getattr(self.album, 'uri', None)
            if uri:
                dfr = common.application.store.find(FavoritesItem,
                                                    FavoritesItem.uri == unicode(uri),
                                                    FavoritesItem.type == u'music_album')
            else:
                dfr = common.application.store.find(FavoritesItem,
                                                    FavoritesItem.foreign_id == self.album.name,
                                                    FavoritesItem.foreign_class == u'MusicAlbum')
            dfr.addCallback(lambda rs: rs.all())
            dfr.addCallback(action.setup)
        elif self.track:
            action = TrackPlayAllAction(self)
            action.title = _('Play Track')
            action.subtitle = _("Play '%s'") % self.track.title
            actions.append(action)

            action = TrackAddToFavoritesAction(self)
            action.untoggled_title = _('Add to Favorites')
            action.untoggled_subtitle = _("Add '%s' To Your Favorites") % self.track.title
            action.toggled_title = _('Remove From Favorites')
            action.toggled_subtitle = _("Remove '%s' From Your Favorites") % self.track.title
            actions.append(action)

            uri = getattr(self.track, 'uri', None)
            if uri:
                dfr = common.application.store.find(FavoritesItem,
                                                    FavoritesItem.uri == unicode(uri),
                                                    FavoritesItem.type == u'music_track')
            else:
                dfr = common.application.store.find(FavoritesItem,
                                                    FavoritesItem.foreign_id == self.track.file_path,
                                                    FavoritesItem.foreign_class == u'MusicTrack')
            dfr.addCallback(lambda rs: rs.all())
            dfr.addCallback(action.setup)

            # FIXME: implement this properly instead of hijacking the app config
            app_config = common.application.config
            if int(app_config.get_option('enable_remove',
                                         'poblesec.main:PoblesecController',
                                         default='0')):
                remove_action = TrackRemoveAction(self)
                remove_action.title = _('Remove Track')
                remove_action.subtitle = _("Remove '%s'") % self.track.title
                actions.append(remove_action)

        else:
            action = TrackPlayAllAction(self)
            action.title = _('Play All')
            actions.append(action)

            action = TrackShuffleAction(self)
            action.title = _('Shuffle & Play')
            actions.append(action)

        return actions

    def node_clicked(self, widget, item):
        if isinstance(item, Action):
            item.run()
        elif hasattr(item, 'action') and issubclass(item.action, Action):
            action = item.action(self)
            dfr = action.run(item)
        else:
            browser = self.frontend.retrieve_controllers('/poblesec/browser')[0]
            path = self.track_controller_path
            title = item.title
            tracks = [t for t in self.model if not isinstance(t, Action)]
            args = {'track': item,
                    'tracks': tracks}
            dfr = browser.history.append_controller(path, title, **args)


class TracksDbController(GenericTracksDbController):

    def initialize(self, album=None, track=None, tracks=None, artist=None):
        dfr = super(TracksDbController, self).initialize(album=album,
                                                         track=track,
                                                         tracks=tracks,
                                                         artist=artist)

        if track:
            return dfr

        def populate_model(self, tracks):
            self.model.extend(tracks)
            return self

        if tracks is not None:
            return dfr.addCallback(populate_model, tracks)

        def got_tracks(tracks):
            self.model.extend(tracks)
            return self

        def sort_tracks_number_title(result_set):
            result_set.order_by(MusicTrack.track_number, MusicTrack.title)
            return result_set.all()

        def sort_tracks_title_client_side(tracks):
            trash_letters = " .,-_"
            tracks.sort(key=lambda t: t.title.strip(trash_letters).lower())
            return tracks

        def get_tracks(self):
            store = common.application.store
            if self.artist is not None:
                # oh, hey, let's fetch all the tracks of the given artist
                dfr = store.find(MusicTrack,
                                 MusicTrack.file_path == File.path,
                                 File.hidden == False,
                                 MusicTrack.file_path == TrackArtist.track_path,
                                 TrackArtist.artist_name == self.artist.name)
                # Add client side ordering - can this be done "server" side? Is
                # there a better way to trash spurious characters?
                dfr.addCallback(lambda result_set: result_set.all())
                dfr.addCallback(sort_tracks_title_client_side)
            elif self.album is not None:
                # FIXME: in this case we should not display an alphabetical
                # fastscroller, which does not make sense for a small number of
                # tracks and will be broken anyway, because the tracks are not
                # ordered alphabetically. See bug #253061 on Launchpad.
                dfr = store.find(MusicTrack, MusicTrack.album_name == self.album.name,
                                 MusicTrack.file_path == File.path,
                                 File.hidden == False)
                dfr.addCallback(sort_tracks_number_title)
            else:
                dfr = store.find(MusicTrack, MusicTrack.title != None,
                                 MusicTrack.file_path == File.path,
                                 File.hidden == False)
                # Add client side ordering - can this be done "server" side? Is
                # there a better way to trash spourious characters?
                dfr.addCallback(lambda result_set: result_set.all())
                dfr.addCallback(sort_tracks_title_client_side)

            dfr.addCallback(got_tracks)
            return dfr

        def retrieve_tracks(controller):
            if tracks is None:
                return album.get_tracks()
            return tracks

        def insert_tracks(tracks):
            self.model.extend(tracks)
            return self

        # retrieve tracks for not-database albums
        if album and getattr(album, 'uri', None):
            dfr.addCallback(retrieve_tracks)
            dfr.addCallback(insert_tracks)
        # retrieve tracks for database albums
        else:
            dfr.addCallback(get_tracks)

        return dfr


class DBTracksViewMode(TracksViewMode):
    def get_label(self, item):
        if isinstance(item, Action):
            return defer.succeed(item.title)
        return super(DBTracksViewMode, self).get_label(item)

    def get_sublabel(self, item):
        if isinstance(item, Action):
            return defer.succeed(item.subtitle)

        def set_values(result_list):
            result_list_len = len(result_list)
            if result_list_len == 0:
                # nothing found
                sublabel = ''
            elif result_list_len == 1:
                sublabel = result_list[0]
            elif result_list_len > 1:
                sublabel = ', '.join(result_list)
            return sublabel

        dfr = item.get_artists()
        dfr.addCallback(set_values)
        return dfr

    def get_default_image(self, item):
        if isinstance(item, Action):
            return item.icon
        return super(DBTracksViewMode, self).get_default_image(item)

    def get_image(self, item, theme):
        if isinstance(item, Action):
            return None
        return super(DBTracksViewMode, self).get_image(item, theme)

    def get_preview_image(self, item, theme):
        if isinstance(item, Action):
            return None
        return super(DBTracksViewMode, self).get_preview_image(item, theme)


class DBTracksInAlbumViewMode(DBTracksViewMode):

    def get_label(self, item):
        if isinstance(item, Action):
            return defer.succeed(item.title)

        # Strip trash letter - defined in TracksViewMode
        title = item.title.strip(self.trash_letters)
        if item.track_number:
            title = '%s. %s' % (str(item.track_number).zfill(2), title)
        return defer.succeed(title)


class TracksPreviewListController(TracksDbController, DoubleLineMenuItemPreviewListController):
    view_mode = DBTracksViewMode
    fastscroller = True

    def initialize(self, album=None, track=None, tracks=None, artist=None):
        # Change the visualization of tracks, depending on whether they
        # are displayed in one album or not
        if album:
            self.view_mode = DBTracksInAlbumViewMode

        return super(TracksPreviewListController, self).initialize(album=album,
                                                                   track=track,
                                                                   tracks=tracks,
                                                                   artist=artist)

class TracksDbCoverflowController(TracksDbController, ImageWithReflectionCoverflowController):
    view_mode = DBTracksViewMode

class TracksDbGridController(TracksDbController, GridItemGridController):
    view_mode = DBTracksViewMode

class TracksDbListSwitcherController(ListSwitcherController):
    modes = [TracksPreviewListController,
             TracksDbCoverflowController,
             TracksDbGridController]
    default_mode = TracksPreviewListController


# A list of important genres
GENRES = [u"Acid Jazz", u"Alternative", u"Alternative & Punk",
          u"Alternative Pop/Rock", u"Alternative Rock", u"AlternRock",
          u"Ambient", u"Blues", u"Blues/R&B", u"Books & Spoken",
          u"Children's Music", u"Classic Rock", u"Classical", u"Comedy",
          u"Dance", u"Easy Listening", u"Electronic", u"Electronica & Dance",
          u"Electronice/Dance", u"Entertainment", u"Folk", u"Funk",
          u"Games", u"Garage/Punk Rock Revival", u"General Alternative",
          u"Hard Rock", u"Hip Hop", u"Hip Hop/Rap", u"Hip-Hop",
          u"Hip-Hop/Rap", u"Holiday", u"House", u"Indi", u"Industrial",
          u"Jazz", u"Latin", u"Metal", u"Music Videos", u"New Age",
          u"Other", u"Pop", u"Punk", u"R&B", u"R&B/Soul", u"Rap", u"Raggae",
          u"Religious", u"Rock", u"Rock & Roll", u"Soundtrack", u"Techno",
          u"Trance", u"World"]


class GenresDbController(HierarchyController):

    empty_label = _('There are no tracks in this section')
    empty_icon = 'elisa.plugins.poblesec.by_genre'

    def initialize(self, all_genres=False):
        self.all_genres = all_genres

        deferred = super(GenresDbController, self).initialize()

        def got_genres(genres, all_genres):
            def iterate(genres, models):
                for genre in genres:
                    link = Link()
                    link.controller_path = '/poblesec/database/music/genre'
                    link.label = genre
                    link.icon = 'elisa.plugins.poblesec.by_genre'
                    link.controller_args = {'genre': genre}
                    models.append(link)
                    yield link

            def iterate_done(result, models):
                self.model.extend(models)
                return self

            models = []

            if not all_genres:
                link = Link()
                link.controller_path = '/poblesec/database/music/genres'
                link.label = _("All genres")
                link.icon = 'elisa.plugins.poblesec.by_genre'
                link.controller_args = {'all_genres': True}
                models.append(link)

            dfr = task.coiterate(iterate(genres, models))
            dfr.addCallback(iterate_done, models)
            return dfr

        def sort_by_genre(result_set):
            result_set.order_by(MusicTrack.genre)
            return result_set

        def set_distinct(result_set):
            result_set.config(distinct=True)
            return result_set

        def get_genres(result_set):
            return result_set.values(MusicTrack.genre)

        def get_tracks(self):
            store = common.application.store
            if all_genres:
                dfr = store.find(MusicTrack, MusicTrack.genre != None)
            else:
                dfr = store.find(MusicTrack, MusicTrack.genre.is_in(GENRES))
            return dfr

        deferred.addCallback(get_tracks)
        deferred.addCallback(set_distinct)
        deferred.addCallback(sort_by_genre)
        deferred.addCallback(get_genres)
        deferred.addCallback(got_genres, all_genres)
        return deferred

    def node_clicked(self, widget, item):
        if isinstance(item, Action):
            item.run()
        else:
            browser = self.frontend.retrieve_controllers('/poblesec/browser')[0]
            path = item.controller_path
            args = item.controller_args
            dfr = browser.history.append_controller(path, item.label, **args)


class GenresDbPreviewListController(GenresDbController, MenuItemPreviewListController):
    view_mode = SectionMenuViewMode

class GenresDbCoverflowController(GenresDbController, ImageWithReflectionCoverflowController):
    view_mode = SectionMenuViewMode

class GenresDbGridController(GenresDbController, GridItemGridController):
    view_mode = SectionMenuViewMode

class GenresDbListSwitcherController(ListSwitcherController):
    modes = [GenresDbPreviewListController,
             GenresDbCoverflowController,
             GenresDbGridController]
    default_mode = GenresDbPreviewListController


class GenreDbController(GenericAlbumsDbController):

    def initialize(self, genre=None):
        self.genre = genre
        store = common.application.store

        deferred = super(GenreDbController, self).initialize()

        def got_albums(albums):
            self.model.extend(albums)
            return self

        def sort_by(result_set):
            result_set.order_by(MusicAlbum.name)
            return result_set.all()

        def distinct_all(result_set):
            result_set.config(distinct=True)
            return result_set.all()

        def get_albums(results):
            return store.find(MusicAlbum, MusicAlbum.name.is_in(results))

        def get_album_names(self):
            dfr = store.find(MusicTrack.album_name,
                    MusicTrack.genre == unicode(genre), \
                    MusicTrack.file_path == File.path, \
                    File.hidden == False)
            dfr.addCallback(distinct_all)
            return dfr

        deferred.addCallback(get_album_names)
        deferred.addCallback(get_albums)
        deferred.addCallback(sort_by)
        deferred.addCallback(got_albums)
        return deferred


class GenrePreviewListController(GenreDbController, DoubleLineMenuItemPreviewListController):
    view_mode = AlbumsDbViewMode
    fastscroller = True

    def item_to_label(self, item):
        if isinstance(item, Action):
            return '#'
        return item.name

class GenreDbCoverflowController(GenreDbController, ImageWithReflectionCoverflowController):
    view_mode = AlbumsDbViewMode

class GenreDbGridController(GenreDbController, GridItemGridController):
    view_mode = AlbumsDbViewMode

class GenreDbListSwitcherController(ListSwitcherController):
    modes = [GenrePreviewListController,
             GenreDbCoverflowController,
             GenreDbGridController]
    default_mode = GenrePreviewListController


class DecadesDbController(HierarchyController):

    def initialize(self, all_genres=False):
        self.all_genres = all_genres

        deferred = super(DecadesDbController, self).initialize()

        def iterate(models):
            # FIXME: What about other decades (10's, 20's, 30's, 40's, ...)?
            #        And what about other centuries?
            for decade in (50, 60, 70, 80, 90):
                link = Link()
                link.label = _("%d's") % decade
                link.icon = 'elisa.plugins.poblesec.by_decade'
                link.controller_args = \
                    {'begin': datetime.datetime(1900 + decade, 1, 1),
                     'end': datetime.datetime(1909 + decade, 12, 31)}
                models.append(link)
                yield link

        def iterate_done(result, models):
            self.model.extend(models)
            return self

        models = []

        def populate(self):
            dfr = task.coiterate(iterate(models))
            dfr.addCallback(iterate_done, models)
            return dfr

        deferred.addCallback(populate)
        return deferred

    def node_clicked(self, widget, item):
        if isinstance(item, Action):
            item.run()
        else:
            browser = self.frontend.retrieve_controllers('/poblesec/browser')[0]
            path = '/poblesec/database/music/time'
            args = item.controller_args
            dfr = browser.history.append_controller(path, item.label, **args)


class DecadesDbPreviewListController(DecadesDbController, MenuItemPreviewListController):
    view_mode = SectionMenuViewMode

class DecadesDbCoverflowController(DecadesDbController, ImageWithReflectionCoverflowController):
    view_mode = SectionMenuViewMode

class DecadesDbGridController(DecadesDbController, GridItemGridController):
    view_mode = SectionMenuViewMode

class DecadesDbListSwitcherController(ListSwitcherController):
    modes = [DecadesDbPreviewListController,
             DecadesDbCoverflowController,
             DecadesDbGridController]
    default_mode = DecadesDbPreviewListController


class TimeDbController(GenericAlbumsDbController):

    empty_icon = 'elisa.plugins.poblesec.by_decade'

    def initialize(self, begin=None, end=None):
        self.begin = begin
        self.end = end

        deferred = super(TimeDbController, self).initialize()

        def got_albums(albums):
            if not albums:
                # FIXME: We should never raise generic exceptions, let's find
                #        a suitable specific exception for this case.
                raise Exception('Missing Data')
            self.model.extend(albums)
            return self

        def sort_by_name(result_set):
            result_set.order_by(MusicAlbum.name)
            return result_set.all()

        def get_albums(self):
            store = common.application.store
            dfr = store.find(MusicAlbum,
                             (MusicAlbum.release_date >= begin) & \
                             (MusicAlbum.release_date <= end))
            return dfr

        def missing_data(failure):
            self.warning("Missing Data for decade.")
            return self

        deferred.addCallback(get_albums)
        deferred.addCallback(sort_by_name)
        deferred.addCallback(got_albums)
        deferred.addErrback(missing_data)
        return deferred


class TimeDbPreviewListController(TimeDbController, DoubleLineMenuItemPreviewListController):
    view_mode = AlbumsDbViewMode

class TimeDbCoverflowController(TimeDbController, ImageWithReflectionCoverflowController):
    view_mode = AlbumsDbViewMode

class TimeDbGridController(TimeDbController, GridItemGridController):
    view_mode = AlbumsDbViewMode

class TimeDbListSwitcherController(ListSwitcherController):
    modes = [TimeDbPreviewListController,
             TimeDbCoverflowController,
             TimeDbGridController]
    default_mode = TimeDbPreviewListController
