# -*- coding: utf-8 -*-
# Elisa - Home multimedia server
# Copyright (C) 2006-2008 Fluendo Embedded S.L. (www.fluendo.com).
# All rights reserved.
#
# This file is available under one of two license agreements.
#
# This file is licensed under the GPL version 3.
# See "LICENSE.GPL" in the root of this distribution including a special
# exception to use Elisa with Fluendo's plugins.
#
# The GPL part of Elisa is also available under a commercial licensing
# agreement from Fluendo.
# See "LICENSE.Elisa" in the root directory of this distribution package
# for details on that license.
#
# Author:  Philippe Normand <philippe@fluendo.com>

from elisa.core import common
from elisa.core.media_uri import MediaUri
from elisa.core.utils.i18n import install_translation

from elisa.plugins.poblesec.base.list_switcher import ListSwitcherController
from elisa.plugins.poblesec.base.preview_list import \
    DoubleLineMenuItemPreviewListController
from elisa.plugins.poblesec.base.coverflow import \
    ImageWithReflectionCoverflowController
from elisa.plugins.poblesec.base.grid import GridItemGridController
from elisa.plugins.poblesec.link import Link

from elisa.plugins.poblesec.actions import Action

from elisa.plugins.database.models import TVShow, TVSeason, TVEpisode
from elisa.plugins.database.video_controller import _get_thumbnail, \
     VideoController, VideoViewMode, VideoTrackViewMode, SynopsisController

from elisa.plugins.database.actions import VideoPlayAllAction, \
     VideoAddToFavoritesAction, TVEpisodeViewSynopsisAction

from elisa.plugins.database.widgets.synopsis import TVEpisodeSynopsisWidget

from elisa.plugins.favorites.models import FavoritesItem

from twisted.internet import defer

import cgi

_ = install_translation('database')

def tvshows_lib_decorator(controller):
    link = Link()
    link.controller_path = '/poblesec/database/tvshows/list'
    link.label = _('TV Shows')
    link.icon = 'elisa.plugins.poblesec.tvshow'
    controller.model.append(link)

    return defer.succeed(None)

class AllTVSeasonsController(VideoController):
    child_controller_path = '/poblesec/database/tvseason/list'
    empty_label = _('There are no TV shows in this section')

    def initialize(self):
        dfr = super(AllTVSeasonsController, self).initialize()
        dfr.addCallback(self._load)
        return dfr

    def _load(self, result):
        def got_items(models):
            self.model.extend(models)
            return self

        def get_items(result_set):
            return result_set.all()

        def sort_by_show_name(result_set):
            result_set.order_by(TVShow.name, TVSeason.number)
            return result_set

        store = common.application.store
        dfr = store.find(TVSeason, TVSeason.tvshow_id == TVShow.id)
        dfr.addCallback(sort_by_show_name)
        dfr.addCallback(get_items)
        dfr.addCallback(got_items)
        return dfr

    def crumb_title(self, item):
        def got_show(show):
            title = "%s Season %s" % (show.name, item.number)
            return title

        dfr = item.tvshow
        dfr.addCallback(got_show)
        return dfr

    def child_controller_args(self, item):
        return defer.succeed(dict(season=item))

    def make_actions(self):
        actions = []
        return actions

class TVSeasonsViewMode(VideoViewMode):

    def get_label(self, item):
        if isinstance(item, Action):
            return defer.succeed(item.title)

        def got_show(show):
            return show.name

        dfr = item.tvshow
        dfr.addCallback(got_show)
        return dfr

    def get_sublabel(self, item):
        if isinstance(item, Action):
            return defer.succeed(item.subtitle)
        return defer.succeed(_("Season %s" % item.number))

    def get_default_image(self, item):
        if isinstance(item, (Action, Link)):
            return item.icon
        resource = 'elisa.plugins.poblesec.tvshow'
        return resource

    def get_image(self, item, theme):
        if isinstance(item, Action):
            return None
        if getattr(item, 'thumbnail_path', None) is not None:
            return defer.succeed(item.thumbnail_path)

        def got_uri(poster_uri):
            if poster_uri:
                return _get_thumbnail(item, poster_uri)
            else:
                return None

        dfr = item.poster_uri()
        dfr.addCallback(got_uri)
        return dfr


    def get_preview_image(self, item, theme):
        if isinstance(item, Action):
            return None
        if getattr(item, 'thumbnail_path', None) is not None:
            return item.thumbnail_path
        else:
            return None

class TVSeasonController(VideoController):
    child_controller_path = '/poblesec/database/tvepisode/track'

    def initialize(self, season):
        self.season = season
        dfr = super(TVSeasonController, self).initialize()
        dfr.addCallback(self._load)
        return dfr

    def _load(self, result):

        def got_items(models):
            self.model.extend(models)
            return self

        def get_items(result_set):
            return result_set.all()

        dfr = self.season.episodes.order_by(TVEpisode.number)
        dfr.addCallback(get_items)
        dfr.addCallback(got_items)
        return dfr

    def crumb_title(self, item):
        title = item.name
        if title is None:
            title = _("Episode %s" % item.number)
        return defer.succeed(title)

    def child_controller_args(self, item):
        return defer.succeed(dict(name=item.name,file_path=item.file_path,
                                  db_model=item))

    def make_actions(self):
        actions = []
        action = VideoPlayAllAction(self)
        action.title = _('Play All')
        action.subtitle = _('Play All Episodes')
        actions.append(action)
        return actions

class TVSeasonViewMode(VideoViewMode):

    def get_label(self, item):
        if isinstance(item, Action):
            return defer.succeed(item.title)

        return defer.succeed(_("Episode %s" % item.number))

    def get_sublabel(self, item):
        if isinstance(item, Action):
            return defer.succeed(item.subtitle)
        return defer.succeed(item.name)

    def get_default_image(self, item):
        if isinstance(item, (Action, Link)):
            return item.icon
        resource = 'elisa.plugins.poblesec.tvshow'
        return resource

    def get_image(self, item, theme):
        if isinstance(item, Action):
            return None
        if getattr(item, 'thumbnail_path', None) is not None:
            return defer.succeed(item.thumbnail_path)

        if item.poster_uri:
            return _get_thumbnail(item, MediaUri(item.poster_uri))
        else:
            return None

    def get_preview_image(self, item, theme):
        if isinstance(item, Action):
            return None
        if getattr(item, 'thumbnail_path', None) is not None:
            return item.thumbnail_path
        else:
            return None

class TVEpisodeTrackController(VideoController):

    def initialize(self, name=None, file_path=None, db_model=None):
        self.name = name
        self.file_path = file_path
        self.db_model = db_model
        return super(TVEpisodeTrackController, self).initialize()

    def is_empty(self):
        return super(TVEpisodeTrackController, self).is_empty() and self.file_path is None

    def make_actions(self):
        actions = []

        action = VideoPlayAllAction(self)
        action.title = _('Play Episode')
        action.subtitle = _("Play '%s'") % self.name
        actions.append(action)

        action = TVEpisodeViewSynopsisAction(self)
        action.title = _('View Synopsis')
        action.crumb_title = _('Synopsis')
        action.subtitle = _('View Synopsis of %s') % self.name
        action.args = {'file_path': self.file_path, 'db_model': self.db_model}
        actions.append(action)

        return actions

class TVEpisodeSynopsisController(SynopsisController):
    synopsis_widget_class = TVEpisodeSynopsisWidget
    video_type = 'tvepisode'

    # Maximum lengths of text that will be displayed, respectively for the
    # overview (aka synopsis) and for the guest star list.
    _overview_max_length = 490
    _guest_stars_max_length = 190

    def set_frontend(self, frontend):
        super(TVEpisodeSynopsisController, self).set_frontend(frontend)

        def got_results(results):
            dfr = results.one()
            dfr.addCallback(got_episode)
            return dfr

        def got_episode(episode):
            overview = episode.overview
            if not overview:
                overview = _("No synopsis available")
            else:
                max_size = self._overview_max_length
                if len(overview) > max_size:
                    overview = overview[:max_size] + "..."
            self._synopsis_widget.overview = overview
            if episode.guest_stars:
                guest_stars = episode.guest_stars
                max_size = self._guest_stars_max_length
                if len(guest_stars) > max_size:
                    guest_stars = guest_stars[:max_size] + "..."
                guest_stars = _("<b>Guest Stars:</b> %s") % cgi.escape(guest_stars)
                self._synopsis_widget.guest_stars = guest_stars

            dfr = episode.season
            dfr.addCallback(got_season, episode)
            return dfr

        def got_season(season, episode):
            dfr = season.tvshow
            dfr.addCallback(got_show, season, episode)

        def got_show(show, season, episode):
            self._synopsis_widget.title = show.name or show.filesystem_name

            if episode.name:
                episode_name = _("Season %d, Episode %d: %s") % (season.number,
                                                                 episode.number,
                                                                 episode.name)
            else:
                episode_name = _("Season %d, Episode %d") % (season.number,
                                                             episode.number)

            self._synopsis_widget.episode_name = episode_name
            cover_uri = episode.poster_uri
            if not cover_uri:
                theme = self.frontend.get_theme()
                icon_path = theme.get_resource('elisa.plugins.poblesec.file_video')
                cover_uri = 'file://%s' % icon_path
            dfr = _get_thumbnail(episode, MediaUri(cover_uri))
            dfr.addCallback(got_thumbnail)
            return dfr

        def got_thumbnail(poster_path):
            self._synopsis_widget.poster_path = poster_path
            self.widget.add(self._synopsis_widget)
            self._synopsis_widget.visible = True

        dfr = common.application.store.find(TVEpisode, file_path=self.file_path)
        dfr.addCallback(got_results)
        return dfr

# All TVSeasons
class AllTVSeasonsVerticalWithPreview(AllTVSeasonsController,
                                      DoubleLineMenuItemPreviewListController):
    fastscroller = False
    view_mode = TVSeasonsViewMode

class AllTVSeasonsCoverflow(AllTVSeasonsController,
                            ImageWithReflectionCoverflowController):
    view_mode = TVSeasonsViewMode

class AllTVSeasonsGrid(AllTVSeasonsController, GridItemGridController):
    view_mode = TVSeasonsViewMode

class AllTVSeasonsListSwitcherController(ListSwitcherController):
    modes = [AllTVSeasonsVerticalWithPreview,
             AllTVSeasonsCoverflow,
             AllTVSeasonsGrid]
    default_mode = AllTVSeasonsVerticalWithPreview


#  TVSeason
class TVSeasonVerticalWithPreview(TVSeasonController,
                                  DoubleLineMenuItemPreviewListController):
    fastscroller = False
    view_mode = TVSeasonViewMode

class TVSeasonCoverflow(TVSeasonController,
                        ImageWithReflectionCoverflowController):
    view_mode = TVSeasonViewMode

class TVSeasonGrid(TVSeasonController, GridItemGridController):
    view_mode = TVSeasonViewMode

class TVSeasonListSwitcherController(ListSwitcherController):
    modes = [TVSeasonVerticalWithPreview,
             TVSeasonCoverflow,
             TVSeasonGrid]
    default_mode = TVSeasonVerticalWithPreview

# TVEpisode tracks
class TVEpisodeTrackVerticalWithPreview(TVEpisodeTrackController, DoubleLineMenuItemPreviewListController):
    view_mode = VideoTrackViewMode

class TVEpisodeTrackCoverflow(TVEpisodeTrackController, ImageWithReflectionCoverflowController):
    view_mode = VideoTrackViewMode

class TVEpisodeTrackGrid(TVEpisodeTrackController, GridItemGridController):
    view_mode = VideoTrackViewMode

class TVEpisodeTrackListSwitcherController(ListSwitcherController):
    modes = [TVEpisodeTrackVerticalWithPreview,
             TVEpisodeTrackCoverflow,
             TVEpisodeTrackGrid]
    default_mode = TVEpisodeTrackVerticalWithPreview
