# -*- coding: utf-8 -*-
# Elisa - Home multimedia server
# Copyright (C) 2006-2008 Fluendo Embedded S.L. (www.fluendo.com).
# All rights reserved.
#
# This file is available under one of two license agreements.
#
# This file is licensed under the GPL version 3.
# See "LICENSE.GPL" in the root of this distribution including a special
# exception to use Elisa with Fluendo's plugins.
#
# The GPL part of Elisa is also available under a commercial licensing
# agreement from Fluendo.
# See "LICENSE.Elisa" in the root directory of this distribution package
# for details on that license.
#
# Author:  Jesus Corrius <jcorrius@fluendo.com>

from elisa.core import common
from elisa.core.media_uri import MediaUri
from elisa.core.utils.i18n import install_translation
from elisa.core.metadata_manager import IncompleteMetadataResponse
from elisa.core.default_config import PICTURES_CACHE
from elisa.core.input_event import *

from elisa.plugins.pigment.pigment_controller import PigmentController

from elisa.plugins.poblesec.base.hierarchy import HierarchyController
from elisa.plugins.poblesec.base.list import GenericListViewMode
from elisa.plugins.poblesec.base.list_switcher import ListSwitcherController
from elisa.plugins.poblesec.base.preview_list import \
    MenuItemPreviewListController, DoubleLineMenuItemPreviewListController
from elisa.plugins.poblesec.base.coverflow import \
    ImageWithReflectionCoverflowController
from elisa.plugins.poblesec.base.grid import GridItemGridController
from elisa.plugins.poblesec.link import Link

from elisa.plugins.poblesec.actions import Action
from elisa.plugins.database.actions import VideoPlayAllAction, \
     VideoShuffleAction, VideoAddToFavoritesAction, MovieViewSynopsisAction

from elisa.plugins.favorites.models import FavoritesItem

from elisa.plugins.base.models.media import PlayableModel

from elisa.plugins.database.models import Movie, Video, File

from elisa.plugins.database.widgets.synopsis import MovieSynopsisWidget, ActionItem

from twisted.internet import defer
from twisted.internet import task

import os, datetime
from random import shuffle

try:
    from hashlib import md5
except ImportError:
    # hashlib is new in Python 2.5
    from md5 import md5

_ = install_translation('database')

def video_lib_decorator(controller):
    link = Link()
    link.controller_path = '/poblesec/database/video_library'
    link.label = _('Video Library')
    link.icon = 'elisa.plugins.poblesec.video_library'
    controller.model.append(link)

    return defer.succeed(None)

def video_lib_unclassified_videos_decorator(controller):
    tracks = Link()
    tracks.controller_path = '/poblesec/database/video/unclassified'
    tracks.label = _('Unorganized videos')
    tracks.icon = 'elisa.plugins.poblesec.unorganized_videos'
    controller.model.append(tracks)
    return defer.succeed(None)

def video_lib_all_movies_decorator(controller):
    tracks = Link()
    tracks.controller_path = '/poblesec/database/movie/list'
    tracks.label = _('Movies')
    tracks.icon = 'elisa.plugins.poblesec.video_section'
    controller.model.append(tracks)
    return defer.succeed(None)

# TODO: factorize this code in the core of Elisa in a caching utilities module
def _thumbnail_file(thumbnail_uri):
    if not os.path.exists(PICTURES_CACHE):
        os.makedirs(PICTURES_CACHE, 0755)
    thumbnail = md5(str(thumbnail_uri)).hexdigest() + '.jpg'
    return os.path.join(PICTURES_CACHE, thumbnail)


def _get_thumbnail(item, thumbnail_uri):
    if hasattr(item, 'thumbnail_path'):
        return defer.succeed(item.thumbnail_path)

    thumbnail_file = _thumbnail_file(thumbnail_uri)
    if os.path.exists(thumbnail_file):
        item.thumbnail_path = thumbnail_file
        return defer.succeed(thumbnail_file)

    if thumbnail_uri.scheme == 'file':
        # Image is local (default picture)
        thumbnail_file = thumbnail_uri._path
        item.thumbnail_path = thumbnail_file
        return defer.succeed(thumbnail_file)

    # Download the thumbnail first
    def got_thumbnail(data_model):
        f = file(thumbnail_file, 'wb')
        f.write(data_model.data)
        f.close()
        item.thumbnail_path = thumbnail_file
        return thumbnail_file

    data_model, dfr = common.application.resource_manager.get(thumbnail_uri)
    dfr.addCallback(got_thumbnail)
    return dfr

class ThumbnailMixin(object):
    thumbnail_type = 'video'

    def __init__(self):
        # FIXME: we need the frontend to get a reference to the gst_metadata
        # instance. This a cheap - UGLY - way to get the frontend without
        # changing a lot of client code. It is really ugly as we assume there
        # is only one frontend, which might not be the case in the future...
        frontend = common.application.interface_controller.frontends.values()[0]
        # Retrieve and store a reference to gst_metadata
        controllers = frontend.retrieve_controllers('/poblesec')
        try:
            self.gst_metadata = controllers[0].gst_metadata
        except AttributeError:
            msg = 'GstMetadata missing: thumbnails will not be generated.'
            self.warning(msg)
            self.gst_metadata = None

    def _updated_thumbnail(self, thumbnail, item):
        if thumbnail != None:
            item.thumbnail_uri = unicode(thumbnail)
        return thumbnail

    def _request_thumbnail(self, path):
        if self.gst_metadata is None:
            msg = 'No gst_metadata.'
            return defer.fail(IncompleteMetadataResponse(msg))

        def got_metadata(metadata):
            return metadata['thumbnail']

        metadata = {'uri': MediaUri("file://%s" % path), 'thumbnail': None,
                    'mime_type': None, 'file_type': self.thumbnail_type}

        dfr = self.gst_metadata.get_metadata(metadata)
        dfr.addCallback(got_metadata)
        return dfr

class VideoViewMode(GenericListViewMode, ThumbnailMixin):

    def get_label(self, item):
        if isinstance(item, Action):
            return defer.succeed(item.title)
        elif isinstance(item, Link):
            return defer.succeed(item.label)
        return defer.succeed(os.path.basename(item.file_path))

    def get_default_image(self, item):
        if isinstance(item, (Action, Link)):
            return item.icon
        resource = 'elisa.plugins.poblesec.file_video'
        return resource

    def get_image(self, item, theme):
        if isinstance(item, (Action, Link)):
            return None
        if hasattr(item, 'thumbnail_uri') and item.thumbnail_uri != None and \
            item.thumbnail_uri != "None":
            return defer.succeed(item.thumbnail_uri)

        dfr = self._request_thumbnail(item.file_path)
        dfr.addCallback(self._updated_thumbnail, item)
        return dfr

    def get_preview_image(self, item, theme):
        if isinstance(item, (Action, Link)):
            return None
        if hasattr(item, 'thumbnail_uri') and item.thumbnail_uri != None and \
            item.thumbnail_uri != "None":
            return item.thumbnail_uri
        else:
            return None

class MovieViewMode(GenericListViewMode):

    def get_label(self, item):
        return defer.succeed(item.title)

    def get_default_image(self, item):
        if isinstance(item, Action):
            return item.icon
        resource = 'elisa.plugins.poblesec.file_video'
        return resource

    def get_image(self, item, theme):
        if isinstance(item, Action):
            return None
        if getattr(item, 'thumbnail_path', None) is not None:
            return defer.succeed(item.thumbnail_path)

        if item.cover_uri:
            return _get_thumbnail(item, MediaUri(item.cover_uri))
        else:
            return None

    def get_preview_image(self, item, theme):
        if isinstance(item, Action):
            return None
        if getattr(item, 'thumbnail_path', None) is not None:
            return item.thumbnail_path
        else:
            return None


class VideoController(HierarchyController):
    child_controller_path = '/poblesec/database/video/track'
    empty_label = _('There are no videos in this section')
    empty_icon = 'elisa.plugins.poblesec.file_video'

    def initialize(self):
        dfr = super(VideoController, self).initialize()
        self.actions.extend(self.make_actions())
        return dfr

    def node_clicked(self, widget, item):
        if isinstance(item, Action):
            dfr = item.run()
        elif hasattr(item, 'action') and issubclass(item.action, Action):
            action = item.action(self)
            dfr = action.run(item)
        else:
            dfr = self.next_level(item)
        return dfr

    def next_level(self, item):

        def got_title(title):
            dfr = self.child_controller_args(item)
            dfr.addCallback(go_next_level, title)
            return dfr

        def go_next_level(args, title):
            controllers = self.frontend.retrieve_controllers('/poblesec/browser')
            history = controllers[0].history
            dfr = history.append_controller(self.child_controller_path,
                                            title, **args)
            return dfr

        dfr = self.crumb_title(item)
        dfr.addCallback(got_title)
        return dfr

    def crumb_title(self, item):
        # We are using the basename of the file_path as the name
        # parameter this should be changed when we can get a proper
        # name
        return defer.succeed(os.path.basename(item.file_path))

    def child_controller_args(self, item):
        return defer.succeed(dict(name=os.path.basename(item.file_path),
                                  file_path=item.file_path,
                                  db_model=item))


    def make_actions(self):
        actions = []

        action = VideoPlayAllAction(self)
        action.title = _('Play All')
        action.subtitle = _('Play All Videos')
        actions.append(action)

        action = VideoShuffleAction(self)
        action.title = _('Shuffle & Play')
        action.subtitle = _('Shuffle & Play All Videos')
        actions.append(action)

        return actions

class AllVideosController(VideoController):

    def initialize(self):
        dfr = super(AllVideosController, self).initialize()
        dfr.addCallback(self._load)
        return dfr

    def _load(self, result):
        def got_items(models):
            self.model.extend(models)
            return self

        def get_items(result_set):
            return result_set.all()

        def sort_by_name(result_set):
            result_set.config(distinct=True)
            result_set.order_by(Video.name, Video.file_path)
            return result_set

        dfr = Video.unclassified_videos()
        dfr.addCallback(sort_by_name)
        dfr.addCallback(get_items)
        dfr.addCallback(got_items)
        return dfr

class AllMoviesController(VideoController):
    child_controller_path = '/poblesec/database/movie/track'

    def initialize(self):
        dfr = super(AllMoviesController, self).initialize()
        dfr.addCallback(self._load)
        return dfr

    def _load(self, result):

        def got_items(models):
            self.model.extend(models)
            return self

        def get_items(result_set):
            return result_set.all()

        def sort_by_title(result_set):
            result_set.order_by(Movie.title, Movie.file_path)
            return result_set

        store = common.application.store

        dfr = store.find(Movie, Movie.file_path == File.path)
        dfr.addCallback(sort_by_title)
        dfr.addCallback(get_items)
        dfr.addCallback(got_items)
        return dfr

    def crumb_title(self, item):
        return defer.succeed(item.title)

    def child_controller_args(self, item):
        return defer.succeed(dict(title=os.path.basename(item.title),
                                  file_path=item.file_path,
                                  db_model=item))

    def make_actions(self):
        actions = []
        return actions


class UnclassifiedVideosController(HierarchyController):

    def initialize(self):
        dfr = super(UnclassifiedVideosController, self).initialize()

        all_videos = Link()
        all_videos.controller_path = '/poblesec/database/video/list'
        all_videos.label = _('All Videos')
        all_videos.icon = 'elisa.plugins.poblesec.file_video'
        self.model.append(all_videos)

        by_date = Link()
        by_date.controller_path = '/poblesec/database/video/by_date'
        by_date.label = _('Browse by date')
        by_date.icon = 'elisa.plugins.poblesec.by_decade'
        self.model.append(by_date)

        return dfr

    def node_clicked(self, widget, item):
        if isinstance(item, Link):
            controllers = self.frontend.retrieve_controllers('/poblesec/browser')
            history = controllers[0].history
            dfr = history.append_controller(item.controller_path,
                                            item.label)
            return dfr

class VideosMonthController(VideoController):

    empty_label = _('There are no videos in this section')
    empty_icon = 'elisa.plugins.poblesec.by_decade'

    def initialize(self, month=None, year=None):
        self.date = datetime.datetime(year, month, 1)

        dfr = super(VideosMonthController, self).initialize()

        def get_videos(result):
            try:
                last = datetime.datetime(year, month+1, 1)
            except ValueError:
                # we reached the month 12 + 1, so we have to skip to
                # january next year
                last = datetime.datetime(year+1, 1, 1)

            dfr = Video.unclassified_videos(Video.creation_time >= self.date,
                                            Video.creation_time < last,
                                            Video.file_path == File.path,
                                            File.hidden == False)
            return dfr

        def got_videos(videos):
            self.model.extend(videos)
            return self

        def sort_videos(result_set):
            result_set.order_by(Video.creation_time, Video.file_path)
            return result_set.all()

        dfr.addCallback(get_videos)
        dfr.addCallback(sort_videos)
        dfr.addCallback(got_videos)

        return dfr

    def make_actions(self):
        actions = []

        action = VideoPlayAllAction(self)
        action.title = _('Play All')
        try:
            subtitle = self.date.strftime(_("Videos from %B, %Y"))
        except TypeError:
            # strftime in python <=2.5 requires a str
            subtitle = self.date.strftime(_("Videos from %B, %Y").encode("utf-8", "replace"))
        action.subtitle = subtitle
        actions.append(action)

        action = VideoShuffleAction(self)
        action.title = _('Shuffle and Play')
        try:
            subtitle = self.date.strftime(_("Videos from %B, %Y"))
        except TypeError:
            # strftime in python <=2.5 requires a str
            subtitle = self.date.strftime(_("Videos from %B, %Y").encode("utf-8", "replace"))
        action.subtitle = subtitle
        actions.append(action)

        return actions

class VideoTimesViewMode(GenericListViewMode):

    def get_label(self, item):
        if isinstance(item, Action):
            return defer.succeed(item.title)
        try:
            return defer.succeed(item.strftime(_('%B, %Y')))
        except TypeError:
            # strftime in python <=2.5 requires a str
            return defer.succeed(item.strftime(_('%B, %Y').encode("utf-8", "replace")))

    def get_default_image(self, item):
        if isinstance(item, Action):
            return item.icon
        resource = 'elisa.plugins.poblesec.video_folders'
        return resource

    def get_image(self, item, theme):
        return None

    def get_preview_image(self, item, theme):
        if isinstance(item, Action):
            return None
        return None

class VideosTimesController(HierarchyController):

    query = "SELECT DISTINCT STRFTIME('%m %Y', creation_time)" \
            " from videos, files where videos.creation_time not NULL" \
            " and files.path = videos.file_path and files.hidden = 0" \
            " and files.path not in (select distinct file_path from movies)"\
            " and files.path not in (select distinct file_path from tvepisodes)"\
            " order by creation_time;"

    empty_label = _('There are no videos in this section')
    empty_icon = 'elisa.plugins.poblesec.by_decade'

    def initialize(self):
        dfr = super(VideosTimesController, self).initialize()

        def get_times(result):
            store = common.application.store
            dfr = store.execute(self.query)
            return dfr

        def get_all(result):
            return result.get_all()

        def iterate(rows):
            for row in rows:
                month, year = row[0].split()
                date = datetime.datetime(int(year), int(month), 1)
                self.model.append(date)
                yield date

        def got_times(times):
            return task.coiterate(iterate(times))

        def done(res):
            return self

        dfr.addCallback(get_times)
        dfr.addCallback(get_all)
        dfr.addCallback(got_times)
        dfr.addCallback(done)
        return dfr

    def make_actions(self):
        actions = []

        action = VideoPlayAllAction(self)
        action.title = _('Play All')
        action.sub_title = _('View all videos')
        actions.append(action)

        return actions

    def node_clicked(self, widget, model):
        if isinstance(model, Action):
            model.run()
        else:
            controllers = self.frontend.retrieve_controllers('/poblesec/browser')
            history = controllers[0].history
            try:
                dfr = history.append_controller('/poblesec/database/video/by_month',
                        model.strftime(_('%B, %Y')), year=model.year, \
                        month=model.month)
            except TypeError:
                # strftime in python <=2.5 requires a str
                month_date = model.strftime(_('%B, %Y').encode("utf-8", "replace"))
                dfr = history.append_controller('/poblesec/database/video/by_month',
                                                month_date, year=model.year,
                                                month=model.month)

class VideoTrackViewMode(VideoViewMode):

    def get_sublabel(self, item):
        if isinstance(item, Action):
            return defer.succeed(item.subtitle)


class VideoTrackController(VideoController):

    def initialize(self, name=None, file_path=None, db_model=None):
        self.name = name
        self.file_path = file_path
        self.db_model = db_model
        dfr = super(VideoTrackController, self).initialize()
        return dfr

    def is_empty(self):
        return super(VideoTrackController, self).is_empty() and self.file_path is None

    def make_actions(self):
        actions = []

        action = VideoPlayAllAction(self)
        action.title = _('Play Video')
        action.subtitle = _("Play '%s'") % self.name
        actions.append(action)

        action = VideoAddToFavoritesAction(self)
        action.untoggled_title = _('Add to Favorites')
        action.untoggled_subtitle = _("Add '%s' To Your Favorites") % self.name
        action.toggled_title = _('Remove From Favorites')
        action.toggled_subtitle = _("Remove '%s' From Your Favorites") % self.name
        actions.append(action)

        dfr = common.application.store.find(FavoritesItem,
                                            FavoritesItem.foreign_id == self.file_path,
                                            FavoritesItem.foreign_class == u'Video')
        dfr.addCallback(lambda rs: rs.all())
        dfr.addCallback(action.setup)

        return actions

class MovieTrackController(VideoController):

    def initialize(self, title=None, file_path=None, db_model=None):
        self.title = title
        self.file_path = file_path
        self.db_model = db_model
        dfr = super(MovieTrackController, self).initialize()
        return dfr

    def is_empty(self):
        return super(MovieTrackController, self).is_empty() and self.file_path is None

    def make_actions(self):
        actions = []

        # TODO: add ViewTrailer action

        action = VideoPlayAllAction(self)
        action.title = _('Play Movie')
        action.subtitle = _("Play '%s'") % self.title
        actions.append(action)

        action = MovieViewSynopsisAction(self)
        action.title = _('View Synopsis')
        action.crumb_title = _('Synopsis')
        action.subtitle = _('View Synopsis of %s') % self.title
        action.args = {'file_path': self.file_path, 'db_model': self.db_model}
        actions.append(action)

        return actions

# Synopsis
class SynopsisController(PigmentController):
    synopsis_widget_class = None

    # currently supported tvepisode and movie
    video_type = ''

    def initialize(self, file_path=None, db_model=None):
        self.file_path = file_path
        self.db_model = db_model
        self._widget_focus_id = self.widget.connect('focus', self._on_focus)
        self._clicked_id = None
        self._synopsis_widget = None

        def created(controller):
            if self.widget.focus:
                controller.widget.focus = self.widget.focus
            return self

        dfr = super(SynopsisController, self).initialize()
        dfr.addCallback(created)
        return dfr

    def _on_focus(self, widget, focus):
        if focus:
            self._synopsis_widget._selector.opacity = 255
        else:
            self._synopsis_widget._selector.opacity = 0

    def set_frontend(self, frontend):
        super(SynopsisController, self).set_frontend(frontend)
        self._synopsis_widget = self.synopsis_widget_class()

        controllers = self.frontend.retrieve_controllers('/poblesec/browser')
        history = controllers[0].history
        prev_controller = history._controllers[history.index-1].list_controller

        # actions
        self._play_action = ActionItem('elisa.plugins.poblesec.play_all',
                                       _("Play Now"))
        self._prev_action = ActionItem('', _('Previous'))
        self._next_action = ActionItem('', _('Next'))

        actions = [self._play_action,]
        self._synopsis_widget.actions.extend(actions)

        self._clicked_id = self._synopsis_widget.actions_list.connect('item-clicked',
                                                                      self._action_clicked)

    def clean(self):
        if self._widget_focus_id is not None:
            self.widget.disconnect(self._widget_focus_id)
            self._widget_focus_id = None

        if self._synopsis_widget:
            if self._clicked_id is not None:
                self._synopsis_widget.actions_list.disconnect(self._clicked_id)
                self._clicked_id = None

            self._synopsis_widget.clean()
            self._synopsis_widget = None

        return super(SynopsisController, self).clean()

    def _action_clicked(self, widget, action):
        callbacks = {self._play_action: self._play_clicked,
                     self._prev_action: self._prev_clicked,
                     self._next_action: self._next_clicked}
        callback = callbacks[action]
        callback()

    def _play_clicked(self):
        poblesec = self.frontend.retrieve_controllers('/poblesec')[0]
        player = poblesec.video_player.player
        poblesec.show_video_player()
        player.play_model(self.db_model)

    def _next_clicked(self):
        return self._next_or_prev_synopsis(1)

    def _prev_clicked(self):
        return self._next_or_prev_synopsis(-1)

    def _next_or_prev_synopsis(self, step):
        controllers = self.frontend.retrieve_controllers('/poblesec/browser')
        history = controllers[0].history

        # go back to actions menu
        history.go_back()

        # go back to movies list
        history.go_back()

        def goto_synopsis(result, movie):
            controller = history.current.list_controller
            controller_path = '/poblesec/database/%s/synopsis' % self.video_type
            dfr = history.append_controller(controller_path,
                                            _('View Synopsis'),
                                            file_path=movie.file_path)
            return dfr

        controller = history.current.list_controller
        # yes, there are actions mixed with models
        items = [ m for m in controller.model if not isinstance(m, Action)]
        item = [ m for m in items if m.file_path == self.file_path ][0]
        index = items.index(item)
        index += step

        if index in range(len(items)):
            new_item = items[index]
            controller_path = '/poblesec/database/%s/track' % self.video_type
            try:
                label = new_item.title
            except AttributeError:
                label = new_item.name
            dfr = history.append_controller(controller_path, label,
                                            file_path=new_item.file_path)
            dfr.addCallback(goto_synopsis, new_item)
            return dfr

    def handle_input(self, manager, input_event):
        actions_list = self._synopsis_widget.actions_list
        if input_event.value == EventValue.KEY_GO_LEFT:
            if actions_list.selected_item_index > 0:
                actions_list.selected_item_index -= 1
                return True
        elif input_event.value == EventValue.KEY_GO_RIGHT:
            if actions_list.selected_item_index < len(actions_list):
                actions_list.selected_item_index += 1
                return True
        elif input_event.value == EventValue.KEY_GO_UP:
            if actions_list.focus:
                # go to top
                return False
        elif input_event.value == EventValue.KEY_OK:
            action = actions_list.model[actions_list.selected_item_index]
            actions_list.emit('item-clicked', action)
            return True

        return super(SynopsisController, self).handle_input(manager,
                                                            input_event)


class MovieSynopsisController(SynopsisController):
    synopsis_widget_class = MovieSynopsisWidget
    video_type = 'movie'

    def set_frontend(self, frontend):
        super(MovieSynopsisController, self).set_frontend(frontend)

        def got_results(results):
            dfr = results.one()
            dfr.addCallback(got_movie)
            return dfr

        def got_movie(movie):
            self._synopsis_widget.title = movie.title
            self._synopsis_widget.overview = movie.short_overview or \
                                             _("No synopsis available")
            cover_uri = movie.cover_uri
            if not cover_uri:
                theme = self.frontend.get_theme()
                icon_path = theme.get_resource('elisa.plugins.poblesec.file_video')
                cover_uri = 'file://%s' % icon_path
            dfr = _get_thumbnail(movie, MediaUri(cover_uri))
            dfr.addCallback(got_thumbnail)
            return dfr

        def got_thumbnail(poster_path):
            self._synopsis_widget.poster_path = poster_path
            self.widget.add(self._synopsis_widget)
            self._synopsis_widget.visible = True

        dfr = common.application.store.find(Movie, file_path=self.file_path)
        dfr.addCallback(got_results)
        return dfr

# All Movies
class AllMoviesVerticalWithPreview(AllMoviesController, MenuItemPreviewListController):
    fastscroller = True
    view_mode = MovieViewMode

    def item_to_label(self, item):
        if isinstance(item, Action):
            return '#'
        return item.title

class AllMoviesCoverflow(AllMoviesController, ImageWithReflectionCoverflowController):
    view_mode = MovieViewMode

class AllMoviesGrid(AllMoviesController, GridItemGridController):
    view_mode = MovieViewMode

class AllMoviesListSwitcherController(ListSwitcherController):
    modes = [AllMoviesVerticalWithPreview,
             AllMoviesCoverflow,
             AllMoviesGrid]
    default_mode = AllMoviesVerticalWithPreview


# All Videos
class AllVideosVerticalWithPreview(AllVideosController, MenuItemPreviewListController):
    fastscroller = True
    view_mode = VideoViewMode

    def item_to_label(self, item):
        if isinstance(item, Action):
            return '#'
        return item.name

class AllVideosCoverflow(AllVideosController, ImageWithReflectionCoverflowController):
    view_mode = VideoViewMode

class AllVideosGrid(AllVideosController, GridItemGridController):
    view_mode = VideoViewMode

class AllVideosListSwitcherController(ListSwitcherController):
    modes = [AllVideosVerticalWithPreview,
             AllVideosCoverflow,
             AllVideosGrid]
    default_mode = AllVideosVerticalWithPreview

# Unclassified Videos
class UnclassifiedVideosVerticalWithPreview(UnclassifiedVideosController,
                                            MenuItemPreviewListController):
    fastscroller = True
    view_mode = VideoViewMode

    def item_to_label(self, item):
        if isinstance(item, Action):
            return '#'
        return item.name

class UnclassifiedVideosCoverflow(UnclassifiedVideosController,
                                  ImageWithReflectionCoverflowController):
    view_mode = VideoViewMode

class UnclassifiedVideosGrid(UnclassifiedVideosController,
                             GridItemGridController):
    view_mode = VideoViewMode

class UnclassifiedVideosListSwitcherController(ListSwitcherController):
    modes = [UnclassifiedVideosVerticalWithPreview,
             UnclassifiedVideosCoverflow,
             UnclassifiedVideosGrid]
    default_mode = UnclassifiedVideosVerticalWithPreview


# Videos by month
class VideosMonthVerticalWithPreview(VideosMonthController, MenuItemPreviewListController):
    view_mode = VideoViewMode

    def item_to_label(self, item):
        if isinstance(item, Action):
            return '#'
        return item.name

class VideosMonthCoverflow(VideosMonthController, ImageWithReflectionCoverflowController):
    view_mode = VideoViewMode

class VideosMonthGrid(VideosMonthController, GridItemGridController):
    view_mode = VideoViewMode

class VideosMonthListSwitcherController(ListSwitcherController):
    modes = [VideosMonthVerticalWithPreview,
             VideosMonthCoverflow,
             VideosMonthGrid]
    default_mode = VideosMonthVerticalWithPreview

# Video Times Overview
class VideoTimesVerticalWithPreview(VideosTimesController, MenuItemPreviewListController):
    view_mode = VideoTimesViewMode

class VideoTimesCoverflow(VideosTimesController, ImageWithReflectionCoverflowController):
    view_mode = VideoTimesViewMode

class VideoTimesGrid(VideosTimesController, GridItemGridController):
    view_mode = VideoTimesViewMode

class VideoTimesListSwitcherController(ListSwitcherController):
    modes = [VideoTimesVerticalWithPreview,
             VideoTimesCoverflow,
             VideoTimesGrid]
    default_mode = VideoTimesVerticalWithPreview

# Videos tracks
class VideoTrackVerticalWithPreview(VideoTrackController, DoubleLineMenuItemPreviewListController):
    view_mode = VideoTrackViewMode

class VideoTrackCoverflow(VideoTrackController, ImageWithReflectionCoverflowController):
    view_mode = VideoTrackViewMode

class VideoTrackGrid(VideoTrackController, GridItemGridController):
    view_mode = VideoTrackViewMode

class VideoTrackListSwitcherController(ListSwitcherController):
    modes = [VideoTrackVerticalWithPreview,
             VideoTrackCoverflow,
             VideoTrackGrid]
    default_mode = VideoTrackVerticalWithPreview

# Movies tracks
class MovieTrackVerticalWithPreview(MovieTrackController, DoubleLineMenuItemPreviewListController):
    view_mode = VideoTrackViewMode

class MovieTrackCoverflow(MovieTrackController, ImageWithReflectionCoverflowController):
    view_mode = VideoTrackViewMode

class MovieTrackGrid(MovieTrackController, GridItemGridController):
    view_mode = VideoTrackViewMode

class MovieTrackListSwitcherController(ListSwitcherController):
    modes = [MovieTrackVerticalWithPreview,
             MovieTrackCoverflow,
             MovieTrackGrid]
    default_mode = MovieTrackVerticalWithPreview

