# -*- coding: utf-8 -*-
# Elisa - Home multimedia server
# Copyright (C) 2006-2008 Fluendo Embedded S.L. (www.fluendo.com).
# All rights reserved.
#
# This file is available under one of two license agreements.
#
# This file is licensed under the GPL version 3.
# See "LICENSE.GPL" in the root of this distribution including a special
# exception to use Elisa with Fluendo's plugins.
#
# The GPL part of Elisa is also available under a commercial licensing
# agreement from Fluendo.
# See "LICENSE.Elisa" in the root directory of this distribution package
# for details on that license.
#
# Authors: Guillaume Emont <guillaume@fluendo.com>

import pgm, gst
from elisa.core import common
from elisa.core.utils import defer
from elisa.core.utils.i18n import install_translation
from elisa.core.components.model import Model
from elisa.core.media_uri import MediaUri
from elisa.core.input_event import *
from elisa.plugins.base.models.media import PlayableModel
from elisa.plugins.base.messages.device import NewDeviceDetected
from elisa.plugins.poblesec.base.hierarchy import HierarchyController
from elisa.plugins.poblesec.base.list import GenericListViewMode
from elisa.plugins.poblesec.base.preview_list import MenuItemPreviewListController
from elisa.plugins.poblesec.widgets.menu_item import MenuItemWidget
from elisa.plugins.poblesec.player_video import Player, PlayerController
from elisa.plugins.poblesec.link import DeviceLink
import platform

_ = install_translation('dvd')

def section_decorator(controller):
    """
    Adds a DVD entry to the video section.
    """
    uri = MediaUri("volumes://?filter=dvd")
    model, dfr = common.application.resource_manager.get(uri, None)

    def add_link(model):
        """
        Adds a link to the controller for the DVD described in C{model}.
        """
        link = DeviceLink(model.udi)
        uri = MediaUri({'scheme': 'dvd',
                        'host': model.device,
                        'path': ''})
        link.controller_args = { 'uri': uri }
        link.controller_path = '/poblesec/video/dvd'
        link.icon = "elisa.plugins.poblesec.dvd"
        link.label = _("DVD")

        controller.model.append(link)

    def do_list_hardware(model):
        devices = sorted(model.devices, key=lambda i: i.name.lower())

        for dev in devices:
            add_link(dev)

    dfr.addCallback(do_list_hardware)

    def on_device_added(message, sender):
        if message.model.protocol == 'dvd':
            add_link(message.model)

    bus = common.application.bus
    bus.register(on_device_added, NewDeviceDetected)

    def on_clean(controller):
        bus.unregister(on_device_added)
        return defer.succeed(None)

    controller.connect('clean', on_clean)

    return dfr

class DvdMenuModel(Model):
    """A simple model for entries in the DVD-specific menu."""
    ACTION_PLAY   = 1
    ACTION_MENU   = 2
    ACTION_RESUME = 3
    ACTION_EJECT  = 4

    def __init__(self, name, action=None):
        super(DvdMenuModel, self).__init__()
        self.name = name
        self.action = action

class DvdHierarchyController(HierarchyController):
    """
    Handles the DVD-specific menu (logic part).

    @ivar uri: The uri of the DVD controlled by this instance
    @type uri: C{elisa.core.media_uri.MediaUri}
    """

    log_category = "dvd_menu"

    start_entry = DvdMenuModel(_("Play DVD"), DvdMenuModel.ACTION_PLAY)
    resume_entry = DvdMenuModel(_("Resume playing"), DvdMenuModel.ACTION_RESUME)
    menu_entry = DvdMenuModel(_("Menu"), DvdMenuModel.ACTION_MENU)
    eject_entry = DvdMenuModel(_("Eject DVD"), DvdMenuModel.ACTION_EJECT)

    def initialize(self, uri):
        dfr = super(DvdHierarchyController, self).initialize()

        self.uri = uri
        self._playing_menu = False

        return dfr

    def set_frontend(self, frontend):
        super(DvdHierarchyController, self).set_frontend(frontend) 

        if not self._is_started():
            self.model.append(self.start_entry)
        else:
            self._set_playing_menu()

    def node_clicked(self, widget, item):
        if item.action == item.ACTION_PLAY:
            self._do_play()
        elif item.action == item.ACTION_MENU:
            self._do_menu()
        elif item.action == item.ACTION_RESUME:
            self._do_resume()
        elif item.action == item.ACTION_EJECT:
            self._do_eject()

    def _is_started(self):
        controller = self._get_player_controller()
        player = controller.player
        played_uri = player.pipeline.get_property('uri')
        return played_uri == str(self.uri) \
                           and player.status != player.STOPPED

    def _get_player_controller(self):
        controllers = self.frontend.retrieve_controllers('/poblesec/dvd_player')
        return controllers[0]

    def _set_playing_menu(self):
        if not self._playing_menu:
            self._playing_menu = True
            self.model[:] = []
            self.model += [self.resume_entry, self.menu_entry]

    def _do_play(self):
        player_controller = self._get_player_controller()

        playable_model = PlayableModel()
        playable_model.uri = self.uri
        self._show_player()
        player_controller.player.play_model(playable_model)

        self._set_playing_menu()

    def _do_menu(self):
        controller = self._get_player_controller()
        if controller.in_menu:
            self._show_player()
        else:
            controller.send_key(controller.MENU_KEY)
            self._show_player()
        controller.player.play()

    def _do_resume(self):
        controller = self._get_player_controller()
        if controller.in_menu:
            controller.send_key(controller.MENU_KEY)
            self._show_player()
        else:
            self._show_player()
        controller.player.play()

    def _do_eject(self):
        self.warning("Eject not implemented yet")

    def _show_player(self):
        controllers = self.frontend.retrieve_controllers('/poblesec')
        main = controllers[0]
        main.show_dvd_player()

class DvdMenuViewMode(GenericListViewMode):

    def get_label(self, item):
        return defer.succeed(item.name)

    def get_sublabel(self, item):
        return defer.succeed(None)

    def get_default_image(self, item):

        if item.action in (DvdMenuModel.ACTION_PLAY,
                           DvdMenuModel.ACTION_RESUME):
            return "elisa.plugins.poblesec.dvd_play"
        elif item.action == DvdMenuModel.ACTION_MENU:
            return "elisa.plugins.poblesec.dvd_menu"
        else:
            raise AssertionError, "No icon defined for action %d." % item.action

    def get_image(self, item, theme):
        return None
    
    def get_preview_image(self, item, theme):
        return None

class DvdController(DvdHierarchyController, MenuItemPreviewListController):
    view_mode = DvdMenuViewMode

class DvdPlayer(Player):

    # We need this hack because playbin needs a dvd:// uri to play the stuff
    # correctly.
    def play_model(self, model):
        if model.uri.scheme == 'file':
            model.uri.scheme = 'dvd'
            # Broken gstreamer elements expect 'dvd://X:/foo/bar' and not
            # 'dvd:///X:/foo/bar'
            if model.uri.path and not model.uri.host:
                model.uri.host = model.uri.path
                model.uri.path=''
                if platform.system() == 'Windows' \
                        and len(model.uri.host) >=3 \
                        and model.uri.host[0] == '/' \
                        and model.uri.host[2] == ':':
                            model.uri.host = model.uri.host.lstrip('/')

        super(DvdPlayer, self).play_model(model)
        

class DvdPlayerController(PlayerController):
    """Specific controller handling a player that behaves for DVDs."""

    PlayerClass = DvdPlayer

    structure_string = \
        "application/x-gst-navigation,event=(string)key-press,key=(string)%s"

    MENU_KEY = 'm'

    log_category = "dvd_player"

    def __init__(self):
        super(DvdPlayerController, self).__init__()

        self.sink_pad = self.player.pgm_sink.get_pad("sink")
        self.sink_pad.add_event_probe(self._sink_pad_event_cb)

        self.in_menu = False

    def initialize(self):
        dfr = super(DvdPlayerController, self).initialize()
        self.player_osd.time_before_hiding = 3
        return dfr

    def _enter_menu(self):
        self.debug("enter menu")
        self.in_menu = True
        self.player.pgm_sink.set_property("events",
            pgm.IMAGE_SINK_MOTION | pgm.IMAGE_SINK_RELEASED)
        self.player_osd.visible = False

    def _leave_menu(self):
        self.debug("leave menu")
        self.in_menu = False
        self.player.pgm_sink.set_property("events", 0)
        self.player_osd.visible = True
        
    # FIXME: very hackish way of knowing^Wguessing whether we're in a menu.
    # Does not allow differentiation between "main" menu and submenus.
    def _sink_pad_event_cb(self, pad, event):
        structure = event.get_structure()
        if structure and structure.has_name('application/x-gst-dvd'):
            if structure['event'] == 'dvd-spu-highlight' and not self.in_menu:
                self._enter_menu()

            elif structure['event'] == 'dvd-spu-reset-highlight' \
                                                and self.in_menu:
                self._leave_menu()
        return True
                
    def send_key(self, key_name):
        structure = gst.structure_from_string(self.structure_string % key_name)
        self.sink_pad.push_event(gst.event_new_navigation(structure))

    def handle_input(self, manager, input_event):
        if not self.in_menu:
            return super(DvdPlayerController, self). \
                                handle_input(manager, input_event)

        # we're in a menu
        key_translation = {
            EventValue.KEY_OK: "Return",
            EventValue.KEY_SPACE: "Return",
            EventValue.KEY_GO_UP: "Up",
            EventValue.KEY_GO_DOWN: "Down",
            EventValue.KEY_GO_LEFT: "Left",
            EventValue.KEY_GO_RIGHT: "Right"}

        if key_translation.has_key(input_event.value):
            self.send_key(key_translation[input_event.value])
            return True

        return False

    # This is a hack in order not to show the OSD when we go in playing mode,
    # because else that happens in transition between menus, which is ugly.
    def _player_status_cb(self, player, status):
        # player_osd.visible determines whether the osd can be shown.
        # player_osd.show() and player_osd.hide() show and hide it with an
        # animation. PlayerController does a player_osd.show() that we don't
        # want when status == PLAYING.
        if status == player.PLAYING and self.player_osd.visible:
            self.player_osd.visible = False
            ret = super(DvdPlayerController, self)._player_status_cb(player,
                                                                     status)
            self.player_osd.hide()
            self.player_osd.visible = True
            return ret

        return super(DvdPlayerController, self)._player_status_cb(player,
                                                                  status)

