# -*- coding: utf-8 -*-
# Elisa - Home multimedia server
# Copyright (C) 2006-2008 Fluendo Embedded S.L. (www.fluendo.com).
# All rights reserved.
#
# This file is available under one of two license agreements.
#
# This file is licensed under the GPL version 3.
# See "LICENSE.GPL" in the root of this distribution including a special
# exception to use Elisa with Fluendo's plugins.
#
# The GPL part of Elisa is also available under a commercial licensing
# agreement from Fluendo.
# See "LICENSE.Elisa" in the root directory of this distribution package
# for details on that license.

import sys
import gobject
import weakref

from twisted.trial.unittest import TestCase
from pkg_resources import resource_filename

from elisa.plugins.pigment.widgets.widget import Widget
from elisa.plugins.pigment.widgets.style import Style
from elisa.plugins.pigment.widgets.theme import Theme
from elisa.plugins.pigment.widgets.const import *
from elisa.plugins.pigment.tests.graph.test_group import GenericTestGroup

import pgm

class TestWidget(TestCase):

    def assertIsProxy(self, obj, proxy):
        return self.assertTrue(proxy in weakref.getweakrefs(obj))

    def test_simple(self):
        widget = Widget()

    def test_hierarchy(self):
        parent = Widget()
        parent.name = 'parent'
        children = []
        grandchildren = []

        for x in range(3):
            child = Widget()
            child.name = 'child_' + str(x)
            parent.add(child)
            children.append(child)

            for x in range(2):
                grandchild = Widget()
                grandchild.name = 'grandchild_' + str(x)
                child.add(grandchild)
                grandchildren.append(grandchild)

                self.assertIsProxy(child, grandchild.get_parent())
                self.assertIsProxy(parent, grandchild.get_root())
                self.assertEquals(len(grandchild.get_children()), 0)
                self.assertEquals(len(grandchild.get_descendants()), 0)

            self.assertIsProxy(parent, child.get_parent())
            self.assertIsProxy(parent, child.get_root())
            self.assertEquals(len(child.get_descendants()), 2)
            self.assertEquals(len(child.get_descendants()),
                              len(child.get_children()))

        self.assertEquals(len(parent.get_children()), 3)
        self.assertEquals(len(parent.get_descendants()), 3 + (3 * 2))

        # focus related tests

        # dictionary holding signal connection ids
        # key:   widget
        # value: id of the gobject signal connection
        signal_ids = {}

        def focus_changed(widget, focus):
            # make sure the signal sends the right focus value
            self.assertEquals(widget.focus, focus)

        for item in parent.get_descendants():
            self.assertEquals(item.focus, False)
            id = item.connect("focus", focus_changed)
            signal_ids[item] = id

        for item in parent.get_descendants():
            item.focus = True
            self.assertEquals(item, parent.focus_child)
            for other in parent.get_descendants():
                if other != item:
                    self.assertEquals(other.focus, False)
                else:
                    self.assertEquals(other.focus, True)

        # disconnect from 'focus' signal of all widgets
        for item, id in signal_ids.iteritems():
            item.disconnect(id)

    def test_state(self):
        widget = Widget()

        self.assertEquals(widget.state, STATE_NORMAL)

        def on_state_changed(widget, state):
            self.assertEquals(state, STATE_ACTIVE)

        widget.connect('state-changed', on_state_changed)
        widget.state = STATE_ACTIVE

    def test_style(self):
        widget = Widget()

        self.assertEquals(widget.style, widget._styles[STATE_NORMAL])

        new_style = Style()

        def on_style_set(widget, style):
            self.assertEquals(style, new_style)

        handler_id = widget.connect('style-set', on_style_set)
        widget.style = new_style
        widget.disconnect(handler_id)

        def on_state_changed_after(widget, state):
            self.assertEquals(widget.style, widget._styles[state])

        handler_id = widget.connect_after('state_changed', on_state_changed_after)
        widget.state = STATE_ACTIVE
        widget.disconnect(handler_id)

    def test_parse_style_key(self):
        widget = Widget()
        widget.attribute = None
        widget.subwidget = Widget()
        widget.subwidget.attribute = None

        key = 'attribute'
        subwidget, attribute, logstr = widget._parse_style_key(key, '')
        self.failUnlessIdentical(subwidget, widget)
        self.failUnlessEqual(attribute, 'attribute')

        key = 'subwidget-attribute'
        subwidget, attribute, logstr = widget._parse_style_key(key, '')
        self.failUnlessIdentical(subwidget, widget.subwidget)
        self.failUnlessEqual(attribute, 'attribute')

        wrong_keys = ('attribute_not_found', 'subwidget_not-found',
                      'subwidget-attribute_not_found')
        for key in wrong_keys:
            self.failUnlessRaises(AttributeError,
                                  widget._parse_style_key, key, '')

    def test_parse_style_value(self):
        widget = Widget()

        value = widget._parse_style_value(1.0, None, '', '')
        self.failUnlessEqual(value, 1.0)

        value = widget._parse_style_value((0, 0, 0, 255), None, '', '')
        self.failUnlessEqual(value, (0, 0, 0, 255))

        value = widget._parse_style_value('something', None, '', '')
        self.failUnlessEqual(value, 'something')

        value = widget._parse_style_value('pgm.center', pgm.Image(), 'alignment', '')
        self.failUnlessEqual(value, pgm.IMAGE_CENTER)

        value = widget._parse_style_value('pgm.right', pgm.Text(), 'alignment', '')
        self.failUnlessEqual(value, pgm.TEXT_ALIGN_RIGHT)

        value = widget._parse_style_value('pgm.bold', pgm.Text(), 'weight', '')
        self.failUnlessEqual(value, pgm.TEXT_WEIGHT_BOLD)

        self.failUnlessRaises(AttributeError, widget._parse_style_value,
                              'pgm.nada', pgm.Text(), 'weight', '')


class GenericTestWidget(GenericTestGroup):
    """
    These are tests that all classes inheriting from Widget should pass.
    You should reimplement this class for anything that inherits from Widget.
    """
    def setUp(self):
        # mock the start_monitoring 
        self.monitoring_mock = Theme.start_monitoring
        Theme.start_monitoring = lambda x: x

    def tearDown(self):
        Theme.start_monitoring = self.monitoring_mock

class GenericTestWidgetImpl(GenericTestWidget, TestCase):
    tested_class = Widget

    def tearDown(self):
        # if the theme was initialized, we have to stop the monitoring or we
        # receive a unclean reactor
        Theme.get_default().stop_monitoring()

