# -*- coding: utf-8 -*-
# Elisa - Home multimedia server
# Copyright (C) 2006-2008 Fluendo Embedded S.L. (www.fluendo.com).
# All rights reserved.
#
# This file is available under one of two license agreements.
#
# This file is licensed under the GPL version 3.
# See "LICENSE.GPL" in the root of this distribution including a special
# exception to use Elisa with Fluendo's plugins.
#
# The GPL part of Elisa is also available under a commercial licensing
# agreement from Fluendo.
# See "LICENSE.Elisa" in the root directory of this distribution package
# for details on that license.
#
# Author: Olivier Tilloy <olivier@fluendo.com>

"""
Plugin management controllers and decorators.
"""

from elisa.core.utils import defer
from elisa.core.utils.sorting import async_ordered_placement
from elisa.core.utils.i18n import install_translation
from elisa.core import common
from elisa.core.plugin_registry import PluginStatusMessage
from elisa.core.input_event import EventValue

from elisa.plugins.base.models.plugin import PluginModel

from elisa.plugins.pigment.widgets.widget import Widget
from elisa.plugins.pigment.widgets.theme import Theme
from elisa.plugins.pigment.pigment_controller import PigmentController
from elisa.plugins.pigment.graph.text import Text
from elisa.plugins.pigment.graph.image import Image

from elisa.plugins.poblesec.link import Link
from elisa.plugins.poblesec.section import SectionMenuController
from elisa.plugins.poblesec.base.hierarchy import HierarchyController
from elisa.plugins.poblesec.base.list import GenericListViewMode
from elisa.plugins.poblesec.base.preview_list import \
    MenuItemPreviewListController, DoubleLineMenuItemPreviewListController
from elisa.plugins.poblesec.widgets.button import TextButton
from elisa.plugins.poblesec.widgets.sliced_image import SlicedImageHorizontal
from elisa.plugins.poblesec.widgets.loading_animation import LoadingAnimation

from elisa.extern import enum

from twisted.internet import task
from twisted.web import error as web_error

import pgm
from pgm.timing import implicit

from distutils.version import LooseVersion
import pkg_resources

import os.path, platform


_ = install_translation('poblesec')


def plugins_decorator(controller):
    link = Link()
    link.controller_path = '/poblesec/plugins_menu'
    link.label = _('Plugins')
    link.icon = 'elisa.plugins.poblesec.plugins_section'
    controller.model.append(link)
    return defer.succeed(None)


class PluginSectionController(SectionMenuController):
    pass


def find_new_plugins_decorator(controller):
    link = Link()
    link.controller_path = '/poblesec/plugins/find'
    link.label = _('Find New Plugins')
    link.icon = 'elisa.plugins.poblesec.find_new_plugins'
    controller.model.append(link)
    return defer.succeed(None)


def plugin_library_decorator(controller):
    link = Link()
    link.controller_path = '/poblesec/plugins/library'
    link.label = _('Plugin Library')
    link.icon = 'elisa.plugins.poblesec.plugin_library'
    controller.model.append(link)
    return defer.succeed(None)


def plugin_updates_decorator(controller):
    link = Link()
    link.controller_path = '/poblesec/plugins/updates'
    link.label = _('Updates')
    link.icon = 'elisa.plugins.poblesec.update'
    controller.model.append(link)
    return defer.succeed(None)


class PluginListViewMode(GenericListViewMode):

    """
    Implementation of the common view modes API.
    """

    plugin_icon = 'elisa.plugins.poblesec.plugins_section'

    def get_label(self, item):
        return defer.succeed(item.title)

    def get_sublabel(self, item):
        author = item.author_name
        if not author:
            author = _('Unknown Author')
        return defer.succeed(author)

    def get_default_image(self, item):
        return self.plugin_icon

    def get_image(self, item, theme):
        return None

    def get_preview_image(self, item, theme):
        try:
            icon_uri = item.icons[0].references[0]
        except IndexError:
            return None
        else:
            cache_file, dfr = item.cache_image(icon_uri)
            if os.path.exists(cache_file):
                return cache_file
            else:
                return None


class FindNewPluginsController(HierarchyController,
                               DoubleLineMenuItemPreviewListController):

    """
    A list controller that displays a list of plugins.

    The list of plugins is downloaded from a set of plugin repositories.
    """

    view_mode = PluginListViewMode
    empty_label = _('There are no new plugins available.')
    empty_icon = view_mode.plugin_icon

    def initialize(self):
        dfr = super(FindNewPluginsController, self).initialize()
        self._plugin_dicts = {}
        self._update_success = True
        dfr.addCallback(self._populate_collection)
        dfr.addCallback(self._connect_bus)
        return dfr

    def clean(self):
        if self._update_success:
            bus = common.application.bus
            bus.unregister(self._plugin_status_changed_cb)
        return super(FindNewPluginsController, self).clean()

    def _populate_collection(self, result):
        def get_plugin_list(cache_file):
            plugins = \
                plugin_registry.get_downloadable_plugins(reload_cache=True)
            installed = list(plugin_registry.get_plugin_names())

            def iterate_plugins(plugins):
                def _cmp(plugin1, plugin2):
                    return cmp(plugin1.title.lower(), plugin2.title.lower())

                is_power_user = common.application.is_power_user()

                for plugin in plugins:
                    name = plugin['name']
                    if name in installed:
                        continue
                    if plugin['quality'] == 'unstable' and not is_power_user:
                        continue
                    model = PluginModel.from_dict(plugin)

                    # Filter out platform-specific plugins if the target
                    # platform does not match.
                    if not model.runs_on_current_platform():
                        continue

                    self._plugin_dicts[name] = plugin
                    yield async_ordered_placement(self.model, model, _cmp)

            dfr = task.coiterate(iterate_plugins(plugins))
            return dfr

        def failed_update(failure):
            failure.trap(web_error.Error)
            self._update_success = False

        plugin_registry = common.application.plugin_registry
        # Query the plugin registry for a list of new plugins
        dfr = plugin_registry.update_cache()
        dfr.addCallback(get_plugin_list)
        dfr.addErrback(failed_update)
        dfr.addCallback(lambda x: result)
        return dfr

    def _connect_bus(self, result):
        if self._update_success:
            bus = common.application.bus
            bus.register(self._plugin_status_changed_cb, PluginStatusMessage)
        return result

    def _plugin_status_changed_cb(self, message, sender):
        # Notification that the status of a plugin has changed.
        # It may mean that a new plugin has been installed, in which case we
        # want to remove it from the list.
        for plugin in self.model:
            if plugin.name == message.plugin_name:
                self.model.remove(plugin)
                if len(self.model) == 0:
                    self.empty_label = _('All the available plugins have ' \
                                         'been successfully installed.')
                    self.toggle_empty_alert(True)
                break

    def set_frontend(self, frontend):
        super(FindNewPluginsController, self).set_frontend(frontend)
        if not self._update_success:
            self.empty_icon = 'elisa.plugins.poblesec.warning'
            self.empty_label = _('Cannot connect to the plugin repository.')
            self.toggle_empty_alert(True)

    def node_clicked(self, widget, item):
        browser = self.frontend.retrieve_controllers('/poblesec/browser')[0]
        path = '/poblesec/plugins/information'
        plugin_dict = self._plugin_dicts[item.name]
        dfr = browser.history.append_controller(path, item.title,
                                                plugin=item, installed=False,
                                                plugin_dict=plugin_dict)
        return dfr


class PluginLibraryViewMode(PluginListViewMode):

    enabled_resource = 'elisa.plugins.poblesec.enabled'
    disabled_resource = 'elisa.plugins.poblesec.disabled'

    def get_default_image(self, item):
        if item.enabled:
            return self.enabled_resource
        else:
            return self.disabled_resource

    def get_image(self, item, theme):
        return None

    def get_preview_image(self, item, theme):
        try:
            image_path = item.icons[0].references[0].path
        except IndexError:
            image_path = theme.get_resource(self.plugin_icon)
        return image_path


class PluginLibraryController(HierarchyController,
                                 DoubleLineMenuItemPreviewListController):

    """
    A list controller that displays a list of plugins.

    The list of plugins is created querying the application's plugin registry.

    @cvar blacklist: a blacklist of plugins we do not want to show in the
                     collection, either because they are part of the
                     infrastructure, or because they do not provide any visible
                     service to the user.
    @type blacklist: C{list} of C{str}
    """

    view_mode = PluginLibraryViewMode

    blacklist = []
    _plugins = ['amazon', 'amp', 'avahi', 'base', 'bluetooth', 'coherence',
                'daap', 'dailymotion', 'database', 'discogs', 'dvd',
                'elisa-updater', 'favorites', 'filtered-shares', 'gnome',
                'gstreamer', 'hal', 'helper', 'http-client', 'httpd', 'ipod',
                'lastfm', 'lirc', 'osso', 'picasa', 'pigment', 'player',
                'poblesec', 'rhythmbox', 'rss', 'search', 'shelf', 'smbwin32',
                'testing', 'tutorial-amazon', 'winremote', 'winscreensaver',
                'wmd']
    blacklist.extend(map(lambda x: 'elisa-plugin-%s' % x, _plugins))

    def initialize(self):
        dfr = super(PluginLibraryController, self).initialize()
        dfr.addCallback(self._populate_collection)
        dfr.addCallback(self._connect_bus)
        return dfr

    def clean(self):
        bus = common.application.bus
        bus.unregister(self._plugin_status_changed_cb)
        return super(PluginLibraryController, self).clean()

    def _blacklisted(self, plugin_name):
        return not plugin_name.startswith('elisa-plugin-') \
            or plugin_name in self.blacklist

    def _populate_collection(self, result):
        # Asynchronous sorted population of the list of plugins.
        plugin_registry = common.application.plugin_registry
        plugins = plugin_registry.get_plugins()

        def iterate_plugins(plugins):
            def _cmp(plugin1, plugin2):
                return cmp(plugin1.title.lower(), plugin2.title.lower())

            for plugin_name, enabled in plugins:
                if self._blacklisted(plugin_name):
                    continue
                plugin = plugin_registry.get_plugin_by_name(plugin_name)
                plugin_registry.get_plugin_metadata(plugin)
                model = PluginModel.from_distribution(plugin)
                model.enabled = enabled
                yield async_ordered_placement(self.model, model, _cmp)

        dfr = task.coiterate(iterate_plugins(plugins))
        dfr.addCallback(lambda x: result)
        return dfr

    def _connect_bus(self, result):
        bus = common.application.bus
        bus.register(self._plugin_status_changed_cb, PluginStatusMessage)
        return result

    def _plugin_status_changed_cb(self, message, sender):
        # Notification that the status of a plugin has changed.
        for index, plugin in enumerate(self.model):
            if plugin.name == message.plugin_name:
                plugin.enabled = \
                    (message.action == PluginStatusMessage.ActionType.ENABLED)
                self.model[index] = plugin
                break

    def node_clicked(self, widget, item):
        browser = self.frontend.retrieve_controllers('/poblesec/browser')[0]
        path = '/poblesec/plugins/actions'
        dfr = browser.history.append_controller(path, item.title, plugin=item)
        return dfr


class PluginAction(Link):

    """
    A contextual action associated to a given plugin or list of plugins.

    @ivar action: a callback invoked to trigger the action
    @type action: C{callable}
    @ivar args:   a list of arguments to pass to the callback
    @type args:   C{list}
    """

    def __init__(self, action=None, args=[]):
        super(PluginAction, self).__init__()
        self.action = action
        if action is None:
            self.action = self._no_action
        self.args = args

    def _no_action(self, item, *args):
        raise NotImplementedError('No action.')


class PluginUpdatesViewMode(PluginListViewMode):

    def get_label(self, item):
        if isinstance(item, PluginAction):
            return defer.succeed(item.label)
        return defer.succeed(item.title)

    def get_sublabel(self, item):
        if isinstance(item, PluginAction):
            return defer.succeed(item.sublabel)
        sublabel = _('Select To Update To Version %s') % item.version
        return defer.succeed(sublabel)

    def get_default_image(self, item):
        if isinstance(item, PluginAction):
            return item.icon
        return super(PluginUpdatesViewMode, self).get_default_image(item)

    def get_image(self, item, theme):
        if isinstance(item, PluginAction):
            return None
        return super(PluginUpdatesViewMode, self).get_image(item, theme)

    def get_preview_image(self, item, theme):
        if isinstance(item, PluginAction):
            return None
        su = super(PluginUpdatesViewMode, self)
        return su.get_preview_image(item, theme)


class PluginUpdatesController(HierarchyController,
                              DoubleLineMenuItemPreviewListController):

    """
    A list controller that displays a list of plugins.

    The plugins listed are those for which a newer version is available for
    download from a set of plugin repositories.
    """

    view_mode = PluginUpdatesViewMode
    empty_label = _('There are no updates available.')
    empty_icon = view_mode.plugin_icon

    def initialize(self):
        dfr = super(PluginUpdatesController, self).initialize()
        self._plugin_dicts = {}
        self._update_success = True
        dfr.addCallback(self._populate_collection)
        return dfr

    def _populate_collection(self, result):
        def get_plugin_list(cache_file):
            plugins = \
                plugin_registry.get_downloadable_plugins(reload_cache=True)
            installed = list(plugin_registry.get_plugin_names())

            def iterate_plugins(plugins):
                def _cmp(plugin1, plugin2):
                    return cmp(plugin1.title.lower(), plugin2.title.lower())

                for plugin in plugins:
                    name = plugin['name']
                    if name not in installed:
                        continue
                    current = plugin_registry.get_plugin_by_name(name)
                    current_version = LooseVersion(current.version)
                    new_version = LooseVersion(plugin['version'])
                    if new_version <= current_version:
                        continue
                    self._plugin_dicts[name] = plugin
                    model = PluginModel.from_dict(plugin)
                    yield async_ordered_placement(self.model, model, _cmp)

            dfr = task.coiterate(iterate_plugins(plugins))
            return dfr

        def failed_update(failure):
            failure.trap(web_error.Error)
            self._update_success = False

        def add_update_all_action(plugins_result):
            if len(self.model) >= 2:
                update_all = PluginAction()
                update_all.label = _('Update All')
                update_all.sublabel = _('Apply All The Available Updates')
                update_all.icon = 'elisa.plugins.poblesec.update'
                self.model.insert(0, update_all)
            return plugins_result

        plugin_registry = common.application.plugin_registry
        # Query the plugin registry for a list of new plugins
        dfr = plugin_registry.update_cache()
        dfr.addCallback(get_plugin_list)
        dfr.addErrback(failed_update)
        dfr.addCallback(add_update_all_action)
        dfr.addCallback(lambda x: result)
        return dfr

    def set_frontend(self, frontend):
        super(PluginUpdatesController, self).set_frontend(frontend)
        if not self._update_success:
            self.empty_icon = 'elisa.plugins.poblesec.warning'
            self.empty_label = _('Cannot connect to the plugin repository.')
            self.toggle_empty_alert(True)

    def _show_restart_popup_cb(self, result):
        main = self.frontend.retrieve_controllers('/poblesec')[0]
        icon = 'elisa.plugins.poblesec.plugin'
        title = _('Need To Restart Elisa')
        text = _('For the update to take effect, please exit and restart' \
                 ' Elisa.')

        buttons = [(_('OK'), main.hide_popup)]
        import sys
        if platform.system() == 'Windows' and hasattr(sys, 'frozen'):
            # Running an installed Elisa on Windows: we are able to restart
            # Elisa automatically.
            buttons[0] = (_('Later'), main.hide_popup)
            from mswin32 import tools
            restart_button = (_('Restart'), lambda: tools.exit(1, True))
            buttons.insert(0, restart_button)

        main.enqueue_popup(icon, title, text, buttons)
        return result

    def _update_plugin(self, item, single_update=True):
        def remove_from_list(result, item):
            self.stop_loading_animation()
            self.model.remove(item)
            updates = [item for item in self.model \
                       if isinstance(item, PluginModel)]
            if len(updates) == 0:
                self.empty_label = _('All the available updates have ' \
                                     'been successfully installed.')
                self.toggle_empty_alert(True)
            return result

        plugin_registry = common.application.plugin_registry
        dfr = plugin_registry.update_plugin(self._plugin_dicts[item.name])
        dfr.addCallback(remove_from_list, item)
        if single_update:
            dfr.addCallback(self._show_restart_popup_cb)
        return dfr

    def _update_all_plugins(self):
        def iterate_updates(updates):
            for item in updates:
                dfr = self._update_plugin(item, single_update=False)
                yield dfr

        updates = [item for item in self.model \
                   if isinstance(item, PluginModel)]
        dfr = task.coiterate(iterate_updates(updates))
        dfr.addCallback(self._show_restart_popup_cb)
        return dfr

    def node_clicked(self, widget, item):
        if isinstance(item, PluginAction):
            return self._update_all_plugins()

        return self._update_plugin(item)


class PluginActionsViewMode(GenericListViewMode):

    """
    Implementation of the common view modes API.
    """

    def get_label(self, item):
        return defer.succeed(item.label)

    def get_default_image(self, item):
        return item.icon

    def get_image(self, item, theme):
        return None

    def get_preview_image(self, item, theme):
        return theme.get_resource('elisa.plugins.poblesec.plugin')


class PluginActionsController(HierarchyController,
                              MenuItemPreviewListController):

    """
    Display a list of actions for a given plugin.

    @ivar plugin: a plugin model
    @type plugin: L{PluginModel}
    """

    view_mode = PluginActionsViewMode

    enable_label = _('Enable Plugin')
    disable_label = _('Disable Plugin')
    enable_icon = 'elisa.plugins.poblesec.enabled'
    disable_icon = 'elisa.plugins.poblesec.disabled'

    def initialize(self, plugin):
        dfr = super(PluginActionsController, self).initialize()
        self.plugin = plugin
        self.switch_state_action = None
        self.use_action = None
        dfr.addCallback(self._connect_bus)
        return dfr

    def clean(self):
        bus = common.application.bus
        bus.unregister(self._plugin_status_changed_cb)
        return super(PluginActionsController, self).clean()

    def _connect_bus(self, result):
        bus = common.application.bus
        bus.register(self._plugin_status_changed_cb, PluginStatusMessage)
        return result

    def _plugin_status_changed_cb(self, message, sender):
        # Notification that the status of a plugin has changed.
        action = self.switch_state_action
        if action is None:
            return

        if message.plugin_name == self.plugin.name:
            self.plugin.enabled = \
                (message.action == PluginStatusMessage.ActionType.ENABLED)
            if self.plugin.enabled:
                action.label = self.disable_label
                if self.use_action is not None:
                    self.model.insert(0, self.use_action)
            else:
                action.label = self.enable_label
                if self.use_action is not None:
                    self.model.remove(self.use_action)
            self.model[self.model.index(action)] = action

    def switch_state(self, item, *args):
        # Enable/disable the plugin
        def change_state(result):
            self.plugin.enabled = not self.plugin.enabled
            if self.plugin.enabled:
                item.label = self.disable_label
                item.icon = self.disable_icon
            else:
                item.label = self.enable_label
                item.icon = self.enable_icon
            self.stop_loading_animation()

        plugin_registry = common.application.plugin_registry
        if self.plugin.enabled:
            dfr = plugin_registry.disable_plugin(self.plugin.name)
        else:
            dfr = plugin_registry.enable_plugin(self.plugin.name)
        dfr.addCallback(change_state)
        return dfr

    def node_clicked(self, widget, item):
        if isinstance(item, PluginAction):
            dfr = item.action(item, item.args)
            return dfr
        elif isinstance(item, Link):
            browser = self.frontend.retrieve_controllers('/poblesec/browser')[0]
            path = item.controller_path
            label = item.label
            args = item.controller_args
            dfr = browser.history.append_controller(path, label, **args)
            return dfr


def use_plugin_decorator(controller):
    # Does the plugin define a 'use me' hook?
    plugin_registry = common.application.plugin_registry
    plugin = plugin_registry.get_plugin_by_name(controller.plugin.name)
    try:
        # The 'use me' hook is a function that takes as a single parameter the
        # frontend and returns a deferred.
        hook = plugin.load_entry_point('elisa.core.plugin_registry', 'use')
    except ImportError:
        return defer.succeed(None)
    else:
        action = PluginAction(lambda item, args: hook(controller.frontend))
        action.label = _('Use Plugin')
        action.icon = 'elisa.plugins.poblesec.use_plugin'
        controller.use_action = action
        if controller.plugin.enabled:
            controller.model.append(action)
        return defer.succeed(None)


def plugin_information_decorator(controller):
    link = Link()
    link.controller_path = '/poblesec/plugins/information'
    link.label = _('Plugin Information')
    link.icon = 'elisa.plugins.poblesec.information'
    link.controller_args = {'plugin': controller.plugin}
    controller.model.append(link)
    return defer.succeed(None)


def enable_plugin_decorator(controller):
    action = PluginAction(action=controller.switch_state)
    if controller.plugin.enabled:
        action.label = controller.disable_label
        action.icon = controller.disable_icon
    else:
        action.label = controller.enable_label
        action.icon = controller.enable_icon
    controller.model.append(action)
    controller.switch_state_action = action
    return defer.succeed(None)


def delete_plugin_decorator(controller):
    action = PluginAction()
    action.label = _('Delete Plugin')
    action.icon = 'elisa.plugins.poblesec.plugin'
    controller.model.append(action)
    return defer.succeed(None)


class PluginInformationWidget(Widget):

    """
    A widget that contains several widgets to display information about a
    plugin: title, author, description, icon.
    """

    def __init__(self):
        super(PluginInformationWidget, self).__init__()

        self.title = Text()
        self.add(self.title)
        self.title.visible = True

        self.author = Text()
        self.add(self.author)
        self.author.visible = True

        self.version = Text()
        self.add(self.version)
        self.version.visible = True

        self.description = Text()
        self.add(self.description)
        self.description.visible = True

        self.image = Image()
        self.add(self.image)
        self.image.visible = True

        self.update_style_properties(self.style.get_items())

    def clean(self):
        self.image.clean()
        self.image = None
        self.description.clean()
        self.description = None
        self.version.clean()
        self.version = None
        self.author.clean()
        self.author = None
        self.title.clean()
        self.title = None
        return super(PluginInformationWidget, self).clean()


class PluginButton(TextButton):
    pass


class UsePluginWidget(Widget):

    """
    A widget that contains several widgets to display actions to perform on a
    plugin and their feedback (use plugin, download & install).
    """

    def __init__(self):
        super(UsePluginWidget, self).__init__()

        self.button = PluginButton()
        self.add(self.button)
        self.button.visible = True

        self._selector = SlicedImageHorizontal()
        self.add(self._selector)
        resource_root = 'elisa.plugins.poblesec.selector'
        theme = Theme.get_default()
        left_cap = theme.get_resource('%s_left_cap' % resource_root)
        right_cap = theme.get_resource('%s_right_cap' % resource_root)
        body = theme.get_resource('%s_body' % resource_root)
        self._selector.left_cap.set_from_file(left_cap)
        self._selector.right_cap.set_from_file(right_cap)
        self._selector.body.set_from_file(body)
        self._selector.visible = True

        self._animated_selector = implicit.AnimatedObject(self._selector)
        self._animated_selector.setup_next_animations(duration=200)

        self.loading_animation = LoadingAnimation()
        self.add(self.loading_animation)
        self.loading_animation.visible = False

        self.label = Text()
        self.add(self.label)
        self.label.visible = False

        self.update_style_properties(self.style.get_items())

    def clean(self):
        self.label.clean()
        self.label = None
        self.loading_animation.clean()
        self.loading_animation = None
        self._animated_selector = None
        self._selector.clean()
        self._selector = None
        self.button.clean()
        self.button = None
        return super(UsePluginWidget, self).clean()

    def do_focus(self, focus):
        if focus:
            self._animated_selector.opacity = 255
        else:
            self._animated_selector.opacity = 0

    def update_style_properties(self, props=None):
        super(UsePluginWidget, self).update_style_properties(props)

        if props is None:
            return

        for key, value in props.iteritems():
            try:
                subwidget, attribute = key.split('-')
            except ValueError:
                continue
            if subwidget == 'button':
                # Resizing and repositioning the button also resizes and
                # repositions the selector.
                try:
                    getattr(self._selector, attribute)
                except AttributeError:
                    # Attribute not found, skipping this property
                    continue
                setattr(self._selector, attribute, value)


class PluginInformationController(PigmentController):

    """
    Display detailed information about a given plugin.

    @ivar plugin:      a plugin model
    @type plugin:      L{PluginModel}
    @ivar installed:   whether the plugin is already installed
    @type installed:   C{bool}
    @ivar plugin_dict: a dictionary representing the plugin as understood by
                       the plugin repository
    @type plugin_dict: C{dict}
    """

    Actions = enum.Enum('INSTALL', 'USE', 'BACK')

    def initialize(self, plugin, installed=True, plugin_dict=None):
        dfr = super(PluginInformationController, self).initialize()
        self.plugin = plugin
        self.installed = installed
        self.plugin_dict = plugin_dict
        dfr.addCallback(self._create_widget)
        return dfr

    def _create_widget(self, result):
        self.information = PluginInformationWidget()
        self.widget.add(self.information)
        self.information.title.label = self.plugin.title
        self.information.author.label = self.plugin.author_name
        self.information.version.label = \
            _('version %s') % str(self.plugin.version)
        self.information.description.label = self.plugin.description

        try:
            img_uri = self.plugin.icons[0].references[0]
        except IndexError:
            theme = Theme.get_default()
            resource = 'elisa.plugins.poblesec.plugin'
            img_file = theme.get_resource(resource)
            self.information.image.set_from_file(img_file)
        else:
            image_path, dfr = self.plugin.cache_image(img_uri)
            dfr.addCallback(self.information.image.set_from_file)

        self.information.size = (1.0, 1.0)
        self.information.position = (0.0, 0.0, 0.0)
        self.information.visible = True

        if self.installed:
            self.actions = None
            return result

        # The plugin is not installed yet, offer the user to download it
        self.actions = UsePluginWidget()
        self.widget.add(self.actions)
        self.actions.size = (1.0, 1.0)
        self.actions.position = (0.0, 0.0, 0.0)
        self._set_action(self.Actions.INSTALL)
        self.actions.visible = True

        self._widget_focus_id = self.widget.connect('focus', self._on_focus)
        self._button_clicked_id = \
            self.actions.button.connect('clicked', self._button_clicked_cb)

        return result

    def clean(self):
        if self.actions is not None:
            self.actions.button.disconnect(self._button_clicked_id)
            self.widget.disconnect(self._widget_focus_id)
            self.actions.clean()
            self.actions = None
        self.information.clean()
        self.information = None
        return super(PluginInformationController, self).clean()

    def _on_focus(self, widget, focus):
        if focus:
            self.actions.focus = focus

    def _button_clicked_cb(self, button, *args):
        self.actions.focus = True
        self._do_action()
        return True

    def _set_action(self, action):
        self._action = action
        if action == self.Actions.INSTALL:
            self.actions.button.label = _('Download & Install')
        elif action == self.Actions.USE:
            self.actions.button.label = _('Use This Plugin')
        elif action == self.Actions.BACK:
            self.actions.button.label = _('Back')

    def _download_failed(self, failure):
        self.warning('Downloading %s %s failed.' % \
                     (self.plugin.name, self.plugin.version))
        return failure

    def _ready_to_use(self, result):
        self.actions.loading_animation.activate(False)
        self.actions.loading_animation.visible = False
        try:
            plugin_registry = common.application.plugin_registry
            plugin = plugin_registry.get_plugin_by_name(self.plugin.name)
            hook = plugin.load_entry_point('elisa.core.plugin_registry', 'use')
        except ImportError:
            self._set_action(self.Actions.BACK)
            return result
        else:
            self._set_action(self.Actions.USE)
            return result

    def _do_action(self):
        plugin_registry = common.application.plugin_registry
        if self._action == self.Actions.INSTALL:
            self.actions.loading_animation.activate(True)
            self.actions.loading_animation.visible = True
            dfr = plugin_registry.download_plugin(self.plugin_dict)
            dfr.addCallback(plugin_registry.install_plugin, self.plugin.name)
            dfr.addErrback(self._download_failed)
            dfr.addBoth(self._ready_to_use)
            return dfr
        elif self._action == self.Actions.USE:
            plugin = plugin_registry.get_plugin_by_name(self.plugin.name)
            # The 'use me' hook is a function that takes as a single parameter
            # the frontend and returns a deferred.
            hook = plugin.load_entry_point('elisa.core.plugin_registry', 'use')
            return hook(self.frontend)
        elif self._action == self.Actions.BACK:
            controller = '/poblesec/browser'
            browser = self.frontend.retrieve_controllers(controller)[0]
            browser.history.go_back()

    def handle_input(self, manager, input_event):
        if input_event.value in (EventValue.KEY_OK, EventValue.KEY_RETURN):
            self._do_action()
            return True
        su = super(PluginInformationController, self)
        return su.handle_input(manager, input_event)
