# -*- coding: utf-8 -*-
# Elisa - Home multimedia server
# Copyright (C) 2006-2008 Fluendo Embedded S.L. (www.fluendo.com).
# All rights reserved.
#
# This file is available under one of two license agreements.
#
# This file is licensed under the GPL version 3.
# See "LICENSE.GPL" in the root of this distribution including a special
# exception to use Elisa with Fluendo's plugins.
#
# The GPL part of Elisa is also available under a commercial licensing
# agreement from Fluendo.
# See "LICENSE.Elisa" in the root directory of this distribution package
# for details on that license.
#
# Author: Olivier Tilloy <olivier@fluendo.com>

from elisa.core.components.model import Model
from elisa.core.media_uri import MediaUri
from elisa.core.default_config import PICTURES_CACHE
from elisa.core.utils import defer
from elisa.core.plugin_registry import get_plugin_toplevel_directory

from elisa.plugins.base.models.image import ImageModel

from distutils.version import LooseVersion

from twisted.web.client import getPage

import pkg_resources

import os

try:
    from hashlib import md5
except ImportError:
    # hashlib is new in Python 2.5
    from md5 import md5


class PluginModel(Model):

    """
    A model that represents a plugin and its metadata as understood by Elisa.

    @note: this model may be enriched along the way with new metadata

    @ivar name:         the name of the plugin
    @type name:         C{unicode}
    @ivar version:      the version number of the plugin
    @type version:      L{distutils.version.LooseVersion}
    @ivar author_name:  the name of the author of the plugin
    @type author_name:  C{unicode}
    @ivar author_email: the e-mail address of the author of the plugin
    @type author_email: C{unicode}
    @ivar platforms:    a list of compatible platforms (typical values are:
                        C{linux}, C{win32}, C{macosx})
    @type platforms:    C{list} of C{unicode}
    @ivar license:      the license of the plugin
    @type license:      C{unicode}
    @ivar website:      the home page of the plugin
    @type website:      C{unicode}
    @ivar category:     a unique category the plugin belongs to
    @type category:     C{unicode}
    @ivar tags:         a list of tags that define the plugin
    @type tags:         C{list} of C{unicode}
    @ivar egg_name:     the egg name
    @type egg_name:     C{unicode}
    @ivar uri:          the URI where the egg can be downloaded from
    @type uri:          C{unicode}
    @ivar title:        a fancy (short) name for the plugin
    @type title:        C{unicode}
    @ivar description:  a detailed description of what the plugin does
    @type description:  C{unicode}
    @ivar icons:        a list of icons for the plugin
    @type icons:        C{list} of
                        L{elisa.plugins.base.models.image.ImageModel}
    @ivar screenshots:  a list of screenshot images for the plugin
    @type screenshots:  C{list} of
                        L{elisa.plugins.base.models.image.ImageModel}

    @ivar enabled:      whether the plugin is currently enabled (this attribute
                        is not part of the plugin metadata)
    @type enabled:      C{bool}
    """

    def __init__(self):
        super(PluginModel, self).__init__()
        self.name = ''
        self.version = LooseVersion()
        self.author_name = ''
        self.author_email = ''
        self.platforms = []
        self.license = ''
        self.website = ''
        self.category = ''
        self.tags = []
        self.egg_name = ''
        self.uri = ''
        self.title = ''
        self.description = ''
        self.icons = []
        self.screenshots = []
        self.enabled = False

    @classmethod
    def from_dict(cls, dictionary):
        """
        Class method that instantiates and returns a plugin model from a
        dictionary as provided by the plugin repository.

        @param dictionary: a dictionary as provided by the plugin repository
        @type dictionary:  C{dict}

        @return:           a plugin model populated accordingly
        @rtype:            L{PluginModel}
        """
        plugin = cls()
        for key, value in dictionary.iteritems():
            if key in ('name', 'author_name', 'author_email', 'license',
                       'website', 'category', 'tags', 'egg_name', 'uri',
                       'description'):
                setattr(plugin, key, value)
            elif key == 'version':
                plugin.version.parse(value)
            elif key == 'platforms':
                # Platforms is a comma-separated list of platforms serialized
                # as a string.
                platforms = value.split(',')
                for platform in platforms:
                    platform = platform.strip()
                    if platform != 'UNKNOWN':
                        plugin.platforms.append(platform)
            elif key == 'summary':
                plugin.title = value
            elif key == 'icons':
                icons = eval(value)
                for icon in icons:
                    model = ImageModel()
                    model.references.append(MediaUri(icon['uri']))
                    plugin.icons.append(model)
            elif key == 'screenshots':
                screenshots = eval(value)
                for screenshot in screenshots:
                    model = ImageModel()
                    model.references.append(MediaUri(screenshot['uri']))
                    plugin.screenshots.append(model)
        return plugin

    def _extract_resources(self, distribution, folder):
        toplevel_directory = get_plugin_toplevel_directory(distribution)
        sub_directory = '%s/%s' % (toplevel_directory, folder)
        if not distribution.resource_isdir(sub_directory):
            return []
        resources = distribution.resource_listdir(sub_directory)
        parsed_resources = []
        requirement = \
            pkg_resources.Requirement.parse(distribution.project_name)
        for resource in resources:
            if resource.startswith('__'):
                continue
            resource_path = '%s/%s' % (sub_directory, resource)
            path = pkg_resources.resource_filename(requirement, resource_path)
            icon_uri = {'scheme': 'file', 'path': path}
            model = ImageModel()
            model.references.append(MediaUri(icon_uri))
            parsed_resources.append(model)
        return parsed_resources

    @classmethod
    def from_distribution(cls, distribution):
        """
        Class method that instantiates and returns a plugin model from a
        distribution as provided by L{pkg_resources}.

        @param distribution: a distribution as provided by L{pkg_resources}
        @type distribution:  L{pkg_resources.Distribution}

        @return:             a plugin model populated accordingly
        @rtype:              L{PluginModel}
        """
        plugin = cls()
        plugin.name = distribution.project_name
        plugin.version.parse(distribution.version)
        plugin.author_name = distribution.author
        plugin.license = distribution.license
        plugin.egg_name = distribution.egg_name()
        plugin.title = distribution.summary
        plugin.description = distribution.description
        plugin.icons = plugin._extract_resources(distribution, 'icons')
        plugin.screenshots = plugin._extract_resources(distribution,
                                                       'screenshots')
        return plugin

    def runs_on_current_platform(self):
        """
        Test whether the plugin is compatible with the current platform.

        @return: C{True} if compatible with the current platform, C{False}
                 otherwise
        @rtype:  C{bool}
        """
        if len(self.platforms) == 0:
            return True

        current_platform = pkg_resources.get_platform()
        for platform in self.platforms:
            if current_platform.startswith(platform):
                return True

        return False

    def cache_image(self, uri):
        """
        Return the path to the image cached on disk corresponding to the given
        URI.

        If the uri points to a local file, just return its path without caching
        it. Otherwise, if the image is not cached yet, download it and cache
        it.

        @param uri: the URI to the image
        @type uri:  L{elisa.core.media_uri.MediaUri}

        @return:    the path to the cache file and a deferred triggered when
                    the image is cached
        @rtype:     (C{str}, L{elisa.core.utils.defer.Deferred})
        """
        if os.path.exists(uri.path) and os.path.isfile(uri.path):
            return (uri.path, defer.succeed(uri.path))

        # TODO: this code should be factorized in a common caching module.
        if not os.path.exists(PICTURES_CACHE):
            os.makedirs(PICTURES_CACHE, 0755)
        cache_file = '%s.%s' % (md5(str(uri)).hexdigest(), uri.extension)
        thumbnail_file = os.path.join(PICTURES_CACHE, cache_file)
        if os.path.exists(thumbnail_file):
            return (thumbnail_file, defer.succeed(thumbnail_file))
        else:
            # Download the thumbnail and cache it
            def got_thumbnail(data):
                fd = file(thumbnail_file, 'wb')
                fd.write(data)
                fd.close()
                return thumbnail_file

            dfr = getPage(str(uri))
            dfr.addCallback(got_thumbnail)
            return (thumbnail_file, dfr)
