# -*- coding: utf-8 -*-
# Elisa - Home multimedia server
# Copyright (C) 2006-2008 Fluendo Embedded S.L. (www.fluendo.com).
# All rights reserved.
#
# This file is available under one of two license agreements.
#
# This file is licensed under the GPL version 3.
# See "LICENSE.GPL" in the root of this distribution including a special
# exception to use Elisa with Fluendo's plugins.
#
# The GPL part of Elisa is also available under a commercial licensing
# agreement from Fluendo.
# See "LICENSE.Elisa" in the root directory of this distribution package
# for details on that license.

from elisa.core import common

from elisa.core.common import application
from elisa.core.media_uri import MediaUri

from elisa.plugins.poblesec.link import Link
from elisa.plugins.poblesec.base.hierarchy import HierarchyController
from elisa.plugins.poblesec.base.list import GenericListViewMode
from elisa.plugins.poblesec.base.preview_list import MenuItemPreviewListController
from elisa.plugins.poblesec.base.coverflow import ImageWithReflectionCoverflowController
from elisa.plugins.poblesec.base.grid import GridItemGridController
from elisa.plugins.poblesec.base.list_switcher import ListSwitcherController

from elisa.plugins.shoutcast.models import ShoutcastGenreModel, \
                                           ShoutcastRadioStationModel

from elisa.plugins.favorites.models import FavoritesItem

from elisa.plugins.poblesec.actions import Action
from elisa.plugins.poblesec.actions import PlayAllAction
from elisa.plugins.poblesec.actions import AddToFavoritesAction

from twisted.internet import defer

from elisa.core.utils.i18n import install_translation

_ = install_translation('shoutcast')
_p = install_translation('poblesec')

import datetime


def shoutcast_decorator(controller):
    link = Link()
    link.controller_path = '/poblesec/shoutcast'
    link.label = _('Shoutcast')
    link.icon = 'elisa.plugins.shoutcast.logo'
    controller.model.append(link)
    return defer.succeed(None)


class StationPlayAction(PlayAllAction):
    def run(self):
        station = self.controller.station
        if station:
            self.controller.play_station(station)


class StationAddToFavoritesAction(AddToFavoritesAction):
    def run(self):
        if self.toggled:
            if self.controller.station:
                uri = unicode(str(self.controller.station.get_playable))
                dfr = common.application.store.find(FavoritesItem,
                                                    FavoritesItem.uri == uri,
                                                    FavoritesItem.foreign_class == u'InternetRadio')
                dfr.addCallback(lambda rs: rs.one())
                dfr.addCallback(self.remove)
        else:
            # expecting an ShoutcastController
            if self.controller.station:
                favorite = FavoritesItem()
                favorite.added_time = datetime.datetime.now()

                favorite.title = unicode(self.controller.station.name)
                favorite.uri = unicode(str(self.controller.station.get_playable))
                favorite.foreign_class = u'InternetRadio'
                application.store.add(favorite)
                self.store(favorite)


class GenericShoutcastController(HierarchyController):
    def initialize(self, uri=None, station=None):
        self.uri = uri
        self.station = station
        dfr = super(GenericShoutcastController, self).initialize()
        self.actions.extend(self.make_actions())
        return dfr

    def is_empty(self):
        return super(GenericShoutcastController, self).is_empty() and self.station is None

    def make_actions(self):
        actions = []

        if self.station:
            # display the contextual action related to the station
            action = StationPlayAction(self)
            action.title = _("Play Station '%s'") % self.station.name
            actions.append(action)

            action = StationAddToFavoritesAction(self)
            action.untoggled_title = _("Add Station '%s' To Your Favorites") % self.station.name
            action.toggled_title = _("Remove Station '%s' From Your Favorites") % self.station.name
            actions.append(action)

            uri = unicode(str(self.station.get_playable))
            dfr = common.application.store.find(FavoritesItem,
                                                FavoritesItem.uri == uri,
                                                FavoritesItem.foreign_class == u'InternetRadio')
            dfr.addCallback(lambda rs: rs.all())
            dfr.addCallback(action.setup)

        return actions

    def node_clicked(self, widget, item):
        if isinstance(item, Action):
            item.run()
        elif isinstance(item, ShoutcastGenreModel):
            browser = self.frontend.retrieve_controllers('/poblesec/browser')[0]
            browser.history.append_controller(self.path, item.name, uri=item.uri)
        elif isinstance(item, ShoutcastRadioStationModel):
            browser = self.frontend.retrieve_controllers('/poblesec/browser')[0]
            browser.history.append_controller(self.path, item.name, uri=None, station=item)

    def play_station(self, station):
        resource, dfr = application.resource_manager.get(station.get_playable, station)

        def play(station):
            player = self.frontend.retrieve_controllers('/poblesec/music_player')[0]
            player.player.play_model(station)

            main = self.frontend.retrieve_controllers('/poblesec')[0]
            main.show_music_player()
            self.stop_loading_animation()

        dfr.addCallback(play)


class ShoutcastController(GenericShoutcastController):

    default_uri = MediaUri("shoutcast://genres")

    def initialize(self, uri=default_uri, station=None):
        dfr = super(ShoutcastController, self).initialize(uri=uri, station=station)

        if uri and not station:
            def resource_loaded(resource):
                if isinstance(resource, ShoutcastGenreModel):
                    self.model.extend(resource.stations)
                else:
                    # FIXME: fallbacks on a list of genres
                    self.model.extend(resource.genres)
                return self

            def load_resource(controller):
                resource, get_dfr = application.resource_manager.get(self.uri)
                return get_dfr

            dfr.addCallback(load_resource)
            dfr.addCallback(resource_loaded)

        return dfr


class ShoutcastViewMode(GenericListViewMode):

    """
    Implementation of the common view modes API.
    """

    def get_label(self, item):
        if isinstance(item, Action):
             return defer.succeed(item.title)
        return defer.succeed(item.name)

    def get_default_image(self, item):
        resource = 'elisa.plugins.poblesec.file'
        if isinstance(item, Action):
            resource = item.icon
        elif isinstance(item, ShoutcastGenreModel):
            resource = 'elisa.plugins.poblesec.folder'
        elif isinstance(item, ShoutcastRadioStationModel):
            resource = 'elisa.plugins.poblesec.radio'
        return resource

    def get_image(self, item, theme):
        return None

    def get_preview_image(self, item, theme):
        return None


class ShoutcastPreviewListController(ShoutcastController, MenuItemPreviewListController):
    view_mode = ShoutcastViewMode

class ShoutcastCoverflowController(ShoutcastController, ImageWithReflectionCoverflowController):
    view_mode = ShoutcastViewMode

class ShoutcastGridController(ShoutcastController, GridItemGridController):
    view_mode = ShoutcastViewMode

class ShoutcastListSwitcherController(ListSwitcherController):
    modes = [ShoutcastPreviewListController,
             ShoutcastCoverflowController,
             ShoutcastGridController]
    default_mode = ShoutcastPreviewListController


def use_me_hook(frontend):
    """
    'Use me' hook that takes the user to the Music/Internet/Shoutcast section.

    @param frontend: the current frontend (poblesec)
    @type frontend:  L{elisa.plugins.pigment.pigment_frontend.PigmentFrontend}

    @return:         a deferred fired when the action is complete
    @rtype:          L{twisted.internet.defer.Deferred}
    """
    browser = frontend.retrieve_controllers('/poblesec/browser')[0]
    browser.history.clear()

    paths = [('/poblesec/music_menu', _p('Music')),
             ('/poblesec/music/internet', _p('Internet')),
             ('/poblesec/shoutcast', _('Shoutcast'))]

    # Select the music section in Elisa's main menu
    for index, link in enumerate(browser.history.current.model):
        if link.controller_path == paths[0][0]:
            browser.history.current.menu.selected_item_index = index
            break

    def append_controller_cb(result, path, label, **kwargs):
        return browser.history.append_controller(path, label, **kwargs)

    dfr = defer.succeed(None)
    for path, label in paths:
        dfr.addCallback(append_controller_cb, path, label)
    return dfr
