# Elisa - Home multimedia server
# Copyright (C) 2006,2007 Fluendo, S.A. (www.fluendo.com).
# All rights reserved.
#
# This file is available under one of two license agreements.
#
# This file is licensed under the GPL version 2.
# See "LICENSE.GPL" in the root of this distribution including a special
# exception to use Elisa with Fluendo's plugins.
#
# The GPL part of Elisa is also available under a commercial licensing
# agreement from Fluendo.
# See "LICENSE.Elisa" in the root directory of this distribution package
# for details on that license.


__maintainer__ = 'Florian Boucault <florian@fluendo.com>'
__maintainer2__ = 'Lionel Martin <lionel@fluendo.com>'


from elisa.core import log
from elisa.core.frontend import Frontend
from elisa.core.backend import Backend
from elisa.core import common
from twisted.internet import reactor, defer

from elisa.core.utils import locale_helper

class InterfaceController(log.Loggable):
    """ InterfaceController is responsible for creating, managing and deleting
    L{elisa.core.frontend.Frontend}s and L{elisa.core.backend.Backend}s
    defined by the user, and to associate them together. It also links the
    chosen L{elisa.base_components.input_provider.InputProvider}s with them.
    """

    def __init__(self):
        log.Loggable.__init__(self)
        self.debug("Creating")
        self._update_poll = None

        # Backends dictionary:
        # key : name defined in config
        # value : backend instance
        self._backends = {}

        # Frontends dictionary:
        # key : name defined in config
        # value : frontend instance
        self._frontends = {}

        # Activities dictionary:
        # key : Activity path
        # value : Activity instance
        self._activities = {}


    def initialize(self):
        """ Initialize various variables internal to the InterfaceController
        such as the backends and the frontends.

        This method is called after the application's configuration is loaded.
        """
        self.info("Initializing")
        application = common.application

        # Create backend instances declared in the configuration
        backend_paths = application.config.get_option('backends',
                                                      section='general',
                                                      default=[])
        for backend_path in backend_paths:
            try:
                self._create_backend(backend_path)
            except Exception, e:
                self.warning("An error occured causing backend '%s' creation " \
                             "to fail" % backend_path)
                application.handle_traceback()

                if backend_path in self._backends:
                    self._backends.pop(backend_path)

        # Create frontend instances declared in the configuration
        frontend_paths = application.config.get_option('frontends',
                                                       section='general',
                                                       default=[])
        for frontend_path in frontend_paths:
            try:
                self._create_frontend(frontend_path)
            except Exception, e:
                self.warning("An error occured causing frontend '%s' creation " \
                             "to fail" % frontend_path)
                application.handle_traceback()

                if frontend_path in self._frontends:
                    self._frontends.pop(frontend_path)

    def _create_backend(self, backend_name):
        self.debug("Initializing backend %s" % backend_name)
        application = common.application
        plugin_registry = application.plugin_registry
        
        master_backend = application.config.get_option('master',
                                                       backend_name, default='')

        # Create Activity and retrieve the root Model from it
        if master_backend == '':
            activity_path = application.config.get_option('activity',
                                                          backend_name,
                                                          default=[])
            try:
                activity = plugin_registry.create_component(activity_path)
            except:
                self.warning("Cannot create activity %r for backend %r",
                             activity_path, backend_name)
                raise

            root_model = activity.get_model()
            activity_key = "%s/%s" % (backend_name, activity.path)
            self._activities[activity_key] = activity
        else:
            root_model = self._backends[master_backend].root_controller.model

        # Create root Controller
        controller_path = application.config.get_option('controller',
                                                        backend_name,
                                                        default=[])
        try:
            root_controller = plugin_registry.create_component(controller_path)
        except:
            self.warning("Cannot create controller %r for backend %r",
                         controller_path, backend_name)
            raise

        backend = Backend(root_controller)
        
        # Create and connect InputProviders found in the config to root
        # Controller
        # FIXME: no checking is done for duplicated InputProviders
        providers_paths = self._create_input_providers(backend_name, backend)
        for provider in providers_paths:
            application.input_manager.subscribe(provider,
                                                backend.dispatch_input)
        
        root_controller.backend = backend
        root_controller.model = root_model
        root_controller.focus()

        self._backends[backend_name] = root_controller.backend


    def _create_frontend(self, frontend_name):
        self.debug("Initializing frontend %s" % frontend_name)
        application = common.application
        plugin_registry = application.plugin_registry


        # Retrieving the backend the new frontend will connect to
        # FIXME: this has to be changed to support remote backends
        backend_name = application.config.get_option('backend', frontend_name,
                                                     default=[])
        if self._backends.has_key(backend_name):
            backend = self._backends[backend_name]
            root_controller = backend.root_controller
        else:
            self.warning("Backend %r not existing for frontend %r", backend_name,
                         frontend_name)
            raise

        if root_controller == None:
            self.warning("Invalid backend %r for frontend %r",
                         backend_name, frontend_name)
            raise

        # Create root View
        view_path = application.config.get_option('view', frontend_name,
                                                  default=[])
        try:
            root_view = plugin_registry.create_component(view_path)
        except:
            self.warning("Cannot create view %r for frontend %r", view_path,
                         frontend_name)
            raise

        # Create theme
        theme_path = application.config.get_option('theme', frontend_name,
                                                   default=[])
        try:
            theme = plugin_registry.create_component(theme_path)
        except:
            self.warning("Cannot create theme %r for frontend %r", theme_path,
                         frontend_name)
            raise

        # Create a Context needed by the root View to render
        try:
            context = plugin_registry.create_component(root_view.context_path)
        except:
            self.warning("Cannot create context %r for frontend %r",
                         root_view.context_path, frontend_name)
            raise
    
        # do the translation things
        languages = application.config.get_option('languages', frontend_name,
                                                  None)
        if not languages:
            languages = []
            locale = locale_helper.get_from_system_locale()
            if locale:
                languages.append(locale)

        # create the frontend
        frontend = Frontend(context, root_view, theme)
        frontend.languages = languages
        frontend.name = frontend_name
        self._frontends[frontend_name] = frontend


        # Create and connect InputProviders found in the config to root Controller
        # FIXME: no checking is done for duplicated InputProviders
        providers_paths = self._create_input_providers(frontend_name, frontend, context.viewport_handle)
        for provider in providers_paths:
            application.input_manager.subscribe(provider, backend.dispatch_input)

        # connect the root controller to the root view
        root_view.frontend = frontend
        root_view.controller = root_controller

        return True


    def _create_input_providers(self, path, origin=None, viewport=None):
        application = common.application
        providers_paths = application.config.get_option("input_providers",
                                                         section=path,
                                                         default=[])
        providers = application.plugin_registry.create_components(providers_paths)
        self.debug("Loading and connecting %s input_providers" % len(providers))
        for provider in providers:
            provider.origin = origin
            provider.viewport = viewport
            application.input_manager.register_component(provider)

        return providers_paths

    def _update_frontends(self):
        for frontend in self._frontends.values():
            frontend.update()

        # FIXME: the framerate is fixed at 50 fps for all the frontends.
        # it does not look like an optimal solution
        self._update_poll = reactor.callLater(1/50.0, self._update_frontends)


    def start(self):
        """
        Start refreshing the frontends periodically.
        """
        self.info("Starting")

        for frontend in self._frontends.values():
            frontend.context.context_handle = frontend.view.context_handle

        # start the update loop
        self._update_poll = reactor.callLater(0.0, self._update_frontends)


    def stop(self):
        """
        Stop refreshing the frontends and terminate the backends.

        @rtype: L{twisted.internet.defer.Deferred}
        """
        dfrs = []
        self.info("Stopping")

        for frontend in self._frontends.values():
            dfr = defer.maybeDeferred(frontend.clean)
            dfrs.append(dfr)

        # FIXME: unsubscribe InputProviders ?

        def all_done(result):
            application = common.application

            if self._update_poll:
                try:
                    self._update_poll.cancel()
                except:
                    pass

            for activity_path, activity in self._activities.iteritems():
                self._activities[activity_path].save_config(application.config)

            return result

        dfr = defer.DeferredList(dfrs)
        dfr.addCallback(all_done)
        return dfr
