# -*- coding: utf-8 -*-
# Elisa - Home multimedia server
# Copyright (C) 2006,2007 Fluendo Embedded S.L. (www.fluendo.com).
# All rights reserved.
#
# This file is available under one of two license agreements.
#
# This file is licensed under the GPL version 2.
# See "LICENSE.GPL" in the root of this distribution including a special
# exception to use Elisa with Fluendo's plugins.
#
# The GPL part of Elisa is also available under a commercial licensing
# agreement from Fluendo.
# See "LICENSE.Elisa" in the root directory of this distribution package
# for details on that license.

"""
Metadata parsing of medias using metadata_provider components
"""

__maintainer__ = 'Philippe Normand <philippe@fluendo.com>'
__maintainer2__ = 'Benjamin Kampmann <benjamin@fluendo.com>'

from elisa.core import common
from elisa.core import manager
from elisa.base_components import metadata_provider
from twisted.internet import defer, threads

class MetadataManager(manager.Manager):
    """
    This manager is responsible for media metadata parsing. It creates
    and handles the L{elisa.base_components.metadata_provider.MetadataProvider}
    components.
    """

    def get_metadata(self, metadata):
        """ Retrieve the metadata of the media located at given URI.

        Metadata is returned as a dictionnary mapping keys depending
        on the content type (example: artist, album, track for 'audio'
        content-type) to actual metadata values.
        If you do a request for metadata, the value of this has to be a None
        Object.

        @param metadata:     a dicitionary containing the already known
                             metadata and empty values for the requested tags
        @type metadata:      dict
        @rtype:              L{twisted.internet.defer.Deferred}
        """
        d = defer.Deferred()

        self.debug("Looking for metadata %s" % metadata)
        mdict = metadata

        # sort the providers for ranking
        ranks = {}
        for provider in self._components:
            rank = provider.get_rank()
            if rank not in ranks:
                ranks[rank] = []
            ranks[rank].append(provider)

        provider = None
        if not ranks:
            d.callback(mdict)
            return d

        found_ranks = ranks.keys()
        found_ranks.sort()
        pos = 0
        prov = 0

        def start_searching(mdict, pos, prov, found_ranks, rank, d):
            #self.debug("Searching for %s" % mdict)
            if not self._still_empty_fields(mdict):
                d.callback(mdict)
                return d

            if pos >= len(found_ranks):
                d.callback(mdict)
                return d

            # get the providers
            providers = ranks[found_ranks[pos]]
            provider = providers[prov]

            # If there is another provider
            if prov < len(providers) -1:
                prov = prov + 1
            else:
                prov = 0
                pos = pos + 1

            if provider.able_to_handle(mdict):
                self.debug("Using metadata provider %s", provider.name)
                dfr = provider.get_metadata(mdict)
                dfr.addCallback(start_searching, pos, prov, found_ranks,
                                rank, d)
            else:
                self.debug("%s not able to handle %r", provider.name, mdict)
                dfr = start_searching(mdict, pos, prov, found_ranks, rank, d)
            return dfr

        def got_result(result):
            if result != None:
                for key, value in result.copy().iteritems():
                    if isinstance(value, basestring):
                        value = value.lower()
                    result[key] = value
            else:
                result = {}
            self.debug("Metadata found: %r", result)
            return result
        
        start_searching(mdict, pos, prov, found_ranks, rank, d)
        d.addCallback(got_result)
        return d

    def _still_empty_fields(self, mdict):
        """ Has the dictionary still empty fields?
            Empty means: that there is a None object as the value for a key
        """
        for key,value in mdict.items():
            if value == None:
                return True

        return False
