# Elisa - Home multimedia server
# Copyright (C) 2006,2007 Fluendo Embedded S.L. (www.fluendo.com).
# All rights reserved.
#
# This file is available under one of two license agreements.
#
# This file is licensed under the GPL version 2.
# See "LICENSE.GPL" in the root of this distribution including a special
# exception to use Elisa with Fluendo's plugins.
#
# The GPL part of Elisa is also available under a commercial licensing
# agreement from Fluendo.
# See "LICENSE.Elisa" in the root directory of this distribution package
# for details on that license.

from elisa.core.tests.elisa_test_case import ElisaTestCase
from elisa.core import plugin
from elisa.core import media_manager, metadata_manager, media_uri, media_scanner
from elisa.core import media_db, db_backend
from elisa.base_components import media_provider, metadata_provider
from elisa.core.tests.test_metadata_provider import FooMetadataProvider
import tempfile, shutil, os

from twisted.internet import defer

class FooMediaProvider(media_provider.MediaProvider):

    def supported_uri_schemes__get(self):
        return {'foo': 0}

class AnotherFooMediaProvider(media_provider.MediaProvider):

    def supported_uri_schemes__get(self):
        return {'foo': 2, 'bar': 0}

class FooPlugin(plugin.Plugin):
    name = 'foo'
    components = {'foo_media_provider': {'path': FooMediaProvider}}

class FooPlugin2(plugin.Plugin):
    name = 'foo2'
    components = {'another_foo_media_provider': {'path':AnotherFooMediaProvider},
                  'foo_metadata_provider': {'path': FooMetadataProvider}}

class LFS(plugin.Plugin):
    name = 'lfs'
    components = {'local_media': {'path': 'elisa.plugins.base.media_providers.local_media:LocalMedia'}}
    
class TestMediaManager(ElisaTestCase):


    def __init__(self, methodName='runTest'):
        ElisaTestCase.__init__(self, methodName=methodName)
        self._db_file = '/tmp/test.db'
        try:
            db = db_backend.DBBackend(db_backend='sqlite',
                                      database=self._db_file)
        except db_backend.DBBackendError, error:
            self.skip = error

    def setUp(self):
        ElisaTestCase.setUp(self)

        from elisa.core.common import application
        application.plugin_registry.register_plugin(LFS)
        application.plugin_registry.register_plugin(FooPlugin)
        application.plugin_registry.register_plugin(FooPlugin2)

        c = application.plugin_registry.create_component

        m_manager = metadata_manager.MetadataManager()
        m_manager.register_component(c('foo2:foo_metadata_provider'))

        self._media_manager = media_manager.MediaManager(m_manager)


        self._media_manager.register_component(c('foo:foo_media_provider'))
        self._media_manager.register_component(c('foo2:another_foo_media_provider'))
        self._media_manager.register_component(c('lfs:local_media'))

        self._media_manager.start()
        self._media_manager._media_scanner._main_loop_running = True
        
        
        self._tmp_dir = tempfile.mkdtemp()
        open(os.path.join(self._tmp_dir, 'test.mp3'), 'w').close()
        
        self.tmp_uri = media_uri.MediaUri(u'file://%s' % self._tmp_dir)

    def tearDown(self):
        self._media_manager.stop()
        self._media_manager._media_scanner._main_loop_running = False
        shutil.rmtree(self._tmp_dir)
        ElisaTestCase.tearDown(self)

    def test_media_provider_access(self):

        self.assertEquals(len(self._media_manager._components), 3)
        for provider in self._media_manager._components:
            self.failUnless(isinstance(provider, media_provider.MediaProvider))

        foo_uri = media_uri.MediaUri(u'foo://bar')
        foo_provider = self._media_manager._get_media_provider(foo_uri)
        self.failUnless(isinstance(foo_provider, FooMediaProvider))

        bar_uri = media_uri.MediaUri(u'bar://boo')
        bar_provider = self._media_manager._get_media_provider(bar_uri)
        self.failUnless(isinstance(bar_provider, AnotherFooMediaProvider))


    def test_media_db_access(self):
        self.failUnless(isinstance(self._media_manager.media_db,
                                   media_db.MediaDB))

    def test_get_tags(self):
        foo_uri = media_uri.MediaUri(u'foo://bar')

        # TODO: test db caching

        dfr = self._media_manager.get_metadata({'uri':foo_uri, 'artist':None,
                                                'album': None})
        dfr.addCallback(lambda tags: self.assertEquals(tags, {'uri': foo_uri,
                                                              'artist':'foo',
                                                              'album': 'bar'}))
        return dfr


    def test_add_source(self):
        self._media_manager.add_source(self.tmp_uri, 'audio')

        def done(r):
            self.failUnless(self._media_manager.get_source_for_uri(self.tmp_uri))
            
        self._media_manager._media_scanner._process_queue()
        done(None)

    def test_remove_source(self):
        self._media_manager.add_source(self.tmp_uri, 'audio')
        self.failUnless(self._media_manager.get_source_for_uri(self.tmp_uri))
        self._media_manager.remove_source(self.tmp_uri)

        def done(r):
            source = self._media_manager.get_source_for_uri(self.tmp_uri)
            self.failIf(source.available)

        self._media_manager._media_scanner._process_queue()
        done(None)

    def test_update_source(self):

        def source_updated(r, source_uri):
            source_updated.called = True

            self.assertEquals(source_uri, self.tmp_uri)

        source_updated.called = False

        self._media_manager.add_source(self.tmp_uri, 'audio')
        
        self._media_manager._media_scanner._process_queue()
        source_updated(None, self.tmp_uri)
        self.check_called(source_updated)

    def test_get_source(self):
        self._media_manager.add_source(self.tmp_uri, 'audio')

        def done(r):
            source = self._media_manager.get_source_for_uri(self.tmp_uri)
            self.failUnless(source)

            self.assertEquals(source.uri, str(self.tmp_uri))

        self._media_manager._media_scanner._process_queue()
        done(None)

    def test_get_media(self):
        self._media_manager.add_source(self.tmp_uri, 'video')

        def done(r):
            source = self._media_manager.get_source_for_uri(self.tmp_uri)
            self.failUnless(source)

            db = self._media_manager.media_db
            foo_uri = media_uri.MediaUri(u'file:///tmp/foo.avi')
            r = db.add_media(foo_uri, 'foo', source, 'video')
            self.assertEquals(r, True)
            
            foo_row = self._media_manager.get_media_information(foo_uri)
            self.assertEquals(foo_row.uri, str(foo_uri))
            
        self._media_manager._media_scanner._process_queue()
        done(None)

        
