# -*- coding: utf-8 -*-
# Elisa - Home multimedia server
# Copyright (C) 2006,2007 Fluendo Embedded S.L. (www.fluendo.com).
# All rights reserved.
#
# This file is available under one of two license agreements.
#
# This file is licensed under the GPL version 2.
# See "LICENSE.GPL" in the root of this distribution including a special
# exception to use Elisa with Fluendo's plugins.
#
# The GPL part of Elisa is also available under a commercial licensing
# agreement from Fluendo.
# See "LICENSE.Elisa" in the root directory of this distribution package
# for details on that license.

from elisa.core.tests.elisa_test_case import ElisaTestCase
from elisa.core import common, plugin
from elisa.core import media_manager, metadata_manager, media_uri, media_scanner
from elisa.core import media_db, media_file
from elisa.base_components import media_provider, metadata_provider
from twisted.internet import defer

class FooMediaProvider(media_provider.MediaProvider):
    name = 'foo_media'

    def supported_uri_schemes__get(self):
        return {'foo': 0}

    def get_media_type(self, uri):
        media_type = {'file_type': 'audio',
                      'mimetype': ''}
        if media_uri.MediaUri(u'foo://foo') == uri:
            media_type['mimetype'] = 'audio/mpeg'
        return media_type

    def is_directory(self, uri):
        is_dir = False
        if media_uri.MediaUri(u'foo://bar') == uri:
            is_dir = True
        return is_dir

    def has_children(self, uri):
        has_children = False
        if media_uri.MediaUri(u'foo://bar') == uri:
            has_children = True
        return has_children

    def get_direct_children(self, uri):
        children = []
        if media_uri.MediaUri(u'foo://bar') == uri:
            # TODO
            pass
        return children

    def next_location(self, uri):
        foo_foo_uri = media_uri.MediaUri(u'foo://foo')
        foo_bar_uri = media_uri.MediaUri(u'foo://bar')

        next = None
        if uri == foo_foo_uri:
            next = foo_bar_uri
        elif uri == foo_bar_uri:
            next = foo_foo_uri
        return next

    def previous_location(self, uri):
        foo_foo_uri = media_uri.MediaUri(u'foo://foo')
        foo_bar_uri = media_uri.MediaUri(u'foo://bar')

        previous = None
        if uri == foo_bar_uri:
            previous = foo_foo_uri
        elif uri == foo_foo_uri:
            previous = foo_bar_uri
        return previous

    already_copied = False

    def copy(self, orig_uri, dest_uri, recursive=False):
        if not self.already_copied and \
               orig_uri == media_uri.MediaUri(u'foo://foo'):
            self.already_copied = True
            return True
        else:
            return False

    def move(self, orig_uri, dest_uri):
        return True

    def delete(self, uri, recursive=False):
        return True

    def open(self, uri, block=True):
        m = media_file.MediaFile(self, 0)
        if block:
            fd = m
        else:
            fd = defer.Deferred()
            fd.callback(m)
        return fd

    def close(self, media_file, block=True):
        if not media_file:
            raise TypeError("No file to close")
        if block:
            d = media_file.close(self)
        else:
            def do_close(r):
                r.close(self)
            d = defer.Deferred()
            d.addCallback(do_close)
            d.callback(media_file)
        return d

    def uri_is_monitorable(self, uri):
        return True

    monitored = {}

    def monitor_uri(self, uri, callback):
        if uri not in self.monitored:
            self.monitored[uri] = True
            r = True
        else:
            r = False
        return r

    def unmonitor_uri(self, uri):
        if uri in self.monitored:
            del self.monitored[uri]
            r = True
        else:
            r = False
        return r

    def uri_is_monitored(self, uri):
        return uri in self.monitored

class TestMediaProvider(ElisaTestCase):

    def setUp(self):
        ElisaTestCase.setUp(self)
        self._foo = FooMediaProvider()
        self._foo_foo_uri = media_uri.MediaUri(u'foo://foo')
        self._foo_bar_uri = media_uri.MediaUri(u'foo://bar')

    def test_supported_uri_schemes(self):
        self.assertEquals(self._foo.supported_uri_schemes, {'foo': 0})

    def test_get_media_type(self):
        self.assertEquals(self._foo.get_media_type(self._foo_foo_uri),
                          {'file_type': 'audio','mimetype': 'audio/mpeg'})

    def test_is_directory(self):
        self.assertEquals(self._foo.is_directory(self._foo_bar_uri), True)

    def test_has_children(self):
        self.assertEquals(self._foo.has_children(self._foo_bar_uri), True)

    def test_get_direct_children(self):
        # TODO
        expected = []
        self.assertEquals(self._foo.get_direct_children(self._foo_bar_uri),
                          expected)

    def test_file_access(self):
        # TODO

        fooko = media_uri.MediaUri(u'foo://foo/fooko.mp3')
        self.failUnless(self._foo.copy(self._foo_foo_uri, fooko))

        self.assertEquals(self._foo.copy(self._foo_bar_uri, fooko), False)

        self.assertEquals(self._foo.move(self._foo_foo_uri, self._foo_bar_uri),
                          True)
        self.assertEquals(self._foo.delete(fooko), True)


        mfile = self._foo.open(self._foo_foo_uri)
        self.failUnless(mfile)

        self.assertRaises(Exception, self._foo.close)

        def opened(mfile):
            opened.called = True

        opened.called = False

        dfr = self._foo.open(self._foo_foo_uri, block=False)
        dfr.addCallback(opened)
        return dfr

    def test_navigation(self):
        # TODO

        bar = self._foo_bar_uri
        foo = self._foo_foo_uri
        self.assertEquals(self._foo.next_location(self._foo_foo_uri),
                          bar)
        self.assertEquals(self._foo.previous_location(self._foo_bar_uri),
                          foo)

    def test_monitoring(self):
        # TODO

        self.failUnless(self._foo.uri_is_monitorable(self._foo_foo_uri))
        self.failIf(self._foo.unmonitor_uri(self._foo_foo_uri))

        def foo_updated(event):
            pass

        self._foo.monitor_uri(self._foo_foo_uri, foo_updated)
        self.assertEquals(self._foo.uri_is_monitored(self._foo_foo_uri), True)

        self._foo.unmonitor_uri(self._foo_foo_uri)

