# Elisa - Home multimedia server
# Copyright (C) 2006,2007 Fluendo Embedded S.L. (www.fluendo.com).
# All rights reserved.
#
# This file is available under one of two license agreements.
#
# This file is licensed under the GPL version 2.
# See "LICENSE.GPL" in the root of this distribution including a special
# exception to use Elisa with Fluendo's plugins.
#
# The GPL part of Elisa is also available under a commercial licensing
# agreement from Fluendo.
# See "LICENSE.Elisa" in the root directory of this distribution package
# for details on that license.

from twisted.trial import unittest
from elisa.core.tests.elisa_test_case import ElisaTestCase
from elisa.core import log

from elisa.core.bus import bus, bus_message



class DataMessage(bus_message.Message):
    def __init__(self, data):
        bus_message.Message.__init__(self)
        self.data = data

class FooMessage(bus_message.Message):
    def __init__(self, foo):
        bus_message.Message.__init__(self)
        self.foo = foo


class TestMessageBus(ElisaTestCase):

    def test_simple_emit(self):
        """ Check a message receiver callback correctly gets called on
        message bus dispatch.
        """

        mybus = bus.Bus()

        received_messages = []

        def on_message(message, sender):
            received_messages.append(message)

        mybus.register(on_message)

        for i in range(5):
            data = '%s. Hello you' % i
            mybus.send_message(DataMessage(data), sender=self)

        self.assertEqual(len(mybus._queue), 5)

        mybus._dispatch_messages()
        self.assertEqual(len(mybus._queue), 0)

        self.assertEqual(len(received_messages), 5)

        mybus.unregister(on_message)

    def test_message_filter(self):
        """
        """
        mybus = bus.Bus()

        received_messages = []

        def on_message(message, sender):
            received_messages.append(message)

        mybus.register(on_message, FooMessage)

        for i in range(5):
            data = '%s. Hello you' % i
            mybus.send_message(DataMessage(data), sender=self)

        sent_foo_messages = []
        for i in range(5):
            data = '%s. Hello you' % i
            msg = FooMessage(data)
            mybus.send_message(msg, sender=self)
            sent_foo_messages.append(msg)

        self.assertEqual(len(mybus._queue), 10)

        mybus._dispatch_messages()
        self.assertEqual(len(mybus._queue), 0)

        self.assertEqual(len(received_messages), 5)
        self.assertEqual(sent_foo_messages, received_messages)

        mybus.unregister(on_message)

    def test_multiple_filters(self):
        """
        """
        mybus = bus.Bus()

        received_messages = []

        def on_message(message, sender):
            received_messages.append(message)

        mybus.register(on_message, DataMessage,FooMessage)

        sent_messages = []

        for i in range(5):
            data = '%s. Hello you' % i
            msg = DataMessage(data)
            mybus.send_message(msg, sender=self)
            sent_messages.append(msg)

        for i in range(5):
            data = '%s. Hello you' % i
            msg = FooMessage(data)
            mybus.send_message(msg, sender=self)
            sent_messages.append(msg)

        self.assertEqual(len(mybus._queue), 10)

        mybus._dispatch_messages()
        self.assertEqual(len(mybus._queue), 0)

        self.assertEqual(len(received_messages), 10)
        self.assertEqual(sent_messages, received_messages)

    def test_decorator(self):
        """
        """
        received_messages = []

        mybus = bus.Bus()

        @bus.bus_listener(mybus, FooMessage)
        def on_message(message, sender):
            received_messages.append(message)

        for i in range(5):
            data = '%s. Hello you' % i
            mybus.send_message(DataMessage(data), sender=self)

        sent_foo_messages = []
        for i in range(5):
            data = '%s. Hello you' % i
            msg = FooMessage(data)
            mybus.send_message(msg, sender=self)
            sent_foo_messages.append(msg)

        self.assertEqual(len(mybus._queue), 10)

        mybus._dispatch_messages()
        self.assertEqual(len(mybus._queue), 0)

        self.assertEqual(len(received_messages), 5)
        self.assertEqual(sent_foo_messages, received_messages)

        mybus.unregister(on_message)
