# -*- coding: utf-8 -*-
# Elisa - Home multimedia server
# Copyright (C) 2006,2007 Fluendo Embedded S.L. (www.fluendo.com).
# All rights reserved.
#
# This file is available under one of two license agreements.
#
# This file is licensed under the GPL version 2.
# See "LICENSE.GPL" in the root of this distribution including a special
# exception to use Elisa with Fluendo's plugins.
#
# The GPL part of Elisa is also available under a commercial licensing
# agreement from Fluendo.
# See "LICENSE.Elisa" in the root directory of this distribution package
# for details on that license.

"""
Functions used to extend setuptools and add elisa specific metadata
when distributing plugins.
"""

__maintainer__ = 'Philippe Normand <philippe@fluendo.com>'

import os
import types
from elisa.core import config
from elisa.core.utils import i18n
from setuptools import find_packages
from distutils.errors import DistutilsSetupError, DistutilsOptionError
from distutils.core import Command

def write_dict(cmd, basename, filename, force=False):
    """
    """
    argname = os.path.splitext(basename)[0]
    arg_value = getattr(cmd.distribution, argname, None)
    text_value = ""
    if arg_value is not None:
        for key, value in arg_value.iteritems():
            
            text_value += "%s = %s\n" % (key, value)
    cmd.write_or_delete_file(argname, filename, text_value, force)

def assert_dict_or_none(dist, attr, value):
    """Verify that value is a dictionary"""
    if type(value) not in (dict, types.NoneType):
        raise DistutilsSetupError("%r must be a dictionary or None value. "\
                                  "Got %r" % (attr,value))

def scan_plugin_conf(conf_path, name_prefix='elisa-plugin-'):
    """ Read the config located at given path and return data suitable
    to distutils' setup().

    @param conf_path:   absolute or relative path to a plugin.conf file
    @type conf_path:    string
    @param name_prefix: plugin's name prefix
    @type name_prefix:  string
    @rtype:             dict
    """
    cfg = config.Config(conf_path)
    section = cfg.get_section('general')

    # Basic distutils supported keywords
    name = section.get('name', default='unknown')
    version = section.get('version', default='0.1')
    description = section.get('description', default='no description')
    long_description = section.get('long_description', default=description)
    author = section.get('author', default='John Doe')
    author_email = section.get('author_email', default='john@doe.com')
    keywords = ','.join(section.get('keywords', default=[]))
    license = section.get('license', default='UNKNOWN')

    # Elisa specific metadata, to go in elisa_infos.txt 
    category_id = section.get('category_id', default='unknown')
    plugin_deps = ','.join([d.strip() for d in
                            section.get('plugin_dependencies', default=[])])
    py_deps = section.get('external_dependencies', default=[])
    for component_path, component_config in cfg.as_dict().iteritems():
        if component_path == 'general':
            continue
        for dep in component_config.get('external_dependencies',[]):
            dep = dep.strip()
            if dep not in py_deps:
                py_deps.append(dep)
                
    py_deps = ','.join(py_deps)
    ext_deps = ''
    elisa_infos = {'plugin_deps': plugin_deps, 'py_deps': py_deps,
                   'ext_deps': ext_deps, 'category_id': category_id}

    # Setuptools specific: entry_points keyword
    package_path = os.path.split(os.path.dirname(conf_path))
    if package_path[0] in ('','.'):
        package_path = package_path[1:]
    package_path = '.'.join(package_path)
    entry_points = {'elisa.plugins': ['%s = %s'  % (name, package_path),]}
    
    infos = {'name': '%s%s' % (name_prefix, name),
             'description': description, 'license': license,
             'keywords': keywords,
             'version': version, 'long_description': long_description,
             'author': author, 'author_email': author_email,
             'elisa_infos': elisa_infos, 'packages': find_packages(),
             'entry_points': entry_points}
    return infos


class build_po(Command):
    """
    Read a set of Elisa translation files, compile po files existing
    in the directories listed in those files and ship mo files as
    package_data of the distribution.

    Translation files have the following format::
    
      #
      # domain path/to/input /path/to/output
      # other-domain path/to/input/only
      #
      #
      # Don't forget the empty line at the end!

    """

    description = "compile Elisa translation file(s)"

    user_options = [
        ('trans-files=', 't', 'Elisa translation files')
        ]

    
    def initialize_options(self):
        self.trans_files = []

    def finalize_options (self):
        self.ensure_filename_list('trans_files')
        if not self.trans_files:
            self.trans_files = ['data/translations.lst',]
            
    def run(self):
        mo_files = i18n.compile_translations_from_files(self.trans_files)
        self.distribution.package_data.update(mo_files)

    def ensure_filename_list(self, option):
        self.ensure_string_list(option)
        for fname in getattr(self, option):
            if not os.path.exists(fname):
                msg = "'%s' does not exist or is not a file" % fname
                raise DistutilsOptionError("error in '%s' option: %s" % (option,
                                                                         msg))
