# -*- coding: utf-8 -*-
# Elisa - Home multimedia server
# Copyright (C) 2006,2007 Fluendo Embedded S.L. (www.fluendo.com).
# All rights reserved.
#
# This file is available under one of two license agreements.
#
# This file is licensed under the GPL version 2.
# See "LICENSE.GPL" in the root of this distribution including a special
# exception to use Elisa with Fluendo's plugins.
#
# The GPL part of Elisa is also available under a commercial licensing
# agreement from Fluendo.
# See "LICENSE.Elisa" in the root directory of this distribution package
# for details on that license.


__maintainer__ = 'Florian Boucault <florian@fluendo.com>'


import os, math
import weakref

import pgm

from pypgmtools.widgets.curved_list import *
from pypgmtools.widgets.grid_view import *
from pypgmtools.timing import implicit
from pypgmtools.graph.image import Image

from elisa.core import common
from elisa.core.media_manager import MediaProviderNotFound
from elisa.core import plugin_registry

from list_cache import CacheList

NodeViewClass = plugin_registry.get_component_class('base:node_view')

class NodeView(NodeViewClass):

    supported_controllers = ('poblenou:node_controller',)
    default_associations = (
    ('poblenou:node_controller','poblenou:node_view'),
    )

    def __init__(self):
        NodeViewClass.__init__(self)
        self.context_path = 'pigment:pigment_context'
        self.context_handle = None

        self._representation = None

        # menu widget
        self._menu = None
        self._animated_menu = None

        # number of preloaded children in terms of graphical ressources
        self._preloaded_children = 25
    
    def clean(self):
        # FIXME: not needed if doing nothing more than calling parent's
        NodeViewClass.clean(self)

    def frontend_changed(self, previous_frontend, new_frontend):
        if new_frontend != previous_frontend:
            if previous_frontend != None:
                signal = previous_frontend.context.canvas_resized
                signal.disconnect(self.canvas_resized)
                previous_frontend.theme_changed.disconnect(self.theme_changed)
                
            if new_frontend == None:
                return

            # FIXME: re-bind the widgets to the new canvas
 
            # connects to the canvas_resized signal of the new context
            signal = new_frontend.context.canvas_resized
            signal.connect(self.canvas_resized)

            # connects to the theme_changed signal of the new context
            new_frontend.theme_changed.connect(self.theme_changed)
            
    def canvas_resized(self, size):
        canvas = self.frontend.context.canvas

        if self._menu != None:
            self._scale_menu()

    def theme_changed(self, new_theme):
        if self.controller.model.thumbnail_source == None and \
               self._representation != None:
            widget = self._representation() 
            if widget != None:
                icon = self.controller.model.theme_icon
                image_path = self.frontend.theme.get_media(icon)
                widget.set_from_fd(os.open(image_path,
                                           os.O_RDONLY))
           
    def controller_changed(self):
        # FIXME: should deal with non-empty controller
        # FIXME: not needed if doing nothing more than calling parent's        
        NodeViewClass.controller_changed(self)

    def _create_list_menu(self):
        canvas = self.frontend.context.canvas
        self._menu = CurvedList(canvas, pgm.DRAWABLE_MIDDLE,
                                orientation = HORIZONTAL)
        self._cache_menu = CacheList(self._preloaded_children,
                                     self._menu)

    def _create_grid_menu(self):
        canvas = self.frontend.context.canvas
        masks = [self.frontend.theme.get_media("grid_mask_left"),
                 self.frontend.theme.get_media("grid_mask_right")]
        self._menu = GridView(canvas,
                              pgm.DRAWABLE_MIDDLE,
                              width = canvas.width,
                              height = canvas.height * (1 - 0.128),
                              columns = 2,
                              rows = 2,
                              duration = 250,
                              transformation = implicit.DECELERATE,
                              scroll_mode = PAGE_WISE,
                              orientation = LEFT_RIGHT,
                              background_masks = None,
                              background_mask_size  = 64.0 / 512.0)
        self._menu.columns, self._menu.rows = self._compute_grid_size()

    def _compute_grid_size(self):
        canvas = self.frontend.context.canvas
        r = float(canvas.width) / float(canvas.height)
        max_columns = 7
        n = len(self)
        w = math.sqrt(n*r)
        h = w/r
        return (math.floor(w), math.ceil(h))

    def _create_menu(self):
        canvas = self.frontend.context.canvas

        old_menu_widget = self._menu

        if self.controller.visualisation_mode == "list":
            self._create_list_menu()
        elif self.controller.visualisation_mode == "grid":
            self._create_grid_menu()

        if old_menu_widget != None:
            old_widgets = old_menu_widget.widgets
            old_animated_widgets = old_menu_widget.animated_widgets
            old_selection = old_menu_widget.selected_item
            self.root.root_group.remove(old_menu_widget)

            for animated_widget in old_animated_widgets:
                animated_widget.stop_animations()

            for widget in old_widgets:
                self._menu.append(widget)

            self._menu.selected_item = old_selection

            if self.controller.visualisation_mode == "grid":
                self._menu.layout()
                # FIXME: grid widget API specific
                self._menu.refresh()
                self._menu.update()

 
        self._menu.fg_color = (255, 255, 255, 255)
        self._menu.bg_color = (0, 0, 0, 0)
        
        self.root.root_group.add(self._menu)

        self._animated_menu = implicit.AnimatedObject(self._menu)
        menu = self._animated_menu
        menu.mode = implicit.REPLACE
        menu.setup_next_animations(duration = 350,
                                   transformation = implicit.DECELERATE)
        self.context_handle = self._animated_menu
       
        self._scale_menu()
        self._menu.visible = True

    def _scale_menu(self):
        # set canvas size dependent properties of menu (size, position)
        canvas = self.frontend.context.canvas

        x = 0.0
        y = canvas.height * (1.0 - 0.200)
        z_step = 500.0
        self._hidden_position = (x, y, 500.0)
        self._position = [(x, y, z) for z in range(-z_step*3, 0.0, z_step)]

        if self.controller.visualisation_mode == "list":
            self._selected_position = (x, y, 0.0)
            self._menu.width = canvas.width
            self._menu.height = canvas.height/7.5
            self._menu.visible_items = 11
        elif self.controller.visualisation_mode == "grid":
            self._selected_position = (0.0, 0.0, 1.0)
            # FIXME: broken in grid widget
            #self._menu.width = canvas.width
            #self._menu.height = canvas.height

        if self.controller.selected:
            self._menu.opacity = 255
            self._menu.position = self._selected_position
            self.hide_parents(2)
            self.hide_children()
        else:
            self._animated_menu.position = self._hidden_position
            self._animated_menu.opacity = 0

    def _create_thumbnail_widget(self, controller):
        widget = Image()
        widget.bg_color = (0, 0, 0, 0)
        widget.layout = pgm.IMAGE_ZOOMED
        widget.alignment = pgm.IMAGE_CENTER
        widget.opacity = 255

        widget.connect("clicked", self._drawable_clicked, controller)
 
        icon = controller.model.theme_icon
        # FIXME: cloning deactivated because set_image_from_image is not ready
        # in Pigment yet; before reactivating it, be sure to use Python's
        # WeakValueDictionary for the caching
        # (see http://docs.python.org/lib/module-weakref.html)
        #widget.set_from_image(self.root.get_icon_drawable(icon))
        image_path = self.frontend.theme.get_media(icon)
        widget.set_from_fd(os.open(image_path, os.O_RDONLY))
        widget.set_name(icon)

        widget.visible = True
         
        uri = controller.model.thumbnail_source
        if uri != None:
            self._thumbnail_source_to_image(uri, widget)

        return widget

    def _thumbnail_source_to_image(self, uri, image):
        try:
            dfr = common.application.media_manager.get_media_type(uri)
        except MediaProviderNotFound, e:
            self.info("Could not make a thumbnail for %s: no MediaProvider"
                      " found for this kind of URI (scheme: %s)" % (uri,
                      uri.scheme))
            # If there is no provider for this uri existing, we can't go on
            return

        def got_media_type(media_type):
            file_type = media_type["file_type"]
            dfr2 = common.application.thumbnailer.get_thumbnail(uri,
                                                                256,
                                                                file_type)
            def display(result):
                image.set_from_fd(os.open(result[0], os.O_RDONLY))
                image.set_name(result[0])

            def error(error):
                try:
                    error.raiseException()
                except Exception, e:
                    self.info(e)

            dfr2.addCallback(display).addErrback(error)

        dfr.addCallback(got_media_type)


    def create_representation(self, controller):
        representation = self._create_thumbnail_widget(controller)
        self._representation = weakref.ref(representation)
        return self._representation()

    def attribute_set(self, key, old_value, new_value):
        if key == 'current_index':
            self.debug("new index for %s: %s" % (self, new_value))

            if self._menu != None:
                if new_value == old_value:
                    return

                self._cache_menu.current_index = new_value
                if self.controller.selected:
                    self._display_description()
                    self._display_arrow()

        elif key == 'selected':
            self.debug("%s (un)selected: %s" % (self, new_value))
            if new_value:
                if self._animated_menu != None:
                    self._animated_menu.opacity = 255
                    self._animated_menu.position = self._selected_position

                self._display_description()
                self._display_arrow()

                self.hide_parents(2)
                self.hide_children()

                self.root.display_loading(self.controller.model.loading)

        elif key == 'loading':
            self.debug("%s loading: %s" % (self, new_value))
            if new_value == old_value or not self.controller.selected:
                return

            if new_value == True:
                self.root.display_loading(True)
            else:
                if self.controller.model.has_children:
                    self.root.display_loading(False)
                else:
                    self.root.display_empty(True)

        elif key == 'thumbnail_source':
            if self._representation != None:
                widget = self._representation() 
                if widget != None:
                    self._thumbnail_source_to_image(new_value, widget)

        elif key == 'visualisation_mode':
            self.info("Switching to %s mode" % new_value)
            self._create_menu()

        elif key == 'has_children':
            if isinstance(self.parent, NodeView):
                self.parent._display_arrow()

            if new_value:
                self.root.display_loading(False)

    def hide_parents(self, depth):
        parent = self.parent
        if isinstance(parent, NodeView):
            parent._animated_menu.opacity = 35 * max(depth, 0)
            parent._animated_menu.position = parent._position[max(depth, 0)]
            parent.hide_parents(depth-1)

    def hide_children(self):
        child_index = self.controller.current_index
        if child_index >= 0 and child_index < len(self):
            self[child_index].hide()
            self[child_index].hide_children()

    def hide(self):
        if self._menu != None:
            self._animated_menu.opacity = 0
            self._animated_menu.position = self._hidden_position

    def child_view_created(self, view, position):
        NodeViewClass.child_view_created(self, view, position)
        if position == self.controller.current_index and self.controller.selected:
            self._display_description()
            self._display_arrow()

        if self._menu == None:
            self._create_menu()

        view_creator = lambda: view.create_representation(view.controller)
        self._cache_menu.insert(position, view_creator)

    def removed(self, elements, position):
        self.debug("%s got a REMOVED at %s for %s elements" % \
                   (self, position, len(elements)))

        # FIXME: only handles removal of 1 element
        if len(elements) > 1:
            self.warning("Support for multiple elements removal not yet \
                          implemented")
            return

        self._cache_menu.pop(position)
        NodeViewClass.removed(self, elements, position)
        
        if len(self) == 0:
            self.debug("Menu of %s is empty: setting it to None" % self)
            self._menu = None
        elif self.controller.selected:
            self.debug("%s is updating the menu" % self)
            self._display_description()
            self._display_arrow()

    def _display_description(self):
        # display the description of currently selected item
        if self.controller.current_index < len(self) and \
           self.controller.selected:
            current_index = self.controller.current_index
            description = self[current_index].controller.model.text
        else:
            description = ""

        self.root.display_description(description)

    def _display_arrow(self):
        # display an animated arrow if the currently selected item has
        # children
        if self.controller.current_index < len(self) and \
           self[self.controller.current_index].controller.model.has_children and \
           self.controller.selected:
            self.root.display_arrow(True)
        else:
            self.root.display_arrow(False)

    def _drawable_clicked(self, drawable, x, y, z, button, time, controller):
        canvas = self.frontend.context.canvas
        self.debug("%s clicked" % controller.model.text)
        
        if drawable.opacity == 0:
            return False

        if self._root._drag_started:
            # do not process the click if there is vertical dragging
            # currently happening
            dy = self._root._drag_start_position[1] - y
            if dy > 0.05:
                return True

            # do not process the click if there is horizontal dragging
            # happening in the current menu level
            if self._root._drag_start_index != self.controller.parent.current_index:
                return True

        if self.controller.root.selected_controller == self.controller.parent:
            # this level is currently selected
            current_index = self.controller.parent.current_index
            clicked_index = self.controller.parent.index(self.controller)
            if current_index != clicked_index:
                # scroll to the clicked entry of this level
                self.controller.parent.current_index = clicked_index
            else:
                # enter the clicked entry of the menu
                self.controller.parent.activate_node()

            return True
    
        return False
