# -*- coding: utf-8 -*-
# Elisa - Home multimedia server
# Copyright (C) 2006,2007 Fluendo Embedded S.L. (www.fluendo.com).
# All rights reserved.
#
# This file is available under one of two license agreements.
#
# This file is licensed under the GPL version 2.
# See "LICENSE.GPL" in the root of this distribution including a special
# exception to use Elisa with Fluendo's plugins.
#
# The GPL part of Elisa is also available under a commercial licensing
# agreement from Fluendo.
# See "LICENSE.Elisa" in the root directory of this distribution package
# for details on that license.

"""
This metadata provider, looks if there is might a Cover (file) in the
directory of the file.
"""


__maintainer__ = 'Benjamin Kampmann <benjamin@fluendo.com>'


from elisa.base_components.metadata_provider import MetadataProvider
import os
from twisted.internet import threads
from elisa.core.media_uri import MediaUri


class CoverInDir(MetadataProvider):
    """
    This CoverInDir Metadata Provider just looks if there are might covers in
    the directory of the URI. If there are more than one found, it uses the
    best suited one.

    @ivar list_of_extentions:   a list of extentions our cover could have.
    @type list_of_extentions:   list

    @ivar list_of_names:        a list of good names for our cover sorted by
                                priority. if there is none found, which is
                                exactly spelled this way, we'll look for one,
                                which contains any of this names...

    @type list_of_names:        list
    """

    # lowercase only
    list_of_extentions = ['png', 'jpg', 'xpm', 'jpeg']

    # lowercase only
    list_of_names = ['cover', 'front', 'cd', 'dvd']
    # TODO: add some more
    # TODO: Should we use multilingual ones, too?


    def get_rank(self):
        ## We'll look here first
        return 120

    def able_to_handle(self, metadata):
        able = False
        content_type = metadata.get('content-type')

        if content_type == 'audio' and metadata.has_key('uri'):
            uri = metadata['uri']
            cover = metadata.get('cover')
            default_image = metadata.get('default_image', '')
            if uri.scheme == 'file' and not cover and not default_image:
                able = True
        return able

    def get_metadata(self, metadata):
        d = threads.deferToThread(self._search_for_cover, metadata)
        return d


    def _search_for_cover(self, metadata):
        parent = metadata['uri'].parent
        directory = parent.path
        
        if os.path.isdir(directory):
            file_list = []
            for onefile in os.listdir(directory):
                if os.path.isdir(onefile):
                    continue
 
                # ???
                if onefile.find('.') == -1:
                    continue
                
                name, ext = os.path.splitext(onefile)
                if ext[1:].lower() in self.list_of_extentions:
                    file_list.append((parent.join(onefile), name.lower()))
                    
            # No suited files found. Return
            if len(file_list) == 0:
                return metadata

            # Only one is found: use it!
            if len(file_list) == 1:
                return self._set_cover(metadata,file_list[0][1])

            # Let's search for each name

            for name in self.list_of_names:
                for filepath,filename in file_list:
                    if name == filename:
                        return self._set_cover(metadata, filepath)

            ## We are still continuing, so let's check for partial names
            for name in self.list_of_names:
                for filepath,filename in file_list:
                    if filename.find(name) != -1:
                        return self._set_cover(metadata, filepath)

        return metadata


    def _set_cover(self, metadata, cover_path):
        if metadata.get('cover', None) == None:
            metadata['cover'] = cover_path


        #### Set globally image if not set yet:
        if metadata.get('default_image', '') == None:
            metadata['default_image'] = cover_path

        return metadata



if __name__ == "__main__":

    from twisted.internet import reactor

    c = CoverInDir()
    uri = MediaUri("file:///tmp/covers/momo.mp3")

    print "rank:",c.get_rank()
    print "Should be able to handle:", c.able_to_handle({'uri':uri})
    print "Should not be able to handle:", c.able_to_handle({'uri' : uri,
                                                        'cover': 'blalbal'})

    def start():

        def print_it(metadata):
            print "metadata is", metadata


        df = c.get_metadata({'uri' : uri})
        df.addCallback(print_it)


    reactor.callWhenRunning(start)

    reactor.run()
