# -*- coding: utf-8 -*-
# Elisa - Home multimedia server
# Copyright (C) 2006,2007 Fluendo Embedded S.L. (www.fluendo.com).
# All rights reserved.
#
# This file is available under one of two license agreements.
#
# This file is licensed under the GPL version 2.
# See "LICENSE.GPL" in the root of this distribution including a special
# exception to use Elisa with Fluendo's plugins.
#
# The GPL part of Elisa is also available under a commercial licensing
# agreement from Fluendo.
# See "LICENSE.Elisa" in the root directory of this distribution package
# for details on that license.

"""
MediaProvider that can access to Elisa's media db
"""


__maintainer__ = 'Lionel Martin <lionel@fluendo.com>'

from elisa.base_components.media_provider import MediaProvider
from elisa.core.media_uri import MediaUri
from elisa.core import common
from elisa.core.utils import deferred_action

from twisted.internet import defer, threads


class ElisaMedia(MediaProvider):
    """
    This class implements elisa metadata-like fs support
    """

    def __init__(self):
        MediaProvider.__init__(self)
        self._def_action = deferred_action.DeferredActionsManager()

    def scannable_uri_schemes__get(self):
        return {}

    def supported_uri_schemes__get(self):
        return { 'elisa': 0 }

    def blocking_get_media_type(self, uri): 
        # ElisaMedia always deal with virtual directories
        return { 'file_type' : 'directory', 'mime_type' : '' }

    def is_directory(self, uri):
        return self._def_action.insert_action(0, self.blocking_has_children_with_types, uri, ('audio'))


    def has_children_with_types(self, uri, media_types):
        return self._def_action.insert_action(0, self.blocking_has_children_with_types, uri, media_types)

    def blocking_has_children_with_types(self, uri, media_types):
        """
        Detect whether the given uri has children for given media
        types which can be of 'directory', 'audio', 'video',
        'picture'. Implies the URI is a directory as well.

        @param uri:         the URI to scan
        @type uri:          L{elisa.core.media_uri.MediaUri}
        @param media_types: the media_types to look for on the directory
        @type media_types:  list of strings
        @rtype:             L{twisted.internet.defer.Deferred}
        """
        if 'directory' in media_types or 'audio' in media_types:
            db = common.application.media_manager.media_db
            return db.has_children(uri)

        return False

    def blocking_get_direct_children(self, uri, children_with_info):
        """
        retrieve the children of uri and fills list

        @param uri:                     the URI to analyze
        @type uri:                      L{elisa.core.media_uri.MediaUri}
        @param list_of_children:        List where the children will be appended
        @type list_of_children:         list
        """
        db = common.application.media_manager.media_db
        list_of_children = db.get_uris_by_meta_uri(uri, children_with_info)
        return children_with_info


    def get_direct_children(self, uri, children_with_info):
        return self._def_action.insert_action(0, self.blocking_get_direct_children,
                                              uri, children_with_info)


    def next_location(self, uri, root=None):
        return self._def_action.insert_action(0, self.blocking_next_location, uri, root)

    def blocking_next_location(self, uri, root):
        db = common.application.media_manager.media_db
        return db.get_next_location(uri, root)

    def uri_is_monitorable(self, uri):
        return False

