package de.lmu.ifi.dbs.elki.utilities.optionhandling.parameters;

/*
 This file is part of ELKI:
 Environment for Developing KDD-Applications Supported by Index-Structures

 Copyright (C) 2014
 Ludwig-Maximilians-Universität München
 Lehr- und Forschungseinheit für Datenbanksysteme
 ELKI Development Team

 This program is free software: you can redistribute it and/or modify
 it under the terms of the GNU Affero General Public License as published by
 the Free Software Foundation, either version 3 of the License, or
 (at your option) any later version.

 This program is distributed in the hope that it will be useful,
 but WITHOUT ANY WARRANTY; without even the implied warranty of
 MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 GNU Affero General Public License for more details.

 You should have received a copy of the GNU Affero General Public License
 along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */

import gnu.trove.list.array.TDoubleArrayList;

import java.util.ArrayList;
import java.util.Iterator;
import java.util.List;

import de.lmu.ifi.dbs.elki.math.linearalgebra.Vector;
import de.lmu.ifi.dbs.elki.utilities.FormatUtil;
import de.lmu.ifi.dbs.elki.utilities.optionhandling.OptionID;
import de.lmu.ifi.dbs.elki.utilities.optionhandling.ParameterException;
import de.lmu.ifi.dbs.elki.utilities.optionhandling.WrongParameterValueException;
import de.lmu.ifi.dbs.elki.utilities.optionhandling.constraints.ParameterConstraint;

/**
 * Parameter class for a parameter specifying a list of vectors.
 * 
 * @author Steffi Wanka
 * @author Erich Schubert
 */
public class VectorListParameter extends ListParameter<VectorListParameter, Vector> {
  /**
   * Constructs a vector list parameter with the given name and description.
   * 
   * @param optionID Option ID
   * @param constraint Constraint
   * @param defaultValue Default value
   */
  public VectorListParameter(OptionID optionID, ParameterConstraint<List<Vector>> constraint, List<Vector> defaultValue) {
    super(optionID, defaultValue);
    addConstraint(constraint);
  }

  /**
   * Constructs a vector list parameter with the given name and description.
   * 
   * @param optionID Option ID
   * @param constraint Constraint
   * @param optional Optional flag
   */
  public VectorListParameter(OptionID optionID, ParameterConstraint<List<Vector>> constraint, boolean optional) {
    super(optionID, optional);
    addConstraint(constraint);
  }

  /**
   * Constructs a vector list parameter with the given name and description.
   * 
   * @param optionID Option ID
   * @param constraint Constraint
   */
  public VectorListParameter(OptionID optionID, ParameterConstraint<List<Vector>> constraint) {
    super(optionID);
    addConstraint(constraint);
  }

  /**
   * Constructs a vector list parameter with the given name and description.
   * 
   * @param optionID Option ID
   * @param defaultValue Default value
   */
  // Indiscernible from optionID, constraints
  /*
   * public VectorListParameter(OptionID optionID, List<Vector> defaultValue) {
   * super(optionID, defaultValue); }
   */

  /**
   * Constructs a vector list parameter with the given name and description.
   * 
   * @param optionID Option ID
   * @param optional Optional flag
   */
  public VectorListParameter(OptionID optionID, boolean optional) {
    super(optionID, optional);
  }

  /**
   * Constructs a vector list parameter with the given name and description.
   * 
   * @param optionID Option ID
   */
  public VectorListParameter(OptionID optionID) {
    super(optionID);
  }

  @Override
  public String getValueAsString() {
    StringBuilder buf = new StringBuilder();
    List<Vector> val = getValue();
    Iterator<Vector> valiter = val.iterator();
    while(valiter.hasNext()) {
      buf.append(FormatUtil.format(valiter.next().getArrayRef(), LIST_SEP));
      // Append separation character
      if(valiter.hasNext()) {
        buf.append(VECTOR_SEP);
      }
    }
    return buf.toString();
  }

  @SuppressWarnings("unchecked")
  @Override
  protected List<Vector> parseValue(Object obj) throws ParameterException {
    try {
      List<?> l = List.class.cast(obj);
      // do extra validation:
      for(Object o : l) {
        List<?> v = List.class.cast(o);
        for(Object c : v) {
          if(!(c instanceof Double)) {
            throw new WrongParameterValueException("Wrong parameter format for parameter \"" + getName() + "\". Given list contains objects of different type!");
          }
        }
      }
      // TODO: can we use reflection to get extra checks?
      // TODO: Should we copy the list and vectors?
      return (List<Vector>) l;
    }
    catch(ClassCastException e) {
      // continue with other attempts.
    }
    if(obj instanceof String) {
      String[] vectors = VECTOR_SPLIT.split((String) obj);
      if(vectors.length == 0) {
        throw new WrongParameterValueException("Wrong parameter format! Given list of vectors for parameter \"" + getName() + "\" is empty!");
      }
      ArrayList<Vector> vecs = new ArrayList<>();

      TDoubleArrayList vectorCoord = new TDoubleArrayList();
      for(String vector : vectors) {
        vectorCoord.clear();
        String[] coordinates = SPLIT.split(vector);
        for(String coordinate : coordinates) {
          try {
            vectorCoord.add(FormatUtil.parseDouble(coordinate));
          }
          catch(NumberFormatException e) {
            throw new WrongParameterValueException("Wrong parameter format! Coordinates of vector \"" + vector + "\" are not valid!");
          }
        }
        vecs.add(new Vector(vectorCoord.toArray()));
      }
      return vecs;
    }
    throw new WrongParameterValueException("Wrong parameter format! Parameter \"" + getName() + "\" requires a list of double values!");
  }

  /**
   * Returns a string representation of the parameter's type.
   * 
   * @return 
   *         &quot;&lt;double_11,...,double_1n:...:double_m1,...,double_mn&gt;&quot
   *         ;
   */
  @Override
  public String getSyntax() {
    return "<double_11,...,double_1n:...:double_m1,...,double_mn>";
  }
}
