/* 
   elmo - ELectronic Mail Operator

   Copyright (C) 2002, 2003 rzyjontko

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; version 2.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software Foundation,
   Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.  

   ----------------------------------------------------------------------

   status line interface
   
*/
/****************************************************************************
 *    IMPLEMENTATION HEADERS
 ****************************************************************************/

#include <string.h>

#include "ecurses.h"
#include "xmalloc.h"
#include "status.h"
#include "gettext.h"
#include "color.h"
#include "interface.h"
#include "ask.h"
#include "str.h"

/****************************************************************************
 *    IMPLEMENTATION PRIVATE DEFINITIONS / ENUMERATIONS / SIMPLE TYPEDEFS
 ****************************************************************************/
/****************************************************************************
 *    IMPLEMENTATION PRIVATE CLASS PROTOTYPES / EXTERNAL CLASS REFERENCES
 ****************************************************************************/
/****************************************************************************
 *    IMPLEMENTATION PRIVATE STRUCTURES / UTILITY CLASSES
 ****************************************************************************/
/****************************************************************************
 *    IMPLEMENTATION REQUIRED EXTERNAL REFERENCES (AVOID)
 ****************************************************************************/
/****************************************************************************
 *    IMPLEMENTATION PRIVATE DATA
 ****************************************************************************/

/* Statusbar window. */
static WINDOW *status_win = NULL;


/* Attributes used to display status information. */
static chtype text_color;
static chtype paren_color;
static chtype percent_color;
static chtype info_color;


/* Used to display current status. */
static str_t *status_text = NULL;

/* Used to display current window information. */
static str_t *current_window = NULL;

/* Used to display percent information. */
static str_t *percents = NULL;

/* Used to display additional information. */
static str_t *add_info = NULL;

/****************************************************************************
 *    INTERFACE DATA
 ****************************************************************************/
/****************************************************************************
 *    IMPLEMENTATION PRIVATE FUNCTION PROTOTYPES
 ****************************************************************************/
/****************************************************************************
 *    IMPLEMENTATION PRIVATE FUNCTIONS
 ****************************************************************************/

static void
change_me (void)
{
        int maxlen, nothing;

        getmaxyx (status_win, nothing, maxlen);
        
        window_mvaddch (status_win, 0, 0, ' ');
        maxlen--;
        
        if (current_window){
                wattrset (status_win, paren_color);
                window_addch (status_win, '(');
                wattrset (status_win, text_color);
                maxlen -= window_addstr (status_win, current_window->str);
                wattrset (status_win, paren_color);
                window_addch (status_win, ')');
                maxlen -= 2;
        }

        if (percents){
                wattrset (status_win, percent_color);
                maxlen -= window_addstr (status_win, percents->str);
        }

        if (status_text){
                wattrset (status_win, text_color);
                maxlen -= window_addstr (status_win, status_text->str);
        }
        
        if (add_info){
                wattrset (status_win, info_color);
                maxlen -= window_addstr (status_win, add_info->str);
        }

        wattrset (status_win, text_color);
        while (maxlen-- > 0)
                window_addch (status_win, ' ');

        wnoutrefresh (status_win);
}

/* This file is generated by interface.pl script from interface.desc,
   and inc.in. */
static WINDOW *interface_init (void);
#include "status.inc"

/****************************************************************************
 *    INTERFACE FUNCTIONS
 ****************************************************************************/

void
status_start (void)
{
        status_win  = interface_init ();
        status_text = str_create_size (COLS + 1);

        str_put_string (status_text, _("   initializing..."));
        change_me ();

        touchwin (status_win);
        wnoutrefresh (status_win);
}



void
status_free_resources (void)
{
        if (status_win)
                delwin (status_win);
        status_win = NULL;
        
        if (status_text)
                str_destroy (status_text);
        status_text = NULL;

        if (current_window)
                str_destroy (current_window);
        current_window = NULL;

        if (percents)
                str_destroy (percents);
        percents = NULL;

        if (add_info)
                str_destroy (add_info);
        add_info = NULL;
}




void
status_put_mailbox (const char *name, int total, int old, int new)
{
        str_clear (status_text);
        str_sprintf (status_text, _("   %s  [msgs:%d"), name, total);
        
        if (old)
                str_sprintf (status_text, _("  old:%d"), old);
        if (new)
                str_sprintf (status_text, _("  new:%d"), new);
        str_put_char (status_text, ']');
        change_me ();
}



void
status_put_mailinfo (const char *from, const char *subject)
{
        str_clear (status_text);
        str_put_string_len (status_text, "   ", 3);

        if (from)
                str_put_string (status_text, from);

        str_put_string_len (status_text, "      ", 6);

        if (subject)
                str_put_string (status_text, subject);

        change_me ();
}



void
status_refresh (void)
{
        change_me ();
}



void
status_switch_window (const char *str, int num)
{
        if (current_window == NULL)
                current_window = str_create ();
        else
                str_clear (current_window);

        str_sprintf (current_window, "%s/%d", str, num);
        change_me ();
}


void
status_show_percentage (int p)
{
        if (percents == NULL)
                percents = str_create ();
        else
                str_clear (percents);
        
        if (p == -1)
                str_put_string (percents, "  --All--");
        else if (p == 0)
                str_put_string (percents, "  --Top--");
        else if (p == 100)
                str_put_string (percents, "  --Bot--");
        else
                str_sprintf (percents, "  --%2d%%--", p);
        change_me ();
}


void
status_add_info (const char *fmt)
{
        if (add_info == NULL)
                add_info = str_create ();
        else
                str_clear (add_info);

        str_sprintf (add_info, "%s", fmt);
        change_me ();
}


/****************************************************************************
 *    INTERFACE CLASS BODIES
 ****************************************************************************/
/****************************************************************************
 *
 *    END MODULE status.c
 *
 ****************************************************************************/
