# This file is part of Email-Reminder.
#
# Email-Reminder is free software; you can redistribute it and/or
# modify it under the terms of the GNU General Public License as
# published by the Free Software Foundation; either version 3 of the
# License, or (at your option) any later version.
#
# Email-Reminder is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
# General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with Email-Reminder; if not, write to the Free Software
# Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA
# 02110-1301, USA.

package EmailReminder::WeeklyEvent;

use strict;
use warnings;

use Date::Manip;
use POSIX;
use Scalar::Util;

use EmailReminder::Event;
use EmailReminder::Utils;

require Exporter;
our @ISA    = ("EmailReminder::Event", "Exporter");
#our @EXPORT = qw(get_name valid_day);

# XML tags, attributes and values
my $DAY_TAG   = 'day';
my $NAME_TAG  = 'name';

# Global date variables
my $current_time = ParseDate("now");
my $current_date = ParseDate(UnixDate($current_time, "\%x"));
my $current_dayofweek = UnixDate($current_time, "\%w");

# Hard-coded value for this event's type (class method)
sub get_type
{
    return 'weekly';
}

# Number of fields this event adds to its parent (class method)
sub get_nb_fields
{
    return EmailReminder::Event->get_nb_fields() + 2;
}

sub get_name
{
    my ($this) = @_;
    my $name  = EmailReminder::Utils::get_node_value($this->{XML_NODE}, $NAME_TAG);
    return $name;
}

sub set_name
{
    my ($this, $new_value) = @_;
    return EmailReminder::Utils::set_node_value($this->{XML_NODE}, $NAME_TAG, $new_value);
}

sub valid_day
{
    my ($new_value) = @_;

    if (!Scalar::Util::looks_like_number($new_value)) {
        # Try to parse as a string
        my $day = UnixDate(ParseDate($new_value), "\%w");
        if ($day) {
            $new_value = $day;
        } else {
            $new_value = 7; # Default: Sunday
        }
    }

    if ($new_value > 7 or $new_value < 1) {
        # Default to Sunday for out of range dates (since zero is
        # both 0 and 7).
        $new_value = 7;
    }

    return $new_value;
}

sub get_day
{
    my ($this) = @_;
    my $day = EmailReminder::Utils::get_node_value($this->{XML_NODE}, $DAY_TAG);
    return valid_day($day);
}

sub set_day
{
    my ($this, $new_value) = @_;
    return EmailReminder::Utils::set_node_value($this->{XML_NODE}, $DAY_TAG, valid_day($new_value));
}

sub get_message_body
{
    my $this = shift;

    # destination user
    my $first_name = shift;

    # event details
    my $when      = $this->{"WHEN"};
    my $name      = $this->get_name();
    
    my $message = <<MESSAGEEND;
Subject: $name

Hi $first_name,

I just want to remind you of the following event $when:

$name
MESSAGEEND

    return $message;
}

# Returns 1 if the event will occur in X days (X is a param)
sub will_occur
{
    my $this = shift;
    my $modifier = shift;
    
    # Apply the modifier to the event date
    my $modified_day = $this->get_day();
    return 0 unless $modified_day;

    if ($modifier) {
        $modified_day -= $modifier;
        while ($modified_day > 7) {
            $modified_day -= 7;
        }
        while ($modified_day < 1) {
            $modified_day += 7;
        }
    }
    
    if ($current_dayofweek == $modified_day) {
        return 1;
    } else {
        return 0;
    }
}

1;
