#!/usr/bin/perl

=head1 NAME

grip-overridereplace.pl - refresh a package to enact an override

=cut

use strict;
use warnings;
use File::Basename;
use Debian::Packages::Compare;

use vars qw/ %tasks %overrides $uri $deb $z @list $suite $dir 
 $prog $our_version $base $grip_name $skip $file @archlist
 %matches $data $file $from $to $package $arch $name /;
 
=head1 Synopsis

 grip-overridereplace.pl -s|--suite STRING -b|--base-path PATH [--grip-name STRING] BINARY
 grip-overridereplace.pl -?|-h|--help|--version

 Commands:
 -s|--suite STRING:        Name of the distribution to override [required]
 -b|--base-path PATH:      path to the top level repository directory [required]

 -?|-h|--help|--version:   print this help message and exit

 Options:
   --grip-name STRING:     alternative name for the grip repository

Only the specified binary package will be affected, each architecture
in turn. Packages are copied out of pool/ into a temporary directory,
removed and then replaced into the architecture concerned.

Overrides take place in the Packages file, not within the binary package
itself - check the results by parsing the relevant Packages file, not
using the output of dpkg -I $deb or other .deb tools.

Note that overrides will need to be enacted for the versions in testing
as well, so repeat the process once you are happy with the effects.

=cut

$prog = basename ($0);
$our_version = "0.1.0";
$grip_name = "grip";

while( @ARGV ) {
	$_= shift( @ARGV );
	last if m/^--$/;
	if (!/^-/) {
		unshift(@ARGV,$_);
		last;
	}
	elsif (/^(-\?|-h|--help|--version)$/) {
		&usageversion();
		exit (0);
	}
	elsif (/^(-b|--base-path)$/) {
		$base = shift;
	}
	elsif (/^(-s|--suite)$/) {
		$suite = shift;
	}
	elsif (/^(--grip-name)$/) {
		$grip_name = shift;
	}
	else {
		die "$prog: Unknown option $_.\n";
	}
}

die "$prog: ERR: Please specify an existing directory for the base-path.\n"
	if (not defined $base);

$base .= '/' if ("$base" !~ m:/$:);
die "$prog: ERR: Please specify an existing directory for the base-path: $base\n"
	if (not -d $base);

die ("$prog: ERR: Specify a distribution name, not a codename. e.g. testing, not lenny.\n")
	if ((not defined $suite) or 
	($suite =~ /etch|sid|lenny|squeeze/));

die ("$prog: ERR: Cannot find Grip configuration directory.\n")
	if (not -d "${base}${grip_name}/conf/");

=head1 Description

Overrides need to be updated from time to time so this script provides
a way to implement overrides restrospectively.

=cut

&set_base($base);

my $a = &get_archlist ($suite, $grip_name);
die ("Unable to obtain list of supported architectures.\n")
	if (not defined $a);

$dir = `mktemp -t -d gripover.XXXXXX`;
chomp ($dir);
$package = $ARGV[0];

foreach $arch (@$a)
{
	# always leave source intact
	next if ($arch eq 'source');
	$data = &get_single_package ($suite, $grip_name, $package, $arch);
	if (not defined $data->{'Filename'})
	{
		print "$prog: ERR: Cannot find package '$package' on $arch\n";
		next;
	}
	$file = $data->{'Filename'};
	$name = $data->{'Package'};
	if ( -f "${base}${grip_name}/$file")
	{
		$from = "${base}${grip_name}/$file";
		$to = basename($file);
		chomp ($to);
		open (FROM, "$from") or die ("Cannot read $from: $!\n");
		my @f=<FROM>;
		close (FROM);
		open (TO, ">$dir/$to") or die ("Cannot write to $dir/$to: $!\n");
		print TO @f;
		close (TO);
	}
	if ( -f "$dir/$to")
	{
		print "Replacing $to for $arch into $suite.\n";
		print "reprepro -v -A $arch -b ${base}${grip_name} remove $suite $name\n";
		system ("reprepro -v -A $arch -b ${base}${grip_name} remove $suite $name");
		print "reprepro -v -A $arch -b ${base}${grip_name} includedeb $suite $dir/$to\n";
		system ("reprepro -v -A $arch -b ${base}${grip_name} includedeb $suite $dir/$to");
		unlink ("$dir/$to");
	}
}

rmdir ($dir);
exit 0;

sub usageversion
{
	print(STDERR <<END)
$prog -s|--suite STRING -b|--base-path PATH [--grip-name STRING] BINARY
$prog -?|-h|--help|--version

Commands:
-s|--suite STRING:        Name of the distribution to override [required]
-b|--base-path PATH:      path to the top level repository directory [required]

-?|-h|--help|--version:   print this help message and exit

Options:
  --grip-name STRING:     alternative name for the grip repository

Only the specified binary package will be affected, each architecture
in turn. Packages are copied out of pool/ into a temporary directory,
removed and then replaced into the architecture concerned.

Overrides take place in the Packages file, not within the binary package
itself - check the results by parsing the relevant Packages file, not
using the output of dpkg -I $deb or other .deb tools.

Note that overrides will need to be enacted for the versions in testing
as well, so repeat the process once you are happy with the effects.

END
	or die "$0: failed to write usage: $!\n";
}

=head1 Copyright and Licence

 Copyright (C) 2007-2009  Neil Williams <codehelp@debian.org>

 This package is free software; you can redistribute it and/or modify
 it under the terms of the GNU General Public License as published by
 the Free Software Foundation; either version 3 of the License, or
 (at your option) any later version.
 
 This program is distributed in the hope that it will be useful,
 but WITHOUT ANY WARRANTY; without even the implied warranty of
 MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 GNU General Public License for more details.
 
 You should have received a copy of the GNU General Public License
 along with this program.  If not, see <http://www.gnu.org/licenses/>.

=cut
