--- linux-2.5.7/drivers/block/Config.in.pre-enbd	Wed Apr 10 00:09:43 2002
+++ linux-2.5.7/drivers/block/Config.in	Tue Apr  9 23:58:20 2002
@@ -40,6 +40,7 @@
 
 tristate 'Loopback device support' CONFIG_BLK_DEV_LOOP
 dep_tristate 'Network block device support' CONFIG_BLK_DEV_NBD $CONFIG_NET
+dep_tristate 'Enhanced network block device support' CONFIG_BLK_DEV_ENBD $CONFIG_NET
 
 tristate 'RAM disk support' CONFIG_BLK_DEV_RAM
 if [ "$CONFIG_BLK_DEV_RAM" = "y" -o "$CONFIG_BLK_DEV_RAM" = "m" ]; then
--- linux-2.5.7/drivers/block/Config.help.pre-enbd	Mon Mar 18 21:37:02 2002
+++ linux-2.5.7/drivers/block/Config.help	Wed Apr 10 00:08:41 2002
@@ -116,6 +116,28 @@
 
   If unsure, say N.
 
+CONFIG_BLK_DEV_ENBD
+  Saying Y here will build in support for the "enhanced network 
+  block device".  Like the plain NBD, this device proxies a remote hard
+  disk or other block device, such as a cdrom or floppy.  The difference
+  between ENBD and NBD is that ENBD is a much more heavyweight solution
+  for an industrial setting - it does automatic reconnects after network
+  brownouts, and uses multiple channels at once to carry the data.  It
+  supports remote ioctls, removable devices, and uses an MD5 sampling
+  technique to accelerate softRAID resyncs.  It will connect through
+  (secure) SSL channels.
+
+  You will need the userspace daemons, which packages are mirrored
+  on several places on the net. The primary source is 
+  ftp://oboe.it.uc3m.es/pub/Programs/nbd/nbd-2.4.28.tgz.
+
+  If you want to compile this driver as a module ( = code which can be
+  inserted in and removed from the running kernel whenever you want),
+  say M here and read <file:Documentation/modules.txt>. The module
+  will be called enbd.o.
+
+  If unsure, say N.
+
 CONFIG_BLK_DEV_XD
   Very old 8 bit hard disk controllers used in the IBM XT computer
   will be supported if you say Y here.
--- linux-2.5.7/drivers/block/Makefile.pre-enbd	Mon Mar 18 21:37:18 2002
+++ linux-2.5.7/drivers/block/Makefile	Wed Apr 10 00:13:49 2002
@@ -31,6 +31,7 @@
 obj-$(CONFIG_BLK_DEV_DAC960)	+= DAC960.o
 
 obj-$(CONFIG_BLK_DEV_NBD)	+= nbd.o
+obj-$(CONFIG_BLK_DEV_ENBD)	+= enbd.o
 
 subdir-$(CONFIG_PARIDE) += paride
 
--- linux-2.5.7/drivers/block/enbd.c.pre-enbd	Wed Apr 10 00:30:45 2002
+++ linux-2.5.7/drivers/block/enbd.c	Wed Apr 10 00:20:57 2002
@@ -0,0 +1,4608 @@
+/*
+ * (Enhanced) Network block device - make block devices work over TCP
+ *
+ * Original NBD Copyright 1997 Pavel Machek <pavel@elf.mj.gts.cz>
+ * Further ENBD Copyrights 1998, 1999, 2000 Peter Breuer <ptb@it.uc3m.es>
+ *
+ *
+ *
+ * ATTENTION: You need the userspace daemons available from
+ *            ftp://oboe.it.uc3m.es/pub/Programs/nbd-2.4.*.tgz
+ *            and then ENBD project on http://freshmeat.net
+ *
+ *
+ *
+ * Development of the ENBD software has been supported by grants and
+ * contributions from Realm Information Technologies, Inc. of 5555
+ * Oakbrook Parkway, NW Norcross, GA and iNsu Innovations Inc.  of
+ * 3465, Boulevard Thimens, Saint-Laurent, Quebec, Canada.
+ * 
+ * ------------ Pavel's history notes ----------------------------------
+ * 97-3-25 compiled 0-th version, not yet tested it 
+ *   (it did not work, BTW) (later that day) HEY! it works!
+ *   (bit later) hmm, not that much... 2:00am next day:
+ *   yes, it works, but it gives something like 50kB/sec
+ * 97-3-28 it's completely strange - when using 1024 byte "packets"
+ *   it gives 50kB/sec and CPU idle; with 2048 bytes it gives
+ *   500kB/sec (and CPU loaded 100% as it should be) (all done
+ *   against localhost)
+ * 97-4-1 complete rewrite to make it possible for many requests at 
+ *   once to be processed
+ * 97-4-1 23:57 rewrite once again to make it work :-(
+ * 97-4-3 00:02 hmm, it does not work.
+ * 97-4-3 23:06 hmm, it will need one more rewrite :-)
+ * 97-4-10 It looks like it's working and stable. But I still do not
+ *  have any recovery from lost connection...
+ * (setq tab-width 4)
+ * 97-4-11 Making protocol independent of endianity etc.
+ * 97-4-15 Probably one more rewrite, since it loses requests under
+ *  heavy loads
+ * 97-9-13 Cosmetic changes
+ *
+ * possible FIXME: make set_sock / set_blksize / set_size / do_it one syscall
+ * why not: would need verify_area and friends, would share yet another 
+ *          structure with userland
+ *
+ * FIXME: not module-safe
+ *
+ * ------------ Peter's history notes ----------------------------------
+ * 98-12-18 modules now OK ptb@it.uc3m.es (Peter Breuer) ported to
+ * 2.0.*. + better debugging. Still possible lockup in connection with APM
+ * and spurious interrupt - only on write. Error treatment should
+ * be improved. After 100 errors from end_request the kernel can
+ * do anything. We should catch it ourselves.
+ * 99-1-sometime fixed lockup by extending semaphore - ptb v1.0.
+ * 99-3-sometime reconnect protocol (client mod agreed by pavel) - ptb v1.1
+ * 99-4-25 add /proc/nbdinfo - ptb v1.1.1
+ * 99-4-sometime add multiplex - ptb v1.2
+ * 99-4-26 fix multiplex and redundancy - ptb v1.2.1
+ * 99-4-29 reentrant client threads - ptb v1.2.2
+ * 99-4-29 socket related stuff placed in user space - amarin v1.3.0
+ * 99-5-3  fix all, all writes had to be before all reads - ptb v1.2.4
+ * 99-5-5  fix out-of-order, async - ptb v1.2.5
+ * 99-5-7  semaphores removed (still works!), fail cases corrected - ptb v1.2.6
+ * 99-5-12 signals unblocked in xmit, blksize != 1024 fixed, ioctls
+ *         added  - ptb v1.2.7
+ * 99-6-1  interaction with client split into two functions - amarin v1.3.0
+ * 99-6-3  reintegrated fully, mem manager fixed, accounts fixed - ptb v1.2.8.3
+ * 99-6-3  extra queue removed, mem manager removed  - ptb v1.2.8.4
+ * 99-7-3  buffer registration introduced - ptb v1.2.8.5
+ * 99-7-3  some client redundancy reestablished - ptb v2.1.1
+ * 99-7-10 encapsulated queue calls. One element rollback buffer - ptb v2.1.2
+ * 99-7-20 timestamp and rollback old abandoned request - ptb v2.1.3
+ * 99-7-24 64bit file sizes and offsets accepted - ptb v2.1.9
+ * 99-7-26 experimental request coalesces - ptb v2.1.10
+ * 99-7-27 partitioning scheme - ptb v2.2.1
+ * 99-8-3  enbd_clr_sock bug in invalidate_device fixed? - ptb v2.2.4
+ * 99-8-5  reverse replace of block_fsync, add sig ioctls - ptb v2.2.5
+ *         reverse bug introduced about v2.2.3 for compound reqs - ptb v2.2.5
+ *         fix clear_que bug (didn't rollback first) from 2.1.3 - ptb v2.2.5
+ * 99-8-22 workaround strange nr_sectors bug - ptb v2.2.6
+ * 99-8-11 fix MY_NBD_SYNC bug. Never sync'ed all - ptb v2.2.7
+ * 99-8-12 wakeups all moved to enqueue - ptb v2.2.7
+ * 99-8-23 remove slot->cli_age - ptb v2.2.7
+ * 99-8-24 first 8 bytes of signature embedded in packets - ptb v2.2.8
+ *         fix SET_SIG define buglet, remove hardcoded constants - ptb v2.2.8
+ *         fix huge bug. Missing copy_fromfs in my_nbd_ack - ptb v2.2.8     
+ *         removed signature embedding and all other decorations - ptb v2.2.8
+ * 99-8-25 recast fix in my_nbd_ack to avoid align. bug - ptb v2.2.9
+ *         put in MKDEVs and put back some hardcode const fixes - ptb v2.2.10
+ * 99-9-29 fix BLKGETSIZE bug - ptb v2.2.14
+ * 99-10-2 run with interrupts on throughout. Think we lose some - ptb v2.2.15
+ * 99-10-8 trim dead code, kernel 2.2 ifdef's - ptb v2.2.17
+ * 99-12-18 further o-o - ptb v2.2.19
+ * 99-12-28 queue account cleanup. endio on queue reqs at reset - ptb v2.2.20
+ *          interruptible semaphores for better client recovery - ptb v2.2.20
+ * 00-1-2   debugging cleanups. Fix race in end_request - ptb v2.2.21
+ * 00-1-4   semaphores simplified. - ptb v2.2.22
+ * 00-6-8   emergency control by write to proc - ptb v2.2.24
+ * 00-7-20  ported to 2.4.0-test1. Possible minor bugs found/fixed - ptb v2.2.24
+ * 00-7-27  changed proc i/f to read_proc from get_info in 2.2/2.4 - ptb v2.2.25
+ * 00-7-30  fixed reads before writes under 2.4 by disabling merge - ptb v2.2.25
+ * 00-7-30  and fixed merge_reqs for 2.4, now that I understand! - ptb v2.2.25
+ * 00-7-30  fixed/introduced possible bug in end_io  for 2.2/2.4 - ptb v2.2.25
+ * 00-7-30 added timeval/zone field in requests and replies - ptb v2.4.0
+ * 00-7-30 fixed hitherto masked bug in read_stat in enbd_client - ptb v2.4.0
+ * 00-7-30 added timeout to net writes in enbd_client - ptb v2.4.0
+ * 00-8-20 display fix for devices over 2GB - ptb v2.4.5
+ * 00-8-23 more 64 bit fixes + error out overrange requests- ptb v2.4.6/2.2.27
+ * 00-8-31 add ENBD_ERR ioctl to error out slot request- ptb v2.4.9
+ * 00-8-31 soften ENBD_SOFT_RESET so doesn't wreck protocol - ptb v2.4.9
+ * 00-9-1  remove %L's from printfs. Kernel 2.2. doesn't - ptb v2.4.10/2.2.27
+ * 00-9-6  add various state flags to help init order - ptb v2.4.11
+ * 00-9-8  add checks for device initialised to set_sock - ptb v2.4.12
+ * 00-9-17 en/disable device as aslot count goes through 0 - ptb v2.4.13/2.2.28
+ * 00-9-21 split read/write dev req counts for accounting - ptb v2.4.14
+ * 00-9-21 renamed sync_intvl to req_timeo - ptb v2.4.14
+ * 00-9-21 made sync_intvl count write blocks - ptb v2.4.14
+ * 00-9-22 repair enable after delayed disable when disabled - ptb v2.4.14
+ * 00-9-22 include sync (nonblocking) after sync_intvl reqs - ptb v2.4.14
+ * 00-9-25 disable sync (nonblocking) after sync_intvl reqs - ptb v2.4.14
+ * 00-9-25 bundle invalidate_buffers in clr_sock - ptb v2.4.14
+ * 00-10-20 implement req_timeo per device + ioctl (Wang Gang) - ptb v2.4.15
+ * 00-10-20 add raid mode (Wang Gang) - ptb v2.4.15
+ * 00-10-26 throttle in do_req  - ptb v2.4.15
+ * 00-10-28 do set_sock on first open and clr_sock on last close - ptb v2.4.15
+ * 00-11-01 make sync_intvl really sync - ptb v2.4.15
+ * 00-11-14 rename throttle to plug, enbd_sync takes arg - ptb v2.4.17
+ * 00-11-19 clr_sock errs req not rollback if show_errs & !aslot - ptb v2.4.17
+ * 00-11-20 removed autodeadlock when disabled in do_req end_req - ptb v2.4.17
+ * 00-11-21 make MY_NBD_SYNC only sync when sync_intvl > 0 - ptb v2.4.17
+ * 00-12-24 make MY_NBD_GET_REQ use a timeout arg - ptb v2.4.18
+ * 01-02-12 ported to 2.4.0 (works). do_nbd_request rewritten - ptb v2.4.20
+ * 01-02-20 managed to get plugging and clustered read/writes OK - ptb v2.4.21
+ * 01-02-21 eliminated slot->buflen for the time being - ptb v2.4.21
+ * 01-02-27 added proper devfs support - ptb v2.4.22
+ * 01-03-15 allowed more devices/in devfs, cleaned up endio - ptb v2.4.23
+ * 01-03-15 added device letter (<= 3 chars) to struct-  - ptb v2.4.23
+ * 01-03-15 added request size check to do_nbd_req - ptb v2.4.23
+ * 01-03-15 increased MAX_SECTORS to 512 by default - ptb v2.4.23
+ * 01-03-15 made major number a module parameter - ptb v2.4.23
+ * 01-03-18 added max_sectors array - ptb v2.4.23
+ * 01-03-23 added devfs links - ptb v2.4.23
+ * 01-04-17 plugging always enabled for 2.4 kernels - ptb v2.4.24
+ * 01-04-17 made SET_RO set_device_ro as well as set local flags - ptb v2.4.25
+ * 01-04-28 impl SET_MD5SUM ioctl and proc support for md5sum - ptb v2.4.25
+ * 01-04-29 added accounting for md5'd reqs - ptb v2.4.25
+ * 01-07-29 added atomic protections for accounting - ptb v2.4.25
+ * 01-08-01 fixed 2.4 smp bugs. Interrupts off in spinlocks - ptb v2.4.25
+ * 01-08-01 removed all semaphores for spinlocks - ptb v2.4.25
+ * 01-08-01 invalidate_buffers in clr_sock (req'd Rogier Wolff) - ptb v2.4.25
+ * 01-08-02 fixed smp deadlock - end_that_request_first slept! ptb v2.4.26
+ * 01-10-16 provisionally added error in device open when notenabled ptb v2.4.27
+ * 01-10-18 added DIRTY flag to save on repeated invalidate_buffers ptb v2.4.27
+ * 01-10-31 increment seqno_out before delivery, so really starts at 1  v2.4.27
+ * 01-11-01 move zeroing of seqno in cmd field to nbe_end_req* ptb v2.4.27
+ * 01-11-18 add speed calculation, dev fields, display in proc ptb v2.4.27
+ * 01-11-20 modifications for compiling into monolithic kernel ptb v2.4.27
+ * 01-12-06 clr requests before reenabling, not after, in enbd_enable ptb 2.4.27
+ * 02-02-21 make enbd_rollback modal, absirbing enbd_error ptb 2.4.27
+ */
+
+#include <linux/major.h>
+#ifndef UNIX98_PTY_MAJOR_COUNT
+  #define UNIX98_PTY_MAJOR_COUNT 8
+  #ifndef UNIX98_NR_MAJORS
+    #define UNIX98_NR_MAJORS=UNIX98_PTY_MAJOR_COUNT
+  #endif
+#endif
+
+#include <linux/module.h>
+
+#if defined(__GNUC__) && __GNUC__ >= 2
+#define _LOOSE_KERNEL_NAMES
+#endif
+
+#include <linux/version.h>
+
+#include <linux/fs.h>
+#include <linux/stat.h>
+#include <linux/errno.h>
+#include <asm/segment.h>
+
+#include <asm/uaccess.h>	/* PTB - when did this arrive in kernel? */
+#include <asm/byteorder.h>
+#include <linux/wrapper.h>
+
+#define MAJOR_NR NBD_MAJOR
+static int major = MAJOR_NR;
+
+#include <linux/proc_fs.h>
+#include <linux/genhd.h>
+#include <linux/hdreg.h>
+
+#include <linux/file.h>		/* PTB - when did this arrive in kernel? */
+
+#include <linux/smp_lock.h>
+
+#include <linux/devfs_fs_kernel.h>
+
+#include <linux/sysctl.h>
+#include <linux/init.h>
+#include <linux/kdev_t.h>
+
+/*                                                       *
+ * PTB --------------- compatibility ------------------- *
+ *                   layer starts here.                  *
+ */
+
+  // BH_Protected disappeared somewhere around 2.4.10
+  #define mark_buffer_protected(rbh) \
+      { \
+  	mark_buffer_dirty (rbh); \
+  	mark_buffer_uptodate (rbh, 1); \
+  	refile_buffer (rbh); \
+       }
+
+  #if defined(__SMP__) || defined(SMP)
+    #if ! defined(CONFIG_SMP)
+      #error CONFIG_SMP not defined
+    #endif
+  #endif
+
+  /* PTB list interface extensions */
+  #define list_head(ptr, type, member) \
+  (list_empty(ptr)?NULL:list_entry(((struct list_head *)ptr)->next,type,member))
+  #define list_tail(ptr, type, member) \
+  (list_empty(ptr)?NULL:list_entry(((struct list_head *)ptr)->prev,type,member))
+
+int linux_version_code = LINUX_VERSION_CODE;
+
+int warning_old_kernel = 0;
+
+/*                                                       *
+ * PTB --------------- compatibility ------------------- *
+ *                   layer ENDS here.                    *
+ */
+
+#include <linux/enbd.h>
+#include <linux/enbd_ioctl.h>
+
+/*
+ * PTB kernel data - 4KB worth
+ * We need space for nda, nda1, .. nda15, ndb, ndb1, ..
+ * The index is exactly the minor number.
+ */
+  static int enbd_blksizes[MAX_NBD * ENBD_MAXCONN];
+  static int enbd_sizes[MAX_NBD * ENBD_MAXCONN];
+  static u64 enbd_bytesizes[MAX_NBD * ENBD_MAXCONN];
+  static int enbd_max_sectors[MAX_NBD * ENBD_MAXCONN];
+
+/*
+ * PTB our data   - about 3KB
+ * These are nda, ndb, ndc, ...
+ * Divide the minor by ENBD_MAXCONN to get this index.
+ */
+  static struct enbd_device enbd_dev[MAX_NBD];
+  static spinlock_t enbd_lock = SPIN_LOCK_UNLOCKED;
+
+/*
+ * PTB User messaging defs.
+ */
+
+  #define ENBD_ID "NBD #%d[%d]: " __FUNCTION__ " "
+
+  #define ENBD_DEBUG(level, s...) \
+  { static int icnt; printk( KERN_DEBUG ENBD_ID, __LINE__, icnt++); printk(s);}
+  #define ENBD_ERROR( s...) \
+  { static int icnt; printk( KERN_ERR   ENBD_ID, __LINE__, icnt++); printk(s);}
+  #define ENBD_ALERT( s...) \
+  { static int icnt; printk( KERN_ALERT ENBD_ID, __LINE__, icnt++); printk(s);}
+  #define ENBD_INFO( s...)  \
+  { static int icnt; printk( KERN_INFO  ENBD_ID, __LINE__, icnt++); printk(s);}
+
+  #define ENBD_FAIL( s ) { \
+    ENBD_DEBUG(1, s " (result %d).\n" , result ); \
+    goto error_out; \
+  }
+  #define ENBD_HARDFAIL( s ) { \
+    ENBD_ERROR( s " (result %d).\n" , result ); \
+    lo->harderror = result; \
+    goto hard_error_out; \
+  }
+
+/*
+ * PTB device parameters. These are module parameters too.
+ */
+
+  static int rahead     = ENBD_RAHEAD_DFLT;/* PTB - read ahead blocks  */
+  static int sync_intvl = ENBD_SYNC_INTVL; /* PTB - sync every n secs/Kreqs */
+  static int speed_lim  = ENBD_SPEED_LIM;  /* PTB - for throttling, in Kw */
+  static int merge_requests               /* PTB - bool, do request coalesce */
+                        = ENBD_MERGE_REQ_DFLT;
+  static int buf_sectors = ENBD_MAX_SECTORS;
+                                          /* PTB - user bufsize required */
+  static int show_errs = 1;	          /* PTB - RAID mode? not usually */
+  static int plug = ENBD_PLUG_DFLT;
+
+  static int md5sum = 0;		  /* PTB - use md5summing write proto */
+  static int md5_on_threshold = 1000;	  /* PTB - reqs reqd to turn md5 on */
+  static int md5_off_threshold = 10;	  /* PTB - errs reqd to turn md5 off */
+  static int enable = 0;		  /* PTB global on/off, not used */
+
+#ifndef NO_BUFFERED_WRITES
+  static int buffer_writes = 0;	          /* PTB - act like ramd on write */
+#endif		/* NO_BUFFERED_WRITES */
+
+#if defined(MODULE)
+  MODULE_PARM (rahead, "i");
+  MODULE_PARM (sync_intvl, "i");
+  MODULE_PARM (speed_lim, "i");
+  MODULE_PARM (merge_requests, "i");
+  MODULE_PARM (buf_sectors, "i");
+  MODULE_PARM (show_errs, "i");
+  MODULE_PARM (plug, "i");
+  #ifndef NO_BUFFERED_WRITES
+    MODULE_PARM (buffer_writes, "i");
+  #endif		/* NO_BUFFERED_WRITES */
+  MODULE_PARM (major, "i");
+  MODULE_PARM (md5sum, "i");
+  MODULE_PARM (md5_on_threshold, "i");
+  MODULE_PARM (md5_off_threshold, "i");
+#endif
+
+#define NO_BUFFERED_WRITES 1
+
+/*                                                       *
+ * PTB --------------- functions ----------------------- *
+ */
+
+/*
+ * PTB count number of blocks in a request. This will be an overestimate
+ * if the number is not an exact multiple. It seems to happen. We 
+ * guarrantee to return -ve only if the request is invalid.
+ *
+ * @req - request we want to count
+ */
+inline long
+nr_blks (struct request *req)
+{
+
+	unsigned log_sectors_per_blk;
+	unsigned sectors_per_blk;
+	int size, minor, nbd;
+	int sectors;
+	struct enbd_device *lo;
+
+	if (!req)
+		return -EINVAL;
+
+	if (major (req->rq_dev) != major) {
+
+		ENBD_ERROR ("req %#x has major %d instead of %d,"
+			   " curr sectors %d, nr sectors %ld\n, buffer %#x",
+			   (unsigned) req, major (req->rq_dev),
+                           major, req->current_nr_sectors, req->nr_sectors,
+			   (unsigned) req->buffer);
+
+		return -EINVAL;
+	}
+
+	minor = minor (req->rq_dev);
+	nbd = minor >> ENBD_SHIFT;
+	lo = &enbd_dev[nbd];
+
+	log_sectors_per_blk = lo->logblksize - 9;
+	sectors_per_blk = 1 << log_sectors_per_blk;
+
+	sectors = req->nr_sectors;
+	size = (sectors + sectors_per_blk - 1) >> log_sectors_per_blk;
+
+	return size;
+}
+
+/*
+ * return a temporary buffer containing the (1 or 2 char) device letter.
+ * This works for i up to 26*26. 0 is "a". The buffer is zero
+ * terminated.
+ *
+ * @i number to be transtaed to x[y] alphabetical  form.
+ */
+static char *
+device_letter (int i)
+{
+
+	static char buf[3];
+	static int cached_i = -1;
+
+	if (cached_i == i)
+		return buf;
+
+	cached_i = i;
+
+	if (i < 26) {
+		buf[0] = 'a' + i;
+		buf[1] = 0;
+		return buf;
+	}
+
+	buf[0] = 'a' + i / 26;
+	buf[1] = 'a' + i % 26;
+	buf[2] = 0;
+	return buf;
+}
+
+/*
+ *  PTB sync the device. Modes:
+ *  @arg = 1:  Do it sync
+ *  @arg = 0:  Do it async
+ *
+ *  We can't call sync_dev outside a process context. I don't know why.
+ *  Death results from a scheduled attempt.
+ *
+ *  Call without the semaphore held, as we lock it and call sync_dev.
+ */
+static void
+enbd_sync (struct enbd_device *lo, long arg)
+{
+	struct inode *inode = lo->inode;
+	short minor, nbd, islot;
+
+	islot = atomic_read (&lo->islot);
+
+	if (!(atomic_read (&lo->flags) & ENBD_INITIALISED) || !inode) {
+		goto fail;
+	}
+
+	minor = minor (inode->i_rdev);
+	nbd = minor >> ENBD_SHIFT;
+
+	// PTB sync_dev is async. fsync_dev is sync.
+	switch (arg) {
+	  case 0:
+		// async
+	        // PTB 2.5.7 doesn't have async sync!	 FIXME
+                // PTB we want sync_buffers(..., 0):
+		break;
+	  default:
+		// sync
+		fsync_dev (mk_kdev (major, nbd << ENBD_SHIFT));
+		invalidate_buffers (mk_kdev (major, nbd << ENBD_SHIFT));
+		break;
+	}
+
+	atomic_clear_mask (ENBD_SYNC_REQD, &lo->flags);
+
+	return;
+
+      fail:
+}
+
+static void
+enbd_async_sync (struct enbd_device *lo)
+{
+	enbd_sync (lo, 0);
+}
+static void
+enbd_sync_sync (struct enbd_device *lo)
+{
+	enbd_sync (lo, 1);
+}
+
+/*
+ *  Do it async if we're enabled, sync if we're not.
+ */
+static void
+enbd_maybe_sync_sync (struct enbd_device *lo)
+{
+
+	if (atomic_read (&lo->flags) & ENBD_ENABLED) {
+		enbd_async_sync (lo);
+		return;
+	}
+	enbd_sync_sync (lo);
+}
+
+/*
+ * PTB - put a request onto the head of a nbd device's queue
+ *     - presumably having taken it off the kernel's queue first!
+ *     - We cannot take the io_spinlock since we are called with it on!
+ *     - and we cannot take the semaphore as we may not sleep!
+ *
+ *     @lo      = the device we are on (could we get it from the req?)
+ *     @req     = the request we shift
+ *     @irqsave = save and restore irqmask when taking our queue spinlock
+ */
+static void
+enbd_enqueue (struct enbd_device *lo, struct request *req, int irqsave)
+{
+	unsigned long req_blks = nr_blks (req);
+	short islot = atomic_read (&lo->islot);
+	int cmd;
+
+	islot = islot;		// PTB stops compiler complaints
+
+	if (req_blks < 0) {
+		ENBD_ERROR ("(%d): invalid req %#x. Not touching!\n",
+			   islot, (unsigned) req);
+		return;
+	}
+
+	// PTB don't want anybody mucking with the request quete but we're
+	// called from the kernel request loop so we can't use the spinlock
+	// and I believe we can't use the semaphore as that may cause us to
+	// sleep. We need a special spinlock here if anything.
+	// PTB experiment since we are called from the kernel loop when the
+	// req is already off the kernel queue, we can't be affected by
+	// other alterations to the kernel queue, and our own alter-egos
+	// never fire on interrupts, so they can't interfere either even
+	// if irqs are allowed.
+
+	/* PTB accounting and nothing more */
+	cmd = rq_data_dir (req);
+
+	atomic_add (req_blks, &lo->requests_in[cmd]);
+	atomic_inc (&lo->countq[cmd]);
+	atomic_inc (&lo->req_in[cmd][req_blks]);
+
+	if (atomic_read (&lo->maxq[cmd]) < atomic_read (&lo->countq[cmd]))
+		atomic_set (&lo->maxq[cmd], atomic_read (&lo->countq[cmd]));
+
+	if (atomic_read (&lo->maxreqblks) < req_blks)
+		atomic_set (&lo->maxreqblks, req_blks);
+
+	// PTB don't need irqsave because no irq uses our spinlock
+	if (irqsave) {
+		unsigned long flags;
+		write_lock_irqsave (&lo->queue_spinlock, flags);
+		list_add (&req->queuelist, &lo->queue);
+		write_unlock_irqrestore (&lo->queue_spinlock, flags);
+	}
+	else {
+		write_lock (&lo->queue_spinlock);
+		list_add (&req->queuelist, &lo->queue);
+		write_unlock (&lo->queue_spinlock);
+	}
+
+	wake_up_interruptible (&lo->wq);
+
+}
+
+/*
+ * PTB - remove a request from anywhere in the nbd device general queue 
+ *     - return 0 for success, -ve for fail
+ *
+ *     We need to hold the queue semaphore when calling this routine
+ *     and the queue spinlock too! It walks the queue.
+ */
+static int
+enbd_remove (struct enbd_device *lo, struct request *req)
+{
+	int cmd;
+
+	if (!req)
+		return -2;
+
+	/* PTB How can I check that the handle is valid first? */
+	// PTB don't want anybody mucking with the request quete
+	// so we have the semaphore at this point. We don't want to take
+	// the io lock as we're not dealing with the kernel queue but with
+	// our own queue and that only needs a semaphore for protection.
+
+	list_del (&req->queuelist);
+	goto success;
+
+      success:
+        /* PTB accounting and nothing more */
+	cmd = rq_data_dir (req);
+	atomic_dec (&lo->countq[cmd]);
+	return 0;
+}
+
+/*
+ *  PTB - Open the device
+ */
+int
+enbd_open (struct inode *inode, struct file *file)
+{
+	int dev;
+	struct enbd_device *lo;
+	int nbd;
+	int part;
+	int islot;
+	char *devnam;
+
+	if (!inode && file) {	/* added by ptb for 2.0.35. Necessary? */
+		inode = file->f_dentry->d_inode;
+	}
+	if (!inode) {
+		ENBD_ERROR ("null inode.\n");
+		return -EINVAL;
+	}
+
+	dev = minor (inode->i_rdev);
+	nbd = dev >> ENBD_SHIFT;
+	part = dev - (nbd << ENBD_SHIFT);
+	islot = part - 1;
+
+	if (nbd >= MAX_NBD) {
+		ENBD_ERROR ("too many (%d) whole devices open\n", nbd);
+		return -ENODEV;
+	}
+
+	lo = &enbd_dev[nbd];
+	devnam = lo->devnam;
+
+    /* PTB try and stop opens before there's a client available but
+     * after setup has occured. This might be a nuisance, as we open
+     * the device just to send it an ioctl saying we are here!
+     */
+
+	// PTB it's a daemon opening the slot? Assume yes.
+	if (islot >= 0) {
+
+		struct enbd_slot *slot = &lo->slots[islot];
+		int refcnt = slot->refcnt++;
+
+		// PTB only one exclusive open allowed on slots
+		if (refcnt > 0) {
+			if (slot->pid != current->pid) {
+				slot->refcnt--;
+				return -EBUSY;
+			}
+		}
+
+		// PTB first time do init on slot
+		if (refcnt <= 0) {
+			static int enbd_set_sock(struct enbd_slot *slot, int arg);
+			enbd_set_sock (slot, 0);
+		}
+		slot->pid = current->pid;
+	}
+
+	if (part == 0) {
+		/* PTB we have got the whole dev's file or inode for 1st time */
+		if (!lo->file || lo->file != file) {
+			lo->file = file;
+			atomic_set (&lo->frstj, jiffies);
+		}
+		if (!lo->inode || lo->inode != inode) {
+			lo->inode = inode;
+		}
+	}
+
+	atomic_inc (&lo->refcnt);
+
+	MOD_INC_USE_COUNT;
+
+	if (!(atomic_read (&lo->flags) & ENBD_INITIALISED)) {	/* PTB 132 */
+		rwlock_init (&lo->queue_spinlock);
+		atomic_set_mask (ENBD_INITIALISED, &lo->flags);
+		if (!(atomic_read (&lo->flags) & ENBD_ENABLED)) {
+			atomic_set_mask (ENBD_ENABLED, &lo->flags);
+			lo->lives++;
+		}
+	}
+
+	return 0;
+}
+
+/*
+ * PTB - complete a transaction irrefutably by taking it out of the
+ *     - slot pending position it is in, and reporting end_request to kernel
+ *
+ *       We are called without the spinlock held, and without the io
+ *       lock held, because our call to end request will take the io
+ *       lock momentarily.
+ */
+void
+enbd_commit (struct enbd_slot *slot, struct request *req)
+{
+
+	struct enbd_device *lo = slot->lo;
+	unsigned long req_blks = nr_blks (req);
+	int errors, cmd;
+
+	if (req_blks < 0) {
+		ENBD_ERROR
+		 ("corrupted req %#x. Not touching with bargepole.\n",
+		  (unsigned) req);
+		return;
+	}
+
+	errors = req->errors;
+
+	// PTB We're the only daemon who can touch this slot so we don't need
+	// no steenking spinlock. And our slot queue doesn't need the
+	// kernels io lock protection either, so no steenking io lock.
+	list_del (&req->queuelist);
+	// PTB NB enbd_end_req_lock needs to be called without the io spinlock on
+	enbd_end_request_lock (req);
+
+	slot->req_age = 0;
+	slot->req -= req_blks;
+
+        /* PTB accounting and nothing more */
+	cmd = rq_data_dir (req);
+
+	atomic_sub (req_blks, &lo->requests_req[cmd]);
+	if (errors < 0) {
+		atomic_add (req_blks, &lo->requests_err);
+		slot->err += req_blks;
+		return;
+	}
+
+	atomic_add (req_blks, &lo->requests_out[cmd]);
+	slot->out += req_blks;
+
+	if (cmd != WRITE)
+		return;
+
+	// PTB non error case writes
+
+	// PTB account the 4 cases for a md5sum'd transaction
+	switch (slot->flags & (ENBD_SLOT_MD5SUM | ENBD_SLOT_MD5_OK)) {
+
+	  case ENBD_SLOT_MD5SUM | ENBD_SLOT_MD5_OK:
+		atomic_add (req_blks, &lo->wrequests_5to);	// 11
+		atomic_add (req_blks, &lo->wrequests_5so);
+		// zero the countdown to turning off md5 as it works
+		atomic_set (&lo->wrequests_5co, 0);
+		break;
+
+	  case ENBD_SLOT_MD5SUM:
+		atomic_add (req_blks, &lo->wrequests_5to);	// 10
+		atomic_add (req_blks, &lo->wrequests_5wo);
+		atomic_inc (&lo->wrequests_5co);
+		if (atomic_read (&lo->wrequests_5co) > md5_off_threshold) {
+			atomic_set (&lo->wrequests_5co, 0);
+			// turn off md5summing as it's not successful
+			atomic_clear_mask (ENBD_MD5SUM, &lo->flags);
+		}
+		break;
+
+	  case ENBD_SLOT_MD5_OK:
+		atomic_add (req_blks, &lo->wrequests_5to);	// 01
+		atomic_add (req_blks, &lo->wrequests_5eo);
+		atomic_inc (&lo->wrequests_5co);
+		if (atomic_read (&lo->wrequests_5co) > md5_off_threshold) {
+			atomic_set (&lo->wrequests_5co, 0);
+			// turn off md5summing as it's errored
+			atomic_clear_mask (ENBD_MD5SUM, &lo->flags);
+		}
+		break;
+
+	  default:
+	  case 0:
+		// PTB nobody asked for a md5 and nobdy gave one back
+		atomic_inc (&lo->wrequests_5no);
+		if (atomic_read (&lo->wrequests_5no) > md5_on_threshold) {
+			atomic_set (&lo->wrequests_5no, 0);
+			// turn on md5summing every so often
+			atomic_set_mask (ENBD_MD5SUM, &lo->flags);
+		}
+		break;
+	}
+
+	// PTB clear the md5sum indicators from the slot afterwards!
+	slot->flags &= ~(ENBD_SLOT_MD5SUM | ENBD_SLOT_MD5_OK);
+
+}
+
+/*
+ * PTB - error out a transaction irrefutably by taking it out of the
+ *     - slot pending position it is in, and reporting end_request to kernel
+ *
+ *     We must be called without the io spinlock held, as we take it
+ */
+void
+enbd_error (struct enbd_slot *slot, struct request *req)
+{
+	struct enbd_device *lo = slot->lo;
+	unsigned long req_blks = nr_blks (req);
+	int cmd;
+
+	if (req_blks < 0) {
+		ENBD_ERROR ("passed illegal request %#x\n", (unsigned) req);
+	}
+
+	req->errors++;
+
+	// PTB We don't need the spinlock since we don't touch our queue,
+	// and we're the only ones working on this slot.
+	list_del (&req->queuelist);
+
+	ENBD_ALERT ("error out req %x from slot %d!\n", (unsigned) req,
+		    slot->i);
+
+	// PTB enbd_end_req_lock needs to be called without the spinlock on
+	enbd_end_request_lock (req);
+
+	/* PTB accounting and nothing more */
+	cmd = rq_data_dir (req);
+	atomic_sub (req_blks, &lo->requests_req[cmd]);
+
+	slot->in -= req_blks;
+	slot->req -= req_blks;
+
+	slot->req_age = 0;
+	slot->err += req_blks;
+	atomic_add (req_blks, &lo->requests_err);
+}
+
+#define ENBD_UNINTERRUPTIBLE 0
+#define ENBD_INTERRUPTIBLE   1
+
+/*
+ * Take a request out of a slot. This must not hold the i/o lock on
+ * entry as we may take it in order to kill the request (end_request)
+ * or we may take the queue lock in order to play with the devices
+ * queue (enbd_enqueue).
+ */
+static void
+enbd_rollback (struct enbd_slot *slot, struct request *req)
+{
+
+	struct enbd_device *lo = slot->lo;
+	unsigned long req_blks;
+	int cmd;
+
+	if (atomic_read (&lo->flags) & ENBD_SHOW_ERRS) {
+		enbd_error (slot, req);
+		return;
+	}
+
+	req_blks = nr_blks (req);
+
+	if (req_blks < 0) {
+		ENBD_ERROR ("passed illegal request %#x\n", (unsigned) req);
+		return;
+	}
+
+	// PTB don't want anybody mucking with the request queue
+	// but classically this was never held so I'll try doing without it
+	// PTB the reason is that we here modify only the slot queue, to
+	// which we have unique access anyway.
+	list_del (&req->queuelist);
+
+	ENBD_ALERT ("rollback req %x from slot %d!\n", (unsigned) req, slot->i);
+
+	/* PTB accounting */
+	slot->in -= req_blks;
+	slot->req -= req_blks;
+
+	/* PTB - pre-decrement count for enqueue */
+	cmd = rq_data_dir (req);
+	atomic_sub (req_blks, &lo->requests_in[cmd]);
+	atomic_dec (&lo->req_in[cmd][req_blks]);
+	atomic_sub (req_blks, &lo->requests_req[cmd]);
+
+	// PTB don't want anybody mucking with the request queue
+	// but classically this was never held 
+	// PTB we have to take the spinlock to protect enbd_enqueue from
+	// the request function's competition. We also HAVE to take the
+	// spinlock as we need to protect ourselves from competing alter
+	// egos
+	// PTB Don't think the io lock is necessary, but it gives enqueue
+	// its normal env back, so I don't have to think about what it does
+	// PTB yeah, enqueue takes a write lock anyway, so no sweat
+
+	enbd_enqueue (lo, req, ENBD_INTERRUPTIBLE);
+
+}
+
+/*
+ * PTB - undo transactions by taking them out of the slot pending
+ *     - position and replacing them on the generic device queue
+ *     - NB we do not hold the io request lock or queue sem when
+ *     -    calling this as we take it internally
+ */
+static void
+enbd_rollback_all (struct enbd_slot *slot)
+{
+
+	struct request *req;
+	short count = 0;
+
+	while (!list_empty (&slot->queue)) {
+
+		if (count++ > 1000)
+			break;
+
+		req = list_head (&slot->queue, struct request, queuelist);
+
+		if (!req)
+			break;
+
+		enbd_rollback (slot, req);
+	}
+
+}
+
+/*     PTB error out all the requests on a slot
+ *     
+ *     We must be called without the io spinlock held, as we take it in
+ *     enbd_error().
+ */
+static void
+enbd_error_all (struct enbd_slot *slot)
+{
+
+	struct request *req;
+	short count = 0;
+
+	while (!list_empty (&slot->queue)) {
+		if (count++ > 1000)
+			break;
+		req = list_head (&slot->queue, struct request, queuelist);
+		if (!req)
+			break;
+		enbd_error (slot, req);
+	}
+}
+
+/*
+ * PTB - let a request onto the slot pending position
+ *     - Can be called without the spinlock and doesn't take the
+ *       spinlock as we  only deal with our  unique slot. If there
+ *       were more than one client per slot this woould be a problem
+ *       but there aren't so it isn't.
+ */
+void
+enbd_accept (struct enbd_slot *slot, struct request *req)
+{
+
+	struct enbd_device *lo = slot->lo;
+	unsigned long req_blks = nr_blks (req);
+	int cmd;
+
+	if (req_blks < 0)
+		return;
+
+        /* PTB accounting and nothing more */
+	cmd = rq_data_dir (req);
+	atomic_add (req_blks, &lo->requests_req[cmd]);
+	/* PTB - Note that this really is slot and not lo.
+	 */
+	list_add (&req->queuelist, &slot->queue);
+
+	slot->req_age = jiffies;
+	slot->in += req_blks;
+	slot->req += req_blks;
+}
+
+/*
+ * PTB - read from userspace to a request buffer. Do it piecewuse
+ *     - to cope with clustered requests.
+ *     - return number of bytes read
+ *
+ *     Unfortunately the only way we can return less than the right
+ *     number of bytes is when the receiving req does not have the
+ *     right number of buffers, because the copy_from_user itself
+ *     doesn't tell us.
+ */
+static int
+copy_from_user_to_req (struct request *req, char *user, int len)
+{
+
+	unsigned size = 0;
+        struct bio *bio /* = req->bio */;
+
+	/* PTB assume user verified */
+
+        rq_for_each_bio(bio, req) {
+
+            int i;
+            struct bio_vec * bvl;
+
+            bio_for_each_segment(bvl, bio, i) {
+
+                struct page *page       = bvl->bv_page;
+                int offset              = bvl->bv_offset;
+                const unsigned current_size
+                                    = bvl->bv_len;
+	        char *buffer;
+                buffer = page_address(page) + offset;
+ 
+		copy_from_user (buffer, user + size, current_size);
+
+		size += current_size;
+	    }
+	}
+	if (size != len) {
+		ENBD_ALERT
+		 ("requested %d and could only read %d bytes to req #%x\n",
+		  len, size, (unsigned) req);
+		ENBD_ALERT
+		 ("request %#x wanted to read user space buffer  %#x\n",
+		  (unsigned) req, (unsigned) user);
+	}
+	return size;
+}
+
+# define REQ_NBD (1 << __REQ_NR_BITS)
+
+/*
+ * PTB - auxiliary function
+ *       we use the low bit (REQ_RW) of the flags and the first high bit
+ *       (REQ_NBD) to designate the type of request.
+ */
+static int
+rq_type (struct request *req)
+{
+        switch (((req->flags & REQ_RW)?1:0) | ((req->flags & REQ_NBD)?2:0)) {
+            case 0:
+                return READ;
+            case 1:
+                return WRITE;
+            case 2:
+                return IOCTL;
+            case 3:
+                return MD5SUM;
+        }
+        // PTB report what we can of the strangeness if it is strange
+        return (req->flags < 4) ? -1: req->flags;
+}
+static void
+set_rq_type (struct request *req, int type)
+{
+        switch (type) {
+            case READ:
+                req->flags &= ~(REQ_RW | REQ_NBD);
+                return;
+            case WRITE:
+                req->flags &= ~REQ_NBD;
+                req->flags |= REQ_RW;
+                return;
+            case IOCTL:
+                req->flags &= ~REQ_RW;
+                req->flags |= REQ_NBD;
+                return;
+            case MD5SUM:
+                req->flags |= REQ_RW|REQ_NBD;
+                return;
+        }
+}
+
+/*
+ * PTB - andres' kernel half of the user-space network handshake, used
+ *     - to complete a transaction.
+ *     - return 0 for success and -ve for fail.
+ */
+int
+enbd_ack (struct enbd_slot *slot, char *buffer)
+{
+	struct enbd_reply reply;
+	struct request *req, *xreq;
+	int result = 0;
+
+	void *user;
+	unsigned long req_blks = 1;
+	struct enbd_device *lo = slot->lo;
+	unsigned buflen = 0;
+	unsigned reqlen;
+        int cmd;
+
+	if (!(slot->flags & ENBD_SLOT_BUFFERED)) {
+		return -EINVAL;
+	}
+	if (slot->buffer != buffer) {
+		if (slot->nerrs++ < 3)
+			ENBD_ALERT ("(%d): user buffer changed\n", slot->i);
+		return -EINVAL;
+	}
+
+	atomic_inc (&lo->cthreads);
+	slot->flags |= ENBD_SLOT_RUNNING;
+	slot->cli_age = jiffies;
+
+	user = slot->buffer;
+	copy_from_user ((char *) &reply, (char *) user,
+			sizeof (struct enbd_reply));
+
+	// PTB we keep tracking the write position in the input buffer
+	buflen += ENBD_BUFFER_DATA_OFFSET;
+
+	// PTB save the reply handle (which is an address) as our req
+	memcpy ((char *) &req, reply.handle, sizeof (req));
+
+	do {
+		struct list_head *pos;
+		int count = 0;
+		xreq = NULL;
+		list_for_each (pos, &slot->queue) {
+			xreq = list_entry (pos, struct request, queuelist);
+			if (count++ > 1000)
+				break;
+			if (xreq == req)
+				/* PTB found it */
+				break;
+		}
+	} while (0);
+
+	if (xreq != req) {
+		if (slot->nerrs++ < 3)
+			ENBD_ALERT
+			 ("fatal: Bad handle (given) %x != %x (found)!\n",
+			  (unsigned) req, (unsigned) xreq);
+		result = -EAGAIN;
+		ENBD_FAIL ("exited wrong request\n");
+		/* PTB we will roll back requests */
+	}
+
+	if (reply.magic != ENBD_REPLY_MAGIC) {
+		if (slot->nerrs++ < 3)
+			ENBD_ALERT ("Not enough reply magic in "
+				   __FUNCTION__ "\n");
+		result = -EAGAIN;
+		ENBD_FAIL ("Not enough reply magic in " __FUNCTION__ "\n");
+		/* PTB we will roll back requests */
+	}
+
+	if (reply.error > 0 || req->errors > 0) {
+		/* PTB wasn't error++'ed before */
+		req->errors++;
+		if (slot->nerrs++ < 3)
+			ENBD_ALERT ("exited with reply error\n");
+		/* PTB we handle this internally */
+		goto success;
+	}
+
+	req_blks = nr_blks (req);
+
+	reqlen = req->nr_sectors;
+	reqlen <<= 9;
+
+	cmd = rq_type (req);
+        switch (cmd) {
+
+		int size;
+
+	  case READ:
+
+		// PTB We have to copy the buffer bit by bit in
+		// case the request is clustered.
+
+		size =
+		 copy_from_user_to_req (req, ((char *) user) + buflen,
+					reqlen);
+		if (size < reqlen) {
+			ENBD_ALERT
+			 ("(%d): copy %dB from user to req %#x failed (%d)\n",
+			  slot->i, reqlen, (unsigned) req, size);
+			// PTB we could try again? We should investigate.
+			ENBD_FAIL
+			 ("exited because of bad copy from user\n");
+		}
+
+		// PTB we keep tracking the write position in the buffer
+		buflen += size;
+
+		break;
+	  case WRITE:
+		// PTB we want to know if the reply is md5summed, and if it is
+		//     whether the md5sum is the same as the one on the
+		//     request. But that's not something we can presently see
+		//     from here as we don't make an md5sum in the kernel.
+		//     So we have to rely on the reply flag from userspace.
+		//     We transmit the information to the slot, as we can't
+		//     keep it on the request.
+
+		switch (reply.flags &
+			(ENBD_REPLY_MD5SUM | ENBD_REPLY_MD5_OK)) {
+
+		  case ENBD_REPLY_MD5SUM | ENBD_REPLY_MD5_OK:
+			// PTB we asked for an md5sum comparison
+			// PTB the two matched, so we skipped writing the request
+			slot->flags |= (ENBD_SLOT_MD5SUM | ENBD_SLOT_MD5_OK);	// 11
+			break;
+		  case ENBD_REPLY_MD5SUM:
+			// PTB the two differed, so we wrote the request
+			slot->flags |= ENBD_SLOT_MD5SUM;
+			slot->flags &= ~ENBD_SLOT_MD5_OK;	// 10
+			break;
+		  case ENBD_REPLY_MD5_OK:
+			// PTB the server refused the md5 request
+			slot->flags &= ~ENBD_SLOT_MD5SUM;
+			slot->flags &= ENBD_SLOT_MD5_OK;	// 01
+			break;
+		  default:
+		  case 0:
+			// PTB mobody asked for an md5sum comparison
+			slot->flags &= ~(ENBD_SLOT_MD5SUM | ENBD_SLOT_MD5_OK);	// 00
+			break;
+		}
+		// PTB now we're all set up to do the accounting in commit etc.
+
+		break;
+
+	  case IOCTL:
+
+		if (!(reply.flags & ENBD_REPLY_IOCTL))
+			ENBD_ALERT
+			 ("ioctl reply to req %#x has no ioctl flag set\n",
+			  (unsigned) req);
+		// PTB the commit should emit the request notification
+		do {
+
+			unsigned cmd = (unsigned) req->special;
+			char *arg = req->buffer;	// PTB saved local address or direct val
+
+			if (cmd == -1) {
+				result = -EINVAL;
+				ENBD_FAIL ("unauthorized remote ioctl\n");
+			}
+
+			if (_IOC_DIR (cmd) & _IOC_READ) {
+				// PTB it would have been nice to save size in req and
+				// in fact we did .. but only approximately, as nr_sectors.
+				// size_t  size = enbd_ioctl_size(cmd, arg);
+				int sectors = req->nr_sectors;
+				// PTB if we are reading, it should be to the local buffer
+				// PTB arg, which points at lo->ctldata or other buffer
+				if (sectors <= 0) {
+					memcpy (arg, &reply.data.ctldta[0],
+						ENBD_CTLDTA_LENGTH);
+				}
+				else {
+					// PTB sectors is an overestimate. SHould be OK
+					// PTB as we are reading from the client buffer
+					// PTB which has plenty of room to spare
+					int size = sectors << 9;
+					copy_from_user (arg,
+							(char *) user +
+							buflen, size);
+					buflen += size;
+				}
+			}
+		} while(0);
+		break;
+	}			// PTB eswitch
+
+      success:
+	slot->nerrs = 0;
+	// PTB - completion (or erroring) of transaction 
+	// enbd_commit will take the io lock to do end_req
+	enbd_commit (slot, req);
+	atomic_dec (&lo->cthreads);
+	slot->flags &= ~ENBD_SLOT_RUNNING;
+	return 0;
+
+      error_out:
+	// PTB we will next do a client rollback on the slot from userspace.
+	//     Right here we just skip the request. 
+	req->errors += req_blks;
+	slot->err += req_blks;
+	atomic_dec (&lo->cthreads);
+	slot->flags &= ~ENBD_SLOT_RUNNING;
+	result = result < 0 ? result : -ENODEV;
+	return result;
+}
+
+/*
+ * PTB - write to userspace from a request buffer. Do it piecewuse
+ *     - to cope with clustered requests.
+ *     - return number of bytes written
+ */
+static int
+copy_to_user_from_req (struct request *req, char *user, int len)
+{
+
+	unsigned size = 0;
+        struct bio *bio /* = req->bio */;
+
+	/* PTB assume user verified */
+
+        rq_for_each_bio(bio, req) {
+
+            int i;
+            struct bio_vec * bvl;
+
+            bio_for_each_segment(bvl, bio, i) {
+
+                struct page *page       = bvl->bv_page;
+                int offset              = bvl->bv_offset;
+                const unsigned current_size
+                                        = bvl->bv_len;
+	        char *buffer;
+                buffer = page_address(page) + offset;
+ 
+		copy_to_user (user + size, buffer, current_size);
+
+		size += current_size;
+            }
+
+	}
+	return size;
+}
+
+/*
+ * PTB - update speed counters if 5s has passed
+ */
+static void
+enbd_speed (struct enbd_speed *spd)
+{
+
+	struct enbd_device *lo = spd->lo;
+
+	// last time we measured
+	int lastjiffy = atomic_read (&spd->jiffy);
+	// jiffies since last time
+	int djiffy = jiffies - lastjiffy;
+
+	// tot blocks since first time
+	int distance = spd->getdistance ? spd->getdistance (lo) : 0;
+	// previous no we measured
+	int lastdist = atomic_read (&spd->distance);
+	// blocks since last time
+	int ddistance = distance - lastdist;
+
+	// write every 5 second in time
+	if (djiffy > 5 * HZ) {
+
+		// jiffies since first time
+		int tjiffy = jiffies - atomic_read (&spd->lo->frstj);
+
+		// max tot speed measured so far
+		int speedmax = atomic_read (&spd->speedmax);
+
+		// last instantaneous speed we measured
+		int lastspeed = atomic_read (&spd->speed);
+
+		// instantaneous read blocks/s
+		int speed = djiffy ? (ddistance * HZ) / djiffy : 0;
+
+		// smoothed KB/s
+		int speedsmoothed =
+		 (djiffy * speed + HZ * lastspeed) / (djiffy + HZ);
+
+		// average speed to now in KB/s
+		int speedav = tjiffy ? (distance * HZ) / tjiffy : 0;
+
+		// smoothing count for max
+		int speedhi =
+		 (speedav > speedsmoothed) ? speedav : speedsmoothed;
+
+		// doing settings
+		atomic_set (&spd->speed, speedsmoothed);
+		if (speedhi > speedmax)
+			atomic_set (&spd->speedmax, speedhi);
+		atomic_set (&spd->distance, distance);
+		atomic_set (&spd->speedav, speedav);
+		atomic_set (&spd->jiffy, jiffies);
+	}
+}
+
+/*
+ * PTB do the devices three speed updates
+ */
+static void
+enbd_set_speed (struct enbd_device *lo)
+{
+	enbd_speed (&lo->wspeed);
+	enbd_speed (&lo->rspeed);
+	enbd_speed (&lo->tspeed);
+}
+static int
+gettdistance (struct enbd_device *lo)
+{
+	return atomic_read (&lo->requests_in[WRITE]) +
+	 atomic_read (&lo->requests_in[READ]);
+}
+static int
+getrdistance (struct enbd_device *lo)
+{
+	return atomic_read (&lo->requests_in[READ]);
+}
+static int
+getwdistance (struct enbd_device *lo)
+{
+	return atomic_read (&lo->requests_in[WRITE]);
+}
+
+/*
+ * PTB auxiliary functions for manipulating the sequence number 
+ */
+static int
+rq_seqno (struct request *req)
+{
+	return req->flags >> (__REQ_NR_BITS + 1);
+}
+static void
+rq_set_seqno (struct request *req, int val)
+{
+	req->flags &= (1 << (__REQ_NR_BITS + 1)) - 1;
+	req->flags |= val << (__REQ_NR_BITS + 1);
+}
+
+/*
+ * PTB - andres' kernel half of the userspace networking. This part
+ *     - initiates the transaction by taking a request off the generic
+ *     - device queue and placing it in the slots pending position.
+ *     - I believe we return 0 for success and -ve for fail.
+ *     - timeo is the number of jiffies we are prepared to wait
+ */
+int
+enbd_get_req (struct enbd_slot *slot, char *buffer)
+{
+	struct enbd_request request;
+	struct request *req;
+	int result = 0;
+	static atomic_t count;
+	unsigned start_time = jiffies;
+	struct enbd_device *lo = slot->lo;
+	unsigned timeout = lo->req_timeo * HZ;
+	int islot = slot->i;
+	// PTB for the new timezone field in requests 
+	extern struct timezone sys_tz;
+	unsigned long flags;
+
+	atomic_inc (&lo->cthreads);	// PTB - client thread enters
+	slot->flags |= ENBD_SLOT_RUNNING;
+	slot->cli_age = jiffies;
+
+	if (!(slot->flags & ENBD_SLOT_BUFFERED)) {
+		ENBD_FAIL ("Our slot has no buffer");
+	}
+	if (slot->buffer != buffer) {
+		ENBD_FAIL ("Our slot has changed its buffer!");
+	}
+
+	atomic_set (&lo->islot, islot);
+
+	if (!list_empty (&slot->queue)) {
+		ENBD_FAIL ("impossible! already treating one request");
+		// PTB we do a nontrivial rollback from the user daemon 
+	}
+	if (!slot->file) {
+		result = -EBADF;
+		ENBD_FAIL ("Our slot has been nofiled");
+	}
+	if (!(atomic_read (&lo->flags) & ENBD_ENABLED)) {
+		result = -ENODEV;
+		ENBD_FAIL ("Our slot has been vamooshed");
+	}
+
+	// PTB don't even bother to look if we're throttling
+	if (lo->speed_lim > 0
+	    && atomic_read (&lo->wspeed.speed) << (lo->logblksize - 10) >
+	    lo->speed_lim) {
+		int delay_jiffies = 1;
+		// PTB delay a jiffy to pretend we did domething
+		ENBD_ALERT
+		 ("throttling on nd%s at jiffies %lu speed %dKB/s\n",
+		  lo->devnam, jiffies,
+		  atomic_read (&lo->wspeed.speed) << (lo->logblksize - 10));
+		interruptible_sleep_on_timeout (&lo->wq, delay_jiffies);
+		enbd_set_speed (lo);
+		result = -ETIME;
+		goto error_out;
+	}
+
+	// PTB take spinlock in order to examine queue
+	atomic_inc (&lo->cwaiters);
+	slot->flags |= ENBD_SLOT_WAITING;
+
+	// we need to protect ourselves against the request fn too
+	read_lock_irqsave (&lo->queue_spinlock, flags);
+	atomic_dec (&lo->cwaiters);
+	slot->flags &= ~ENBD_SLOT_WAITING;
+
+	// PTB - now spin until request arrives to treat 
+	while (slot->file && list_empty (&lo->queue)) {
+
+		read_unlock_irqrestore (&lo->queue_spinlock, flags);
+
+		atomic_inc (&lo->cwaiters);
+		slot->flags |= ENBD_SLOT_WAITING;
+
+		interruptible_sleep_on_timeout (&lo->wq,
+						start_time + timeout - jiffies);
+
+		slot->flags &= ~ENBD_SLOT_WAITING;
+		atomic_dec (&lo->cwaiters);
+		atomic_inc (&count);
+
+		// PTB Have to take the spinlock again to check at the queue
+		atomic_inc (&lo->cwaiters);
+		slot->flags |= ENBD_SLOT_WAITING;
+		// we need to protect ourselves against the request fn too
+		read_lock_irqsave (&lo->queue_spinlock, flags);
+		atomic_dec (&lo->cwaiters);
+		slot->flags &= ~ENBD_SLOT_WAITING;
+
+		// PTB fail for recheck if we're inactive too long 
+
+		if (jiffies >= start_time + timeout
+		    && list_empty (&lo->queue)) {
+
+			result = -ETIME;
+
+			// PTB we will exit with one code or another, so up
+			// spinlock
+			read_unlock_irqrestore (&lo->queue_spinlock, flags);
+
+			do {
+				static int
+                                      enbd_clr_sock (struct enbd_slot *slot);
+				int siz =
+				 lo->blksize + sizeof (struct enbd_request);
+				// PTB verify the buffer is still OK - holds one block 
+				if (access_ok(VERIFY_WRITE,slot->buffer,siz))
+                                    break;
+
+				result = -EINVAL;
+
+				// PTB clr_sock takes both the io lock and the spinlock
+				enbd_clr_sock (slot);	// PTB TEST
+				ENBD_FAIL ("Our process has died or lost its buffer");
+			} while (0);
+
+			// PTB we may do a rollback from the user daemon here
+			// but it'll be trivial - without effect - as we don't
+			// have a request in our slot to treat.
+
+			goto error_out;
+		}
+
+	}
+
+	// PTB we still have the (read) spinlock here
+
+	if (!(atomic_read (&lo->flags) & ENBD_ENABLED)) {
+		read_unlock_irqrestore (&lo->queue_spinlock, flags);
+		result = -ENODEV;
+		ENBD_FAIL ("Our slot vaporized while we slept!");
+	}
+	if (!slot->file) {
+		read_unlock_irqrestore (&lo->queue_spinlock, flags);
+		result = -EBADF;
+		ENBD_FAIL ("Our slot nofiled itself while we slept!");
+	}
+
+	if (!list_empty (&slot->queue)) {
+		read_unlock_irqrestore (&lo->queue_spinlock, flags);
+		result = -EINVAL;
+		ENBD_FAIL ("impossible! already treating one request");
+		// PTB we do a nontrivial rollback from the user daemon 
+	}
+
+	// PTB now let's relinquish the read lock and try for the write lock
+	read_unlock_irqrestore (&lo->queue_spinlock, flags);
+
+	write_lock_irqsave (&lo->queue_spinlock, flags);
+	// PTB got the write lock
+
+	if (list_empty (&lo->queue)) {
+		write_unlock_irqrestore (&lo->queue_spinlock, flags);
+		// PTB - somebody else did it while we waited on spinlock. OK 
+		result = -EINVAL;
+		ENBD_FAIL ("ho hum beaten to the punch");
+		// PTB we may do a trivial rollback from the user daemon 
+	}
+
+	// PTB cli/sti here looks unnec. hardware interrupts return here 
+	// AMARIN begin uninterruptible code 
+
+	// PTB we have the (write) spinlock
+
+	// PTB oldest=last element in queue 
+	req = list_tail (&lo->queue, struct request, queuelist);
+
+	// PTB this is where we free the req from our queue. We need to be
+	// holding our spinlock at this point
+
+	// PTB - must succeed as have the spinlock 
+	result = enbd_remove (lo, req);
+	// PTB now holding irqs off in enbd_remove 
+
+	// AMARIN end uninterruptable code 
+	// PTB uh - maybe cli/sti is needed? interrupts can muck the queue?
+	//        - Nah! I've left them enabled so we can see any errors.
+
+	write_unlock_irqrestore (&lo->queue_spinlock, flags);
+
+	request.magic = ENBD_REQUEST_MAGIC;
+	request.flags = 0;
+
+	switch (rq_type (req)) {
+
+	  case IOCTL:
+
+		request.type = IOCTL;
+		request.len = 0;
+
+		do {
+			// PTB this is our special ioctl kernel request
+
+			unsigned cmd = (unsigned) req->special;
+			char *arg = req->buffer;
+
+			// PTB the arg was (either a literal or) the lo->ctldta buffer
+
+			if (_IOC_DIR (cmd) & _IOC_READ) {
+				// PTB we're in get_req, transferring stored ioctl
+				size_t size = req->nr_sectors << 9;
+				if (size <= 0) {
+					// PTB we copy the data during the request copy
+					if (_IOC_DIR (cmd) & _IOC_WRITE) {
+						memcpy (&request.data.
+							ctldta[0], arg,
+							ENBD_CTLDTA_LENGTH);
+					}
+					request.len = 0;
+				}
+				else {
+					// PTB we copy to the user buffer later
+					request.len = size;
+				}
+			}
+			// PTB we store the weirded ioctl id
+			// PTB this composition is our private invention
+			request.from = (((u64) cmd) << 32)
+			 // PTB really want this to go to a 64 bit request.special
+			 | ((u64) (unsigned long) arg);
+			// PTB note that we recorded the "original" buffer,
+			// which is the lo->ctldata stuff, so we can put it
+			// back there later
+		} while (0);
+		break;
+
+	  case READ:
+	  case WRITE:
+
+		// PTB we might check first if it's actually our own ack
+		do {
+			int minor = minor (req->rq_dev);
+			int nbd = minor >> ENBD_SHIFT;
+			int islot = minor - (nbd << ENBD_SHIFT) - 1;
+			if (islot != -1) {
+				ENBD_ALERT
+				 ("received  req type %d directed at slot %d\n",
+				  rq_type (req), islot);
+				// We have to go looking at other peoples slots.
+			}
+		} while (0);
+
+		request.type = rq_data_dir (req);
+		request.from = req->sector;
+		request.from <<= 9;
+		request.len = req->nr_sectors;
+		request.len <<= 9;
+		if (atomic_read (&lo->flags) & ENBD_MD5SUM) {
+			// PTB set the please do md5sum flag on the request until we
+			// learn to do it ourselves in-kernel during the memcopy below.
+			request.flags |= ENBD_REQUEST_MD5SUM;
+		}
+		break;
+          case MD5SUM:
+                break;
+
+	  default:
+		ENBD_ALERT ("received unknown req %#x flags %#x\n",
+			   (unsigned) req, rq_type (req));
+		break;
+	}
+
+	request.seqno = rq_seqno (req);
+	// PTB we need to erase the extra seqno info so that on error or on ack
+	// the kernel can use the right internal array, but I'll try and
+	// catch this in the ack function instad
+
+	do_gettimeofday (&request.time);
+	request.zone = sys_tz;
+
+	// PTB I don't know how to fill tz_minuteswest otherwise (tz_dsttime = 0
+	// always)
+
+	memcpy (request.handle, (char *) &req, sizeof (req));
+
+	copy_to_user (slot->buffer, (char *) &request, sizeof (request));
+
+	// PTB the type is always 2bit since it's been created by a mask
+	switch (request.type) {
+		int err;
+	  case READ:
+		break;
+
+	  case IOCTL:
+		if (request.len > 0) {
+			char *arg =
+			 (char *) slot->buffer + ENBD_BUFFER_DATA_OFFSET;
+			copy_to_user (arg, req->buffer, request.len);
+		}
+		break;
+
+	  case WRITE:
+		err = copy_to_user_from_req (req,
+					     (char *) slot->buffer +
+					     ENBD_BUFFER_DATA_OFFSET,
+					     request.len);
+		if (err < request.len) {
+			// PTB buffer had missing BHS's
+			ENBD_ERROR
+			 ("req %#x only offered %d bytes of %d for copy to user\n",
+			  (unsigned) req, result, request.len);
+			// PTB this request is badly damaged. We'd better shoot it.
+			if (req) {
+				if (req->errors == 0) {
+					req->errors++;
+					enbd_end_request_lock (req);
+				}
+			}
+			ENBD_FAIL
+			 ("kernel failed to keep req while we copied from it");
+		}
+		break;
+          case MD5SUM:
+                break;
+          default:
+		ENBD_ERROR
+		 ("req %#x was type %#x\n", (unsigned)req, rq_type(req));
+		ENBD_FAIL ("unknown req type");
+                break;
+	}
+
+	// PTB enbd_accept does not take spinlock and does not need to as
+	// the req is already free of the shared queue and only needs
+	// to be placed on the unique slot queue.
+
+	enbd_accept (slot, req);
+        ENBD_DEBUG(1,
+                "slotted OK request %#x type %d for sector %ld-%ld slot %d\n",
+                        (unsigned)req, rq_type(req), req->sector,
+                        req->sector + req->nr_sectors - 1, slot->i);
+
+	atomic_dec (&lo->cthreads);	// PTB - client thread leaves normally 
+	slot->flags &= ~ENBD_SLOT_RUNNING;
+
+	return 0;
+
+      error_out:
+	// PTB accounting - a fail to get a request is not an errored request 
+	atomic_dec (&lo->cthreads);	// PTB - client thread leaves abnormally 
+	slot->flags &= ~ENBD_SLOT_RUNNING;
+	result = result < 0 ? result : -ENODEV;
+
+	return result;
+}
+
+/*
+ * PTB error out the pending requests on the kernel queue
+ * We have to be called WITHOUT the io request lock held.
+ * We sleep imbetween clearing each request, for "safety".
+ */
+static int
+enbd_clr_kernel_queue (void)
+{
+
+	int count = 0;
+	unsigned long flags;
+        request_queue_t *q = BLK_DEFAULT_QUEUE(major);
+
+	spin_lock_irqsave (q->queue_lock, flags);
+
+	while (!QUEUE_EMPTY && count++ < 1000) {
+		struct request *req;
+		req = CURRENT;
+		if (!req) {	// PTB impossible
+			spin_unlock_irqrestore (q->queue_lock, flags);
+			ENBD_ALERT
+			 ("impossible! kernel queue empty after tested nonemty!\n");
+			goto fail;
+		}
+		blkdev_dequeue_request (req);
+		spin_unlock_irqrestore (q->queue_lock, flags);
+		req->errors++;
+		schedule ();
+		// enbd_end_request should be called with the spinlock held
+		spin_lock_irqsave (q->queue_lock, flags);
+		enbd_end_request (req);
+	}
+	spin_unlock_irqrestore (q->queue_lock, flags);
+	goto success;
+
+      fail:
+	/* PTB fall thru */
+      success:
+	ENBD_ALERT ("removed %d requests\n", count);
+	return count;
+
+}
+
+/*
+ * PTB error out the pending requests on the nbd queue and kernel queue
+ * Note that we take the queue spinlock for this
+ */
+int
+enbd_clr_queue (struct enbd_device *lo)
+{
+	int count = 0;
+
+	while (1) {
+
+		struct request *req;
+		unsigned long req_blks = 1;
+
+                // PTB can't allow new requests via interrupts
+		write_lock (&lo->queue_spinlock);
+		if (list_empty (&lo->queue)) {
+			write_unlock(&lo->queue_spinlock);
+			break;
+		}
+		req = list_head (&lo->queue, struct request, queuelist);
+
+		req_blks = nr_blks (req);
+
+		req->errors += req_blks + 1;
+		atomic_add (req_blks, &lo->requests_err);
+
+		/* PTB - must succeed as have the spinlock */
+		enbd_remove (lo, req);
+		/* PTB now hold irqs off in enbd_remove */
+		write_unlock(&lo->queue_spinlock);
+		count++;
+
+		// PTB enbd_end_req_lock needs to be called without the io spinlock
+		enbd_end_request_lock (req);
+
+	}
+	ENBD_ALERT ("unqueued %d reqs\n", count);
+	return count;
+}
+
+#undef ENBD_FAIL
+#define ENBD_FAIL( s... ) { \
+  ENBD_ERROR( s); printk("\n"); \
+  goto error_out; \
+}
+
+#ifndef NO_BUFFERED_WRITES
+  /*
+   * Magic function from rd.c that we hope saves a buffer head
+   * permanently somewhere in the kernel VM system.
+   */
+static int
+buffered_write_pagecache_IO (struct buffer_head *sbh, int nbd)
+{
+	struct address_space *mapping;
+	unsigned long index;
+	int offset, size, err;
+	struct enbd_device *lo = &enbd_dev[nbd];
+	err = 0;
+
+	// PTB we need to save the /dev/nda inode
+	if (!lo->inode) {
+		err = -ENODEV;
+		goto out;
+	}
+	mapping = lo->inode->i_mapping;
+
+	// PTB index appears to be the page number
+	index = sbh->b_rsector >> (PAGE_CACHE_SHIFT - 9);
+	// PTB offset is in bytes, and says where in the page the sector starts
+	offset = (sbh->b_rsector << 9) & ~PAGE_CACHE_MASK;
+	// PTB well, an abbreviation for the buffer size, in bytes
+	size = sbh->b_size;
+
+	do {
+		// PTB we mark each page that we should write to Uptodate
+
+		int count;
+		struct page **hash;
+		struct page *page;
+		char *src, *dst;
+
+		int unlock = 0;
+
+		// PTB ummm, how much of the page is left to traverse
+		count = PAGE_CACHE_SIZE - offset;
+		// PTB reduce it to how much we actually need to traverse
+		if (count > size)
+			count = size;
+		// PTB say NOW? that we have traversed what we want of the page
+		size -= count;
+
+		hash = page_hash (mapping, index);
+		page = __find_get_page (mapping, index, hash);
+
+		if (!page) {
+			// PTB we get to make a new page
+			page = grab_cache_page (mapping, index);
+			if (!page) {
+				// PTB failed to get new page
+				err = -ENOMEM;
+				goto out;
+			}
+			// PTB magic
+			if (!Page_Uptodate (page)) {
+				memset (kmap (page), 0, PAGE_CACHE_SIZE);
+				kunmap (page);
+				SetPageUptodate (page);
+			}
+			// PTB the new page is locked. We need to unlock it later
+			unlock = 1;
+		}
+
+		// PTB prepare already for next page
+		index++;
+
+		// PTB set up for copy
+		dst = kmap (page);
+		dst += offset;
+		src = bh_kmap (sbh);
+
+		// PTB prepare for next round
+		offset = 0;
+
+		// PTB do a copy
+		memcpy (dst, src, count);
+
+		kunmap (page);
+		bh_kunmap (sbh);
+
+		if (unlock) {
+			UnlockPage (page);
+		}
+		SetPageDirty (page);
+		__free_page (page);
+
+	} while (size > 0);
+
+      out:
+	return err;
+
+}
+static int
+buffered_write (struct request *req)
+{
+
+	struct buffer_head *bh;
+	int dev = minor (req->rq_dev);
+	int nbd = dev >> ENBD_SHIFT;
+	int err = 0;
+
+	// PTB go through and copy and protect the written buffers
+	for (bh = req->bh; bh; bh = bh->b_reqnext) {
+		struct buffer_head *rbh;
+		rbh =
+		 getblk (bh->b_rdev, bh->b_rsector / (bh->b_size >> 9),
+			 bh->b_size);
+		if (bh != rbh) {
+			char *bdata = bh_kmap (bh);
+			memcpy (rbh->b_data, bdata, rbh->b_size);
+			ENBD_ALERT ("got new bh sector %lu on write\n",
+				   bh->b_rsector);
+		}
+		bh_kunmap (bh);
+		mark_buffer_protected (rbh);	// PTB equals dirty, uptodate
+		err = buffered_write_pagecache_IO (bh, nbd);
+		if (err < 0) {
+			break;
+		}
+		brelse (rbh);
+	}
+	return err;
+}
+
+#endif		/* NO_BUFFERED_WRITES */
+
+/*
+ * PTB - kernel function to take reqs off the kernel queue. Runs with
+ * io lock held.
+ */
+static void
+do_nbd_request (request_queue_t * q)
+{
+	struct request *req;
+	int dev, nbd;
+	unsigned long flags;
+
+	while (!QUEUE_EMPTY) {
+
+		struct enbd_device *lo = NULL;
+
+		req = CURRENT;
+
+		dev = minor (req->rq_dev);
+		nbd = dev >> ENBD_SHIFT;
+
+		if (nbd >= MAX_NBD) {
+			ENBD_FAIL ("minor too big in do_nbd_request.");
+		}
+		lo = &enbd_dev[nbd];
+
+		atomic_inc (&lo->kthreads);	/* PTB - one kernel thread enters */
+		if (atomic_read (&lo->kthreads) > atomic_read (&lo->kmax))
+			atomic_set (&lo->kmax, atomic_read (&lo->kthreads));
+
+		if (!lo->file) {
+			ENBD_FAIL ("Request when device not ready.");
+		}
+
+                if (!(req->flags & REQ_CMD)) {
+			ENBD_FAIL ("Request doesn't have the CMD bit set.");
+                }
+
+		// PTB - should use is_read_only(MKDEV(major,nbd<<ENBD_SHIFT))
+		// FIXME run over all the partitions too
+		if (rq_data_dir (req) == WRITE
+		    && (atomic_read (&lo->flags) & ENBD_READ_ONLY)) {
+			ENBD_FAIL ("write on read-only device");
+		}
+		flags = atomic_read (&lo->flags);
+		if (!(flags & ENBD_INITIALISED)) {
+			ENBD_FAIL ("device not initialised.");
+		}
+		if (!(flags & ENBD_ENABLED)) {
+			ENBD_FAIL ("device not enabled.");
+		}
+		if (flags & ENBD_INVALID) {
+			ENBD_FAIL ("device invalidated.");
+		}
+		if (lo->nslot <= 0) {
+			ENBD_ERROR
+			 ("device nd%s has no (%d) clients registered\n",
+			  lo->devnam, lo->nslot);
+			ENBD_FAIL ("device has no clients registered yet.");
+		}
+		if (lo->aslot <= 0 && (flags & ENBD_SHOW_ERRS)) {
+			ENBD_FAIL ("device presently has no active clients.");
+		}
+		if (req->sector + req->nr_sectors > lo->sectors) {
+			ENBD_FAIL ("overrange request");
+		}
+		if (req->sector > ~(1 << (sizeof (int) * 8 - 1))) {
+			ENBD_FAIL ("overrange request");
+		}
+		if (req->sector < 0) {
+			ENBD_FAIL ("underrange request");
+		}
+		req->errors = 0;
+		blkdev_dequeue_request (req);
+                // PTB in 2.5 we can release the iolock briefly here
+                spin_unlock_irq(q->queue_lock);
+		// PTB we are the only reader and writer of lo->seqno
+		if (rq_data_dir (req) == WRITE && rq_seqno (req) == 0) {
+			// PTB it's a new request never seen before
+			atomic_inc (&lo->seqno_out);
+			// PTB we have to be careful to change this back before
+			// giving it back to the kernel, as the kernel uses it.
+			// We patch it back again in enbd_end_request.
+			rq_set_seqno (req, atomic_read (&lo->seqno_out));
+		}
+
+		// PTB normal sequence is to queue request locally
+		enbd_enqueue (lo, req, ENBD_UNINTERRUPTIBLE);
+                ENBD_DEBUG(1,
+                        "queued OK request %#x flags %#lx for sector %ld-%ld\n",
+                         (unsigned)req, req->flags, req->sector,
+                         req->sector + req->nr_sectors - 1);
+		goto accounting;
+
+	      accounting:
+		atomic_dec (&lo->kthreads);
+                // PTB regain the iolock for another turn
+                spin_lock_irq(q->queue_lock);
+		continue;	// PTB next request
+
+	      error_out:
+		// PTB can rely on req being nonnull here
+		req->errors++;
+		blkdev_dequeue_request (req);
+		ENBD_ALERT ("ending req %x with prejudice\n",
+			   (unsigned) req);
+		enbd_end_request (req);
+                // PTB more accounting
+		if (lo) {
+			int req_blks = nr_blks (req);
+			atomic_add (req_blks, &lo->requests_err);
+			atomic_dec (&lo->kthreads);
+		} else {
+                        ENBD_ALERT("failed to account one orphan errored req\n");
+                }
+                // PTB regain the iolock for another turn
+                spin_lock(q->queue_lock);
+                continue;
+	}
+        return;
+}
+
+/*
+ * PTB rollback all requests on a given slot and then invalidate it
+ * (so the requests can't go back until somebody reactivates the slot)
+ * At least rollback (which we call takes both the io spinlock and our
+ * spinlock, so we can hold neither when we are called. Soft_reset
+ * (which we call) also calls rollback, so has the same problem.
+ */
+static int
+   /* introduced by PTB for better modularity */
+enbd_clr_sock (struct enbd_slot *slot)
+   /* PTB arg introduced for multiplexing */
+   /* PTB the nbd arg if -ve, means don't invalidate buffers also */
+{
+	int i = 0;
+	struct enbd_device *lo = slot->lo;
+	int islot = slot->i;
+	unsigned long flags;
+
+	enbd_rollback_all (slot);
+
+	slot->file = NULL;
+	slot->bufsiz = 0;
+	slot->flags = 0;
+	slot->buffer = NULL;
+
+	write_lock_irqsave (&lo->queue_spinlock, flags);
+
+	/* PTB reset lo->aslot */
+
+	if (lo->aslot > 0) {
+
+		/* PTB grr .. do this the hard way since I don't seem to count right */
+		lo->aslot = 0;
+		for (i = 0; i < lo->nslot; i++) {
+			struct enbd_slot *sloti = &lo->slots[i];
+			if (sloti->file)
+				lo->aslot++;
+		}
+
+		if (lo->aslot <= 0) {
+			// PTB if we are the last client alive, diasable device as we die
+			atomic_clear_mask (ENBD_ENABLED, &lo->flags);
+			if (atomic_read (&lo->flags) & ENBD_SHOW_ERRS) {
+				// PTB soft_reset will invalidate_buffers
+				static int enbd_soft_reset (struct enbd_device *lo);
+				// PTB soft reset will take the io spinlock and our spinlock
+				// so we must release our spinlock and never
+				// have the io spinlock at this point
+				write_unlock_irqrestore (&lo->queue_spinlock, flags);
+				enbd_soft_reset (lo);
+				write_lock_irqsave (&lo->queue_spinlock, flags);
+			}
+		}
+                else if (!(atomic_read (&lo->flags) & ENBD_ENABLED)) {
+			// PTB we must not call reenable as that clears the queue
+			// PTB this is silly. We're already enabled or should have been!
+			atomic_set_mask (ENBD_ENABLED, &lo->flags);
+			lo->lives++;
+			ENBD_ALERT ("enabled device nd%s\n", lo->devnam);
+		}
+	}
+
+	/* PTB reset lo->islot, for no good reason */
+
+	if (atomic_read (&lo->islot) == islot) {
+		for (i = 0; i++ < lo->nslot;) {
+			atomic_inc (&lo->islot);
+			if (atomic_read (&lo->islot) >= lo->nslot)
+				atomic_set (&lo->islot, 0);
+			if (lo->slots[atomic_read (&lo->islot)].file)
+				break;
+		}
+	}
+
+	lo->harderror = 0;
+
+	write_unlock_irqrestore (&lo->queue_spinlock, flags);
+
+	/* PTB don't clear whole device queue as we might still be open */
+
+	return 0;
+}
+
+/*
+ * PTB - check all slots for old requests and roll them back. 
+ * At least rollback (which we call takes both the io spinlock and our
+ * spinlock, so we can hold neither when we are called.
+ */
+static void
+enbd_rollback_old (struct enbd_device *lo)
+{
+
+	int islot;
+
+	for (islot = 0; islot < lo->nslot; islot++) {
+		struct enbd_slot *slot = &lo->slots[islot];
+		if (slot->req_age > 0
+		    && slot->req_age < jiffies - lo->req_timeo * HZ) {
+			enbd_rollback_all (slot);
+		}
+	}
+
+}
+
+/*
+ * PTB - register a socket to a slot.
+ *     - Return 0 for success and -ve for failure.
+ *       Nowadays this doesn't do very much!
+ */
+static int
+enbd_set_sock (struct enbd_slot *slot, int arg)
+{
+
+	struct enbd_device *lo = slot->lo;
+	int islot = slot->i;
+	unsigned long flags;
+
+	if (!(atomic_read (&lo->flags) & ENBD_INITIALISED)) {
+		ENBD_ALERT ("device nd%s not initialised yet!\n",
+			   lo->devnam);
+		return -ENODEV;
+	}
+	if (!(atomic_read (&lo->flags) & ENBD_SIZED)) {
+		ENBD_ALERT ("device nd%s not sized yet!\n", lo->devnam);
+		return -EINVAL;
+	}
+	if (!(atomic_read (&lo->flags) & ENBD_BLKSIZED)) {
+		ENBD_ALERT ("device nd%s not blksized yet!\n", lo->devnam);
+		return -EINVAL;
+	}
+	if (!(atomic_read (&lo->flags) & ENBD_SIGNED)) {
+		ENBD_ALERT ("setting unsigned device nd%s! But harmless.\n",
+			   lo->devnam);
+		return -EINVAL;
+	}
+
+	slot = &lo->slots[islot];
+
+	if (slot->file) {
+		static int my_nbd_set_sig (struct enbd_slot *slot, int *sig);
+		int error;	/* PTB need something better */
+		// PTB it could be that the last daemon got kill -9 and that's all
+		// PTB the device is signed at this point
+		if (arg && my_nbd_set_sig (slot, (int *) arg) >= 0) {
+			// PTB signature matches, so error pending requests to let
+			// last accessor release, then try again
+			enbd_clr_sock (slot);
+			if (slot->file) {
+				error = -EBUSY;
+				goto error_out;
+			}
+			error = 0;
+			goto success_out;
+		}
+
+	      error_out:
+		error = -EBUSY;
+		// PTB but it sure would mess up the counts if we didn't exit now
+		return error;
+
+	      success_out:
+	}
+
+	// PTB this is a queue critical code region for the flags business
+	write_lock_irqsave (&lo->queue_spinlock, flags);
+
+	// PTB file has to be nonzero to indicate we're all set up. 
+	slot->file = (void *) 1;
+
+	if (++lo->aslot > 0) {
+		// PTB if this is the first slot, we might call reenable and
+		// thus clr queue too, but reenable takes the spinlock
+		if (!(atomic_read (&lo->flags) & ENBD_ENABLED)) {
+			atomic_set_mask (ENBD_ENABLED, &lo->flags);
+			lo->lives++;
+		}
+	}
+	if (islot >= lo->nslot) {
+		lo->nslot = islot + 1;
+		ENBD_INFO ("increased socket count to %d\n", lo->nslot);
+	}
+
+	lo->harderror = 0;
+
+	write_unlock_irqrestore (&lo->queue_spinlock, flags);
+	// PTB end of queue critical region 
+
+	return 0;
+}
+
+/*
+ * PTB - return the index i of 2^i + j, 0 <= j < 2^i
+ */
+static inline unsigned
+log2 (unsigned arg)
+{
+	unsigned log = 0;
+	while ((arg >>= 1) > 0)
+		log++;
+	return log;
+}
+
+/*
+ * PTB - set the blksize in bytes of the block device. Return 0 for
+ *     - success and -ve for failure.
+ */
+static int
+enbd_set_blksize (struct enbd_device *lo, unsigned int arg)
+{
+	int nbd = lo->nbd;
+	if (arg > PAGE_SIZE || arg < 512 || (arg & (arg - 1))) {
+		ENBD_ERROR ("blksize too big (%u)\n", arg);
+		return -EINVAL;
+	}
+	lo->blksize = enbd_blksizes[nbd << ENBD_SHIFT] = arg;
+	lo->logblksize = log2 (lo->blksize);
+	atomic_set_mask (ENBD_BLKSIZED, &lo->flags);
+	return 0;
+}
+
+/*
+ * PTB - set the size in bytes of the block device. Return 0 for
+ *     - success and -ve for failure.
+ */
+static int
+enbd_set_size (struct enbd_device *lo, u64 arg)
+{
+	int nbd = lo->nbd;
+	lo->bytesize = enbd_bytesizes[nbd << ENBD_SHIFT] = arg;
+	lo->size = enbd_sizes[nbd << ENBD_SHIFT] = arg >> 10;
+	lo->sectors = lo->size << 1;
+
+	atomic_set_mask (ENBD_SIZED, &lo->flags);
+	return 0;
+}
+
+/* WG */
+static int
+my_nbd_set_intvl (struct enbd_device *lo, int arg)
+{
+	if (arg <= 0) {
+		ENBD_ERROR ("bad pulse interval/req timeout value (%d)\n", arg);
+		return -EINVAL;
+	}
+	lo->req_timeo = arg;
+	return 0;
+}
+
+static int
+my_nbd_set_spid (struct enbd_slot *slot, int arg)
+{
+	short spid = arg;
+	if (arg < 0 || arg >= (1 << (sizeof (short) * 8))) {
+		ENBD_ERROR ("bad spid value (%d)\n", arg);
+		return -EINVAL;
+	}
+	slot->spid = spid;
+	return 0;
+}
+
+static int
+my_nbd_set_bufferwr (struct enbd_device *lo, int arg)
+{
+	if (arg) {
+		atomic_set_mask (ENBD_BUFFERWR, &lo->flags);
+	}
+	else {
+		atomic_clear_mask (ENBD_BUFFERWR, &lo->flags);
+	}
+	return 0;
+}
+
+static int
+my_nbd_set_invalid (struct enbd_device *lo, int arg)
+{
+	// PTB we handle the event ourself exactly when it happens
+	// instead of letting the kernel have check_media defined
+	// and doing it there (and reporting 0 to the kernel)
+
+	if (arg == 0) {
+		atomic_clear_mask (ENBD_INVALID, &lo->flags);
+	}
+	else {
+		if (!(atomic_read (&lo->flags) & ENBD_INVALID)) {
+			kdev_t dev = mk_kdev (major, lo->nbd << ENBD_SHIFT);
+			// PTB - clear buffers now instead of waiting for kernel
+			atomic_set_mask (ENBD_INVALID, &lo->flags);
+			destroy_buffers (dev);
+		}
+	}
+	return 0;
+}
+
+/*
+ * PTB - if we're not signed, accept new sig and return success.
+ *     - if we are signed, compare the offer and return success if equal,
+ *     - and -ve for failure.
+ */
+static int
+my_nbd_set_sig (struct enbd_slot *slot, int *sig)
+{
+	int err = 0;
+	int buf[ENBD_SIGLEN / sizeof (int)];
+	int islot = slot->i;
+	struct enbd_device *lo = slot->lo;
+
+	if (!access_ok (VERIFY_READ, (char *) sig, ENBD_SIGLEN)) {
+		err = -EINVAL;
+		return err;
+	}
+	if (!(atomic_read (&lo->flags) & ENBD_SIGNED)) {
+		/* PTB first time grab sig */
+		copy_from_user ((char *) lo->signature, (char *) sig,
+				ENBD_SIGLEN);
+		atomic_set_mask (ENBD_SIGNED, &lo->flags);
+		return 0;
+	}
+	copy_from_user ((char *) buf, (char *) sig, ENBD_SIGLEN);
+
+	/* PTB test for equality */
+
+	if (memcmp (&buf[0], &lo->signature[0], ENBD_SIGLEN / sizeof (int))
+	    != 0) {
+		err = -EINVAL;
+		ENBD_ALERT ("(%d): failed sigcheck wth %d\n", islot, err);
+		return err;
+	}
+	err = 0;
+	return err;
+}
+
+/*
+ * PTB - register a userspace buffer to a slot. Return 0 for success
+ *     - and -ve for failure. Null arg acts as erase.
+ */
+static int
+my_nbd_reg_buf (struct enbd_slot *slot, char *buffer)
+{
+
+	int err = 0, siz;
+	struct enbd_device *lo = slot->lo;
+
+	if (!buffer) {
+		slot->flags &= ~ENBD_SLOT_BUFFERED;
+		slot->buffer = NULL;
+		slot->bufsiz = 0;
+		return 0;
+	}
+
+	siz = lo->max_sectors << 9;
+
+	/* verify the buffer is in the process space */
+	if (!access_ok (VERIFY_WRITE, buffer, siz)) {
+		err = -EINVAL;
+		return err;
+	}
+	/* PTB hope the buffer is as big as it should be - FIXME */
+	slot->buffer = buffer;
+	slot->bufsiz = siz;
+
+	/* PTB let the device bufsiz be min of registered nonzero bufsizes */
+	if (!lo->bufsiz) {
+		// PTB first time
+		lo->bufsiz = siz;
+	}
+	else {
+		if (lo->bufsiz > siz)
+			lo->bufsiz = siz;
+	}
+
+	// PTB just in case the buffer really is small, we reset all the
+	//     kernels request maxima if we have to adjust the device max
+	if (lo->max_sectors < (lo->bufsiz >> 9)) {
+		int j;
+		lo->max_sectors = lo->bufsiz >> 9;
+		for (j = 0; j < ENBD_MAXCONN; j++) {
+			enbd_max_sectors[(lo->nbd << ENBD_SHIFT) + j] =
+			 lo->max_sectors;
+		}
+	}
+
+	slot->flags |= ENBD_SLOT_BUFFERED;
+	return 0;
+}
+
+static struct timer_list reenable_timer;
+/*
+ * PTB - set the enabled flag on a device and then clear all queues
+ * ( call without the spinlock held ) 
+ */
+static void
+enbd_reenable (struct enbd_device *lo)
+{
+
+	int m, n;
+	unsigned long flags;
+
+	if (!(atomic_read (&lo->flags) & ENBD_INITIALISED))
+		return;
+	if (lo->aslot <= 0)
+		return;
+
+	// PTB disable part
+	write_lock_irqsave (&lo->queue_spinlock, flags);
+	if ((atomic_read (&lo->flags) & ENBD_ENABLED)) {
+		atomic_clear_mask (ENBD_ENABLED, &lo->flags);
+	}
+	write_unlock_irqrestore (&lo->queue_spinlock, flags);
+
+	// PTB clear queue part
+	m = enbd_clr_queue (lo);
+	// PTB - have to call clr_kernel_queue without the io_spinlock held
+	n = enbd_clr_kernel_queue ();
+	if (m + n > 0) {
+		// PTB - schedule ourselves to try and reenable again later
+		ENBD_ALERT
+		 ("cleared %d+%d kernel requests, rescheduling enable\n",
+		  m, n);
+		reenable_timer.data = (unsigned long) lo;
+		reenable_timer.expires = jiffies + 1 * HZ;
+		add_timer (&reenable_timer);
+		return;
+	}
+
+	// PTB reenable part
+	write_lock_irqsave (&lo->queue_spinlock, flags);
+	if ((atomic_read (&lo->flags) & ENBD_ENABLED)) {
+		write_unlock_irqrestore (&lo->queue_spinlock, flags);
+		return;
+	}
+	atomic_set_mask (ENBD_ENABLED, &lo->flags);
+	lo->lives++;
+	write_unlock_irqrestore (&lo->queue_spinlock, flags);
+
+}
+static struct timer_list reenable_timer = {
+	{NULL, NULL},
+	0,
+	0,
+	(void (*)(unsigned long)) enbd_reenable,
+};
+
+/*
+ * PTB - this unsets the enabled flag on the device and then clears the
+ *     - queue for the device.
+ */
+static void
+enbd_disable (struct enbd_device *lo)
+{
+	int i;
+	if (!(atomic_read (&lo->flags) & ENBD_INITIALISED)) {
+		return;
+	}
+	atomic_clear_mask (ENBD_ENABLED, &lo->flags);
+	for (i = 0; i < 100; i++) {
+		if (enbd_clr_queue (lo) <= 0)
+			break;
+	}
+}
+
+/*
+ * PTB - drains device queue. Disables device.
+ * At least rollback (which we call takes both the io spinlock and our
+ * spinlock, so we can hold neither when we are called. Also
+ * invalidate buffers, on request of Rogier Wolff.
+ */
+static int
+enbd_soft_reset (struct enbd_device *lo)
+{
+	int j;
+	if (!(atomic_read (&lo->flags) & ENBD_INITIALISED && lo->nslot > 0)) {
+		return -EINVAL;
+	}
+	// We push back the requests in the slot, in order to be able to
+	// vamoosh them in a moment. This is a race, surely? We ought to
+	// do this atomically or dsiable the slots first.
+	for (j = 0; j < lo->nslot; j++) {
+		struct enbd_slot *slot = &lo->slots[j];
+		enbd_rollback_all (slot);
+	}
+	// disable unsets the enabled flag and clears the queue
+	enbd_disable (lo);
+	// PTB WE used to renable in 5s but I couldn't see how that was
+	// triggered, so I got rid of the code (temporarily)
+	//lo->flags &= ~ENBD_SIGNED;   /* PTB unsign the device */
+	// PTB put back invalidate buffers for use when called from
+	// clr_sock from enbd_release on request of Rogier Wolff.
+	for (j = 0; j < lo->nslot; j++) {
+		invalidate_buffers (mk_kdev (major, (lo->nbd << ENBD_SHIFT) + j));
+	}
+	return 0;
+}
+
+/*
+ * PTB - added a device/module reset for tidyness in face of rampant hacking
+ *     - this does a soft_reset of all devices, followed bu a clr sock
+ *     - on each, and then clears the kernel queue. It unsets the
+ *     - enabled flag on each device.
+ *       We have to be called without either the spinlock or the
+ *       spinlock held, as we call soft_reset which takes both, as
+ *       does clr_sock
+ */
+int
+enbd_hard_reset (void)
+{
+	int i;
+	int err = 0;
+
+	for (i = 0; i < MAX_NBD; i++) {
+		struct enbd_device *lo = &enbd_dev[i];
+		int j;
+		enbd_soft_reset (lo);
+		for (j = 0; j < lo->nslot; j++) {
+			struct enbd_slot *slot = &lo->slots[j];
+			//  PTB this takes the io spinlock and our spinlock.
+			enbd_clr_sock (slot);
+		}
+	}
+	// PTB - have to call clr_kernel_queue without the io_spinlock held
+	enbd_clr_kernel_queue ();
+
+#ifdef MODULE
+	// PTB I'd alert if it were negative, if only I had access to count!
+	while (MOD_IN_USE)
+		MOD_DEC_USE_COUNT;
+#endif
+
+	return err;
+}
+
+/*
+ * PTB - generic ioctl handling
+ */
+static int
+enbd_ioctl (struct inode *inode, struct file *file,
+	   unsigned int cmd, unsigned long arg)
+{
+	struct enbd_device *lo = 0;	// PTB device pointer
+	int minor = -1;		// PTB minor on which we got the ioctl
+	int islot = -1;		// PTB slot number 0, 1, ...
+	int nbd = -1;		// PTB the count for the device group
+	struct enbd_slot *slot = 0;	// PTB slot pointer
+	unsigned start_time, timeout;
+	size_t size;		// PTB for use in ioctls
+
+	if (!suser ()) {
+		ENBD_ERROR ("caller must be root.\n");
+		return -EACCES;
+	}
+	if (!inode) {
+		ENBD_ERROR ("given bad inode.\n");
+		return -EINVAL;
+	}
+	if (major (inode->i_rdev) != major) {
+		ENBD_ERROR ("pseudo-major %d != %d\n",
+			   major (inode->i_rdev), major);
+		return -ENODEV;
+	}
+	minor = minor (inode->i_rdev);
+	nbd = minor >> ENBD_SHIFT;
+	if (nbd >= MAX_NBD) {
+		ENBD_ERROR ("tried to open too many devices, %d\n", minor);
+		return -ENODEV;
+	}
+	lo = &enbd_dev[nbd];
+	lo->harderror = 0;
+	islot = minor % ENBD_MAXCONN - 1;
+	slot = &lo->slots[islot];
+
+	// PTB these are all always local ioctls
+	switch (cmd) {
+		int err;
+		int intval;
+
+	  case ENBD_CLEAR_SOCK:
+		err = enbd_clr_sock (slot);
+		return err;
+
+	  case ENBD_SET_SOCK:
+		err = enbd_set_sock (slot, arg);
+		return err;
+
+	  case BLKBSZGET:
+		if (!(atomic_read (&lo->flags) & ENBD_BLKSIZED)) {
+			return -EINVAL;
+		}
+		err = put_user (lo->blksize, (long *) arg);
+		return err;
+
+	  case BLKBSZSET:
+		if (!arg)
+			return -EINVAL;
+		if (get_user (intval, (int *) arg))
+			return -EFAULT;
+		err = enbd_set_blksize (lo, intval);
+		return err;
+
+	  case ENBD_SET_SIZE:
+		err = enbd_set_size (lo, (u64) arg);
+		return err;
+
+	  case ENBD_SET_SECTORS:
+		err = enbd_set_size (lo, ((u64) arg) << 9);
+		return err;
+
+	  case MY_NBD_SET_INTVL:	/* WG */
+		err = my_nbd_set_intvl (lo, arg);
+		return err;
+
+	  case MY_NBD_SET_SPID:
+		err = my_nbd_set_spid (slot, arg);
+		return err;
+
+	  case MY_NBD_SET_BUFFERWR:
+		err = my_nbd_set_bufferwr (lo, arg);
+		return err;
+
+	  case MY_NBD_REG_BUF:	/* PTB register your buffer per socket here */
+		if (!arg) {
+			/* PTB serves as existence check for this ioctl */
+			return 0;
+		}
+		err = my_nbd_reg_buf (slot, (char *) arg);
+		return err;
+
+	  case MY_NBD_SET_SIG:
+		err = my_nbd_set_sig (slot, (int *) arg);
+		return err;
+
+	  case MY_NBD_GET_REQ:
+		err = enbd_get_req (slot, (char *) arg);
+		return err;
+
+	  case MY_NBD_CLR_REQ:
+		enbd_rollback_all (slot);
+		err = 0;
+		return err;
+
+	  case MY_NBD_ERR_REQ:
+		enbd_error_all (slot);
+		err = 0;
+		return err;
+
+	  case MY_NBD_SYNC:
+		err = 0;
+
+		// PTB error too old reqs if show_errs is set, else roll them back
+
+		enbd_rollback_old (lo);
+
+		enbd_set_speed (lo);
+
+		// PTB wait   max(sync_intvl,1) s,
+		if (atomic_read (&lo->flags) & ENBD_ENABLED) {
+			// we stay here for 1s or more
+			int intvl = (sync_intvl > 0) ? sync_intvl : 1;
+			// PTB jiffies we're prepared to throttle for
+			unsigned long timeout = intvl * HZ;
+			// ptb jiffies at which we abort
+			unsigned long timedue = jiffies + timeout;
+
+			while (1) {
+				if (jiffies >= timedue)
+					break;
+				interruptible_sleep_on_timeout (&lo->wq, 1);
+			}
+		}
+		else {
+			err = -EINVAL;
+		}
+
+		return err;
+
+	  case MY_NBD_ACK:
+		err = enbd_ack (slot, (char *) arg);
+		return err;
+
+		/* let this be compiled in always - it's useful. PTB */
+	  case ENBD_PRINT_DEBUG:
+		ENBD_INFO
+		 ("device %d: head = %x, tail = %x, in = %d, out = %d\n",
+		  minor, (int) list_head (&lo->queue, struct request,
+					  queuelist),
+		  (int) list_tail (&lo->queue, struct request, queuelist),
+		  atomic_read (&lo->requests_in[READ]) +
+		  atomic_read (&lo->requests_in[WRITE]),
+		  atomic_read (&lo->requests_out[READ]) +
+		  atomic_read (&lo->requests_out[WRITE]));
+		err = 0;
+		return err;
+	  case ENBD_HARD_RESET:	/* PTB - debugging */
+		err = enbd_hard_reset ();
+		return err;
+
+	  case ENBD_RESET:	/* PTB - debugging */
+		err = enbd_soft_reset (lo);
+		// PTB we reenable in 5s
+		reenable_timer.data = (unsigned long) lo;
+		reenable_timer.expires = jiffies + 5 * HZ;
+		add_timer (&reenable_timer);
+		return err;
+
+	  case ENBD_SET_MD5SUM:	/* PTB - change to do/plead md5summing */
+		if (arg) {
+			atomic_set_mask (ENBD_MD5SUM, &lo->flags);
+		}
+		else {
+			atomic_clear_mask (ENBD_MD5SUM, &lo->flags);
+		}
+		err = 0;
+		return err;
+
+	  case MY_NBD_SET_SHOW_ERRS:	/* PTB/WG - change show error status */
+		if (arg) {
+			atomic_set_mask (ENBD_SHOW_ERRS, &lo->flags);
+		}
+		else {
+			atomic_clear_mask (ENBD_SHOW_ERRS, &lo->flags);
+		}
+		return 0;
+
+	  case MY_NBD_INVALIDATE:
+		err = my_nbd_set_invalid (lo, (int) arg);
+		return err;
+	}
+
+	// PTB these are the standard ioctls, and we might get them from
+	// the other side
+
+	switch (cmd) {
+		int err;
+		int intval;
+
+	  case BLKROSET:	/* PTB - change ro status */
+		if (get_user (intval, (int *) arg))
+			return -EFAULT;
+		// PTB local flags
+		if (intval) {
+			atomic_set_mask (ENBD_READ_ONLY, &lo->flags);
+		}
+		else {
+			atomic_clear_mask (ENBD_READ_ONLY, &lo->flags);
+		}
+		// PTB which device really doesn't matter. We do the checking.
+		set_device_ro (mk_kdev (major, nbd << ENBD_SHIFT), intval);
+		return 0;
+
+	  case BLKROGET:
+		intval = (atomic_read (&lo->flags) & ENBD_READ_ONLY) != 0;
+		return put_user (intval, (int *) arg);
+
+	  case BLKFLSBUF:
+		enbd_maybe_sync_sync (lo);	// PTB normally fsync_dev
+		// PTB device likely has buffers or caches in kernel
+		invalidate_buffers (inode->i_rdev);
+#ifndef NO_BUFFERED_WRITES
+		if (atomic_read (&lo->flags) & ENBD_BUFFERWR) {
+			// PTB got this from rd.c
+			destroy_buffers (inode->i_rdev);
+		}
+#endif		/* NO_BUFFERED_WRITES */
+		return 0;
+
+	  case HDIO_GETGEO:
+		if (!arg) {
+			return -EINVAL;
+		}
+		do {
+			struct hd_geometry *geo =
+			 (struct hd_geometry *) arg;
+			int sectors = enbd_sizes[nbd << ENBD_SHIFT] << 1;
+			unsigned short c;
+			unsigned char h, s;
+			if (sectors < (1 << 22)) {
+				h = 4;
+				s = 16;
+				c = sectors >> 6;
+			}
+			else {
+				h = 255;
+				s = 63;
+				c = (sectors / h) / s;
+			}
+			err = 0;
+			if ((err = put_user (c, &geo->cylinders), err < 0)
+			    || (err = put_user (h, &geo->heads), err < 0)
+			    || (err = put_user (s, &geo->sectors), err < 0)
+			    || (err = put_user (h, &geo->start), err < 0)) {
+				return err;
+			}
+		} while (0);
+		return 0;
+
+	  case BLKGETSIZE:	/* PTB 132 */
+		/* PTB return nr sectors */
+		if (!(atomic_read (&lo->flags) & ENBD_SIZED)) {
+			return -ENODEV;
+		}
+		err = put_user ((unsigned long) lo->sectors,
+			   (unsigned long *) arg);
+		// PTB this check is silly here and seems to trigger!
+		if (lo->size != 0
+		    && (u32) (lo->bytesize >> 10) != lo->size) {
+			ENBD_ALERT
+			 ("bytes %luKB mismatch with KB %u in BLKGETSIZE\n",
+			  (unsigned long) (lo->bytesize >> 10), lo->size);
+		}
+		return err;
+
+#ifdef BLKGETSIZE64
+	  case BLKGETSIZE64:
+		/* PTB return real size in bytes */
+		if (!(atomic_read (&lo->flags) & ENBD_SIZED)) {
+			return -ENODEV;
+		}
+		err = put_user (lo->bytesize, (u64 *) arg);
+		return err;
+#endif
+
+	}
+
+	cmd = enbd_ioctl_convert (cmd);
+	if (cmd == -1) {
+		ENBD_ALERT ("unauthorized ioctl\n");
+		return -EINVAL;
+	}
+
+	// PTB here we have to treat remote ioctls. We should probably make
+	// a request and put it on the local queue, but where can we get
+	// the request from? We might have to keep one in reserve.
+	// That's not a bad idea, because
+	// we generate it here and we delete it here, and the daemon code
+	// is all set up to read that sort of thing. So that's what we do ...
+
+	timeout = lo->req_timeo * HZ;
+	start_time = jiffies;
+
+	// PTB we'll spinlock on our queue just to have any old lock. 
+	// PTB we're waiting to be able to write the req safely
+	write_lock (&lo->queue_spinlock);
+	while (lo->req.rq_status == RQ_ACTIVE) {
+		int err;
+		if (jiffies >= start_time + timeout) {
+			// PTB it takes too long to get the ioctl lock
+			ENBD_ALERT
+			 ("took too long to get a spare ioctl: TIMEOUT\n");
+			write_unlock (&lo->queue_spinlock);
+			return -ETIME;
+		}
+		write_unlock (&lo->queue_spinlock);
+		err = interruptible_sleep_on_timeout (&lo->req_wq,
+						      start_time +
+						      timeout - jiffies);
+		write_lock (&lo->queue_spinlock);
+	}
+
+	// PTB lock is now held, prepare the fake request for our queue
+
+	memset (&lo->req, 0, sizeof (struct request));
+	lo->req.rq_status = RQ_ACTIVE;
+	set_rq_type(&lo->req, IOCTL);
+
+	// PTB 1 block announced for accounting visibility
+	//lo->req.nr_sectors = lo->logblksize - 9;
+
+	lo->req.errors = 0;
+
+	// PTB this is the fixed-up command
+	lo->req.special = (void *) cmd;
+
+	// PTB this is (arg if it is direct, else) the address of a local buffer
+	// PTB we need to store the arg or its dereference somewhere local
+	// for a while until the cnb-client thread can enter and pick it
+	// up. The alternative is to block the ioctl here until it is
+	// picked up, which IS possible.
+	if (_IOC_DIR (cmd) & _IOC_READ) {
+		// PTB indirect
+
+		int err;
+		char *buf;
+
+		size = enbd_ioctl_size_user (cmd, (char *) arg);
+
+		if (size < 0) {
+			// PTB unauthorized ioctl
+			return -EINVAL;
+		}
+
+		if (size > ENBD_CTLDTA_LENGTH) {
+			// PTB we have to use an extra buffer or else block
+			// here and rendezvous directly with the get_req call
+			buf = kmalloc (size, GFP_KERNEL);
+			if (!buf) {
+				write_unlock (&lo->queue_spinlock);
+				return -ENOMEM;
+			}
+			// PTB nr_sectors is at least one iff we kmalloc'ed
+			lo->req.nr_sectors = (size + 511) >> 9;
+		}
+		else {
+			buf = &lo->ctldta[0];
+			lo->req.nr_sectors = 0;
+		}
+
+		if (_IOC_DIR (cmd) & _IOC_WRITE) {
+			err =
+			 enbd_ioctl_copy_from_user (cmd, buf, (char *) arg,
+						   size);
+			if (err < 0) {
+				write_unlock (&lo->queue_spinlock);
+				if (lo->req.nr_sectors > 0)
+					kfree (buf);
+				return err;
+			}
+		}
+
+		// PTB always record which buffer we are using
+		lo->req.buffer = buf;
+
+	}
+	else {
+		// PTB direct - we just need to remember the value
+		size = 0;
+		lo->req.buffer = (char *) arg;
+	}
+
+	// PTB point the request buffer vaguely in the direction of where
+	// the data is, but it doesn't matter.
+	lo->req.rq_dev = mk_kdev (major, minor);
+
+	write_unlock (&lo->queue_spinlock);
+	// PTB Lock released. Now everyone knows this ioctl req is being used.
+
+	// PTB we queue the request for treatment and wait till treated
+	enbd_enqueue (lo, &lo->req, ENBD_UNINTERRUPTIBLE);
+
+	// PTB gain the lock again to watch for when req released in enbd_commit
+	write_lock (&lo->queue_spinlock);
+
+	while (lo->req.rq_status == RQ_ACTIVE) {
+		int err;
+		if (jiffies >= start_time + timeout) {
+			// PTB it takes too long 
+			ENBD_ALERT
+			 ("deleting queued ioctl from queue for timeout!\n");
+			// PTB FIXME really need a spinlock per slot
+			list_del (&lo->req.queuelist);
+			lo->req.rq_status = 0;
+			lo->req.errors++;
+			write_unlock (&lo->queue_spinlock);
+			// PTB FIXME want some kind of error accounting increment
+			ENBD_ALERT
+			 ("took too long to treat queued ioctl: TIMEOUT\n");
+			if (lo->req.nr_sectors > 0) {
+				kfree (lo->req.buffer);
+			}
+			return -ETIME;
+		}
+		write_unlock (&lo->queue_spinlock);
+		err = interruptible_sleep_on_timeout (&lo->req_wq, 1);
+		write_lock (&lo->queue_spinlock);
+	}
+
+	// PTB lock is now held for normal exit
+	if (_IOC_DIR (cmd) & _IOC_READ) {
+		int err;
+		// PTB if we are reading, it should be to the local buffer
+		// PTB the buffer points at lo->ctldta or kmalloced area
+		err =
+		 enbd_ioctl_copy_to_user (cmd, (char *) arg, lo->req.buffer,
+					 size);
+		if (lo->req.nr_sectors > 0) {
+			kfree (lo->req.buffer);
+		}
+		if (err < size) {
+			return -ENOMEM;
+		}
+	}
+	// PTB inspect error status and then release lock
+	lo->req.rq_status = 0;
+	if (lo->req.errors != 0) {
+		write_unlock (&lo->queue_spinlock);
+		return -EINVAL;
+	}
+	write_unlock (&lo->queue_spinlock);
+	return 0;
+
+}
+
+/*
+ * PTB - release the device. This happens when the last process closes
+ * or dies.
+ */
+static int
+enbd_release (struct inode *inode, struct file *file)
+{
+	struct enbd_device *lo;
+	int dev;
+	int nbd;
+	int islot;
+
+	if (!inode) {
+		ENBD_ALERT ("null inode.\n");
+		return -ENODEV;
+	}
+	dev = minor (inode->i_rdev);
+	nbd = dev >> ENBD_SHIFT;
+
+	if (nbd >= MAX_NBD) {
+		ENBD_ALERT ("too many open devices.\n");
+		return -ENODEV;
+	}
+
+	lo = &enbd_dev[nbd];
+
+	islot = dev % ENBD_MAXCONN - 1;
+
+	// PTB it's a daemon closing the slot?
+	if (islot >= 0) {
+		struct enbd_slot *slot = &lo->slots[islot];
+		if (--slot->refcnt <= 0) {
+			// PTB this was the last ref, so it's certain the daemon died
+			enbd_clr_sock (slot);
+			slot->pid = 0;
+		}
+	}
+
+      sync:
+	if (atomic_read (&lo->refcnt) > 1) {
+		// PTB can try async flush since will get another chance later
+	        // PTB 2.5.7 doesn't have async sync!	 FIXME
+                // PTB we want sync_buffers(..., 0):
+                // or write_unlocked_buffers
+	}
+	else {
+		// PTB need to really turn off now and flush
+		fsync_dev (inode->i_rdev);
+	}
+
+	if (atomic_read (&lo->flags) & ENBD_RLSE_REQD) {
+		// delayed release attempt followed up now
+		atomic_clear_mask (ENBD_RLSE_REQD, &lo->flags);
+		atomic_dec (&lo->refcnt);
+		if (MOD_IN_USE)
+			MOD_DEC_USE_COUNT;
+		goto sync;
+	}
+
+	if (atomic_read (&lo->refcnt) <= 0) {
+		ENBD_ALERT ("refcount(%d) <= 0\n",
+			   atomic_read (&lo->refcnt));
+		return 0;
+	}
+	if (!list_empty (&lo->queue) && atomic_read (&lo->refcnt) < 2) {
+		ENBD_ALERT
+		 ("Some requests are waiting queued -> cannot turn off.\n");
+		// PTB signal that we want to release but can't
+		atomic_set_mask (ENBD_RLSE_REQD, &lo->flags);
+		return -EBUSY;
+	}
+	if (atomic_read (&lo->requests_req[READ]) +
+	    atomic_read (&lo->requests_req[WRITE]) > 0
+	    && atomic_read (&lo->refcnt) < 2) {
+		ENBD_ALERT
+		 ("Some requests are still in progress -> cannot turn off.\n");
+		// PTB signal that we want to release but can't
+		atomic_set_mask (ENBD_RLSE_REQD, &lo->flags);
+		return -EBUSY;
+	}
+
+	/* POSSIBLE change socket here PTB */
+
+	atomic_dec (&lo->refcnt);
+	if (MOD_IN_USE)
+		MOD_DEC_USE_COUNT;
+
+	// PTB invalidate buffers on last close
+	if (atomic_read (&lo->refcnt) <= 0 || !MOD_IN_USE) {
+		//invalidate_buffers (lo->inode->i_rdev);     
+
+		lo->bufsiz = 0;
+		atomic_set (&lo->seqno_out, 0);
+	}
+
+	return 0;
+}
+
+static struct block_device_operations enbd_blkops = {
+        owner:                  THIS_MODULE,
+	open:                   enbd_open,
+	release:                enbd_release,
+	ioctl:                  enbd_ioctl,
+	check_media_change:     NULL,
+	revalidate:             NULL,
+};
+
+/*
+ * Pavel - And here should be modules and kernel interface 
+ *  (Just smiley confuses emacs :-)
+ */
+
+/*
+ * PTB This is just to get a nice limited width integer printout in proc!
+ * use endpos (<= 8) spaces at most. We serve from a static buffer size 16.
+ */
+char *
+display (unsigned n, int endpos)
+{
+	// PTB  use endpos (<= 8) spaces at most
+	static char buf[16];
+	int units = 0;
+	int decimals = 0;
+	int decpos = endpos;
+	int wholepart = n, fractionpart = 0;
+	buf[endpos--] = 0;
+	// PTB  find the right units to display. U or K or M or G.
+	while (n >= 1 << 10) {
+		decimals = n & ((1 << 10) - 1);
+		n >>= 10;
+		units++;
+	}
+	switch (units) {
+	  case 0:
+		break;
+	  case 1:
+		buf[endpos--] = 'K';
+		break;
+	  case 2:
+		buf[endpos--] = 'M';
+		break;
+	  case 3:
+		buf[endpos--] = 'G';
+		break;
+	  case 4:
+		buf[endpos--] = 'T';
+		break;
+	}
+	// after this wholepart = n && fractionpart = decimals
+	fractionpart = wholepart & ((1 << (units * 10)) - 1);
+	wholepart >>= units * 10;
+	// PTB  write the whole digits (something between 0 and 1023 inclusive)
+	if (n == 0) {
+		buf[endpos--] = '0';
+	}
+	else {
+		while (endpos >= 0 && n > 0) {
+			buf[endpos--] = '0' + n % 10;
+			n /= 10;
+		}
+	}
+	// PTB if there is space and cause, add decimal digits
+	if (endpos >= 1 && units > 0) {
+		int k = 0;
+		char unitchar = buf[--decpos];
+		buf[decpos + k++] = '.';
+		while (endpos >= k) {
+			int digit = (decimals * 10) >> 10;
+			buf[decpos + k++] = '0' + digit;
+			decimals -= (digit << 10) / 10;
+			decimals *= 10;
+		}
+		buf[decpos + k++] = unitchar;
+		buf[decpos + k] = 0;
+	}
+	// PTB report the start position
+	return buf + endpos + 1;
+}
+
+int
+enbd_read_proc (char *buf, char **start, off_t offset, int len, int *eof,
+	       void *data)
+{
+
+#ifndef MIN
+#define MIN(x,y) ((x)<(y)?(x):(y))
+#endif
+
+	const int limit = MIN (PAGE_SIZE, len) - 80;
+	static int i;
+	struct enbd_device *lo;
+	static int last;
+	static void *next_label;
+	static char *next_label_name;
+	static int total;
+	unsigned long flags;
+
+	if (offset > 0 && !next_label) {
+		*eof = 1;
+		*start = buf;
+		return 0;
+	}
+
+	if (offset <= 0) {
+		// PTB do static inits first time through
+		last = -1;
+		i = 0;
+		next_label = NULL;
+		next_label_name = NULL;
+		total = 0;
+	}
+
+	// PTB  start this bytecount
+	len = 0;
+
+#define ENBD_PROC_LABEL(n) \
+        next_label = &&label_##n; \
+        next_label_name = "label_" #n; \
+        if (len > limit) { \
+            *start = (char *) len; \
+            total += len; \
+            return len;\
+        } \
+        label_##n:
+
+	for ( /* static init */ ; i < MAX_NBD; i++) {
+
+		char *devnam;
+
+		lo = &enbd_dev[i];
+		devnam = lo->devnam;
+		if (lo->nslot <= 0) {
+			next_label = NULL;
+			continue;
+		}
+
+		// PTB computed goto next not-done
+		if (next_label) {
+			void *label = next_label;
+			next_label = NULL;
+			next_label_name = NULL;
+			len = 0;
+			goto *label;
+		}
+
+		ENBD_PROC_LABEL (1);
+
+		if (last == i - 2) {
+			char *prevdevnam = device_letter (i - 1);
+			len +=
+			 sprintf (buf + len, "Device %s:\tClosed\n",
+				  prevdevnam);
+		}
+		if (last < i - 2) {
+			char lastdevnam[3];
+			char prevdevnam[3];
+			strncpy (lastdevnam, device_letter (last + 1), 3);
+			strncpy (prevdevnam, device_letter (i - 1), 3);
+			len +=
+			 sprintf (buf + len, "Device %s-%s:\tClosed\n",
+				  lastdevnam, prevdevnam);
+		}
+
+		ENBD_PROC_LABEL (2);
+
+		len +=
+		 sprintf (buf + len, "Device %s:\tOpen " "\n", devnam);
+
+		ENBD_PROC_LABEL (3);
+
+		len += sprintf (buf + len,
+				"[%s] State:\t%s%s%s%s%s%s%s%s%s%s%slast error %d, lives %d, bp %d\n",
+				devnam, atomic_read (&lo->flags)
+				& ENBD_INITIALISED ? "" : "uninitialized, ",
+				atomic_read (&lo->flags)
+				& ENBD_WRITE_NOCHK ? "noverify, " :
+				"verify, ", atomic_read (&lo->flags)
+				& ENBD_READ_ONLY ? "ro, " : "rw, ",
+				merge_requests ? "merge requests, " : "",
+#ifndef NO_BUFFERED_WRITES
+				atomic_read (&lo->flags)
+				& ENBD_BUFFERWR ? "buffer writes, " : "",
+#else
+				"",
+#endif		/* NO_BUFFERED_WRITES */
+				atomic_read (&lo->flags)
+				& ENBD_ENABLED ? "enabled, " :
+				"disabled, ", atomic_read (&lo->flags)
+				& ENBD_INVALID ? "invalid, " : "",
+				atomic_read (&lo->flags)
+				& ENBD_SHOW_ERRS ? "show_errs, " : "",
+				plug ? "plug, " : "",
+				atomic_read (&lo->flags)
+				& ENBD_SYNC ? "sync, " : "",
+				atomic_read (&lo->flags)
+				& ENBD_MD5SUM ? "md5sum, " : "",
+				lo->harderror,
+				lo->lives -
+				((atomic_read (&lo->flags) & ENBD_ENABLED) ?
+				 1 : 0), 0	//atomic_read(&buffermem_pages)
+		 );
+
+		ENBD_PROC_LABEL (4);
+
+		do {		// PTB begin long do once block
+			int countq[2] = { 0, 0 };
+			int cmd;
+
+			struct list_head *pos;
+
+			read_lock_irqsave (&lo->queue_spinlock, flags);
+
+			list_for_each (pos, &lo->queue) {
+				struct request *req =
+				 list_entry (pos, struct request, queuelist);
+				if (countq[READ] + countq[WRITE] > 1000)
+					break;
+
+				cmd = rq_data_dir (req);
+				countq[cmd]++;
+			}
+
+			read_unlock_irqrestore (&lo->queue_spinlock,
+						flags);
+
+			len += sprintf (buf + len,
+					"[%s] Queued:\t+%dR/%dW curr (check %dR/%dW) +%dR/%dW max\n",
+					devnam,
+					atomic_read (&lo->countq[READ]),
+					atomic_read (&lo->countq[READ]),
+					countq[READ], countq[WRITE],
+					atomic_read (&lo->maxq[READ]),
+					atomic_read (&lo->maxq[WRITE]));
+		} while (0);	// PTB end long do once block
+
+		ENBD_PROC_LABEL (5);
+
+		len += sprintf (buf + len,
+				"[%s] Buffersize:\t%d\t(sectors=%d, blocks=%d)\n",
+				devnam, lo->bufsiz, lo->max_sectors,
+				lo->max_sectors / (lo->blksize >> 9));
+		len +=
+		 sprintf (buf + len, "[%s] Blocksize:\t%d\t(log=%d)\n",
+			  devnam, lo->blksize, lo->logblksize);
+		len +=
+		 sprintf (buf + len, "[%s] Size:\t%luKB\n", devnam,
+			  (unsigned long) (lo->bytesize >> 10));
+		len +=
+		 sprintf (buf + len, "[%s] Blocks:\t%u\n", devnam,
+			  lo->size >> (lo->logblksize - 10));
+
+		ENBD_PROC_LABEL (6);
+
+		len +=
+		 sprintf (buf + len, "[%s] Sockets:\t%d", devnam,
+			  lo->nslot);
+
+		ENBD_PROC_LABEL (7);
+
+		do {		// PTB begin short do once block
+			int j;
+			for (j = 0; j < lo->nslot; j++) {
+				struct enbd_slot *slotj = &lo->slots[j];
+				if (j != atomic_read (&lo->islot))
+					len +=
+					 sprintf (buf + len, "\t(%s)",
+						  slotj->file ? "+" : "-");
+				else
+					len +=
+					 sprintf (buf + len, "\t(%s)",
+						  slotj->file ? "*" : ".");
+			}
+		} while (0);	// PTB end short do once block
+
+		len += sprintf (buf + len, "\n");
+
+		ENBD_PROC_LABEL (8);
+
+		len += sprintf (buf + len, "[%s] Requested:\t%s", devnam,
+				display (atomic_read
+					 (&lo->requests_in[READ]) +
+					 atomic_read (&lo->requests_in
+						      [WRITE]), 7));
+
+		ENBD_PROC_LABEL (9);
+
+		do {		// PTB begin short do once block
+			int j;
+			char buff[2][8];
+			for (j = 0; j < lo->nslot; j++) {
+				struct enbd_slot *slotj = &lo->slots[j];
+				len +=
+				 sprintf (buf + len, "\t(%s)",
+					  display (slotj->in, 5));
+			}
+			strncpy (buff[0],
+				 display (atomic_read
+					  (&lo->requests_in[READ]), 6), 7);
+			strncpy (buff[1],
+				 display (atomic_read
+					  (&lo->requests_in[WRITE]), 6),
+				 7);
+			len +=
+			 sprintf (buf + len, "\t%sR/%sW", buff[0],
+				  buff[1]);
+			enbd_set_speed (lo);
+			len += sprintf (buf + len, "\tmax %d",
+					atomic_read (&lo->maxreqblks));
+		} while (0);	// PTB end short do once block
+
+		len += sprintf (buf + len, "\n");
+		len += sprintf (buf + len, "[%s] Despatched:\t%s", devnam,
+				display (atomic_read
+					 (&lo->requests_out[READ]) +
+					 atomic_read (&lo->requests_out
+						      [WRITE]), 7));
+
+		ENBD_PROC_LABEL (10);
+
+		do {		// PTB begin short do once block
+			int j;
+			char buff[2][8];
+			for (j = 0; j < lo->nslot; j++) {
+				struct enbd_slot *slotj = &lo->slots[j];
+				len +=
+				 sprintf (buf + len, "\t(%s)",
+					  display (slotj->out, 5));
+			}
+			strncpy (buff[0],
+				 display (atomic_read
+					  (&lo->requests_out[READ]), 6),
+				 7);
+			strncpy (buff[1],
+				 display (atomic_read
+					  (&lo->requests_out[WRITE]), 6),
+				 7);
+			len +=
+			 sprintf (buf + len, "\t%sR/%sW", buff[0],
+				  buff[1]);
+			len +=
+			 sprintf (buf + len, "\tmd5 %sW",
+				  display (atomic_read
+					   (&lo->wrequests_5to), 5));
+			len +=
+			 sprintf (buf + len, " (%s eq,",
+				  display (atomic_read
+					   (&lo->wrequests_5so), 5));
+			len +=
+			 sprintf (buf + len, " %s ne,",
+				  display (atomic_read
+					   (&lo->wrequests_5wo), 5));
+			len +=
+			 sprintf (buf + len, " %s dn)",
+				  display (atomic_read
+					   (&lo->wrequests_5eo), 5));
+		} while (0);	// PTB end short do once block
+
+		len += sprintf (buf + len, "\n");
+		len += sprintf (buf + len, "[%s] Errored:\t%s", devnam,
+				display (atomic_read (&lo->requests_err),
+					 7));
+
+		ENBD_PROC_LABEL (11);
+
+		do {		// PTB begin short do once block
+			int j;
+			char buff[2][8];
+			int toterrs = 0;
+
+			for (j = 0; j < lo->nslot; j++) {
+				struct enbd_slot *slotj = &lo->slots[j];
+				len +=
+				 sprintf (buf + len, "\t(%s)",
+					  display (slotj->err, 5));
+				toterrs += slotj->err;
+			}
+			strncpy (buff[0], display (toterrs, 6), 7);
+			strncpy (buff[1],
+				 display (atomic_read (&lo->requests_err) -
+					  toterrs, 6), 7);
+			len +=
+			 sprintf (buf + len, "\t%s+%s\n", buff[0],
+				  buff[1]);
+		} while (0);	// PTB end short do once block
+
+		ENBD_PROC_LABEL (12);
+
+		do {		// PTB begin long do once block
+			int pending_rblks = 0;	/* PTB  reads not reached the slots yet */
+			int pending_wblks = 0;	/* PTB  writes not reached the slots yet */
+			int blks = 0;
+
+			read_lock_irqsave (&lo->queue_spinlock, flags);
+
+			do {	// PTB begin short do once block
+				struct list_head *pos;
+
+				int count = 0;
+				struct request *req;
+
+				list_for_each (pos, &lo->queue) {
+					req =
+					 list_entry (pos, struct request,
+						     queuelist);
+					if (count++ > 1000)
+						break;
+					blks = nr_blks (req);
+					if (blks > 0) {
+						switch (rq_data_dir (req)) {
+						  case READ:
+							pending_rblks +=
+							 blks;
+							break;
+						  case WRITE:
+							pending_wblks +=
+							 blks;
+							break;
+						}
+					}
+				}
+			} while (0);	// PTB end short do once block
+
+			read_unlock_irqrestore (&lo->queue_spinlock,
+						flags);
+			len +=
+			 sprintf (buf + len, "[%s] Pending:\t%d", devnam,
+				  atomic_read (&lo->requests_req[READ]) +
+				  atomic_read (&lo->requests_req[WRITE]));
+
+			do {	// PTB begin short do once block
+				int j;
+				for (j = 0; j < lo->nslot; j++) {
+					struct enbd_slot *slotj =
+					 &lo->slots[j];
+					len +=
+					 sprintf (buf + len, "\t(%d)",
+						  slotj->req);
+				}
+			} while (0);	// PTB end short do once block
+
+			len += sprintf (buf + len,
+					"\t%dR/%dW+%dR/%dW\n",
+					atomic_read (&lo->requests_req[READ]),
+					atomic_read (&lo->requests_req[WRITE]),
+					pending_rblks, pending_wblks);
+
+		} while (0);	// PTB end long do once block
+
+		ENBD_PROC_LABEL (13);
+
+		do {		// PTB begin long do once block
+			char buff[10][8];
+			int shift = lo->logblksize;
+
+			strncpy (buff[0],
+				 display (atomic_read (&lo->wspeed.speed)
+					  << shift, 5), 7);
+			strncpy (buff[1],
+				 display (atomic_read (&lo->wspeed.speedav)
+					  << shift, 5), 7);
+			strncpy (buff[2],
+				 display (atomic_read
+					  (&lo->wspeed.speedmax) << shift,
+					  5), 7);
+
+			strncpy (buff[3],
+				 display (atomic_read (&lo->rspeed.speed)
+					  << shift, 5), 7);
+			strncpy (buff[4],
+				 display (atomic_read (&lo->rspeed.speedav)
+					  << shift, 5), 7);
+			strncpy (buff[5],
+				 display (atomic_read
+					  (&lo->rspeed.speedmax) << shift,
+					  5), 7);
+
+			strncpy (buff[6],
+				 display (atomic_read (&lo->tspeed.speed)
+					  << shift, 5), 7);
+			strncpy (buff[7],
+				 display (atomic_read (&lo->tspeed.speedav)
+					  << shift, 5), 7);
+			strncpy (buff[8],
+				 display (atomic_read
+					  (&lo->tspeed.speedmax) << shift,
+					  5), 7);
+
+			len +=
+			 sprintf (buf + len, "[%s] B/s now:", devnam);
+			len +=
+			 sprintf (buf + len, "\t%s\t(%sR+%sW)\n", buff[6],
+				  buff[3], buff[0]);
+			len +=
+			 sprintf (buf + len, "[%s] B/s ave:", devnam);
+			len +=
+			 sprintf (buf + len, "\t%s\t(%sR+%sW)\n", buff[7],
+				  buff[4], buff[1]);
+			len +=
+			 sprintf (buf + len, "[%s] B/s max:", devnam);
+			len +=
+			 sprintf (buf + len, "\t%s\t(%sR+%sW)\n", buff[8],
+				  buff[5], buff[2]);
+			if (lo->speed_lim > 0) {
+				char buff[8];
+				strncpy (buff,
+					 display (lo->speed_lim << 10, 5),
+					 7);
+				len +=
+				 sprintf (buf + len, "[%s] B/s lim:",
+					  devnam);
+				len +=
+				 sprintf (buf + len, "\t%s\t(%sR+%sW)\n",
+					  buff, buff, buff);
+			}
+		} while (0);	// PTB end long do once block
+
+		do {		// PTB begin short do once block
+			int blks;
+			int tot_reqs = 0;
+
+			len +=
+			 sprintf (buf + len, "[%s] Spectrum:", devnam);
+			for (blks = 0;
+			     blks <= atomic_read (&lo->maxreqblks); blks++) {
+				tot_reqs +=
+				 atomic_read (&lo->req_in[READ][blks]) +
+				 atomic_read (&lo->req_in[WRITE][blks]);
+			}
+
+			for (blks = 0;
+			     blks <= atomic_read (&lo->maxreqblks); blks++) {
+				int req_blks =
+				 atomic_read (&lo->req_in[READ][blks])
+				 + atomic_read (&lo->req_in[WRITE][blks]);
+				int percent =
+				 tot_reqs >
+				 0 ? (100 * req_blks) / tot_reqs : 0;
+				if (percent <= 0)
+					continue;
+				len +=
+				 sprintf (buf + len, "\t%u%%%d", percent,
+					  blks);
+			}
+			len += sprintf (buf + len, "\n");
+		} while (0);	// PTB end short do once block
+
+		ENBD_PROC_LABEL (14);
+
+		len += sprintf (buf + len, "[%s] Kthreads:\t%d", devnam,
+				atomic_read (&lo->kthreads));
+		len +=
+		 sprintf (buf + len, "\t(%d waiting/%d running/%d max)\n",
+			  atomic_read (&lo->kwaiters),
+			  atomic_read (&lo->kthreads) -
+			  atomic_read (&lo->kwaiters),
+			  atomic_read (&lo->kmax));
+
+		ENBD_PROC_LABEL (15);
+
+		len += sprintf (buf + len, "[%s] Cthreads:\t%d", devnam,
+				atomic_read (&lo->cthreads));
+
+		ENBD_PROC_LABEL (16);
+
+		do {
+			int j;
+			for (j = 0; j < lo->nslot; j++) {
+				struct enbd_slot *slotj = &lo->slots[j];
+				int state =
+				 ((slotj->flags & ENBD_SLOT_RUNNING) ? 1 :
+				  0) +
+				 ((slotj->flags & ENBD_SLOT_WAITING) ? 2 :
+				  0);
+				char *desc = "?";
+				switch (state) {
+				  case 0:
+					desc = "-";
+					break;	/* PTB not in */
+				  case 1:
+					desc = "*";
+					break;	/* PTB in and not waiting */
+				  case 2:
+					desc = "?";
+					break;	/* PTB impossible */
+				  case 3:
+					desc = "+";
+					break;	/* PTB in and waiting */
+				}
+				len += sprintf (buf + len, "\t(%s)", desc);
+			}
+		} while (0);
+
+		len += sprintf (buf + len, "\n");
+
+		ENBD_PROC_LABEL (17);
+
+		last = i;
+		len += sprintf (buf + len, "[%s] Cpids:\t%d", devnam,
+				atomic_read (&lo->cthreads));
+
+		do {
+			int j;
+			for (j = 0; j < lo->nslot; j++) {
+				struct enbd_slot *slotj = &lo->slots[j];
+				len +=
+				 sprintf (buf + len, "\t(%u)", slotj->pid);
+			}
+			len += sprintf (buf + len, "\n");
+		} while (0);
+
+		do {
+			int j, k;
+			for (j = 0; j < lo->nslot; j++) {
+				struct enbd_slot *slotj = &lo->slots[j];
+				if (slotj->spid != 0)
+					break;
+			}
+			if (j < lo->nslot) {
+				len +=
+				 sprintf (buf + len, "[%s] Kpids:\t%d",
+					  devnam,
+					  atomic_read (&lo->cthreads));
+				for (k = 0; k < lo->nslot; k++) {
+					struct enbd_slot *slotk =
+					 &lo->slots[k];
+					len +=
+					 sprintf (buf + len, "\t(%u)",
+						  slotk->spid);
+				}
+				len += sprintf (buf + len, "\n");
+			}
+		} while (0);
+
+		ENBD_PROC_LABEL (18);
+
+		ENBD_PROC_LABEL (19);
+
+		// PTB have to tell loop head that we are not reentering 
+		next_label = NULL;
+		next_label_name = NULL;
+	}
+
+	ENBD_PROC_LABEL (20);
+
+	if (last == i - 2) {
+		char *prevnam = device_letter (i - 1);
+		len +=
+		 sprintf (buf + len, "Device %s:\tClosed\n", prevnam);
+	}
+
+	if (last < i - 2) {
+		char lastnam[3];
+		char prevnam[3];
+		strncpy (lastnam, device_letter (last + 1), 3);
+		strncpy (prevnam, device_letter (i - 1), 3);
+		len += sprintf (buf + len, "Device %s-%s:\tClosed\n",
+				lastnam, prevnam);
+	}
+
+	ENBD_PROC_LABEL (21);
+
+	// PTB re-init vital statics for next time 
+	next_label = NULL;
+	next_label_name = NULL;
+
+	*eof = 1;
+	*start = buf;
+	total += len;
+
+	return len;
+}
+
+/*
+ * PTB read an int from a string. Return number of ints read (0 or 1).
+ */
+static int
+sscani (char *buf, int len, int *n)
+{
+
+	int i, a = 0;
+	short has_digits = 0;
+	short is_signed = 0;
+
+	// PTB look for first significant character
+	for (i = 0; i < len; i++) {
+		char c = buf[i];
+		if (c == ' ' || c == '\t') {
+			if (is_signed)
+				return 0;
+		}
+		else if (c == '-') {
+			if (is_signed)
+				return 0;
+			is_signed = -1;
+		}
+		else if (c == '+') {
+			if (is_signed)
+				return 0;
+			is_signed = 1;
+		}
+		else if (c >= '0' && c <= '9') {
+			is_signed = 1;
+			has_digits = 1;
+			break;
+		}
+		else {
+			return 0;
+		}
+	}
+	// PTB i now points at first digit if there is one
+	if (!has_digits)
+		return 0;
+	for (; i < len; i++) {
+		char c = buf[i];
+		if (c < '0' || c > '9')
+			break;
+		a *= 10;
+		a += c - '0';
+	}
+	if (is_signed >= 0)
+		*n = a;
+	else
+		*n = -a;
+	return 1;
+}
+
+/*
+ * look for a 1 or 2 letter device code ("a" or "aa") and save the
+ * device number to which it refers. Return number of device letter
+ * codes found (0 or 1).
+ */
+static int
+sscana (char *buf, int len, int *n)
+{
+
+	int i, a = 0;
+	short has_letters = 0;
+
+	for (i = 0; i < len; i++) {
+		char c = buf[i];
+		if (c >= 'a' && c <= 'z') {
+			has_letters = 1;
+			break;
+		}
+		else if (c == ' ') {
+			if (has_letters)
+				return 0;
+		}
+		else {
+			return 0;
+		}
+	}
+	if (!has_letters)
+		return 0;
+	for (; i < len; i++) {
+		char c = buf[i];
+		if (c < 'a' || c > 'z')
+			break;
+		a *= 26;
+		a += c - 'a';
+	}
+	*n = a;
+	return 1;
+}
+
+/*
+ * read an integer (or 2-letter ascii) arg into an int. Return numner
+ * of integers read (0 or 1) and -1 for no keymatch. The first arg is a
+ * preceding key.
+ * @i is the integer value that results
+ * @j is an index if one one supplied (foo[j] = i ), else -1
+ */
+static int
+getarg (const char *buffer, int buflen, const char *key, int *i, int *j)
+{
+
+	int keylen;
+
+	void skip_ws (void) {
+		while (buflen > 0) {
+			if (*buffer != ' ' && *buffer != '\t')
+				break;
+			buffer++;
+			buflen--;
+	        }
+        };
+
+        skip_ws ();
+
+	keylen = strlen (key);
+	if (strncmp (buffer, key, keylen))
+		return -1;
+
+	buffer += keylen;
+	buflen -= keylen;
+
+	skip_ws ();
+
+	*j = -1;
+	if (*buffer == '[') {
+		char *closing;
+		int indexlen;
+
+		buffer++;
+		buflen--;
+
+		skip_ws ();
+
+		closing = strchr (buffer, ']');
+		if (!closing)
+			return -1;
+		indexlen = closing - buffer;
+		*closing = 0;
+
+		if (sscani ((char *) buffer, indexlen, j) < 1)
+			return 0;
+		if (sscana ((char *) buffer, buflen, j) < 1)
+			return 0;
+
+		buffer = closing;
+		buflen -= indexlen;
+
+		buffer++;
+		buflen--;
+
+		skip_ws ();
+	}
+
+	if (*buffer != '=')
+		return -1;
+
+	buffer++;
+	buflen--;
+
+	skip_ws ();
+
+	if (sscani ((char *) buffer, buflen, i) < 1)
+		return 0;
+	if (sscana ((char *) buffer, buflen, i) < 1)
+		return 0;
+	return 1;
+}
+
+static void
+set_sync_intvl (int sync_intvl, int i)
+{
+	void set_si (void) {
+		struct enbd_device *lo = &enbd_dev[i];
+		if (sync_intvl) {
+			atomic_set_mask (ENBD_SYNC, &lo->flags);
+		}
+		else {
+			atomic_clear_mask (ENBD_SYNC, &lo->flags);
+		}
+	};
+
+	if (i >= 0 && i < MAX_NBD) {
+		set_si ();
+		return;
+	}
+	for (i = 0; i < MAX_NBD; i++) {
+		set_si ();
+	}
+}
+
+static void
+set_speed_lim (int speed_lim, int i)
+{
+	void set_sl (void) {
+		struct enbd_device *lo = &enbd_dev[i];
+		 lo->speed_lim = speed_lim;
+	};
+
+	if (i >= 0 && i < MAX_NBD) {
+		set_sl ();
+		return;
+	}
+	for (i = 0; i < MAX_NBD; i++) {
+		set_sl ();
+	}
+}
+
+static void
+set_show_errs (int show_errs, int i)
+{
+	void set_se (void) {
+		struct enbd_device *lo = &enbd_dev[i];
+		if (show_errs) {
+			atomic_set_mask (ENBD_SHOW_ERRS, &lo->flags);
+			return;
+		};
+		atomic_clear_mask (ENBD_SHOW_ERRS, &lo->flags);
+	};
+
+	if (i >= 0 && i < MAX_NBD) {
+		set_se ();
+		return;
+	}
+	for (i = 0; i < MAX_NBD; i++) {
+		set_se ();
+	}
+}
+
+static void
+set_md5sum (int md5sum, int i)
+{
+
+	void set_md5 (void) {
+		struct enbd_device *lo;
+		 lo = &enbd_dev[i];
+		if (md5sum) {
+			atomic_set_mask (ENBD_MD5SUM, &lo->flags);
+			return;
+		};
+		atomic_clear_mask (ENBD_MD5SUM, &lo->flags);
+	};
+
+	if (i >= 0 && i < MAX_NBD) {
+		set_md5 ();
+		return;
+	}
+	for (i = 0; i < MAX_NBD; i++) {
+		set_md5 ();
+	}
+}
+
+static void
+set_enable (int enable, int i)
+{
+	void set_e (void) {
+		struct enbd_device *lo = &enbd_dev[i];
+		if (enable != 0) {
+			enbd_reenable (lo);
+			return;
+		};
+		atomic_clear_mask (ENBD_ENABLED, &lo->flags);
+	};
+
+	if (i >= 0 && i < MAX_NBD) {
+		set_e ();
+		return;
+	}
+	for (i = 0; i < MAX_NBD; i++) {
+		set_e ();
+	}
+}
+
+/*  
+ * PTB - write a 0 with echo -n 0 to /proc/nbdinfo to do a hard reset.
+ */
+static int
+enbd_write_proc (struct file *file, const char *buffer, unsigned long count,
+		void *data)
+{
+
+	switch (count) {
+
+		int i;
+
+	  case 2:
+		if (buffer[1] != '\n')
+			break;
+		/* else fallthru to case 1 */
+	  case 1:
+		switch (*buffer) {
+		  case '1':
+			enbd_hard_reset ();
+			break;
+		  case '0':
+			for (i = 0; i < MAX_NBD; i++) {
+				//  PTB this takes the io spinlock and our spinlock.
+				struct enbd_device *lo = &enbd_dev[i];
+				enbd_soft_reset (lo);
+				reenable_timer.data = (unsigned long) lo;
+				reenable_timer.expires = jiffies + 5 * HZ;
+				add_timer (&reenable_timer);
+			}
+			break;
+		}
+		break;
+	  default:
+		do {
+			int index;
+
+			if (getarg (buffer, count, "merge_requests",
+				    &merge_requests, &index) >= 0) {
+				// merge_requests
+				break;
+			}
+			if (getarg (buffer, count, "sync_intvl",
+				    &sync_intvl, &index) >= 0
+			    || getarg (buffer, count, "sync",
+				       &sync_intvl, &index) >= 0) {
+				// sync_intvl
+				set_sync_intvl (sync_intvl, index);
+				break;
+			}
+			if (getarg (buffer, count, "speed_lim",
+				    &speed_lim, &index) >= 0) {
+				// speed_lim
+				set_speed_lim (speed_lim, index);
+				break;
+			}
+			if (getarg (buffer, count, "show_errs",
+				    &show_errs, &index) >= 0) {
+				// show_errs
+				set_show_errs (show_errs, index);
+				break;
+			}
+			if (getarg (buffer, count, "plug",
+				    &plug, &index) >= 0) {
+				// plug
+				break;
+			}
+			if (getarg (buffer, count, "md5sum",
+				    &md5sum, &index) >= 0) {
+				// md5sum
+				set_md5sum (md5sum, index);
+				break;
+			}
+#ifndef NO_BUFFERED_WRITES
+			if (getarg (buffer, count, "buffer_writes",
+				    &buffer_writes, &index) >= 0) {
+				// buffer_writes
+				set_buffer_writes (buffer_writes, index);
+				break;
+			}
+#endif		/* NO_BUFFERED_WRITES */
+			if (getarg (buffer, count, "enable",
+				    &enable, &i) >= 0) {
+				// enable
+				set_enable (enable, index);
+				break;
+			}
+			ENBD_ERROR ("illegal %ld character command\n",
+				   count);
+			return -EINVAL;
+		} while (0);
+		break;
+	}
+	return count;
+}
+
+#ifdef MODULE
+MODULE_AUTHOR ("Peter T. Breuer, Andres Marin");
+MODULE_DESCRIPTION ("Enhanced Network Block Device " ENBD_VERSION);
+#endif		/* MODULE */
+
+// PTB we steal these from the queue struct at init
+static merge_requests_fn *ll_merge_requests_fn;
+static merge_request_fn *ll_front_merge_fn;
+static merge_request_fn *ll_back_merge_fn;
+
+/* PTB -
+ * These functions are needed when the kernel does request merging in
+ * order to stop it making requests that are bigger than our buffer.
+ *
+ * To turn OFF merging (once these functions are in place), set
+ * merge_requests=0.
+ */
+static int
+enbd_merge_requests_fn (request_queue_t * q, struct request *req,
+		       struct request *req2)
+{
+	int dev = minor (req->rq_dev);
+	int nbd = dev >> ENBD_SHIFT;
+	struct enbd_device *lo = &enbd_dev[nbd];
+
+	if (!merge_requests)
+		return 0;
+
+	if (!ll_merge_requests_fn)
+		return 0;
+
+	if (req->nr_sectors + req2->nr_sectors > lo->max_sectors)
+		return 0;
+
+	if (req->nr_sectors + req2->nr_sectors >
+	    ((merge_requests + 1) << (lo->logblksize - 9)))
+		return 0;
+
+	return ll_merge_requests_fn (q, req, req2);
+}
+static int
+enbd_front_merge_fn (request_queue_t * q, struct request *req, struct bio * bio)
+{
+	int dev = minor (req->rq_dev);
+	int nbd = dev >> ENBD_SHIFT;
+	struct enbd_device *lo = &enbd_dev[nbd];
+
+	if (!merge_requests)
+		return 0;
+
+	if (!ll_front_merge_fn)
+		return 0;
+
+	if (req->nr_sectors > lo->max_sectors)
+		return 0;
+
+	if (req->nr_sectors > ((merge_requests + 1) << (lo->logblksize - 9)))
+                return 0;
+
+	return ll_front_merge_fn (q, req, bio);
+}
+static int
+enbd_back_merge_fn (request_queue_t * q, struct request *req,
+		   struct bio * bio)
+{
+	int dev = minor (req->rq_dev);
+	int nbd = dev >> ENBD_SHIFT;
+	struct enbd_device *lo = &enbd_dev[nbd];
+
+	if (!merge_requests)
+		return 0;
+
+	if (!ll_back_merge_fn)
+		return 0;
+
+	if (req->nr_sectors > lo->max_sectors)
+		return 0;
+
+	if (req->nr_sectors >
+	    ((merge_requests + 1) << (lo->logblksize - 9))) return 0;
+
+        return ll_back_merge_fn (q, req, bio);
+}
+
+// PTB - and now to play with the sysctl interface ...
+static struct ctl_table_header *enbd_table_header;
+// the above was set by the register call of the root table
+static ctl_table enbd_table[] = {
+	{1, "rahead",
+	 &rahead, sizeof (int), 0644, NULL, &proc_dointvec},
+	{2, "plug",
+	 &plug, sizeof (int), 0644, NULL, &proc_dointvec},
+	{3, "sync_intvl",
+	 &sync_intvl, sizeof (int), 0644, NULL, &proc_dointvec},
+	{4, "merge_requests",
+	 &merge_requests, sizeof (int), 0644, NULL, &proc_dointvec},
+	{5, "md5sum",
+	 &md5sum, sizeof (int), 0644, NULL, &proc_dointvec},
+	{8, "md5_on_threshold",
+	 &md5_on_threshold, sizeof (int), 0644, NULL, &proc_dointvec},
+	{9, "md5_off_threshold",
+	 &md5_off_threshold, sizeof (int), 0644, NULL, &proc_dointvec},
+	{10, "speed_lim",
+	 &speed_lim, sizeof (int), 0644, NULL, &proc_dointvec},
+	{0}
+};
+static ctl_table enbd_dir_table[] = {
+	{6, "nbd", NULL, 0, 0555, enbd_table},
+	{0}
+};
+static ctl_table enbd_root_table[] = {
+	{CTL_DEV, "dev", NULL, 0, 0555, enbd_dir_table},
+	{0}
+};
+
+#ifdef CONFIG_DEVFS_FS
+static devfs_handle_t devfs_handle;
+static devfs_handle_t devfs_handles[MAX_NBD];
+#endif
+
+int __init
+enbd_init (void)
+{
+	int i, j;
+	int err = 0;
+
+	ENBD_INFO ("Network Block Device originally by pavel@elf.mj.gts.cz\n");
+	ENBD_INFO ("Network Block Device port to 2.0 by ptb@it.uc3m.es\n");
+	ENBD_INFO ("Network Block Device move networking to user space by "
+		  "amarin@it.uc3m.es\n");
+	ENBD_INFO ("Enhanced Network Block Device " ENBD_VERSION " by "
+		  "ptb@it.uc3m.es\n");
+	if (register_blkdev (major, "nbd", &enbd_blkops)) {
+		ENBD_ERROR ("Unable to register major number %d for NBD\n",
+			   major);
+		return -EIO;
+	}
+#ifdef MODULE
+	ENBD_INFO ("registered device at major %d\n", major);
+#endif
+	blksize_size[major] = enbd_blksizes;	/* blksize in B */
+	blk_size[major] = enbd_sizes;	/* size in KB */
+
+// PTB - set up kernel queue struct with default methods
+	blk_init_queue (BLK_DEFAULT_QUEUE (major), do_nbd_request, &enbd_lock);
+
+        (BLK_DEFAULT_QUEUE (major))->max_sectors
+                       = buf_sectors;	/* max per request */
+
+// PTB - I think that put:
+//     - q->plug_device_fn    = generic_plug_device    (static ll_rw_blk)
+//     - q->plug_tq.routine   = generic_unplug_device  (static ll_rw_blk)
+//     - q->back_merge_fn     = ll_back_merge_fn       (static ll_rw_blk)
+//     - q->front_merge_fn    = ll_front_merge_fn      (static ll_rw_blk)
+//     - q->merge_requests_fn = ll_merge_requests_fn   (static ll_rw_blk)
+//     - q->request_fn        = do_nbd_request         (param)
+
+// PTB - we have to do some more init magic in 2.4.*. This says that we
+//     - take all stuff off the kernel queue before processing it, so in
+//     - particular it's OK for kernel to do merges with the queue head.
+	blk_queue_headactive (BLK_DEFAULT_QUEUE (major), 0);
+
+// LA - moved the next #if higher;
+//    - kernel 2.2.* doesn't know about plug_device_fn
+
+	// PTB control merge attempts so we don't overflow our buffer
+	ll_merge_requests_fn = (BLK_DEFAULT_QUEUE (major))->merge_requests_fn;
+	ll_front_merge_fn = (BLK_DEFAULT_QUEUE (major))->front_merge_fn;
+	ll_back_merge_fn = (BLK_DEFAULT_QUEUE (major))->back_merge_fn;
+
+// JSA - Add this line because under >=2.4.1, merge optimizations are in flux
+// PTB - however it's not this which does damage, I believe. Data: plugging
+//     - simply has to be enabled in these kernels. Without it, requests just
+//     - sit on the kernel queue and never come off and into our request_fn.
+// PTB - commented the ifdef again after talks with Jens Axboe.
+//     - Apparently plug_fn will disappear in 2.4.4 and merge functions are
+//       the only way to control merges, so they MUST be included.
+
+// PTB - The functions below just impose our own stricter size limit before
+//     - calling the defaults if all seems OK sizewise.
+
+	(BLK_DEFAULT_QUEUE (major))->merge_requests_fn = &enbd_merge_requests_fn;
+	(BLK_DEFAULT_QUEUE (major))->front_merge_fn = &enbd_front_merge_fn;
+	(BLK_DEFAULT_QUEUE (major))->back_merge_fn = &enbd_back_merge_fn;
+
+
+	for (i = 0; i < MAX_NBD; i++) {
+		struct enbd_device *lo = &enbd_dev[i];
+		memset (lo, 0, sizeof (struct enbd_device));
+		lo->magic = ENBD_DEV_MAGIC;
+		strncpy (lo->devnam, device_letter (i), 4);
+		for (j = 0; j < ENBD_MAXCONN; j++) {	/* PTB */
+			struct enbd_slot *slot = &lo->slots[j];
+			slot->lo = lo;
+			slot->i = j;
+			INIT_LIST_HEAD (&slot->queue);
+		}
+		lo->blksize = 1024;	/* PTB 132 */
+		lo->logblksize = 10;	/* PTB */
+		lo->bytesize = 0x7fffffff00000;	/* PTB 132 */
+		lo->size = 0x7fffffff;	/* PTB (bytesizes >> 10) */
+		lo->sectors = 0xfffffffe;	/* PTB sectors */
+		lo->nbd = i;
+		lo->req_timeo = ENBD_REQ_TIMEO;	/* PTB default pulse intvl */
+		lo->max_sectors = buf_sectors;
+                register_disk(NULL, mk_kdev(major,i << ENBD_MAXCONN), 1,
+                        &enbd_blkops, lo->bytesize >> 9);
+		// speed struct inits
+		lo->wspeed.getdistance = getwdistance;
+		lo->rspeed.getdistance = getrdistance;
+		lo->tspeed.getdistance = gettdistance;
+		lo->wspeed.lo = lo;
+		lo->rspeed.lo = lo;
+		lo->tspeed.lo = lo;
+
+		INIT_LIST_HEAD (&lo->queue);
+		init_waitqueue_head (&lo->wq);
+		INIT_LIST_HEAD (&lo->req.queuelist);
+		init_waitqueue_head (&lo->req_wq);
+		for (j = 0; j < ENBD_MAXCONN; j++) {
+			enbd_blksizes[i * ENBD_MAXCONN + j] = lo->blksize;
+			enbd_bytesizes[i * ENBD_MAXCONN + j] = lo->bytesize;
+			enbd_sizes[i * ENBD_MAXCONN + j] = lo->size;
+			enbd_max_sectors[i * ENBD_MAXCONN + j] = lo->max_sectors;
+		}
+		if (md5sum) {
+			atomic_set_mask (ENBD_MD5SUM, &lo->flags);
+		}
+		if (sync_intvl) {
+			atomic_set_mask (ENBD_SYNC, &lo->flags);
+		}
+		if (show_errs) {
+			atomic_set_mask (ENBD_SHOW_ERRS, &lo->flags);
+		}
+		if (buffer_writes) {
+			atomic_set_mask (ENBD_BUFFERWR, &lo->flags);
+		}
+
+		//enbd_partitions[i << ENBD_SHIFT].nr_sects = lo->sectors; 
+
+	}
+
+	do {
+		struct proc_dir_entry *res =
+		 create_proc_read_entry ("nbdinfo", 0, NULL,
+					 &enbd_read_proc, NULL);
+		if (!res) {
+			ENBD_ALERT ("creation of proc entry failed\n");
+			err = -EINVAL;
+			return err;
+		}
+		// PTB additional write_proc entry in struct
+		res->write_proc = &enbd_write_proc;
+	} while (0);
+
+#ifdef CONFIG_DEVFS_FS
+
+	devfs_handle = devfs_mk_dir (NULL, "nd", NULL);
+	if (devfs_handle) {
+		for (i = 0; i < MAX_NBD; i++) {
+			struct enbd_device *lo = &enbd_dev[i];
+			int j;
+			// PTB make the directory "a" "b" etc.
+			devfs_handles[i] =
+			 devfs_mk_dir (devfs_handle, lo->devnam, NULL);
+			// PTB add the blk specials, called "0" "1" to ENBD_MAXCONN-1
+			if (devfs_handles[i]) {
+				devfs_register_series
+				 (devfs_handles[i], "%u",
+				  ENBD_MAXCONN, DEVFS_FL_DEFAULT,
+				  major, i * ENBD_MAXCONN,
+				  S_IFBLK | S_IRUSR | S_IWUSR,
+				  &enbd_blkops, NULL);
+			}
+			// PTB do the whole disk symlink ..
+			devfs_mk_symlink (devfs_handles[i], "disk",
+					  DEVFS_FL_DEFAULT, "0",
+					  NULL, NULL);
+			// PTB .. and the channel symlinks
+			for (j = 1; j < MAX_NBD; j++) {
+				char link[4];
+				char name[8];
+				sprintf (link, "%u", j);
+				sprintf (name, "chan%u", j);
+				devfs_mk_symlink (devfs_handles[i],
+						  name,
+						  DEVFS_FL_DEFAULT,
+						  link, NULL, NULL);
+			}
+		}
+	}
+#endif		/* CONFIG_DEVFS_FS */
+
+	// PTB - sysctl interface
+	enbd_table_header = register_sysctl_table (enbd_root_table, 1);
+
+	return err;
+}
+
+void __exit
+enbd_cleanup (void)
+{
+	int i;
+
+	for (i = 0; i < MAX_NBD; i++) {
+
+		struct enbd_device *lo = &enbd_dev[i];
+		int j;
+
+		if (!(atomic_read (&lo->flags) & ENBD_INITIALISED))
+			continue;
+
+		ENBD_INFO ("invalidating buffers on device nd%s%d-%d\n",
+			  lo->devnam, 0, ENBD_MAXCONN);
+
+		for (j = 0; j < ENBD_MAXCONN; j++) {
+			int minor = i * ENBD_MAXCONN + j;
+			destroy_buffers (mk_kdev (major, minor));
+		}
+
+		ENBD_INFO ("destroying buffers on device nd%s%d-%d\n",
+			  lo->devnam, 0, ENBD_MAXCONN);
+
+		for (j = 0; j < ENBD_MAXCONN; j++) {
+			int minor = i * ENBD_MAXCONN + j;
+			destroy_buffers (mk_kdev (major, minor));
+		}
+	}
+
+	unregister_sysctl_table (enbd_table_header);
+
+#ifdef CONFIG_DEVFS_FS
+	if (devfs_handle) {
+		for (i = 0; i < MAX_NBD; i++) {
+			int j;
+			if (!devfs_handles[i])
+				continue;
+			for (j = 0; j < ENBD_MAXCONN; j++) {
+				devfs_handle_t x;
+				char s[3];
+				s[0] = '0' + j;
+				s[1] = 0;
+				if (j >= 10) {
+					s[0] = '1';
+					s[1] = '0' + (j - 10);
+					s[2] = 0;
+				}
+				x = devfs_find_handle (devfs_handles[i],
+						       s, major,
+						       i * ENBD_MAXCONN + j,
+						       DEVFS_SPECIAL_BLK,
+						       0);
+				if (x)
+					devfs_unregister (x);
+			}
+			// PTB should we also search for links? No - they're not inodes
+			devfs_unregister (devfs_handles[i]);
+		}
+		devfs_unregister (devfs_handle);
+	}
+#endif
+
+	remove_proc_entry ("nbdinfo", &proc_root);
+
+	for (i = 0; i < MAX_NBD; i++) {
+		struct enbd_device *lo = &enbd_dev[i];
+		atomic_clear_mask (ENBD_ENABLED, &lo->flags);
+		if (lo->blockmap) {
+			kfree (lo->blockmap);
+			lo->blockmap = NULL;
+		}
+		enbd_sync_sync (lo);
+		del_timer (&lo->run_queue);
+	}
+
+	blk_cleanup_queue (BLK_DEFAULT_QUEUE (major));
+	blk_size[major] = NULL;
+
+	if (unregister_blkdev (major, "nbd") != 0) {
+		ENBD_ALERT ("cleanup_module failed\n");
+	}
+	else {
+		ENBD_INFO ("module cleaned up.\n");
+	}
+}
+
+module_init (enbd_init);
+module_exit (enbd_cleanup);
+
+
+/* Compile line:
+
+ *  gcc -O2 -D__KERNEL__ -DMODULE -xc -c nbd.c -o nbd.o
+ *
+ *  (possibly with -DMODVERSIONS also). PTB
+ */
--- linux-2.5.7/include/linux/enbd.h.pre-enbd	Wed Apr 10 00:31:01 2002
+++ linux-2.5.7/include/linux/enbd.h	Wed Apr 10 00:28:55 2002
@@ -0,0 +1,387 @@
+#ifndef LINUX_ENBD_H
+#define LINUX_ENBD_H
+
+/* unsigned comments are Pavel's originals for 2.1.*
+ *   pavel@atrey.karlin.mff.cuni.cz (Pavel Machek)
+ * comments marked PTB are from
+ *   ptb@it.uc3m.es (Peter T. Breuer)
+ * comments marked AMARIN are from
+ *   amarin@it.uc3m.es (Andres Marin Lopez)
+ */
+
+#include <asm/types.h>
+
+#ifndef ENBD_VERSION
+#define ENBD_VERSION "2.4.27 $Date$"
+#endif /*ENBD_VERSION*/
+
+  /*
+   * Third type of request apart from READ or WRITE
+   */
+  #ifndef IOCTL
+  # define IOCTL 2
+  #endif
+  /*
+   * and fourth ..
+   */
+  #ifndef MD5SUM
+  # define MD5SUM 3
+  #endif
+
+  /*
+   * We need an extra bit of req->flags
+   * */
+  #define REQ_NBD (1 << __REQ_NR_BITS)
+
+/* PTB - new style ioctl assignments */
+  #define ENBD_SET_SOCK         _IOW(0xab, 0x00, int)
+  #define ENBD_TEST_IOCTL1      _IOW(0xab, 0x01, int)
+  #define ENBD_SET_SIZE         _IOW(0xab, 0x02, int)
+  #define ENBD_DO_IT            _IOW(0xab, 0x03, int)
+  #define ENBD_CLEAR_SOCK       _IOW(0xab, 0x04, int)
+  #define ENBD_CLEAR_QUE        _IO (0xab, 0x05)
+  #define ENBD_PRINT_DEBUG      _IO (0xab, 0x06)
+  #define ENBD_TEST_IOCTL2      _IOR(0xab, 0x07, int)
+  #define ENBD_HARD_RESET       _IO (0xab, 0x09)
+  #define ENBD_DEC_USE_COUNT    _IO (0xab, 0x09)
+  #define MY_NBD_ACK           _IOW(0xab, 0x0a, char *)
+  #define MY_NBD_GET_REQ       _IOW(0xab, 0x0b, char *)
+  #define MY_NBD_REG_BUF       _IOW(0xab, 0x0c, char *)
+  #define MY_NBD_CLR_REQ       _IOW(0xab, 0x0d, int)
+  #define MY_NBD_SYNC          _IOW(0xab, 0x0e, int)
+  #define ENBD_SET_SECTORS      _IOW(0xab, 0x0f, int)
+  #define MY_NBD_SET_SIG       _IOW(0xab, 0x10, int *)
+  #define ENBD_RESET            _IO (0xab, 0x11)
+  #define ENBD_TEST_IOCTL3      _IOWR(0xab, 0x12, int)
+  #define MY_NBD_ERR_REQ       _IOW(0xab, 0x13, int)
+  #define MY_NBD_SET_INTVL     _IOW(0xab, 0x14, int)
+  #define MY_NBD_SET_SHOW_ERRS _IOW(0xab, 0x15, int) 
+  #define ENBD_SET_MD5SUM       _IOW(0xab, 0x16, int) 
+  #define MY_NBD_SET_BUFFERWR  _IOW(0xab, 0x17, int) 
+  #define MY_NBD_INVALIDATE    _IOW(0xab, 0x18, int)
+  #define MY_NBD_SET_SPID      _IOW(0xab, 0x19, int) 
+  #define MY_NBD_SET_RQ_HANDLE _IOW(0xab, 0x1a, void*) 
+  #define MY_NBD_SET_RQ_SEQNO  _IOW(0xab, 0x1b, int) 
+  #define MY_NBD_SET_RQ_DIGEST _IOWR(0xab, 0x1d, enbd_digest_t) 
+  #define ENBD_TEST_IOCTL4      _IOR(0xab, 0x1e, char[256])
+  #define ENBD_TEST_IOCTL5      _IOWR(0xab, 0x1f, char[256])
+  #define ENBD_TEST_IOCTL6      _IO(0xab, 0x20) // special r 256B
+  #define ENBD_TEST_IOCTL7      _IO(0xab, 0x21) // special rw 256B
+  #define ENBD_SET_BLKSIZE      _IOW(0xab, 0x22, int)
+  #define ENBD_GET_BLKSIZE      _IOR(0xab, 0x23, long)
+
+#define MAX_NBD 16          /* PTB MAX was 128, but that's a lot */
+#define ENBD_SHIFT 4         /* PTB 16 partitions/sockets/slots per device */
+                            /* PTB number of socket slots per device */
+#define ENBD_MAXCONN (1<<ENBD_SHIFT)
+#define ENBD_SIGLEN 128      /* PTB length of sig on device */
+#define ENBD_MAX_SECTORS 512 /* PTB max number of 512B sectors in a buffer */
+
+
+#ifdef MAJOR_NR
+  /* PTB we are included from the kernel nbd.c file so put kernel stuff here */
+
+  #include <linux/locks.h>
+  #include <linux/config.h>
+
+  #define ENDREQ_NOCURRENT
+  #define LOCAL_END_REQUEST
+  #include <linux/blk.h>
+
+  static void end_request(struct request *req, int uptodate) {  
+     /* unlock chained buffers */
+     struct bio *bio;
+     while ((bio = req->bio) != NULL) {
+             blk_finished_io(bio_sectors(bio));
+             req->bio = bio->bi_next;
+             bio->bi_next = NULL;
+             bio_endio(bio, uptodate);
+     }
+     blkdev_release_request(req);
+  }                 
+ /* 
+  * PTB This takes the spinlock itself! So call it with the io spinlock
+  * not held.
+  */
+  static void end_request_lock(struct request *req, int uptodate) {  
+     unsigned long flags;
+     request_queue_t *q = req->q;
+     static void rq_set_seqno(struct request *, int);
+
+     rq_set_seqno(req, 0); // PTB Zero extra seqno info
+     /* unlock chained buffers */
+     spin_lock_irqsave(q->queue_lock, flags);
+     end_request(req, uptodate);
+     spin_unlock_irqrestore(q->queue_lock, flags);
+
+  }
+
+ /*
+  * PTB Call this only with the io spinlock * held.
+  */
+  static void enbd_end_request(struct request *req) {
+
+    // PTB the kernel has only 2 queues, read and write, and it uses
+    // the cmd field to determine to which the req belongs. We add a
+    // seqno to it in enbd_do_req, so we reestablish it here.
+    static void rq_set_seqno(struct request *, int);
+
+    rq_set_seqno(req, 0); // PTB Zero extra seqno info
+    end_request( req, (req->errors == 0) ? 1 : 0 );
+  }
+ /* 
+  * PTB This takes the spinlock itself! So call it with the io spinlock
+  * not held.
+  */
+  static void enbd_end_request_lock(struct request *req) {  
+
+    // PTB the kernel has only 2 queues, read and write, and it uses
+    // the cmd field to determine to which the req belongs. We add a
+    // seqno to it in enbd_do_req, so we reestablish it here.
+    static void rq_set_seqno(struct request *, int);
+
+    rq_set_seqno(req, 0); // PTB Zero extra seqno info
+    end_request_lock( req, !req->errors );
+  }
+
+
+  /* PTB various defaults */
+  #define ENBD_RAHEAD_DFLT    24 /* PTB slow medium                  */
+  #define ENBD_SYNC_INTVL      0 /* PTB sync every nK reqs (default disable) */
+  #define ENBD_REQ_TIMEO       5 /* PTB client inactivity chk intvl (rollback) */
+  #define ENBD_SPEED_LIM  100000 /* PTB limit to 100M write reqs/s */
+  #define ENBD_MERGE_REQ_DFLT  0 /* PTB until accounting fixed! */
+   /* PTB Jens Axboe says that plug should always be set in 2.4.* */
+  #define ENBD_PLUG_DFLT       1 
+  #define ENBD_MD5SUM_DFLT     0 
+
+    struct enbd_slot {
+      struct file * file;      /* PTB add - for refcnt, NULL if slot empty */
+      struct socket * sock;    /* PTB add                          */
+      int in;                  /* PTB add - tot blocks entered     */
+      int out;                 /* PTB add - tot blocks released    */
+      int err;                 /* PTB add - tot blocks errored     */
+      int req;                 /* PTB add - tot blocks pending     */
+      char * buffer;           /* PTB add - user space buffer      */
+      int  bufsiz;             /* PTB add - user space buffer size */
+      struct list_head queue;
+      unsigned long req_age;   /* PTB add - age of pending req     */
+      unsigned long cli_age;   /* PTB add - age of client          */
+      struct enbd_device *lo;   /* PTB add - parent device          */
+    #define ENBD_SLOT_RUNNING   0x0001
+    #define ENBD_SLOT_WAITING   0x0002
+    #define ENBD_SLOT_BUFFERED  0x0004
+    #define ENBD_SLOT_MD5SUM    0x8000 /* slot reply has a digest in it ..*/
+    #define ENBD_SLOT_MD5_OK   0x10000 /* .. and equaled req's */
+      int flags;               /* PTB add */
+      int i;                   /* PTB add - slot number */
+      int buflen;              /* PTB add - buffer byte count */
+      int pid;                 /* PTB add - client process */
+      int refcnt;              /* PTB add - so can set_sock/clr_sock ourself */
+      int nerrs;               /* PTB add - local error count */
+      int spid;                /* PTB add - server pid */
+    };
+
+
+  struct enbd_speed {
+    atomic_t speed;                      /* PTB add - current speed in KB/s */
+    atomic_t speedmax;                   /* PTB add - max speed */
+    atomic_t speedav;                    /* PTB add - average speed */
+    atomic_t distance;                   /* PTB add - last distance measure */
+    atomic_t jiffy;                      /* PTB add - last jiffies speed set */
+    struct enbd_device *lo;               /* parent */
+    int (*getdistance)(struct enbd_device*);
+                                         /* get current total measure */
+  };
+
+  struct enbd_device {
+      atomic_t refcnt;	
+    #define ENBD_READ_ONLY   0x0001
+    #define ENBD_WRITE_NOCHK 0x0002
+    #define ENBD_INITIALISED 0x0004
+    #define ENBD_SIGNED      0x0008
+    #define ENBD_ENABLED     0x0010
+    #define ENBD_SIZED       0x0020
+    #define ENBD_BLKSIZED    0x0040
+    #define ENBD_SYNC_REQD   0x0080
+    #define ENBD_RLSE_REQD   0x0100
+    #define ENBD_SHOW_ERRS   0x0200
+    #define ENBD_SYNC        0x0400
+    #define ENBD_MD5SUM      0x8000
+
+
+    #define ENBD_INVALID    0x40000         /* remote resource vanished */
+
+    #define ENBD_BUFFERWR  0x100000         /* buffer writes to device */
+      atomic_t flags;
+      int harderror;		           /* Code of hard error	    */
+      int magic;			   /* FIXME: not if debugging is off  */
+      struct list_head queue;
+      rwlock_t queue_spinlock;             /* PTB add - spinlock */
+      int nslot;                           /* PTB add - total slots */
+      atomic_t islot;                      /* PTB add - current slot */
+      int aslot;                           /* PTB add - total active slots*/
+      atomic_t requests_in[2];             /* PTB add - blocks put on queue */
+      atomic_t requests_out[2];            /* PTB add - blocks out from queue */
+      atomic_t requests_err;               /* PTB add - blocks erred on queue */
+      atomic_t wrequests_5so;              /* PTB add - write blocks md5 skip */
+      atomic_t wrequests_5wo;              /* PTB add - write blocks md5 wr */
+      atomic_t wrequests_5eo;              /* PTB add - write blocks md5 refus*/
+      atomic_t wrequests_5to;              /* PTB add - write blocks md5sum */
+      atomic_t wrequests_5co;              /* PTB add - write blocks md5 tot */
+      atomic_t wrequests_5no;              /* PTB add - write blocks not md5 */
+      atomic_t requests_req[2];            /* PTB add - read blocks pending */
+      atomic_t kwaiters;                   /* PTB add - kernel thrds waiting */
+      atomic_t kthreads;                   /* PTB add - kernel threads in */
+      atomic_t maxq[2];                    /* PTB add - max req queue depth */
+      atomic_t countq[2];                  /* PTB add - request queue depth */
+      atomic_t errors;                     /* PTB add - tot requests errored */
+      atomic_t seqno_out;                  /* PTB add - sequence number */
+      atomic_t cwaiters;                   /* PTB add - client thrds waiting */
+      atomic_t cthreads;                   /* PTB add - client threads in */
+      atomic_t req_in[2][1 + ENBD_MAX_SECTORS/2];
+      wait_queue_head_t wq;                /* PTB add */
+      struct enbd_slot slots[ENBD_MAXCONN];  /* PTB add - client array */
+      unsigned blksize;                    /* PTB add - device blksize in B */
+      u64 bytesize;                        /* PTB add - device size in B */
+      u64 sectors;                         /* PTB add - device size (sectors) */
+      unsigned size;                       /* PTB add - device size in blks */
+      unsigned logblksize;                 /* PTB add - log2 blksize */
+      unsigned nbd;                        /* PTB add - this array index */
+      int signature[ENBD_SIGLEN/sizeof(int)];
+                                           /* PTB add - server sig */
+      struct file * file;                  /* PTB add - for ref */
+      struct inode * inode;                /* PTB add - for ref */
+      int  bufsiz;                         /* PTB add - userspace buffer size */
+      atomic_t kmax;                       /* PTB add - max kernel threads */
+      char *blockmap;                      /* PTB add - map of block states */
+      unsigned long disabled;              /* PTB add - when was it disabled */
+      int req_timeo;                       /* PTB add - inactivity timeout */
+      struct timer_list run_queue;         /* PTB add - run queue */
+      struct tq_struct task_queue;         /* PTB add - task queue */
+      char devnam[4];                      /* PTB add - drive letters */
+      atomic_t maxreqblks;                 /* PTB add - maximum req size seen */
+      int max_sectors;                     /* PTB add - max req size allowed! */
+      int lives;                           /* PTB add - # times enabled */
+      atomic_t seqno_in;                   /* PTB add - wreq sequence number */
+      // PTB speed measurement settings
+      struct enbd_speed tspeed;
+      struct enbd_speed wspeed;
+      struct enbd_speed rspeed;
+      atomic_t frstj;                      /* PTB add - first jiffies */
+      int speed_lim;                       /* PTB add - speed_lim in KB/s */
+      struct request req;                  /* PTB fake request for ioctls */
+      wait_queue_head_t req_wq;
+      char  ctldta[4*4];                   /* PTB ioctl data buffer */
+    };
+
+#endif  /* MAJOR_NR */
+
+
+
+/* Pavel - This now IS in some kind of include file...	*/
+
+/* PTB 132 */ 
+#define ENBD_INIT_MAGIC 0x12345678       /* AMARIN */
+#define ENBD_REQUEST_MAGIC 0x25609513
+#define ENBD_REPLY_MAGIC 0x67446698     
+/* Pavel - Do *not* use magics: 0x12560953 0x96744668. 
+ */
+
+#define ENBD_DEV_MAGIC 0x68797548
+
+#define ENBD_REQUEST_MAGIC_T  __u32
+#define ENBD_REQUEST_TYPE_T   __u32
+#define ENBD_HANDLE_LENGTH    sizeof(void*)      /* PTB number of bytes */
+#define ENBD_REQUEST_FROM_T   __u64
+#define ENBD_REQUEST_LEN_T    __u32
+#define ENBD_REQUEST_FLAGS_T  __u32
+#define ENBD_REQUEST_TIME_T   struct timeval
+#define ENBD_REQUEST_ZONE_T   struct timezone
+#define ENBD_REQUEST_SPECIAL_T  __u32
+
+#define ENBD_REPLY_MAGIC_T    __u32
+#define ENBD_REPLY_ERROR_T    __s32
+#define ENBD_REPLY_FLAGS_T    __u32
+#define ENBD_REPLY_TIME_T     struct timeval
+#define ENBD_REPLY_ZONE_T     struct timezone
+
+typedef char enbd_handle_t[ENBD_HANDLE_LENGTH];
+
+#define ENBD_REQUEST_HANDLE_T enbd_handle_t
+#define ENBD_REPLY_HANDLE_T   enbd_handle_t
+
+  typedef __u32 enbd_digest_t[4];
+  typedef __u32 enbd_ctldta_t[4];
+
+  #define ENBD_DIGEST_T   enbd_digest_t
+  #define ENBD_CTLDTA_T   enbd_ctldta_t
+
+#define ENBD_REQUEST_DIGEST_T enbd_digest_t
+#define ENBD_REPLY_DIGEST_T   enbd_digest_t
+
+#define ENBD_REQUEST_CTLDTA_T   ENBD_CTLDTA_T
+#define ENBD_REPLY_CTLDTA_T   ENBD_CTLDTA_T
+
+#define ENBD_DIGEST_BITS      128
+#define ENBD_DIGEST_LENGTH    ((ENBD_DIGEST_BITS)/8)
+#define ENBD_CTLDTA_LENGTH    (128/8)
+#define ENBD_REQUEST_SEQNO_T  __u32
+
+struct enbd_request {
+  ENBD_REQUEST_MAGIC_T  magic;
+  ENBD_REQUEST_TYPE_T   type;			/* == READ || == WRITE 	*/
+  ENBD_REQUEST_HANDLE_T handle;
+  ENBD_REQUEST_FROM_T   from;                    /* 64 bit PTB 132 */
+  ENBD_REQUEST_LEN_T    len;
+
+
+
+#define ENBD_REQUEST_ERRORED    0x0800
+#define ENBD_REQUEST_MD5SUM     0x8000         /* has a digest in it ..*/
+#define ENBD_REQUEST_MD5_OK    0x10000         /* .. and equaled req's */
+#define ENBD_REQUEST_IOCTL     0x40000         /* ioctl in len, arg in from */
+  ENBD_REQUEST_FLAGS_T  flags;
+  ENBD_REQUEST_TIME_T   time;
+  ENBD_REQUEST_ZONE_T   zone;
+  ENBD_REQUEST_SEQNO_T  seqno;
+  union {
+       ENBD_REQUEST_DIGEST_T digest;
+       ENBD_REQUEST_CTLDTA_T ctldta;
+  } data;
+  ENBD_REQUEST_SPECIAL_T special;
+  char dummy[0] __attribute__ ((aligned (64)));
+} __attribute__ ((packed)) ;
+
+  #define ENBD_REQUEST_LENGTH sizeof(struct enbd_request)
+
+struct enbd_reply {
+  ENBD_REPLY_MAGIC_T    magic;
+  ENBD_REPLY_ERROR_T    error;		      /* 0 = ok, else error	*/
+  ENBD_REPLY_HANDLE_T   handle;	              /* handle you got from request */
+
+
+
+#define ENBD_REPLY_ERRORED      0x0800
+#define ENBD_REPLY_MD5SUM       0x8000         /* has a digest in it .. */
+#define ENBD_REPLY_MD5_OK      0x10000         /* .. and equaled req's  */
+#define ENBD_REPLY_CLOSE       0x20000         /* close cmd from server */
+#define ENBD_REPLY_IOCTL       0x40000         /* ioctl in len, arg in from */
+  ENBD_REPLY_FLAGS_T    flags;
+  ENBD_REPLY_TIME_T     time;
+  ENBD_REPLY_ZONE_T     zone;
+  union {
+       ENBD_REPLY_DIGEST_T digest;
+       ENBD_REPLY_CTLDTA_T ctldta;
+  } data;
+  char dummy[0] __attribute__ ((aligned (64)));
+} __attribute__ ((packed)) ;
+
+  #define ENBD_REPLY_LENGTH sizeof(struct enbd_reply)
+
+  #define ENBD_BUFFER_DATA_OFFSET \
+   ((ENBD_REQUEST_LENGTH>ENBD_REPLY_LENGTH)?ENBD_REQUEST_LENGTH:ENBD_REPLY_LENGTH)
+
+#endif /* LINUX_ENBD_H */
+
+
--- linux-2.5.7/include/linux/enbd_ioctl.h.pre-enbd	Wed Apr 10 00:31:08 2002
+++ linux-2.5.7/include/linux/enbd_ioctl.h	Tue Apr  9 23:55:57 2002
@@ -0,0 +1,374 @@
+#ifndef ENBD_IOCTL_H
+#define ENBD_IOCTL_H 1
+
+int enbd_ioctl_convert(int ioctl);
+int enbd_ioctl_revert(int ioctl);
+int enbd_ioctl_size (int cmd, char *arg);
+int enbd_ioctl_size_user (int cmd, char *arg);
+#ifdef __KERNEL__
+int enbd_ioctl_copy_to_user (int cmd, char *arg, char *buf, int size);
+int enbd_ioctl_copy_from_user (int cmd, char *buf, char *arg, int size);
+#endif
+
+#endif
+#ifndef __KERNEL__
+#include <sys/time.h>
+#include <unistd.h>
+#endif
+
+#include <linux/ioctl.h>
+#include <linux/fs.h>
+#include <linux/fd.h>
+#include <linux/cdrom.h>
+#include <linux/enbd.h>
+
+// PTB conversion table entries
+struct ioctl_conv {
+   unsigned int old; // ioctl id, _IO or _IOR or _IOW or _IOWR
+   unsigned int new; // ioctl id
+};
+
+// PTB extended conversion table entries
+struct ioctl_special {
+    int new;
+    int (*size) (char *arg);
+    int (*size_user) (char *arg);
+    int (*ioctl_copy_from_user)(char *buf, char*arg, int size);
+    int (*ioctl_copy_to_user)(char *arg, char*buf, int size);
+};
+
+
+/*
+ * This is the whitelist of remote ioctls - an entry here tells the
+ * driver that it's OK to send this ioctl out over the net, because we
+ * have the right info on it.
+ *
+ * "The right info" is what is on the right hand side of the table (a 0
+ * stands for a repetition of the LHS info). We have to fixup something
+ * that a lot of kernel authors forgot to do or got worng - namely
+ * declare their ioctls in a way that conveys information about their
+ * intended mode of use (see iotcl.h in the kernel sources).
+ *
+ * We need all ioctls to be delared as either
+ *
+ *    _IO(class,id)         -- default. Means no args. The call is enough.
+ *    _IOW(class,id,type)   -- we write a value to kernel that is sizeof(type)
+ *    _IOR(class,id,type)   -- we read a value from kernel sizeof(type)
+ *    _IOWR(class,id,type)  -- ibid, but both ways
+ *
+ *  The "R" bit is crucial because it lets us know that the data is
+ *  _indirected_. I.e. it's an address of somewhere in userspace where
+ *  we want to read data from or write data to. 
+ *
+ *  The "type" part should be the type of the indirected argument, NOT
+ *  the type of its address!
+ *
+ *  Kernel authors typically make two mistakes:
+ *
+ *     1) they write _IO instead of _IOR or IOWR, and hence forget the
+ *        type info. Well, not telling me if the argument data is
+ *        direct or indirectly accessible was already bad enough!
+ *     2) they get the type argument _wrong_ when they do remember to
+ *        put it. They write "int *" instead  of "int", for example,
+ *        when the argument to the ioctl is a pointer to an integer.
+ *        OK, so it's a natural mistake to make! But in that case the
+ *        argument should be "int" so that the kernel macro picks up
+ *        sizeof(int) instead of sizeof(int*).
+ *
+ *  Those "errors" have to be repaired via this table. Wrong at left,
+ *  corrected at right. A 0 for the new entry indicates that the old
+ *  was alright. If there isn't an entry, the ioctl won't be treated.
+ *  If the size info works out at the max for the field  (2^14 - 1)
+ *  then a extra table is consulted for size and copy methods.
+ */
+
+
+// PTB the space before the final comma is important as the ##
+// discards the preceding token when D is empty
+#define _NEW_IO_(B,C,D...) C(_IOC_TYPE(B), _IOC_NR(B) , ## D)
+#define _NEW_IO(B,D...)   _IO(_IOC_TYPE(B), _IOC_NR(B) , ## D)
+#define _NEW_IOW(B,D...)  _IOW(_IOC_TYPE(B), _IOC_NR(B) , ## D)
+#define _NEW_IOR(B,D...)  _IOR(_IOC_TYPE(B), _IOC_NR(B) , ## D)
+#define _NEW_IOWR(B,D...) _IOWR(_IOC_TYPE(B), _IOC_NR(B) , ## D)
+#define _NEW_IORS(B) _IOC(_IOC_READ,_IOC_TYPE(B), _IOC_NR(B), _IOC_SIZEMASK)
+#define _NEW_IOWRS(B) _IOC(_IOC_READ|_IOC_WRITE,_IOC_TYPE(B), _IOC_NR(B), _IOC_SIZEMASK)
+
+static struct ioctl_conv ioctl_conv_tab[] = {
+// fs.h
+   { BLKROSET, _NEW_IOW(BLKROSET,int), },
+   { BLKROGET, _NEW_IOR(BLKROGET,int), },
+//#define BLKRRPART  _IO(0x12,95) /* re-read partition table */
+   { BLKRRPART,  0, },
+   { BLKGETSIZE, _NEW_IOR(BLKGETSIZE,int), },
+//#define BLKFLSBUF  _IO(0x12,97) /* flush buffer cache */
+   { BLKFLSBUF,  0, },
+#ifdef BLKRASET
+   { BLKRASET,   _NEW_IOW(BLKRASET,int), },
+#endif /* BLKRASET */
+#ifdef BLKRAGET
+   { BLKRAGET,   _NEW_IOR(BLKRAGET,int), },
+#endif /* BLKRAGET */
+#ifdef BLKFRASET
+   { BLKFRASET,  _NEW_IOW(BLKFRASET,int), },
+#endif /* BLKFRASET */
+#ifdef BLKFRAGET
+   { BLKFRAGET,  _NEW_IOR(BLKFRAGET,int), },
+#endif /* BLKFRAGET */
+   { BLKSECTSET, _NEW_IOW(BLKSECTSET,int), },
+   { BLKSECTGET, _NEW_IOR(BLKSECTGET,int), },
+   { BLKSSZGET,  _NEW_IOR(BLKSSZGET,int), },
+//  fd.h
+   { FDCLRPRM,  0, },
+   { FDSETPRM, _NEW_IOWR(FDSETPRM, struct floppy_struct), },
+   { FDDEFPRM, _NEW_IOWR(FDDEFPRM, struct floppy_struct), }, 
+   { FDGETPRM, _NEW_IOR(FDGETPRM, struct floppy_struct), }, 
+   { FDMSGON,   0, }, 
+   { FDMSGOFF,  0, }, 
+   { FDFMTBEG,  0, }, 
+   { FDFMTTRK, _NEW_IOWR(FDFMTTRK, struct format_descr), }, 
+   { FDFMTEND,  0, }, 
+   { FDSETEMSGTRESH, _NEW_IOW(FDSETEMSGTRESH, unsigned), }, 
+   { FDFLUSH,  0, }, 
+   { FDSETMAXERRS, _NEW_IOWR(FDSETMAXERRS, struct floppy_max_errors), }, 
+   { FDGETMAXERRS, _NEW_IOR(FDGETMAXERRS, struct floppy_max_errors), }, 
+   { FDGETDRVTYP, _NEW_IOR(FDGETDRVTYP, floppy_drive_name), }, // 16 bytes
+   { FDSETDRVPRM, _NEW_IOWR(FDSETDRVPRM, struct floppy_drive_params), }, 
+   { FDGETDRVPRM, _NEW_IOR(FDGETDRVPRM, struct floppy_drive_params), }, 
+   { FDGETDRVSTAT, _NEW_IOR(FDGETDRVSTAT, struct floppy_drive_struct), }, 
+   { FDPOLLDRVSTAT, _NEW_IOR(FDPOLLDRVSTAT, struct floppy_drive_struct), }, 
+   { FDRESET,  0, }, 
+   { FDGETFDCSTAT, _NEW_IOR(FDGETFDCSTAT, struct floppy_fdc_state), }, 
+   { FDWERRORCLR,  0, }, 
+   { FDWERRORGET, _NEW_IOR(FDWERRORGET, struct floppy_write_errors), }, 
+   { FDRAWCMD, _NEW_IOWR(FDRAWCMD, struct floppy_raw_cmd[1]) },  // FIXME linked list
+   { FDTWADDLE,  0, }, 
+   { FDEJECT,  0, }, 
+// cdrom.h
+   { CDROMPAUSE, _NEW_IO(CDROMPAUSE), }, 
+   { CDROMRESUME, _NEW_IO(CDROMRESUME), }, 
+   { CDROMPLAYMSF, _NEW_IOR(CDROMPLAYMSF, struct cdrom_msf), }, 
+   { CDROMPLAYTRKIND, _NEW_IOR(CDROMPLAYTRKIND, struct cdrom_ti), }, 
+   { CDROMREADTOCHDR, _NEW_IOWR(CDROMREADTOCHDR, struct cdrom_tochdr), }, 
+   { CDROMREADTOCENTRY, _NEW_IOWR(CDROMREADTOCENTRY, struct cdrom_tocentry), }, 
+   { CDROMSTOP, _NEW_IO(CDROMSTOP), }, 
+   { CDROMSTART, _NEW_IO(CDROMSTART), }, 
+   { CDROMEJECT, _NEW_IO(CDROMEJECT), }, 
+   { CDROMVOLCTRL, _NEW_IOR(CDROMVOLCTRL, struct cdrom_volctrl), }, 
+   { CDROMSUBCHNL, _NEW_IOWR(CDROMSUBCHNL, struct cdrom_subchnl), }, 
+   { CDROMREADMODE2, _NEW_IOR(CDROMREADMODE2, struct cdrom_read), },  // INDIRECT 2336B
+   { CDROMREADMODE1, _NEW_IOR(CDROMREADMODE1, struct cdrom_read), },  // INDIRECT 2048B
+   { CDROMREADAUDIO, _NEW_IOR(CDROMREADAUDIO, struct cdrom_read_audio), }, 
+   { CDROMEJECT_SW, _NEW_IO(CDROMEJECT_SW), }, 
+   { CDROMMULTISESSION, _NEW_IOWR(CDROMMULTISESSION, struct cdrom_multisession), }, 
+   { CDROM_GET_MCN, _NEW_IOWR(CDROM_GET_MCN, struct cdrom_mcn), }, 
+   { CDROMRESET, _NEW_IO(CDROMRESET), }, 
+   { CDROMVOLREAD, _NEW_IOWR(CDROMVOLREAD, struct cdrom_volctrl), }, 
+   { CDROMREADRAW, _NEW_IOR(CDROMREADRAW, struct cdrom_read), },  // INDIRECT 2352B
+   // aztcd.c optcd.c
+   { CDROMREADCOOKED, _NEW_IOR(CDROMREADCOOKED, struct cdrom_msf), }, // INDIRECT FIXME
+   { CDROMSEEK, _NEW_IOR(CDROMSEEK, struct cdrom_msf), }, 
+   // scsi-cd.c
+   { CDROMPLAYBLK, _NEW_IOWR(CDROMPLAYBLK, struct cdrom_blk), }, 
+   // optcd.c
+   { CDROMREADALL, _NEW_IOR(CDROMREADALL, char[2646]), }, 
+   // ide-cd.c
+   { CDROMGETSPINDOWN, _NEW_IOWR(CDROMGETSPINDOWN, char), },  // one byte
+   { CDROMSETSPINDOWN, _NEW_IOWR(CDROMSETSPINDOWN, char), },  // one byte
+   // cdrom.c
+   { CDROMCLOSETRAY, _NEW_IO(CDROMCLOSETRAY), },
+   { CDROM_SET_OPTIONS, _NEW_IOW(CDROM_SET_OPTIONS, int), },
+   { CDROM_CLEAR_OPTIONS, _NEW_IOW(CDROM_CLEAR_OPTIONS, int), },
+   { CDROM_SELECT_SPEED, _NEW_IOW(CDROM_SELECT_SPEED, int), }, // FIXME - don't know
+   { CDROM_SELECT_DISC, _NEW_IOW(CDROM_SELECT_DISC, int), },
+   { CDROM_MEDIA_CHANGED, _NEW_IOW(CDROM_MEDIA_CHANGED, int), },
+   { CDROM_DRIVE_STATUS, _NEW_IOW(CDROM_DRIVE_STATUS, int), },
+   { CDROM_DISC_STATUS, _NEW_IO(CDROM_DISC_STATUS), },
+   { CDROM_CHANGER_NSLOTS, _NEW_IO(CDROM_CHANGER_NSLOTS), },
+   { CDROM_LOCKDOOR, _NEW_IOW(CDROM_LOCKDOOR, int), },
+   { CDROM_DEBUG, _NEW_IOW(CDROM_DEBUG, int), },
+   { CDROM_GET_CAPABILITY, _NEW_IO(CDROM_GET_CAPABILITY), },
+   // sbpcd
+   { CDROMAUDIOBUFSIZ, _NEW_IOW(CDROMAUDIOBUFSIZ, int), },
+   // dvd
+   { DVD_READ_STRUCT, _NEW_IOR(DVD_READ_STRUCT, dvd_struct), },
+   { DVD_WRITE_STRUCT, _NEW_IOWR(DVD_WRITE_STRUCT, dvd_struct), },
+   { DVD_AUTH, _NEW_IOWR(DVD_AUTH, dvd_authinfo), },
+   { CDROM_SEND_PACKET, _NEW_IOR(CDROM_SEND_PACKET, struct cdrom_generic_command), },
+   { CDROM_NEXT_WRITABLE, _NEW_IOWR(CDROM_NEXT_WRITABLE, long), },
+   { CDROM_LAST_WRITTEN, _NEW_IOWR(CDROM_LAST_WRITTEN, long), },
+   // PTB local test ioctls
+   { ENBD_TEST_IOCTL1, 0, },   // write an int
+   { ENBD_TEST_IOCTL2, 0, },   // read an int
+   { ENBD_TEST_IOCTL3, 0, },   // write and read an int
+   { ENBD_TEST_IOCTL4, 0, },   // read 256B
+   { ENBD_TEST_IOCTL5, 0, },   // r/w 256B
+   { ENBD_TEST_IOCTL6, _NEW_IORS(ENBD_TEST_IOCTL6), },    // read special
+   { ENBD_TEST_IOCTL7, _NEW_IORS(ENBD_TEST_IOCTL7), },    // r/w special
+   // PTB we must terminate with a 0,0 entry.
+   {0 , 0, },
+};
+
+/*
+ * This should be the table of special methods for certain ioctls.
+ * The "new" code is the real index. It will have a size count of
+ * _IOC_SIZEMASK but the rest of it should be meaningful. The size is
+ * gotten by dynamic lookup using the size() function.
+ */
+static struct ioctl_special ioctl_special_tab[] = {
+    // PTB last entry must be all zeros
+    { 0, NULL, NULL, NULL, NULL, },
+};
+
+
+static struct ioctl_conv *
+ioctl_lookup_old (int ioctl)
+{
+	int i;
+        unsigned old;
+        if (ioctl == -1)
+            return NULL;
+	for (i = 0; old = ioctl_conv_tab[i].old, old; i++) {
+		if (old == ioctl)
+		        return &ioctl_conv_tab[i];
+	}
+	// PTB not there
+	return NULL;
+}
+
+int
+enbd_ioctl_convert (int ioctl)
+{
+	struct ioctl_conv *conv = ioctl_lookup_old (ioctl);
+	if (!conv)
+		// PTB not there
+		return -1;
+        return conv->new ? : ioctl;
+}
+
+static struct ioctl_conv *
+ioctl_lookup_new (int ioctl)
+{
+	int i = 0;
+	unsigned old;
+	for (i = 0; old = ioctl_conv_tab[i].old, old; i++) {
+		unsigned new = ioctl_conv_tab[i].new;
+		if (new == ioctl || (new == 0 && old == ioctl)) 
+			return &ioctl_conv_tab[i];
+	}
+	// PTB not there
+	return NULL;
+}
+
+int
+enbd_ioctl_revert (int ioctl)
+{
+	struct ioctl_conv *conv = ioctl_lookup_new (ioctl);
+	if (!conv)
+		// PTB not there
+		return -1;
+	return conv->old;
+}
+
+static struct ioctl_special *
+ioctl_special_lookup_new (int ioctl)
+{
+	int i;
+        unsigned new;
+	for (i = 0; new = ioctl_special_tab[i].new, new; i++) {
+		if (new == ioctl) 
+			return &ioctl_special_tab[i];
+	}
+	// PTB not there
+	return NULL;
+}
+
+int
+enbd_ioctl_size (int cmd, char *arg)
+{
+	int size = _IOC_SIZE (cmd);
+	if (size == _IOC_SIZEMASK) {
+		// PTB special hadling required. 
+                struct ioctl_special *special = ioctl_special_lookup_new(cmd);
+                if (!special)
+                        return -1;
+	        return special->size (arg);
+	}
+	return size;
+}
+
+int
+enbd_ioctl_size_user (int cmd, char *arg)
+{
+	int size = _IOC_SIZE (cmd);
+	if (size == _IOC_SIZEMASK) {
+		// PTB special hadling required. 
+                struct ioctl_special *special = ioctl_special_lookup_new(cmd);
+                if (!special)
+                        return -1;
+	        return special->size_user (arg);
+	}
+	return size;
+}
+
+
+#ifdef __KERNEL__
+int
+enbd_ioctl_copy_to_user (int cmd, char *arg, char *buf, int size)
+{
+
+	if (_IOC_SIZE (cmd) == _IOC_SIZEMASK) {
+                struct ioctl_special *special = ioctl_special_lookup_new(cmd);
+                if (!special)
+                        return -1;
+	        return special->ioctl_copy_to_user (arg, buf, size);
+	}
+
+	if (_IOC_DIR (cmd) & _IOC_READ) {
+		// indirect
+		copy_to_user (arg, buf, size);
+		return size;
+	}
+
+	return -1;
+}
+
+
+
+int
+enbd_ioctl_copy_from_user (int cmd, char *buf, char *arg, int size)
+{
+
+	if (_IOC_SIZE (cmd) == _IOC_SIZEMASK) {
+                struct ioctl_special *special = ioctl_special_lookup_new(cmd);
+                if (!special)
+                        return -1;
+	        return special->ioctl_copy_from_user (buf, arg, size);
+	}
+
+	if (_IOC_DIR (cmd) & _IOC_READ) {
+		// indirect
+		copy_from_user (buf, arg, size);
+		return size;
+	}
+
+	// direct
+	if (size > sizeof (arg)) {
+		return -1;
+	}
+
+	memcpy (buf, &arg, size);
+	return size;
+}
+#endif /* __KERNEL__ */
+
+
+/*
+static
+int ioctl_init(struct ioctl_conv *self, int old, int new) {
+    self->old = old;
+    self->new = new;
+    self->serialize = ioctl_serialize;
+    self->deserialize = ioctl_deserialize;
+    self->size = ioctl_size;
+}
+*/
