/* 
 * Copyright (C) 1999-2001 Peter T. Breuer <ptb@it.uc3m.es>
 */

struct timespec;
#include <sys/time.h>
#include <unistd.h>
#include <setjmp.h>
#include <signal.h>
#include <syslog.h>
#include <stdio.h>
#include <netinet/in.h>
#include "cliserv.h"
#include "alarm.h"
#include "time.h"

extern int debug_level;

#ifndef MY_NAME
#define MY_NAME "nbd/alarm"
#endif

/*
 * PTB - an initial attempt at timeouts. The idea is to set a timeout and
 * if it expires, then jump to a particular place to deal with it. The
 * code sequence is:
 *
 *     jmp_buf my_env;
 *     ...
 *     bounce((unsigned long) &my_env);
 *     if (setjmp(my_jmp)) {
 *        ... handle alarm
 *     }
 *     ...
 *     set_alarm(timeout);
 *     ...
 *     unset_alarm();    
 */
static jmp_buf * trampoline = 0;
inline void setbounce(jmp_buf *jmp) {
   trampoline = jmp;                // PTB set the trampoline
}
inline jmp_buf * getbounce() {
   return trampoline;               // PTB get the trampoline
}
void
bounce(int sig) {
   // PTB jump using trampoline
   jmp_buf * jmpbufp = trampoline;  // PTB save old trampoline dest
   trampoline = 0;                  // PTB wipe out trampoline dest
   if (jmpbufp)
     longjmp(*jmpbufp, (int)sig);   // PTB jump to old trampoline dest
}
void
set_alarm(unsigned long timeout) {
  alarm(timeout);
  if (timeout > 0) {
       // PTB sigalarm will make us fail to the trampoline return
       signal(SIGALRM, bounce); 
  } else {
       // PTB sigalarm is ignored
       signal(SIGALRM, SIG_IGN);
  }
}
void unset_alarm() {
    set_alarm(0);
}


static struct alarm * jumpoff;

/* A more sophisticated timeout interface, with luck. Thank goodness for
 * the inline keyword. Gcc had better honour it strictly!
 *
 *     struct alarm my_env;
 *     ...
 *     if (catch_alarm(&my_env, timeout)) {
 *        ... handle alarm
 *     } else {
 *        ...
 *        uncatch_alarm(&my_env);    
 *     }
 */
void uncatch_alarm(struct alarm * alarm) {
   // PTB unspool the timer chain till we unhook the alarm

   void unhook() {
      // PTB get rid of the bottom element in the timer chain

      jumpoff = jumpoff->prev;

      if (jumpoff) {
        struct timeval tv;
        setbounce(&jumpoff->jmp);
        mygettimeofday(&tv, NULL);
        // PTB overgenerous
        if (timercmp(&tv,&jumpoff->timedue,<)) {
          // reset the alarm for what's remaining of it
           long timeout = jumpoff->timedue.tv_sec - tv.tv_sec;
           set_alarm(timeout);
        } else {
          // alarm expired, so signal for it!
          kill(getpid(),SIGALRM);
          // proabably better to call bounce (SIGALRM) directly
        }
        DEBUG("unset alarm -\n");
        return;
      }
      DEBUG("unset alarm -\n");
      unset_alarm();
   }

   while (jumpoff) {
      if (jumpoff == alarm) {
         unhook();
         break;
      }
      unhook();
   }
}

int
catch_alarm(struct alarm * alarm, unsigned long timeout) {

    // chain an extra timer. This preempts previous timers
    // but when we release this timer we check the previous one
    // and signal if it expired meanwhile.

    // fill the env with the data
    if (setjmp(alarm->jmp)) {
      // caught the alarm. restore prev env
      uncatch_alarm(alarm);
      return 1;
    }
    DEBUG("set alarm %ld\n", timeout);
    // save prev env
    alarm->prev    = jumpoff;
    alarm->timeout = timeout;
    mygettimeofday(&alarm->timedue, NULL);
    alarm->timedue.tv_sec += timeout;
    // tell handler which env to use now
    setbounce(&alarm->jmp);
    jumpoff = alarm;
    set_alarm(timeout);
    return 0;
}



