
#include <stdlib.h>
#include <stdio.h>
#include <unistd.h>
#include <errno.h>
#include <asm/atomic.h>
#include <syslog.h>
#include <string.h>
#include <netinet/in.h>
#include "list.h"
#include "lock.h"
#include "hash.h"
#define MY_NAME "nbd/hash"
#include "cliserv.h"

static int hash_fn(struct nbd_hash *hash, struct hash_datum *hkey) {
    unsigned char b = hkey->dptr[0];
    return b & (hash->nbuckets - 1);
}

static int fls (int x) {
	int r = 1;

	if (!x)
		return 0;
	if (x & 0xffff0000) {
		x >>= 16;
		r += 16;
	}
	if (x & 0xff00) {
		x >>= 8;
		r += 8;
	}
	if (x & 0xf0) {
		x >>= 4;
		r += 4;
	}
	if (x & 0xc) {
		x >>= 2;
		r += 2;
	}
	if (x & 0x2) {
		x >>= 1;
		r += 1;
	}
	return r;
}

static void
hash_free (struct nbd_hash *hash, char *dptr, int dsize)
{

    struct list_head *pos;
    int zone = fls (dsize - 1) - 1;
    int offset = dptr - (char *) &hash->data[0];

    if (zone < 0)
	return;

  retry:

    list_for_each (pos, &hash->free[zone]) {

	if (hash->freecount[zone] > 1) {
            // only try merge if we won't make a hole

	    switch ((offset >> (zone + 1)) & 1) {
              // look for left-right "buddies"

	      case 1:		// left hand part
		if ((char *) pos + (1 << (zone + 1)) == dptr) {
		    // combine them
		    list_del (pos);
		    --hash->freecount[zone++];
		    DEBUG
		     ("combining %d/%d with unit size %d from zone %d rem %d\n",
		      dsize, 1 << zone, 1 << zone, zone-1,
		      hash->freecount[zone-1]);
		    dsize +=  1 << zone;
                    // move dptr backwards to start of merged area
                    offset -= 1 << zone;
		    dptr = (char *) pos;
		    goto retry;
		}
		break;

	      case 0:		// right hand part
		if (dptr + (1 << (zone + 1)) == (char *) pos) {
		    // combine them
		    list_del (pos);
		    --hash->freecount[zone++];
		    DEBUG
		     ("combining %d/%d with unit size %d from zone %d rem %d\n",
		      dsize, 1 << zone, 1 << zone, zone-1,
		      hash->freecount[zone-1]);
		    dsize += 1 << zone;
		    goto retry;
		}
		break;
	    }
        }

        // default - drop in right position. Could do this binarywise
	if (dptr < (char *) pos)
	    break;
    }

    INIT_LIST_HEAD ((struct list_head *) dptr);
    list_add ((struct list_head *) dptr, pos->prev);
    hash->freecount[zone]++;
    DEBUG ("freeing unit size %d/%d into zone %d tot %d\n",
	   dsize, 1 << (zone + 1), zone, hash->freecount[zone]);
}

static char *
hash_alloc (struct nbd_hash *hash, int dsize)
{
    struct list_head *pos;
    int zone = fls (dsize - 1) - 1;
    // dsize =   2    -> zone = 0 size = 2
    // dsize = 3-4    -> zone = 1 size = 4
    // dsize = 5-8    -> zone = 2 size = 8
    // dsize = 9-16   -> zone = 3 size = 16
    // dsize = 17-32  -> zone = 4 size = 32
    // dsize = 33-64  -> zone = 5 size = 64
    // dsize = 65-128 -> zone = 6 size = 128
    int retry = 0;

  retry:
    list_for_each (pos, &hash->free[zone]) {
        // these entries are really of size 1<<(zone+1)
        list_del (pos);
        hash->freecount[zone]--;
        DEBUG("allocating unit size %d/%d from zone %d rem %d\n",
                dsize, 1 << (zone+1), zone, hash->freecount[zone]);
        // now cut up the excess
	while (retry-- > 0) {
            DEBUG("splitting off %d in zone %d for zone %d\n",
                    1<<zone, zone, zone - 1);
	    hash_free (hash, (char*)pos + (1<<zone), 1<<zone);
            zone--;
	}
	return (char *)pos;
    }
    // no space
    if (++zone < hash->nfreebkt) {
        DEBUG("nothing in zone %d, looking in zone %d for %dB\n",
                zone - 1, zone, dsize);
        retry++;
	goto retry;
    }
    DEBUG("FAILED to allocate unit size %d (%d entries)\n", dsize, hash->count);
    return NULL;
}

static struct hash_datum hash_fetch(struct nbd_hash *hash, struct hash_datum hkey) {

    struct list_head *pos;
    int h = hash_fn(hash, &hkey);
    struct hash_datum res = { dsize: 0, dptr: NULL };

    hash->lock.down(&hash->lock);
    list_for_each(pos, &hash->hash[h]) {

        struct nbd_hash_entry * he = list_entry(pos, struct nbd_hash_entry, q);

        if (he->k.dsize != hkey.dsize)
            continue;
        if (memcmp(he->k.dptr, hkey.dptr, hkey.dsize) != 0)
            continue;
        res = he->d;
        break;
    }
    hash->lock.up(&hash->lock);
    return res;
}

static struct hash_datum hash_firstkey(struct nbd_hash *hash) {
    struct list_head * pos = &hash->used;
    struct hash_datum res = { dsize: 0, dptr: NULL };

    hash->lock.down(&hash->lock);
    if (pos->prev != pos) {
        struct nbd_hash_entry * he =
            list_entry(pos->prev, struct nbd_hash_entry, u);
        res = he->k;
    }
    hash->lock.up(&hash->lock);
    return res;
}

static struct
hash_datum hash_nextkey(struct nbd_hash *hash, struct hash_datum hkey) {
    struct list_head * pos;
    int h = hash_fn(hash, &hkey);
    struct hash_datum res = { dsize: 0, dptr: NULL };

    hash->lock.down(&hash->lock);
    list_for_each(pos, &hash->hash[h]) {

        struct nbd_hash_entry * he = list_entry(pos, struct nbd_hash_entry, q);
        struct list_head * upos;

        if (he->k.dsize != hkey.dsize)
            continue;
        if (memcmp(he->k.dptr, hkey.dptr, hkey.dsize) != 0)
            continue;
        upos = &he->u;
        if (upos->prev != &hash->used) {
            he = list_entry(upos->prev, struct nbd_hash_entry, u);
            res = he->k;
        }
        break;
    }
    hash->lock.up(&hash->lock);
    return res;
}

static int hash_delete(struct nbd_hash *hash, struct hash_datum hkey) {

    struct list_head *pos;
    int h = hash_fn(hash, &hkey);
    int err = -EINVAL;

    hash->lock.down(&hash->lock);
    list_for_each(pos, &hash->hash[h]) {

        struct nbd_hash_entry * he = list_entry(pos, struct nbd_hash_entry, q);

        if (he->k.dsize != hkey.dsize)
            continue;
        if (memcmp(he->k.dptr, hkey.dptr, hkey.dsize) != 0)
            continue;

        list_del(&he->q);
        list_del(&he->u);

        hash_free(hash, he->k.dptr, he->k.dsize);
        hash_free(hash, he->d.dptr, he->d.dsize);
        hash_free(hash, (char*)he, sizeof(*he));
        hash->count--;
        DEBUG("hashcount at %d after delete\n", hash->count);
        err = 0;
        break;
    }
    hash->lock.up(&hash->lock);
    return err;
}

//static int hash_trim(struct nbd_hash *hash) {
//
//    int n = 0;
//    struct list_head * pos = &hash->used;
//
//    DEBUG("at %d entries in %dB\n", hash->count, hash->dsize);
//
//    if (pos->prev != pos) {
//
//        struct nbd_hash_entry * he =
//            list_entry(pos->prev, struct nbd_hash_entry, u);
//
//        list_del(&he->q);
//
//        hash_free(hash, he->k.dptr, he->k.dsize);
//        hash_free(hash, he->d.dptr, he->d.dsize);
//
//        list_del(&he->u);
//        hash_free(hash, (char*)he, sizeof(*he));
//        hash->count--;
//        n++;
//    }
//    return n;
//}

static int
hash_shrink (struct nbd_hash *hash)
{
	struct hash_datum fhkey = hash->firstkey (hash);
        int count = 0;
        int reqd = hash->hiwater > 0 ? hash->count - hash->lowater : 0;
        if (reqd <= 0 && hash->count > 0)
                reqd = 1;
        if (count >= reqd) {
                DEBUG("returned from shrink with %d removals\n", count);
                return count;
        }
        if (hash->shrink_test)
	for (; fhkey.dptr; fhkey = hash->nextkey (hash, fhkey)) {
		struct hash_datum fhdata = hash->fetch (hash, fhkey);
		if (!fhdata.dptr)
			break;
		if (hash->shrink_test(fhdata.dptr)) {
			// done or timed out
			int err = hash->delete (hash, fhkey);
			if (err >= 0) {
                                if (++count >= reqd) {
                                        DEBUG("returned from shrink with %d removals\n", count);
                                        return count;
                                }
			}
		}
		// next key
	}
        if (count >= reqd) {
                DEBUG("returned from shrink with %d removals\n", count);
                return count;
        }
	// give up and drop the first available
	fhkey = hash->firstkey (hash);
	for (; fhkey.dptr; fhkey = hash->nextkey (hash, fhkey)) {
		int err = hash->delete (hash, fhkey);
		if (err >= 0) {
                        if (++count >= reqd) {
                                DEBUG("returned from shrink with %d removals\n", count);
                                return count;
                        }
		}
	}
        DEBUG("returned from shrink with %d removals\n", count);
        return count;
}

static int hash_store(struct nbd_hash *hash, struct hash_datum hkey, struct hash_datum hdata, unsigned flags) {

    struct list_head *pos;
    int h = hash_fn(hash, &hkey);
    struct nbd_hash_entry * he;

    hash->lock.down(&hash->lock);
    list_for_each(pos, &hash->hash[h]) {

        he = list_entry(pos, struct nbd_hash_entry, q);

        if (he->k.dsize != hkey.dsize)
            continue;
        if (memcmp(he->k.dptr, hkey.dptr, hkey.dsize) != 0)
            continue;
        if (flags == NBD_HASH_INSERT)
            return -1;
        // remove old data and store new
        hash_free(hash, he->d.dptr, he->d.dsize);
        // go looking for data space
        he->d.dptr = hash_alloc(hash, hdata.dsize);
        if (!he->d.dptr) {
            hash->lock.up(&hash->lock);
            return -ENOMEM;
        }
	he->d.dsize = hdata.dsize;
	memcpy (he->d.dptr, hdata.dptr, hdata.dsize);
        hash->lock.up(&hash->lock);
        return 1;
    }

    // OK - new data
    if ((hash->mode & NBD_HASH_MODE_LIMIT) && hash->count >= hash->hiwater && hash->hiwater > 0) {
        if (hash->lowater > 9 * hash->hiwater / 10) {
            PERR("lowater was too high at %d/%d\n",
                    hash->lowater, hash->hiwater);
            hash->lowater = 9 * hash->hiwater / 10;
            PERR("lowater lowered to %d/%d\n", hash->lowater, hash->hiwater);
        }
        while (hash->count >= hash->lowater) {
            if (hash_shrink(hash) < 1) {
                hash->lock.up(&hash->lock);
                break;
            }
        }
        if (hash->count >= hash->hiwater) {
                return -ENOMEM;
        }
    }
    he = (struct nbd_hash_entry *)hash_alloc(hash, sizeof(*he));
    while (!he) {
        if (hash->count < hash->hiwater) {
            int diff = (100 * (hash->hiwater - hash->lowater)) / (1 + hash->hiwater);
            hash->hiwater = hash->count;
            hash->lowater = hash->hiwater - (diff * hash->hiwater) / 100;
            if (hash->lowater <= 0) hash->lowater = hash->hiwater / 3;
        }
        if (!(hash->mode & NBD_HASH_MODE_LIMIT) || hash_shrink(hash) < 1) {
            hash->lock.up(&hash->lock);
            return -ENOMEM;
        }
        he = (struct nbd_hash_entry *)hash_alloc(hash, sizeof(*he));
    }

    he->k.dptr = hash_alloc(hash, hkey.dsize);
    while (!he->k.dptr) {
        if (hash->count < hash->hiwater) {
            int diff = (100 * (hash->hiwater - hash->lowater)) / (1 + hash->hiwater);
            hash->hiwater = hash->count;
            hash->lowater = hash->hiwater - (diff * hash->hiwater) / 100;
            if (hash->lowater <= 0) hash->lowater = hash->hiwater / 3;
        }
        if (!(hash->mode & NBD_HASH_MODE_LIMIT) || hash_shrink(hash) < 1) {
            hash_free (hash, (char *)he, sizeof(*he));
            hash->lock.up(&hash->lock);
            return -ENOMEM;
        }
        he->k.dptr = hash_alloc(hash, hkey.dsize);
    }
    he->k.dsize = hkey.dsize;
    memcpy (he->k.dptr, hkey.dptr, hkey.dsize);

    he->d.dptr = hash_alloc(hash, hdata.dsize);
    while (!he->d.dptr) {
        if (hash->count < hash->hiwater) {
            int diff = (100 * (hash->hiwater - hash->lowater)) / (1 + hash->hiwater);
            hash->hiwater = hash->count;
            hash->lowater = hash->hiwater - (diff * hash->hiwater) / 100;
            if (hash->lowater < 0) hash->lowater = hash->hiwater / 3;
        }
        if (!(hash->mode & NBD_HASH_MODE_LIMIT) || hash_shrink(hash) < 1) {
            hash_free (hash, he->k.dptr, he->k.dsize);
            hash_free (hash, (char *)he, sizeof(*he));
            hash->lock.up(&hash->lock);
            return -ENOMEM;
        }
        he->d.dptr = hash_alloc(hash, hdata.dsize);
    }
    he->d.dsize = hdata.dsize;
    memcpy (he->d.dptr, hdata.dptr, hdata.dsize);

    list_add(&he->q, &hash->hash[h]);
    list_add(&he->u, &hash->used);
    ++hash->count;
    if (hash->count > hash->max) {
        DEBUG("hashcount max at %d entries after store\n", hash->count);
        hash->max = hash->count;
    }
    DEBUG("hashcount at %d after store\n", hash->count);
    hash->lock.up(&hash->lock);
    return 0;
}

static int
hash_register_shrink_test(struct nbd_hash *hash, int (*test)(void *)) {
    hash->lock.down(&hash->lock);
    hash->shrink_test = test;
    hash->lock.up(&hash->lock);
    return 0;
}

static int
hash_unregister_shrink_test(struct nbd_hash *hash) {
    hash->lock.down(&hash->lock);
    hash->shrink_test = NULL;
    hash->lock.up(&hash->lock);
    return 0;
}

void init_hash(struct nbd_hash *hash, int size, int hfd, int off, int limit, unsigned mode) {

    /*
     * PTB size is the total area we have to live in. We have to take away
     * the infrastructure costs from that in order to get the area
     * dsize available to us for data.
     */

    int zone, i, dsize;
    char * dptr;

    hash->mode = mode;
    hash->hfd = hfd;
    hash->off = off;
    hash->nbuckets = sizeof(hash->hash) / sizeof(hash->hash[0]); // 64
    hash->nfreebkt = sizeof(hash->free) / sizeof(hash->free[0]); // 32
    hash->hiwater = limit;
    hash->lowater = limit > 10 ? limit - 10 : limit / 3;
    hash->dsize = size - ((char *)&hash->data[0] - (char *)hash);
    hash->fetch = hash_fetch;
    hash->store = hash_store;
    hash->delete = hash_delete;
    hash->firstkey = hash_firstkey;
    hash->nextkey = hash_nextkey;
    //hash->shrink = hash_shrink;
    hash->shrink_test = NULL;
    hash->register_shrink_test = hash_register_shrink_test;
    hash->unregister_shrink_test = hash_unregister_shrink_test;
    INIT_LIST_HEAD(&hash->used);

    for (i = 0; i < hash->nfreebkt; i++) {
        INIT_LIST_HEAD(&hash->free[i]);
        hash->freecount[i] = 0;
    }
    for (i = 0; i < hash->nbuckets; i++) {
        INIT_LIST_HEAD(&hash->hash[i]);
    }

    dsize= hash->dsize;
    dptr = (char *)&hash->data[0];
    zone = fls (hash->dsize) - 1;

    while (zone >= 0 && dsize >= sizeof(struct list_head)) {
        MSG("pre-seeding hash heap with unit size %d\n", 1 << zone);
        hash_free(hash, dptr, 1 << zone);

        dptr += 1 << zone;
        dsize -= 1 << zone;
        zone = fls (dsize) - 1;
    }
    hash->count = 0;
    init_lock(&hash->lock, hfd, off + ((char *)&hash->lock - (char *)hash));
}
