/* 
 * Copyright (C) 1999-2001 Peter T. Breuer <ptb@it.uc3m.es>
 */


#include <net/if.h>             /* ifreq, IFNAMSIZ */
#include <string.h>             /* strncpy */
#include <sys/socket.h>         /* inet_aton */
#include <netinet/in.h>         /* inet_aton */
#include <arpa/inet.h>          /* inet_aton */
#include <unistd.h>             /* close */
#include <sys/ioctl.h>          /* ioctl */
#include <errno.h>              /* errno */
#include <malloc.h>             /* free */

#include "config.h"
#include "interface.h"


// PTB compatibility stuff from glibc2 for glibc1 
#ifndef HAVE_IF_NAMEINDEX

/* Variable to signal whether SIOCGIFCONF is not available.  */
static int old_siocgifconf;

/* Return a socket of any type.  The socket can be used in subsequent
   ioctl calls to talk to the kernel.  */
static int
__opensock (void) {

  /* Cache the last AF that worked, to avoid many redundant calls to
     socket().  */
  static int sock_af = -1;
  int fd = -1;

  if (sock_af != -1) {
      fd = socket (sock_af, SOCK_DGRAM, 0);
      if (fd != -1)
        return fd;
  }
  if (sock_af != -1)
    fd = socket (sock_af, SOCK_DGRAM, 0);

  if (fd == -1) {
#ifdef AF_INET
      fd = socket (sock_af = AF_INET, SOCK_DGRAM, 0);
#endif
#ifdef AF_INET6
      if (fd < 0)
        fd = socket (sock_af = AF_INET6, SOCK_DGRAM, 0);
#endif
#ifdef AF_IPX
      if (fd < 0)
        fd = socket (sock_af = AF_IPX, SOCK_DGRAM, 0);
#endif
#ifdef AF_AX25
      if (fd < 0)
        fd = socket (sock_af = AF_AX25, SOCK_DGRAM, 0);
#endif
#ifdef AF_APPLETALK
      if (fd < 0)
        fd = socket (sock_af = AF_APPLETALK, SOCK_DGRAM, 0);
#endif
  }
  return fd;
}


unsigned int
if_nametoindex (const char *ifname) {

  struct ifreq ifr;
  int fd = __opensock ();

  if (fd < 0)
    return 0;
  strncpy (ifr.ifr_name, ifname, sizeof (ifr.ifr_name));
  if (ioctl (fd, SIOCGIFINDEX, &ifr) < 0) {
      int saved_errno = errno;
      close (fd);
      if (saved_errno == EINVAL)
        errno = ENOSYS;
      return 0;
  }
  close (fd);
  return ifr.ifr_ifindex;
}

void
if_freenameindex (struct if_nameindex *ifn) {

  struct if_nameindex *ptr = ifn;

  while (ptr->if_name || ptr->if_index) {
      if (ptr->if_name)
        free (ptr->if_name);
      ++ptr;
  }
  free (ifn);
}

struct if_nameindex *
if_nameindex (void) {

  int fd = __opensock ();
  struct ifconf ifc;
  unsigned int nifs, i;
  int rq_len;
  struct if_nameindex *idx = NULL;

# define RQ_IFS 4
  if (fd < 0)
    return NULL;
  ifc.ifc_buf = NULL;
  /* We may be able to get the needed buffer size directly, rather than
     guessing.  */
  if (! old_siocgifconf) {
      ifc.ifc_buf = NULL;
      ifc.ifc_len = 0;
      if (ioctl (fd, SIOCGIFCONF, &ifc) < 0 || ifc.ifc_len == 0) {

          old_siocgifconf = 1;
          rq_len = RQ_IFS * sizeof (struct ifreq);
        }
      else
        rq_len = ifc.ifc_len;
  } else
    rq_len = RQ_IFS * sizeof (struct ifreq);

  /* Read all the interfaces out of the kernel.  */
  do {
      ifc.ifc_buf = alloca (ifc.ifc_len = rq_len);
      if (ifc.ifc_buf == NULL || ioctl (fd, SIOCGIFCONF, &ifc) < 0) {
          close (fd);
          return NULL;
      }
      rq_len *= 2;
    } while (ifc.ifc_len == rq_len && old_siocgifconf);

  nifs = ifc.ifc_len / sizeof (struct ifreq);
  idx = malloc ((nifs + 1) * sizeof (struct if_nameindex));
  if (idx == NULL) {
      close (fd);
      errno = ENOBUFS;
      return NULL;
  }
  for (i = 0; i < nifs; ++i) {

      struct ifreq *ifr = &ifc.ifc_req[i];

      idx[i].if_name = strdup (ifr->ifr_name);
      if (idx[i].if_name == NULL || ioctl (fd, SIOCGIFINDEX, ifr) < 0) {

          int saved_errno = errno;
          unsigned int j;

          for (j =  0; j < i; ++j)
            free (idx[j].if_name);
          free (idx);
          close (fd);
          if (saved_errno == EINVAL)
            saved_errno = ENOSYS;
          else if (saved_errno == ENOMEM)
            saved_errno = ENOBUFS;
          errno = saved_errno;
          return NULL;
      }
      idx[i].if_index = ifr->ifr_ifindex;
  }
  idx[i].if_index = 0;
  idx[i].if_name = NULL;
  close (fd);
  return idx;
}

#endif // ! HAVE_IF_NAMEINDEX


  static int INET_rresolve(char *name, size_t len, struct sockaddr_in *sin) {
      //unsigned long ad = (unsigned long) sin->sin_addr.s_addr;
      strncpy(name, inet_ntoa(sin->sin_addr), len);
      return 0;
  }
  /* Display an Internet socket address. */
  static char *INET_sprint(struct interface *ife) {
      struct sockaddr * sap = &ife->addr;
      static char buff[128];
      struct sockaddr_in * sap_in = (struct sockaddr_in *)sap;
      int err = INET_rresolve(buff, sizeof(buff), sap_in);
      if (err != 0)
        return NULL;
      return buff;
  }
  /* Display an Internet socket netmask. */
  static char *INET_sprintmask(struct interface *ife) {
      struct sockaddr * sap = &ife->netmask;
      static char buff[128];
      struct sockaddr_in * sap_in = (struct sockaddr_in *)sap;
      if (INET_rresolve(buff, sizeof(buff), sap_in) != 0)
         return NULL;
      return buff;
  }
  /* Display an Internet socket address. */
  static unsigned long INET_addr(struct interface *ife) {
    char * buf =  ife->sprintaddr(ife);
    struct in_addr in;
    if (!buf)
      return -1;
    //PERR("INET_print looks at %s\n", buf);
    if (!inet_aton(buf,&in))
      return -1;
    return in.s_addr;
  }
  /* Display an Internet socket netmask. */
  static unsigned long INET_mask(struct interface *ife) {
    char * buf =  ife->sprintmask(ife);
    struct in_addr in;
    if (!buf)
      return -1;
    //PERR("INET_printmask looks at %s\n", buf);
    if (!inet_aton(buf,&in))
      return -1;
    return in.s_addr;
  }
  static char * INET_name(struct interface *ife) {
    return (char*)ife->name;
  }

  static int
  get_addr(struct interface *ife) {
    struct ifreq ifr;
    ifr.ifr_addr.sa_family = AF_INET;
    strncpy(ifr.ifr_name, ife->name, sizeof(ifr.ifr_name));
    if (ioctl(ife->fd, SIOCGIFADDR, &ifr) < 0) {
      ife->has_ip = 0;
      memset(&ife->addr, 0, sizeof(struct sockaddr));
      //MSG("server (%d) finds if %s down\n", self->i, ifname);
      return -1;
    }
    ife->has_ip = 1;
    ife->addr = ifr.ifr_addr;
    //MSG("server (%d) finds if %s src addr %s\n", self->i,
    //        ifname, INET_sprint(&ife->addr));
    return 0;
  }
  static int
  get_dstaddr(struct interface *ife) {
    struct ifreq ifr;
    ifr.ifr_addr.sa_family = AF_INET;
    strncpy(ifr.ifr_name, ife->name, sizeof(ifr.ifr_name));
    if (ioctl(ife->fd, SIOCGIFDSTADDR, &ifr) < 0) {
       memset(&ife->dstaddr, 0, sizeof(struct sockaddr));
       return -1;
    } 
    ife->dstaddr = ifr.ifr_dstaddr;
    //MSG("server (%d) finds if %s dest addr %s\n", self->i,
    //   ifname, INET_sprint(&ife->dstaddr));
    return 0;
  }
  static int
  get_brdaddr(struct interface *ife) {
    struct ifreq ifr;
    ifr.ifr_addr.sa_family = AF_INET;
    strncpy(ifr.ifr_name, ife->name, sizeof(ifr.ifr_name));
    if (ioctl(ife->fd, SIOCGIFBRDADDR, &ifr) < 0) {
       memset(&ife->broadaddr, 0, sizeof(struct sockaddr));
       return -1;
    } 
    ife->broadaddr = ifr.ifr_broadaddr;
    //MSG("server (%d) finds if %s bcst addr %s\n", self->i,
    //   ifname, INET_sprint(&ife->broadaddr));
    return 0;
  }
  static int
  get_netmask(struct interface *ife) {
    struct ifreq ifr;
    ifr.ifr_addr.sa_family = AF_INET;
    strncpy(ifr.ifr_name, ife->name, sizeof(ifr.ifr_name));
    if (ioctl(ife->fd, SIOCGIFNETMASK, &ifr) < 0) {
       memset(&ife->netmask, 0, sizeof(struct sockaddr));
       return -1;
    } 
    ife->netmask = ifr.ifr_netmask;
    //MSG("server (%d) finds if %s netmask %s\n", self->i,
    //    ifname, INET_sprintmask(&ife->netmask));
    return 0;
  }
  static int
  get_metric(struct interface *ife) {
    struct ifreq ifr;
    ifr.ifr_addr.sa_family = AF_INET;
    strncpy(ifr.ifr_name, ife->name, sizeof(ifr.ifr_name));
    if (ioctl(ife->fd, SIOCGIFMETRIC, &ifr) < 0) {
       ife->metric = -1;
       return -1;
    } 
    ife->metric = ifr.ifr_metric;
    return 0;
  }
  static int
  get_mtu(struct interface *ife) {
    struct ifreq ifr;
    ifr.ifr_addr.sa_family = AF_INET;
    strncpy(ifr.ifr_name, ife->name, sizeof(ifr.ifr_name));
    if (ioctl(ife->fd, SIOCGIFMTU, &ifr) < 0) {
       ife->mtu = -1;
       return -1;
    } 
    ife->mtu = ifr.ifr_mtu;
    return 0;
  }
  static int
  get_hwaddr(struct interface *ife) {
    struct ifreq ifr;
    ifr.ifr_addr.sa_family = AF_INET;
    strncpy(ifr.ifr_name, ife->name, sizeof(ifr.ifr_name));
    if (ioctl(ife->fd, SIOCGIFHWADDR, &ifr) < 0) {
       memset(ife->hwaddr, 0, sizeof(ife->hwaddr));
       return -1;
    } 
    // FIXME .. only up to 32 bytes, sa_family + 14 bytes sa_data
    *(struct sockaddr *)ife->hwaddr = ifr.ifr_hwaddr;
    return 0;
  }
    
  /*
   * return -1 for no more interfaces, 0 for an interface
   * and check ife->has_ip afterwards to see if it was up
   *
   * FIXME .. what happens when we have an IPX interface? Do we manage
   * to skip it OK?
   */
  static int
  next_interface(struct interface *ife) {

    struct if_nameindex * ifc;
    char *ifname;
    int ifidx;

    if (!ife->ifs || ife->fd < 0 || ife->i < 0)
      return -1;

    // initialize locals
    ifc    = ife->ifs + ife->i++;
    ifname = ifc->if_name;
    ifidx  = ifc->if_index;

    if (ifidx <= 0 || !ifname) {
       // reached the end
       ife->close(ife);
       return -1; // got no interface
    }

    //MSG("server (%d) found interface #%d named %s\n", self->i, ifidx, ifname);
    strncpy(ife->name,ifname,sizeof(ife->name));

    if (get_addr(ife) < 0) 
      return 0;   // has_ip = 0

    // has_ip = 1

    get_dstaddr(ife);
    get_brdaddr(ife);
    get_netmask(ife);
    get_mtu(ife);
    get_metric(ife);
    get_hwaddr(ife);

    return 0;
  }

  static int rewind_interface(struct interface * ife) {
    if (!ife->ifs || ife->fd < 0 || ife->i < 0)
      return -1;
    ife->i = 0;
    return 0;
  }

  static int open_interface(struct interface * ife) {

    if (!ife->ifs) {
      ife->ifs = if_nameindex();
      if (!ife->ifs)
         return -1;
    }
    if (ife->fd < 0) {
      ife->fd  = socket(AF_INET, SOCK_DGRAM, 0);
      if (ife->fd < 0)
         return -1;
    }
    ife->i = 0;

    return 0;
  }

  static int close_interface(struct interface * ife) {

    int errs = 0;
    if (ife->ifs)
      if_freenameindex(ife->ifs);
    ife->ifs = NULL;

    if (ife->fd >= 0)
      if (close(ife->fd) < 0)
        errs++;
    ife->fd = -1;

    ife->i = -1;

    return -errs;
  }

  void init_interface(struct interface * ife) {

    ife->ifs = NULL;
    ife->fd  = -1;
    ife->i   = 0;
    ife->has_ip
             = 0;

    ife->next= next_interface;
    ife->getaddr
             = INET_addr;
    ife->getmask
             = INET_mask;
    ife->sprintaddr
             = INET_sprint;
    ife->sprintmask
             = INET_sprintmask;
    ife->getname
             = INET_name;
    ife->open
             = open_interface;
    ife->close
             = close_interface;
    ife->rewind
             = rewind_interface;
  }

/*
 * Gaaaaah. Have to also go through the routing table because if there's
 * a gateway in our ifmask, then effectively we have the netmask of the route
 * to the gateway, not the ifmask of our interface.
 */
