#ifndef __KERNEL__
struct timespec;
#include <sys/time.h>
#include <unistd.h>
#endif

#include <sys/types.h>
#include <linux/ioctl.h>

//#include <linux/fs.h> // BLK ioctls
#define BLKROSET   _IO(0x12,93) /* set device read-only (0 = read-write) */
#define BLKROGET   _IO(0x12,94) /* get read-only status (0 = read_write) */
#define BLKRRPART  _IO(0x12,95) /* re-read partition table */
#define BLKGETSIZE _IO(0x12,96) /* return device size /512 (long *arg) */
#define BLKFLSBUF  _IO(0x12,97) /* flush buffer cache */
#define BLKRASET   _IO(0x12,98) /* set read ahead for block device */
#define BLKRAGET   _IO(0x12,99) /* get current read ahead setting */
#define BLKFRASET  _IO(0x12,100)/* set filesystem (mm/filemap.c) read-ahead */
#define BLKFRAGET  _IO(0x12,101)/* get filesystem (mm/filemap.c) read-ahead */
#define BLKSECTSET _IO(0x12,102)/* set max sectors per request (ll_rw_blk.c) */
#define BLKSECTGET _IO(0x12,103)/* get max sectors per request (ll_rw_blk.c) */
#define BLKSSZGET  _IO(0x12,104)/* get block device sector size */
#define BLKPG      _IO(0x12,105)/* See blkpg.h */
#define BLKELVGET  _IOR(0x12,106,size_t)/* elevator get */
#define BLKELVSET  _IOW(0x12,107,size_t)/* elevator set */
#define BLKBSZGET  _IOR(0x12,112,size_t)
#define BLKBSZSET  _IOW(0x12,113,size_t)
#define BLKGETSIZE64 _IOR(0x12,114,size_t)      /* return device size in bytes (u64 *arg) */
#define BMAP_IOCTL 1            /* obsolete - kept for compatibility */
#define FIBMAP     _IO(0x00,1)  /* bmap access */
#define FIGETBSZ   _IO(0x00,2)  /* get the block size used for bmap */

#include <linux/fd.h> // floppy ioctls
#ifndef _CADDR_T
#define caddr_t char*
#endif
#include <linux/enbd.h>

/*
 * From the GCC manual:
 *
 * Many functions do not examine any values except their arguments,
 * and have no effects except the return value.  Basically this is
 * just slightly more strict class than the `pure' attribute above,
 * since function is not allowed to read global memory.
 *
 * Note that a function that has pointer arguments and examines the
 * data pointed to must _not_ be declared `const'.  Likewise, a
 * function that calls a non-`const' function usually must not be
 * `const'.  It does not make sense for a `const' function to return
 * `void'.
 */
#ifndef __attribute_const__
# define __attribute_const__    /* unimplemented */
#endif


#include <linux/cdrom.h>


// PTB conversion table entries
struct ioctl_conv {
   unsigned int old; // ioctl id, _IO or _IOR or _IOW or _IOWR
   unsigned int new; // ioctl id
};

// PTB extended conversion table entries
struct ioctl_special {
    int new;
    int (*size) (char *arg);
    int (*size_user) (char *arg);
    int (*ioctl_copy_from_user)(char *buf, char*arg, int size);
    int (*ioctl_copy_to_user)(char *arg, char*buf, int size);
};


/*
 * This is the whitelist of remote ioctls - an entry here tells the
 * driver that it's OK to send this ioctl out over the net, because we
 * have the right info on it.
 *
 * "The right info" is what is on the right hand side of the table (a 0
 * stands for a repetition of the LHS info). We have to fixup something
 * that a lot of kernel authors forgot to do or got worng - namely
 * declare their ioctls in a way that conveys information about their
 * intended mode of use (see iotcl.h in the kernel sources).
 *
 * We need all ioctls to be delared as either
 *
 *    _IO(class,id)         -- default. Means no args. The call is enough.
 *    _IOW(class,id,type)   -- we write a value to kernel that is sizeof(type)
 *    _IOR(class,id,type)   -- we read a value from kernel sizeof(type)
 *    _IOWR(class,id,type)  -- ibid, but both ways
 *
 *  The "R" bit is crucial because it lets us know that the data is
 *  _indirected_. I.e. it's an address of somewhere in userspace where
 *  we want to read data from or write data to. 
 *
 *  The "type" part should be the type of the indirected argument, NOT
 *  the type of its address!
 *
 *  Kernel authors typically make two mistakes:
 *
 *     1) they write _IO instead of _IOR or IOWR, and hence forget the
 *        type info. Well, not telling me if the argument data is
 *        direct or indirectly accessible was already bad enough!
 *     2) they get the type argument _wrong_ when they do remember to
 *        put it. They write "int *" instead  of "int", for example,
 *        when the argument to the ioctl is a pointer to an integer.
 *        OK, so it's a natural mistake to make! But in that case the
 *        argument should be "int" so that the kernel macro picks up
 *        sizeof(int) instead of sizeof(int*).
 *
 *  Those "errors" have to be repaired via this table. Wrong at left,
 *  corrected at right. A 0 for the new entry indicates that the old
 *  was alright. If there isn't an entry, the ioctl won't be treated.
 *  If the size info works out at the max for the field  (2^14 - 1)
 *  then a extra table is consulted for size and copy methods.
 */


// PTB the space before the final comma is important as the ##
// discards the preceding token when D is empty
#define _NEW_IO_(B,C,D...) C(_IOC_TYPE(B), _IOC_NR(B) , ## D)
#define _NEW_IO(B,D...)   _IO(_IOC_TYPE(B), _IOC_NR(B) , ## D)
#define _NEW_IOW(B,D...)  _IOW(_IOC_TYPE(B), _IOC_NR(B) , ## D)
#define _NEW_IOR(B,D...)  _IOR(_IOC_TYPE(B), _IOC_NR(B) , ## D)
#define _NEW_IOWR(B,D...) _IOWR(_IOC_TYPE(B), _IOC_NR(B) , ## D)
#define _NEW_IORS(B) _IOC(_IOC_READ,_IOC_TYPE(B), _IOC_NR(B), _IOC_SIZEMASK)
#define _NEW_IOWRS(B) _IOC(_IOC_READ|_IOC_WRITE,_IOC_TYPE(B), _IOC_NR(B), _IOC_SIZEMASK)

static struct ioctl_conv ioctl_conv_tab[] = {
// fs.h
   { BLKROSET, _NEW_IOW(BLKROSET,int), },
   { BLKROGET, _NEW_IOR(BLKROGET,int), },
//#define BLKRRPART  _IO(0x12,95) /* re-read partition table */
   { BLKRRPART,  0, },
   { BLKGETSIZE, _NEW_IOR(BLKGETSIZE,int), },
//#define BLKFLSBUF  _IO(0x12,97) /* flush buffer cache */
   { BLKFLSBUF,  0, },
   { BLKRASET,   _NEW_IOW(BLKRASET,int), },
   { BLKRAGET,   _NEW_IOR(BLKRAGET,int), },
   { BLKFRASET,  _NEW_IOW(BLKFRASET,int), },
   { BLKFRAGET,  _NEW_IOR(BLKFRAGET,int), },
   { BLKSECTSET, _NEW_IOW(BLKSECTSET,int), },
   { BLKSECTGET, _NEW_IOR(BLKSECTGET,int), },
   { BLKSSZGET,  _NEW_IOR(BLKSSZGET,int), },
//  fd.h
   { FDCLRPRM,  0, },
   { FDSETPRM, _NEW_IOWR(FDSETPRM, struct floppy_struct), },
   { FDDEFPRM, _NEW_IOWR(FDDEFPRM, struct floppy_struct), }, 
   { FDGETPRM, _NEW_IOR(FDGETPRM, struct floppy_struct), }, 
   { FDMSGON,   0, }, 
   { FDMSGOFF,  0, }, 
   { FDFMTBEG,  0, }, 
   { FDFMTTRK, _NEW_IOWR(FDFMTTRK, struct format_descr), }, 
   { FDFMTEND,  0, }, 
   { FDSETEMSGTRESH, _NEW_IOW(FDSETEMSGTRESH, unsigned), }, 
   { FDFLUSH,  0, }, 
   { FDSETMAXERRS, _NEW_IOWR(FDSETMAXERRS, struct floppy_max_errors), }, 
   { FDGETMAXERRS, _NEW_IOR(FDGETMAXERRS, struct floppy_max_errors), }, 
   { FDGETDRVTYP, _NEW_IOR(FDGETDRVTYP, floppy_drive_name), }, // 16 bytes
   { FDSETDRVPRM, _NEW_IOWR(FDSETDRVPRM, struct floppy_drive_params), }, 
   { FDGETDRVPRM, _NEW_IOR(FDGETDRVPRM, struct floppy_drive_params), }, 
   { FDGETDRVSTAT, _NEW_IOR(FDGETDRVSTAT, struct floppy_drive_struct), }, 
   { FDPOLLDRVSTAT, _NEW_IOR(FDPOLLDRVSTAT, struct floppy_drive_struct), }, 
   { FDRESET,  0, }, 
   { FDGETFDCSTAT, _NEW_IOR(FDGETFDCSTAT, struct floppy_fdc_state), }, 
   { FDWERRORCLR,  0, }, 
   { FDWERRORGET, _NEW_IOR(FDWERRORGET, struct floppy_write_errors), }, 
   { FDRAWCMD, _NEW_IOWR(FDRAWCMD, struct floppy_raw_cmd[1]) },  // FIXME linked list
   { FDTWADDLE,  0, }, 
   { FDEJECT,  0, }, 
// cdrom.h
   { CDROMPAUSE, _NEW_IO(CDROMPAUSE), }, 
   { CDROMRESUME, _NEW_IO(CDROMRESUME), }, 
   { CDROMPLAYMSF, _NEW_IOR(CDROMPLAYMSF, struct cdrom_msf), }, 
   { CDROMPLAYTRKIND, _NEW_IOR(CDROMPLAYTRKIND, struct cdrom_ti), }, 
   { CDROMREADTOCHDR, _NEW_IOWR(CDROMREADTOCHDR, struct cdrom_tochdr), }, 
   { CDROMREADTOCENTRY, _NEW_IOWR(CDROMREADTOCENTRY, struct cdrom_tocentry), }, 
   { CDROMSTOP, _NEW_IO(CDROMSTOP), }, 
   { CDROMSTART, _NEW_IO(CDROMSTART), }, 
   { CDROMEJECT, _NEW_IO(CDROMEJECT), }, 
   { CDROMVOLCTRL, _NEW_IOR(CDROMVOLCTRL, struct cdrom_volctrl), }, 
   { CDROMSUBCHNL, _NEW_IOWR(CDROMSUBCHNL, struct cdrom_subchnl), }, 
   { CDROMREADMODE2, _NEW_IOR(CDROMREADMODE2, struct cdrom_read), },  // INDIRECT 2336B
   { CDROMREADMODE1, _NEW_IOR(CDROMREADMODE1, struct cdrom_read), },  // INDIRECT 2048B
   { CDROMREADAUDIO, _NEW_IOR(CDROMREADAUDIO, struct cdrom_read_audio), }, 
   { CDROMEJECT_SW, _NEW_IO(CDROMEJECT_SW), }, 
   { CDROMMULTISESSION, _NEW_IOWR(CDROMMULTISESSION, struct cdrom_multisession), }, 
   { CDROM_GET_MCN, _NEW_IOWR(CDROM_GET_MCN, struct cdrom_mcn), }, 
   { CDROMRESET, _NEW_IO(CDROMRESET), }, 
   { CDROMVOLREAD, _NEW_IOWR(CDROMVOLREAD, struct cdrom_volctrl), }, 
   { CDROMREADRAW, _NEW_IOR(CDROMREADRAW, struct cdrom_read), },  // INDIRECT 2352B
   // aztcd.c optcd.c
   { CDROMREADCOOKED, _NEW_IOR(CDROMREADCOOKED, struct cdrom_msf), }, // INDIRECT FIXME
   { CDROMSEEK, _NEW_IOR(CDROMSEEK, struct cdrom_msf), }, 
   // scsi-cd.c
   { CDROMPLAYBLK, _NEW_IOWR(CDROMPLAYBLK, struct cdrom_blk), }, 
   // optcd.c
   { CDROMREADALL, _NEW_IOR(CDROMREADALL, char[2646]), }, 
   // ide-cd.c
#ifdef CDROMGETSPINDOWN
   { CDROMGETSPINDOWN, _NEW_IOWR(CDROMGETSPINDOWN, char), },  // one byte
#endif
#ifdef CDROMSETSPINDOWN
   { CDROMSETSPINDOWN, _NEW_IOWR(CDROMSETSPINDOWN, char), },  // one byte
#endif
   // cdrom.c
   { CDROMCLOSETRAY, _NEW_IO(CDROMCLOSETRAY), },
   { CDROM_SET_OPTIONS, _NEW_IOW(CDROM_SET_OPTIONS, int), },
   { CDROM_CLEAR_OPTIONS, _NEW_IOW(CDROM_CLEAR_OPTIONS, int), },
   { CDROM_SELECT_SPEED, _NEW_IOW(CDROM_SELECT_SPEED, int), }, // FIXME - don't know
   { CDROM_SELECT_DISC, _NEW_IOW(CDROM_SELECT_DISC, int), },
   { CDROM_MEDIA_CHANGED, _NEW_IOW(CDROM_MEDIA_CHANGED, int), },
   { CDROM_DRIVE_STATUS, _NEW_IOW(CDROM_DRIVE_STATUS, int), },
   { CDROM_DISC_STATUS, _NEW_IO(CDROM_DISC_STATUS), },
   { CDROM_CHANGER_NSLOTS, _NEW_IO(CDROM_CHANGER_NSLOTS), },
   { CDROM_LOCKDOOR, _NEW_IOW(CDROM_LOCKDOOR, int), },
   { CDROM_DEBUG, _NEW_IOW(CDROM_DEBUG, int), },
   { CDROM_GET_CAPABILITY, _NEW_IO(CDROM_GET_CAPABILITY), },
   // sbpcd
   { CDROMAUDIOBUFSIZ, _NEW_IOW(CDROMAUDIOBUFSIZ, int), },
   // dvd
#ifdef DVD_READ_STRUCT
   { DVD_READ_STRUCT, _NEW_IOR(DVD_READ_STRUCT, dvd_struct), },
#endif
#ifdef DVD_WRITE_STRUCT
   { DVD_WRITE_STRUCT, _NEW_IOWR(DVD_WRITE_STRUCT, dvd_struct), },
#endif
#ifdef DVD_AUTH
   { DVD_AUTH, _NEW_IOWR(DVD_AUTH, dvd_authinfo), },
#endif
#ifdef CDROM_SEND_PACKET
   { CDROM_SEND_PACKET, _NEW_IOR(CDROM_SEND_PACKET, struct cdrom_generic_command), },
#endif
#ifdef CDROM_NET_WRITABLE
   { CDROM_NEXT_WRITABLE, _NEW_IOWR(CDROM_NEXT_WRITABLE, long), },
#endif
#ifdef CDROM_LAST_WRITTEN
   { CDROM_LAST_WRITTEN, _NEW_IOWR(CDROM_LAST_WRITTEN, long), },
#endif
   // PTB local test ioctls
   { ENBD_TEST_IOCTL1, 0, },   // write an int
   { ENBD_TEST_IOCTL2, 0, },   // read an int
   { ENBD_TEST_IOCTL3, 0, },   // write and read an int
   { ENBD_TEST_IOCTL4, 0, },   // read 256B
   { ENBD_TEST_IOCTL5, 0, },   // r/w 256B
   { ENBD_TEST_IOCTL6, _NEW_IORS(ENBD_TEST_IOCTL6), },    // read special
   { ENBD_TEST_IOCTL7, _NEW_IORS(ENBD_TEST_IOCTL7), },    // r/w special
   // PTB we must terminate with a 0,0 entry.
   {0 , 0, },
};

/*
 * This should be the table of special methods for certain ioctls.
 * The "new" code is the real index. It will have a size count of
 * _IOC_SIZEMASK but the rest of it should be meaningful. The size is
 * gotten by dynamic lookup using the size() function.
 */
static struct ioctl_special ioctl_special_tab[] = {
    // PTB last entry must be all zeros
    { 0, NULL, NULL, NULL, NULL, },
};


static struct ioctl_conv *
ioctl_lookup_old (int ioctl)
{
	int i;
        unsigned old;
        if (ioctl == -1)
            return NULL;
	for (i = 0; old = ioctl_conv_tab[i].old, old; i++) {
		if (old == ioctl)
		        return &ioctl_conv_tab[i];
	}
	// PTB not there
	return NULL;
}

int
nbd_ioctl_convert (int ioctl)
{
	struct ioctl_conv *conv = ioctl_lookup_old (ioctl);
	if (!conv)
		// PTB not there
		return -1;
        return conv->new ? : ioctl;
}

static struct ioctl_conv *
ioctl_lookup_new (int ioctl)
{
	int i = 0;
	unsigned old;
	for (i = 0; old = ioctl_conv_tab[i].old, old; i++) {
		unsigned new = ioctl_conv_tab[i].new;
		if (new == ioctl || (new == 0 && old == ioctl)) 
			return &ioctl_conv_tab[i];
	}
	// PTB not there
	return NULL;
}

int
nbd_ioctl_revert (int ioctl)
{
	struct ioctl_conv *conv = ioctl_lookup_new (ioctl);
	if (!conv)
		// PTB not there
		return -1;
	return conv->old;
}

static struct ioctl_special *
ioctl_special_lookup_new (int ioctl)
{
	int i;
        unsigned new;
	for (i = 0; new = ioctl_special_tab[i].new, new; i++) {
		if (new == ioctl) 
			return &ioctl_special_tab[i];
	}
	// PTB not there
	return NULL;
}

int
nbd_ioctl_size (int cmd, char *arg)
{
	int size = _IOC_SIZE (cmd);
	if (size == _IOC_SIZEMASK) {
		// PTB special hadling required. 
                struct ioctl_special *special = ioctl_special_lookup_new(cmd);
                if (!special)
                        return -1;
	        return special->size (arg);
	}
	return size;
}

int
nbd_ioctl_size_user (int cmd, char *arg)
{
	int size = _IOC_SIZE (cmd);
	if (size == _IOC_SIZEMASK) {
		// PTB special hadling required. 
                struct ioctl_special *special = ioctl_special_lookup_new(cmd);
                if (!special)
                        return -1;
	        return special->size_user (arg);
	}
	return size;
}


#ifdef __KERNEL__
int
nbd_ioctl_copy_to_user (int cmd, char *arg, char *buf, int size)
{

	if (_IOC_SIZE (cmd) == _IOC_SIZEMASK) {
                struct ioctl_special *special = ioctl_special_lookup_new(cmd);
                if (!special)
                        return -1;
	        return special->ioctl_copy_to_user (arg, buf, size);
	}

	if (_IOC_DIR (cmd) & _IOC_READ) {
		// indirect
		copy_to_user (arg, buf, size);
		return size;
	}

	return -1;
}



int
nbd_ioctl_copy_from_user (int cmd, char *buf, char *arg, int size)
{

	if (_IOC_SIZE (cmd) == _IOC_SIZEMASK) {
                struct ioctl_special *special = ioctl_special_lookup_new(cmd);
                if (!special)
                        return -1;
	        return special->ioctl_copy_from_user (buf, arg, size);
	}

	if (_IOC_DIR (cmd) & _IOC_READ) {
		// indirect
		copy_from_user (buf, arg, size);
		return size;
	}

	// direct
	if (size > sizeof (arg)) {
		return -1;
	}

	memcpy (buf, &arg, size);
	return size;
}
#endif /* __KERNEL__ */


/*
static
int ioctl_init(struct ioctl_conv *self, int old, int new) {
    self->old = old;
    self->new = new;
    self->serialize = ioctl_serialize;
    self->deserialize = ioctl_deserialize;
    self->size = ioctl_size;
}
*/
