/* 
 * Copyright (C) 1999-2001 Peter T. Breuer <ptb@it.uc3m.es>
 */

#ifndef MY_NAME
  #define MY_NAME "enbd-netserver"
#endif
#define ISCLIENT

#define _USE_BSD
#include <sys/ioctl.h>
#include <sys/socket.h>
#include <sys/types.h>
#include <sys/wait.h>                  

#include <netinet/tcp.h>
#include <netinet/in.h>         /* sockaddr_in, htons, in_addr */
#include <netdb.h>              /* hostent, gethostby*, getservby* */
#include <stdio.h>
#include <string.h>
#include <fcntl.h>
#include <syslog.h>
#include <stdlib.h>
#include <signal.h>
#include <setjmp.h>
#include <sys/time.h>
#include <sys/errno.h>
#include <sys/stat.h>
#include <ctype.h>
#include <sys/mman.h>
#include <unistd.h>

#include <pthread.h>

#ifdef USING_SSL
#include "sslincl.h"
#endif

#include <linux/ioctl.h>
#include <linux/enbd.h>
#include "cliserv.h"


#include "stream.h"
//#include "netserver.h"
//#include "cacheserver.h"
#include "server_stub.h"
#include "pidfile.h"
#include "enbd-client.h"
#include "socket.h"
#include "select.h"
#include "ioctl.h"
#include "time.h"


/*
 * static (const) test to see if reply struct is close packed. This
 * function should optimize out into a constant, the amount by which
 * the struct is not close packed.
 */
static inline const short sane_reply_test(void) __attribute__ ((const));
static inline const short sane_reply_test(void)
{
    static const struct enbd_reply * _ex = NULL;
    static const short struct_rpl_is_not_packed = 
              ((char *)&_ex->dummy0 - (char *)_ex) - (
              sizeof(_ex->magic) + 
              sizeof(_ex->error) + 
              sizeof(_ex->handle) + 
              sizeof(_ex->flags) + 
              sizeof(_ex->time) + 
              sizeof(_ex->zone) + 
              sizeof(_ex->data) + 
              0);
    return struct_rpl_is_not_packed;
}
 
/*
 * negative return for network failure, and nnegative but short for
 * remote failure. Otherwise we report how many bytes we got.
 */
static int
recv_reply (struct nbd_stub_server* self, struct enbd_reply *reply,
            char *buf, u32 handle,
            int cmd, unsigned len, unsigned long long from,
            unsigned flags
          )
{
    int result;
    int tot = 0; // bytes of message received
    struct nbd_stream * stream = (struct nbd_stream *)self->media;

    DEBUG("net_recv_reply enters for req type %d with handle %#lx len %d"
          " offset %Ld flags %#x\n",
           cmd, (unsigned long)handle, len, from, flags);

    if (sane_reply_test() != 0) {
        // PTB network transmit is iffy, as we send the struct, not the
        // fields. This test should be a constant, so no penalty.
        static short count;
        if (count++ <= 0) {
            PERR("Warning! Reply struct is not close packed (diff %d != 0).\n",
              sane_reply_test());
        }
    }

    if (self->magic != ENBD_CLIENT_CACHE_NETMAGIC) {
      PERR("netserver %p has bad magic!\n", self);
      goto fail;
    }

    // PTB - read the reply header from net
    result = stream->read(stream, (char *)reply, sizeof (struct enbd_reply));

    DEBUG("client (%d) net_recv_reply reads reply header size %d from net"
        " (result %d) for cmd %d sector %u len %d\n",
           self->i, sizeof(struct enbd_reply), result, cmd,
           (unsigned)(from >> 9), len);

    DEBUG("client (%d) net_recv_reply checks reply (result %d)\n", self->i, result);

    // PTB - check result before magic!
    //if (result == 0)
    //  goto success;  // We think 0 bytes read is OK??? Surely means again?

    if (result < sizeof (struct enbd_reply))
            goto fail;

    //if (result < 0)
    //  goto fail;

    // PTB case result > 0

    // PTB convert interesting fields back to host order
    reply->magic = ntohl(reply->magic);
    reply->error = ntohl(reply->error);

    DEBUG("client (%d) net_recv_reply sees fields:\n\t magic %#x\n"
            "\terror %#x\n\thandle %#lx\n\tflags %d\n\tdigest %.8x%.8x%.8x%.8x\n"
            "on cmd %d len %d sector %lu\n",
            self->i,
            reply->magic,
            reply->error,
            (unsigned long) reply->handle,
            reply->flags,
      reply->data.digest[0], reply->data.digest[1],
      reply->data.digest[2], reply->data.digest[3],
            cmd, len, (unsigned long)(from >> 9));

    // PTB - check the magic
    if (reply->magic != ENBD_REPLY_MAGIC) {
       DEBUG("client (%d) net_recv_reply reports bad magic %#x instead of %#x"
            " with error %#x handle %#lx flags %d cmd %d len %d sector %lu\n",
            self->i,
            reply->magic, ENBD_REPLY_MAGIC,
            reply->error, (unsigned long) reply->handle, reply->flags,
            cmd, len, (unsigned long)(from >> 9));
       DEBUG("client (%d) net_recv_reply reports addresses "
            "magic %p error %p handle %#lx flags %p\n", self->i,
            &reply->magic, &reply->error,
            (unsigned long)reply->handle, &reply->flags);
       goto fail;
    }

    if (reply->error != 0) {
            DEBUG("received remote failure notification\n");
            // We want to treat this as a short remote read
            goto success;
    }

    // PTB - deal with data
    switch (cmd){
       case READ:
       // PTB - read some more bytes from stream to rest of buffer

       if (flags & ENBD_REQUEST_ERRORED) {
           PERR("client (%d) net_recv_reply does not get from net"
            " for req with handle %#lx flags %x cmd %d len %d sector %u"
                " because request already errored\n",
               self->i,
               (unsigned long) reply->handle, reply->flags,
               cmd, len, (unsigned)(from >> 9));
         // PTB leave tot short so we can judge what happened
         goto success;
       }
       // PTB some kind of generic function
       result = stream->read(stream,buf,len);
       if (result < 0)
             goto fail;
       tot += result;

       goto success;
       break;

       case WRITE:
         // PTB should return len, but seems to hurt elsewhere
         // tot = len;
       goto success;
       break;

       case CKSUM:
         DEBUG("net_recv_reply receives CKSUM %.8x%.8x%.8x%.8x for" 
          " req type %d with handle" " %#lx len %d offset %Ld\n",
       reply->data.digest[0], reply->data.digest[1],
       reply->data.digest[2], reply->data.digest[3],
          cmd, (unsigned long)handle, len, from);
          
       goto success;
       break;

       case SPECIAL:
       break;

       case IOCTL:
         DEBUG("net_recv_reply receives IOCTL for" 
               " req type %d with handle" " %#lx len %d offset %Ld\n",
               cmd, (unsigned long)handle, len, from);
         if (len > 0) {
               int ioctl = from >> 32;
               if (_IOC_DIR(ioctl) & _IOC_READ) {
                       DEBUG("recv_reply waits for %dB data (%#x) for IOCTL %#x\n",
                               len, *(unsigned*)buf, ioctl);
                       result = stream->read(stream,buf,len);
                       DEBUG("recv_reply receives %dB of %dB data (%#x..) into buf %#x\n",
                               result, len, *(unsigned*)buf, (unsigned)buf);
                       if (result < len)
                               goto fail;
                       tot += result;
               }
         }

         goto success;
         break;

    } // endsw

    success:
    DEBUG("net_recv_reply exits %d for req type %d with handle %#lx "
          "len %d offset %Ld\n", tot, cmd, (unsigned long)handle, len, from);
    return tot;

    // PTB it looks like inval is returned from here for a zero read from
    // the net, which may be precipitious. Perhaps the server is slow.
    fail:
    DEBUG("net_recv_reply exits INVAL for req type %d with handle %#lx "
         "len %d offset %Ld\n", cmd, (unsigned long)handle, len, from);
    return -EINVAL;  // PTB should cause packet to be dropped 

}

/*
 * static (const) test to see if request struct is close packed. This
 * function should optimize out into a constant, the amount by which
 * the struct is not close packed.
 */
static inline const short sane_request_test(void) __attribute__ ((const));
static inline const short sane_request_test(void)
{
    static const struct enbd_request * _ex = NULL;
    static const short struct_req_is_not_packed = 
              ((char *)&_ex->dummy0 - (char *)_ex) - (
              sizeof(_ex->magic) + 
              sizeof(_ex->type) + 
              sizeof(_ex->handle) + 
              sizeof(_ex->from) + 
              sizeof(_ex->len) + 
              sizeof(_ex->flags) + 
              sizeof(_ex->time) + 
              sizeof(_ex->zone) + 
              sizeof(_ex->seqno) + 
              sizeof(_ex->data) + 
              sizeof(_ex->special) + 
              0);
    return struct_req_is_not_packed;
}


static int
send_request (struct nbd_stub_server * self,
        char *buf,
        int type,
        unsigned flags,
        u64 from,
        int len,
        int seqno,
        u32 handle,
        long special,
        char *data
        )
{

    int result = 0;
    struct nbd_stream * stream = (struct nbd_stream *) self->media;
    struct enbd_request request;
    struct timezone zone;
    struct timeval time;

    if (sane_request_test() != 0) {
        // PTB network transmit is iffy, as we send the struct, not the
        // fields. This test should be a constant, so no penalty.
        static short count;
        if (count++ <= 0) {
            PERR("Warning! Request struct is not packed (diff %d != 0).\n",
              sane_request_test());
        }
    }

    if (self->magic != ENBD_CLIENT_CACHE_NETMAGIC) {
      PERR("netserver %p has bad magic!\n", self);
      result = -EINVAL;
      goto fail;
    }

    DEBUG("send_req enters with req type %d handle %#lx len %d off %Ld\n",
           type, (unsigned long)handle, len, from);

    if (type != IOCTL && from + len > self->size){
      DEBUG("client (%d) net_send_req fails OVERRANGE req %#lx len %d\n",
                    self->i, (unsigned long)handle, len);
      result = -EINVAL;
      goto fail;
    }

    if (type==WRITE && !(self->flags & ENBD_CLIENT_CACHE_WRITETHROUGH)){
      DEBUG("client (%d) net_send_req fails WRITE to RO req %#lx len %d\n",
                    self->i, (unsigned long)handle, len);
      result = -EINVAL;
      goto fail;
    }

    if (flags != 0) {
      PERR("send_req sees bad flags on req type %d handle %#lx len %d off %Ld\n",
           type, (unsigned long)handle, len, (unsigned long long)from);
    }

    // PTB prepare the network order version that we'll send out
    request.magic = htonl(ENBD_REQUEST_MAGIC);
    request.type  = htonl(type);
    request.from  = htonll(from);
    request.len   = htonl(len);
    request.seqno = htonl(seqno);

    mygettimeofday(&time, &zone); 
    request.zone = zone.tz_minuteswest;

    request.time = time.tv_sec;
    request.time *= 1000000;
    request.time += time.tv_usec;

    request.time = htonll(request.time);
    request.zone = htonll(request.zone);

    //PTB fill the handle with what we should fill it with
    DEBUG("copying handle %#lx\n", (unsigned long)handle);
    request.handle = htonl(handle);
    DEBUG("copied handle %#lx\n", (unsigned long)handle);

    // PTB ummmm let's not forget the payload
    if (data)
        memcpy(&request.data, data, sizeof(request.data));
    else
        memset(&request.data, 0, sizeof(request.data));

    // PTB and what about special stuff?
    memcpy(&request.special, &special, sizeof(request.special));

    DEBUG("send_req will write req with handle %lx (len %d)\n",
            (unsigned long)handle, sizeof(request));
    // PTB write the request itself
    result = stream->write(stream,(char*)&request,sizeof(struct enbd_request));

    DEBUG("send_req wrote %dB of %dB for req with handle %lx\n",
            result, sizeof(request), (unsigned long)handle);

    if (result < sizeof (struct enbd_request))
        goto fail;

    switch (type) {

      case READ:

      case CKSUM:
        goto success;
        break;

      case SPECIAL:
        DEBUG("sent special req %#lx ok (from=%Ld, len=%d)\n",
                (unsigned long)handle, from, len);
        goto success;
        break;

      case IOCTL:
        if (len > 0) {
               int ioctl = from >> 32;
               if ((_IOC_DIR(ioctl) & _IOC_READ)
               && (_IOC_DIR(ioctl) & _IOC_WRITE)) {
                       DEBUG("send_req sends %dB data (%#x..) for IOCTL req handle %#lx from buf %#x\n",
                        len, *(unsigned*)buf, (unsigned long)handle, (unsigned)buf);
                       result = stream->write(stream, buf, len);
                       DEBUG("send_req sent %dB data of %d for IOCTL req\n",
                              result, len);
               }
        }
        goto success;
        break;

      case WRITE:
        if (len <= 0) {
            result = len;
            goto success;
        }
        DEBUG("writes to stream %x from buf %x len %d\n",
            (unsigned)stream, (unsigned)buf, len);

        // PTB write any data we need to write
        result = stream->write(stream, buf, len);
        DEBUG("returns %d from write_str %x from buf %x len %d\n",
            result, (unsigned)stream, (unsigned)buf, len);
        goto success;
        break;

      default:
        PERR("unknown req type %#x\n", type);
        goto fail;
        break;
    }

  success:
    // PTB begin debug
      DEBUG("returns %d on req type %d with handle"
      " %#lx flags %x for %d bytes from sector %d\n",
           result, type, (unsigned long)handle,
           flags,len,
           (unsigned)(from>>9));
    // PTB end debug
    return result;

  fail:
    PERR("fails req type %d with handle %#lx len %d sector %Ld\n",
           type, (unsigned long)handle, len, (unsigned long long)(from >> 9));
    //flags |= ENBD_REQUEST_ERRORED;
    return (result<0)?result:-EINVAL;

}

static int
net_read(struct nbd_stub_server * self, char *buf, int len, s64 from) {

     struct enbd_reply  rpl;
     static int count;
     u32 handle;
     int err = 0;
     int i;
     const int retries = 3;

     DEBUG("net_read (%d) enters for sector %Ld len %d\n", self->i,
           from >>9, len);

     if (self->magic != ENBD_CLIENT_CACHE_NETMAGIC) {
       PERR("netserver %p has bad magic!\n", self);
       err = -1;
       goto fail;
     }

     handle = self->handle;

     // PTB also need to do this in net_write!
     // remove all pending wracks from the stream in case we're async
     while ((self->flags & ENBD_CLIENT_ASYNC)
       && (self->flags & ENBD_CLIENT_WRPLY_PEND)) {
       // PTB get (for discard) last times pending ack
       err = recv_reply(self, &rpl, buf, handle, WRITE, 0, -1, 0);
       DEBUG("seeing if (%d) can discard one of %d write acks pending\n",
             self->i, 1);
       if (err >= 0) {
         self->flags &= ~ENBD_CLIENT_WRPLY_PEND;
         DEBUG("(%d) yes it can, result %d\n", self->i, err);
       } else {
         DEBUG("(%d) no it can't, result %d\n", self->i, err);
         // PTB wait one jiffy
         microsleep(50000);
         // PTB we hope to cause a timeout eventually
         // PTB same as a goto fail?
       }
     }

     handle = count;

     // PTB read requests get an artificial handle
     //     why can't we use the real one from the kernel?
     //     Well, apparently the read request from the stub doesn't pass
     //     the handle to us!
     count++;

     for (i = 0; i < retries; i++) {

       err = send_request(self, buf, READ, 0, from, len, self->seqno,
               handle, 0, NULL);

       if (err >= 0)
         break;
       // PTB wait 1s before retry
       //flags |= ENBD_REQUEST_ERRORED;
       microsleep(1000000);
     }

     if (i >= retries) {
       // PTB rollback kernel requests, unplug from kernel
       // PTB err was negative anyway when we get here
       goto fail;             // PTB force -1 for renegogiate
     }
   
     err = recv_reply(self, &rpl, buf, handle, READ, len, from, 0);

     // PTB the error received above must indicate whether the error
     // was on the remote or on the net. Let's use short nonnegative
     // values to indicate a failed remote (i.e. error reply).
   
     if (err < 0)
       goto fail;

     DEBUG("net_read (%d) exits OK\n", self->i);
     return err;

  fail:
     PERR("net_read (%d) exits FAIL\n", self->i);
     if (err >= 0)
       err = -1;
     return err;
}


static int
net_write(struct nbd_stub_server * self, char *buf, int len, s64 from) {

     //struct enbd_request req;
     struct enbd_reply   rpl;
     static int count;
     u32 handle;
     //typeof(((struct enbd_request *) NULL)->data) data = {{0,0,0,0}};
     //u32 digest[4] = {0,0,0,0};
     int err = 0;
     int i;
     const int retries = 3;

     ENBD_REQUEST_FLAGS_T flags = 0;

     DEBUG("net_write (%d) enters for sector %d len %d\n",
           self->i, (int)(from>>9), len);

     //req.len   = len;
     //req.from  = from;
     //req.type  = WRITE;
     //req.magic = htonl(ENBD_REQUEST_MAGIC);
     //req.handle = self->handle;

     handle = self->handle;

     //memcpy(&req.data.digest[0], &self->digest[0], ENBD_DIGEST_LENGTH);
     //req.flags = 0;
     //req.seqno = self->seqno;

     if (self->magic != ENBD_CLIENT_CACHE_NETMAGIC) {
       PERR("netserver %p has bad magic!\n", self);
       err = -1;
       goto fail;
     }

     // PTB also need to do this in net_read!
     // remove a pending reply from the stream in case we're async
     while ((self->flags & ENBD_CLIENT_ASYNC)
     && (self->flags & ENBD_CLIENT_WRPLY_PEND)) {
       DEBUG("seeing if (%d) can discard one of %d write acks pending\n",
             self->i, 1);
       // PTB get (for discard) last times pending ack
       err = recv_reply(self, &rpl, buf, handle, WRITE, 0, -1, 0);
       if (err >= 0) {
          self->flags &= ~ENBD_CLIENT_WRPLY_PEND;
          DEBUG("(%d) yes it can, result %d\n", self->i, err);
       } else {
         DEBUG("(%d) no it can't, result %d\n", self->i, err);
         // PTB wait one jiffy
         microsleep(50000);
       }
     }

     // PTB write requests get an artificial handle
     //     why can't we use the real one from the kernel?
     //     Well, apparently the write request from the stub doesn't pass
     //     the handle to us! It could.

     handle = count; // this is a void* sized target
     count++;

     for (i = 0; i < retries; i++) {

       flags = 0;
       //req.handle = handle;

       //memcpy(&req.data.digest[0],&data.digest[0],ENBD_DIGEST_LENGTH);
       //req.flags = flags;

       //err = send_request(self, buf, &req);
       err = send_request(self, buf, WRITE, 0, from, len,
               self->seqno, handle, 0, NULL);

       //memcpy(&data.digest[0],&req.data.digest[0],ENBD_DIGEST_LENGTH);
       //flags = req.flags;

       if (err >= 0)
         break;
       flags |= ENBD_REQUEST_ERRORED;
       microsleep(1000000);
     }
     if (i >= retries) {
       // PTB rollback kernel requests, unplug from kernel
       goto fail;             // PTB force -1 for renegogiate
     }

     // PTB if we're running async, don't expect a reply until next time
     if (self->flags & ENBD_CLIENT_ASYNC) {
       self->flags |= ENBD_CLIENT_WRPLY_PEND;
       DEBUG("(%d) wrote new async request, # %d\n", self->i, 1);
       goto success;
     }

     err = recv_reply(self, &rpl, buf, handle, WRITE, len, from, flags);
     if (err < 0)
      goto fail;

  success:
     DEBUG("net_write exits (%s) OK %d\n",
            (self->flags & ENBD_CLIENT_ASYNC)?"async":"sync", len);
     return len;

  fail:
     DEBUG("net_write exits FAIL %d\n", err);
     return err;
}

static int
net_check(struct nbd_stub_server * self) {
    // PTB make dummy request of type READ len 0 to send out
    
    int err = 0;
    static u64 count;
    //struct enbd_request req;
    struct enbd_reply rpl;
    u64 from;

    DEBUG("check enters\n");
    //memset(&req,0,sizeof(req));

    if (self->magic != ENBD_CLIENT_CACHE_NETMAGIC) {
      PERR("netserver %p has bad magic!\n", self);
      err = -1;
      goto fail;
    }

    // PTB increment offset by one each time
    from = (count++ << 9) % self->size;
    //req.type = READ;
    //req.len  = 0;

    DEBUG("sending check on sector %Ld\n", from >> 9);
    //err = send_request(self, NULL, &req);
    err = send_request(self, NULL, READ, 0, from, 0, 0, 0, 0, NULL);
    if (err < 0) {
       goto fail;
    }
    rpl.error = 0;
    err = recv_reply(self, &rpl, NULL, 0, READ, 0, from, 0);
    DEBUG("received check on sector %Ld err %d\n", from >> 9, err);

    // PTB the recv_reply can fail, in which case we need to maybe fail
    if (err < 0)
       goto fail;
    if (rpl.error != 0) {
        err = 1;
        goto fail;
    }

    DEBUG("check exits OK\n");
    return 0;

 fail:
    DEBUG("check exits FAIL\n");
    return err;
}

static int
net_ioctl(struct nbd_stub_server *self, int ioctl, char *arg) {
    // PTB make request of type IOCTL to send out
    struct enbd_reply   rpl;
    u64 from;
    int len;
    int err = 0;

    char *buf = NULL;
    int size;

    DEBUG("net_ioctl enters\n");

    if (self->magic != ENBD_CLIENT_CACHE_NETMAGIC) {
      PERR("netserver %p has bad magic!\n", self);
      err = -ELIBBAD;
      goto fail;
    }

    // PTB local ioctls
    switch (ioctl) {
        case MY_NBD_SET_RQ_HANDLE:
            self->handle = (unsigned long)arg;
            return 0;
        case MY_NBD_SET_RQ_SEQNO:
            self->seqno = (unsigned long)arg;
            return 0;
        case MY_NBD_SET_RQ_DIGEST:
            memcpy(&self->digest[0], arg, ENBD_DIGEST_LENGTH);
            return 0;
        default:
            // PTB carry on to handle the remote ioctls
            break;
    }

    // PTB remote ioctls - assume flags and arg is already correct
    
    size = nbd_ioctl_size(ioctl, arg);

    DEBUG("net_ioctl enters for remote ioctl %#x arg %#x size %d\n",
            ioctl, (unsigned)arg, size);
    
    from  = (((u64)(unsigned long)ioctl) << 32)
              | (((u64)(unsigned long)arg) & 0xffffffffL);

    if (size <= 0) {
            len   = 0; 
            if (_IOC_DIR(ioctl) & _IOC_READ) {
                    buf = NULL;
            }
    } else {
    // PTB sometimes have to copy the ioctl data into a buffer too
            len   = ((size + 511) >> 9) << 9;
            if (_IOC_DIR(ioctl) & _IOC_READ) {
                    buf = arg;
            }
    }


    if (_IOC_DIR(ioctl) & _IOC_READ) {
        if (len > 0) {
               DEBUG("net_ioctl remote ioctl %#x contains %dB buffered (%#x..) at #%x\n",
            (unsigned)ioctl, len, *(unsigned*)buf, (unsigned)buf);
        }
    }
    DEBUG("net_ioctl sends request for ioctl id %#x arg %#x\n",
            (unsigned)ioctl, (unsigned)arg);

    err = send_request(self, buf, IOCTL, 0, from, len, 0, 0, 0, NULL);

    if (err < 0) {
       DEBUG("send_req IOCTL failed (%d)\n", err);
       goto fail;
    }

    DEBUG("net_ioctl waits on reply to request id %#x arg %#x len %d\n",
            (unsigned)ioctl, (unsigned)arg, len);

    err = recv_reply(self, &rpl, buf, 0, IOCTL, len, from, 0);

    if (err < 0) {
       DEBUG("recv_reply for IOCTL failed (%d)\n", err);
       goto fail;
    }

    if (rpl.error != 0) {
       // remote error
       err = rpl.error;
       if (err > 0)
            err = -err;
       DEBUG("recv_reply for IOCTL failed (%d)\n", err);
       goto fail;
    }

    if (err < len) {
        DEBUG("net_ioctl got %dB in reply to request for %dB\n", err, len);
        goto fail;
    }

    if (len > 0) {
        DEBUG("net_ioctl got %dB (%#x..) at %#x in reply to request for %dB\n",
            err, *(unsigned*)buf, (unsigned)buf, len);
    } else {
        DEBUG("net_ioctl got 0B in reply to request for %dB\n", len);
    }

    DEBUG("net_ioctl IOCTL exits OK (%d)\n", err);
    return err;

 fail:
    DEBUG("net_ioctl exits FAIL (%d)\n", err);
    return err;
}

static int
net_sum(struct nbd_stub_server *self, enbd_digest_t digest, int len, s64 from) {

    // PTB make request of type CKSUM to send out

    struct enbd_reply   rpl;
    unsigned flags = 0;
    int err = 0;
    // PTB this is so we can pass the address of the union container,
    // not the union contents
    const int data_offset = ({
        const struct enbd_request *_x = NULL;
        (char*)&_x->data.digest[0] - (char*)&_x->data;
    });

    DEBUG("sum enters\n");

    if (self->magic != ENBD_CLIENT_CACHE_NETMAGIC) {
      PERR("netserver %p has bad magic!\n", self);
      err = -1;
      goto fail;
    }

    err = send_request(self, NULL, CKSUM, flags, from, len, self->seqno,
            0, 0, ((char *)digest) - data_offset);
    if (err < 0) {
       flags |= ENBD_REQUEST_ERRORED;
       PERR("send_req CKSUM failed (%d)\n", err);
       goto fail;
    }
    err = recv_reply(self, &rpl, NULL, 0, CKSUM, len, from, flags);

    if (err < 0) {
       PERR("recv_reply for CKSUM failed (%d)\n", err);
       goto fail;
    }

    memcpy(&digest[0], &rpl.data.digest[0], ENBD_DIGEST_LENGTH);
    DEBUG("sum exits OK (%d)\n", err);
    return err;

 fail:
    PERR("sum exits FAIL (%d)\n", err);
    return err;
}


static int
net_sync(struct nbd_stub_server * self, unsigned how) {
    if (self->magic != ENBD_CLIENT_CACHE_NETMAGIC) {
      PERR("netserver %p has bad magic!\n", self);
      return -1;
    }
    return 0;
}

static int
net_special(struct nbd_stub_server * self, int rw, int len, u64 from) {
    // PTB make dummy request of type SPECIAL to send out
    
    int err = 0;
    //struct enbd_request req;
    struct enbd_reply rpl;

    DEBUG("special enters\n");
    //memset(&req,0,sizeof(req));

    if (self->magic != ENBD_CLIENT_CACHE_NETMAGIC) {
      PERR("netserver %p has bad magic!\n", self);
      err = -1;
      goto fail;
    }

    DEBUG("sending special len %d on sector %Ld\n", len, from >> 9);
    //err = send_request(self, NULL, &req);
    err = send_request(self, NULL, SPECIAL, 0, from, len, 0, 0, 0, NULL);
    if (err < 0) {
       goto fail;
    }
    rpl.error = 0;
    err = recv_reply(self, &rpl, NULL, 0, rw, 0, from, 0);
    DEBUG("received special len %d on sector %Ld err %d\n", len, from>>9, err);

    // PTB the recv_reply can fail, in which case we need to maybe fail
    if (err < 0)
       goto fail;
    if (rpl.error != 0) {
        err = 1;
        goto fail;
    }

    DEBUG("special exits OK\n");
    return 0;

 fail:
    DEBUG("special exits FAIL\n");
    return err;
}


static unsigned long setflags(struct nbd_stub_server *self, unsigned long flags) {
     self->flags |= flags;
     return self->flags;
}
static unsigned long resetflags(struct nbd_stub_server *self, unsigned long flags) {
     self->flags &= ~flags;
     return self->flags;
}
static unsigned long getflags(struct nbd_stub_server *self) {
     return self->flags;
}



void
init_netserver(struct nbd_stub_server *self, short i,
     s64 size, void * stream, short ro, void *alternate, short async) {
       self->media = stream; // struct nbd_stream
       self->size  = size;
       self->flags = 0;
       self->i     = i;
       self->altmedia
                   = alternate; // struct nbd_stub_server
       self->nextmedia
                   = NULL;
       self->write = net_write;
       self->read  = net_read;
       self->check = net_check;
       self->sync  = net_sync;
       self->sum   = net_sum ;
       self->ioctl = net_ioctl;
       self->special
                   = net_special;
       if (!ro)
         self->flags |= ENBD_CLIENT_CACHE_WRITETHROUGH;
       self->flags |= ENBD_CLIENT_CACHE_READTHROUGH;
       if (async)
         self->flags |= ENBD_CLIENT_ASYNC;
       self->magic = ENBD_CLIENT_CACHE_NETMAGIC;

       self->setflags = setflags;
       self->getflags = getflags;
       self->resetflags = resetflags;
}

