/* 
 * Copyright (C) 1999-2001 Peter T. Breuer <ptb@it.uc3m.es>
 */

#ifndef MY_NAME
  #define MY_NAME "nbd-client-server"
#endif
#define ISCLIENT

/*
 * This is the read/write interface to the server. Underneath it calls
 * either the cacheserver or the netserver. This interface is called
 * directly by the nbd-client top-level code whenever it needs to do a
 * read or write (or presumably, a sync).
 *
 * int (*write)();
 * int (*read)();
 * int (*check)();
 * int (*sync)();
 */

#include <stdio.h>
#include <string.h>
#include <fcntl.h>
#include <syslog.h>
#include <stdlib.h>
#include <setjmp.h>
#include <sys/time.h>
#include <sys/errno.h>
#include <sys/stat.h>
#include <ctype.h>
#include <sys/mman.h>
#include <sys/types.h>
#include <unistd.h>

#include <pthread.h>

#ifdef USING_SSL
#include "sslincl.h"
#endif

#include <linux/ioctl.h>
#include <linux/enbd.h>
#include "cliserv.h"


#ifndef NBD_NO_CACHING
  #include "cache/cache.h"
#endif
#include "stream.h"
//#include "netserver.h"
//#include "cacheserver.h"
#include "server_stub.h"
#include "pidfile.h"
#include "nbd-client.h"
#include "socket.h"
#include "select.h"

static int
readgen(struct nbd_stub_server * self, char *buf, struct enbd_request *req) {

     struct nbd_stub_server * firstserver 
            = (struct nbd_stub_server *)self->media;
     struct nbd_stub_server * altserver 
            = (struct nbd_stub_server *)self->altmedia;
     struct nbd_stub_server * nextserver 
            = (struct nbd_stub_server *)self->nextmedia;
     int result;
     int tot = 0;

     DEBUG("readgen enters\n");

     if (self->magic != NBD_CLIENT_CACHE_GENMAGIC) {
         PERR("server %x with bad magic\n", (unsigned)self);
         return -EINVAL;
     }

     while (req->len > 0) {
       int res1, res2;

       if (firstserver && (firstserver->flags & NBD_CLIENT_CACHE_ENABLED)) {

       // PTB if net enabled, try net first
         DEBUG("readgen calls firstserver net\n");
         res1 = firstserver->read(firstserver, buf, req);
         if (res1 < 0)
             break;
         req->len  -= res1;
         req->from += res1;
         tot += res1;
       }

       if (req->len > 0) {
           if (altserver && (altserver->flags & NBD_CLIENT_CACHE_ENABLED)) {
               // try and recover
             int len  = req->len;
             req->len = 1024; // FIXME want blksize!!
             res2 = altserver->read(altserver, buf, req);
             req->len = len;
             if (res2 < 1024) {
               break;
             }
             req->len  -= res2;
             req->from += res2;
             tot       += res2;
           }
       }

     } // ewhile

     if (req->len > 0)
       result = -EIO;

     if (result < 0) {
       PERR("readgen exits FAIL (%d)\n", result);
       return result;
     }
     DEBUG("readgen exits OK (%d)\n", result);
     return result;
}




static int
writegen(struct nbd_stub_server * self, char *buf, struct enbd_request *req) {

     struct nbd_stub_server * firstserver 
            = (struct nbd_stub_server *)self->media;
     struct nbd_stub_server * altserver 
            = (struct nbd_stub_server *)self->altmedia;
     struct nbd_stub_server * nextserver 
            = (struct nbd_stub_server *)self->nextmedia;
     int err, result;


     if (self->magic != NBD_CLIENT_CACHE_GENMAGIC) {
         PERR("server %x with bad magic\n", (unsigned)self);
         return -EINVAL;
     }

     DEBUG("write to sector %Ld len %d\n", req->from >> 9, req->len);

     err = -1;
     if (firstserver && (firstserver->flags & NBD_CLIENT_CACHE_ENABLED)) {
       err = firstserver->write(firstserver, buf, req);
       if (err < 0) {
         PERR("write first server fails sector %Ld len %d\n",
            req->from >> 9, req->len);
       }
     }

     if (err < 0
     && altserver && (altserver->flags & NBD_CLIENT_CACHE_ENABLED)) {
       err = altserver->write(altserver, buf, req);
       if (err < 0) {
         PERR("write alt server fails sector %Ld len %d\n",
            req->from >> 9, req->len);
       }
     }

     if (err >= 0
     && nextserver && (nextserver->flags & NBD_CLIENT_CACHE_ENABLED)) {
       err = nextserver->write(nextserver, buf, req);
       if (err < 0) {
         PERR("write next server fails sector %Ld len %d\n", 
            req->from >> 9, req->len);
       }
     }

     if (err >= 0)
       result = err;
     if (err < 0) {
       PERR("exits FAIL\n");
       return err;
     }
     DEBUG("exits OK\n");
     return result;
}


static int
checknet(struct nbd_stub_server * self, struct enbd_reply *reply) {
    struct nbd_stub_server * netserver 
            = (struct nbd_stub_server *)self->media;
    // PTB make dummy request of type READ len 0 to send out
    if (self->magic != NBD_CLIENT_CACHE_GENMAGIC) {
         PERR("server %x with bad magic\n", (unsigned)self);
         return -EINVAL;
    }
    return netserver->check(netserver, reply);
}


static int
syncgen(struct nbd_stub_server * self, unsigned how) {

     struct nbd_stub_server * firstserver 
            = (struct nbd_stub_server *)self->media;
     struct nbd_stub_server * altserver 
            = (struct nbd_stub_server *)self->altmedia;
     struct nbd_stub_server * nextserver 
            = (struct nbd_stub_server *)self->nextmedia;
     int err, result;

     if (self->magic != NBD_CLIENT_CACHE_GENMAGIC) {
         PERR("server %x with bad magic\n", (unsigned)self);
         return -EINVAL;
     }

     err = -1;
     if (firstserver && (firstserver->flags & NBD_CLIENT_CACHE_ENABLED)) {
       err = firstserver->sync(firstserver, how);
     }

     if (err < 0
     && altserver && (altserver->flags & NBD_CLIENT_CACHE_ENABLED)) {
       err = altserver->sync(altserver, how);
     }

     if (err >= 0
     && nextserver && (nextserver->flags & NBD_CLIENT_CACHE_ENABLED)) {
       err = nextserver->sync(nextserver, how);
     }

     if (err >= 0)
       result = err;
     if (err < 0) {
       PERR("syncgen exits FAIL\n");
       return err;
     }
     DEBUG("syncgen exits OK\n");
     return result;
}

static int
ioctlgen(struct nbd_stub_server * self, int ioctl, char * arg) {

     int err;
     struct nbd_stub_server * firstserver 
            = (struct nbd_stub_server *)self->media;

     if (self->magic != NBD_CLIENT_CACHE_GENMAGIC) {
         PERR("server %x with bad magic\n", (unsigned)self);
         return -EINVAL;
     }

     err = -1;
     if (firstserver) {
       err = firstserver->ioctl(firstserver, ioctl, arg);
     }

     if (err < 0) {
       PERR("ioctlgen exits FAIL (%d)\n", err);
       return err;
     }
     MSG("ioctlgen exits OK\n");
     return err;
}

static int
sumgen(struct nbd_stub_server *self, u32 digest[4], int len, s64 from) {

     struct nbd_stub_server * firstserver 
            = (struct nbd_stub_server *)self->media;
     struct nbd_stub_server * altserver 
            = (struct nbd_stub_server *)self->altmedia;
     struct nbd_stub_server * nextserver 
            = (struct nbd_stub_server *)self->nextmedia;
     int err, result;

     if (self->magic != NBD_CLIENT_CACHE_GENMAGIC) {
         PERR("server %x with bad magic\n", (unsigned)self);
         return -EINVAL;
     }

     err = -1;
     if (firstserver && (firstserver->flags & NBD_CLIENT_CACHE_ENABLED)) {
       err = firstserver->sum(firstserver, digest, len, from);
     }

     if (err < 0
     && altserver && (altserver->flags & NBD_CLIENT_CACHE_ENABLED)) {
       err = altserver->sum(altserver, digest, len, from);
     }

     if (err >= 0
     && nextserver && (nextserver->flags & NBD_CLIENT_CACHE_ENABLED)) {
       err = nextserver->sum(nextserver, digest, len, from);
     }

     if (err >= 0)
       result = err;
     if (err < 0) {
       PERR("sumgen exits FAIL (%d)\n", err);
       return err;
     }
     DEBUG("received server md5sum %.8x%.8x%.8x%.8x\n",
        digest[0], digest[1], digest[2], digest[3]);
     DEBUG("sumgen exits OK (%d)\n", result);
     return result;
}


/*
 * Turns on the cache if the cache bit is set and the cache was not
 * already flagged as set (it disregards the previous setting when we
 * are not flagged as itialized yet). Hopefully this will keep flags
 * and cache state in step.
 */
static int cache_enable(struct nbd_stub_server *self) {

  struct nbd_stub_server * cacheserver 
            = (struct nbd_stub_server *)self->altmedia;
  struct nbd_stub_server * netserver 
            = (struct nbd_stub_server *)self->media;

  // PTB the first time through, set things up anyway despite flags
  if ((self->flags & NBD_CLIENT_CACHE_INITIALIZED)
  && (self->flags & NBD_CLIENT_CACHE_ENABLED))
    return 0;

  self->flags |= NBD_CLIENT_CACHE_ENABLED;

  // PTB this is slightly wrong, as the cache leads through to the net
  DEBUG("disable netserver (media), enable cacheserver (altmedia)\n");
  netserver->flags   &= ~NBD_CLIENT_CACHE_ENABLED;
  cacheserver->flags |= NBD_CLIENT_CACHE_ENABLED;

  // PTB this is what should happen
  //self->nextmedia     = netserver;
  //netserver->flags   |= NBD_CLIENT_CACHE_ENABLED;

  //self->read = readcache;
  //self->write= writecache;
  //self->sync = synccache;
  return 0;
}

static int cache_disable(struct nbd_stub_server *self) {

  struct nbd_stub_server * cacheserver 
            = (struct nbd_stub_server *)self->altmedia;
  struct nbd_stub_server * netserver 
            = (struct nbd_stub_server *)self->media;

  // PTB the first time through, set things up anyway despite flags
  if ((self->flags & NBD_CLIENT_CACHE_INITIALIZED)
  && !(self->flags & NBD_CLIENT_CACHE_ENABLED))
    return 0;

  self->flags &= ~NBD_CLIENT_CACHE_ENABLED;

  // PTB this is slightly wrong, as the cache leads through to the net
  DEBUG("enable netserver (media), disable cacheserver (altmedia)\n");
  netserver->flags   |= NBD_CLIENT_CACHE_ENABLED;
  cacheserver->flags &= ~NBD_CLIENT_CACHE_ENABLED;

  // PTB this is what should happen

  //self->read = readnet;
  //self->write= writenet;
  //self->sync = syncnet;
  return 0;
}

/*
 * Accepts the bits of the flag that are to be set and sets them. Also
 * turns on the cache if the cache bit is set and the cache was not
 * already flagged as set (it disregards the previous setting when we
 * are not flagged as itialized yet). Hopefully this will keep flags
 * and cache state in step.
 */
static unsigned long setflags(struct nbd_stub_server *self, unsigned long flags) {
     if ((flags & NBD_CLIENT_CACHE_ENABLED) 
     && ! (self->flags & NBD_CLIENT_CACHE_ENABLED)) 
        cache_enable(self);
     return self->flags |= flags;
}

/*
 * accepts the bits of the flag that are to be reset and resets them.
 * Note that therefore one passes a 1 in the position that is to be
 * reset! This turns off the cache if the cache was set. It does not
 * disable the cache unless the flag was already set for the cache
 * (except when the initialized bit is not yet set, when it does just
 * what it is told).
 */
static unsigned long resetflags(struct nbd_stub_server *self, unsigned long flags) {
     if ((flags & NBD_CLIENT_CACHE_ENABLED) 
     &&  (self->flags & NBD_CLIENT_CACHE_ENABLED)) 
        cache_disable(self);
     return self->flags &= ~flags;
}
static unsigned long getflags(struct nbd_stub_server *self) {
     return self->flags;
}

void init_stub_server(struct nbd_stub_server *self,
      unsigned long flags, int i, u64 size, int journal,
      //struct nbd_stream * stream, struct nbd_cache * cache
    struct nbd_stub_server * netserver, struct nbd_stub_server * cacheserver
    ) {

  self->flags  = 0;
  if (!(flags & NBD_CLIENT_SERVER_RO))
     self->flags |= NBD_CLIENT_CACHE_WRITETHROUGH;
  self->i      = i;
  self->size   = size;
  if (journal >= -1) {
     self->flags |= NBD_CLIENT_CACHE_WRITEBACK | NBD_CLIENT_CACHE_READBACK;
     self->flags |= NBD_CLIENT_CACHE_ENABLED;
     DEBUG("journal set so cache flagged on\n");
  }
  self->flags |= NBD_CLIENT_CACHE_READTHROUGH;

  self->check   = checknet;
  self->setflags= setflags;
  self->getflags= getflags;
  self->resetflags
                = resetflags;

  DEBUG("server media set to 0x%lx\n", (unsigned long)netserver);
  self->media   = netserver;    // this is the first call
  DEBUG("server altmedia set to 0x%lx\n", (unsigned long)cacheserver);
  self->altmedia= cacheserver;  // this is the alternate call, if first fails
  DEBUG("server nextmedia set to 0x%lx\n", (unsigned long)NULL);
  self->nextmedia               // this is the next call, if first succeeds
                = NULL;

  self->read    = readgen;
  self->write   = writegen;
  self->sync    = syncgen;
  self->sum     = sumgen;
  self->ioctl   = ioctlgen;

  self->magic   = NBD_CLIENT_CACHE_GENMAGIC;

  if (self->flags & NBD_CLIENT_CACHE_ENABLED) {
    cache_enable(self);  // this points write at writecache
    DEBUG("cache enabled\n");
  } else {
    cache_disable(self); // this points write at writenet
    DEBUG("cache disabled\n");
  }
  self->flags |= NBD_CLIENT_CACHE_INITIALIZED;
}

