#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <gtk/gtk.h>
#include <gdk/gdkkeysyms.h>

#include "../include/string.h"
#include "../include/disk.h"

#include "guiutils.h"
#include "toolbar.h"

#include "edvtypes.h"
#include "edvcfg.h"
#include "edvstatusbar.h"
#include "archiver.h"
#include "archivercb.h"
#include "archiveropcb.h"
#include "archivercontents.h"
#include "archiverdnd.h"
#include "endeavour.h"
#include "edvcb.h"
#include "edvutils.h"
#include "edvutilsgtk.h"
#include "edvcfglist.h"
#include "config.h"


#include "images/icon_new_20x20.xpm"
#include "images/icon_open_20x20.xpm"
#include "images/icon_save_20x20.xpm"
#include "images/icon_executable_20x20.xpm"
#include "images/icon_terminal_20x20.xpm"
#include "images/icon_close_20x20.xpm"
#include "images/icon_exit_20x20.xpm"
#include "images/icon_archive_add_20x20.xpm"
#include "images/icon_archive_extract_20x20.xpm"
#include "images/icon_cancel_20x20.xpm"
#include "images/icon_select_20x20.xpm"
#include "images/icon_reload_20x20.xpm"
#include "images/icon_mimetypes_20x20.xpm"
#include "images/icon_search_20x20.xpm"
#include "images/icon_processes_20x20.xpm"
#include "images/icon_tool_bar_20x20.xpm"
#include "images/icon_status_bar_20x20.xpm"
#include "images/icon_options2_20x20.xpm"
#include "images/icon_customize_20x20.xpm"
#include "images/icon_insecure_20x20.xpm"
#include "images/icon_secure_20x20.xpm"
#include "images/icon_help_20x20.xpm"
#include "images/icon_about_20x20.xpm"


#include "images/endeavour_20x20.xpm"
#include "images/image_browser_20x20.xpm"
#include "images/archiver_20x20.xpm"
#include "images/icon_trash_20x20.xpm"
#include "images/icon_trash_empty_20x20.xpm"
#include "images/archiver_48x48.xpm"


static edv_archiver_opid_struct *EDVArchiverMatchOPID(
        edv_archiver_struct *archiver, gint op
);

void EDVArchiverSetTitle(
	edv_archiver_struct *archiver, const gchar *path
);

gchar *EDVArchiverCurrentLocation(edv_archiver_struct *archiver);
void EDVArchiverSetLocation(
        edv_archiver_struct *archiver,
        const gchar *path, gbool record_history
);
void EDVArchiverUpdateLocationIcon(
        edv_archiver_struct *archiver, const gchar *path
);

void EDVArchiverToolbarRegenerate(edv_archiver_struct *archiver);
void EDVArchiverContentsMenuRegenerate(edv_archiver_struct *archiver);

void EDVArchiverSyncConfiguration(edv_archiver_struct *archiver);

void EDVArchiverSelectArchive(
	edv_archiver_struct *archiver, const gchar *arch_obj
);

edv_archiver_opid_struct **EDVArchiverOPIDListNew(
	edv_archiver_struct *archiver,      /* Can be NULL. */
	gint *total
);
void EDVArchiverOPIDListDelete(
        edv_archiver_opid_struct **list, gint total
);

static void EDVArchiverCreateMenuBar(
        edv_archiver_struct *archiver, GtkWidget *parent
);

edv_archiver_struct *EDVArchiverNew(gpointer core_ptr);
void EDVArchiverReset(edv_archiver_struct *archiver, gbool need_unmap);
void EDVArchiverSyncData(edv_archiver_struct *archiver);
void EDVArchiverUpdateMenus(edv_archiver_struct *archiver);
void EDVArchiverUpdateContentsHeadings(
        edv_archiver_struct *archiver,
        const gint *contents_heading, gint total_contents_headings
);
void EDVArchiverSetBusy(edv_archiver_struct *archiver, gbool is_busy);
void EDVArchiverMap(edv_archiver_struct *archiver);
void EDVArchiverUnmap(edv_archiver_struct *archiver);
void EDVArchiverDelete(edv_archiver_struct *archiver);



/*
 *      Returns the operation id structure on the given archiver that
 *      matches the given id.
 */
static edv_archiver_opid_struct *EDVArchiverMatchOPID(
        edv_archiver_struct *archiver, gint op
)
{
	gint i;
        edv_archiver_opid_struct *opid;


        if(archiver == NULL)
            return(NULL);

        for(i = 0; i < archiver->total_opids; i++)
        {
            opid = archiver->opid[i];
            if((opid != NULL) ? (opid->op == op) : FALSE)
                return(opid);
        }

        return(NULL);
}


/*
 *      Updates the WM title on the given recycle bin's toplevel window.
 *
 *      The given path is optional, it can be either the current
 *      selected location or the last selected object.
 */
void EDVArchiverSetTitle(
	edv_archiver_struct *archiver, const gchar *path
)
{
        gbool include_user_name, include_location;
	const gchar *prog_name = "Archiver";
        gchar *title;
        GtkWidget *w;
        edv_core_struct *core_ptr;


        if(archiver == NULL)
            return;

        core_ptr = (edv_core_struct *)archiver->core_ptr;
        if(core_ptr == NULL)
            return;

	if(core_ptr->title != NULL)
	    prog_name = core_ptr->title;

        w = archiver->toplevel;
        if(w == NULL)
            return;

        include_user_name = EDVCFGItemListGetValueI(
            core_ptr->cfg_list, EDV_CFG_PARM_ARCHIVER_TITLE_WITH_USER
        );
        include_location = EDVCFGItemListGetValueI(
            core_ptr->cfg_list, EDV_CFG_PARM_ARCHIVER_TITLE_WITH_LOCATION
        );

        /* Generate new title string. */
        if(include_user_name && include_location &&
           (path != NULL)
        )
            title = g_strdup_printf(
                "%s: %s %s",
                prog_name, core_ptr->effective_user_id_location_str,
                path
            );
        else if(include_location && (path != NULL))
            title = g_strdup_printf(
                "%s: %s",
                prog_name, path
            );
        else if(include_user_name)
            title = g_strdup_printf(
                "%s: %s",
                prog_name, core_ptr->effective_user_id_location_str
            );
        else
            title = g_strdup(prog_name);

        /* Set new title. */
        gtk_window_set_title(GTK_WINDOW(w), title);

        /* Deallocate title string, it is no longer needed. */
        g_free(title);
}


/*
 *      Returns the path to the current location of the archiver.
 *
 *      Can return NULL on error.
 */
gchar *EDVArchiverCurrentLocation(edv_archiver_struct *archiver)
{
        GtkCombo *combo;


        if(archiver == NULL)
            return(NULL);

        combo = (GtkCombo *)archiver->location_combo;
        if(combo == NULL)
            return(NULL);

        return(gtk_entry_get_text(GTK_ENTRY(combo->entry)));
}

/*
 *      Sets the location combo of the given archiver to the value of
 *      path.
 *
 *      If record_history is TRUE then the current value will be recorded
 *      on the combo's list before setting of the new value.
 *      Duplicate values will not be recorded.
 */
void EDVArchiverSetLocation(
        edv_archiver_struct *archiver,
        const gchar *path, gbool record_history
)
{
        GtkCombo *combo;
        gchar *new_path;


        if((archiver == NULL) || (path == NULL))
            return;

        combo = (GtkCombo *)archiver->location_combo;
        if(combo == NULL)
            return;


        /* Copy given path as the new path. */
        new_path = g_strdup(path);

        /* Simplify new path, reducing occurances of /.. */
        SimplifyPath(new_path);


        /* Check for no change in value. */
        if(new_path != NULL)
        {
            const gchar *cstrptr = gtk_entry_get_text(GTK_ENTRY(combo->entry));
            if(cstrptr != NULL)
            {
                /* No change in value? */
                if(!strcmp(cstrptr, new_path))
                {
                    g_free(new_path);
                    new_path = NULL;
                    return;
                }
            }
        }

        /* Get old value from combo's entry and record it on the list. */
        if(record_history)
        {
            gchar *old_path;
            const gchar *cstrptr = gtk_entry_get_text(GTK_ENTRY(combo->entry));
            old_path = (cstrptr != NULL) ? g_strdup(cstrptr) : NULL;
            if(old_path != NULL)
            {
                GUIComboAddItem(combo, old_path);
                g_free(old_path);
                old_path = NULL;
            }
        }

        /* Set new value on entry and deallocate coppied new_path. */
        if(new_path != NULL)
        {
            gtk_entry_set_text(GTK_ENTRY(combo->entry), new_path);
            g_free(new_path);
            new_path = NULL;
        }
}

/*
 *      Updates the location bar's icon pixmap with respect to the path
 *      specified by path.
 *
 *      If the given path is NULL then the icon will be set as unknown.
 */
void EDVArchiverUpdateLocationIcon(
        edv_archiver_struct *archiver, const gchar *path
)
{
        gint type, width, height;
        guint permissions;
        GdkPixmap *pixmap_closed, *pixmap_opened;
        GdkBitmap *mask_closed, *mask_opened;
        GtkWidget *parent, *w;
        edv_core_struct *core_ptr;


        if(archiver == NULL)
            return;

        core_ptr = (edv_core_struct *)archiver->core_ptr;
        if(core_ptr == NULL)
            return;

        /* Get icon fixed widget as the parent. */
        parent = archiver->location_icon_fixed;
        if(parent == NULL)
            return;


        /* Get type and permissions of object that the given path reffers
         * to.
         */
        type = EDV_OBJECT_TYPE_FILE;
        permissions = 0x00000000;
        if(path != NULL)
        {
            struct stat lstat_buf;

            /* Try to get local stats of the given path. */
            if(!lstat(path, &lstat_buf))
            {
                type = EDVObjectGetTypeFromStatMode(lstat_buf.st_mode);
                permissions = EDVObjectGetPermissionsFromStatMode(lstat_buf.st_mode);
            }
        }

        /* Get pixmap and mask for icon. */
        EDVMatchObjectIcon(
            core_ptr->device, core_ptr->total_devices,
            core_ptr->mimetype, core_ptr->total_mimetypes,
            type,
            path,
            TRUE, permissions,
            0,                  /* Small icons. */
            &pixmap_closed, &mask_closed,
            &pixmap_opened, &mask_opened,
            NULL, NULL
        );
        /* If `opened' pixmap is not available then revert to `closed'. */
        if(pixmap_opened == NULL)
        {
            pixmap_opened = pixmap_closed;
            mask_opened = mask_closed;
        }

        /* If no `opened' version of pixmap available then give up. */
        if(pixmap_opened == NULL)
            return;


        /* Create new icon pixmap widget as needed. */
        w = archiver->location_icon_pm;
        if(w == NULL)
        {
            archiver->location_icon_pm = w = gtk_pixmap_new(
                pixmap_opened, mask_opened
            );
            gtk_fixed_put(GTK_FIXED(parent), w, 0, 0);
            gtk_widget_show(w);
        }
        else
        {
            gtk_pixmap_set(GTK_PIXMAP(w), pixmap_opened, mask_opened);
        }

        gdk_window_get_size(
            (GdkWindow *)pixmap_opened, &width, &height
        );

        /* Adjust size of fixed widget to fit pixmap. */
/*
        gtk_widget_set_usize(parent, width, height);
        gtk_widget_queue_resize(parent);
 */

        /* Set new shape of fixed widget parent. */
/*
        gtk_widget_shape_combine_mask(parent, mask_opened, 0, 0);
 */
}


/*
 *      Recreates the tool bar on the given recycle bin.
 */
void EDVArchiverToolbarRegenerate(edv_archiver_struct *archiver)
{
        gint i;
        GtkWidget *w, *parent;
        edv_core_struct *core_ptr;
        edv_intlist_struct *intlist;
        edv_archiver_opid_struct *opid;
        gint tb_display, tb_relief;
        toolbar_item_struct **item, *item_ptr;
        gint total_items;


        if(archiver == NULL)
            return;

        core_ptr = (edv_core_struct *)archiver->core_ptr;
        if(core_ptr == NULL)
            return;

        /* Get recycle bin tool bar display and relief settings. */
        tb_display = EDVCFGItemListGetValueI(
            core_ptr->cfg_list, EDV_CFG_PARM_ARCHIVER_TOOL_BAR_DISPLAY
        );
        tb_relief = EDVCFGItemListGetValueI(
            core_ptr->cfg_list, EDV_CFG_PARM_ARCHIVER_TOOL_BAR_RELIEF
        );


        /* Get parent for tool bar. */
        parent = archiver->tool_bar_handle;
        if(parent == NULL)
            return;


        /* Delete old tool bar (if any). */
        ToolBarDelete(archiver->toolbar);
        archiver->toolbar = NULL;


        /* Get tool bar arrangement from configuration list. */
        intlist = EDVCFGItemListGetValueIntList(
            core_ptr->cfg_list, EDV_CFG_PARM_ARCHIVER_TOOL_BAR
        );
        if(intlist == NULL)
            return;

        /* Create tool bar items list. */
        total_items = intlist->total;
        if(total_items > 0)
        {
            item = (toolbar_item_struct **)g_malloc0(
                total_items * sizeof(toolbar_item_struct *)
            );
            if(item == NULL)
                total_items = 0;
        }
        else
        {
            item = NULL;
        }
        for(i = 0; i < total_items; i++)
        {
            /* Get opid structure that matches the operation id specified
             * on the intlist.
             */
            opid = EDVArchiverMatchOPID(archiver, intlist->i[i]);
            if(opid == NULL)
                continue;

            switch(opid->op)
            {
              case EDV_ARCHIVER_OP_NONE:
                item[i] = item_ptr = NULL;
                break;

              default:
                item[i] = item_ptr = ToolBarItemNew(
                    opid->toolbar_item_type,
                    opid->button_name,
                    opid->button_icon_data,
                    opid->tooltip,
                    opid->op,
                    EDVArchiverOPCB, opid,
                    EDVArchiverOPEnterCB, opid,
                    EDVArchiverOPLeaveCB, opid
                );
                break;
            }

        }

        /* Create new tool bar. */
        archiver->toolbar = ToolBarNew(
            item, total_items, parent,
            tb_display, tb_relief, FALSE
        );
        ToolBarMap(archiver->toolbar);


        /* Deallocate tool bar items list, it is no longer needed. */
        for(i = 0; i < total_items; i++)
            ToolBarItemDelete(item[i]);
        g_free(item);
        item = NULL;
        total_items = 0;


        /* Need to tell toplevel to resize since tool bar may change size. */
        w = archiver->toplevel;
        if(w != NULL)
            gtk_widget_queue_resize(w);
}

/*
 *      Recreates the contents clist right click menu on the given
 *      recycle bin.
 */
void EDVArchiverContentsMenuRegenerate(edv_archiver_struct *archiver)
{
        if(archiver == NULL)
            return;


        /* Destroy existing menu if it already exists. */
        if(1)
        {
            GtkWidget **w;

#define DO_DESTROY_WIDGET       \
{ \
 if((*w) != NULL) \
 { \
  GtkWidget *tmp_w = *w; \
  (*w) = NULL; \
  gtk_widget_destroy(tmp_w); \
 } \
}

            w = &archiver->contents_clist_menu;
            DO_DESTROY_WIDGET

#undef DO_DESTROY_WIDGET
        }


        /* Create new menu. */
        if(1)
        {
            gint op;
            guint8 **icon_data = NULL;
            GtkAccelGroup *accelgrp = NULL;     /* No acceleraters wanted. */
            GtkWidget *menu, *w, **w_record, **fw_record;

#define DO_ADD_MENU_ITEM_LABEL  \
{ \
 edv_archiver_opid_struct *opid = EDVArchiverMatchOPID(archiver, op); \
 if(opid != NULL) \
 { \
  w = GUIMenuItemCreate( \
   menu, GUI_MENU_ITEM_TYPE_LABEL, accelgrp, \
   (u_int8_t **)((icon_data != NULL) ? icon_data : opid->menu_icon_data), \
   opid->menu_name, \
   opid->accel_key, opid->accel_mods, \
   (void **)fw_record, \
   (gpointer)opid, EDVArchiverMenuItemCB \
  ); \
  GUISetMenuItemCrossingCB( \
   w, \
   (gpointer)EDVArchiverMenuItemEnterCB, (gpointer)opid, \
   (gpointer)EDVArchiverMenuItemLeaveCB, (gpointer)opid \
  ); \
  if(w_record != NULL) *w_record = w; \
 } \
}

#define DO_ADD_MENU_SEP \
{ \
 w = GUIMenuItemCreate( \
  menu, GUI_MENU_ITEM_TYPE_SEPARATOR, NULL, \
  NULL, NULL, 0, 0, NULL, \
  NULL, NULL \
 ); \
}

            /* Create menu. */
            archiver->contents_clist_menu = menu = GUIMenuCreate();

            op = EDV_ARCHIVER_OP_ADD;
            w_record = &archiver->contents_clist_add_mi;
            fw_record = NULL;
            DO_ADD_MENU_ITEM_LABEL

            op = EDV_ARCHIVER_OP_EXTRACT;
            w_record = &archiver->contents_clist_extract_mi;
            fw_record = NULL;
            DO_ADD_MENU_ITEM_LABEL

            op = EDV_ARCHIVER_OP_EXTRACT_ALL;
            w_record = &archiver->contents_clist_extract_all_mi;
            fw_record = NULL;
            DO_ADD_MENU_ITEM_LABEL

            op = EDV_ARCHIVER_OP_DELETE;
            w_record = &archiver->contents_clist_delete_mi;
            fw_record = NULL;
            DO_ADD_MENU_ITEM_LABEL

#undef DO_ADD_MENU_ITEM_LABEL
#undef DO_ADD_MENU_SEP
        }
}


/*
 *	Sets values on the given recycle bin to the configuration list on
 *	the core structure of the given recycle bin.
 */
void EDVArchiverSyncConfiguration(edv_archiver_struct *archiver)
{
        GtkWidget *w;
        edv_core_struct *core_ptr;


        if(archiver == NULL)
            return;

        core_ptr = (edv_core_struct *)archiver->core_ptr;
        if(core_ptr == NULL)
            return;

        /* Get size of toplevel. */
        w = archiver->toplevel;
        if(w != NULL)
        {
            GdkWindow *window = w->window;
            gint x = 0, y = 0, width, height;


            if(window != NULL)
                gdk_window_get_root_origin(window, &x, &y);
            EDVCFGItemListSetValueI(
                core_ptr->cfg_list, EDV_CFG_PARM_ARCHIVER_X,
                x, FALSE
            );
            EDVCFGItemListSetValueI(
                core_ptr->cfg_list, EDV_CFG_PARM_ARCHIVER_Y,
                y, FALSE
            );
	    gtk_widget_set_uposition(w, x, y);

            width = w->allocation.width;
            height = w->allocation.height;
            EDVCFGItemListSetValueI(
                core_ptr->cfg_list, EDV_CFG_PARM_ARCHIVER_WIDTH,
                width, FALSE
            );
            EDVCFGItemListSetValueI(
                core_ptr->cfg_list, EDV_CFG_PARM_ARCHIVER_HEIGHT,
                height, FALSE
            );
	    gtk_widget_set_usize(w, width, height);
        }

        /* EDV_CFG_PARM_ARCHIVER_CONTENTS_COLUMN_WIDTH parameter is updated
         * as they change in EDVArchiverResizeColumnCB() or when the
         * configuration list is changed directly.
         */

        /* EDV_CFG_PARM_ARCHIVER_CONTENTS_COLUMN parameter is updated
         * when the configuration list is changed directly.
         */


}

/*
 *	Clears the contents clist and loads the listing of the archive
 *	specified by arch_obj if it exists.
 */
void EDVArchiverSelectArchive(
        edv_archiver_struct *archiver, const gchar *arch_obj
)
{
        gchar *darch_obj;
	GtkCList *clist;


        if((archiver == NULL) || (arch_obj == NULL))
            return;

	clist = (GtkCList *)archiver->contents_clist;
	if(clist == NULL)
	    return;

        darch_obj = g_strdup(arch_obj);
        if(darch_obj == NULL)
            return;

        /* Update title and location bar. */
        EDVArchiverSetTitle(archiver, darch_obj);
        EDVArchiverSetLocation(archiver, darch_obj, FALSE);
        EDVArchiverUpdateLocationIcon(archiver, darch_obj);

        /* Clear contents list and get listing of the newly set archive. */
	gtk_clist_freeze(clist);
	EDVArchiverContentsDoUpdate(archiver, TRUE);
	gtk_clist_thaw(clist);


        g_free(darch_obj);
        darch_obj = NULL;
}


/*
 *      Creates a new standard list of recycle bin op ids.
 */
edv_archiver_opid_struct **EDVArchiverOPIDListNew(
        edv_archiver_struct *archiver,      /* Can be NULL. */
        gint *total
)
{
        gint i;
        edv_archiver_opid_struct **list, *opid;
        gint op, toolbar_item_type;
        gbool allow_multiple;
        void (*func_cb)(gpointer);
        void (*enter_func_cb)(gpointer);
        const gchar *button_name, *menu_name, *tooltip;
        gint accel_key;
        guint accel_mods;
        guint8 **button_icon_data, **menu_icon_data;


        if(total == NULL)
            return(NULL);

        *total = 0;
        list = NULL;

#define DO_APPEND_OPID  \
{ \
 i = *total; \
 *total = i + 1; \
 list = (edv_archiver_opid_struct **)g_realloc( \
  list, \
  (*total) * sizeof(edv_archiver_opid_struct *) \
 ); \
 if(list == NULL) \
 { \
  *total = 0; \
  return(list); \
 } \
 else \
 { \
  list[i] = opid = (edv_archiver_opid_struct *)g_malloc0( \
   sizeof(edv_archiver_opid_struct) \
  ); \
  if(opid != NULL) \
  { \
   opid->archiver = archiver; \
   opid->op = op; \
   opid->allow_multiple = allow_multiple; \
   opid->toolbar_item_type = toolbar_item_type; \
   opid->func_cb = func_cb; \
   opid->enter_func_cb = enter_func_cb; \
   opid->button_name = (button_name != NULL) ? g_strdup(button_name) : NULL; \
   opid->menu_name = (menu_name != NULL) ? g_strdup(menu_name) : NULL; \
   opid->tooltip = (tooltip != NULL) ? g_strdup(tooltip) : NULL; \
   opid->accel_key = accel_key; \
   opid->accel_mods = accel_mods; \
   opid->button_icon_data = button_icon_data; \
   opid->menu_icon_data = menu_icon_data; \
  } \
 } \
}

        func_cb = EDVArchiverOPCB;
        enter_func_cb = EDVArchiverOPEnterCB;


        op = EDV_ARCHIVER_OP_SEPARATOR;
	toolbar_item_type = TOOLBAR_ITEM_SEPARATOR;
        allow_multiple = TRUE;
        button_icon_data = NULL;
        menu_icon_data = NULL;
#ifdef PROG_LANGUAGE_ENGLISH
        button_name = "Separator";
        menu_name = "Separator";
        tooltip = "Separator";
#endif
#ifdef PROG_LANGUAGE_SPANISH
        button_name = "Separador";
        menu_name = "Separador";
        tooltip = "Separador";
#endif
#ifdef PROG_LANGUAGE_FRENCH
        button_name = "Sparateur";
        menu_name = "Sparateur";
        tooltip = "Sparateur";
#endif
        accel_key = 0;
        accel_mods = 0;
        DO_APPEND_OPID


        op = EDV_ARCHIVER_OP_CLOSE;
	toolbar_item_type = TOOLBAR_ITEM_BUTTON;
        allow_multiple = FALSE;
        button_icon_data = (guint8 **)icon_close_20x20_xpm;
        menu_icon_data = button_icon_data;
#ifdef PROG_LANGUAGE_ENGLISH
        button_name = "Close";
        menu_name = "Close";
        tooltip = "Close this window";
#endif
#ifdef PROG_LANGUAGE_SPANISH
        button_name = "Cierre";
        menu_name = "Cierre";
        tooltip = "Cierre esta ventana";
#endif
#ifdef PROG_LANGUAGE_FRENCH
        button_name = "Proche";
        menu_name = "Proche";
        tooltip = "Fermer cette fentre";
#endif
        accel_key = 'w';
        accel_mods = GDK_CONTROL_MASK;
        DO_APPEND_OPID

        op = EDV_ARCHIVER_OP_EXIT;
        toolbar_item_type = TOOLBAR_ITEM_BUTTON;
        allow_multiple = FALSE;
        button_icon_data = (guint8 **)icon_exit_20x20_xpm;
        menu_icon_data = button_icon_data;
#ifdef PROG_LANGUAGE_ENGLISH
        button_name = "Exit";
        menu_name = "Exit";
        tooltip = "Close all windows in the application and exit";
#endif
#ifdef PROG_LANGUAGE_SPANISH
        button_name = "La Salida";
        menu_name = "La Salida";
        tooltip = "Cierra todas ventanas en la aplicacin y la";
#endif
#ifdef PROG_LANGUAGE_FRENCH
        button_name = "Sortie";
        menu_name = "Sortie";
        tooltip = "Ferme toutes fentres dans l'application et sortie";
#endif
        accel_key = 0;
        accel_mods = 0;
        DO_APPEND_OPID


        op = EDV_ARCHIVER_OP_SYNC_DISKS;
        toolbar_item_type = TOOLBAR_ITEM_BUTTON;
        allow_multiple = FALSE;
        button_icon_data = (guint8 **)icon_save_20x20_xpm;
        menu_icon_data = button_icon_data;
#ifdef PROG_LANGUAGE_ENGLISH
        button_name = "Sync Disks";
        menu_name = "Sync Disks";
        tooltip = "Flush any pending data transfers to disk";
#endif
#ifdef PROG_LANGUAGE_SPANISH
        button_name = "Discos Sincro";
        menu_name = "Discos Sincro";
        tooltip = "Limpan cualquiera transferencias pendientes de datos al disco";
#endif
#ifdef PROG_LANGUAGE_FRENCH
        button_name = "Disq Syn";
        menu_name = "Les Disques De Syn";
        tooltip = "Rincer des transferts de donnes en attente au disque";
#endif
        accel_key = 0;
        accel_mods = 0;
        DO_APPEND_OPID

        op = EDV_ARCHIVER_OP_RUN;
        toolbar_item_type = TOOLBAR_ITEM_BUTTON;
        allow_multiple = FALSE;
        button_icon_data = (guint8 **)icon_executable_20x20_xpm;
        menu_icon_data = button_icon_data;
#ifdef PROG_LANGUAGE_ENGLISH
        button_name = "Run";
        menu_name = "Run...";
        tooltip = "Run a program";
#endif
#ifdef PROG_LANGUAGE_SPANISH
        button_name = "Corra";
        menu_name = "Corra...";
        tooltip = "Corra un programa";
#endif
#ifdef PROG_LANGUAGE_FRENCH
        button_name = "Courir";
        menu_name = "Courir...";
        tooltip = "Excute un programme";
#endif
        accel_key = 0;
        accel_mods = 0;
        DO_APPEND_OPID

        op = EDV_ARCHIVER_OP_RUN_TERMINAL;
        toolbar_item_type = TOOLBAR_ITEM_BUTTON;
        allow_multiple = FALSE;
        button_icon_data = (guint8 **)icon_terminal_20x20_xpm;
        menu_icon_data = button_icon_data;
#ifdef PROG_LANGUAGE_ENGLISH
        button_name = "Terminal";
        menu_name = "Terminal";
        tooltip = "Run a terminal";
#endif
#ifdef PROG_LANGUAGE_SPANISH
        button_name = "La Terminal";
        menu_name = "La Terminal";
        tooltip = "Corre una terminal";
#endif
#ifdef PROG_LANGUAGE_FRENCH
        button_name = "Le Terminal";
        menu_name = "Le Terminal";
        tooltip = "Courir un terminal";
#endif
        accel_key = 0;
        accel_mods = 0;
        DO_APPEND_OPID

        op = EDV_ARCHIVER_OP_WRITE_PROTECT;
        toolbar_item_type = TOOLBAR_ITEM_BUTTON;
        allow_multiple = FALSE;
        button_icon_data = (guint8 **)icon_secure_20x20_xpm;
        menu_icon_data = button_icon_data;
#ifdef PROG_LANGUAGE_ENGLISH
        button_name = "Protect";
        menu_name = "Write Protect";
        tooltip = "Enable/disable write protect";
#endif
#ifdef PROG_LANGUAGE_SPANISH
        button_name = "Proteja";
        menu_name = "Escriba Proteja";
        tooltip = "Habilite/incapacita escribe protege";
#endif
#ifdef PROG_LANGUAGE_FRENCH
        button_name = "Protge";
        menu_name = "Protge En criture";
        tooltip = "Rend capable/rend infirme protge en criture";
#endif
        accel_key = 0;
        accel_mods = 0;
        DO_APPEND_OPID


        op = EDV_ARCHIVER_OP_NEW;
        toolbar_item_type = TOOLBAR_ITEM_BUTTON;
        allow_multiple = FALSE;
        button_icon_data = (guint8 **)icon_new_20x20_xpm;
        menu_icon_data = button_icon_data;
#ifdef PROG_LANGUAGE_ENGLISH
        button_name = "New";
        menu_name = "New...";
        tooltip = "Create a new archive";
#endif
#ifdef PROG_LANGUAGE_SPANISH
        button_name = "Nuevo";
        menu_name = "Nuevo...";
        tooltip = "Crea un archivo nuevo";
#endif
#ifdef PROG_LANGUAGE_FRENCH
        button_name = "Nouveau";
        menu_name = "Nouveau...";
        tooltip = "Crer une nouvelle archive";
#endif
        accel_key = 'n';
        accel_mods = GDK_CONTROL_MASK;
        DO_APPEND_OPID

        op = EDV_ARCHIVER_OP_OPEN;
        toolbar_item_type = TOOLBAR_ITEM_BUTTON;
        allow_multiple = FALSE;
        button_icon_data = (guint8 **)icon_open_20x20_xpm;
        menu_icon_data = button_icon_data;
#ifdef PROG_LANGUAGE_ENGLISH
        button_name = "Open";
        menu_name = "Open...";
        tooltip = "Open an archive";
#endif
#ifdef PROG_LANGUAGE_SPANISH
        button_name = "Abierto";
        menu_name = "Abierto...";
        tooltip = "Abre un archivo";
#endif
#ifdef PROG_LANGUAGE_FRENCH
        button_name = "Ouvert";
        menu_name = "Ouvert...";
        tooltip = "Ouvrir une archive";
#endif
        accel_key = 'o';
        accel_mods = GDK_CONTROL_MASK;
        DO_APPEND_OPID

        op = EDV_ARCHIVER_OP_ADD;
        toolbar_item_type = TOOLBAR_ITEM_BUTTON;
        allow_multiple = FALSE;
        button_icon_data = (guint8 **)icon_archive_add_20x20_xpm;
        menu_icon_data = button_icon_data;
#ifdef PROG_LANGUAGE_ENGLISH
        button_name = "Add";
        menu_name = "Add...";
        tooltip = "Add objects to archive";
#endif
#ifdef PROG_LANGUAGE_SPANISH
        button_name = "Agregue";
        menu_name = "Agregue...";
        tooltip = "Agregue objetos al archivo";
#endif
#ifdef PROG_LANGUAGE_FRENCH
        button_name = "Ajouter";
        menu_name = "Ajouter...";
        tooltip = "Ajoute des objets  l'archive";
#endif
        accel_key = '+';
        accel_mods = GDK_CONTROL_MASK;
        DO_APPEND_OPID

        op = EDV_ARCHIVER_OP_EXTRACT;
        toolbar_item_type = TOOLBAR_ITEM_BUTTON;
        allow_multiple = FALSE;
        button_icon_data = (guint8 **)icon_archive_extract_20x20_xpm;
        menu_icon_data = button_icon_data;
#ifdef PROG_LANGUAGE_ENGLISH
        button_name = "Extract";
        menu_name = "Extract...";
        tooltip = "Extract selected object(s) from archive";
#endif
#ifdef PROG_LANGUAGE_SPANISH
        button_name = "Extracto";
        menu_name = "El Extracto...";
        tooltip = "El Extracto escogi se opone(s) del archivo";
#endif
#ifdef PROG_LANGUAGE_FRENCH
        button_name = "L'Extrait";
        menu_name = "L'Extrait...";
        tooltip = "L'Extrait a choisi l'objet (s) de l'archive";
#endif
        accel_key = GDK_Return;
        accel_mods = GDK_CONTROL_MASK;
        DO_APPEND_OPID

        op = EDV_ARCHIVER_OP_EXTRACT_ALL;
        toolbar_item_type = TOOLBAR_ITEM_BUTTON;
        allow_multiple = FALSE;
        button_icon_data = (guint8 **)icon_archive_extract_20x20_xpm;
        menu_icon_data = button_icon_data;
#ifdef PROG_LANGUAGE_ENGLISH
        button_name = "ExtractAll";
        menu_name = "Extract All...";
        tooltip = "Extract all objects from archive";
#endif
#ifdef PROG_LANGUAGE_SPANISH
        button_name = "ExtrTodo";
        menu_name = "Extraiga Todo...";
        tooltip = "Extrae todo se opone del archivo";
#endif
#ifdef PROG_LANGUAGE_FRENCH
        button_name = "ExtrTout";
        menu_name = "Extraire Tout...";
        tooltip = "Extraire tous objets de l'archive";
#endif
        accel_key = GDK_Return;
        accel_mods = GDK_CONTROL_MASK | GDK_SHIFT_MASK;
        DO_APPEND_OPID

        op = EDV_ARCHIVER_OP_DELETE;
        toolbar_item_type = TOOLBAR_ITEM_BUTTON;
        allow_multiple = FALSE;
        button_icon_data = (guint8 **)icon_cancel_20x20_xpm;
        menu_icon_data = button_icon_data;
#ifdef PROG_LANGUAGE_ENGLISH
        button_name = "Delete";
        menu_name = "Delete";
        tooltip = "Delete selected object(s) from archive";
#endif
#ifdef PROG_LANGUAGE_SPANISH
        button_name = "Borre";
        menu_name = "Borre";
        tooltip = "Borre escogido se opone(s) del archivo";
#endif
#ifdef PROG_LANGUAGE_FRENCH
        button_name = "Effacer";
        menu_name = "Effacer";
        tooltip = "Efface l'objet choisi (s) de l'archive";
#endif
        accel_key = GDK_Delete;
        accel_mods = GDK_CONTROL_MASK;
        DO_APPEND_OPID

        op = EDV_ARCHIVER_OP_SELECT_ALL;
        toolbar_item_type = TOOLBAR_ITEM_BUTTON;
        allow_multiple = FALSE;
        button_icon_data = (guint8 **)icon_select_20x20_xpm;
        menu_icon_data = NULL;
#ifdef PROG_LANGUAGE_ENGLISH
        button_name = "SelAll";
        menu_name = "Select All";
        tooltip = "Select all objects";
#endif
#ifdef PROG_LANGUAGE_SPANISH
        button_name = "SelAll";
        menu_name = "Escoja Todo";
        tooltip = "Escoge todo se opone";
#endif
#ifdef PROG_LANGUAGE_FRENCH
        button_name = "SelAll";
        menu_name = "Choisir Tout";
        tooltip = "Choisir tous objets";
#endif
        accel_key = 'a';
        accel_mods = GDK_CONTROL_MASK;
        DO_APPEND_OPID

        op = EDV_ARCHIVER_OP_UNSELECT_ALL;
        toolbar_item_type = TOOLBAR_ITEM_BUTTON;
        allow_multiple = FALSE;
        button_icon_data = (guint8 **)icon_select_20x20_xpm;
        menu_icon_data = NULL;
#ifdef PROG_LANGUAGE_ENGLISH
        button_name = "UnSelAll";
        menu_name = "Unselect All";
        tooltip = "Unselect all objects";
#endif
#ifdef PROG_LANGUAGE_SPANISH
        button_name = "UnSelAll";
        menu_name = "Unescoja Todo";
        tooltip = "Unescoge todo se opone";
#endif
#ifdef PROG_LANGUAGE_FRENCH
        button_name = "UnSelAll";
        menu_name = "Unchoisir Tout";
        tooltip = "Unchoisir tous objets";
#endif
        accel_key = 'u';
        accel_mods = GDK_CONTROL_MASK;
        DO_APPEND_OPID

        op = EDV_ARCHIVER_OP_INVERT_SELECTION;
        toolbar_item_type = TOOLBAR_ITEM_BUTTON;
        allow_multiple = FALSE;
        button_icon_data = (guint8 **)icon_select_20x20_xpm;
        menu_icon_data = NULL;
#ifdef PROG_LANGUAGE_ENGLISH
        button_name = "InvSel";
        menu_name = "Invert Selection";
        tooltip =
 "Select all unselected objects and unselect all selected objects";
#endif
#ifdef PROG_LANGUAGE_SPANISH
        button_name = "InvSel";
        menu_name = "Invierta Seleccin";
        tooltip =
 "Selectos todos objetos y unselect no seleccionados todos objetos escogidos";
#endif
#ifdef PROG_LANGUAGE_FRENCH
        button_name = "InvSel";
        menu_name = "Inverser Slection";
        tooltip =
 "Choisir tous objets dslections et deslectionner tous objets choisis";
#endif
        accel_key = 'i';
        accel_mods = GDK_CONTROL_MASK;
        DO_APPEND_OPID

        op = EDV_ARCHIVER_OP_FIND;
        toolbar_item_type = TOOLBAR_ITEM_BUTTON;
        allow_multiple = FALSE;
        button_icon_data = (guint8 **)icon_search_20x20_xpm;;
        menu_icon_data = button_icon_data;
#ifdef PROG_LANGUAGE_ENGLISH
        button_name = "Find";
        menu_name = "Find...";
        tooltip = "Find object(s) by specific attributes";
#endif
#ifdef PROG_LANGUAGE_SPANISH
        button_name = "Hallazgo";
        menu_name = "El Hallazgo";
        tooltip = "El Hallazgo se opone (s) por atributos especificados";
#endif
#ifdef PROG_LANGUAGE_FRENCH
        button_name = "Dcouverte";
        menu_name = "La Dcouverte";
        tooltip = "L'objet de Dcouverte (s) par les attributs spcifis ";
#endif
        accel_key = 'f';
        accel_mods = GDK_CONTROL_MASK;
        DO_APPEND_OPID

        op = EDV_ARCHIVER_OP_HISTORY;
        toolbar_item_type = TOOLBAR_ITEM_BUTTON;
        allow_multiple = FALSE;
        button_icon_data = (guint8 **)icon_processes_20x20_xpm;
        menu_icon_data = NULL;
#ifdef PROG_LANGUAGE_ENGLISH
        button_name = "History";
        menu_name = "History...";
        tooltip = "History of commands processed by this application";
#endif
#ifdef PROG_LANGUAGE_SPANISH
        button_name = "Historia";
        menu_name = "La Historia...";
        tooltip = "La Historia de rdenes procesadas por esta aplicacin";
#endif
#ifdef PROG_LANGUAGE_FRENCH
        button_name = "L'Histoire";
        menu_name = "L'Histoire...";
        tooltip = "L'Histoire d'ordres traits par cette application";
#endif
        accel_key = 0;
        accel_mods = 0;
        DO_APPEND_OPID

        op = EDV_ARCHIVER_OP_REFRESH;
        toolbar_item_type = TOOLBAR_ITEM_BUTTON;
        allow_multiple = FALSE;
        button_icon_data = (guint8 **)icon_reload_20x20_xpm;
        menu_icon_data = button_icon_data;
#ifdef PROG_LANGUAGE_ENGLISH
        button_name = "Refresh";
        menu_name = "Refresh";
        tooltip = "Refresh archive objects listing";
#endif
#ifdef PROG_LANGUAGE_SPANISH
        button_name = "Refresque";
        menu_name = "Refresque";
        tooltip = "Refresque objetos de archivo que listan";
#endif
#ifdef PROG_LANGUAGE_FRENCH
        button_name = "Rafrachir";
        menu_name = "Rafrachir";
        tooltip = "Rafrachit les objets d'archive numrants";
#endif
        accel_key = GDK_F5;
        accel_mods = 0;
        DO_APPEND_OPID

        op = EDV_ARCHIVER_OP_REFRESH_ALL;
        toolbar_item_type = TOOLBAR_ITEM_BUTTON;
        allow_multiple = FALSE;
        button_icon_data = (guint8 **)icon_reload_20x20_xpm;
        menu_icon_data = button_icon_data;
#ifdef PROG_LANGUAGE_ENGLISH
        button_name = "Refresh All";
        menu_name = "Refresh All";
        tooltip = "Refresh all listings";
#endif
#ifdef PROG_LANGUAGE_SPANISH
        button_name = "Refresque Todo";
        menu_name = "Refresque Todo";
        tooltip = "Refresca todas listas";
#endif
#ifdef PROG_LANGUAGE_FRENCH
        button_name = "Rafrachir Tout";
        menu_name = "Rafrachir Tout";
        tooltip = "Rafrachir toutes listes";
#endif
        accel_key = 0;
        accel_mods = 0;
        DO_APPEND_OPID

        op = EDV_ARCHIVER_OP_SHOW_TOOL_BAR;
        toolbar_item_type = TOOLBAR_ITEM_TOGGLE_BUTTON;
        allow_multiple = FALSE;
        button_icon_data = (guint8 **)icon_tool_bar_20x20_xpm;
        menu_icon_data = button_icon_data;
#ifdef PROG_LANGUAGE_ENGLISH
        button_name = "ToolBar";
        menu_name = "Tool Bar";
        tooltip = "Show/hide tool bar";
#endif
#ifdef PROG_LANGUAGE_SPANISH
        button_name = "ToolBar";
        menu_name = "Tool Bar";
        tooltip = "La Exposicin/la barra de herramienta de cuero";
#endif
#ifdef PROG_LANGUAGE_FRENCH
        button_name = "ToolBar";
        menu_name = "Tool Bar";
        tooltip = "La barre d'outil de montre/peau";
#endif
        accel_key = 0;
        accel_mods = 0;
        DO_APPEND_OPID

        op = EDV_ARCHIVER_OP_SHOW_LOCATION_BAR;
        toolbar_item_type = TOOLBAR_ITEM_TOGGLE_BUTTON;
        allow_multiple = FALSE;
        button_icon_data = (guint8 **)icon_tool_bar_20x20_xpm;
        menu_icon_data = button_icon_data;
#ifdef PROG_LANGUAGE_ENGLISH
        button_name = "LocBar";
        menu_name = "Location Bar";
        tooltip = "Show/hide location bar";
#endif
#ifdef PROG_LANGUAGE_SPANISH
        button_name = "LocBar";
        menu_name = "Location Bar";
        tooltip = "La exposicin/la barra de la ubicacin del cuero";
#endif
#ifdef PROG_LANGUAGE_FRENCH
        button_name = "LocBar";
        menu_name = "Location Bar";
        tooltip = "La barre d'emplacement de montre/peau";
#endif
        accel_key = 0;
        accel_mods = 0;
        DO_APPEND_OPID

        op = EDV_ARCHIVER_OP_SHOW_FIND_BAR;
        toolbar_item_type = TOOLBAR_ITEM_TOGGLE_BUTTON;
        allow_multiple = FALSE;
        button_icon_data = (guint8 **)icon_tool_bar_20x20_xpm;
        menu_icon_data = button_icon_data;
#ifdef PROG_LANGUAGE_ENGLISH
        button_name = "FindBar";
        menu_name = "Find Bar";
        tooltip = "Show/hide find bar";
#endif
#ifdef PROG_LANGUAGE_SPANISH
        button_name = "FindBar";
        menu_name = "Find Bar";
        tooltip = "La exposicin/la barra del hallazgo del cuero";
#endif
#ifdef PROG_LANGUAGE_FRENCH
        button_name = "FindBar";
        menu_name = "Find Bar";
        tooltip = "La barre de dcouverte de montre/peau";
#endif
        accel_key = 0;
        accel_mods = 0;
        DO_APPEND_OPID

        op = EDV_ARCHIVER_OP_SHOW_STATUS_BAR;
        toolbar_item_type = TOOLBAR_ITEM_TOGGLE_BUTTON;
        allow_multiple = FALSE;
        button_icon_data = (guint8 **)icon_status_bar_20x20_xpm;
        menu_icon_data = button_icon_data;
#ifdef PROG_LANGUAGE_ENGLISH
        button_name = "StatusBar";
        menu_name = "Status Bar";
        tooltip = "Show/hide status bar";
#endif
#ifdef PROG_LANGUAGE_SPANISH
        button_name = "StatusBar";
        menu_name = "Status Bar";
        tooltip = "La exposicin/la barra de la posicin del cuero";
#endif
#ifdef PROG_LANGUAGE_FRENCH
        button_name = "StatusBar";
        menu_name = "Status Bar";
        tooltip = "La barre de statut de montre/peau";
#endif
        accel_key = 0;
        accel_mods = 0;
        DO_APPEND_OPID


        op = EDV_ARCHIVER_OP_MIME_TYPES;
        toolbar_item_type = TOOLBAR_ITEM_BUTTON;
        allow_multiple = FALSE;
        button_icon_data = (guint8 **)icon_mimetypes_20x20_xpm;
        menu_icon_data = button_icon_data;
#ifdef PROG_LANGUAGE_ENGLISH
        button_name = "MIMETypes";
        menu_name = "MIME Types...";
        tooltip = "View/modify MIME Types";
#endif
#ifdef PROG_LANGUAGE_SPANISH
        button_name = "MIMETypes";
        menu_name = "MIME Types...";
        tooltip = "El panorama/modifica MIME Types";
#endif
#ifdef PROG_LANGUAGE_FRENCH
        button_name = "MIMETypes";
        menu_name = "MIME Types...";
        tooltip = "Vue/modifie MIME Types";
#endif
        accel_key = 'm';
        accel_mods = GDK_CONTROL_MASK;
        DO_APPEND_OPID


        op = EDV_ARCHIVER_OP_NEW_BROWSER;
        toolbar_item_type = TOOLBAR_ITEM_BUTTON;
        allow_multiple = FALSE;
        button_icon_data = (guint8 **)endeavour_20x20_xpm;
        menu_icon_data = button_icon_data;
#ifdef PROG_LANGUAGE_ENGLISH
        button_name = "FBrowser";
        menu_name = "New File Browser";
        tooltip = "Create a new file browser";
#endif
#ifdef PROG_LANGUAGE_SPANISH
        button_name = "FBrowser";
        menu_name = "El Examinador Nuevo Del Archivo";
        tooltip = "Crea a un examinador nuevo del archivo";
#endif
#ifdef PROG_LANGUAGE_FRENCH
        button_name = "FBrowser";
        menu_name = "Nouveau Browser de Fichier";
        tooltip = "Crer un nouveau browser de fichier";
#endif
        accel_key = 0;
        accel_mods = 0;
        DO_APPEND_OPID

        op = EDV_ARCHIVER_OP_NEW_IMBR;
        toolbar_item_type = TOOLBAR_ITEM_BUTTON;
        allow_multiple = FALSE;
        button_icon_data = (guint8 **)image_browser_20x20_xpm;
        menu_icon_data = button_icon_data;
#ifdef PROG_LANGUAGE_ENGLISH
        button_name = "IBrowser";
        menu_name = "New Image Browser";
        tooltip = "Create a new image browser";
#endif
#ifdef PROG_LANGUAGE_SPANISH
        button_name = "IBrowser";
        menu_name = "El Examinador Nuevo De La Imagen";
        tooltip = "Crea a un examinador nuevo de la imagen";
#endif
#ifdef PROG_LANGUAGE_FRENCH
        button_name = "IBrowser";
        menu_name = "Nouveau Browser D'Image";
        tooltip = "Crer un nouveau browser d'image";
#endif
        accel_key = 0;
        accel_mods = 0;
        DO_APPEND_OPID

        op = EDV_ARCHIVER_OP_NEW_ARCHIVER;
        toolbar_item_type = TOOLBAR_ITEM_BUTTON;
        allow_multiple = FALSE;
        button_icon_data = (guint8 **)archiver_20x20_xpm;
        menu_icon_data = button_icon_data;
#ifdef PROG_LANGUAGE_ENGLISH
        button_name = "Archiver";
        menu_name = "New Archiver";
        tooltip = "Create a new archiver";
#endif
#ifdef PROG_LANGUAGE_SPANISH
        button_name = "Archiver";
        menu_name = "Archiver Nuevo";
        tooltip = "Crea un archiver nuevo";
#endif
#ifdef PROG_LANGUAGE_FRENCH
        button_name = "Archiver";
        menu_name = "Nouveau Archiver";
        tooltip = "Crer un nouveau archiver";
#endif
        accel_key = 0;
        accel_mods = 0;
        DO_APPEND_OPID

        op = EDV_ARCHIVER_OP_RECYCLE_BIN;
        toolbar_item_type = TOOLBAR_ITEM_BUTTON;
        allow_multiple = FALSE;
        button_icon_data = (guint8 **)icon_trash_20x20_xpm;
        menu_icon_data = button_icon_data;
#ifdef PROG_LANGUAGE_ENGLISH
        button_name = "RecBin";
        menu_name = "Recycle Bin";
        tooltip = "Recycle bin (trash can)";
#endif
#ifdef PROG_LANGUAGE_SPANISH
        button_name = "RecBin";
        menu_name = "El Cajn De La Recirculacin";
        tooltip = "Cajn de Recirculacin (basura puede)";
#endif
#ifdef PROG_LANGUAGE_FRENCH
        button_name = "RecBin";
        menu_name = "Recycler L'Huche";
        tooltip = "Recycler l'huche (la poubelle)";
#endif
        accel_key = 0;
        accel_mods = 0;
        DO_APPEND_OPID

        op = EDV_ARCHIVER_OP_OPTIONS;
        toolbar_item_type = TOOLBAR_ITEM_BUTTON;
        allow_multiple = FALSE;
        button_icon_data = (guint8 **)icon_options2_20x20_xpm;
        menu_icon_data = button_icon_data;
#ifdef PROG_LANGUAGE_ENGLISH
        button_name = "Options";
        menu_name = "Options...";
        tooltip = "Modify this application's options";
#endif
#ifdef PROG_LANGUAGE_SPANISH
        button_name = "Las Opciones";
        menu_name = "Las Opciones...";
        tooltip = "Modifican estas opciones de la aplicacin";
#endif
#ifdef PROG_LANGUAGE_FRENCH
        button_name = "Les Options";
        menu_name = "Les Options";
        tooltip = "Modifier ces options de l'application";
#endif
        accel_key = 'o';
        accel_mods = GDK_CONTROL_MASK;
        DO_APPEND_OPID

        op = EDV_ARCHIVER_OP_CUSTOMIZE;
        toolbar_item_type = TOOLBAR_ITEM_BUTTON;
        allow_multiple = FALSE;
        button_icon_data = (guint8 **)icon_customize_20x20_xpm;
        menu_icon_data = button_icon_data;
#ifdef PROG_LANGUAGE_ENGLISH
        button_name = "Custom";
        menu_name = "Customize...";
        tooltip = "Customize the look and feel of this application";
#endif
#ifdef PROG_LANGUAGE_SPANISH
        button_name = "Personal";
        menu_name = "Personaliza...";
        tooltip = "Personaliza la mirada y se siente de esta aplicacin";
#endif
#ifdef PROG_LANGUAGE_FRENCH
        button_name = "Personnal";
        menu_name = "Personnalise...";
        tooltip = "Personnalise le regard et sens de cette application";
#endif
        accel_key = 't';
        accel_mods = GDK_CONTROL_MASK;
        DO_APPEND_OPID


        op = EDV_ARCHIVER_OP_HELP_CONTENTS;
        toolbar_item_type = TOOLBAR_ITEM_BUTTON;
        allow_multiple = FALSE;
        button_icon_data = (guint8 **)icon_help_20x20_xpm;
        menu_icon_data = NULL;
#ifdef PROG_LANGUAGE_ENGLISH
        button_name = "Help";
        menu_name = "Contents";
        tooltip = "Index of all help pages";
#endif
#ifdef PROG_LANGUAGE_SPANISH
        button_name = "La Ayuda";
        menu_name = "El Contenido";
        tooltip = "El Indice de toda ayuda pagina";
#endif
#ifdef PROG_LANGUAGE_FRENCH
        button_name = "L'Aide";
        menu_name = "Les Contenus";
        tooltip = "L'Index de toutes pages d'aide";
#endif
        accel_key = GDK_F1;
        accel_mods = 0;
        DO_APPEND_OPID

        op = EDV_ARCHIVER_OP_HELP_FILE_BROWSER;
        toolbar_item_type = TOOLBAR_ITEM_BUTTON;
        allow_multiple = FALSE;
        button_icon_data = (guint8 **)icon_help_20x20_xpm;
        menu_icon_data = NULL;
        button_name = "Help";
        menu_name = "File Browser";
        tooltip = "Help on using the file browser";
        accel_key = 0;
        accel_mods = 0;
        DO_APPEND_OPID

        op = EDV_ARCHIVER_OP_HELP_IMAGE_BROWSER;
        toolbar_item_type = TOOLBAR_ITEM_BUTTON;
        allow_multiple = FALSE;
        button_icon_data = (guint8 **)icon_help_20x20_xpm;
        menu_icon_data = NULL;
        button_name = "Help";
        menu_name = "Image Browser";
        tooltip = "Help on using the image browser";
        accel_key = 0;
        accel_mods = 0;
        DO_APPEND_OPID

        op = EDV_ARCHIVER_OP_HELP_ARCHIVER;
        toolbar_item_type = TOOLBAR_ITEM_BUTTON;
        allow_multiple = FALSE;
        button_icon_data = (guint8 **)icon_help_20x20_xpm;
        menu_icon_data = NULL;
        button_name = "Help";
        menu_name = "Archiver";
        tooltip = "Help on using the archiver";
        accel_key = 0;
        accel_mods = 0;
        DO_APPEND_OPID

        op = EDV_ARCHIVER_OP_HELP_RECYCLE_BIN;
        toolbar_item_type = TOOLBAR_ITEM_BUTTON;
        allow_multiple = FALSE;
        button_icon_data = (guint8 **)icon_help_20x20_xpm;
        menu_icon_data = NULL;
        button_name = "Help";
        menu_name = "Recycle Bin";
        tooltip = "Help on using the recycle bin";
        accel_key = 0;
        accel_mods = 0;
        DO_APPEND_OPID

        op = EDV_ARCHIVER_OP_HELP_KEYS_LIST;
        toolbar_item_type = TOOLBAR_ITEM_BUTTON;
        allow_multiple = FALSE;
        button_icon_data = (guint8 **)icon_help_20x20_xpm;
        menu_icon_data = NULL;
#ifdef PROG_LANGUAGE_ENGLISH
        button_name = "HelpKeys";
        menu_name = "Keys List";
        tooltip = "List of all keyboard key operations";
#endif
#ifdef PROG_LANGUAGE_SPANISH
        button_name = "HelpKeys";
        menu_name = "La Lista De Llaves";
        tooltip = "La Lista de todo teclado las operaciones claves";
#endif
#ifdef PROG_LANGUAGE_FRENCH
        button_name = "HelpKeys";
        menu_name = "La Liste De Clefs";
        tooltip = "La Liste de tout clavier toutes oprations principales";
#endif
        accel_key = 0;
        accel_mods = 0;
        DO_APPEND_OPID

        op = EDV_ARCHIVER_OP_HELP_COMMON_OPERATIONS;
        toolbar_item_type = TOOLBAR_ITEM_BUTTON;
        allow_multiple = FALSE;
        button_icon_data = (guint8 **)icon_help_20x20_xpm;
        menu_icon_data = NULL;
#ifdef PROG_LANGUAGE_ENGLISH
        button_name = "HelpMIME";
        menu_name = "Mime Types";
        tooltip = "Help on setting up MIME Types";
#endif
#ifdef PROG_LANGUAGE_SPANISH
        button_name = "HelpMIME";
        menu_name = "Mime Types";
        tooltip = "Ayude a establecer MIME Types";
#endif
#ifdef PROG_LANGUAGE_FRENCH
        button_name = "HelpMIME";
        menu_name = "Mime Types";
        tooltip = "Aider sur montant MIME Types";
#endif
        accel_key = 0;
        accel_mods = 0;
        DO_APPEND_OPID

        op = EDV_ARCHIVER_OP_HELP_ABOUT;
        toolbar_item_type = TOOLBAR_ITEM_BUTTON;
        allow_multiple = FALSE;
        button_icon_data = (guint8 **)icon_about_20x20_xpm;
        menu_icon_data = (guint8 **)icon_about_20x20_xpm;
#ifdef PROG_LANGUAGE_ENGLISH
        button_name = "About";
        menu_name = "About";
        tooltip = "About this application";
#endif
#ifdef PROG_LANGUAGE_SPANISH
        button_name = "Acerca";
        menu_name = "Acerca De";
        tooltip = "Acerca de esta aplicacin";
#endif
#ifdef PROG_LANGUAGE_FRENCH
        button_name = "De";
        menu_name = "De";
        tooltip = "De cette application";
#endif
        accel_key = 0;
        accel_mods = 0;
        DO_APPEND_OPID


#undef DO_APPEND_OPID

        return(list);
}

/*
 *      Deallocates the given list of recycle bin op ids.
 */
void EDVArchiverOPIDListDelete(
        edv_archiver_opid_struct **list, gint total
)
{
        gint i;
        edv_archiver_opid_struct *opid;


        if(list == NULL)
            return;

        for(i = 0; i < total; i++)
        {
            opid = list[i];
            if(opid == NULL)
                continue;

            g_free(opid->button_name);
            g_free(opid->menu_name);
            g_free(opid->tooltip);

            g_free(opid);
        }
        g_free(list);
        list = NULL;
        total = 0;
}


/*
 *      Creates the menu bar for the given recycle bin.
 *
 *      This function should be called by EDVArchiverNew().
 */
static void EDVArchiverCreateMenuBar(
        edv_archiver_struct *archiver, GtkWidget *parent
)
{
        gint op;
        guint8 **icon_data = NULL;
        GtkAccelGroup *accelgrp;
        GtkWidget *menu_bar, *menu, *w, **w_record, **fw_record;


        /* Get keyboard accelerator group. */
        accelgrp = archiver->accelgrp;

        /* Create menu bar. */
        menu_bar = (GtkWidget *)GUIMenuBarCreate(NULL);
        archiver->menu_bar = menu_bar;
        gtk_container_add(GTK_CONTAINER(parent), menu_bar);
        gtk_widget_show(menu_bar);

#define DO_ADD_MENU_ITEM_LABEL  \
{ \
 edv_archiver_opid_struct *opid = EDVArchiverMatchOPID(archiver, op); \
 if(opid != NULL) \
 { \
  w = GUIMenuItemCreate( \
   menu, GUI_MENU_ITEM_TYPE_LABEL, accelgrp, \
   (u_int8_t **)((icon_data != NULL) ? icon_data : opid->menu_icon_data), \
   opid->menu_name, \
   opid->accel_key, opid->accel_mods, \
   (void **)fw_record, \
   (gpointer)opid, EDVArchiverMenuItemCB \
  ); \
  GUISetMenuItemCrossingCB( \
   w, \
   (gpointer)EDVArchiverMenuItemEnterCB, (gpointer)opid, \
   (gpointer)EDVArchiverMenuItemLeaveCB, (gpointer)opid \
  ); \
  if(w_record != NULL) *w_record = w; \
 } \
}
#define DO_ADD_MENU_ITEM_SUBMENU        \
{ \
 edv_archiver_opid_struct *opid = EDVArchiverMatchOPID(archiver, op); \
 if(opid != NULL) \
 { \
  w = GUIMenuItemCreate( \
   menu, GUI_MENU_ITEM_TYPE_SUBMENU, accelgrp, \
   (u_int8_t **)((icon_data != NULL) ? icon_data : opid->menu_icon_data), \
   opid->menu_name, \
   opid->accel_key, opid->accel_mods, \
   (void **)fw_record, \
   (gpointer)opid, NULL \
  ); \
  GUISetMenuItemCrossingCB( \
   w, \
   (gpointer)EDVArchiverMenuItemEnterCB, (gpointer)opid, \
   (gpointer)EDVArchiverMenuItemLeaveCB, (gpointer)opid \
  ); \
  if(w != NULL) \
   GUIMenuItemSetSubMenu(w, submenu); \
  if(w_record != NULL) *w_record = w; \
 } \
}

#define DO_ADD_MENU_ITEM_CHECK  \
{ \
 edv_archiver_opid_struct *opid = EDVArchiverMatchOPID(archiver, op); \
 if(opid != NULL) \
 { \
  w = GUIMenuItemCreate( \
   menu, GUI_MENU_ITEM_TYPE_CHECK, accelgrp, \
   (u_int8_t **)((icon_data != NULL) ? icon_data : opid->menu_icon_data), \
   opid->menu_name, \
   opid->accel_key, opid->accel_mods, \
   (void **)fw_record, \
   (gpointer)opid, EDVArchiverMenuItemCB \
  ); \
  GUISetMenuItemCrossingCB( \
   w, \
   (gpointer)EDVArchiverMenuItemEnterCB, (gpointer)opid, \
   (gpointer)EDVArchiverMenuItemLeaveCB, (gpointer)opid \
  ); \
  if(w_record != NULL) *w_record = w; \
 } \
}

#define DO_ADD_MENU_SEP \
{ \
 w = GUIMenuItemCreate( \
  menu, GUI_MENU_ITEM_TYPE_SEPARATOR, NULL, \
  NULL, NULL, 0, 0, NULL, \
  NULL, NULL \
 ); \
}

        /* Create file menu. */
        menu = GUIMenuCreateTearOff();
        if(menu != NULL)
        {
            op = EDV_ARCHIVER_OP_NEW;
            w_record = &archiver->new_mi;
            fw_record = NULL;
            DO_ADD_MENU_ITEM_LABEL

            op = EDV_ARCHIVER_OP_OPEN;
            w_record = &archiver->open_mi;
            fw_record = NULL;
            DO_ADD_MENU_ITEM_LABEL

            DO_ADD_MENU_SEP

            op = EDV_ARCHIVER_OP_SYNC_DISKS;
            w_record = &archiver->sync_disks_mi;
            fw_record = NULL;
            DO_ADD_MENU_ITEM_LABEL

            DO_ADD_MENU_SEP

            op = EDV_ARCHIVER_OP_HISTORY;
            w_record = &archiver->history_mi;
            fw_record = NULL;
            DO_ADD_MENU_ITEM_LABEL

            DO_ADD_MENU_SEP

            op = EDV_ARCHIVER_OP_RUN;
            w_record = &archiver->run_mi;
            fw_record = NULL;
            DO_ADD_MENU_ITEM_LABEL

            DO_ADD_MENU_SEP

            op = EDV_ARCHIVER_OP_CLOSE;
            w_record = &archiver->close_mi;
            fw_record = NULL;
            DO_ADD_MENU_ITEM_LABEL
        }
        GUIMenuAddToMenuBar(
            menu_bar, menu,
#ifdef PROG_LANGUAGE_ENGLISH
            "File",
#endif
#ifdef PROG_LANGUAGE_SPANISH
	    "Archivo",
#endif
#ifdef PROG_LANGUAGE_FRENCH
	    "Fichier",
#endif
            GUI_MENU_BAR_ALIGN_LEFT
        );


        /* Create edit menu. */
        menu = GUIMenuCreateTearOff();
        if(menu != NULL)
        {
            op = EDV_ARCHIVER_OP_ADD;
            w_record = &archiver->edit_add_mi;
            fw_record = NULL;
            DO_ADD_MENU_ITEM_LABEL

            op = EDV_ARCHIVER_OP_EXTRACT;
            w_record = &archiver->edit_extract_mi;
            fw_record = NULL;
            DO_ADD_MENU_ITEM_LABEL

            op = EDV_ARCHIVER_OP_EXTRACT_ALL;
            w_record = &archiver->edit_extract_all_mi;
            fw_record = NULL;
            DO_ADD_MENU_ITEM_LABEL

            op = EDV_ARCHIVER_OP_DELETE;
            w_record = &archiver->edit_delete_mi;
            fw_record = NULL;
            DO_ADD_MENU_ITEM_LABEL

            DO_ADD_MENU_SEP

            op = EDV_ARCHIVER_OP_SELECT_ALL;
            w_record = &archiver->select_all_mi;
            fw_record = NULL;
            DO_ADD_MENU_ITEM_LABEL

            op = EDV_ARCHIVER_OP_UNSELECT_ALL;
            w_record = &archiver->unselect_all_mi;
            fw_record = NULL;
            DO_ADD_MENU_ITEM_LABEL

            op = EDV_ARCHIVER_OP_INVERT_SELECTION;
            w_record = &archiver->invert_selection_mi;
            fw_record = NULL;
            DO_ADD_MENU_ITEM_LABEL

            op = EDV_ARCHIVER_OP_FIND;
            w_record = &archiver->find_mi;
            fw_record = NULL;
            DO_ADD_MENU_ITEM_LABEL
        }
        GUIMenuAddToMenuBar(
            menu_bar, menu,
#ifdef PROG_LANGUAGE_ENGLISH
            "Edit",
#endif
#ifdef PROG_LANGUAGE_SPANISH
	    "Redacta",
#endif
#ifdef PROG_LANGUAGE_FRENCH
	    "diter",
#endif
            GUI_MENU_BAR_ALIGN_LEFT
        );


        /* Create view menu. */
        menu = GUIMenuCreateTearOff();
        if(menu != NULL)
        {
            op = EDV_ARCHIVER_OP_REFRESH;
            w_record = &archiver->refresh_mi;
            fw_record = NULL;
            DO_ADD_MENU_ITEM_LABEL

            op = EDV_ARCHIVER_OP_REFRESH_ALL;
            w_record = &archiver->refresh_all_mi;
            fw_record = NULL;
            DO_ADD_MENU_ITEM_LABEL

            DO_ADD_MENU_SEP

            op = EDV_ARCHIVER_OP_SHOW_TOOL_BAR;
            w_record = &archiver->show_tool_bar_micheck;
            fw_record = NULL;
            DO_ADD_MENU_ITEM_CHECK

            op = EDV_ARCHIVER_OP_SHOW_LOCATION_BAR;
            w_record = &archiver->show_location_bar_micheck;
            fw_record = NULL;
            DO_ADD_MENU_ITEM_CHECK

	    op = EDV_ARCHIVER_OP_SHOW_FIND_BAR;
            w_record = &archiver->show_find_bar_micheck;
            fw_record = NULL;
            DO_ADD_MENU_ITEM_CHECK

            op = EDV_ARCHIVER_OP_SHOW_STATUS_BAR;
            w_record = &archiver->show_status_bar_micheck;
            fw_record = NULL;
            DO_ADD_MENU_ITEM_CHECK

            DO_ADD_MENU_SEP

            op = EDV_ARCHIVER_OP_MIME_TYPES;
            w_record = &archiver->mime_types_mi;
            fw_record = NULL;
            DO_ADD_MENU_ITEM_LABEL
        }
        GUIMenuAddToMenuBar(
            menu_bar, menu,
#ifdef PROG_LANGUAGE_ENGLISH
            "View",
#endif
#ifdef PROG_LANGUAGE_SPANISH
	    "Panorama",
#endif
#ifdef PROG_LANGUAGE_FRENCH
	    "Vue",
#endif
            GUI_MENU_BAR_ALIGN_LEFT
        );


        /* Create settings menu. */
        menu = GUIMenuCreateTearOff();
        if(menu != NULL)
        {
            op = EDV_ARCHIVER_OP_WRITE_PROTECT;
            w_record = &archiver->write_protect_mi;
            fw_record = NULL;
            DO_ADD_MENU_ITEM_LABEL

            DO_ADD_MENU_SEP

            op = EDV_ARCHIVER_OP_OPTIONS;
            w_record = &archiver->options_mi;
            fw_record = NULL;
            DO_ADD_MENU_ITEM_LABEL

            op = EDV_ARCHIVER_OP_CUSTOMIZE;
            w_record = &archiver->customize_mi;
            fw_record = NULL;
            DO_ADD_MENU_ITEM_LABEL
        }
        GUIMenuAddToMenuBar(
            menu_bar, menu,
#ifdef PROG_LANGUAGE_ENGLISH
            "Settings",
#endif
#ifdef PROG_LANGUAGE_SPANISH
	    "Colocaciones",
#endif
#ifdef PROG_LANGUAGE_FRENCH
	    "Montages",
#endif
            GUI_MENU_BAR_ALIGN_LEFT
        );


        /* Create windows menu. */
        menu = GUIMenuCreateTearOff();
        if(menu != NULL)
        {
            op = EDV_ARCHIVER_OP_NEW_BROWSER;
            w_record = &archiver->window_new_browser_mi;
            fw_record = NULL;
            DO_ADD_MENU_ITEM_LABEL

            op = EDV_ARCHIVER_OP_NEW_IMBR;
            w_record = &archiver->window_new_imbr_mi;
            fw_record = NULL;
            DO_ADD_MENU_ITEM_LABEL

            op = EDV_ARCHIVER_OP_NEW_ARCHIVER;
            w_record = &archiver->window_new_archiver_mi;
            fw_record = NULL;
            DO_ADD_MENU_ITEM_LABEL

            DO_ADD_MENU_SEP

            op = EDV_ARCHIVER_OP_RECYCLE_BIN;
            w_record = &archiver->window_recycle_bin_mi;
            fw_record = NULL;
            DO_ADD_MENU_ITEM_LABEL

        }
        GUIMenuAddToMenuBar(
            menu_bar, menu,
#ifdef PROG_LANGUAGE_ENGLISH
            "Windows",
#endif
#ifdef PROG_LANGUAGE_SPANISH
	    "Ventanas",
#endif
#ifdef PROG_LANGUAGE_FRENCH
	    "Fentres",
#endif
            GUI_MENU_BAR_ALIGN_LEFT
        );


        /* Create help menu. */
        menu = GUIMenuCreateTearOff();
        if(menu != NULL)
        {
            op = EDV_ARCHIVER_OP_HELP_CONTENTS;
            w_record = &archiver->help_contents_mi;
            fw_record = NULL;
            DO_ADD_MENU_ITEM_LABEL

            DO_ADD_MENU_SEP

            op = EDV_ARCHIVER_OP_HELP_ARCHIVER;
            w_record = &archiver->help_archiver_mi;
            fw_record = NULL;
            DO_ADD_MENU_ITEM_LABEL

            op = EDV_ARCHIVER_OP_HELP_KEYS_LIST;
            w_record = &archiver->help_keys_list_mi;
            fw_record = NULL;
            DO_ADD_MENU_ITEM_LABEL

            op = EDV_ARCHIVER_OP_HELP_COMMON_OPERATIONS;
            w_record = &archiver->help_common_operations_mi;
            fw_record = NULL;
            DO_ADD_MENU_ITEM_LABEL

            DO_ADD_MENU_SEP

            op = EDV_ARCHIVER_OP_HELP_ABOUT;
            w_record = &archiver->help_about_mi;
            fw_record = NULL;
            DO_ADD_MENU_ITEM_LABEL
        }
        GUIMenuAddToMenuBar(
            menu_bar, menu,
#ifdef PROG_LANGUAGE_ENGLISH
            "Help",
#endif
#ifdef PROG_LANGUAGE_SPANISH
	    "Ayuda",
#endif
#ifdef PROG_LANGUAGE_FRENCH
	    "L'Aide",
#endif
            GUI_MENU_BAR_ALIGN_RIGHT
        );



#undef DO_ADD_MENU_ITEM_LABEL
#undef DO_ADD_MENU_ITEM_SUBMENU
#undef DO_ADD_MENU_ITEM_CHECK
#undef DO_ADD_MENU_SEP

}

/*
 *      Allocate and create a new new edv_archiver_struct.
 */
edv_archiver_struct *EDVArchiverNew(gpointer core_ptr)
{
	gint	toplevel_x = 0,
		toplevel_y = 0,
		toplevel_width = EDV_DEF_ARCHIVER_WIDTH,
		toplevel_height = EDV_DEF_ARCHIVER_HEIGHT;
	gbool	tool_bar_map_state = TRUE,
		location_bar_map_state = TRUE,
		find_bar_map_state = TRUE,
		status_bar_map_state = TRUE;
	gint i, state;
	edv_status_bar_struct *status_bar;
	const gchar	*font_name = NULL,
			*fg_color_name = NULL,
			*bg_color_name = NULL,
                        *sfg_color_name = NULL,
                        *sbg_color_name = NULL,
                        *bg_pixmap_path = NULL,
                        *sbg_pixmap_path = NULL,
                        *wm_name = NULL,
                        *wm_class = NULL;
	gpointer combo_rtn;
	GList *glist;
        GdkColormap *colormap = NULL;
	GdkCursor *cursor;
	GtkRcStyle *rcstyle;
        GtkWidget *w, *parent, *parent2, *parent3, *parent4;
        GtkCList *clist;
	GtkCombo *combo;
        GdkWindow *window;
        GtkAccelGroup *accelgrp;
        const GtkTargetEntry dnd_tar_types[] = {
{"text/plain",				0,	EDV_DND_TYPE_INFO_TEXT_PLAIN},
{"text/uri-list",			0,	EDV_DND_TYPE_INFO_TEXT_URI_LIST},
{"STRING",				0,	EDV_DND_TYPE_INFO_STRING}
        };
        const GtkTargetEntry dnd_src_types[] = {
{EDV_DND_TYPE_STRING_ARCHIVE_OBJECT,	0,	EDV_DND_TYPE_INFO_ARCHIVE_OBJECT}
        };
        edv_archiver_struct *archiver = (edv_archiver_struct *)g_malloc0(
            sizeof(edv_archiver_struct)
        );
        if(archiver == NULL)
            return(archiver);


        /* Get values from core structure. */
        if(core_ptr != NULL)
        {
            edv_core_struct *c_ptr = EDV_CORE(core_ptr);

	    font_name = c_ptr->font_name;
	    fg_color_name = c_ptr->fg_color_name;
	    bg_color_name = c_ptr->bg_color_name;
            sfg_color_name = c_ptr->sfg_color_name;
            sbg_color_name = c_ptr->sbg_color_name;
            bg_pixmap_path = c_ptr->bg_pixmap_path;
            sbg_pixmap_path = c_ptr->sbg_pixmap_path;
            wm_name = c_ptr->wm_name;
            wm_class = c_ptr->wm_class;
	    if(c_ptr->geometry_set)
            {
                GdkRectangle *rect = &c_ptr->geometry;
                toplevel_x = rect->x;
                toplevel_y = rect->y;
                toplevel_width = rect->width;
                toplevel_height = rect->height;
            }
            else
            {
		toplevel_x = EDVCFGItemListGetValueI(
		    c_ptr->cfg_list, EDV_CFG_PARM_ARCHIVER_X
		);
                toplevel_y = EDVCFGItemListGetValueI(
                    c_ptr->cfg_list, EDV_CFG_PARM_ARCHIVER_Y
                );
                toplevel_width = EDVCFGItemListGetValueI(
                    c_ptr->cfg_list, EDV_CFG_PARM_ARCHIVER_WIDTH
                );
                toplevel_height = EDVCFGItemListGetValueI(
                    c_ptr->cfg_list, EDV_CFG_PARM_ARCHIVER_HEIGHT
                );
	    }

            tool_bar_map_state = EDVCFGItemListGetValueI(
                c_ptr->cfg_list, EDV_CFG_PARM_ARCHIVER_SHOW_TOOL_BAR
            );
            location_bar_map_state = EDVCFGItemListGetValueI(
                c_ptr->cfg_list, EDV_CFG_PARM_ARCHIVER_SHOW_LOCATION_BAR
            );
            find_bar_map_state = EDVCFGItemListGetValueI(
                c_ptr->cfg_list, EDV_CFG_PARM_ARCHIVER_SHOW_FIND_BAR
            );
            status_bar_map_state = EDVCFGItemListGetValueI(
                c_ptr->cfg_list, EDV_CFG_PARM_ARCHIVER_SHOW_STATUS_BAR
            );
        }

        /* Begin resetting values. */
        archiver->initialized = TRUE;
        archiver->map_state = FALSE;
        archiver->processing = FALSE;
	archiver->ignore_input = FALSE;
        archiver->busy_count = 0;
        archiver->core_ptr = core_ptr;
        archiver->opid = NULL;
        archiver->total_opids = 0;
        archiver->tool_bar_map_state = tool_bar_map_state;
        archiver->location_bar_map_state = location_bar_map_state;
	archiver->find_bar_map_state = find_bar_map_state;
        archiver->status_bar_map_state = status_bar_map_state;
	archiver->contents_clist_selected_row = -1;
	archiver->last_recbin_items = -1;
	archiver->last_write_protect_state = -1;
	archiver->findbar = NULL;
	archiver->status_bar = NULL;

        /* Create operation ids. */
        archiver->opid = EDVArchiverOPIDListNew(
            archiver, &archiver->total_opids
        );

        /* Keyboard accelerator group. */
        archiver->accelgrp = accelgrp = gtk_accel_group_new();

        /* Begin creating widgets. */
        /* Toplevel. */
        archiver->toplevel = w = gtk_window_new(GTK_WINDOW_TOPLEVEL);
        gtk_widget_set_uposition(
            w, toplevel_x, toplevel_y
        );
        gtk_widget_set_usize(
            w, toplevel_width, toplevel_height
        );
        gtk_window_set_wmclass(GTK_WINDOW(w), wm_name, wm_class);
        gtk_window_set_policy(
            GTK_WINDOW(w),
            TRUE, TRUE, TRUE
        );
        gtk_widget_realize(w);
        window = w->window;
        if(window != NULL)
        {
            GdkGeometry geometry;

            geometry.min_width = 100;
            geometry.min_height = 70;

            geometry.base_width = 0;
            geometry.base_height = 0;

            geometry.width_inc = 1;
            geometry.height_inc = 1;
/*
            geometry.min_aspect = 1.3;
            geometry.max_aspect = 1.3;
 */
            gdk_window_set_geometry_hints(
                window, &geometry,
                GDK_HINT_MIN_SIZE |
                GDK_HINT_BASE_SIZE |
                /* GDK_HINT_ASPECT | */
                GDK_HINT_RESIZE_INC
            );

            GUISetWMIcon(window, (u_int8_t **)archiver_48x48_xpm);

            colormap = gdk_window_get_colormap(window);
        }
        gtk_signal_connect(
            GTK_OBJECT(w), "map_event",
            GTK_SIGNAL_FUNC(EDVArchiverMapCB), archiver
        );
        gtk_signal_connect(
            GTK_OBJECT(w), "unmap_event",
            GTK_SIGNAL_FUNC(EDVArchiverUnmapCB), archiver
        );
        gtk_signal_connect(
            GTK_OBJECT(w), "delete_event",
            GTK_SIGNAL_FUNC(EDVArchiverDeleteEventCB), archiver
        );
        gtk_signal_connect(
            GTK_OBJECT(w), "destroy",
            GTK_SIGNAL_FUNC(EDVArchiverDestroyCB), archiver
        );
        gtk_accel_group_attach(accelgrp, GTK_OBJECT(w));
        gtk_container_set_border_width(GTK_CONTAINER(w), 0);
        parent = w;


        /* Main vbox. */
	archiver->main_vbox = w = gtk_vbox_new(FALSE, 0);
        gtk_container_add(GTK_CONTAINER(parent), w);
        gtk_widget_show(w);
        parent = w;


        /* Vbox for tool bars. */
        archiver->bars_vbox = w = gtk_vbox_new(FALSE, 0);
        gtk_box_pack_start(GTK_BOX(parent), w, FALSE, FALSE, 0);
        gtk_widget_show(w);
        parent2 = w;


        /* Handle for menu bar. */
        archiver->menu_bar_handle = w = gtk_handle_box_new();
        gtk_box_pack_start(GTK_BOX(parent2), w, FALSE, FALSE, 0);
        gtk_signal_connect_after(
            GTK_OBJECT(w), "child_attached",
	    GTK_SIGNAL_FUNC(EDVArchiverHandleChildAttachedCB), archiver
        );
        gtk_signal_connect_after(
            GTK_OBJECT(w), "child_detached",
	    GTK_SIGNAL_FUNC(EDVArchiverHandleChildDetachedCB), archiver
        );
        gtk_widget_show(w);
        /* Create menu bar. */
        EDVArchiverCreateMenuBar(archiver, w);


        /* Handle for tool bar. */
        archiver->tool_bar_handle = w = gtk_handle_box_new();
        gtk_box_pack_start(GTK_BOX(parent2), w, FALSE, FALSE, 0);
        gtk_signal_connect_after(
            GTK_OBJECT(w), "child_attached",
            GTK_SIGNAL_FUNC(EDVArchiverHandleChildAttachedCB), archiver
        );
        gtk_signal_connect_after(
            GTK_OBJECT(w), "child_detached",
            GTK_SIGNAL_FUNC(EDVArchiverHandleChildDetachedCB), archiver
        );
        if(archiver->tool_bar_map_state)
            gtk_widget_show(w);
        /* Build tool bar later. */
        archiver->toolbar = NULL;


        /* Location bar handle box. */
        archiver->location_bar_handle = w = gtk_handle_box_new();
        gtk_signal_connect_after(
            GTK_OBJECT(w), "child_attached",
            GTK_SIGNAL_FUNC(EDVArchiverHandleChildAttachedCB), archiver
        );
        gtk_signal_connect_after(
            GTK_OBJECT(w), "child_detached",
            GTK_SIGNAL_FUNC(EDVArchiverHandleChildDetachedCB), archiver
        );
        gtk_box_pack_start(GTK_BOX(parent), w, FALSE, FALSE, 0);
        if(archiver->location_bar_map_state)
            gtk_widget_show(w);
        parent2 = w;

        /* Location bar hbox for location bar. */
        archiver->location_bar_hbox = w = gtk_hbox_new(FALSE, 0);
        gtk_container_add(GTK_CONTAINER(parent2), w);
        gtk_widget_show(w);
        parent2 = w;


        /* Hbox for location icon and combo. */
        w = gtk_hbox_new(FALSE, 0);     /* Don't need spacing between icon
                                         * and combo because combo has it
                                         * already.
                                         */
        gtk_box_pack_start(GTK_BOX(parent2), w, TRUE, TRUE, 0);
        gtk_widget_show(w);
        parent3 = w;

        /* Vbox to orient location icon fixed parent. */
        w = gtk_vbox_new(TRUE, 0);
        gtk_widget_set_usize(w, 20, 20);
        gtk_box_pack_start(GTK_BOX(parent3), w, FALSE, FALSE, 0);
        gtk_widget_show(w);
        parent4 = w;
        /* Location icon fixed parent. */
        archiver->location_icon_fixed = w = gtk_fixed_new();
        gtk_widget_add_events(
            w,
	    GDK_BUTTON_PRESS_MASK | GDK_BUTTON_RELEASE_MASK |
	    GDK_POINTER_MOTION_MASK |
            GDK_ENTER_NOTIFY_MASK | GDK_LEAVE_NOTIFY_MASK
        );
        gtk_box_pack_start(GTK_BOX(parent4), w, TRUE, FALSE, 0);
        gtk_signal_connect(
            GTK_OBJECT(w), "enter_notify_event",
            GTK_SIGNAL_FUNC(EDVArchiverLocBarIconCrossingCB), archiver
        );
        gtk_signal_connect(
            GTK_OBJECT(w), "leave_notify_event",
            GTK_SIGNAL_FUNC(EDVArchiverLocBarIconCrossingCB), archiver
        );
	gtk_widget_realize(w);
	cursor = EDVGetCursor(
	    (edv_core_struct *)core_ptr, EDV_CURSOR_CODE_HAND
	);
	gdk_window_set_cursor(w->window, cursor);
	GUISetWidgetTip(
	    w,
#ifdef PROG_LANGUAGE_ENGLISH
"Drag this to create a link to this location"
#endif
#ifdef PROG_LANGUAGE_SPANISH
"Arrastre esto crear un eslabn a esta ubicacin"
#endif
#ifdef PROG_LANGUAGE_FRENCH
"Traner ceci pour crer un lien  cet emplacement"
#endif
	);
        gtk_widget_show(w);

        /* Reset location icon pixmap. */
        archiver->location_icon_pm = NULL;

        /* Location combo. */
        glist = NULL;
        w = (GtkWidget *)GUIComboCreate(
            "Location:",
            "/",
            glist,
            20,         /* Maximum items. */
            &combo_rtn,
            archiver,
            EDVArchiverComboActivateCB,
            NULL
        );
        gtk_box_pack_start(GTK_BOX(parent3), w, TRUE, TRUE, 0);
        gtk_widget_show(w);

	archiver->location_combo = w = (GtkWidget *)combo_rtn;
        combo = GTK_COMBO(w);
        gtk_combo_set_case_sensitive(combo, TRUE);
	if(1)
	{
	    /* Need to set different dnd_src_types for the drag source. */
	    const GtkTargetEntry dnd_src_types[] = {
{"text/plain",                          0,      EDV_DND_TYPE_INFO_TEXT_PLAIN},
{"text/uri-list",                       0,      EDV_DND_TYPE_INFO_TEXT_URI_LIST},
{"STRING",                              0,      EDV_DND_TYPE_INFO_STRING}
	    };
	    GUIDNDSetSrc(
		archiver->location_icon_fixed,
		dnd_src_types,
		sizeof(dnd_src_types) / sizeof(GtkTargetEntry),
		GDK_ACTION_COPY | GDK_ACTION_MOVE |
		GDK_ACTION_LINK,			/* Actions. */
		GDK_BUTTON1_MASK | GDK_BUTTON2_MASK,	/* Buttons. */
		NULL,
		EDVLocBarIconDragDataGet,
		NULL,
		NULL,
		combo
	    );
	}
        EDVEntrySetDND((edv_core_struct *)core_ptr, combo->entry);


        /* Find bar handle box. */
        archiver->find_bar_handle = w = gtk_handle_box_new();
        gtk_signal_connect_after(
            GTK_OBJECT(w), "child_attached",
            GTK_SIGNAL_FUNC(EDVArchiverHandleChildAttachedCB), archiver
        );
        gtk_signal_connect_after(
            GTK_OBJECT(w), "child_detached",
            GTK_SIGNAL_FUNC(EDVArchiverHandleChildDetachedCB), archiver
        );
        gtk_box_pack_start(GTK_BOX(parent), w, FALSE, FALSE, 0);
        if(archiver->find_bar_map_state)
            gtk_widget_show(w);
        parent2 = w;

        /* Create find bar. */
        archiver->findbar = EDVFindBarNew(
            core_ptr, parent2,
            EDV_FINDBAR_ROLE_ARCHIVE_OBJECT,
            EDVArchiverBarStatusMessageCB,
            EDVArchiverBarStatusProgressCB,
            EDVArchiverFindBarLocationCB,
            EDVArchiverFindBarStartCB,
            EDVArchiverFindBarEndCB,
            EDVArchiverFindBarMatchCB,
            archiver
        );
        EDVFindBarMap(archiver->findbar);


        /* Scrolled window for contents clist. */
        w = gtk_scrolled_window_new(NULL, NULL);
        gtk_scrolled_window_set_policy(
            GTK_SCROLLED_WINDOW(w),
            GTK_POLICY_AUTOMATIC,
            GTK_POLICY_AUTOMATIC
        );
        gtk_box_pack_start(GTK_BOX(parent), w, TRUE, TRUE, 0);
        gtk_widget_show(w);
        parent2 = w;

        /* Contents clist. */
        archiver->contents_clist = w = gtk_clist_new(
	    EDV_ARCHIVER_CONTENTS_CLIST_COLUMNS_MAX
	);
        clist = GTK_CLIST(w);
        gtk_widget_add_events(
            w,
            GDK_KEY_PRESS_MASK | GDK_KEY_RELEASE_MASK |
            GDK_BUTTON_PRESS_MASK
        );
        gtk_signal_connect(
            GTK_OBJECT(w), "key_press_event",
            GTK_SIGNAL_FUNC(EDVArchiverKeyEventCB), archiver
        );
        gtk_signal_connect(
            GTK_OBJECT(w), "key_release_event",
            GTK_SIGNAL_FUNC(EDVArchiverKeyEventCB), archiver
        );
        gtk_signal_connect_after(
            GTK_OBJECT(w), "button_press_event",
            GTK_SIGNAL_FUNC(EDVArchiverButtonPressEventCB), archiver
        );
        gtk_clist_set_selection_mode(clist, GTK_SELECTION_EXTENDED);
        for(i = 0; i < clist->columns; i++)
            gtk_clist_set_column_width(clist, i, 10);
        gtk_clist_set_row_height(clist, EDV_LIST_ROW_SPACING);
        gtk_clist_set_shadow_type(clist, GTK_SHADOW_IN);
        gtk_container_add(GTK_CONTAINER(parent2), w);
        gtk_signal_connect(
            GTK_OBJECT(w), "resize_column",
            GTK_SIGNAL_FUNC(EDVArchiverResizeColumnCB), archiver
        );
        gtk_signal_connect(
            GTK_OBJECT(w), "click_column",
            GTK_SIGNAL_FUNC(EDVArchiverClickColumnCB), archiver
        );
        gtk_signal_connect(
            GTK_OBJECT(w), "select_row",
            GTK_SIGNAL_FUNC(EDVArchiverSelectRowCB), archiver
        );
        gtk_signal_connect(
            GTK_OBJECT(w), "unselect_row",
            GTK_SIGNAL_FUNC(EDVArchiverUnselectRowCB), archiver
        );
        /* Set up DND for the contents list. */
        GUIDNDSetSrc(
            w,
            dnd_src_types,
            sizeof(dnd_src_types) / sizeof(GtkTargetEntry),
            GDK_ACTION_COPY,			/* Actions. */
            GDK_BUTTON1_MASK | GDK_BUTTON2_MASK,/* Buttons. */
            NULL,
            EDVArchiverContentsDragDataGetCB,
            EDVArchiverContentsDragDataDeleteCB,
            NULL,
            archiver
        );
        GUIDNDSetTar(
            w,
            dnd_tar_types,
            sizeof(dnd_tar_types) / sizeof(GtkTargetEntry),
	    GDK_ACTION_COPY,			/* Actions. */
	    GDK_ACTION_COPY,			/* Default action if same. */
	    GDK_ACTION_COPY,			/* Default action. */
            EDVArchiverContentsDragDataReceivedCB,
            archiver
        );
        gtk_signal_connect_after(
	    GTK_OBJECT(w), "drag_motion",
	    GTK_SIGNAL_FUNC(EDVArchiverContentsDragMotionCB), archiver
	);
        rcstyle = gtk_rc_style_new();
        rcstyle->font_name = (font_name != NULL) ? g_strdup(font_name) : NULL;
        state = GTK_STATE_NORMAL;
        rcstyle->color_flags[state] =
            ((fg_color_name != NULL) ? GTK_RC_FG | GTK_RC_TEXT : 0) |
            ((bg_color_name != NULL) ? GTK_RC_BG | GTK_RC_BASE : 0);
        if(fg_color_name != NULL)
        {
            gdk_color_parse(fg_color_name, &rcstyle->fg[state]);
            gdk_color_parse(fg_color_name, &rcstyle->text[state]);
        }
        if(bg_color_name != NULL)
        {
            gdk_color_parse(bg_color_name, &rcstyle->bg[state]);
            gdk_color_parse(bg_color_name, &rcstyle->base[state]);
        }
        if(bg_pixmap_path != NULL)
            rcstyle->bg_pixmap_name[state] = g_strdup(bg_pixmap_path);
        state = GTK_STATE_SELECTED;
        rcstyle->color_flags[state] =
            ((sfg_color_name != NULL) ? GTK_RC_FG | GTK_RC_TEXT : 0) |
            ((sbg_color_name != NULL) ? GTK_RC_BG | GTK_RC_BASE : 0);
        if(sfg_color_name != NULL)
        {
            gdk_color_parse(sfg_color_name, &rcstyle->fg[state]);
            gdk_color_parse(sfg_color_name, &rcstyle->text[state]);
        }
        if(sbg_color_name != NULL)
        {
            gdk_color_parse(sbg_color_name, &rcstyle->bg[state]);
            gdk_color_parse(sbg_color_name, &rcstyle->base[state]);
        }
        if(sbg_pixmap_path != NULL)
            rcstyle->bg_pixmap_name[state] = g_strdup(sbg_pixmap_path);
        gtk_widget_modify_style(w, rcstyle);
        GUIRCStyleDeallocUnref(rcstyle);
        gtk_widget_show(w);




        /* Create tool bar. */
        EDVArchiverToolbarRegenerate(archiver);

        /* Create contents clist menu. */
        EDVArchiverContentsMenuRegenerate(archiver);


        /* Create status bar. */
        archiver->status_bar = status_bar = EDVStatusBarNew(
            core_ptr, archiver->main_vbox
        );
        if(status_bar != NULL)
        {
            if(archiver->status_bar_map_state)
                EDVStatusBarMap(status_bar);
            else
                EDVStatusBarUnmap(status_bar);
        }



        /* Reset all values. */
        EDVArchiverReset(archiver, FALSE);

        /* Update menus. */
        EDVArchiverUpdateMenus(archiver);


        return(archiver);
}

/*
 *      Resets all values on the given recycle bin.
 */
void EDVArchiverReset(edv_archiver_struct *archiver, gbool need_unmap)
{
        if(archiver == NULL)
            return;

        if(!archiver->initialized)
            return;


        EDVArchiverSetTitle(archiver, NULL);


        if(need_unmap)
            EDVArchiverUnmap(archiver);
}

/*
 *      Syncronizes all data in the recycle bin's widgets to internal data
 *      on the recycle bin.
 */
void EDVArchiverSyncData(edv_archiver_struct *archiver)
{
        if(archiver == NULL)
            return;

        if(!archiver->initialized)
            return;

/* This function may be removed in the future, since there really is
 * no floating data from widgets to be synced to disk on the recycle bin.
 */


}

/*
 *      Updates all widgets on the given recycle bin to reflect the current
 *      data.
 */
void EDVArchiverUpdateMenus(edv_archiver_struct *archiver)
{
        gbool write_protect, write_protect_changed = FALSE;
        gbool sensitive, state;
        guint8 **icon_data, **recbin_icon_data;
        GtkWidget *w;
	GtkCList *clist;
	const gchar *location;
        gint selected_obj_row = -1;
        toolbar_struct *tb;
        edv_core_struct *core_ptr;


        if(archiver == NULL)
            return;

        if(!archiver->initialized)
            return;

        core_ptr = (edv_core_struct *)archiver->core_ptr;
        if(core_ptr == NULL)
            return;

        tb = archiver->toolbar;

#define DO_SET_SENSITIVE        \
{ \
 if(w != NULL) \
  gtk_widget_set_sensitive(w, sensitive); \
}
#define DO_SHOW \
{ \
 if(w != NULL) \
  gtk_widget_show(w); \
}
#define DO_HIDE \
{ \
 if(w != NULL) \
  gtk_widget_hide(w); \
}
#define DO_SET_CHECK_STATE      \
{ \
 if((w != NULL) ? GTK_IS_CHECK_MENU_ITEM(w) : FALSE) \
  GTK_CHECK_MENU_ITEM(w)->active = state; \
}

        /* Get global write protect state. */
        write_protect = EDVCFGItemListGetValueI(
            core_ptr->cfg_list, EDV_CFG_PARM_WRITE_PROTECT
        );
        if(archiver->last_write_protect_state < 0)
        {
            write_protect_changed = TRUE;
        }
        else
        {
            if((write_protect && !archiver->last_write_protect_state) ||
               (!write_protect && archiver->last_write_protect_state)
            )
                write_protect_changed = TRUE;
        }
        archiver->last_write_protect_state = write_protect ? 1 : 0;

        /* Get last selected row on the contents clist. */
	clist = (GtkCList *)archiver->contents_clist;
	selected_obj_row = EDVCListGetSelectedLast(clist, NULL);

        /* Update recycle bin icon data, setting it to not NULL if it
         * needs to be changed and update the recycle bin's last_recbin_items
         * to match the value on the core structure.
         */
        icon_data = NULL;
        if(archiver->last_recbin_items < 0)
        {
            if(core_ptr->last_recbin_items > 0)
                icon_data = (guint8 **)icon_trash_20x20_xpm;
            else
                icon_data = (guint8 **)icon_trash_empty_20x20_xpm;
        }
        else
        {
            if(!archiver->last_recbin_items &&
               core_ptr->last_recbin_items
            )
                icon_data = (guint8 **)icon_trash_20x20_xpm;
            else if(archiver->last_recbin_items &&
                    !core_ptr->last_recbin_items
            )
                icon_data = (guint8 **)icon_trash_empty_20x20_xpm;
        }
        archiver->last_recbin_items = core_ptr->last_recbin_items;
        recbin_icon_data = icon_data;

	/* Get current location. */
	location = EDVArchiverCurrentLocation(archiver);



        /* Begin updating menu items and tool bar items. */

        /* Edit. */
	/* Add. */
	sensitive = (location != NULL) ? (*location != '\0') : FALSE;
	w = archiver->edit_add_mi;
	DO_SET_SENSITIVE
        ToolBarItemSetSensitiveID(
            tb, EDV_ARCHIVER_OP_ADD, sensitive
        );
	/* Extract all. */
	w = archiver->edit_extract_all_mi;
        DO_SET_SENSITIVE
        ToolBarItemSetSensitiveID(
            tb, EDV_ARCHIVER_OP_EXTRACT_ALL, sensitive
        );

	/* Extract. */
        sensitive = (selected_obj_row < 0) ? FALSE : TRUE;
        w = archiver->edit_extract_mi;
        DO_SET_SENSITIVE
        ToolBarItemSetSensitiveID(
            tb, EDV_ARCHIVER_OP_EXTRACT, sensitive
        );
	/* Delete. */
        w = archiver->edit_delete_mi;
        DO_SET_SENSITIVE
        ToolBarItemSetSensitiveID(
            tb, EDV_ARCHIVER_OP_DELETE, sensitive
        );


        /* View. */
	/* Show tool bar. */
        w = archiver->show_tool_bar_micheck;
        state = EDVCFGItemListGetValueI(
            core_ptr->cfg_list, EDV_CFG_PARM_ARCHIVER_SHOW_TOOL_BAR
        );
        DO_SET_CHECK_STATE
        ToolBarItemSetToggleID(
            tb, EDV_ARCHIVER_OP_SHOW_TOOL_BAR, state
        );
	/* Show location bar. */
        w = archiver->show_location_bar_micheck;
        state = EDVCFGItemListGetValueI(
            core_ptr->cfg_list, EDV_CFG_PARM_ARCHIVER_SHOW_LOCATION_BAR
        );
        DO_SET_CHECK_STATE
        ToolBarItemSetToggleID(
            tb, EDV_ARCHIVER_OP_SHOW_LOCATION_BAR, state
        );
	/* Show find bar. */
        w = archiver->show_find_bar_micheck;
        state = EDVCFGItemListGetValueI(
            core_ptr->cfg_list, EDV_CFG_PARM_ARCHIVER_SHOW_FIND_BAR
        );
        DO_SET_CHECK_STATE
        ToolBarItemSetToggleID(
            tb, EDV_ARCHIVER_OP_SHOW_FIND_BAR, state
        );
	/* Show status bar. */
        w = archiver->show_status_bar_micheck;
        state = EDVCFGItemListGetValueI(
            core_ptr->cfg_list, EDV_CFG_PARM_ARCHIVER_SHOW_STATUS_BAR
        );
        DO_SET_CHECK_STATE
        ToolBarItemSetToggleID(
            tb, EDV_ARCHIVER_OP_SHOW_STATUS_BAR, state
        );


	/* Settings. */
        if(write_protect_changed)
	{
	    w = archiver->write_protect_mi;
	    if(w != NULL)
		GUIMenuItemSetPixmap(
		    w,
		    (u_int8_t **)(write_protect ?
			icon_secure_20x20_xpm : icon_insecure_20x20_xpm)
		);
            ToolBarItemUpdateByID(
                tb, EDV_ARCHIVER_OP_WRITE_PROTECT,
                NULL,
                (guint8 **)(write_protect ?
                    icon_secure_20x20_xpm : icon_insecure_20x20_xpm),
                NULL
            );
	}


        /* Windows. */
        if(recbin_icon_data != NULL)
	{
            GUIMenuItemSetPixmap(
                archiver->window_recycle_bin_mi, recbin_icon_data
            );
            ToolBarItemUpdateByID(
                tb, EDV_ARCHIVER_OP_RECYCLE_BIN,
                NULL, recbin_icon_data, NULL
            );
	}


        /* Update contents clist menu. */
        sensitive = (selected_obj_row < 0) ?
            FALSE : TRUE;
        w = archiver->contents_clist_extract_mi;
        DO_SET_SENSITIVE
        w = archiver->contents_clist_delete_mi;
        DO_SET_SENSITIVE






        /* Update status bar. */
        EDVStatusBarUpdateMenus(archiver->status_bar);



#undef DO_SHOW
#undef DO_HIDE
#undef DO_SET_SENSITIVE
#undef DO_SET_CHECK_STATE
}

/*
 *      Marks the recycle bin as busy or ready.
 */
void EDVArchiverSetBusy(edv_archiver_struct *archiver, gbool is_busy)
{
        GdkCursor *cursor;
        GtkWidget *w;
        edv_core_struct *core_ptr;


        if(archiver == NULL)
            return;

        if(!archiver->initialized)
            return;

        core_ptr = (edv_core_struct *)archiver->core_ptr;
        if(core_ptr == NULL)
            return;

        w = archiver->toplevel;
        if(w != NULL)
        {
            if(is_busy)
            {
                /* Increase busy count. */
                archiver->busy_count++;

                /* If already busy then don't change anything. */
                if(archiver->busy_count > 1)
                    return;

                cursor = EDVGetCursor(core_ptr, EDV_CURSOR_CODE_BUSY);
            }
            else
            {
                /* Reduce busy count. */
                archiver->busy_count--;
                if(archiver->busy_count < 0)
                    archiver->busy_count = 0;

                /* If still busy do not change anything. */
                if(archiver->busy_count > 0)
                    return;

                cursor = NULL;  /* Use default cursor. */
            }

            /* Update toplevel window's cursor. */
            if(w->window != NULL)
            {
                gdk_window_set_cursor(w->window, cursor);
                gdk_flush();
            }
        }
}

/*
 *      Maps the given recycle bin.
 */
void EDVArchiverMap(edv_archiver_struct *archiver)
{
        GtkWidget *w;


        if(archiver == NULL)
            return;

        if(!archiver->initialized)
            return;

	w = archiver->toplevel;
	gtk_widget_show_raise(w);
	archiver->map_state = TRUE;
}

/*
 *      Unmaps the given recycle bin.
 */
void EDVArchiverUnmap(edv_archiver_struct *archiver)
{
        GtkWidget *w;


        if(archiver == NULL)
            return;

        if(!archiver->initialized)
            return;

        if(archiver->map_state)
        {
            w = archiver->toplevel;
            if(w != NULL)
                gtk_widget_hide(w);

            archiver->map_state = FALSE;
        }
}

/*
 *      Destroys all resources on the given recycle bin and deallocates the
 *      structure itself.
 */
void EDVArchiverDelete(edv_archiver_struct *archiver)
{
        GtkWidget **w;


        if(archiver == NULL)
            return;

        if(archiver->initialized)
        {
#define DO_DESTROY_WIDGET       \
{ \
 if((*w) != NULL) \
 { \
  GtkWidget *tmp_w = *w; \
  (*w) = NULL; \
  gtk_widget_destroy(tmp_w); \
 } \
}

            /* Reset last selected item references. */
            archiver->contents_clist_selected_row = -1;

	    /* Destroy right-click menus. */
            w = &archiver->contents_clist_menu;
            archiver->contents_clist_add_mi = NULL;
            archiver->contents_clist_extract_mi = NULL;
            archiver->contents_clist_extract_all_mi = NULL;
            archiver->contents_clist_delete_mi = NULL;
            DO_DESTROY_WIDGET


            /* Delete status bar. */
            EDVStatusBarDelete(archiver->status_bar);
            archiver->status_bar = NULL;

            /* Delete find bar. */
            EDVFindBarDelete(archiver->findbar);
            archiver->findbar = NULL;

            /* Delete tool bar. */
            ToolBarDelete(archiver->toolbar);
            archiver->toolbar = NULL;


            /* Destroy menu bar and submenus. */
            w = &archiver->menu_bar;
            archiver->new_mi = NULL;
            archiver->open_mi = NULL;
            archiver->sync_disks_mi = NULL;
	    archiver->history_mi = NULL;
            archiver->close_mi = NULL;

            archiver->edit_add_mi = NULL;
            archiver->edit_extract_mi = NULL;
            archiver->edit_extract_all_mi = NULL;
            archiver->edit_delete_mi = NULL;
            archiver->select_all_mi = NULL;
            archiver->unselect_all_mi = NULL;
            archiver->invert_selection_mi = NULL;
            archiver->find_mi = NULL;

            archiver->refresh_mi = NULL;
            archiver->refresh_all_mi = NULL;
            archiver->show_tool_bar_micheck = NULL;
	    archiver->show_location_bar_micheck = NULL;
            archiver->show_find_bar_micheck = NULL;
            archiver->show_status_bar_micheck = NULL;
	    archiver->mime_types_mi = NULL;

            archiver->write_protect_mi = NULL;
            archiver->options_mi = NULL;
            archiver->customize_mi = NULL;

            archiver->window_new_browser_mi = NULL;
	    archiver->window_new_imbr_mi = NULL;
            archiver->window_new_archiver_mi = NULL;
            archiver->window_recycle_bin_mi = NULL;

            archiver->help_contents_mi = NULL;
	    archiver->help_archiver_mi = NULL;
	    archiver->help_keys_list_mi = NULL;
            archiver->help_common_operations_mi = NULL;
            archiver->help_about_mi = NULL;

            DO_DESTROY_WIDGET


            /* Destroy toplevel and the rest of the widgets. */
            w = &archiver->toplevel;
            archiver->main_vbox = NULL;
            archiver->menu_bar = NULL;
            archiver->bars_vbox = NULL;
            archiver->tool_bar_handle = NULL;
            archiver->menu_bar_handle = NULL;
            archiver->contents_clist = NULL;
            DO_DESTROY_WIDGET

            if(archiver->accelgrp != NULL)
            {
                gtk_accel_group_unref(archiver->accelgrp);
                archiver->accelgrp = NULL;
            }

#undef DO_DESTROY_WIDGET
        }


        /* Deallocate other memory on recycle bin structure. */

        /* Operation id structures. */
        EDVArchiverOPIDListDelete(
            archiver->opid, archiver->total_opids
        );



        /* Deallocate structure itself. */
        g_free(archiver);
}
