#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <sys/stat.h>
#include <gtk/gtk.h>

#include "../include/string.h"
#include "../include/fio.h"
#include "../include/disk.h"

#include "guiutils.h"
#include "cdialog.h"
#include "fb.h"
#include "progressdialog.h"

#include "edvtypes.h"
#include "edvcfg.h"
#include "edvconfirm.h"
#include "edvobj.h"
#include "edvarch.h"
#include "edvarchfio.h"
#include "edvarchop.h"
#include "archiveopts.h"
#include "archiver.h"
#include "archivercb.h"
#include "archiveropcb.h"
#include "archivercontents.h"
#include "endeavour.h"
#include "edvcb.h"
#include "edvop.h"
#include "edvutils.h"
#include "edvutilsgtk.h"
#include "edvcfglist.h"
#include "config.h"


void EDVArchiverMenuItemCB(GtkWidget *widget, gpointer data);
gint EDVArchiverMenuItemEnterCB(
        GtkWidget *widget, GdkEventCrossing *crossing, gpointer data
);
gint EDVArchiverMenuItemLeaveCB(
        GtkWidget *widget, GdkEventCrossing *crossing, gpointer data
);
void EDVArchiverOPCB(gpointer data);
void EDVArchiverOPEnterCB(gpointer data);
void EDVArchiverOPLeaveCB(gpointer data);

void EDVArchiverFindBarStartCB(gpointer fb, gpointer data);
void EDVArchiverFindBarEndCB(
        gpointer fb, gint total_matches, gpointer data
);
gchar *EDVArchiverFindBarLocationCB(gpointer fb, gpointer data);
void EDVArchiverFindBarMatchCB(
        const gchar *path, const struct stat *lstat_buf, const gchar *excerpt,
        gpointer data
);

void EDVArchiverBarStatusMessageCB(
        gpointer bar, const gchar *message, gpointer data
);
void EDVArchiverBarStatusProgressCB(
        gpointer bar, gfloat progress, gpointer data
);

void EDVArchiverOPSyncDisks(edv_archiver_struct *archiver);
void EDVArchiverOPWriteProtect(edv_archiver_struct *archiver);

void EDVArchiverOPNew(edv_archiver_struct *archiver);
void EDVArchiverOPOpen(edv_archiver_struct *archiver);

void EDVArchiverOPClose(edv_archiver_struct *archiver);
void EDVArchiverOPExit(edv_archiver_struct *archiver);

static edv_archive_object_struct **EDVArchiverGetSelectedObjects(
        edv_archiver_struct *archiver, gint *total
);
void EDVArchiverOPAdd(edv_archiver_struct *archiver);
void EDVArchiverOPExtract(edv_archiver_struct *archiver);
void EDVArchiverOPExtractAll(edv_archiver_struct *archiver);
void EDVArchiverOPDelete(edv_archiver_struct *archiver);
void EDVArchiverOPSelectAll(edv_archiver_struct *archiver);
void EDVArchiverOPUnselectAll(edv_archiver_struct *archiver);
void EDVArchiverOPInvertSelection(edv_archiver_struct *archiver);

void EDVArchiverOPRefresh(edv_archiver_struct *archiver);
void EDVArchiverOPRefreshAll(edv_archiver_struct *archiver);




/*
 *      Menu item activate callback.
 *
 *      The given client data must be a edv_archiver_opid_struct *.
 */
void EDVArchiverMenuItemCB(GtkWidget *widget, gpointer data)
{
        EDVArchiverOPCB(data);
}

/*
 *      Menu item "enter_notify_event" signal callback.
 *
 *      The given client data must be a edv_archiver_opid_struct *.
 */
gint EDVArchiverMenuItemEnterCB(
        GtkWidget *widget, GdkEventCrossing *crossing, gpointer data
)
{
        EDVArchiverOPEnterCB(data);
        return(TRUE);
}

/*
 *      Menu item "leave_notify_event" signal callback.
 *
 *      The given client data must be a edv_archiver_opid_struct *.
 */
gint EDVArchiverMenuItemLeaveCB(
        GtkWidget *widget, GdkEventCrossing *crossing, gpointer data
)
{
        EDVArchiverOPLeaveCB(data);
        return(TRUE);
}

/*
 *      Operation id callback nexus.
 *
 *      The given client data must be a edv_archiver_opid_struct *.
 */
void EDVArchiverOPCB(gpointer data)
{
        static gbool reenterent = FALSE;
        edv_archiver_struct *archiver;
        edv_core_struct *core_ptr;
        edv_archiver_opid_struct *opid = (edv_archiver_opid_struct *)data;
        if(opid == NULL)
            return;

        archiver = (edv_archiver_struct *)opid->archiver;
        if(archiver == NULL)
            return;

        if(archiver->processing)
            return;

        core_ptr = (edv_core_struct *)archiver->core_ptr;
        if(core_ptr == NULL)
            return;

        if(reenterent)
            return;
        else
            reenterent = TRUE;

        /* Handle by operation id code. */
        switch(opid->op)
        {
          case EDV_ARCHIVER_OP_NEW:
            EDVArchiverOPNew(archiver);
            break;

          case EDV_ARCHIVER_OP_OPEN:
            EDVArchiverOPOpen(archiver);
            break;


	  case EDV_ARCHIVER_OP_CLOSE:
	    EDVArchiverOPClose(archiver);
	    break;

	  case EDV_ARCHIVER_OP_EXIT:
	    EDVArchiverOPExit(archiver);
            break;


          case EDV_ARCHIVER_OP_ADD:
	    EDVArchiverOPAdd(archiver);
	    break;

          case EDV_ARCHIVER_OP_EXTRACT:
	    EDVArchiverOPExtract(archiver);
	    break;

	  case EDV_ARCHIVER_OP_EXTRACT_ALL:
	    EDVArchiverOPExtractAll(archiver);
	    break;

          case EDV_ARCHIVER_OP_DELETE:
            EDVArchiverOPDelete(archiver);
            break;

	  case EDV_ARCHIVER_OP_SELECT_ALL:
	    EDVArchiverOPSelectAll(archiver);
	    break;

          case EDV_ARCHIVER_OP_UNSELECT_ALL:
            EDVArchiverOPUnselectAll(archiver);
            break;

          case EDV_ARCHIVER_OP_INVERT_SELECTION:
            EDVArchiverOPInvertSelection(archiver);
            break;

          case EDV_ARCHIVER_OP_FIND:
            EDVDoMapArchiverFindWin(core_ptr, archiver);
            break;

          case EDV_ARCHIVER_OP_HISTORY:
            EDVDoMapHistoryListWin(core_ptr, archiver->toplevel);
            break;

	  case EDV_ARCHIVER_OP_SYNC_DISKS:
	    EDVArchiverOPSyncDisks(archiver);
	    break;

          case EDV_ARCHIVER_OP_WRITE_PROTECT:
            EDVArchiverOPWriteProtect(archiver);
            break;

          case EDV_ARCHIVER_OP_RUN:
            EDVDoMapRunDlg(core_ptr);
            break;

          case EDV_ARCHIVER_OP_RUN_TERMINAL:
            EDVDoRunTerminal(core_ptr, NULL);
            break;


          case EDV_ARCHIVER_OP_REFRESH:
	    EDVArchiverOPRefresh(archiver);
	    break;

          case EDV_ARCHIVER_OP_REFRESH_ALL:
            EDVArchiverOPRefreshAll(archiver);
            break;

          case EDV_ARCHIVER_OP_SHOW_TOOL_BAR:
            if(core_ptr != NULL)
            {
                gbool state = !EDVCFGItemListGetValueI(
                    core_ptr->cfg_list, EDV_CFG_PARM_ARCHIVER_SHOW_TOOL_BAR
                );
                EDVCFGItemListSetValueI(
                    core_ptr->cfg_list, EDV_CFG_PARM_ARCHIVER_SHOW_TOOL_BAR,
                    state, FALSE
                );
                EDVReconfiguredEmit(core_ptr);
            }
            break;

          case EDV_ARCHIVER_OP_SHOW_LOCATION_BAR:
            if(core_ptr != NULL)
            {
                gbool state = !EDVCFGItemListGetValueI(
                    core_ptr->cfg_list, EDV_CFG_PARM_ARCHIVER_SHOW_LOCATION_BAR
                );
                EDVCFGItemListSetValueI(
                    core_ptr->cfg_list, EDV_CFG_PARM_ARCHIVER_SHOW_LOCATION_BAR,
                    state, FALSE
                );
                EDVReconfiguredEmit(core_ptr);
            }
            break;

          case EDV_ARCHIVER_OP_SHOW_FIND_BAR:
            if(core_ptr != NULL)
            {
                gbool state = !EDVCFGItemListGetValueI(
                    core_ptr->cfg_list, EDV_CFG_PARM_ARCHIVER_SHOW_FIND_BAR
                );
                EDVCFGItemListSetValueI(
                    core_ptr->cfg_list, EDV_CFG_PARM_ARCHIVER_SHOW_FIND_BAR,
                    state, FALSE
                );
                EDVReconfiguredEmit(core_ptr);
            }
            break;

          case EDV_ARCHIVER_OP_SHOW_STATUS_BAR:
            if(core_ptr != NULL)
            {
                gbool state = !EDVCFGItemListGetValueI(
                    core_ptr->cfg_list, EDV_CFG_PARM_ARCHIVER_SHOW_STATUS_BAR
                );
                EDVCFGItemListSetValueI(
                    core_ptr->cfg_list, EDV_CFG_PARM_ARCHIVER_SHOW_STATUS_BAR,
                    state, FALSE
                );
                EDVReconfiguredEmit(core_ptr);
            }
            break;


          case EDV_ARCHIVER_OP_MIME_TYPES:
            EDVDoMapMIMETypesListWin(core_ptr, archiver->toplevel);
            break;


          case EDV_ARCHIVER_OP_NEW_BROWSER:
            EDVDoNewBrowser(core_ptr);
            break;

          case EDV_ARCHIVER_OP_NEW_IMBR:
            EDVDoNewImbr(core_ptr);
            break;

          case EDV_ARCHIVER_OP_NEW_ARCHIVER:
            EDVDoNewArchiver(core_ptr);
            break;

          case EDV_ARCHIVER_OP_RECYCLE_BIN:
            EDVDoMapRecBin(core_ptr);
            break;


          case EDV_ARCHIVER_OP_OPTIONS:
            EDVDoMapOptionsWin(core_ptr, archiver->toplevel);
            break;

          case EDV_ARCHIVER_OP_CUSTOMIZE:
            EDVDoMapCustomizeWin(core_ptr, archiver->toplevel);
            break;


          case EDV_ARCHIVER_OP_HELP_ABOUT:
            EDVDoHelpAbout(core_ptr, archiver->toplevel);
            break;
          case EDV_ARCHIVER_OP_HELP_CONTENTS:
            EDVDoHelp(core_ptr, "Contents", archiver->toplevel);
            break;
          case EDV_ARCHIVER_OP_HELP_FILE_BROWSER:
            EDVDoHelp(core_ptr, "File Browser", archiver->toplevel);
            break;
          case EDV_ARCHIVER_OP_HELP_IMAGE_BROWSER:
            EDVDoHelp(core_ptr, "Image Browser", archiver->toplevel);
            break;
          case EDV_ARCHIVER_OP_HELP_ARCHIVER:
            EDVDoHelp(core_ptr, "Archiver", archiver->toplevel);
            break;
          case EDV_ARCHIVER_OP_HELP_RECYCLE_BIN:
            EDVDoHelp(core_ptr, "Recycle Bin", archiver->toplevel);
            break;
          case EDV_ARCHIVER_OP_HELP_KEYS_LIST:
            EDVDoHelp(core_ptr, "Keys List", archiver->toplevel);
            break;
          case EDV_ARCHIVER_OP_HELP_COMMON_OPERATIONS:
            EDVDoHelp(core_ptr, "Common Operations", archiver->toplevel);
            break;


          default:
            if(1)
            {
                gchar text[256];

                sprintf(
                    text,
                    "Unsupported archiver operation `%i'\n",
                    opid->op
                );

                CDialogSetTransientFor(archiver->toplevel);
                CDialogGetResponse(
                    "Operation Error",
                    text,
                    NULL,
                    CDIALOG_ICON_ERROR,
                    CDIALOG_BTNFLAG_OK,
                    CDIALOG_BTNFLAG_OK
                );
                CDialogSetTransientFor(NULL);
            }
            break;
	}

        reenterent = FALSE;
}

/*
 *      Operation id enter notify callback nexus.
 *
 *      The given client data must be a edv_archiver_opid_struct *.
 */
void EDVArchiverOPEnterCB(gpointer data)
{
        const gchar *tooltip;
        edv_archiver_struct *archiver;
        edv_archiver_opid_struct *opid = (edv_archiver_opid_struct *)data;
        if(opid == NULL)
            return;

        archiver = (edv_archiver_struct *)opid->archiver;
        if(archiver == NULL)
            return;

        tooltip = opid->tooltip;
        if(tooltip != NULL)
            EDVStatusBarMessage(archiver->status_bar, tooltip, FALSE);
}

/*
 *      Operation id leave notify callback nexus.
 */
void EDVArchiverOPLeaveCB(gpointer data)
{
        edv_archiver_struct *archiver;
        edv_archiver_opid_struct *opid = (edv_archiver_opid_struct *)data;
        if(opid == NULL)
            return;

        archiver = (edv_archiver_struct *)opid->archiver;
        if(archiver == NULL)
            return;

        EDVStatusBarMessage(archiver->status_bar, NULL, FALSE);
}


/*
 *      Find bar get current location callback.
 */
gchar *EDVArchiverFindBarLocationCB(gpointer fb, gpointer data)
{
        return(EDVArchiverCurrentLocation(
            (edv_archiver_struct *)data
        ));
}

/*
 *      Find bar start find callback.
 */
void EDVArchiverFindBarStartCB(gpointer fb, gpointer data)
{
        GtkCList *clist;
        edv_archiver_struct *archiver = (edv_archiver_struct *)data;
        if(archiver == NULL)
            return;

        clist = (GtkCList *)archiver->contents_clist;
        if(clist == NULL)
            return;

        EDVArchiverSetBusy(archiver, TRUE);
/*      GUIBlockInput(archiver->toplevel, TRUE); */

        gtk_clist_freeze(clist);
        gtk_clist_unselect_all(clist);
        gtk_clist_thaw(clist);

        EDVArchiverUpdateMenus(archiver);
}

/*
 *      Find bar end find callback.
 */
void EDVArchiverFindBarEndCB(
        gpointer fb, gint total_matches, gpointer data
)
{
        edv_archiver_struct *archiver = (edv_archiver_struct *)data;
        if(archiver == NULL)
            return;

/*      GUIBlockInput(archiver->toplevel, FALSE); */
        EDVArchiverSetBusy(archiver, FALSE);
}


/*
 *      Find bar match callback.
 */
void EDVArchiverFindBarMatchCB(
        const gchar *path, const struct stat *lstat_buf, const gchar *excerpt,
        gpointer data
)
{
        gint row;
        GtkCList *clist;
        edv_archiver_struct *archiver = (edv_archiver_struct *)data;
        if((path == NULL) || (archiver == NULL))
            return;

        clist = (GtkCList *)archiver->contents_clist;
        if(clist == NULL)
            return;

        row = EDVArchiverContentsFindRowByPath(archiver, path);
        if((row >= 0) && (row < clist->rows))
            gtk_clist_select_row(clist, row, 0);
}


/*
 *      Opaque status bar message update callback, the first argument
 *      bar is completely ignored.
 */
void EDVArchiverBarStatusMessageCB(
        gpointer bar, const gchar *message, gpointer data
)
{
        static gbool reenterent = FALSE;
        edv_archiver_struct *archiver = (edv_archiver_struct *)data;
        if(archiver == NULL)
            return;

        if(archiver->processing)
            return;

        if(reenterent)
            return;
        else
            reenterent = TRUE;

        EDVStatusBarMessage(archiver->status_bar, message, FALSE);

        reenterent = FALSE;
}

/*
 *      Opaque status bar progress update callback, the first argument
 *      bar is completely ignored.
 */
void EDVArchiverBarStatusProgressCB(
        gpointer bar, gfloat progress, gpointer data
)
{
        static gbool reenterent = FALSE;
        edv_archiver_struct *archiver = (edv_archiver_struct *)data;
        if(archiver == NULL)
            return;

        if(archiver->processing)
            return;

        if(reenterent)
            return;
        else
            reenterent = TRUE;

        EDVStatusBarProgress(archiver->status_bar, progress, FALSE);

        reenterent = FALSE;
}


/*
 *      Flush data transfer output to disks.
 */
void EDVArchiverOPSyncDisks(edv_archiver_struct *archiver)
{
        if(archiver == NULL)
            return;

        EDVStatusBarMessage(
            archiver->status_bar,
            "Flushing all write buffers to disk...",
            TRUE
        );

        EDVArchiverSetBusy(archiver, TRUE);

        EDVDoSyncDisks((edv_core_struct *)archiver->core_ptr);

        EDVArchiverSetBusy(archiver, FALSE);

        EDVStatusBarMessage(
            archiver->status_bar,
            "Disk sync done",
            FALSE
        );
        EDVStatusBarProgress(archiver->status_bar, 0.0, FALSE);
}

/*
 *      Write protect toggle.
 */
void EDVArchiverOPWriteProtect(edv_archiver_struct *archiver)
{
        gbool write_protect;
        edv_core_struct *core_ptr;


        if(archiver == NULL)
            return;

        core_ptr = (edv_core_struct *)archiver->core_ptr;
        if(core_ptr == NULL)
            return;

        /* Get new write protect state, changing from the previous one. */
        write_protect = EDVCFGItemListGetValueI(
            core_ptr->cfg_list, EDV_CFG_PARM_WRITE_PROTECT
        ) ? FALSE : TRUE;

        /* Update write protect in the configuration list. */
        EDVCFGItemListSetValueI(
            core_ptr->cfg_list, EDV_CFG_PARM_WRITE_PROTECT,
            write_protect, FALSE
        );

        /* Notify all resources about change in write protect. */
        EDVWriteProtectChangedEmit(core_ptr, write_protect);
}


/*
 *	New archive callback.
 */
void EDVArchiverOPNew(edv_archiver_struct *archiver)
{
        static gbool reenterent = FALSE;
        gbool status;
        gint i;
        GtkWidget *toplevel;
        fb_type_struct **ftype = NULL, *ftype_rtn = NULL;
        gint total_ftypes = 0;
        gchar **path_rtn = NULL;
        gint total_path_rtns = 0;
        edv_core_struct *core_ptr;
        const gchar *ext_list[] = EDV_ARCHIVER_ARCHIVE_FILE_EXTENSIONS;


        if(archiver == NULL)
            return;


        if(FileBrowserIsQuery())
            return;

        toplevel = archiver->toplevel;

        core_ptr = (edv_core_struct *)archiver->core_ptr;
        if(core_ptr == NULL)
            return;

        /* Check and warn if write protect is enabled. */
        if(EDVCheckWriteProtect(core_ptr, TRUE))
            return;

        if(reenterent)
            return;
        else
            reenterent = TRUE;


        /* Create file types list. */
        for(i = 0; ext_list[i] != NULL; i += 2)
            FileBrowserTypeListNew(
                &ftype, &total_ftypes,
                ext_list[i], ext_list[i + 1]
            );

        /* Query user for new icon file. */
        FileBrowserSetTransientFor(toplevel);
        status = FileBrowserGetResponse(
            "Create Archive",
            "Create", "Cancel",
            NULL,               /* Startup path. */
            ftype, total_ftypes,
            &path_rtn, &total_path_rtns,
            &ftype_rtn
        );
        FileBrowserSetTransientFor(NULL);

        /* Got user response? */
        if(status)
        {
            const gchar *new_path = (total_path_rtns > 0) ?
                path_rtn[0] : NULL;


            if((new_path != NULL) ? (*new_path != '\0') : FALSE)
            {
		FILE *fp;
		struct stat stat_buf;


		/* Make sure that the new archive to be created does
		 * not exist.
		 */
		if(stat(new_path, &stat_buf))
		{
		    /* Does not exist, so `touch' the location to create
		     * am empty archive.
		     */
		    fp = FOpen(new_path, "wb");
		    if(fp != NULL)
		    {
			/* Created new file, do not put anything in the
			 * file though. Just close it immediatly.
			 */
			fstat(fileno(fp), &stat_buf);
			FClose(fp);
			fp = NULL;

			EDVArchiverSetBusy(archiver, TRUE);
			GUIBlockInput(toplevel, TRUE);

			/* Clear the contents clist and load the listing
			 * of the new archive specified by new_path.
			 */
			EDVArchiverSelectArchive(archiver, new_path);

			/* Report new object added. */
			EDVObjectAddedEmit(
			    core_ptr, new_path, &stat_buf
			);

                        GUIBlockInput(toplevel, FALSE);
                        EDVArchiverSetBusy(archiver, FALSE);
		    }
		    else
		    {
			/* Unable to create new archive. */
			gchar *buf = g_strdup_printf(
"Unable to create new archive:\n\
\n\
    %s\n",
			    new_path
			);
			CDialogSetTransientFor(toplevel);
			CDialogGetResponse(
			    "Create Error",
			    buf,
			    NULL,
			    CDIALOG_ICON_ERROR,
			    CDIALOG_BTNFLAG_OK,
			    CDIALOG_BTNFLAG_OK
			);
			CDialogSetTransientFor(NULL);
			g_free(buf);
		    }
		}
		else
		{
		    /* An object already exists at the specified 
		     * location.
		     */
		    CDialogSetTransientFor(toplevel);
		    CDialogGetResponse(
			"Create Error",
			"An object already exists at the specified location",
			NULL,
			CDIALOG_ICON_WARNING,
			CDIALOG_BTNFLAG_OK,
			CDIALOG_BTNFLAG_OK
		    );
		    CDialogSetTransientFor(NULL);
		}

		EDVArchiverUpdateMenus(archiver);
            }
        }

        /* Deallocate file types list. */
        FileBrowserDeleteTypeList(ftype, total_ftypes);

        reenterent = FALSE;
}

/*
 *	Open archive callback.
 */
void EDVArchiverOPOpen(edv_archiver_struct *archiver)
{
        static gbool reenterent = FALSE;
	gbool status;
	gint i;
        GtkWidget *toplevel;
        fb_type_struct **ftype = NULL, *ftype_rtn = NULL;
        gint total_ftypes = 0;
        gchar **path_rtn = NULL;
        gint total_path_rtns = 0;
        edv_core_struct *core_ptr;
	const gchar *ext_list[] = EDV_ARCHIVER_ARCHIVE_FILE_EXTENSIONS;


        if(archiver == NULL)
            return;

        if(FileBrowserIsQuery())
            return;

	toplevel = archiver->toplevel;

        core_ptr = (edv_core_struct *)archiver->core_ptr;
        if(core_ptr == NULL)
            return;

	if(reenterent)
	    return;
	else
	    reenterent = TRUE;


        /* Create file types list. */
        for(i = 0; ext_list[i] != NULL; i += 2)
	    FileBrowserTypeListNew(
		&ftype, &total_ftypes,
		ext_list[i], ext_list[i + 1]
	    );
        FileBrowserTypeListNew(
            &ftype, &total_ftypes,
            "*.*", "All files"
        );


        /* Query user for new icon file. */
        FileBrowserSetTransientFor(toplevel);
        status = FileBrowserGetResponse(
            "Open Archive",
            "Open", "Cancel",
            NULL,		/* Startup path. */
            ftype, total_ftypes,
            &path_rtn, &total_path_rtns,
            &ftype_rtn
        );
        FileBrowserSetTransientFor(NULL);

        /* Got user response? */
        if(status)
        {
            const gchar *new_path = (total_path_rtns > 0) ?
                path_rtn[0] : NULL;


            if((new_path != NULL) ? (*new_path != '\0') : FALSE)
            {
                EDVArchiverSetBusy(archiver, TRUE);
                GUIBlockInput(toplevel, TRUE);

                /* Clear the contents clist and load the listing of the
                 * new archive specified by new_path.
                 */
		EDVArchiverSelectArchive(archiver, new_path);

                GUIBlockInput(toplevel, FALSE);
                EDVArchiverSetBusy(archiver, FALSE);

                EDVArchiverUpdateMenus(archiver);
	    }
	}

        /* Deallocate file types list. */
        FileBrowserDeleteTypeList(ftype, total_ftypes);

	reenterent = FALSE;
}


/*
 *      Close recycle bin.
 */
void EDVArchiverOPClose(edv_archiver_struct *archiver)
{
        if(archiver == NULL)
            return;

        /* Set current properties of the recycle bin to the configuration
         * list.
         */
        EDVArchiverSyncConfiguration(archiver);

        /* Unmap the archiver. */
        EDVArchiverUnmap(archiver);
}

/*
 *      Close all windows.
 */
void EDVArchiverOPExit(edv_archiver_struct *archiver)
{
        edv_core_struct *core_ptr;


        if(archiver == NULL)
            return;

        core_ptr = (edv_core_struct *)archiver->core_ptr;
        if(core_ptr == NULL)
            return;

        /* Set current properties of the recycle bin to the configuration
         * list.
         */
        EDVArchiverSyncConfiguration(archiver);

        /* Unmap this recycle bin. */
        EDVArchiverUnmap(archiver);

        /* Mark need_close_all_windows on the core structure to TRUE, this
         * will be checked during the management timeout. In which case
         * all windows will be deleted.
         */
        core_ptr->need_close_all_windows = TRUE;
}


/*
 *	Returns a list of dynamically allocated archive object stat
 *	structures from the selected rows on the archiver's contents
 *	clist.
 *
 *	The calling function must deallocate the returned list and
 *	each structure.
 */
static edv_archive_object_struct **EDVArchiverGetSelectedObjects(
	edv_archiver_struct *archiver, gint *total
)
{
	gint i;
	GList *glist;
	GtkCList *clist;
	edv_archive_object_struct **list = NULL, *obj;


	if(total != NULL)
	    *total = 0;

	if((archiver == NULL) || (total == NULL))
	    return(list);

        clist = (GtkCList *)archiver->contents_clist;
        if(clist == NULL)
            return(list);

        /* Generate a list of archive object stat structures from the
         * selected rows on the contents clist. Each structure must
         * be coppied.
         */
        glist = clist->selection;
        while(glist != NULL)
        {
            obj = (edv_archive_object_struct *)gtk_clist_get_row_data(
                clist, (gint)glist->data
            );
            if(obj != NULL)
            {
                i = *total;
                *total = i + 1;
		list = (edv_archive_object_struct **)g_realloc(
		    list,
		    (*total) * sizeof(edv_archive_object_struct *)
		);
		if(list == NULL)
		    *total = 0;
		else
		    list[i] = EDVArchObjectCopy(obj);
            }
            glist = glist->next;
        }

	return(list);
}

/*
 *	Add callback.
 */
void EDVArchiverOPAdd(edv_archiver_struct *archiver)
{
        gbool yes_to_all = FALSE;
        gbool recurse = TRUE;
	gint compression = 50;		/* 0 to 100. */
	gbool dereference_links = FALSE;

        GtkWidget *toplevel;
        edv_core_struct *core_ptr;
        gchar *arch_obj = NULL, *add_dir = NULL;
        const gchar *cstrptr, *error_mesg;
        gint i, status;

        fb_type_struct **ftype = NULL, *ftype_rtn = NULL;
        gint total_ftypes = 0;
        gchar **path_rtn = NULL;
        gint total_path_rtns = 0;

	gchar **tar_path = NULL;
	gint total_tar_paths = 0;
        gchar **new_aobj = NULL;
	gint total_new_aobjs = 0;


        if(archiver == NULL)
            return;

        core_ptr = (edv_core_struct *)archiver->core_ptr;
        if(core_ptr == NULL)
            return;

        toplevel = archiver->toplevel;

        /* Check and warn if write protect is enabled. */
        if(EDVCheckWriteProtect(core_ptr, TRUE))
            return;

#define DO_FREE_LOCALS  \
{ \
 for(i = 0; i < total_new_aobjs; i++) \
  g_free(new_aobj[i]); \
 g_free(new_aobj); \
 new_aobj = NULL; \
 total_new_aobjs = 0; \
 \
 for(i = 0; i < total_tar_paths; i++) \
  g_free(tar_path[i]); \
 g_free(tar_path); \
 tar_path = NULL; \
 total_tar_paths = 0; \
 \
 g_free(arch_obj); \
 arch_obj = NULL; \
 g_free(add_dir); \
 add_dir = NULL; \
 \
 /* Deallocate file types list. */ \
 FileBrowserDeleteTypeList(ftype, total_ftypes); \
 ftype = NULL; \
 total_ftypes = 0; \
}

        /* Get current archive path. */
        cstrptr = EDVArchiverCurrentLocation(archiver);
        arch_obj = (cstrptr != NULL) ? g_strdup(cstrptr) : NULL;
        if(arch_obj == NULL)
        {
            DO_FREE_LOCALS
            return;
        }


        /* Generate suggested add location based on parent directory
         * of current archive.
         */
        cstrptr = GetParentDir(arch_obj);
	add_dir = (cstrptr != NULL) ? g_strdup(cstrptr) : NULL;


        EDVArchiverSetBusy(archiver, TRUE);


        /* Begin querying user for add path. */

        /* Generate file types list. */
        FileBrowserTypeListNew(
            &ftype, &total_ftypes,
            "*.*", "All files"
        );

        /* Query user for extract destination. */
        FileBrowserSetTransientFor(toplevel);
        status = (gint)FileBrowserGetResponse(
            "Add Objects To Archive",
            "Add", "Cancel",
            add_dir,
            ftype, total_ftypes,
            &path_rtn, &total_path_rtns,
            &ftype_rtn
        );
        FileBrowserSetTransientFor(NULL);

        /* User canceled? */
        if(!status)
        {
            EDVArchiverSetBusy(archiver, FALSE);
            DO_FREE_LOCALS
            return;
        }

        /* Get list of target objects to add. */
        if(total_path_rtns > 0)
        {
	    total_tar_paths = total_path_rtns;
	    tar_path = (gchar **)g_realloc(
		tar_path, total_tar_paths * sizeof(gchar *)
	    );
	    for(i = 0; i < total_tar_paths; i++)
	    {
		cstrptr = path_rtn[i];
		if((cstrptr != NULL) ? (*cstrptr == '\0') : TRUE)
		    continue;

		tar_path[i] = g_strdup(cstrptr);

                /* Path from file browser may have tailing deliminators,
                 * get rid of them.
                 */
                StripPath(tar_path[i]);
            }
        }

        /* Get additional add to archive options. */
        if(!EDVArchiveAddQueryOptions(
            core_ptr, toplevel, arch_obj,
	    &recurse, &compression, &dereference_links
        ))
        {
            /* User canceled. */
            EDVArchiverSetBusy(archiver, FALSE);
            DO_FREE_LOCALS
            return;
        }


	if(TRUE)
	{
	    /* Add selected object to the archive. */
	    status = EDVArchOPAdd(
		core_ptr, arch_obj, tar_path, total_tar_paths,
		&new_aobj, &total_new_aobjs,
		toplevel, TRUE, TRUE, &yes_to_all,
		recurse, compression, dereference_links
	    );

            /* Get error message (if any) that might have occured in the
             * above operation.
             */
            error_mesg = EDVArchOPGetError();
            if(error_mesg != NULL)
            {
                CDialogSetTransientFor(toplevel);
                CDialogGetResponse(
                    "Operation Error",
                    error_mesg,
                    NULL,
                    CDIALOG_ICON_ERROR,
                    CDIALOG_BTNFLAG_OK,
                    CDIALOG_BTNFLAG_OK
                );
                CDialogSetTransientFor(NULL);
            }

	    /* Report that the archive has been modified, this will
	     * cause the archiver to reload listing.
	     */
	    if(!status)
	    {
		struct stat lstat_buf;

		if(!lstat(arch_obj, &lstat_buf))
		    EDVObjectModifiedEmit(
			core_ptr, arch_obj, arch_obj, &lstat_buf
		    );
	    }
	}

        /* Update status bar. */
        if(TRUE)
        {
            gchar *buf;

            switch(status)
            {
              case 0: case -5:
                buf = g_strdup_printf(
                    "Added %i object%s",
                    total_new_aobjs,
                    (total_new_aobjs == 1) ? "" : "s"
                );
                break;

              case -4:  /* Cancel. */
                buf = g_strdup_printf(
                    "Add operation canceled"
                );
                break;

              default:  /* Error. */
                buf = g_strdup("Unable to add object");
                break;
            }
            EDVStatusBarMessage(archiver->status_bar, buf, FALSE);
            g_free(buf);
        }

        EDVArchiverSetBusy(archiver, FALSE);


        /* Unmap progress dialog, it may have been mapped if any
         * operations occured in the above call.
         */
        ProgressDialogBreakQuery(TRUE);
        ProgressDialogSetTransientFor(NULL);

	DO_FREE_LOCALS
#undef DO_FREE_LOCALS
}

/*
 *      Extract callback.
 */
void EDVArchiverOPExtract(edv_archiver_struct *archiver)
{
        gbool yes_to_all = FALSE;
	gbool preserve_directories = TRUE;

        gint objects_extracted = 0;
        GtkWidget *toplevel;
        edv_archive_object_struct **list = NULL;
        edv_core_struct *core_ptr;
        gchar *arch_obj = NULL, *dest_path = NULL;
        const gchar *cstrptr;
        gint status, total = 0;

        fb_type_struct **ftype = NULL, *ftype_rtn = NULL;
        gint total_ftypes = 0;
        gchar **path_rtn = NULL;
        gint total_path_rtns = 0;


        if(archiver == NULL)
            return;

        core_ptr = (edv_core_struct *)archiver->core_ptr;
        if(core_ptr == NULL)
            return;

        toplevel = archiver->toplevel;

        /* Check and warn if write protect is enabled. */
        if(EDVCheckWriteProtect(core_ptr, TRUE))
            return;

#define DO_FREE_LOCALS  \
{ \
 gint i; \
 \
 for(i = 0; i < total; i++) \
  EDVArchObjectDelete(list[i]); \
 g_free(list); \
 list = NULL; \
 total = 0; \
 \
 g_free(arch_obj); \
 arch_obj = NULL; \
 g_free(dest_path); \
 dest_path = NULL; \
 \
 /* Deallocate file types list. */ \
 FileBrowserDeleteTypeList(ftype, total_ftypes); \
 ftype = NULL; \
 total_ftypes = 0; \
}

        /* Get current archive path. */
        cstrptr = EDVArchiverCurrentLocation(archiver);
        arch_obj = (cstrptr != NULL) ? g_strdup(cstrptr) : NULL;
        if(arch_obj == NULL)
        {
            DO_FREE_LOCALS
            return;
        }

        /* Get list of selected archive objects. */
        list = EDVArchiverGetSelectedObjects(archiver, &total);
        if((list == NULL) || (total <= 0))
        {
            DO_FREE_LOCALS
            return;
        }

	/* Generate suggested extract location based on parent directory
	 * of current archive.
	 */
	cstrptr = GetParentDir(arch_obj);
	dest_path = (cstrptr != NULL) ? g_strdup(cstrptr) : NULL;


        EDVArchiverSetBusy(archiver, TRUE);


	/* Begin querying user for extract destination. */

	/* Generate file types list. */
        FileBrowserTypeListNew(
            &ftype, &total_ftypes,
            "*.*", "All files"
        );

        /* Query user for extract destination. */
        FileBrowserSetTransientFor(toplevel);
        status = (gint)FileBrowserGetResponse(
            "Select Extract Destination",
            "Select", "Cancel",
            dest_path,
            ftype, total_ftypes,
            &path_rtn, &total_path_rtns,
            &ftype_rtn
        );
        FileBrowserSetTransientFor(NULL);

	/* User canceled? */
	if(!status)
	{
            EDVArchiverSetBusy(archiver, FALSE);
	    DO_FREE_LOCALS
	    return;
	}

	/* Get new destination. */
	g_free(dest_path);
	dest_path = NULL;
	if(total_path_rtns > 0)
	{
	    cstrptr = path_rtn[0];
	    if((cstrptr != NULL) ? (*cstrptr != '\0') : FALSE)
	    {
		g_free(dest_path);
		dest_path = g_strdup(cstrptr);

		/* Path from file browser may have tailing deliminators,
		 * get rid of them.
		 */
		StripPath(dest_path);
	    }
	}
	/* No destination path selected or destination path is not
	 * a directory?
	 */
        if(dest_path == NULL)
        {
            CDialogSetTransientFor(toplevel);
            CDialogGetResponse(
                "Invalid Destination",
                "The given destination directory is invalid",
                NULL,
                CDIALOG_ICON_WARNING,
                CDIALOG_BTNFLAG_OK,
                CDIALOG_BTNFLAG_OK
            );
            CDialogSetTransientFor(NULL);

            EDVArchiverSetBusy(archiver, FALSE);
            DO_FREE_LOCALS
            return;
        }

	/* Get additional extract from archive options. */
	if(!EDVArchiveExtractQueryOptions(
	    core_ptr, toplevel, arch_obj, &preserve_directories
	))
	{
	    /* User canceled. */
	    EDVArchiverSetBusy(archiver, FALSE);
            DO_FREE_LOCALS
            return;
	}


	if(1)
	{
	    const gchar *error_mesg;
	    gchar **new_path = NULL;
	    gint total_new_paths = 0;

	    /* Extract. */
	    status = EDVArchOPExtract(
		core_ptr, arch_obj, list, total,
		dest_path, &new_path, &total_new_paths,
		toplevel, TRUE, TRUE, &yes_to_all,
		preserve_directories
	    );

            /* Get error message (if any) that might have occured in the
             * above operation.
             */
            error_mesg = EDVArchOPGetError();
            if(error_mesg != NULL)
            {
                CDialogSetTransientFor(toplevel);
                CDialogGetResponse(
                    "Operation Error",
                    error_mesg,
                    NULL,
                    CDIALOG_ICON_ERROR,
                    CDIALOG_BTNFLAG_OK,
                    CDIALOG_BTNFLAG_OK
                );
                CDialogSetTransientFor(NULL);
            }

            /* Report new disk objects added? */
	    if((new_path != NULL) && !status)
	    {
		gint i;
		struct stat lstat_buf;

		for(i = 0; i < total_new_paths; i++)
		{
		    cstrptr = new_path[i];
		    if(cstrptr == NULL)
			continue;
		    if(lstat(cstrptr, &lstat_buf))
			continue;

		    objects_extracted++;

		    EDVObjectAddedEmit(
			core_ptr, cstrptr, &lstat_buf
		    );
		}
	    }

            /* Deallocate coppies of new disk object's paths. */
	    if(new_path != NULL)
	    {
                gint i;

                for(i = 0; i < total_new_paths; i++)
		    g_free(new_path[i]);
		g_free(new_path);
		new_path = NULL;
		total_new_paths = 0;
	    }
        }

        /* Update status bar. */
        if(1)
        {
            gchar *buf;

            switch(status)
            {
              case 0: case -5:
                buf = g_strdup_printf(
                    "Extracted %i object%s",
                    objects_extracted,
                    (objects_extracted == 1) ? "" : "s"
                );
                break;

              case -4:  /* Cancel. */
                buf = g_strdup_printf(
                    "Extract operation canceled"
                );
                break;

              default:  /* Error. */
                buf = g_strdup_printf(
                    "Unable to extract object%s",
                    (total == 1) ? "" : "s"
                );
                break;
            }
            EDVStatusBarMessage(archiver->status_bar, buf, FALSE);
            g_free(buf);
        }

        EDVArchiverSetBusy(archiver, FALSE);


        /* Unmap progress dialog, it may have been mapped if any
         * operations occured in the above call.
         */
        ProgressDialogBreakQuery(TRUE);
        ProgressDialogSetTransientFor(NULL);


        DO_FREE_LOCALS
#undef DO_FREE_LOCALS
}

/*
 *	Extract all callback.
 */
void EDVArchiverOPExtractAll(edv_archiver_struct *archiver)
{
        gbool yes_to_all = FALSE;
        gbool preserve_directories = TRUE;

        gint objects_extracted = 0;
        GtkWidget *toplevel;
	GtkCList *clist;
        edv_archive_object_struct **list = NULL;
        edv_core_struct *core_ptr;
        gchar *arch_obj = NULL, *dest_path = NULL;
        const gchar *cstrptr;
        gint status, total = 0;

        fb_type_struct **ftype = NULL, *ftype_rtn = NULL;
        gint total_ftypes = 0;
        gchar **path_rtn = NULL;
        gint total_path_rtns = 0;


        if(archiver == NULL)
            return;

        core_ptr = (edv_core_struct *)archiver->core_ptr;
        if(core_ptr == NULL)
            return;

	clist = (GtkCList *)archiver->contents_clist;
	if(clist == NULL)
	    return;

        toplevel = archiver->toplevel;

        /* Check and warn if write protect is enabled. */
        if(EDVCheckWriteProtect(core_ptr, TRUE))
            return;

#define DO_FREE_LOCALS  \
{ \
 gint i; \
 \
 for(i = 0; i < total; i++) \
  EDVArchObjectDelete(list[i]); \
 g_free(list); \
 list = NULL; \
 total = 0; \
 \
 g_free(arch_obj); \
 arch_obj = NULL; \
 g_free(dest_path); \
 dest_path = NULL; \
 \
 /* Deallocate file types list. */ \
 FileBrowserDeleteTypeList(ftype, total_ftypes); \
 ftype = NULL; \
 total_ftypes = 0; \
}

        /* Get current archive path. */
        cstrptr = EDVArchiverCurrentLocation(archiver);
        arch_obj = (cstrptr != NULL) ? g_strdup(cstrptr) : NULL;
        if(arch_obj == NULL)
        {
            DO_FREE_LOCALS
            return;
        }

        /* Get list of all archive objects. */
	total = clist->rows;
	if(total > 0)
	{
	    gint i;
	    edv_archive_object_struct *obj;

	    list = (edv_archive_object_struct **)g_malloc0(
		total * sizeof(edv_archive_object_struct *)
	    );
	    if(list == NULL)
		total = 0;
	    for(i = 0; i < total; i++)
	    {
		obj = (edv_archive_object_struct *)gtk_clist_get_row_data(
		    clist, i
		);
		if(obj != NULL)
		    list[i] = EDVArchObjectCopy(obj);
	    }
	}
        if((list == NULL) || (total <= 0))
        {
            DO_FREE_LOCALS
            return;
        }

        /* Generate suggested extract location based on parent directory
         * of current archive.
         */
        cstrptr = GetParentDir(arch_obj);
        dest_path = (cstrptr != NULL) ? g_strdup(cstrptr) : NULL;


        EDVArchiverSetBusy(archiver, TRUE);


        /* Begin querying user for extract destination. */

        /* Generate file types list. */
        FileBrowserTypeListNew(
            &ftype, &total_ftypes,
            "*.*", "All files"
        );

        /* Query user for extract destination. */
        FileBrowserSetTransientFor(toplevel);
        status = (gint)FileBrowserGetResponse(
            "Select Extract Destination",
            "Select", "Cancel",
            dest_path,
            ftype, total_ftypes,
            &path_rtn, &total_path_rtns,
            &ftype_rtn
        );
        FileBrowserSetTransientFor(NULL);

        /* User canceled? */
        if(!status)
        {
            EDVArchiverSetBusy(archiver, FALSE);
            DO_FREE_LOCALS
            return;
        }

        /* Get new destination. */
        g_free(dest_path);
        dest_path = NULL;
        if(total_path_rtns > 0)
        {
            cstrptr = path_rtn[0];
            if((cstrptr != NULL) ? (*cstrptr != '\0') : FALSE)
            {
                g_free(dest_path);
                dest_path = g_strdup(cstrptr);

                /* Path from file browser may have tailing deliminators,
                 * get rid of them.
                 */
                StripPath(dest_path);
            }
        }
        /* No destination path selected or destination path is not
         * a directory?
         */
        if(dest_path == NULL)
        {
            CDialogSetTransientFor(toplevel);
            CDialogGetResponse(
                "Invalid Destination",
                "The given destination directory is invalid",
                NULL,
                CDIALOG_ICON_WARNING,
                CDIALOG_BTNFLAG_OK,
                CDIALOG_BTNFLAG_OK
            );
            CDialogSetTransientFor(NULL);

            EDVArchiverSetBusy(archiver, FALSE);
            DO_FREE_LOCALS
            return;
        }

        /* Get additional extract from archive options. */
        if(!EDVArchiveExtractQueryOptions(
            core_ptr, toplevel, arch_obj, &preserve_directories
        ))
        {
            /* User canceled. */
            EDVArchiverSetBusy(archiver, FALSE);
            DO_FREE_LOCALS
            return;
        }


        if(1)
        {
	    const gchar *error_mesg;
            gchar **new_path = NULL;
            gint total_new_paths = 0;

            /* Extract. */
            status = EDVArchOPExtract(
                core_ptr, arch_obj, list, total,
                dest_path, &new_path, &total_new_paths,
                toplevel, TRUE, TRUE, &yes_to_all,
		preserve_directories
            );

            /* Get error message (if any) that might have occured in the
             * above operation.
             */
            error_mesg = EDVArchOPGetError();
            if(error_mesg != NULL)
            {
                CDialogSetTransientFor(toplevel);
                CDialogGetResponse(
                    "Operation Error",
                    error_mesg,
                    NULL,
                    CDIALOG_ICON_ERROR,
                    CDIALOG_BTNFLAG_OK,
                    CDIALOG_BTNFLAG_OK
                );
                CDialogSetTransientFor(NULL);
            }

            /* Report new disk objects added? */
            if((new_path != NULL) && !status)
            {
                gint i;
                struct stat lstat_buf;

                for(i = 0; i < total_new_paths; i++)
                {
                    cstrptr = new_path[i];
                    if(cstrptr == NULL)
                        continue;
                    if(lstat(cstrptr, &lstat_buf))
                        continue;

                    objects_extracted++;

                    EDVObjectAddedEmit(
                        core_ptr, cstrptr, &lstat_buf
                    );
                }
            }

            /* Deallocate coppies of new disk object's paths. */
            if(new_path != NULL)
            {
                gint i;

                for(i = 0; i < total_new_paths; i++)
                    g_free(new_path[i]);
                g_free(new_path);
                new_path = NULL;
                total_new_paths = 0;
            }
        }

        /* Update status bar. */
        if(1)
        {
            gchar *buf;

            switch(status)
            {
              case 0: case -5:
                buf = g_strdup_printf(
                    "Extracted %i object%s",
                    objects_extracted,
                    (objects_extracted == 1) ? "" : "s"
                );
                break;

              case -4:  /* Cancel. */
                buf = g_strdup_printf(
                    "Extract operation canceled"
                );
                break;

              default:  /* Error. */
                buf = g_strdup_printf(
                    "Unable to extract object%s",
                    (total == 1) ? "" : "s"
                );
                break;
            }
            EDVStatusBarMessage(archiver->status_bar, buf, FALSE);
            g_free(buf);
        }

        EDVArchiverSetBusy(archiver, FALSE);


        /* Unmap progress dialog, it may have been mapped if any
         * operations occured in the above call.
         */
        ProgressDialogBreakQuery(TRUE);
        ProgressDialogSetTransientFor(NULL);


        DO_FREE_LOCALS
#undef DO_FREE_LOCALS
}

/*
 *	Delete callback.
 */
void EDVArchiverOPDelete(edv_archiver_struct *archiver)
{
	gbool yes_to_all = FALSE;
	gint objects_deleted = 0;
	GtkWidget *toplevel;
	edv_archive_object_struct **list = NULL, *obj;
	edv_core_struct *core_ptr;
	gchar *arch_obj = NULL;
	const gchar *cstrptr, *src_path, *error_mesg;
	gint i, status, total = 0;


	if(archiver == NULL)
	    return;

	core_ptr = (edv_core_struct *)archiver->core_ptr;
	if(core_ptr == NULL)
	    return;

	toplevel = archiver->toplevel;

        /* Check and warn if write protect is enabled. */
        if(EDVCheckWriteProtect(core_ptr, TRUE))
            return;

#define DO_FREE_LOCALS	\
{ \
 src_path = NULL; \
 \
 for(i = 0; i < total; i++) \
  EDVArchObjectDelete(list[i]); \
 g_free(list); \
 list = NULL; \
 total = 0; \
 \
 g_free(arch_obj); \
 arch_obj = NULL; \
}

	/* Get current archive path. */
	cstrptr = EDVArchiverCurrentLocation(archiver);
	arch_obj = (cstrptr != NULL) ? g_strdup(cstrptr) : NULL;
	if(arch_obj == NULL)
	{
            DO_FREE_LOCALS
            return;
        }

	/* Get list of selected archive objects. */
	list = EDVArchiverGetSelectedObjects(archiver, &total);
	if((list == NULL) || (total <= 0))
	{
	    DO_FREE_LOCALS
	    return;
	}

	/* Get source path to first object. */
	if(total == 1)
	    src_path = (list[0] != NULL) ? list[0]->full_path : NULL;
	else
	    src_path = NULL;


        EDVArchiverSetBusy(archiver, TRUE);


	/* Get delete confirmation. */
	status = EDVConfirmArchiveDelete(
	    core_ptr, toplevel, src_path, total
	);
	switch(status)
	{
          case CDIALOG_RESPONSE_YES_TO_ALL:
	    yes_to_all = TRUE;
          case CDIALOG_RESPONSE_YES:
	    break;

	  default:
            EDVArchiverSetBusy(archiver, FALSE);
	    DO_FREE_LOCALS
	    return;
	    break;
	}


	/* Iterate through selected objects. */
	status = 0;
	for(i = 0; i < total; i++)
	{
	    obj = list[i];
	    if(obj == NULL)
		continue;

            /* Delete archieve object. */
	    status = EDVArchOPDelete(
		core_ptr, arch_obj, obj, toplevel,
		TRUE, TRUE, &yes_to_all
	    );

            /* Get error message (if any) that might have occured in the
             * above operation.
             */
            error_mesg = EDVArchOPGetError();
            if(error_mesg != NULL)
            {
                CDialogSetTransientFor(toplevel);
                CDialogGetResponse(
                    "Operation Error",
                    error_mesg,
                    NULL,
                    CDIALOG_ICON_ERROR,
                    CDIALOG_BTNFLAG_OK,
                    CDIALOG_BTNFLAG_OK
                );
                CDialogSetTransientFor(NULL);
            }


            /* Archive object was delected successfully? */
            if(!status)
            {
                /* Report archive object removed. */
		EDVArchiveObjectRemovedNotifyCB(
		    core_ptr, arch_obj, obj->full_path
		);

                /* Increment number of archive objects deleted. */
                objects_deleted++;
            }


            /* Error occured (not 0) and it was not a user `no' or
             * `not available' (-5) response?
             */
            if(status && (status != -5))
            {
                /* Skip rest of the disk object paths due to error. */
                break;
            }
	}


        /* Update status bar. */
        if(1)
        {
            gchar *buf;

            switch(status)
            {
              case 0: case -5:
                buf = g_strdup_printf(
                    "Deleted %i object%s",
                    objects_deleted,
                    (objects_deleted == 1) ? "" : "s"
                );
                break;

              case -4:  /* Cancel. */
                buf = g_strdup_printf(
                    "Delete operation canceled"
                );
                break;

              default:  /* Error. */
                buf = g_strdup_printf(
                    "Unable to delete object%s",
                    (total == 1) ? "" : "s"
                );
                break;
            }
            EDVStatusBarMessage(archiver->status_bar, buf, FALSE);
            g_free(buf);
        }

        EDVArchiverSetBusy(archiver, FALSE);


        /* Unmap progress dialog, it may have been mapped if any
         * operations occured in the above call.
         */
        ProgressDialogBreakQuery(TRUE);
        ProgressDialogSetTransientFor(NULL);


	DO_FREE_LOCALS
#undef DO_FREE_LOCALS
}

/*
 *	Select all.
 */
void EDVArchiverOPSelectAll(edv_archiver_struct *archiver)
{
        edv_core_struct *core_ptr;
        GtkCList *clist;


        if(archiver == NULL)
            return;

        core_ptr = (edv_core_struct *)archiver->core_ptr;
        if(core_ptr == NULL)
            return;

        /* Get contents clist. */
        clist = (GtkCList *)archiver->contents_clist;
        if(clist == NULL)
            return;

        EDVArchiverSetBusy(archiver, TRUE);

        /* Select all rows on clist. */
        gtk_clist_freeze(clist);
        gtk_clist_select_all(clist);
        gtk_clist_thaw(clist);

        /* Assume highest row index as the last selected row. */
        archiver->contents_clist_selected_row = clist->rows - 1;

        EDVStatusBarMessage(
            archiver->status_bar, "All objects selected", FALSE
        );

        EDVArchiverSetBusy(archiver, FALSE);
}

/*
 *	Unselect all.
 */
void EDVArchiverOPUnselectAll(edv_archiver_struct *archiver)
{
        edv_core_struct *core_ptr;
        GtkCList *clist;


        if(archiver == NULL)
            return;

        core_ptr = (edv_core_struct *)archiver->core_ptr;
        if(core_ptr == NULL)
            return;

        /* Get contents clist. */
        clist = (GtkCList *)archiver->contents_clist;
        if(clist == NULL)
            return;

        EDVArchiverSetBusy(archiver, TRUE);

        /* Unselect all rows on clist. */
        gtk_clist_freeze(clist);
        gtk_clist_unselect_all(clist);
        gtk_clist_thaw(clist);

        /* Mark contents clist's row as unselected. */
	archiver->contents_clist_selected_row = -1;

        EDVStatusBarMessage(
            archiver->status_bar, "All objects unselected", FALSE
        );

        EDVArchiverSetBusy(archiver, FALSE);
}

/*
 *      Invert selection callback.
 */
void EDVArchiverOPInvertSelection(edv_archiver_struct *archiver)
{
        edv_core_struct *core_ptr;
        GtkCList *clist;
        GList *glist, *selection;
        gint row, total_rows;


        if(archiver == NULL)
            return;

        core_ptr = (edv_core_struct *)archiver->core_ptr;
        if(core_ptr == NULL)
            return;

        /* Get contents clist. */
        clist = (GtkCList *)archiver->contents_clist;
        if(clist == NULL)
            return;

        EDVArchiverSetBusy(archiver, TRUE);
        gtk_clist_freeze(clist);

        /* Get copy of selected rows list from clist. */
        selection = (clist->selection != NULL) ?
            g_list_copy(clist->selection) : NULL;

        for(row = 0, total_rows = clist->rows;
            row < total_rows;
            row++
        )
        {
            for(glist = selection;
                glist != NULL;
                glist = glist->next
            )
            {
                if(row == (gint)glist->data)
                {
                    gtk_clist_unselect_row(clist, row, 0);
                    break;
                }
            }
            /* Row not selected? */
            if(glist == NULL)
                gtk_clist_select_row(clist, row, 0);
        }

        g_list_free(selection);

        gtk_clist_thaw(clist);
        EDVStatusBarMessage(
            archiver->status_bar, "Selection inverted", FALSE
        );
        EDVArchiverSetBusy(archiver, FALSE);
}

/*
 *	Refresh callback.
 */
void EDVArchiverOPRefresh(edv_archiver_struct *archiver)
{
        GtkCList *clist;


        if(archiver == NULL)
            return;

        EDVArchiverSetBusy(archiver, TRUE);
        GUIBlockInput(archiver->toplevel, TRUE);

        /* Update contents clist. */
        clist = (GtkCList *)archiver->contents_clist;
        if(clist != NULL)
        {
            gfloat last_vpos, last_hpos;


            /* Record last scroll position. */
            last_hpos = (clist->hadjustment != NULL) ?
                clist->hadjustment->value : 0.0;
            last_vpos = (clist->vadjustment != NULL) ?
                clist->vadjustment->value : 0.0;

            gtk_clist_freeze(clist);
            EDVArchiverContentsDoUpdate(archiver, TRUE);
            gtk_clist_thaw(clist);

            /* Scroll back to original position. */
            EDVScrollCListToPosition(
                clist, last_hpos, last_vpos
            );
        }

        GUIBlockInput(archiver->toplevel, FALSE);
        EDVArchiverSetBusy(archiver, FALSE);


        EDVArchiverUpdateMenus(archiver);

        EDVStatusBarMessage(
            archiver->status_bar, "Refreshed contents listing", FALSE
        );
}

/*
 *      Refresh all callback.
 */
void EDVArchiverOPRefreshAll(edv_archiver_struct *archiver)
{
	/* Refresh all is the same as a regular refresh for the recycle
	 * bin lists.
	 */
	EDVArchiverOPRefresh(archiver);
}
