#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <gtk/gtk.h>

#include "guiutils.h"
#include "fprompt.h"
#include "deskicon.h"


static void DeskIconFPromptApplyCB(
        gpointer data, const gchar *s
);
static void DeskIconDoRename(deskicon_struct *di);

static void DeskIconDrawIcon(deskicon_struct *di);
static gint DeskIconEventCB(
        GtkWidget *widget, GdkEvent *event, gpointer data
);
static void DeskIconSyncMoveLabelToIcon(
	deskicon_struct *di, gint x, gint y
);

void DeskIconGetPosition(
	deskicon_struct *di, gint *x, gint *y
);
void DeskIconSetPosition(
	deskicon_struct *di, gint x, gint y
);

static void DeskIconLoadIconPixmap(
        deskicon_struct *di, guchar **icon_data
);

deskicon_struct *DeskIconNew(
        gint x, gint y,
        guchar **icon_data, const gchar *label
);
void DeskIconSet(
        deskicon_struct *di,
        guchar **icon_data, const gchar *label
);
void DeskIconSetMapRefCB(
        deskicon_struct *di,
        void (*map_ref_cb)(gpointer, gpointer),
        gpointer client_data
);
void DeskIconSetMovedCB(
        deskicon_struct *di,
        void (*moved_cb)(gpointer, gint, gint, gpointer),
        gpointer client_data
);
void DeskIconSetMenu(deskicon_struct *di, GtkWidget *w);
GtkWidget *DeskIconGetMenu(deskicon_struct *di);
void DeskIconMap(deskicon_struct *di);
void DeskIconUnmap(deskicon_struct *di);
void DeskIconDelete(deskicon_struct *di);


#define MAX(a,b)	(((a) > (b)) ? (a) : (b))
#define MIN(a,b)	(((a) < (b)) ? (a) : (b))
#define CLIP(a,l,h)	(MIN(MAX((a),(l)),(h)))
#define ABSOLUTE(x)	(((x) < 0) ? ((x) * -1) : (x))


/*
 *	Float prompt renaming apply callback.
 */
static void DeskIconFPromptApplyCB(
	gpointer data, const gchar *s
)
{
	deskicon_struct *di = (deskicon_struct *)data;
	if((di == NULL) || (s == NULL))
	    return;

	DeskIconSet(di, NULL, s);
}

/*
 *	Maps the floating prompt to rename the desktop icon.
 */
static void DeskIconDoRename(deskicon_struct *di)
{
	gint x, y, width;
	GdkWindow *window;
	GtkWidget *w;


        if((di == NULL) || FPromptIsQuery())
            return;

	w = di->label_toplevel;
	if(w == NULL)
	    return;

	window = w->window;
	if(window == NULL)
	    return;

        gdk_window_get_root_origin(window, &x, &y);

	w = di->label;
        if(w == NULL)
            return;

	width = MAX(w->allocation.width, 100);
	x = x - 2 - ((width - w->allocation.width) / 2);
	y = y - 4;

	FPromptSetPosition(x, y);
	FPromptMapQuery(
	    NULL, di->label_str, NULL,
	    FPROMPT_MAP_NO_MOVE,
	    width, -1,
	    0,		/* Flags. */
	    di,
	    NULL,
	    DeskIconFPromptApplyCB,
	    NULL
	);

}

/*
 *	Redraws the icon toplevel on the given desktop icon.
 */
static void DeskIconDrawIcon(deskicon_struct *di)
{
	gint state, width, height;
	GdkGC *gc;
	GdkPixmap *pixmap;
	GdkBitmap *mask;
	GdkWindow *window;
	GtkWidget *w;
	GtkStyle *style;


	if(di == NULL)
	    return;

	w = di->icon_da;
	pixmap = di->icon_pixmap;
	mask = di->icon_mask;
	if((w == NULL) || (pixmap == NULL))
	    return;

	gc = di->gc;
	window = w->window;
	style = gtk_widget_get_style(w);
	if((gc == NULL) || (window == NULL) || (style == NULL))
	    return;

	state = GTK_WIDGET_STATE(w);
	width = di->icon_width;
	height = di->icon_height;


	/* Begin drawing pixmap. */
	gdk_gc_set_function(gc, GDK_COPY);
	gdk_gc_set_fill(gc, GDK_SOLID);
        gdk_gc_set_clip_mask(gc, mask);
        gdk_gc_set_clip_origin(gc, 0, 0);
        gdk_gc_set_foreground(gc, &style->bg[state]);
	gdk_draw_pixmap(
	    (GdkDrawable *)window, gc,
	    (GdkDrawable *)pixmap,
	    0, 0,
	    0, 0,
	    width, height
	);

	/* Draw selection? */
	if(state == GTK_STATE_SELECTED)
	{
	    gdk_gc_set_function(gc, GDK_INVERT);
            gdk_gc_set_foreground(gc, &style->white);

            gdk_draw_rectangle(
                (GdkDrawable *)window, gc, TRUE,
		0, 0, width, height
	    );
	}
}

/*
 *	Event signal handler.
 */
static gint DeskIconEventCB(
	GtkWidget *widget, GdkEvent *event, gpointer data
)
{
	static gbool reenterent = FALSE;
	gint status = FALSE;
	gint etype;
	GdkEventButton *button;
	GdkEventMotion *motion;
	deskicon_struct *di = (deskicon_struct *)data;
	if((widget == NULL) || (event == NULL) || (di == NULL))
	    return(status);

	if(reenterent)
	    return(status);
	else
	    reenterent = TRUE;

	/* Get event type. */
	etype = event->type;

	/* Handle by event type. */
	switch(etype)
	{
	  case GDK_EXPOSE:
	    DeskIconDrawIcon(di);
	    status = TRUE;
	    break;

	  case GDK_BUTTON_PRESS:
	    button = (GdkEventButton *)event;
            switch(button->button)
            {
	      case 1:
		di->button_down = button->button;
		di->last_x = (gint)button->x_root;
		di->last_y = (gint)button->y_root;
		if(di->icon_toplevel != NULL)
		    gtk_widget_set_state(di->icon_toplevel, GTK_STATE_SELECTED);
		DeskIconDrawIcon(di);
		gdk_pointer_grab(
		    button->window, TRUE,
		    GDK_BUTTON_PRESS_MASK | GDK_BUTTON_RELEASE_MASK |
		    GDK_POINTER_MOTION_MASK,
		    NULL, NULL, button->time
		);
		break;
	      case 2:
		DeskIconDoRename(di);
		break;
	      case 3:
		if(di->menu != NULL)
		    gtk_menu_popup(
			GTK_MENU(di->menu), NULL, NULL,
			NULL, NULL,
			button->button, button->time
		    );
		break;
	    }
	    status = TRUE;
	    break;

          case GDK_2BUTTON_PRESS:
            button = (GdkEventButton *)event;
            switch(button->button)
            {
              case 1:
		di->button_down = 0;
		if(di->icon_toplevel != NULL)
		    gtk_widget_set_state(di->icon_toplevel, GTK_STATE_NORMAL);
		DeskIconDrawIcon(di);
		gdk_pointer_ungrab(button->time);
		/* Call map reference callback? */
		if(di->map_ref_cb != NULL)
		    di->map_ref_cb(
			(gpointer)di, di->map_ref_client_data
		    );
		break;
	    }
	    status = TRUE;
	    break;

          case GDK_BUTTON_RELEASE:
            button = (GdkEventButton *)event;
            /* Set icon values to reflect it as not being dragged. */
            di->button_down = 0;
	    if(di->icon_toplevel != NULL)
		gtk_widget_set_state(di->icon_toplevel, GTK_STATE_NORMAL);
            DeskIconDrawIcon(di);
	    gdk_pointer_ungrab(button->time);
            /* Call moved callback? */
            if(di->moved_cb != NULL)
	    {
		GtkWidget *w = di->icon_toplevel;
                GdkWindow *window = (w != NULL) ? w->window : NULL;
                if(window != NULL)
                {
                    gint cx, cy;

                    gdk_window_get_position(window, &cx, &cy);
		    di->moved_cb(
			(gpointer)di, (gint)cx, (gint)cy,
			di->moved_client_data
		    );
		}
	    }
	    status = TRUE;
            break;

	  case GDK_MOTION_NOTIFY:
	    motion = (GdkEventMotion *)event;
	    if(di->button_down)
	    {
		gint x, y, dx, dy;
		GdkModifierType mask;
		GdkWindow *window = motion->window;
		GtkWidget *w;


		/* If this event is a motion hint then request additional
		 * motion events to be sent.
		 */
		if(motion->is_hint)
		{
		    gdk_window_get_pointer(
                        window, &x, &y, &mask
                    );
		}
		x = (gint)motion->x_root;
		y = (gint)motion->y_root;

		/* Calculate motion movement. */
		dx = x - di->last_x;
                dy = y - di->last_y;


		/* Get icon toplevel window and move it. */
		w = di->icon_toplevel;
		window = (w != NULL) ? w->window : NULL;
		if(window != NULL)
		{
		    gint cx, cy;

		    gdk_window_get_position(window, &cx, &cy);
		    cx += dx;
		    cy += dy;
		    gdk_window_move(window, cx, cy);
		    DeskIconSyncMoveLabelToIcon(di, cx, cy);
		}

		/* Record last motion coordinates. */
		di->last_x = x;
		di->last_y = y;

		status = TRUE;
	    }
	    break;

	}

	reenterent = FALSE;
	return(status);
}

/*
 *	Moves label toplevel to the location aligned with the current
 *	icon toplevel position.
 */
static void DeskIconSyncMoveLabelToIcon(
	deskicon_struct *di, gint x, gint y
)
{
	gint width, height, nx, ny;
	GdkWindow *window;
	GtkWidget *w;


	if(di == NULL)
	    return;

	/* Get icon toplevel window. */
	w = di->icon_toplevel;
	window = (w != NULL) ? w->window : NULL;
	if(window == NULL)
	    return;

	/* Get size of icon toplevel. */
	width = di->icon_width;
	height = di->icon_height;

	/* Get label toplevel. */
	w = di->label_toplevel;
	window = (w != NULL) ? w->window : NULL;
        if(window == NULL)
            return;

	nx = x + (width / 2) - (w->allocation.width / 2);
	ny = y + height;
	gtk_widget_set_uposition(w, nx, ny);
	gdk_window_move(window, nx, ny);

	gtk_widget_queue_resize(di->label);
	gtk_widget_queue_resize(w);
}


/*
 *	Returns the position of the icon toplevel of the desktop icon.
 */
void DeskIconGetPosition(
        deskicon_struct *di, gint *x, gint *y
)
{
	gint cx, cy;
	GdkWindow *window;
	GtkWidget *w;


	if(x != NULL)
	    *x = 0;
	if(y != NULL)
	    *y = 0;

	if(di == NULL)
	    return;

	w = di->icon_toplevel;
	window = (w != NULL) ? w->window : NULL;
	if(window == NULL)
	    return;

	gdk_window_get_root_origin(window, &cx, &cy);
	if(x != NULL)
            *x = cx;
        if(y != NULL)
            *y = cy;
}

/*
 *	Moves the desktop icon to the specified position.
 */
void DeskIconSetPosition(
        deskicon_struct *di, gint x, gint y
)
{
        GdkWindow *window;
        GtkWidget *w;


        if(di == NULL)
            return;

        w = di->icon_toplevel;
        window = (w != NULL) ? w->window : NULL;
        if(window == NULL)
            return;

	gtk_widget_set_uposition(w, x, y);
	gdk_window_move(window, x, y);
	DeskIconSyncMoveLabelToIcon(di, x, y);
}

/*
 *	Loads a new icon pixmap and mask pair from the given icon_data to
 *	the desktop icon structure.
 *
 *	If an existing icon pixmap and mask pair is on the desktop icon
 *	structure then they will be unref'ed.
 */
static void DeskIconLoadIconPixmap(
	deskicon_struct *di, guchar **icon_data
)
{
	gint width, height;
	GdkWindow *window;
	GdkPixmap *pixmap, *old_pixmap;
	GdkBitmap *mask, *old_mask;
	GtkWidget *w;
	GtkStyle *style;


	if((di == NULL) || (icon_data == NULL))
	    return;

	w = di->icon_da;
	if(w == NULL)
	    return;

        style = gtk_widget_get_style(w);
	window = w->window;
	if((style == NULL) || (window == NULL))
	    return;

	/* Record old pixmap and mask pair that will be unref'ed later on
	 * successful load.
	 */
	old_pixmap = di->icon_pixmap;
	old_mask = di->icon_mask;

	/* Load new icon pixmap and mask pair from the given data. */
	pixmap = gdk_pixmap_create_from_xpm_d(
	    window, &mask,
	    &style->bg[GTK_STATE_NORMAL], (gchar **)icon_data
	);
	if(pixmap == NULL)
	    return;


	/* Loaded icon pixmap and mask pair  successfully, now begin
	 * updating values.
	 */

	/* Size. */
	gdk_window_get_size((GdkWindow *)pixmap, &width, &height);
	di->icon_width = width;
	di->icon_height = height;

	/* Pixmap and mask. */
	di->icon_pixmap = pixmap;
	di->icon_mask = mask;


	/* Update size and shape of toplevel widget and drawing area. */
	w = di->icon_toplevel;
	if(w != NULL)
	{
	    gtk_widget_set_usize(w, width, height);
	    gtk_widget_shape_combine_mask(w, mask, 0, 0);
	}
        w = di->icon_da;
        if(w != NULL)
        {
	    gtk_drawing_area_size(GTK_DRAWING_AREA(w), width, height);
            gtk_widget_set_usize(w, width, height);
            gtk_widget_shape_combine_mask(w, mask, 0, 0);
        }

	/* Unref old icon pixmap and mask. */
	if(old_pixmap != NULL)
	    gdk_pixmap_unref(old_pixmap);
	if(old_mask != NULL)
	    gdk_bitmap_unref(old_mask);
}

/*
 *	Creates a new desktop icon.
 */
deskicon_struct *DeskIconNew(
        gint x, gint y,
        guchar **icon_data, const gchar *label
)
{
	GdkWindow *window;
	GtkWidget *w, *parent;
	deskicon_struct *di = (deskicon_struct *)g_malloc0(
	    sizeof(deskicon_struct)
	);
	if(di == NULL)
	    return(di);

	/* Reset values. */
	di->initialized = TRUE;
	di->map_state = FALSE;
	di->icon_pixmap = NULL;
	di->icon_mask = NULL;
	di->label_str = (label != NULL) ? g_strdup(label) : NULL;
	di->menu = NULL;
	di->button_down = 0;
	di->last_x = 0;
	di->last_y = 0;
	di->map_ref_cb = NULL;
        di->map_ref_client_data = NULL;
        di->moved_cb = NULL;
        di->moved_client_data = NULL;

        /* Icon toplevel. */
        di->icon_toplevel = w = gtk_window_new(GTK_WINDOW_POPUP);
        gtk_widget_add_events(
            w,
	    GDK_BUTTON_PRESS_MASK | GDK_BUTTON_RELEASE_MASK |
	    GDK_POINTER_MOTION_MASK | GDK_POINTER_MOTION_HINT_MASK |
	    GDK_STRUCTURE_MASK
        );
        gtk_signal_connect(
            GTK_OBJECT(w), "button_press_event",
            GTK_SIGNAL_FUNC(DeskIconEventCB), di
        );
        gtk_signal_connect(
            GTK_OBJECT(w), "button_release_event",
            GTK_SIGNAL_FUNC(DeskIconEventCB), di
        );
        gtk_signal_connect(
            GTK_OBJECT(w), "motion_notify_event",
            GTK_SIGNAL_FUNC(DeskIconEventCB), di
        );
        gtk_widget_realize(w);
        window = w->window;
        if(window != NULL)
        {
	    gdk_window_move(window, x, y);

	    /* No decorations. */
            gdk_window_set_decorations(window, 0);

	    /* No functions, we'll do all the WM managing ourselves. */
            gdk_window_set_functions(window, 0);
        }
        gtk_container_border_width(GTK_CONTAINER(w), 0);
	parent = w;

	/* Create graphics context. */
	di->gc = gdk_gc_new(window);

	/* Icon drawing area. */
	di->icon_da = w = gtk_drawing_area_new();
	gtk_drawing_area_size(GTK_DRAWING_AREA(w), 48, 48);
        gtk_widget_add_events(
            w,
            GDK_EXPOSURE_MASK
        );
        gtk_signal_connect(
            GTK_OBJECT(w), "expose_event",
            GTK_SIGNAL_FUNC(DeskIconEventCB), di
        );
	gtk_container_add(GTK_CONTAINER(parent), w);
	gtk_widget_show(w);




        /* Label toplevel. */
        di->label_toplevel = w = gtk_window_new(GTK_WINDOW_POPUP);
        gtk_widget_add_events(
            w,
	    GDK_BUTTON_PRESS_MASK | GDK_BUTTON_RELEASE_MASK |
	    GDK_POINTER_MOTION_MASK | GDK_POINTER_MOTION_HINT_MASK |
	    GDK_STRUCTURE_MASK
        );
        gtk_signal_connect(
            GTK_OBJECT(w), "button_press_event",
            GTK_SIGNAL_FUNC(DeskIconEventCB), di
        );
        gtk_signal_connect(
            GTK_OBJECT(w), "button_release_event",
            GTK_SIGNAL_FUNC(DeskIconEventCB), di
        );
        gtk_signal_connect(
            GTK_OBJECT(w), "motion_notify_event",
            GTK_SIGNAL_FUNC(DeskIconEventCB), di
        );
        gtk_widget_realize(w);
        window = w->window;
        if(window != NULL)
        {
	    /* Do not set position for label toplevel, it follows the
	     * pixmap toplevel.
	     */

            /* No decorations. */
            gdk_window_set_decorations(window, 0);

            /* No functions, we'll do all the WM managing ourselves. */
            gdk_window_set_functions(window, 0);
        }
        gtk_container_border_width(GTK_CONTAINER(w), 0);
	parent = w;

	/* Get resources for creating label. */
	if(label != NULL)
	{
	    gint state;
	    GdkColor *c;
	    GtkRcStyle *rcstyle = gtk_rc_style_new();

	    di->label = w = gtk_label_new(label);
	    gtk_container_add(GTK_CONTAINER(parent), w);
	    gtk_widget_show(w);

	    state = GTK_STATE_NORMAL;
	    rcstyle->color_flags[state] = GTK_RC_BG | GTK_RC_FG;
	    c = &rcstyle->bg[state];
	    c->red = 1.0 * (guint16)-1;
	    c->green = 1.0 * (guint16)-1;
	    c->blue = 1.0 * (guint16)-1;
            c = &rcstyle->fg[state];
            c->red = 0.0 * (guint16)-1;
            c->green = 0.0 * (guint16)-1;
            c->blue = 0.0 * (guint16)-1;
	    /* Modify style of label and parent. */
	    gtk_widget_modify_style(w, rcstyle);
            gtk_widget_modify_style(parent, rcstyle);

	    GUIRCStyleDeallocUnref(rcstyle);
	}


	/* Load given icon data. */
	DeskIconLoadIconPixmap(di, icon_data);

	/* Set given position. */
	DeskIconSetPosition(di, x, y);

	return(di);
}

/*
 *	Updates the icon and/or label. Either may be NULL implying do not
 *	change.
 */
void DeskIconSet(
        deskicon_struct *di,
        guchar **icon_data, const gchar *label
)
{
	GtkWidget *w, *parent;


	if(di == NULL)
	    return;

	if((di->icon_toplevel == NULL) || (di->label_toplevel == NULL))
	    return;

        /* Load given icon data. */
        DeskIconLoadIconPixmap(di, icon_data);

        /* Get resources for creating label. */
	parent = di->label_toplevel;
        if((label != NULL) && (parent != NULL))
        {
            gint state;
            GdkColor *c;
            GtkRcStyle *rcstyle = gtk_rc_style_new();

	    /* Label widget not already created? */
	    if(di->label == NULL)
	    {
		w = gtk_label_new(label);
		gtk_container_add(GTK_CONTAINER(parent), w);
		gtk_widget_show(w);
	    }
	    else
	    {
		w = di->label;
		gtk_label_set_text(GTK_LABEL(w), label);
	    }

            state = GTK_STATE_NORMAL;
            rcstyle->color_flags[state] = GTK_RC_BG | GTK_RC_FG;
            c = &rcstyle->bg[state];
            c->red = 1.0 * (guint16)-1;
            c->green = 1.0 * (guint16)-1;
            c->blue = 1.0 * (guint16)-1;
            c = &rcstyle->fg[state];
            c->red = 0.0 * (guint16)-1;
            c->green = 0.0 * (guint16)-1;
            c->blue = 0.0 * (guint16)-1;
            /* Modify style of label and parent. */
            gtk_widget_modify_style(w, rcstyle);
            gtk_widget_modify_style(parent, rcstyle);

            GUIRCStyleDeallocUnref(rcstyle);

	    /* Record label. */
	    g_free(di->label_str);
	    di->label_str = g_strdup(label);
        }

	/* Get icon toplevel again (after changes) and move the label
	 * to align properly with it.
	 */
	if(1)
	{
	    gint cx, cy;
	    DeskIconGetPosition(di, &cx, &cy);
	    DeskIconSyncMoveLabelToIcon(di, cx, cy);

	    DeskIconDrawIcon(di);
	}
}

/*
 *	Sets the map reference callback function.
 */
void DeskIconSetMapRefCB(
        deskicon_struct *di,
        void (*map_ref_cb)(gpointer, gpointer),
        gpointer client_data
)
{
	if(di == NULL)
	    return;

	di->map_ref_cb = map_ref_cb;
	di->map_ref_client_data = client_data;
}

/*
 *      Sets the moved callback function.
 */
void DeskIconSetMovedCB(
        deskicon_struct *di,
        void (*moved_cb)(gpointer, gint, gint, gpointer),
        gpointer client_data
)
{
        if(di == NULL)
            return;

        di->moved_cb = moved_cb;
        di->moved_client_data = client_data;
}

/*
 *	Sets the menu, destroying the previous menu (if any).
 */
void DeskIconSetMenu(deskicon_struct *di, GtkWidget *w)
{
	if(di == NULL)
	    return;

	if(di->menu != NULL)
	    gtk_widget_destroy(di->menu);
	di->menu = w;

	/* Is given widget valid and a GtkMenu? If it is not then
	 * explicitly set it to NULL.
	 */
	if(w != NULL)
	{
	    if(!GTK_IS_MENU(w))
		di->menu = w = NULL;
	}
}

/*
 *	Returns the current menu (if any).
 */
GtkWidget *DeskIconGetMenu(deskicon_struct *di)
{
	return((di != NULL) ? di->menu : NULL);
}


/*
 *	Maps the desktop icon.
 */
void DeskIconMap(deskicon_struct *di)
{
	GtkWidget *w;


	if(di == NULL)
	    return;

	if(!di->map_state)
	{
	    gint x, y;


	    w = di->icon_toplevel;
	    if(w != NULL)
	    {
		gtk_widget_show(w);
                if(w->window != NULL)
                    gdk_window_lower(w->window);
            }

            w = di->label_toplevel;
            if(w != NULL)
	    {
                gtk_widget_show(w);
		if(w->window != NULL)
		    gdk_window_lower(w->window);
	    }

	    di->map_state = TRUE;

            DeskIconGetPosition(di, &x, &y);
            DeskIconSetPosition(di, x, y);
	}
}

/*
 *	Unmaps the desktop icon.
 */
void DeskIconUnmap(deskicon_struct *di)
{
        GtkWidget *w;


        if(di == NULL)
            return;

        if(di->map_state)
        {
            w = di->icon_toplevel;
            if(w != NULL)
                gtk_widget_hide(w);

            w = di->label_toplevel;
            if(w != NULL)
                gtk_widget_hide(w);

            di->map_state = FALSE;
        }
}

/*
 *	Deallocates all resources on the given desktop icon and
 *	deallocates the structure itself.
 */
void DeskIconDelete(deskicon_struct *di)
{
        GdkPixmap **pixmap;
        GdkBitmap **mask;
	GtkWidget **w;


	if(di == NULL)
	    return;

        if(di->initialized)
        {
#define DO_UNREF_PIXMAP	\
{ \
 if((*pixmap) != NULL) \
 { \
  GdkPixmap *tpm = *pixmap; \
  (*pixmap) = NULL; \
  gdk_pixmap_unref(tpm); \
 } \
}
#define DO_UNREF_MASK	\
{ \
 if((*mask) != NULL) \
 { \
  GdkBitmap *tbm = *mask; \
  (*mask) = NULL; \
  gdk_bitmap_unref(tbm); \
 } \
}
#define DO_DESTROY_WIDGET       \
{ \
 if((*w) != NULL) \
 { \
  GtkWidget *tmp_w = *w; \
  (*w) = NULL; \
  gtk_widget_destroy(tmp_w); \
 } \
}

            /* Begin destroying widgets. */
            w = &di->menu;
            DO_DESTROY_WIDGET

            w = &di->label;
            DO_DESTROY_WIDGET
            w = &di->label_toplevel;
            DO_DESTROY_WIDGET

            w = &di->icon_da;
            DO_DESTROY_WIDGET
            w = &di->icon_toplevel;
            DO_DESTROY_WIDGET

	    pixmap = &di->icon_pixmap;
	    DO_UNREF_PIXMAP
            mask = &di->icon_mask;
            DO_UNREF_MASK

	    if(di->gc != NULL)
	    {
		gdk_gc_unref(di->gc);
		di->gc = NULL;
	    }

	    g_free(di->label_str);
	    di->label_str = NULL;

#undef DO_DESTROY_WIDGET
#undef DO_UNREF_PIXMAP
#undef DO_UNREF_MASK
	}

	/* Deallocate structure itself. */
	g_free(di);
}
