#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <gtk/gtk.h>
#include <gdk/gdkkeysyms.h>

#include "../include/string.h"
#include "../include/disk.h"

#include "guiutils.h"
#include "cdialog.h"
#include "fb.h"
#include "fprompt.h"

#include "edvtypes.h"
#include "edvcfg.h"
#include "edvdevices.h"
#include "deviceswin.h"
#include "endeavour.h"
#include "edvcb.h"
#include "edvutils.h"
#include "edvutilsgtk.h"
#include "edvcfglist.h"
#include "config.h"

#include "images/icon_ok_20x20.xpm"
#include "images/icon_select_20x20.xpm"
#include "images/icon_cancel_20x20.xpm"
#include "images/icon_properties2_48x48.xpm"


static void EDVDevicesEditIconUpdateDisplay(
        edv_device_editwin_icon_struct *ei, gint state
);
static void EDVDevicesEditIconChangeCB(
        GtkWidget *widget, gpointer data
);
static void EDVDevicesEditIconPrevCB(
        GtkWidget *widget, gpointer data
);
static void EDVDevicesEditIconNextCB(
        GtkWidget *widget, gpointer data
);

static void EDVDevicesEditBrowseCB(gpointer button, gpointer data);

static gint EDVDevicesEditWinDeleteEventCB(
        GtkWidget *widget, GdkEvent *event, gpointer data
);
static void EDVDevicesEditWinOKCB(
        GtkWidget *widget, gpointer data
);
static void EDVDevicesEditWinApplyCB(
	GtkWidget *widget, gpointer data
);
static void EDVDevicesEditWinCancelCB(
        GtkWidget *widget, gpointer data
);

void EDVDevicesEditWinFetchValues(
	edv_device_editwin_struct *ew, gint dev_num
);
void EDVDevicesEditWinDeviceRemovedCB(
        edv_device_editwin_struct *ew, gint dev_num
);

static edv_device_editwin_icon_struct *EDVDevicesEditIconNew(
	gpointer editwin, GtkWidget *parent, const gchar *title,
	gint btn_width, gint btn_height
);
static void EDVDevicesEditIconDelete(edv_device_editwin_icon_struct *ei);

edv_device_editwin_struct *EDVDevicesEditWinNew(
        gpointer core_ptr, gpointer listwin
);
void EDVDevicesEditWinMap(
        edv_device_editwin_struct *ew
);
void EDVDevicesEditWinUnmap(
        edv_device_editwin_struct *ew
);
void EDVDevicesEditWinDelete(
        edv_device_editwin_struct *ew
);


#define EDITWIN_WIDTH	500
#define EDITWIN_HEIGHT	-1

#define EDITWIN_BTN_WIDTH	(100 + (2 * 3))
#define EDITWIN_BTN_HEIGHT	(30 + (2 * 3))

#ifdef PROG_LANGUAGE_ENGLISH
# define EDITWIN_TITLE	"Device"
#endif
#ifdef PROG_LANGUAGE_SPANISH
# define EDITWIN_TITLE	"El Artefacto"
#endif
#ifdef PROG_LANGUAGE_FRENCH
# define EDITWIN_TITLE  "Appareil"
#endif


/*
 *	Updates the icon displayed on the button of the given edit window
 *	icon structure with the corresponding pixmap and mask pair of
 *	the given state.
 */
static void EDVDevicesEditIconUpdateDisplay(
	edv_device_editwin_icon_struct *ei, gint state
)
{
        const gchar *label_str = "";
        GdkPixmap *pixmap;
        GdkBitmap *mask;
	GtkWidget *w, *parent;


	if(ei == NULL)
	    return;

        if((state < 0) || (state >= EDV_DEVICE_TOTAL_ICON_STATES))
	    return;

	/* Get references to pixmap and mask pair specified by the given
	 * state. 
	 */
	pixmap = ei->pixmap[state];
	mask = ei->mask[state];

	/* Get button widget as the parent to create GtkPixmap in. */
	parent = ei->button;
	if(parent == NULL)
	    return;

	/* Referenced pixmap valid? */
	if(pixmap != NULL)
        {
	    /* Referenced pixmap is valid, now check if the GtkPixmap that
	     * is to be used to display the pixmap has been created yet.
	     * if not then create a new one.
	     */
	    w = ei->gtk_pixmap;
	    if(w == NULL)
	    {
		ei->gtk_pixmap = w = gtk_pixmap_new(pixmap, mask);
		gtk_container_add(GTK_CONTAINER(parent), w);
		gtk_widget_show(w);
	    }
	    else
	    {
		/* GtkPixmap exists, so just update it. */
		gtk_pixmap_set(GTK_PIXMAP(w), pixmap, mask);
		gtk_widget_show(w);
	    }
        }
	else
	{
	    /* No referenced pixmap, so unmap the the GtkPixmap. */
	    w = ei->gtk_pixmap;
	    if(w != NULL)
	    {
		gtk_widget_hide(w);
	    }
	}


	/* Update state label, get the currect label string to display
	 * for the state label based on the given state.
	 */
        switch(state)
        {
          case EDV_DEVICE_ICON_STATE_UNMOUNTED:
	    label_str =
#ifdef PROG_LANGUAGE_ENGLISH
		"Unmounted";
#endif
#ifdef PROG_LANGUAGE_SPANISH
                "No El Monte";
#endif
#ifdef PROG_LANGUAGE_FRENCH
                "Pas Mont";
#endif
	    break;
          case EDV_DEVICE_ICON_STATE_SELECTED:
            label_str =
#ifdef PROG_LANGUAGE_ENGLISH
                "Selected";
#endif
#ifdef PROG_LANGUAGE_SPANISH
                "Escogi";
#endif
#ifdef PROG_LANGUAGE_FRENCH
                "Choisi";
#endif
            break;
          case EDV_DEVICE_ICON_STATE_STANDARD:
            label_str =
#ifdef PROG_LANGUAGE_ENGLISH
                "Standard";
#endif
#ifdef PROG_LANGUAGE_SPANISH
                "Estndar";
#endif
#ifdef PROG_LANGUAGE_FRENCH
                "Norme";
#endif
            break;
	}
	w = ei->state_label;
	if(w != NULL)
	    gtk_label_set_text(GTK_LABEL(w), label_str);

}

/*
 *	Edit window icon change callback.
 */
static void EDVDevicesEditIconChangeCB(
        GtkWidget *widget, gpointer data
)
{
	static gbool reenterent = FALSE;
	gint state, status;
	const gchar *cstrptr, *old_path;
	gchar *old_parent_path;
	GdkWindow *window;
	GtkAdjustment *adj;
	GtkStyle *style;
	GtkWidget *w, *toplevel;
	fb_type_struct **ftype = NULL, *ftype_rtn = NULL;
	gint total_ftypes = 0;
	gchar **path_rtn = NULL;
	gint total_path_rtns = 0;
	edv_core_struct *core_ptr;
        edv_device_editwin_struct *ew;
        edv_device_editwin_icon_struct *ei =
            (edv_device_editwin_icon_struct *)data;
        if(ei == NULL)
            return;

	if(FileBrowserIsQuery())
	    return;

	adj = ei->adj;
	if(adj == NULL)
	    return;

	/* Get current state and make sure it is in bounds. */
	state = (gint)adj->value;
	if((state < 0) || (state >= EDV_DEVICE_TOTAL_ICON_STATES))
	    return;

        ew = (edv_device_editwin_struct *)ei->editwin;
        if(ew == NULL)
            return;

        /* Get values from device edit window's toplevel widget. */
        toplevel = w = ew->toplevel;
        if(w == NULL)
            return;
        style = gtk_widget_get_style(w);
        window = w->window;
        if((style == NULL) || (window == NULL))
            return;

	core_ptr = (edv_core_struct *)ew->core_ptr;
	if(core_ptr == NULL)
	    return;

	if(reenterent)
	    return;
	else
	    reenterent = TRUE;


	/* Get current path to the icon file for this state (can be
	 * NULL).
	 */
	old_path = ei->path[state];
	if(old_path != NULL)
	{
	    cstrptr = GetParentDir(old_path);
	    old_parent_path = (cstrptr != NULL) ?
		g_strdup(cstrptr) : NULL;
	}
	else
	{
	    old_parent_path = NULL;
	}

	/* Create file types list. */
	FileBrowserTypeListNew(
	    &ftype, &total_ftypes,
	    ".xpm", "XPM Image"
	);
        FileBrowserTypeListNew(
            &ftype, &total_ftypes,
            "*.*", "All files"
        );

	/* Current old_path must be considered invalid at this point. */
	old_path = NULL;


	/* Query user for new icon file. */
	FileBrowserSetTransientFor(toplevel);
	status = FileBrowserGetResponse(
	    "Select Icon",
	    "Select", "Cancel",
	    old_parent_path,
	    ftype, total_ftypes,
	    &path_rtn, &total_path_rtns,
	    &ftype_rtn
	);
	FileBrowserSetTransientFor(NULL);

        /* Got user response? */
        if(status)
	{
	    const gchar *new_path = (total_path_rtns > 0) ?
		path_rtn[0] : NULL;


	    if((new_path != NULL) ? (*new_path != '\0') : FALSE)
	    {
                GdkPixmap *old_pixmap = ei->pixmap[state];
                GdkBitmap *old_mask = ei->mask[state];


                /* Load new pixmap and mask. */
                ei->pixmap[state] = gdk_pixmap_create_from_xpm(
                    window, &ei->mask[state],
                    &style->bg[GTK_STATE_NORMAL],
                    new_path
                );
                if(ei->pixmap[state] != NULL)
                {
                    /* New pixmap and mask loaded successfully. */

                    /* Unref old pixmap and mask. */
                    if(old_pixmap != NULL)
                    {
                        gdk_pixmap_unref(old_pixmap);
                        old_pixmap = NULL;
                    }
                    if(old_mask != NULL)
                    {
                        gdk_bitmap_unref(old_mask);
                        old_mask = NULL;
                    }

                    /* Set new icon file path. */
                    g_free(ei->path[state]);
                    ei->path[state] = g_strdup(new_path);

                    /* Update displayed icon on the given edit icon
                     * structure.
                     */
                    EDVDevicesEditIconUpdateDisplay(ei, state);
                }
                else
                {
                    /* Load failed, unref current pixmap and mask pair. */
                    gchar *buf;


                    /* Unref current pixmap and mask. */
                    if(old_pixmap != NULL)
                    {
                        gdk_pixmap_unref(old_pixmap);
                        old_pixmap = NULL;
                    }
                    if(old_mask != NULL)
                    {
                        gdk_bitmap_unref(old_mask);
                        old_mask = NULL;
                    }

                    /* Mark that current pixmap and mask pair have been
                     * unloaded.
                     */
                    ei->pixmap[state] = NULL;
                    ei->mask[state] = NULL;

                    /* Unset path. */
                    g_free(ei->path[state]);
                    ei->path[state] = NULL;

                    /* Update displayed icon on the given edit icon
                     * structure.
                     */
                    EDVDevicesEditIconUpdateDisplay(ei, state);


                    /* Notify user about failed load. */
                    buf = g_strdup_printf(
"Unable to load icon file:\n\
\n\
    %s\n\
\n\
Please verify that the specified path is correct and\n\
the file is a valid xpm image file.",
                        new_path
                    );
                    CDialogSetTransientFor(toplevel);
                    CDialogGetResponse(
                        "Load Failed", buf, NULL,
                        CDIALOG_ICON_ERROR,
                        CDIALOG_BTNFLAG_OK,
                        CDIALOG_BTNFLAG_OK
                    );
                    CDialogSetTransientFor(NULL);
                    g_free(buf);
                }
            }
        }


	/* Deallocate file types list. */
	FileBrowserDeleteTypeList(ftype, total_ftypes);

	/* Deallocate coppied parent path. */
	g_free(old_parent_path);
	old_parent_path = NULL;


	reenterent = FALSE;
}

/*
 *      Prev icon edit window icon callback.
 */
static void EDVDevicesEditIconPrevCB(
        GtkWidget *widget, gpointer data
)
{
        static gbool reenterent = FALSE;
	GtkAdjustment *adj;
        edv_device_editwin_struct *ew;
        edv_device_editwin_icon_struct *ei =
            (edv_device_editwin_icon_struct *)data;
        if(ei == NULL)
            return;

        ew = (edv_device_editwin_struct *)ei->editwin;
        if(ew == NULL)
            return;

	adj = ei->adj;
	if(adj == NULL)
	    return;

        if(reenterent)
            return;
        else
            reenterent = TRUE;

        /* Increment adjustment position. */
        if(adj->value > adj->lower)
            adj->value -= adj->step_increment;
        if(adj->value < adj->lower)
            adj->value = adj->lower;

	EDVDevicesEditIconUpdateDisplay(ei, (gint)adj->value);

        reenterent = FALSE;
}

/*
 *	Next icon edit window icon callback.
 */
static void EDVDevicesEditIconNextCB(
        GtkWidget *widget, gpointer data
)
{
        static gbool reenterent = FALSE;
        GtkAdjustment *adj;
	edv_device_editwin_struct *ew;
	edv_device_editwin_icon_struct *ei =
	    (edv_device_editwin_icon_struct *)data;
	if(ei == NULL)
	    return;

	ew = (edv_device_editwin_struct *)ei->editwin;
	if(ew == NULL)
	    return;

        adj = ei->adj;
        if(adj == NULL)
            return;

        if(reenterent)
            return;
        else
            reenterent = TRUE;

        /* Increment adjustment position. */
        if(adj->value < adj->upper)
            adj->value += adj->step_increment;
        if(adj->value > adj->upper)
            adj->value = adj->upper;

        EDVDevicesEditIconUpdateDisplay(ei, (gint)adj->value);

        reenterent = FALSE;
}


/*
 *      Browse for program browse button "clicked" signal callback.
 */
static void EDVDevicesEditBrowseCB(gpointer button, gpointer data)
{
        gbool status;
        GtkWidget *toplevel;
        fb_type_struct **ftype = NULL, *ftype_rtn = NULL;
        gint total_ftypes = 0;
        gchar **path_rtn = NULL;
        gint total_path_rtns = 0;
        GtkEntry *entry = (GtkEntry *)data;
        if(entry == NULL)
            return;


        if(FileBrowserIsQuery())
            return;

        toplevel = gtk_widget_get_toplevel(GTK_WIDGET(entry));

        FileBrowserTypeListNew(
            &ftype, &total_ftypes,
            "*.*", "All files"
        );

        /* Query user for program path. */
        FileBrowserSetTransientFor(toplevel);
        status = FileBrowserGetResponse(
            "Select Program",
            "Select", "Cancel",
            gtk_entry_get_text(entry),          /* Startup path. */
            ftype, total_ftypes,
            &path_rtn, &total_path_rtns,
            &ftype_rtn
        );
        FileBrowserSetTransientFor(NULL);

        /* Got user response? */
        if(status)
        {
            const gchar *new_path = (total_path_rtns > 0) ?
                path_rtn[0] : NULL;
            if((new_path != NULL) ? (*new_path != '\0') : FALSE)
                gtk_entry_set_text(entry, new_path);
        }

        /* Deallocate file types list. */
        FileBrowserDeleteTypeList(ftype, total_ftypes);
}


/*
 *	Devices edit window "delete_event" signal callback.
 */
static gint EDVDevicesEditWinDeleteEventCB(
	GtkWidget *widget, GdkEvent *event, gpointer data
)
{
	edv_device_editwin_struct *ew = (edv_device_editwin_struct *)data;
	if(ew == NULL)
	    return(TRUE);

	EDVDevicesEditWinCancelCB(NULL, ew);

	return(TRUE);
}


/*
 *      Devices edit window OK button callback.
 */
static void EDVDevicesEditWinOKCB(
        GtkWidget *widget, gpointer data
)
{
        static gbool reenterent = FALSE;
        edv_device_editwin_struct *ew = (edv_device_editwin_struct *)data;
        if(ew == NULL)
            return;

        if(reenterent)
            return;
        else
            reenterent = TRUE;

	EDVDevicesEditWinApplyCB(NULL, ew);
        EDVDevicesEditWinUnmap(ew);

        reenterent = FALSE;
}

/*
 *      Devices edit window Apply button callback.
 */
static void EDVDevicesEditWinApplyCB(
        GtkWidget *widget, gpointer data
)
{
        static gbool reenterent = FALSE;
	GtkWidget *w;
	GtkEntry *entry;
	GtkCombo *combo;
	edv_core_struct *core_ptr;
	gint dev_num;
	edv_device_struct *dev_ptr;
	edv_device_listwin_struct *lw;
	edv_device_editwin_icon_struct *ei;
        edv_device_editwin_struct *ew = (edv_device_editwin_struct *)data;
        if(ew == NULL)
            return;

        core_ptr = (edv_core_struct *)ew->core_ptr;
        if(core_ptr == NULL)
            return;

	lw = (edv_device_listwin_struct *)ew->listwin;
	if(lw == NULL)
	    return;

        if(reenterent)
            return;
        else
            reenterent = TRUE;

	/* Get device index and pointer to structure on the core's
	 * list.
	 */
	dev_num = ew->dev_num;
	if((dev_num >= 0) && (dev_num < core_ptr->total_devices))
	    dev_ptr = core_ptr->device[dev_num];
	else
	    dev_ptr = NULL;
	if(dev_ptr == NULL)
	{
	    CDialogSetTransientFor(ew->toplevel);
	    CDialogGetResponse(
"Apply Failed",
"This device no longer exists, these values will be discarded.",
		NULL,
		CDIALOG_ICON_ERROR,
		CDIALOG_BTNFLAG_OK,
		CDIALOG_BTNFLAG_OK
	    );
	    CDialogSetTransientFor(NULL);

	    reenterent = FALSE;
	    return;
	}


	/* Begin applying values. */

        /* Descriptive name. */
        entry = (GtkEntry *)ew->name_entry;
        if(entry != NULL)
        {
            const gchar *cstrptr = gtk_entry_get_text(entry);
            if(cstrptr != NULL)
            {
                g_free(dev_ptr->name);
                dev_ptr->name = g_strdup(cstrptr);
            }
        }

	/* Device path. */
	entry = (GtkEntry *)ew->device_path_entry;
	if(entry != NULL)
	{
	    const gchar *cstrptr = gtk_entry_get_text(entry);
	    if(cstrptr != NULL)
	    {
		g_free(dev_ptr->device_path);
		dev_ptr->device_path = g_strdup(cstrptr);
	    }
	}

        /* Mount path. */
        entry = (GtkEntry *)ew->mount_path_entry;
        if(entry != NULL)
        {
            const gchar *cstrptr = gtk_entry_get_text(entry);
            if(cstrptr != NULL)
            {
                g_free(dev_ptr->mount_path);
                dev_ptr->mount_path = g_strdup(cstrptr);
            }
        }

	/* File system type. */
	combo = (GtkCombo *)ew->fs_type_combo;
	if(combo != NULL)
	{
	    entry = (GtkEntry *)combo->entry;
            dev_ptr->fs_type = EDVDeviceGetFSFromString(
                gtk_entry_get_text(entry)
            );
	}

        /* Mount command. */
        entry = (GtkEntry *)ew->command_mount_entry;
        if(entry != NULL)
        {
            const gchar *cstrptr = gtk_entry_get_text(entry);
            if(cstrptr != NULL)
            {
                g_free(dev_ptr->command_mount);
                dev_ptr->command_mount = g_strdup(cstrptr);
            }
        }
        /* Unmount command. */
        entry = (GtkEntry *)ew->command_unmount_entry;
        if(entry != NULL)
        {
            const gchar *cstrptr = gtk_entry_get_text(entry);
            if(cstrptr != NULL)
            {
                g_free(dev_ptr->command_unmount);
                dev_ptr->command_unmount = g_strdup(cstrptr);
            }
        }
        /* Eject command. */
        entry = (GtkEntry *)ew->command_eject_entry;
        if(entry != NULL)
        {
            const gchar *cstrptr = gtk_entry_get_text(entry);
            if(cstrptr != NULL)
            {
                g_free(dev_ptr->command_eject);
                dev_ptr->command_eject = g_strdup(cstrptr);
            }
        }

        /* Check command. */
        entry = (GtkEntry *)ew->command_check_entry;
        if(entry != NULL)
        {
            const gchar *cstrptr = gtk_entry_get_text(entry);
            if(cstrptr != NULL)
            {
                g_free(dev_ptr->command_check);
                dev_ptr->command_check = g_strdup(cstrptr);
            }
        }
        /* Tools command. */
        entry = (GtkEntry *)ew->command_tools_entry;
        if(entry != NULL)
        {
            const gchar *cstrptr = gtk_entry_get_text(entry);
            if(cstrptr != NULL)
            {
                g_free(dev_ptr->command_tools);
                dev_ptr->command_tools = g_strdup(cstrptr);
            }
        }
        /* Format command. */
        entry = (GtkEntry *)ew->command_format_entry;
        if(entry != NULL)
        {
            const gchar *cstrptr = gtk_entry_get_text(entry);
            if(cstrptr != NULL)
            {
                g_free(dev_ptr->command_format);
                dev_ptr->command_format = g_strdup(cstrptr);
            }
        }

        /* No unmount. */
        w = ew->no_unmount_check;
        if((w != NULL) ? GTK_IS_TOGGLE_BUTTON(w) : FALSE)
        {
            dev_ptr->no_unmount = GTK_TOGGLE_BUTTON(w)->active;
        }
        /* Read only. */
        w = ew->read_only_check;
        if((w != NULL) ? GTK_IS_TOGGLE_BUTTON(w) : FALSE)
        {
            dev_ptr->read_only = GTK_TOGGLE_BUTTON(w)->active;
        }
	/* Unlisted. */
	w = ew->unlisted_check;
	if((w != NULL) ? GTK_IS_TOGGLE_BUTTON(w) : FALSE)
        {
            dev_ptr->unlisted = GTK_TOGGLE_BUTTON(w)->active;
        }

	/* Begin applying icons file paths. */

	/* Small icon. */
        ei = ew->icon_small;
        if(ei != NULL)
        {
	    gint i;
	    const gchar *cstrptr;

	    for(i = 0; i < EDV_DEVICE_TOTAL_ICON_STATES; i++)
	    {
		cstrptr = ei->path[i];

		g_free(dev_ptr->small_icon_file[i]);
		dev_ptr->small_icon_file[i] = (cstrptr != NULL) ?
		    g_strdup(cstrptr) : NULL;
	    }
	}

        /* Medium icon. */
        ei = ew->icon_medium;
        if(ei != NULL)
        {
            gint i;
            const gchar *cstrptr;

            for(i = 0; i < EDV_DEVICE_TOTAL_ICON_STATES; i++)
            {
                cstrptr = ei->path[i];

                g_free(dev_ptr->medium_icon_file[i]);
                dev_ptr->medium_icon_file[i] = (cstrptr != NULL) ?
                    g_strdup(cstrptr) : NULL;
            }
        }

        /* Large icon. */
        ei = ew->icon_large;
        if(ei != NULL)
        {
            gint i;
            const gchar *cstrptr;

            for(i = 0; i < EDV_DEVICE_TOTAL_ICON_STATES; i++)
            {
                cstrptr = ei->path[i];

                g_free(dev_ptr->large_icon_file[i]);
                dev_ptr->large_icon_file[i] = (cstrptr != NULL) ?
                    g_strdup(cstrptr) : NULL;
            }
        }


	/* Re-realize the device. */
	EDVDeviceRealize(dev_ptr, TRUE);

        /* Update all device mount states and stats. */
        EDVDevicesListUpdateMountStates(
            core_ptr->device, core_ptr->total_devices
        );
        EDVDevicesListUpdateStats(
            core_ptr->device, core_ptr->total_devices
        );

	/* Send device modified signal to all of endeavour's resources. */
	EDVDeviceModifiedEmit(core_ptr, dev_num, dev_ptr);

        reenterent = FALSE;
}

/*
 *      Devices edit window Cancel button callback.
 */
static void EDVDevicesEditWinCancelCB(
        GtkWidget *widget, gpointer data
)
{
        static gbool reenterent = FALSE;
        edv_device_editwin_struct *ew = (edv_device_editwin_struct *)data;
        if(ew == NULL)
            return;

        if(reenterent)
            return;
        else
            reenterent = TRUE;

	EDVDevicesEditWinUnmap(ew);

        reenterent = FALSE;
}

/*
 *	Loads the device values from the structure specified by the index
 *	dev_num on the core structure on the given edit window.
 */
void EDVDevicesEditWinFetchValues(
        edv_device_editwin_struct *ew, gint dev_num
)
{
	GtkWidget *w;
	GtkEntry *entry;
	GtkCombo *combo;
	edv_device_struct *dev_ptr;
	edv_core_struct *core_ptr;
	edv_device_editwin_icon_struct *ei;


	if(ew == NULL)
	    return;

	core_ptr = (edv_core_struct *)ew->core_ptr;
	if(core_ptr == NULL)
	    return;

	/* Get pointer to device structure on core's devices list. */
	if((dev_num < 0) || (dev_num >= core_ptr->total_devices))
	    return;
	else
	    dev_ptr = core_ptr->device[dev_num];
	if(dev_ptr == NULL)
	    return;


	/* Begin fetching values. */

	/* Set device index on core structure from the given dev_num. */
	ew->dev_num = dev_num;

        /* Descriptive name. */
        entry = (GtkEntry *)ew->name_entry;
        if(entry != NULL)
        {
            gtk_entry_set_text(
                entry,
                (dev_ptr->name != NULL) ? dev_ptr->name : ""
            );
        }

        /* Device path. */
        entry = (GtkEntry *)ew->device_path_entry;
        if(entry != NULL)
        {
	    gtk_entry_set_text(
		entry,
		(dev_ptr->device_path != NULL) ? dev_ptr->device_path : ""
	    );
	}

        /* Mount path. */
        entry = (GtkEntry *)ew->mount_path_entry;
        if(entry != NULL)
        {
            gtk_entry_set_text(
                entry,
                (dev_ptr->mount_path != NULL) ? dev_ptr->mount_path : ""
            );
        }


	/* File system type. */
	combo = (GtkCombo *)ew->fs_type_combo;
	if(combo != NULL)
	{
	    entry = (GtkEntry *)combo->entry;
	    gtk_entry_set_text(
		entry, EDVDeviceGetFSStringFromNumber(dev_ptr->fs_type)
	    );
	}

        /* Mount command. */
        entry = (GtkEntry *)ew->command_mount_entry;
        if(entry != NULL)
            gtk_entry_set_text(
                entry,
                (dev_ptr->command_mount != NULL) ? dev_ptr->command_mount : ""
            );
        /* Unmount command. */
        entry = (GtkEntry *)ew->command_unmount_entry;
        if(entry != NULL)
            gtk_entry_set_text(
                entry,
                (dev_ptr->command_unmount != NULL) ? dev_ptr->command_unmount : ""
            );
        /* Eject command. */
        entry = (GtkEntry *)ew->command_eject_entry;
        if(entry != NULL)
            gtk_entry_set_text(
                entry,
                (dev_ptr->command_eject != NULL) ? dev_ptr->command_eject : ""
            );

        /* Check command. */
        entry = (GtkEntry *)ew->command_check_entry;
        if(entry != NULL)
            gtk_entry_set_text(
                entry,
                (dev_ptr->command_check != NULL) ? dev_ptr->command_check : ""
            );
        /* Tools command. */
        entry = (GtkEntry *)ew->command_tools_entry;
        if(entry != NULL)
            gtk_entry_set_text(
                entry,
                (dev_ptr->command_tools != NULL) ? dev_ptr->command_tools : ""
            );
        /* Format command. */
        entry = (GtkEntry *)ew->command_format_entry;
        if(entry != NULL)
            gtk_entry_set_text(
                entry,
                (dev_ptr->command_format != NULL) ? dev_ptr->command_format : ""
            );

        /* No unmount. */
        w = ew->no_unmount_check;
        if((w != NULL) ? GTK_IS_TOGGLE_BUTTON(w) : FALSE)
            GTK_TOGGLE_BUTTON(w)->active = dev_ptr->no_unmount;
        /* Read only. */
        w = ew->read_only_check;
        if((w != NULL) ? GTK_IS_TOGGLE_BUTTON(w) : FALSE)
            GTK_TOGGLE_BUTTON(w)->active = dev_ptr->read_only;
        /* Unlisted. */
        w = ew->unlisted_check;
        if((w != NULL) ? GTK_IS_TOGGLE_BUTTON(w) : FALSE)
            GTK_TOGGLE_BUTTON(w)->active = dev_ptr->unlisted;

	/* Begin loading icons. */

#define DO_LOAD_ICON	\
{ \
 for(i = 0; i < EDV_DEVICE_TOTAL_ICON_STATES; i++) \
 { \
  /* Increase ref count on source pixmap because it is about to be \
   * transfered. \
   */ \
  if(pixmap[i] != NULL) \
   gdk_pixmap_ref(pixmap[i]); \
\
  /* Unref old pixmap if any on the edit window icon structure. */ \
  if(ei->pixmap[i] != NULL) \
   gdk_pixmap_unref(ei->pixmap[i]); \
\
  /* Transfer source pixmap to target pixmap. */ \
  ei->pixmap[i] = pixmap[i]; \
\
\
  /* Increase ref count on source bitmap because it is about to be \
   * transfered.\
   */ \
  if(mask[i] != NULL) \
   gdk_bitmap_ref(mask[i]); \
\
  /* Unref old bitmap if any on the edit window icon structure. */ \
  if(ei->mask[i] != NULL) \
   gdk_bitmap_unref(ei->mask[i]); \
\
  /* Transfer source bitmap to target bitmap. */ \
  ei->mask[i] = mask[i]; \
\
\
  /* Deallocate old icon file path and copy new icon file path. */ \
  g_free(ei->path[i]); \
  ei->path[i] = (path[i] != NULL) ? g_strdup(path[i]) : NULL; \
\
 } \
}

	/* Small icons. */
	ei = ew->icon_small;
	if(ei != NULL)
	{
	    gint i;
	    GdkPixmap **pixmap = dev_ptr->small_pixmap;
	    GdkBitmap **mask = dev_ptr->small_mask;
	    gchar **path = dev_ptr->small_icon_file;

	    if(ei->adj != NULL)
		ei->adj->value = 0.0;

	    DO_LOAD_ICON

	    if(ei->adj != NULL)
		EDVDevicesEditIconUpdateDisplay(
		    ei, (gint)ei->adj->value
		);
	}

        /* Medium icons. */
        ei = ew->icon_medium;
        if(ei != NULL)
        {
            gint i;
            GdkPixmap **pixmap = dev_ptr->medium_pixmap;
            GdkBitmap **mask = dev_ptr->medium_mask;
            gchar **path = dev_ptr->medium_icon_file;

            if(ei->adj != NULL)
                ei->adj->value = 0.0;

            DO_LOAD_ICON

            if(ei->adj != NULL)
                EDVDevicesEditIconUpdateDisplay(
                    ei, (gint)ei->adj->value
                );
        }

        /* Large icons. */
        ei = ew->icon_large;
        if(ei != NULL)
        {
            gint i;
            GdkPixmap **pixmap = dev_ptr->large_pixmap;
            GdkBitmap **mask = dev_ptr->large_mask;
            gchar **path = dev_ptr->large_icon_file;

            if(ei->adj != NULL)
                ei->adj->value = 0.0;

            DO_LOAD_ICON

            if(ei->adj != NULL)
                EDVDevicesEditIconUpdateDisplay(
                    ei, (gint)ei->adj->value
                );
        }

#undef DO_LOAD_ICON

	if(ew->toplevel != NULL)
	    gtk_widget_queue_resize(ew->toplevel);
}


/*
 *	Device removed callback.
 */
void EDVDevicesEditWinDeviceRemovedCB(
        edv_device_editwin_struct *ew, gint dev_num
)
{
	if(ew == NULL)
	    return;

	/* If the removed device is the device that this edit window is
	 * displaying values for then this edit window needs to be
	 * unmapped.
	 */
	if(ew->dev_num == dev_num)
	{
	    ew->dev_num = -1;
	    EDVDevicesEditWinUnmap(ew);
	}
}

/*
 *	Creates a new edit window icon structure and parents it to the
 *	given GtkBox parent.
 */
static edv_device_editwin_icon_struct *EDVDevicesEditIconNew(
        gpointer editwin, GtkWidget *parent, const gchar *title,
	gint btn_width, gint btn_height
)
{
	gint bw = 20, bh = 20;
	GtkAdjustment *adj;
	GtkWidget *w, *parent2, *parent3, *parent4;
	edv_device_editwin_icon_struct *ei = (edv_device_editwin_icon_struct *)g_malloc0(
	    sizeof(edv_device_editwin_icon_struct)
	);
	if(ei == NULL)
	    return(ei);

	/* Reset values. */
	ei->editwin = editwin;

	/* Create adjustment for keeping track of displayed icon. */
	ei->adj = adj = (GtkAdjustment *)gtk_adjustment_new(
	    0.0,	/* Value. */
	    0.0,	/* Lower. */
	    (gfloat)EDV_DEVICE_TOTAL_ICON_STATES - 1.0,	/* Upper. */
	    1.0, 1.0,	/* Step and page increments. */
	    1.0		/* Page size. */
	);

	/* Create toplevel. */
	ei->toplevel = w = gtk_vbox_new(FALSE, 0);
	gtk_box_pack_start(GTK_BOX(parent), w, FALSE, FALSE, 0);
	gtk_widget_show(w);
	parent2 = w;

        /* Hbox for title label. */
        w = gtk_hbox_new(FALSE, 0);
        gtk_box_pack_start(GTK_BOX(parent2), w, FALSE, FALSE, 0);
        gtk_widget_show(w);
        parent3 = w;
	/* Label. */
	w = gtk_label_new(title);
        gtk_box_pack_start(GTK_BOX(parent3), w, FALSE, FALSE, 0);
        gtk_widget_show(w);


        /* Hbox for button. */
        w = gtk_hbox_new(TRUE, 0);
        gtk_box_pack_start(GTK_BOX(parent2), w, FALSE, FALSE, 0);
        gtk_widget_show(w);
        parent3 = w;

	/* Create button. */
	ei->button = w = gtk_button_new();
	gtk_widget_set_usize(w, btn_width, btn_height);
        gtk_signal_connect(
            GTK_OBJECT(w), "clicked",
            GTK_SIGNAL_FUNC(EDVDevicesEditIconChangeCB), ei
        );
	gtk_box_pack_start(GTK_BOX(parent3), w, TRUE, FALSE, 0);
        gtk_widget_show(w);
	parent3 = w;
/*
	ei->gtk_pixmap = w = gtk_pixmap_new(NULL, NULL);
	gtk_container_add(GTK_CONTAINER(parent3), w);
	gtk_widget_show(w);
 */


	/* Hbox for buttons and labels. */
	w = gtk_hbox_new(FALSE, 0);
        gtk_box_pack_start(GTK_BOX(parent2), w, FALSE, FALSE, 0);
        gtk_widget_show(w);
        parent3 = w;


	/* Left button. */
        ei->left_btn = w = gtk_button_new();
        gtk_signal_connect(
            GTK_OBJECT(w), "clicked",
            GTK_SIGNAL_FUNC(EDVDevicesEditIconPrevCB), ei
	);
        gtk_widget_set_usize(w, bw, bh);
        gtk_box_pack_start(GTK_BOX(parent3), w, FALSE, FALSE, 0);
        gtk_widget_show(w);
        parent4 = w;
        /* Arrow. */
        w = gtk_arrow_new(GTK_ARROW_LEFT, GTK_SHADOW_OUT);
        gtk_container_add(GTK_CONTAINER(parent4), w);
        gtk_widget_show(w);

	/* Frame for state label. */
	w = gtk_frame_new(NULL);
	gtk_frame_set_shadow_type(GTK_FRAME(w), GTK_SHADOW_OUT);
	gtk_widget_set_usize(w, 80, -1);
	gtk_box_pack_start(GTK_BOX(parent3), w, FALSE, FALSE, 0);
	gtk_widget_show(w);
	parent4 = w;
	/* State label. */
	ei->state_label = w = gtk_label_new("");
	gtk_container_add(GTK_CONTAINER(parent4), w);
        gtk_widget_show(w);

        /* Right button. */
        ei->right_btn = w = gtk_button_new();
        gtk_signal_connect(
            GTK_OBJECT(w), "clicked",
            GTK_SIGNAL_FUNC(EDVDevicesEditIconNextCB), ei
        );
        gtk_widget_set_usize(w, bw, bh);
        gtk_box_pack_start(GTK_BOX(parent3), w, FALSE, FALSE, 0);
        gtk_widget_show(w);
        parent4 = w;
        /* Arrow. */
        w = gtk_arrow_new(GTK_ARROW_RIGHT, GTK_SHADOW_OUT);
        gtk_container_add(GTK_CONTAINER(parent4), w);
        gtk_widget_show(w);


	return(ei);
}

/*
 *	Deallocates all resources of the given edit window icon structure
 *	and deallocates the structure itself.
 */
static void EDVDevicesEditIconDelete(edv_device_editwin_icon_struct *ei)
{
	gint i;
	GtkWidget **w;
	GdkPixmap *pixmap;
	GdkBitmap *mask;


	if(ei == NULL)
	    return;

#define DO_DESTROY_WIDGET       \
{ \
 if((*w) != NULL) \
 { \
  GtkWidget *tmp_w = *w; \
  (*w) = NULL; \
  gtk_widget_destroy(tmp_w); \
 } \
}

	/* Unref all loaded pixmap and mask pairs. */
	for(i = 0; i < EDV_DEVICE_TOTAL_ICON_STATES; i++)
        {
	    pixmap = ei->pixmap[i];
	    ei->pixmap[i] = NULL;
	    if(pixmap != NULL)
		gdk_pixmap_unref(pixmap);

	    mask = ei->mask[i];
            ei->mask[i] = NULL;
            if(mask != NULL)
                gdk_bitmap_unref(mask);
	}

	/* Destroy all widgets. */
	w = &ei->gtk_pixmap;
	DO_DESTROY_WIDGET

	w = &ei->button;
	DO_DESTROY_WIDGET
	w = &ei->left_btn;
	DO_DESTROY_WIDGET
	w = &ei->right_btn;
	DO_DESTROY_WIDGET
        w = &ei->state_label;
	DO_DESTROY_WIDGET
        w = &ei->toplevel;
	DO_DESTROY_WIDGET

	if(ei->adj != NULL)
	{
	    gtk_object_unref(GTK_OBJECT(ei->adj));
	    ei->adj = NULL;
	}

#undef DO_DESTROY_WIDGET

        /* Deallocate all icon paths. */
        for(i = 0; i < EDV_DEVICE_TOTAL_ICON_STATES; i++)
        {
	    g_free(ei->path[i]);
	    ei->path[i] = NULL;
	}

	/* Deallocate structure itself. */
	g_free(ei);
}


/*
 *	Creates a new devices edit window.
 */
edv_device_editwin_struct *EDVDevicesEditWinNew(
        gpointer core_ptr, gpointer listwin
)
{
        gint i;
        const edv_fs_type_struct *fs_type_ptr;
        const edv_fs_type_struct fs_type_list[] = EDV_FS_TYPE_LIST;
        gint /*border_minor = 2, */ border_major = 5;
	gpointer combo_rtn, entry_rtn, browse_rtn;
	GList *glist;
        GdkWindow *window;
        GtkAccelGroup *accelgrp;
        GtkWidget *w, *parent, *parent2, *parent3, *parent4;
	GtkCombo *combo;
        edv_device_editwin_struct *ew = (edv_device_editwin_struct *)g_malloc0(
            sizeof(edv_device_editwin_struct)
        );
        if(ew == NULL)
            return(ew);


        /* Reset values. */
        ew->initialized = TRUE;
        ew->map_state = FALSE;
        ew->core_ptr = core_ptr;
	ew->listwin = listwin;
	ew->dev_num = -1;


        /* Keyboard accelerator group. */
        ew->accelgrp = accelgrp = gtk_accel_group_new();


        /* Toplevel. */
        ew->toplevel = w = gtk_window_new(GTK_WINDOW_DIALOG);
	gtk_widget_set_usize(w, EDITWIN_WIDTH, EDITWIN_HEIGHT);
        gtk_widget_realize(w);
        gtk_window_set_title(GTK_WINDOW(w), EDITWIN_TITLE);
        window = w->window;
        if(window != NULL)
        {
            gdk_window_set_decorations(
                window,
                GDK_DECOR_TITLE | GDK_DECOR_MENU | GDK_DECOR_MINIMIZE
            );
            gdk_window_set_functions(
                window,
                GDK_FUNC_MOVE | GDK_FUNC_MINIMIZE | GDK_FUNC_CLOSE
            );
            GUISetWMIcon(window, (u_int8_t **)icon_properties2_48x48_xpm);
        }
        gtk_signal_connect(
            GTK_OBJECT(w), "delete_event",
            GTK_SIGNAL_FUNC(EDVDevicesEditWinDeleteEventCB), ew
        );
        gtk_container_border_width(GTK_CONTAINER(w), 0);
        gtk_accel_group_attach(accelgrp, GTK_OBJECT(w));
        parent = w;


        /* Main vbox. */
        ew->main_vbox = w = gtk_vbox_new(FALSE, border_major);
	gtk_container_border_width(GTK_CONTAINER(w), border_major);
        gtk_container_add(GTK_CONTAINER(parent), w);
        gtk_widget_show(w);
        parent = w;


	/* Hbox to split things into two columns. */
        w = gtk_hbox_new(FALSE, border_major);
	gtk_box_pack_start(GTK_BOX(parent), w, TRUE, TRUE, 0);
        gtk_widget_show(w);
        parent2 = w;


	/* Left side vbox. */
	w = gtk_vbox_new(FALSE, border_major);
        gtk_box_pack_start(GTK_BOX(parent2), w, TRUE, TRUE, 0);
        gtk_widget_show(w);
        parent3 = w;


        /* Identification frame. */
        w = gtk_frame_new(
#ifdef PROG_LANGUAGE_ENGLISH
            "Identification"
#endif
#ifdef PROG_LANGUAGE_SPANISH
            "Identificacin"
#endif
#ifdef PROG_LANGUAGE_FRENCH
            "Identification"
#endif
	);
        gtk_box_pack_start(GTK_BOX(parent3), w, FALSE, FALSE, 0);
        gtk_widget_show(w);
        parent4 = w;

        w = gtk_vbox_new(FALSE, border_major);
        gtk_container_border_width(GTK_CONTAINER(w), border_major);
        gtk_container_add(GTK_CONTAINER(parent4), w);
        gtk_widget_show(w);
        parent4 = w;

        /* Name entry. */
        w = (GtkWidget *)GUIPromptBar(
            NULL,
#ifdef PROG_LANGUAGE_ENGLISH
            "Name:",
#endif
#ifdef PROG_LANGUAGE_SPANISH
            "El Nombre:",
#endif
#ifdef PROG_LANGUAGE_FRENCH
	    "Nom:",
#endif
	    NULL, &entry_rtn
        );
        ew->name_entry = (GtkWidget *)entry_rtn;
        gtk_box_pack_start(GTK_BOX(parent4), w, FALSE, FALSE, 0);
        EDVEntrySetDND((edv_core_struct *)core_ptr, (GtkWidget *)entry_rtn);
	GUISetWidgetTip(
	    (GtkWidget *)entry_rtn,
#ifdef PROG_LANGUAGE_ENGLISH
"Enter the name of this device (this value will only be used for\
 display purposes)"
#endif
#ifdef PROG_LANGUAGE_SPANISH
"Entre el nombre de este artefacto (este valor hace slo sea usado\
 para propsitos de despliegue)"
#endif
#ifdef PROG_LANGUAGE_FRENCH
"Entrer le nom de cet appareil (cette valeur sera seulement utilise\
 pour les buts d'exposition)"
#endif
	);
        gtk_widget_show(w);

	/* Device path entry. */
	w = (GtkWidget *)GUIPromptBar(
	    NULL,
#ifdef PROG_LANGUAGE_ENGLISH
            "Device Path:",
#endif
#ifdef PROG_LANGUAGE_SPANISH
            "El Sendero Del Artefacto:",
#endif
#ifdef PROG_LANGUAGE_FRENCH
            "Sentier D'Appareil:",
#endif
	    NULL, &entry_rtn
	);
	ew->device_path_entry = (GtkWidget *)entry_rtn;
        gtk_box_pack_start(GTK_BOX(parent4), w, FALSE, FALSE, 0);
        EDVEntrySetDND((edv_core_struct *)core_ptr, (GtkWidget *)entry_rtn);
        GUISetWidgetTip(
            (GtkWidget *)entry_rtn,
#ifdef PROG_LANGUAGE_ENGLISH
"Enter the path to the device object"
#endif
#ifdef PROG_LANGUAGE_SPANISH
"Entre el sendero al objeto del artefacto"
#endif
#ifdef PROG_LANGUAGE_FRENCH
"Le objeto d'al de sendero d'el de Entre del artefacto"
#endif
        );
        gtk_widget_show(w);

        /* Mount path entry. */
        w = (GtkWidget *)GUIPromptBar(
            NULL,
#ifdef PROG_LANGUAGE_ENGLISH
            "Mount Path:",
#endif
#ifdef PROG_LANGUAGE_SPANISH
            "Monte Sendero:",
#endif
#ifdef PROG_LANGUAGE_FRENCH
            "Monter Le Sentier:",
#endif
	    NULL, &entry_rtn
        );
        ew->mount_path_entry = (GtkWidget *)entry_rtn;
        gtk_box_pack_start(GTK_BOX(parent4), w, FALSE, FALSE, 0);
        EDVEntrySetDND((edv_core_struct *)core_ptr, (GtkWidget *)entry_rtn);
        GUISetWidgetTip(
            (GtkWidget *)entry_rtn,
#ifdef PROG_LANGUAGE_ENGLISH
"Enter the path to the directory object that this device is to\
 be mounted on"
#endif
#ifdef PROG_LANGUAGE_SPANISH
"Entre el sendero al objeto de la gua que este artefacto\
 deber ser montado en"
#endif
#ifdef PROG_LANGUAGE_FRENCH
"Entrer le sentier  l'objet d'annuaire que cet appareil va\
 tre mont sur"
#endif
        );
        gtk_widget_show(w);


        /* File system type frame. */
        w = gtk_frame_new(
#ifdef PROG_LANGUAGE_ENGLISH
            "FileSystem:"
#endif
#ifdef PROG_LANGUAGE_SPANISH
            "FileSystem:"
#endif
#ifdef PROG_LANGUAGE_FRENCH
            "FileSystem:"
#endif
	);
        gtk_box_pack_start(GTK_BOX(parent3), w, FALSE, FALSE, 0);
        gtk_widget_show(w);
        parent4 = w;

        w = gtk_vbox_new(FALSE, border_major);
        gtk_container_border_width(GTK_CONTAINER(w), border_major);
        gtk_container_add(GTK_CONTAINER(parent4), w);
        gtk_widget_show(w);
        parent4 = w;

        /* Create filesystem strings glist. */
        glist = NULL;
	for(i = 0; fs_type_list[i].fs_type; i++)
        {
            fs_type_ptr = &fs_type_list[i];
            if(fs_type_ptr->name != NULL)
		glist = g_list_append(glist, g_strdup(fs_type_ptr->name));
        }
        /* Filesystem type combo. */
        w = (GtkWidget *)GUIComboCreate(
#ifdef PROG_LANGUAGE_ENGLISH
            "Type:",
#endif
#ifdef PROG_LANGUAGE_SPANISH
            "El Tipo:",
#endif
#ifdef PROG_LANGUAGE_FRENCH
            "Type:",
#endif
            EDV_FS_TYPE_MSDOS_NAME,
            glist,		/* Initial glist of items. */
            i,			/* Maximum items. */
            &combo_rtn,
            ew,
            NULL,
            NULL
        );
        combo = (GtkCombo *)combo_rtn;
        ew->fs_type_combo = (GtkWidget *)combo_rtn;
        gtk_entry_set_editable(GTK_ENTRY(combo->entry), FALSE);
/*
        gtk_signal_connect(
            GTK_OBJECT(combo->entry), "changed",
            GTK_SIGNAL_FUNC(EDVMimeTypesEditWinClassChangedCB), ew
        );
 */
        gtk_box_pack_start(GTK_BOX(parent4), w, FALSE, FALSE, 0);
        GUISetWidgetTip(
            combo->entry,
#ifdef PROG_LANGUAGE_ENGLISH
"Select the filesystem type of the media used with this device"
#endif
#ifdef PROG_LANGUAGE_SPANISH
"Escoja el de tipo filesystem de los medios usados con este\
 artefacto"
#endif
#ifdef PROG_LANGUAGE_FRENCH
"Choisir le type de filesystem du presse utilis avec cet\
 appareil"
#endif
        );
        gtk_widget_show(w);

        /* Deallocate class strings glist. */
        g_list_foreach(glist, (GFunc)g_free, NULL);
        g_list_free(glist);



        /* Commands frame. */
        w = gtk_frame_new(
#ifdef PROG_LANGUAGE_ENGLISH
            "Commands"
#endif
#ifdef PROG_LANGUAGE_SPANISH
            "Las Ordenes"
#endif
#ifdef PROG_LANGUAGE_FRENCH
            "Ordres"
#endif
	);
        gtk_box_pack_start(GTK_BOX(parent3), w, FALSE, FALSE, 0);
        gtk_widget_show(w);
        parent4 = w;

        w = gtk_vbox_new(FALSE, border_major);
        gtk_container_border_width(GTK_CONTAINER(w), border_major);
        gtk_container_add(GTK_CONTAINER(parent4), w);
        gtk_widget_show(w);
        parent4 = w;

        /* Mount command entry. */
        w = (GtkWidget *)GUIPromptBarWithBrowse(
            NULL,
#ifdef PROG_LANGUAGE_ENGLISH
            "Mount:",
#endif
#ifdef PROG_LANGUAGE_SPANISH
            "El Monte:",
#endif
#ifdef PROG_LANGUAGE_FRENCH
            "Mont:",
#endif
	    NULL, &entry_rtn, &browse_rtn,
	    NULL, NULL
        );
        ew->command_mount_entry = (GtkWidget *)entry_rtn;
        gtk_box_pack_start(GTK_BOX(parent4), w, FALSE, FALSE, 0);
        EDVEntrySetDND((edv_core_struct *)core_ptr, (GtkWidget *)entry_rtn);
        GUISetWidgetTip(
            (GtkWidget *)entry_rtn,
#ifdef PROG_LANGUAGE_ENGLISH
            "Enter the command that will be executed to mount\
 this device (or leave it blank if it is not mountable)"
#endif
#ifdef PROG_LANGUAGE_SPANISH
            "Entre la orden que ser ejecutada al mount este\
 artefacto (o le sale blanco si no es mountable)"
#endif
#ifdef PROG_LANGUAGE_FRENCH
            "Entrer l'ordre qui sera excut  moun cet\
 appareil (ou le part le vide si ne ce pas est mountable)"
#endif
        );
        gtk_widget_show(w);
	w = (GtkWidget *)browse_rtn;
	if(w != NULL)
	    gtk_signal_connect(
		GTK_OBJECT(w), "clicked",
		GTK_SIGNAL_FUNC(EDVDevicesEditBrowseCB), entry_rtn
	    );

        /* Unmount command entry. */
        w = (GtkWidget *)GUIPromptBarWithBrowse(
            NULL,
#ifdef PROG_LANGUAGE_ENGLISH
            "Unmount:",
#endif
#ifdef PROG_LANGUAGE_SPANISH
            "El Unmoute:",
#endif
#ifdef PROG_LANGUAGE_FRENCH
            "Unmont:",
#endif
            NULL, &entry_rtn, &browse_rtn,
            NULL, NULL
        );
        ew->command_unmount_entry = (GtkWidget *)entry_rtn;
        gtk_box_pack_start(GTK_BOX(parent4), w, FALSE, FALSE, 0);
        EDVEntrySetDND((edv_core_struct *)core_ptr, (GtkWidget *)entry_rtn);
        GUISetWidgetTip(
            (GtkWidget *)entry_rtn,
#ifdef PROG_LANGUAGE_ENGLISH
            "Enter the command that will be executed to unmount\
 this device (or leave it blank if it is not unmountable)"
#endif
#ifdef PROG_LANGUAGE_SPANISH
	    "Entre la orden que ser ejecutada al unmount este\
 artefacto (o le sale blanco si no es unmountable)"
#endif
#ifdef PROG_LANGUAGE_FRENCH
            "Entrer l'ordre qui sera excut  unmount cet\
 appareil (ou le part le vide si ne ce pas est unmountable)"
#endif
        );
        gtk_widget_show(w);
        w = (GtkWidget *)browse_rtn;
        if(w != NULL)
            gtk_signal_connect(
                GTK_OBJECT(w), "clicked",
                GTK_SIGNAL_FUNC(EDVDevicesEditBrowseCB), entry_rtn
            );

        /* Eject command entry. */
        w = (GtkWidget *)GUIPromptBarWithBrowse(
            NULL,
#ifdef PROG_LANGUAGE_ENGLISH
            "Eject:",
#endif
#ifdef PROG_LANGUAGE_SPANISH
            "Expulse:",
#endif
#ifdef PROG_LANGUAGE_FRENCH
            "Ejecter:",
#endif
            NULL, &entry_rtn, &browse_rtn,
            NULL, NULL
        );
        ew->command_eject_entry = (GtkWidget *)entry_rtn;
        gtk_box_pack_start(GTK_BOX(parent4), w, FALSE, FALSE, 0);
        EDVEntrySetDND((edv_core_struct *)core_ptr, (GtkWidget *)entry_rtn);
        GUISetWidgetTip(
            (GtkWidget *)entry_rtn,
#ifdef PROG_LANGUAGE_ENGLISH
            "Enter the command that will be executed to eject\
 the media from the device (or leave it blank if the device does not\
 support ejecting)"
#endif
#ifdef PROG_LANGUAGE_SPANISH
            "Entre la orden que ser ejecutada a eject los\
 medios del artefacto (o le sale blanco si el artefacto hace\
 expulsar de apoyo de not)"
#endif
#ifdef PROG_LANGUAGE_FRENCH
	    "Entrer l'ordre qui sera excut  eject le presse\
 de l'appareil (ou le part le vide si l'appareil fait\
 l'jecter de soutien de not)"
#endif
        );
        gtk_widget_show(w);
        w = (GtkWidget *)browse_rtn;
        if(w != NULL)
            gtk_signal_connect(
                GTK_OBJECT(w), "clicked",
                GTK_SIGNAL_FUNC(EDVDevicesEditBrowseCB), entry_rtn
            );


        /* Check command entry. */
        w = (GtkWidget *)GUIPromptBarWithBrowse(
            NULL,
#ifdef PROG_LANGUAGE_ENGLISH
            "Check:",
#endif
#ifdef PROG_LANGUAGE_SPANISH
            "Cheque:",
#endif
#ifdef PROG_LANGUAGE_FRENCH
            "Chque:",
#endif
            NULL, &entry_rtn, &browse_rtn,
            NULL, NULL
        );
        ew->command_check_entry = (GtkWidget *)entry_rtn;
        gtk_box_pack_start(GTK_BOX(parent4), w, FALSE, FALSE, 0);
        EDVEntrySetDND((edv_core_struct *)core_ptr, (GtkWidget *)entry_rtn);
        GUISetWidgetTip(
            (GtkWidget *)entry_rtn,
#ifdef PROG_LANGUAGE_ENGLISH
            "Enter the command that will be executed to run the\
 file system check program for this device (or leave it blank if\
 you do not have one)"
#endif
#ifdef PROG_LANGUAGE_SPANISH
            "Entre la orden que se ejecutar para correr el\
 programa de cheque de sistema de archivo para este artefacto\
 (o le sale blanco si usted no tiene uno)"
#endif
#ifdef PROG_LANGUAGE_FRENCH
            "Entrez l'ordre qui sera excut courir le\
 programme de chque de systme de fichier pour cet appareil\
 (ou le part le vide si vous ne celui avez pas)"
#endif
        );
        gtk_widget_show(w);
        w = (GtkWidget *)browse_rtn;
        if(w != NULL)
            gtk_signal_connect(
                GTK_OBJECT(w), "clicked",
                GTK_SIGNAL_FUNC(EDVDevicesEditBrowseCB), entry_rtn
            );

	/* Tools command entry. */
        w = (GtkWidget *)GUIPromptBarWithBrowse(
            NULL,
#ifdef PROG_LANGUAGE_ENGLISH
            "Tools:",
#endif
#ifdef PROG_LANGUAGE_SPANISH
            "Las Herramientas:",
#endif
#ifdef PROG_LANGUAGE_FRENCH
            "Outils:",
#endif
            NULL, &entry_rtn, &browse_rtn,
            NULL, NULL
        );
        ew->command_tools_entry = (GtkWidget *)entry_rtn;
        gtk_box_pack_start(GTK_BOX(parent4), w, FALSE, FALSE, 0);
        EDVEntrySetDND((edv_core_struct *)core_ptr, (GtkWidget *)entry_rtn);
	GUISetWidgetTip(
	    (GtkWidget *)entry_rtn,
#ifdef PROG_LANGUAGE_ENGLISH
	    "Enter the command that will be executed to run the\
 tools program for this device (or leave it blank if you do not\
 have one)"
#endif
#ifdef PROG_LANGUAGE_SPANISH
            "Entre la orden que se ejecutar para correr el\
 programa de herramientas para este artefacto (o le sale\
 blanco si usted no tiene uno)"
#endif
#ifdef PROG_LANGUAGE_FRENCH
            "Entrer l'ordre qui sera excut courir le\
 programme d'outils pour cet appareil (ou le part le vide\
 si vous ne celui avez pas)"
#endif
	);
        gtk_widget_show(w);
        w = (GtkWidget *)browse_rtn;
        if(w != NULL)
            gtk_signal_connect(
                GTK_OBJECT(w), "clicked",
                GTK_SIGNAL_FUNC(EDVDevicesEditBrowseCB), entry_rtn
            );

        /* Format command entry. */
        w = (GtkWidget *)GUIPromptBarWithBrowse(
            NULL,
#ifdef PROG_LANGUAGE_ENGLISH
            "Format:",
#endif
#ifdef PROG_LANGUAGE_SPANISH
            "Formato:",
#endif
#ifdef PROG_LANGUAGE_FRENCH
            "Format:",
#endif
            NULL, &entry_rtn, &browse_rtn,
            NULL, NULL
        );
        ew->command_format_entry = (GtkWidget *)entry_rtn;
        gtk_box_pack_start(GTK_BOX(parent4), w, FALSE, FALSE, 0);
        EDVEntrySetDND((edv_core_struct *)core_ptr, (GtkWidget *)entry_rtn);
        GUISetWidgetTip(
            (GtkWidget *)entry_rtn,
#ifdef PROG_LANGUAGE_ENGLISH
"Enter the command that will be executed to run the\
 format program for this device (or leave it blank if you do not\
 have one)"
#endif
#ifdef PROG_LANGUAGE_SPANISH
"Entre la orden que se ejecutar para correr the el\
 programa de formato para este artefacto (o le sale\
 blanco si usted hace not tiene uno)"
#endif
#ifdef PROG_LANGUAGE_FRENCH
"Entrer l'ordre qui sera excut courir le programme\
 de format pour cet appareil (ou le part le vide si\
 vous ne celui avez pas)"
#endif
        );
        gtk_widget_show(w);
        w = (GtkWidget *)browse_rtn;
        if(w != NULL)
            gtk_signal_connect(
                GTK_OBJECT(w), "clicked",
                GTK_SIGNAL_FUNC(EDVDevicesEditBrowseCB), entry_rtn
            );


        /* Options frame. */
        w = gtk_frame_new(
#ifdef PROG_LANGUAGE_ENGLISH
            "Options"
#endif
#ifdef PROG_LANGUAGE_SPANISH
            "Las Opciones"
#endif
#ifdef PROG_LANGUAGE_FRENCH
            "Options"
#endif
	);
        gtk_box_pack_start(GTK_BOX(parent3), w, FALSE, FALSE, 0);
        gtk_widget_show(w);
        parent4 = w;

        w = gtk_hbox_new(FALSE, border_major);
        gtk_container_border_width(GTK_CONTAINER(w), border_major);
        gtk_container_add(GTK_CONTAINER(parent4), w);
        gtk_widget_show(w);
        parent4 = w;

        /* No unmount check. */
        ew->no_unmount_check = w = gtk_check_button_new_with_label(
#ifdef PROG_LANGUAGE_ENGLISH
            "No Unmount"
#endif
#ifdef PROG_LANGUAGE_SPANISH
            "Ningn Unmount"
#endif
#ifdef PROG_LANGUAGE_FRENCH
            "Aucun Unmount"
#endif
        );
        gtk_box_pack_start(GTK_BOX(parent4), w, FALSE, FALSE, 0);
        GUISetWidgetTip(
            w,
"Check this to locally mark the device as \"fixed\", so that it cannot\
 be mounted or unmounted"
        );
        gtk_widget_show(w);

        /* Read only check. */
        ew->read_only_check = w = gtk_check_button_new_with_label(
#ifdef PROG_LANGUAGE_ENGLISH
            "Read Only"
#endif
#ifdef PROG_LANGUAGE_SPANISH
            "Lea Slo"
#endif
#ifdef PROG_LANGUAGE_FRENCH
            "Lire Seulement"
#endif
        );
        gtk_box_pack_start(GTK_BOX(parent4), w, FALSE, FALSE, 0);
	GUISetWidgetTip(
	    w,
"Check this to always mount the device as read only"
	);
        gtk_widget_show(w);

        /* Unlisted check. */
        ew->unlisted_check = w = gtk_check_button_new_with_label(
#ifdef PROG_LANGUAGE_ENGLISH
            "Unlisted"
#endif
#ifdef PROG_LANGUAGE_SPANISH
            "Unlisted"
#endif
#ifdef PROG_LANGUAGE_FRENCH
            "Unlisted"
#endif
        );
        gtk_box_pack_start(GTK_BOX(parent4), w, FALSE, FALSE, 0);
        GUISetWidgetTip(
            w,
"Check this if you do not want this device to be listed on any\
 of the \"brief\" device listings (such as the devices popup list)"
        );
        gtk_widget_show(w);



        /* Right side vbox. */
        w = gtk_vbox_new(FALSE, border_major);
        gtk_box_pack_start(GTK_BOX(parent2), w, FALSE, FALSE, 0);
        gtk_widget_show(w);
        parent3 = w;

        /* Icons frame. */
        w = gtk_frame_new(
#ifdef PROG_LANGUAGE_ENGLISH
	    "Icons"
#endif
#ifdef PROG_LANGUAGE_SPANISH
            "Icons"
#endif
#ifdef PROG_LANGUAGE_FRENCH
            "Icnes"
#endif
	);
        gtk_box_pack_start(GTK_BOX(parent3), w, TRUE, TRUE, 0);
        gtk_widget_show(w);
        parent4 = w;

        w = gtk_vbox_new(FALSE, border_major);
        gtk_container_border_width(GTK_CONTAINER(w), border_major);
        gtk_container_add(GTK_CONTAINER(parent4), w);
        gtk_widget_show(w);
        parent4 = w;


	/* Create icon structures. */
	ew->icon_large = EDVDevicesEditIconNew(
	    ew, parent4,
#ifdef PROG_LANGUAGE_ENGLISH
            "Large 48x48:",
#endif
#ifdef PROG_LANGUAGE_SPANISH
            "Grande 48x48:",
#endif
#ifdef PROG_LANGUAGE_FRENCH
            "Grand 48x48:",
#endif
	    54, 54
	);
        ew->icon_medium = EDVDevicesEditIconNew(
            ew, parent4,
#ifdef PROG_LANGUAGE_ENGLISH
            "Medium 32x32:",
#endif
#ifdef PROG_LANGUAGE_SPANISH
            "Medio 32x32:",
#endif
#ifdef PROG_LANGUAGE_FRENCH
            "Moyen 32x32:",
#endif
	    38, 38
        );
        ew->icon_small = EDVDevicesEditIconNew(
            ew, parent4,
#ifdef PROG_LANGUAGE_ENGLISH
            "Small 20x20:",
#endif
#ifdef PROG_LANGUAGE_SPANISH
            "Pequeo 20x20:",
#endif
#ifdef PROG_LANGUAGE_FRENCH
            "Petit 20x20:",
#endif
	    26, 26
        );


        /* Horizontal separator. */
        w = gtk_hseparator_new();
        gtk_box_pack_start(GTK_BOX(parent), w, FALSE, FALSE, 0);
        gtk_widget_show(w);


	/* Hbox for buttons. */
	w = gtk_hbox_new(TRUE, 0);
        gtk_box_pack_start(GTK_BOX(parent), w, FALSE, FALSE, 0);
        gtk_widget_show(w);
	parent2 = w;

        /* OK button. */
        ew->ok_btn = w = GUIButtonPixmapLabelH(
            (u_int8_t **)icon_ok_20x20_xpm, "OK", NULL
        );
        gtk_widget_set_usize(w, EDITWIN_BTN_WIDTH, EDITWIN_BTN_HEIGHT);
        GTK_WIDGET_SET_FLAGS(w, GTK_CAN_DEFAULT);
        gtk_box_pack_start(GTK_BOX(parent2), w, TRUE, FALSE, 0);
        gtk_signal_connect(
            GTK_OBJECT(w), "clicked",
            GTK_SIGNAL_FUNC(EDVDevicesEditWinOKCB), ew
        );
        gtk_widget_show(w);

        /* Apply button. */
        ew->apply_btn = w = GUIButtonPixmapLabelH(
            (u_int8_t **)icon_select_20x20_xpm,
#ifdef PROG_LANGUAGE_ENGLISH
            "Apply",
#endif
#ifdef PROG_LANGUAGE_SPANISH
            "Aplique",
#endif
#ifdef PROG_LANGUAGE_FRENCH
            "S'appliquer",
#endif
	    NULL
        );
        gtk_widget_set_usize(w, EDITWIN_BTN_WIDTH, EDITWIN_BTN_HEIGHT);
        GTK_WIDGET_SET_FLAGS(w, GTK_CAN_DEFAULT);
        gtk_box_pack_start(GTK_BOX(parent2), w, TRUE, FALSE, 0);
        gtk_signal_connect(
            GTK_OBJECT(w), "clicked",
            GTK_SIGNAL_FUNC(EDVDevicesEditWinApplyCB), ew
        );
        gtk_widget_show(w);

        /* Cancel button. */
        ew->cancel_btn = w = GUIButtonPixmapLabelH(
            (u_int8_t **)icon_cancel_20x20_xpm,
#ifdef PROG_LANGUAGE_ENGLISH
            "Cancel",
#endif
#ifdef PROG_LANGUAGE_SPANISH
            "Cancele",
#endif
#ifdef PROG_LANGUAGE_FRENCH
            "Annuler",
#endif
	    NULL
        );
        gtk_widget_set_usize(w, EDITWIN_BTN_WIDTH, EDITWIN_BTN_HEIGHT);
        GTK_WIDGET_SET_FLAGS(w, GTK_CAN_DEFAULT);
        gtk_box_pack_start(GTK_BOX(parent2), w, TRUE, FALSE, 0);
        gtk_signal_connect(
            GTK_OBJECT(w), "clicked",
            GTK_SIGNAL_FUNC(EDVDevicesEditWinCancelCB), ew
        );
        gtk_accel_group_add(
            accelgrp, GDK_Escape, 0, GTK_ACCEL_VISIBLE,
            GTK_OBJECT(w), "clicked"
        );
        gtk_widget_show(w);


	return(ew);
}

/*
 *	Maps the devices edit window.
 */
void EDVDevicesEditWinMap(
        edv_device_editwin_struct *ew
)
{
        GtkWidget *w;

        if(ew == NULL)
            return;

	/* Set OK button as the default on map. */
        w = ew->ok_btn;
        if(w != NULL)
        {
            gtk_widget_grab_focus(w);
            gtk_widget_grab_default(w);
        }

	w = ew->toplevel;
	gtk_widget_show_raise(w);
	ew->map_state = TRUE;
}

/*
 *	Unmaps the device edit window.
 */
void EDVDevicesEditWinUnmap(
        edv_device_editwin_struct *ew
)
{
	GtkWidget *w;

	if(ew == NULL)
	    return;

	w = ew->toplevel;
	if(w == NULL)
	    return;

	if(ew->map_state)
	{
	    gtk_widget_hide(w);
	    ew->map_state = FALSE;
	}
}

/*
 *	Deallocates all resources of the given devices edit window and
 *	deallocates the structure itself.
 */
void EDVDevicesEditWinDelete(
        edv_device_editwin_struct *ew
)
{
        GtkWidget **w;


        if(ew == NULL)
            return;

        if(ew->initialized)
        {
#define DO_DESTROY_WIDGET       \
{ \
 if((*w) != NULL) \
 { \
  GtkWidget *tmp_w = *w; \
  (*w) = NULL; \
  gtk_widget_destroy(tmp_w); \
 } \
}

	    /* Begin destroying widgets. */

	    EDVDevicesEditIconDelete(ew->icon_small);
	    ew->icon_small = NULL;
            EDVDevicesEditIconDelete(ew->icon_medium);
	    ew->icon_medium = NULL;
            EDVDevicesEditIconDelete(ew->icon_large);
	    ew->icon_large = NULL;

	    w = &ew->ok_btn;
	    DO_DESTROY_WIDGET
            w = &ew->apply_btn;
            DO_DESTROY_WIDGET
	    w = &ew->cancel_btn;
	    DO_DESTROY_WIDGET

            w = &ew->name_entry;
            DO_DESTROY_WIDGET
	    w = &ew->device_path_entry;
	    DO_DESTROY_WIDGET
            w = &ew->mount_path_entry;
            DO_DESTROY_WIDGET
            w = &ew->fs_type_combo;
            DO_DESTROY_WIDGET
            w = &ew->command_mount_entry;
            DO_DESTROY_WIDGET
            w = &ew->command_unmount_entry;
            DO_DESTROY_WIDGET
	    w = &ew->command_eject_entry;
            DO_DESTROY_WIDGET
            w = &ew->command_check_entry;
            DO_DESTROY_WIDGET
	    w = &ew->command_tools_entry;
	    DO_DESTROY_WIDGET
            w = &ew->command_format_entry;
            DO_DESTROY_WIDGET
            w = &ew->no_unmount_check;
            DO_DESTROY_WIDGET
            w = &ew->read_only_check;
            DO_DESTROY_WIDGET
            w = &ew->unlisted_check;
            DO_DESTROY_WIDGET

            w = &ew->toplevel;
            DO_DESTROY_WIDGET

            if(ew->accelgrp != NULL)
            {
                gtk_accel_group_unref(ew->accelgrp);
                ew->accelgrp = NULL;
            }

#undef DO_DESTROY_WIDGET
        }

        /* Deallocate structure itself. */
	g_free(ew);
}
