#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <time.h>
#include <sys/time.h>
#include <sys/types.h>
#include <signal.h>
#include <unistd.h>
#include <gtk/gtk.h>

#include "../../include/string.h"
#include "../../include/fio.h"
#include "../../include/disk.h"
#include "../../include/prochandle.h"

#include "../guiutils.h"
#include "../cdialog.h"
#include "../pdialog.h"
#include "../fb.h"
#include "../progressdialog.h"

#include "../edvtypes.h"
#include "../edvcfglist.h"
#include "../lib/endeavour2.h"

#include "downloadcfgio.h"
#include "config.h"

#include "../images/icon_planet_32x32.xpm"
#include "../images/icon_ftp_48x48.xpm"
#include "../images/pdi_file01_20x20.xpm"
#include "../images/pdi_file02_20x20.xpm"
#include "../images/pdi_file03_20x20.xpm"
#include "../images/pdi_file04_20x20.xpm"
#include "../images/pdi_file05_20x20.xpm"
#include "../images/pdi_file06_20x20.xpm"
#include "../images/pdi_folder_32x32.xpm"
#include "../images/pdi_folderfile_32x32.xpm"

#define DOWNLOAD_PROG	"/usr/bin/wget"


static glong CurrentMillitime(void);

static void FSCKManagerSignalCB(int s);
static gchar *DownloadBrowseTargetPathCB(
	gpointer d, gpointer data, gint prompt_num
);

static gulong DownloadTryGetTotalLength(FILE *fp);
static gint DownloadMonitor(
	edv_context_struct *ctx, gint p,
	const gchar *src, const gchar *tar,
	const gchar *stdout_file
);


/*
 *      Returns the current time in milliseconds.
 */
static glong CurrentMillitime(void)
{
#ifdef __MSW__
        SYSTEMTIME t;   /* Current time of day structure. */

        GetSystemTime(&t);
        return(
            (glong)(
                (((((t.wHour * 60.0) + t.wMinute) * 60) + t.wSecond) * 1000) +
                t.wMilliseconds
            )
        );
#else
        struct timeval tv[1];

        if(gettimeofday(tv, NULL) < 0)
            return(-1);

        return(((tv->tv_sec % 86400) * 1000) + (tv->tv_usec / 1000));
#endif
}

/*
 *      UNIX signal callback.
 */
static void FSCKManagerSignalCB(int s)
{
        switch(s)
        {
          case SIGINT:
          case SIGTERM:
          case SIGSEGV:
            exit(1);
            break;
        }
}

/*
 *	Prompt dialog browse target path callback.
 */
static gchar *DownloadBrowseTargetPathCB(
        gpointer d, gpointer data, gint prompt_num
)
{
        gboolean status;
        fb_type_struct **ftype = NULL, *ftype_rtn = NULL;
        gint total_ftypes = 0;
        gchar **path_rtn = NULL;
        gint total_path_rtns = 0;
        GtkWidget *toplevel = NULL;

        if(FileBrowserIsQuery())
            return(NULL);

        /* Create file types list. */
        FileBrowserTypeListNew(
            &ftype, &total_ftypes,
            "*.*", "All files"
        );

        /* Query user for path. */
        FileBrowserSetTransientFor(toplevel);
        status = FileBrowserGetResponse(
#ifdef PROG_LANGUAGE_ENGLISH
            "Select Path",
            "Select", "Cancel",
#endif
#ifdef PROG_LANGUAGE_SPANISH
            "El Sendero Selecto",
            "Selecto", "Cancele",
#endif
#ifdef PROG_LANGUAGE_FRENCH
            "Choisir Le Sentier",
            "Choisir", "Annuler",
#endif
	    PDialogGetPromptValue(prompt_num),
            ftype, total_ftypes,
            &path_rtn, &total_path_rtns,
            &ftype_rtn
        );
        FileBrowserSetTransientFor(NULL);

        /* Deallocate file types list. */
        FileBrowserDeleteTypeList(ftype, total_ftypes);

	return((total_path_rtns > 0) ? path_rtn[0] : NULL);
}

/*
 *	Reads the first 1024 bytes (or less) of the file specified by
 *	fp, searching for the total length header (prefixed by the
 *	string "\nLength:".
 *
 *	This function may return (gulong)-1, indicating that a permanent
 *	error has occured and not to call it anymore.  It normally
 *	returns the total length (or 0 if it was not found).
 */
static gulong DownloadTryGetTotalLength(FILE *fp)
{
	gulong total_length = 0;
	gchar *buf;
	gint buf_len, bytes_read;
	struct stat stat_buf;


	if(fp == NULL)
	    return((gulong)-1);

	if(fstat(fileno(fp), &stat_buf))
	    return((gulong)-1);

	buf_len = stat_buf.st_size;
	if(buf_len <= 0)
	    return((gulong)-1);

	/* Allocate read buffer. */
	buf = (gchar *)g_malloc((buf_len + 1) * sizeof(gchar));
	if(buf == NULL)
	    return((gulong)-1);

	/* Read the beginning of the file, searching for the content
	 * length "\nLength:" string.
	 */
	rewind(fp);
	bytes_read = fread(buf, sizeof(gchar), buf_len, fp);
	if(bytes_read > 0)
	{
	    gchar *strptr, num_str[80];


	    /* Null terminate buffer. */
	    if(bytes_read >= buf_len)
		buf[buf_len - 1] = '\0';
	    else
		buf[bytes_read] = '\0';

	    /* Look for the content length string and that we got the
	     * entire line.
	     */
	    strptr = strstr(buf, "\nLength:");
	    if((strptr != NULL) ?
		(strchr(strptr + 1, '\n') != NULL) : FALSE
	    )
	    {
		/* Found string, now seek past it to the next space
		 * space separated string.
		 */
		while(!ISBLANK(*strptr) && (*strptr != '\0'))
		    strptr++;
		while(ISBLANK(*strptr))
		    strptr++;

		/* Now strptr is positioned at the string we want,
		 * it is assumed to be a number so copy it to num_str.
		 */
		strncpy(num_str, strptr, 80);
		num_str[80 - 1] = '\0';
		/* Null terminate number string. */
		strptr = num_str;
		while(!ISBLANK(*strptr) && (*strptr != '\0'))
                    strptr++;
		*strptr = '\0';

		/* Now we need to get rid of the ',' characters (if any)
		 * in the number string.
		 */
		substr(num_str, ",", "");
		total_length = atol(num_str);
		/* Number string was 0 or contained no numbers?
		 * Then set total_length to 1, indicating that although
		 * we got the total length, it was undeciperable
		 * (which can happen if the total length is not given).
		 */
		if(total_length == 0)
		    total_length = (gulong)-1;

	    }	/* Look for the content length string. */
	}

	/* Deallocate read buffer. */
	g_free(buf);

	return(total_length);
#undef buf_len
}


/*
 *	Monitors the download process p.
 *
 *	Returns:
 *
 *	0	Success
 *	-2	Process exited but object not downloaded completely
 *	-3	Object does not exist or unable to download from remote
 *		server (downloaded object will not exist)
 *	-4	User aborted
 */
static gint DownloadMonitor(
	edv_context_struct *ctx, gint p,
	const gchar *src, const gchar *tar,
	const gchar *stdout_file
)
{
	FILE *fp;
	gboolean	notified_object_added = FALSE,
			user_abort = FALSE;
	const gchar *cstrptr;
	gint check_count = 0, notify_count = 0;
        gulong last_length = 0, current_length = 0, total_length = 0;
	glong last_millitime = CurrentMillitime();
	gchar *title, *message, *p1, *p2, *tar_name;
        u_int8_t        **start_icon_data[3],
                        **icon_data[6],
                        **end_icon_data[3];
	struct stat stat_buf;


	/* Get just the name of the target. */
	cstrptr = strrchr(tar, '/');
	if(cstrptr != NULL)
	    tar_name = g_strdup(cstrptr + 1);
	else
	    tar_name = g_strdup(tar);

	/* Set up icon data. */
        start_icon_data[0] = (u_int8_t **)icon_planet_32x32_xpm;
        start_icon_data[1] = (u_int8_t **)icon_planet_32x32_xpm;
        start_icon_data[2] = (u_int8_t **)icon_planet_32x32_xpm;

        icon_data[0] = (u_int8_t **)pdi_file01_20x20_xpm;
        icon_data[1] = (u_int8_t **)pdi_file02_20x20_xpm;
        icon_data[2] = (u_int8_t **)pdi_file03_20x20_xpm;
        icon_data[3] = (u_int8_t **)pdi_file04_20x20_xpm;
        icon_data[4] = (u_int8_t **)pdi_file05_20x20_xpm;
        icon_data[5] = (u_int8_t **)pdi_file06_20x20_xpm;

        end_icon_data[0] = (u_int8_t **)pdi_folder_32x32_xpm;
        end_icon_data[1] = (u_int8_t **)pdi_folder_32x32_xpm;
        end_icon_data[2] = (u_int8_t **)pdi_folderfile_32x32_xpm;

	/* Format title. */
	title = g_strdup_printf(
#ifdef PROG_LANGUAGE_ENGLISH
	    "Downloading %s",
#endif
#ifdef PROG_LANGUAGE_SPANISH
	    "Cargar %s",
#endif
#ifdef PROG_LANGUAGE_SPANISH
	    "Chargement %s",
#endif
	    tar_name
	);

	/* Format source and target paths for message. */
	p1 = EDVCopyShortenPath(
	    src, EDV_DEF_PROGRESS_BAR_PATH_DISPLAY_MAX - 10
	);
	p2 = EDVCopyShortenPath(
	    tar, EDV_DEF_PROGRESS_BAR_PATH_DISPLAY_MAX - 10
	);
	/* Format initial message. */
	message = g_strdup_printf(
#ifdef PROG_LANGUAGE_ENGLISH
"Downloading:\n\
\n\
    %s (connecting...)\n\
\n\
To:\n\
\n\
    %s\n",
#endif
#ifdef PROG_LANGUAGE_SPANISH
"Cargar:\n\
\n\
    %s (conectar...)\n\
\n\
A:\n\
\n\
    %s\n",
#endif
#ifdef PROG_LANGUAGE_FRENCH
"Chargement:\n\
\n\
    %s (connecter...)\n\
\n\
A:\n\
\n\
    %s\n",
#endif
	    p1, p2
	);

	/* Map progress dialog. */
	ProgressDialogSetWMIconData(
	    (const u_int8_t **)icon_ftp_48x48_xpm
	);
        ProgressDialogMapAnimation(
	    title,
	    message,
#ifdef PROG_LANGUAGE_ENGLISH
            "Stop",
#endif
#ifdef PROG_LANGUAGE_SPANISH
	    "Parada",
#endif
#ifdef PROG_LANGUAGE_FRENCH
	    "Arrt",
#endif
            start_icon_data, 3,
            icon_data, 6,
            end_icon_data, 3,
            500,
            10000
        );

	g_free(title);
	title = NULL;
	g_free(message);
	message = NULL;


	/* Open stdout file. */
	fp = FOpen(stdout_file, "rb");

	/* While the download process is running... */
        while(1)
        {
	    /* User aborted? */
            if(ProgressDialogStopCount() > 0)
            {
                kill(p, SIGINT);
		user_abort = TRUE;
                break;
            }

	    /* Time to do some checks (at least every 1 second)? */
	    if(check_count >= 50)
	    {
		/* Check if download process is done. */
		if(!ExecProcessExists(p))
		    break;

		/* Need to get total length? Only call this to try and
		 * get the total length if we have not obtained it yet.
		 * If total_length is non-zero or (gulong)-1 then it
		 * means it has already been obtained or is not
		 * available. In either case we should not try to get
		 * it anymore.
		 */
		if(total_length == 0)
		    total_length = DownloadTryGetTotalLength(fp);

		/* Make sure the target object exists and get its 
		 * current length.
		 */
		if(!stat(tar, &stat_buf))
		{
		    glong current_millitime, delta_millitime;
		    glong time_left_seconds;
		    gulong delta_length, remaining_length;
		    gchar time_left_str[80];

		    /* Update current length and time lapse between
		     * length checks.
		     */
		    last_length = current_length;
		    current_length = stat_buf.st_size;
		    delta_length = current_length - last_length;
		    remaining_length = total_length - current_length;

		    current_millitime = CurrentMillitime();
		    if((current_millitime > last_millitime) &&
		       (last_millitime > 0)
		    )
			delta_millitime = current_millitime - last_millitime;
		    else
			delta_millitime = 0;
		    last_millitime = current_millitime;

		    /* Calculate time left in seconds, if the result
		     * is 0 then it means the transfer has stalled.
		     */
		    if((total_length > 0) && (total_length != (gulong)-1) &&
		       (delta_length > 0) && (remaining_length > 0)
		    )
			time_left_seconds = (glong)(
			    (gfloat)delta_millitime / (gfloat)delta_length *
			    (gfloat)remaining_length / 1000.0f
			);
		    else
			time_left_seconds = 0;

		    /* Format time left string. */
		    if(time_left_seconds > 3600)
			sprintf(
			    time_left_str,
#ifdef PROG_LANGUAGE_ENGLISH
			    "%ld:%.2ld:%.2ld left",
#endif
#ifdef PROG_LANGUAGE_SPANISH
                            "%ld:%.2ld:%.2ld restante",
#endif
#ifdef PROG_LANGUAGE_FRENCH
                            "%ld:%.2ld:%.2ld rester",
#endif

			    (time_left_seconds / 3600),
			    (time_left_seconds / 60) % 60,
			    (time_left_seconds / 1) % 60
			);
		    else if(time_left_seconds > 0)
			sprintf(
                            time_left_str,
#ifdef PROG_LANGUAGE_ENGLISH
                            "%ld:%.2ld left",
#endif
#ifdef PROG_LANGUAGE_SPANISH
                            "%ld:%.2ld restante",
#endif
#ifdef PROG_LANGUAGE_FRENCH
                            "%ld:%.2ld rester",
#endif
                            (time_left_seconds / 60),
                            (time_left_seconds / 1) % 60
                        );
		    else
#ifdef PROG_LANGUAGE_ENGLISH
			strcpy(time_left_str, "stalled");
#endif
#ifdef PROG_LANGUAGE_SPANISH
			strcpy(time_left_str, "atascado");
#endif
#ifdef PROG_LANGUAGE_FRENCH
			strcpy(time_left_str, "cal");
#endif

/*
printf("Time left %ld seconds (%ld left) %ld %ld\n",
 time_left_seconds, remaining_length, current_length, total_length);
 */

		    /* Need to notify about the object being
		     * added? Note that since we stat() it we know
		     * it exists.
		     */
		    if(!notified_object_added)
		    {
			EDVNotifyQueueObjectAdded(ctx, tar);
			EDVContextFlush(ctx);
			notified_object_added = TRUE;
		    }

                    /* Update title string. */
                    g_free(title);
                    if((total_length > 0) && (total_length != (gulong)-1))
                        title = g_strdup_printf(
#ifdef PROG_LANGUAGE_ENGLISH
			    "Downloading %s %.0f%%",
#endif
#ifdef PROG_LANGUAGE_SPANISH
                            "Cargar %s %.0f%%",
#endif
#ifdef PROG_LANGUAGE_FRENCH
                            "Chargement %s %.0f%%",
#endif
			    tar_name,
			    (gfloat)current_length /
				(gfloat)total_length * 100.0f
			);
		    else
			title = g_strdup_printf(
#ifdef PROG_LANGUAGE_ENGLISH
			    "Downloading %s",
#endif
#ifdef PROG_LANGUAGE_SPANISH
                            "Cargar %s",
#endif
#ifdef PROG_LANGUAGE_FRENCH
                            "Chargement %s",
#endif
			    tar_name
			);

		    /* Update message string. */
		    g_free(message);
		    if((total_length > 0) && (total_length != (gulong)-1))
			message = g_strdup_printf(
#ifdef PROG_LANGUAGE_ENGLISH
"Downloading:\n\
\n\
    %s (%ld bytes)\n\
\n\
To:\n\
\n\
    %s (%ld bytes) %s\n",
#endif
#ifdef PROG_LANGUAGE_SPANISH
"Cargar:\n\
\n\
    %s (%ld bytes)\n\
\n\
A:\n\
\n\
    %s (%ld bytes) %s\n",
#endif
#ifdef PROG_LANGUAGE_FRENCH
"Chargement:\n\
\n\
    %s (%ld bytes)\n\
\n\
A:\n\
\n\
    %s (%ld bytes) %s\n",
#endif
			    p1, total_length, p2, current_length,
			    time_left_str
			);
		    else
                        message = g_strdup_printf(
#ifdef PROG_LANGUAGE_ENGLISH
"Downloading:\n\
\n\
    %s (??? bytes)\n\
\n\
To:\n\
\n\
    %s (%ld bytes)\n",
#endif
#ifdef PROG_LANGUAGE_SPANISH
"Cargar:\n\
\n\
    %s (??? bytes)\n\
\n\
A:\n\
\n\
    %s (%ld bytes)\n",
#endif
#ifdef PROG_LANGUAGE_FRENCH
"Chargement:\n\
\n\
    %s (??? bytes)\n\
\n\
A:\n\
\n\
    %s (%ld bytes)\n",
#endif
                            p1, p2, current_length
                        );
		}	/* If total length is known, then get current length. */

		check_count = 0;	/* Reset timmer. */
	    }
	    else
	    {
		/* Not time to check, so just increment check timmer. */
		check_count++;
	    }

            /* Time to do notifies (every 10 seconds or longer)? */
            if(notify_count >= 500)
	    {
		if(notified_object_added)
		{
		    /* Send object modified notify to Endeavour. */
		    EDVNotifyQueueObjectModified(ctx, tar, NULL);
		    EDVContextFlush(ctx);
		}

		notify_count = 0;	/* Reset timmer. */
	    }
	    else
	    {
		/* Not time to notify, so just increment notify timmer. */
                notify_count++;
	    }


	    /* Manage GTK+ events. */
            if(gtk_events_pending() > 0)
                gtk_main_iteration();

	    /* Update progress and set message (if it is not NULL).
	     * If the total length is know then we use a known progress
	     * method, otherwise we use the unknown method.
	     */
	    if((total_length > 0) && (total_length != (gulong)-1))
		ProgressDialogUpdate(
		    title, message, NULL, NULL,
		    (gfloat)current_length / (gfloat)total_length,
		    EDV_DEF_PROGRESS_BAR_TICKS,
		    TRUE
		);
	    else
		ProgressDialogUpdateUnknown(
		    title, message, NULL, NULL,
		    TRUE
		);

	    /* Reset title and message strings. */
            if(title != NULL)
            {
                g_free(title);
                title = NULL;
            }
	    if(message != NULL)
	    {
		g_free(message);
		message = NULL;
	    }

	    usleep(20000);
        }

	/* Need to unmap progress dialog just in case. */
	ProgressDialogBreakQuery(TRUE);

	/* Close standard output file. */
	FClose(fp);

	/* Deallocate progress message. */
	g_free(title);
        g_free(message);
	g_free(p1);
	g_free(p2);
	g_free(tar_name);


	/* Check if the object was downloaded by checking if the
	 * target object exists.  Also make sure that the user did not
	 * abort.
	 */
	if(stat(tar, &stat_buf) && !user_abort)
	{
	    /* Unable to stat destination file, this suggests that the
	     * object does not exist on the remote server or that the
	     * remote server is not responding.
	     */
	    gchar *buf = g_strdup_printf(
#ifdef PROG_LANGUAGE_ENGLISH
"Unable to download:\n\
\n\
    %s\n\
\n\
Object does not exist or remote server is not responding",
#endif
#ifdef PROG_LANGUAGE_SPANISH
"Incapaz de cargar:\n\
\n\
    %s\n\
\n\
Opngase no exista ni camarero remoto no responde",
#endif
#ifdef PROG_LANGUAGE_SPANISH
"Incapable pour charger:\n\
\n\
    %s\n\
\n\
L'objet n'existe pas ou le serveur loign ne rpond pas",
#endif
		src
	    );
            CDialogSetTransientFor(NULL);
            CDialogGetResponse(
#ifdef PROG_LANGUAGE_ENGLISH
		"Download Failed",
		buf,
"The object may not exist on the remote server or\n\
the remote server is not responding.  Please verify\n\
that the URL that reffers to the object is correct\n\
and that the object actually exists on the remote\n\
server.  If the remote server is not responding then\n\
you may want to try again later.",
#endif
#ifdef PROG_LANGUAGE_SPANISH
                "Cargue Fallado",
                buf,
"El objeto no puede existir en el camarero remoto ni\n\
el camarero remoto no responde. Verifique por favor\n\
que el URL ese reffers al objeto es correcto y que el\n\
objeto existe verdaderamente en el remoto camarero.\n\
Si el camarero remoto no responde entonces\N\ usted\n\
puede querer tratar otra vez luego.",
#endif
#ifdef PROG_LANGUAGE_FRENCH
                "Charger A Echou",
                buf,
"L'objet ne peut pas exister sur le serveur loign ou\n\
le serveur loign ne rpond pas. S'il vous plat\n\
vrifier que le URL ce reffers  l'objet est exact\n\
et que l'objet existe en fait sur l'loign le serveur.\n\
Si le serveur loign ne rpond pas alors\n\ vous pouvez\n\
vouloir essayer encore plus tard.",
#endif
		CDIALOG_ICON_WARNING,
		CDIALOG_BTNFLAG_OK | CDIALOG_BTNFLAG_HELP,
		CDIALOG_BTNFLAG_OK
	    );
	    CDialogSetTransientFor(NULL);
	    g_free(buf);
	    return(-3);		/* Target object does not exist. */
	}
	else
	{
	    current_length = stat_buf.st_size;

	    /* Target object exists, suggesting that we downloaded
	     * it successfully (but also that the user may have
	     * aborted). So note that the target object size might be
	     * smaller than the one reported in the header.
	     *
	     * Notify Endeavour one last time about object modified,
	     * but notify it as an object added since 
	     */
	    if(notified_object_added)
		EDVNotifyQueueObjectModified(ctx, tar, NULL);
	    else
		EDVNotifyQueueObjectAdded(ctx, tar);
	    EDVContextFlush(ctx);

	    /* Total length known and does not match the current
	     * length? Also make sure the user did not abort.
	     */
	    if((total_length > 0) && (total_length != (gulong)-1) &&
	       (total_length != current_length) && !user_abort
	    )
	    {
		gchar *buf = g_strdup_printf(
#ifdef PROG_LANGUAGE_ENGLISH
"The downloaded object has a size of %ld bytes, but the\n\
expected size of the object should be %ld bytes.",
#endif
#ifdef PROG_LANGUAGE_SPANISH
"El objeto cargado tiene un tamao de byte de %ld, pero el\n\
el tamao esperado del objeto debe ser los byte de %ld.",
#endif
#ifdef PROG_LANGUAGE_FRENCH
"L'objet charg a une taille d'octets de %ld, mais le\n\
la taille prvue de l'objet doit tre les octets de %ld.",
#endif
		    current_length, total_length
		);

		CDialogSetTransientFor(NULL);
		CDialogGetResponse(
#ifdef PROG_LANGUAGE_ENGLISH
		    "Download Warning",
		    buf,
"The download may have ended prematurly, it is recommended\n\
that you resume the download by downloading the object again.",
#endif
#ifdef PROG_LANGUAGE_SPANISH
                    "Cargue Advertir",
                    buf,
"El carga puede haber finalizado prematuramente, se\n\
recomienda que usted reasume el carga cargando el objeto\n\
otra vez.",
#endif
#ifdef PROG_LANGUAGE_FRENCH
                    "Charger L'Avertissement",
                    buf,
"Le chargement a pu terminer prmaturument, il est\n\
recommand que vous reprenez le chargement en chargeant\n\
l'objet encore.",
#endif
		    CDIALOG_ICON_WARNING,
		    CDIALOG_BTNFLAG_OK | CDIALOG_BTNFLAG_HELP,
		    CDIALOG_BTNFLAG_OK
		);
		CDialogSetTransientFor(NULL);
		g_free(buf);
		return(-2);	/* Target object smaller in size. */
	    }

	    /* We know the object appears to have been downloaded, but
	     * we need to report if it was successful or the user
	     * aborted.
	     */
	    return(user_abort ? -4 : 0);
	}
}

/*
 *	Returns:
 *
 *	0	Success
 *	1	General error (object probably not downloaded)
 *	2	Object not downloaded (object not exist, server problem, etc)
 *	3	Download program that will be used to download the
 *		object does not exist
 *	4	User abort
 */
int main(int argc, char *argv[])
{
	gboolean initialized_gtk = FALSE;
	gboolean	need_confirmation = FALSE,
			beep_when_complete = FALSE,
			open_object = FALSE,
			download_last_object = FALSE;
	gint i, status = 0;
	const gchar *arg_ptr;
	const gchar *download_prog = DOWNLOAD_PROG;
	gchar *source_url = NULL;
	gchar *destination_path = NULL;
	gchar *destination_file = NULL;
	gchar *stdout_file = NULL;
        edv_context_struct *ctx = NULL;

#define DO_SHUTDOWN	\
{ \
 if(ctx != NULL) \
 { \
  EDVContextSync(ctx); \
  EDVContextDelete(ctx); \
  ctx = NULL; \
 } \
\
 g_free(source_url); \
 source_url = NULL; \
 g_free(destination_path); \
 destination_path = NULL; \
 g_free(destination_file); \
 destination_file = NULL; \
\
 if(stdout_file != NULL) \
 { \
  unlink(stdout_file); \
  g_free(stdout_file); \
  stdout_file = NULL; \
 } \
\
 if(initialized_gtk) \
 { \
  /* Shutdown dialogs. */ \
  CDialogShutdown(); \
  PDialogShutdown(); \
  FileBrowserShutdown(); \
  ProgressDialogShutdown(); \
  initialized_gtk = FALSE; \
 } \
}

        /* Set up time zone. */
        tzset();

        /* Set up signal callbacks. */
        signal(SIGINT, FSCKManagerSignalCB);
        signal(SIGTERM, FSCKManagerSignalCB);
        signal(SIGKILL, FSCKManagerSignalCB);
        signal(SIGSEGV, FSCKManagerSignalCB);
        signal(SIGSTOP, FSCKManagerSignalCB);
        signal(SIGCONT, FSCKManagerSignalCB);
        signal(SIGPIPE, FSCKManagerSignalCB);


	/* Parse arguments. */
	for(i = 1; i < argc; i++)
	{
	    arg_ptr = argv[i];
	    if(arg_ptr == NULL)
		continue;

	    /* Help. */
	    if(!strcasecmp(arg_ptr, "--help") ||
	       !strcasecmp(arg_ptr, "-help") ||
               !strcasecmp(arg_ptr, "--h") ||
               !strcasecmp(arg_ptr, "-h") ||
               !strcasecmp(arg_ptr, "-?")
	    )
	    {
		printf("%s", PROG_HELP_MESG);
		status = 0;
		DO_SHUTDOWN
		return(status);
	    }
            /* Version. */
            else if(!strcasecmp(arg_ptr, "--version") ||
                    !strcasecmp(arg_ptr, "-version") ||
                    !strcasecmp(arg_ptr, "--v") ||
                    !strcasecmp(arg_ptr, "-v")
            )
            {
                printf("%s %s\n%s", PROG_NAME, PROG_VERSION, PROG_COPYRIGHT);
		status = 0;
		DO_SHUTDOWN
		return(status);
            }
	    /* Beep when download is complete? */
            else if(!strcasecmp(arg_ptr, "--beep") ||
                    !strcasecmp(arg_ptr, "-beep") ||
                    !strcasecmp(arg_ptr, "--b") ||
                    !strcasecmp(arg_ptr, "-b")
            )
            {
                beep_when_complete = TRUE;
            }
	    /* Need confirmation? */
            else if(!strcasecmp(arg_ptr, "--confirm") ||
                    !strcasecmp(arg_ptr, "-confirm") ||
                    !strcasecmp(arg_ptr, "--c") ||
                    !strcasecmp(arg_ptr, "-c")
            )
            {
		need_confirmation = TRUE;
	    }
            /* Open object after (successful and complete) download? */
            else if(!strcasecmp(arg_ptr, "--open") ||
                    !strcasecmp(arg_ptr, "-open") ||
                    !strcasecmp(arg_ptr, "--o") ||
                    !strcasecmp(arg_ptr, "-o")
            )
            {
                open_object = TRUE;
            }
            /* (Re)download last object? */
            else if(!strcasecmp(arg_ptr, "--last") ||
                    !strcasecmp(arg_ptr, "-last") ||
                    !strcasecmp(arg_ptr, "--l") ||
                    !strcasecmp(arg_ptr, "-l")
            )
            {
                download_last_object = TRUE;
            }

	    /* All else assume source url or destination path. */
	    else
	    {
		if(source_url == NULL)
		    source_url = g_strdup(arg_ptr);
		else if(destination_path == NULL)
		    destination_path = g_strdup(arg_ptr);
	    }
	}


        /* Initialize GTK+ as needed. */
        if(!initialized_gtk)
        {
            if(!gtk_init_check(&argc, &argv))
            {
                fprintf(
                    stderr,
 "This program requires X.\n"
                );
		status = 1;
		DO_SHUTDOWN
                return(status);
            }
            initialized_gtk = TRUE;
        }

        /* Initialize GDK RGB buffers system. */
        gdk_rgb_init();

        /* Initialize dialogs. */
        CDialogInit();
        PDialogInit();
	FileBrowserInit();
        ProgressDialogInit();

        /* Initialize Endeavour context. */
        ctx = EDVContextNew();
        EDVContextLoadConfigurationFile(ctx, NULL);


	/* (Re)download last object? */
	if(download_last_object)
	{
	    /* Read last download configuration file and see if there
	     * was an object recorded, make sure that we at least have
	     * the source URL.
	     */
	    gint download_status = 0;
	    DownloadCFGGetLast(
		ctx,
		&source_url,
		&destination_path,
		&download_status
	    );
	    if(source_url == NULL)
	    {
		CDialogSetTransientFor(NULL);
		CDialogGetResponse(
#ifdef PROG_LANGUAGE_ENGLISH
		    "No Object To Download",
"There is no record of a previously downloaded object\n\
to (re)download.",
		    NULL,
#endif
#ifdef PROG_LANGUAGE_SPANISH
                    "Ningn Objeto A Carge",
"No hay el registro de un previamente cargado se opone",
                    NULL,
#endif
#ifdef PROG_LANGUAGE_FRENCH
                    "Aucun Objet A Charg",
"Il n'y a pas de dossier d'un objet prcdemment charg",
                    NULL,
#endif
		    CDIALOG_ICON_INFO,
		    CDIALOG_BTNFLAG_OK,
		    CDIALOG_BTNFLAG_OK
		);
		CDialogSetTransientFor(NULL);

                status = 2;
                DO_SHUTDOWN
                return(status);
	    }
	}

	/* Use current working directory as destination path if the
	 * destination path is not obtained from the command line.
	 */
	if((destination_path != NULL) ? (*destination_path == '\0') : TRUE)
        {
            gchar cwd[PATH_MAX];
            if(getcwd(cwd, PATH_MAX) != NULL)
            {
                cwd[PATH_MAX - 1] = '\0';
                g_free(destination_path);
                destination_path = g_strdup(cwd);
            }
        }


	/* No source url given? then query user for url. */
	if((source_url != NULL) ? (*source_url == '\0') : TRUE)
	{
	    gchar **strv;
	    gint strc;

	    PDialogDeleteAllPrompts();
	    PDialogAddPrompt(
#ifdef PROG_LANGUAGE_ENGLISH
		NULL, "URL:", NULL
#endif
#ifdef PROG_LANGUAGE_SPANISH
                NULL, "URL:", NULL
#endif
#ifdef PROG_LANGUAGE_FRENCH
                NULL, "URL:", NULL
#endif
	    );
	    PDialogSetPromptTip(
		-1,
#ifdef PROG_LANGUAGE_ENGLISH
"Enter the URL (starting with http:// or ftp://) of the\
 object that you want to download"
#endif
#ifdef PROG_LANGUAGE_SPANISH
"Entre el URL (comenzar con http:// o el ftp://) del objeto eso\
 usted quiere cargar"
#endif
#ifdef PROG_LANGUAGE_FRENCH
"Entrer le URL (le commencer avec http:// ou ftp://) de l'objet\n\
 cela vous voulez charger"
#endif
	    );
            PDialogAddPromptWithBrowse(
#ifdef PROG_LANGUAGE_ENGLISH
                NULL, "To:", destination_path,
#endif
#ifdef PROG_LANGUAGE_SPANISH
                NULL, "A:", destination_path,
#endif
#ifdef PROG_LANGUAGE_FRENCH
                NULL, "A:", destination_path,
#endif
		NULL, DownloadBrowseTargetPathCB
            );
            PDialogSetPromptTip(
                -1,
#ifdef PROG_LANGUAGE_ENGLISH
"Enter the full path of the directory that you want to download the\
 object to or leave it blank to download to the current directory"
#endif
#ifdef PROG_LANGUAGE_SPANISH
"Entre el sendero repleto de la gua que usted quiere para cargar\
 objeto de the a o salir este blanco para usar la gua actual"
#endif
#ifdef PROG_LANGUAGE_FRENCH
"Entrer le sentier plein de l'annuaire que vous voulez charger\
 l'objet de the  ou part ce vide pour utiliser l'annuaire actuel"
#endif
            );
	    PDialogSetSize(450, -1);
	    strv = PDialogGetResponse(
#ifdef PROG_LANGUAGE_ENGLISH
		PROG_NAME,
"Enter the URL (starting with http:// or ftp://) of the object that\n\
you want to download. If you leave the To: field blank then the\n\
object will be downloaded to the current directory.",
		NULL,
#endif
#ifdef PROG_LANGUAGE_SPANISH
                PROG_NAME,
"Entre el URL (comenzar con http:// o el ftp://) del objeto eso\n\
usted quiere cargar. Si usted sale el A: blanco de campo entonces\n\
el gua actual se usar en lugar.",
                NULL,
#endif
#ifdef PROG_LANGUAGE_FRENCH
                PROG_NAME,
"Entrer le URL (le commencer avec http:// ou ftp://) de l'objet\n\
cela vous voulez charger. Si vous partez A: le vide de domaine\n\
alors le l'annuaire actuel sera utilis plutt.",
                NULL,
#endif
		PDIALOG_ICON_PLANET,
		"Download", "Cancel",
		PDIALOG_BTNFLAG_SUBMIT | PDIALOG_BTNFLAG_CANCEL,
		PDIALOG_BTNFLAG_SUBMIT,
		&strc
	    );
	    if((strv != NULL) && (strc > 0))
	    {
		/* Get source URL from first string. */
		const gchar *cstrptr = strv[0];
		if((cstrptr != NULL) ? (*cstrptr != '\0') : FALSE)
		{
		    g_free(source_url);
		    source_url = g_strdup(cstrptr);
		}

		/* Get destination path from second string (if any). */
		if(strc > 1)
		{
		    cstrptr = strv[1];
		    if((cstrptr != NULL) ? (*cstrptr != '\0') : FALSE)
		    {
			g_free(destination_path);
			destination_path = g_strdup(cstrptr);
		    }
		}
	    }

	    /* If no source url was obtained than that implies the user
	     * has canceled.
	     */
	    if((source_url != NULL) ? (*source_url == '\0') : TRUE)
	    {
		status = 4;
		DO_SHUTDOWN
		return(status);
	    }
	}


	/* Use current working directory as destination path if the
	 * destination path is not given.  Note that we have to do this
	 * again since the user may have been queried above and no
	 * destination path was given.
	 */
        if((destination_path != NULL) ? (*destination_path == '\0') : TRUE)
        {
            gchar cwd[PATH_MAX];
            if(getcwd(cwd, PATH_MAX) != NULL)
            {
                cwd[PATH_MAX - 1] = '\0';
		g_free(destination_path);
		destination_path = g_strdup(cwd);
            }
        }


	/* Change working directory to the destination path only if the
	 * destination path is given, otherwise use current directory.
	 */
	if(ISPATHDIR(destination_path))
	{
	    const gchar *cstrptr = strrchr(source_url, '/');
	    if(cstrptr != NULL)
		destination_file = g_strdup_printf(
		    "%s/%s", destination_path, cstrptr + 1
		);
	    else
		destination_file = g_strdup(destination_path);

	    chdir(destination_path);
	}
	else if(destination_path != NULL)
	{
	    /* Destination path is given, but it does not appear to be
	     * a directory, so instead get its parent path and change
	     * to its directory.
	     */
	    const gchar	*cstrptr = strrchr(source_url, '/'),
			*parent = GetParentDir(destination_path);
	    if(parent != NULL)
		chdir(parent);

            if(cstrptr != NULL)
                destination_file = g_strdup_printf(
                    "%s/%s", parent, cstrptr + 1
                );
            else
                destination_file = g_strdup(destination_path);
	}
	else
	{
	    gchar cwd[PATH_MAX];
	    if(getcwd(cwd, PATH_MAX) != NULL)
	    {
		cwd[PATH_MAX - 1] = '\0';
		if(destination_path == NULL)
		    destination_path = g_strdup(cwd);
	    }
	}


	/* Need to make initial confirmation? */
	if(need_confirmation)
	{
	    gint response;
            gchar *buf = g_strdup_printf(
#ifdef PROG_LANGUAGE_ENGLISH
"Download:\n\
\n\
    %s\n\
\n\
To:\n\
\n\
    %s\n",
#endif
#ifdef PROG_LANGUAGE_SPANISH
"Cargue:\n\
\n\
    %s\n\
\n\
A:\n\
\n\
    %s\n",
#endif
#ifdef PROG_LANGUAGE_FRENCH
"Chargement:\n\
\n\
    %s\n\
\n\
A:\n\
\n\
    %s\n",
#endif
		source_url, destination_file
	    );
            CDialogSetTransientFor(NULL);
            response = CDialogGetResponseIconData(
#ifdef PROG_LANGUAGE_ENGLISH
                "Confirm Download",
                buf,
                NULL,
#endif
#ifdef PROG_LANGUAGE_SPANISH
                "Confirme Cargue",
                buf,
                NULL,
#endif
#ifdef PROG_LANGUAGE_FRENCH
                "Confirmer Le Chargement",
                buf,
                NULL,
#endif
                (guint8 **)icon_planet_32x32_xpm,
                CDIALOG_BTNFLAG_YES | CDIALOG_BTNFLAG_NO,
                CDIALOG_BTNFLAG_YES
            );
            CDialogSetTransientFor(NULL);
            g_free(buf);

	    switch(response)
	    {
	      case CDIALOG_RESPONSE_NO:
	      case CDIALOG_RESPONSE_NOT_AVAILABLE:
		status = 4;
		DO_SHUTDOWN
		return(status);
		break;
	    }
	}

	/* Destination file already exists? */
	if(!access(destination_file, F_OK))
	{
	    gint response;
	    gchar *buf = g_strdup_printf(
#ifdef PROG_LANGUAGE_ENGLISH
"Resume download of existing object:\n\
\n\
    %s\n\
\n\
Click on \"Yes\" to resume download.\n\
Click on \"No\" to overwrite local object with remote object.\n\
Click on \"Cancel\" to cancel the download.",
#endif
#ifdef PROG_LANGUAGE_SPANISH
"Reasuma cargue de existente se opone:\n\
\n\
    %s\n\
\n\
El chasquido en el \"Si\" de para reasumir carga.\n\
El chasquido en el \"No\" de para escribir para reemplazar objeto\n\
local con objeto remoto.\n\
El chasquido en el \"Cancela\" de para cancelar el carga.",
#endif
#ifdef PROG_LANGUAGE_FRENCH
"Reprendre le chargement d'objet existant:\n\
\n\
    %s\n\
\n\
Le dclic sur \"Oui\" reprendre le chargement.\n\
Le dclic sur \"Non\" superposer l'objet local avec l'objet loign.\n\
Le dclic sur \"Annule\" pour annuler le chargement.",
#endif
		destination_file
	    );
	    CDialogSetTransientFor(NULL);
	    response = CDialogGetResponseIconData(
#ifdef PROG_LANGUAGE_ENGLISH
		"Confirm Resume",
		buf,
"The object that you are trying to download appears to\n\
already exist locally.  If the existing local object\n\
was only partially downloaded (e.g. due to an interrupted\n\
transfer) you should click on \"Yes\" to resume\n\
downloading it.  If you wish to overwrite the existing\n\
local object with the object you are downloading then\n\
click on \"No\".  If you are not sure what to do or you\n\
do not want to download the object then click on \"Cancel\".",
#endif
#ifdef PROG_LANGUAGE_SPANISH
                "Confirme Reasuma",
                buf,
"El objeto que usted tratan de cargar aparece a\n\
existe ya localmente. Si el objeto local existente\n\
era slo parcialmente cargado (e.g. debido a un interrumpido\n\
la transferencia) usted debe chasquear en \"S\" reasumir\n\
cargarlo. Si usted desea escribir para  reemplazar el\n\
existente objeto local con el objeto que usted cargan\n\
entonces el chasquido en \"No\". Si usted est no seguro lo\n\
que hacer ni usted quiere no cargar el objeto entonces\n\
chasquido en \"Cancela\".",
#endif
#ifdef PROG_LANGUAGE_FRENCH
                "Confirmer Reprendre",
                buf,
"L'objet que vous essayez charger apparat \n\ existe dj\n\
localement. Si l'existant local objet seulement a t charg\n\
(e.g. grce  un interrompu transfert) vous doit cliqueter\n\
sur \"Oui\" reprendre chargement il. Si vous souhaitez\n\
superposer l'existant l'objet local avec l'objet que vous\n\
chargez alors le dclic sur \"No\". Si vous n'tes pas sr ce\n\
que de faire ou vous ne veut pas charger l'objet l'alors\n\
dclic sur \"Annule\".",
#endif
                (guint8 **)icon_planet_32x32_xpm,
                CDIALOG_BTNFLAG_YES | CDIALOG_BTNFLAG_NO |
		CDIALOG_BTNFLAG_CANCEL | CDIALOG_BTNFLAG_HELP,
                CDIALOG_BTNFLAG_YES
            );
            CDialogSetTransientFor(NULL);
            g_free(buf);

	    switch(response)
	    {
	      case CDIALOG_RESPONSE_CANCEL:
	      case CDIALOG_RESPONSE_NOT_AVAILABLE:
		status = 4;
		DO_SHUTDOWN
                return(status);
		break;

	      case CDIALOG_RESPONSE_NO:
		/* Overwrite, so remove the destination file. */
		unlink(destination_file);
		/* Do not notify Endeavour about the removed file. */
		break;
            }
	}

	/* Check if the download program does not exist. */
	if(access(download_prog, F_OK))
	{
	    gchar *buf = g_strdup_printf(
#ifdef PROG_LANGUAGE_ENGLISH
"Unable to find download program:\n\n    %s",
#endif
#ifdef PROG_LANGUAGE_SPANISH
"Incapaz de encontrar carga el programa:\n\n    %s",
#endif
#ifdef PROG_LANGUAGE_FRENCH
"Incapable pour trouver le programme de chargement:\n\n    %s",
#endif
		download_prog
	    );
            CDialogSetTransientFor(NULL);
            CDialogGetResponse(
#ifdef PROG_LANGUAGE_ENGLISH
                "No Download Program",
                buf,
"This program is used to perform the actual download,\n\
it must be installed on your system at the location\n\
specified above before you can download.",
#endif
#ifdef PROG_LANGUAGE_SPANISH
                "No Cargue El Programa",
                buf,
"Este programa se usa para realizar el verdadero carga,\n\
se debe instalar en su sistema en la ubicacin\n\
especificado encima de antes usted puede cargar.",
#endif
#ifdef PROG_LANGUAGE_FRENCH
                "Aucun Programme De Chargement",
                buf,
"Ce programme est utilis pour excuter le chargement\n\
vritable, il doit tre install sur votre systme \n\
l'emplacement spcifi au-dessus d'avant de vous pouvez\n\
charger.",
#endif
                CDIALOG_ICON_ERROR,
                CDIALOG_BTNFLAG_OK | CDIALOG_BTNFLAG_HELP,
                CDIALOG_BTNFLAG_OK
            );
            CDialogSetTransientFor(NULL);
	    g_free(buf);

	    status = 3;
	    DO_SHUTDOWN
	    return(status);
	}

	/* Generate tempory stdout file. */
	stdout_file = EDVTmpName(NULL);


	/* Execute download program and monitor download. */
	if(destination_file != NULL)
	{
	    gchar *cmd = g_strdup_printf(
		"%s \"%s\" --tries=1 --progress=dot -c",
		download_prog, source_url
	    );
	    /* Execute the download command.
	     *
	     * Note that wget writes stdout output to stderr.
	     */
	    gint p = ExecOE(cmd, NULL, stdout_file);
	    if(p <= 0)
	    {
		/* Unable to execute download command. */
                gchar *buf = g_strdup_printf(
#ifdef PROG_LANGUAGE_ENGLISH
"Unable to execute download command:\n\n    %s",
#endif
#ifdef PROG_LANGUAGE_SPANISH
"Incapaz de ejecutar carga la orden:\n\n    %s",
#endif
#ifdef PROG_LANGUAGE_FRENCH
"Incapable pour excuter l'ordre de chargement:\n\n    %s",
#endif
		    cmd
		);
		CDialogSetTransientFor(NULL);
		CDialogGetResponse(
#ifdef PROG_LANGUAGE_ENGLISH
		    "Download Failed",
		    buf,
		    NULL,
#endif
#ifdef PROG_LANGUAGE_SPANISH
                    "Cargue Fallado",
                    buf,
                    NULL,
#endif
#ifdef PROG_LANGUAGE_FRENCH
                    "Charger A Echou",
                    buf,
                    NULL,
#endif
		    CDIALOG_ICON_ERROR,
		    CDIALOG_BTNFLAG_OK,
		    CDIALOG_BTNFLAG_OK
		);
		CDialogSetTransientFor(NULL);
		g_free(buf);
		status = 1;
	    }
	    else
	    {
		/* Download command executed, now monitor the download
	         * and handle the result.
		 */
		gint download_status = DownloadMonitor(
                    ctx, p, source_url, destination_file, stdout_file
                );
		switch(download_status)
		{
		  case 0:	/* Success. */
		    if(beep_when_complete)
			gdk_beep();
		    if(open_object)
			EDVOpen(ctx, destination_file, NULL);
		    break;

		  case -2:	/* Process exited but object not completely downloaded. */
		    status = 2;
                    if(beep_when_complete)
                        gdk_beep();
		    break;

		  case -3:	/* Object not downloaded (does not exist). */
		    status = 2;
		    break;

		  case -4:	/* User abort. */
		    status = 4;
		    break;
		}

		/* Record the last downloaded URL regardless of the
		 * above result.
		 */
		DownloadCFGSetLast(
		    ctx, source_url, destination_path, download_status
		);

	    }
	}	/* Execute download program and monitor download. */


	DO_SHUTDOWN
	return(status);
#undef DO_SHUTDOWN
}
