#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <gtk/gtk.h>

#include "../include/string.h"
#include "../include/disk.h"

#include "cdialog.h"

#include "edvtypes.h"
#include "edvcfg.h"
#include "edvobj.h"
#include "edvrecbin.h"
#include "edvrecbinfio.h"
#include "edvconfirm.h"
#include "endeavour.h"
#include "edvcfglist.h"
#include "config.h"


#include "images/icon_copy_file_32x32.xpm"
#include "images/icon_move_file_32x32.xpm"
#include "images/icon_link2_32x32.xpm"
#include "images/icon_trash_32x32.xpm"
#include "images/icon_archive_add_32x32.xpm"
#include "images/icon_archive_extract_32x32.xpm"
#include "images/icon_recover_32x32.xpm"
#include "images/icon_purge_32x32.xpm"


gint EDVConfirmLink(
        edv_core_struct *core_ptr, GtkWidget *toplevel,
        const gchar *src_path, gint total_objects,
        const gchar *tar_path
);
gint EDVConfirmMove(
        edv_core_struct *core_ptr, GtkWidget *toplevel,
        const gchar *src_path, gint total_objects,
        const gchar *tar_path
);
gint EDVConfirmCopy(
        edv_core_struct *core_ptr, GtkWidget *toplevel,
        const gchar *src_path, gint total_objects,
        const gchar *tar_path
);
gint EDVConfirmDelete(
        edv_core_struct *core_ptr, GtkWidget *toplevel,
        const gchar *src_path, gint total_objects
);

gint EDVConfirmArchiveAdd(
        edv_core_struct *core_ptr, GtkWidget *toplevel,
        const gchar *src_path, gint total_objects,
        const gchar *arch_path
);
gint EDVConfirmArchiveExtract(
        edv_core_struct *core_ptr, GtkWidget *toplevel,
        const gchar *src_path, gint total_objects,
        const gchar *tar_path
);
gint EDVConfirmArchiveDelete(
        edv_core_struct *core_ptr, GtkWidget *toplevel,
        const gchar *src_path, gint total_objects
);

gint EDVConfirmRecover(
        edv_core_struct *core_ptr, GtkWidget *toplevel,
        const gchar *src_path,		/* String containing index number. */
	gint total_objects,
        const gchar *tar_path
);
gint EDVConfirmPurge(
        edv_core_struct *core_ptr, GtkWidget *toplevel,
        const gchar *src_path,		/* String containing index number. */
	gint total_objects
);

gint EDVConfirmDND(
        edv_core_struct *core_ptr,
	gint gdk_action, guint info,
        GtkWidget *toplevel,
        edv_dnd_object_struct **dnd_object, gint total_dnd_objects,
        const gchar *tar_path
);


/*
 *	Confirm link.
 *
 *	Returns one of CDIALOG_RESPONSE_*.
 *	Confirmation will always be CDIALOG_RESPONSE_YES if configuration
 *	specifies no confirmation for this operation.
 */
gint EDVConfirmLink(
        edv_core_struct *core_ptr, GtkWidget *toplevel,
        const gchar *src_path, gint total_objects,
        const gchar *tar_path
)
{
	gint response;
	const gchar *title;
	gchar *buf;


	if(core_ptr == NULL)
	    return(CDIALOG_RESPONSE_NOT_AVAILABLE);

	if(!EDVCFGItemListGetValueI(
	    core_ptr->cfg_list, EDV_CFG_PARM_CONFIRM_LINK
	))
	    return(CDIALOG_RESPONSE_YES);

	/* Set up confirmation title and message. */

#ifdef PROG_LANGUAGE_ENGLISH
	title = "Confirm Link";
#endif
#ifdef PROG_LANGUAGE_SPANISH
        title = "Confirme el Eslabn";
#endif
#ifdef PROG_LANGUAGE_FRENCH
        title = "Confirmer le Lien";
#endif

	if((total_objects == 1) && (src_path != NULL))
	    buf = g_strdup_printf(
#ifdef PROG_LANGUAGE_ENGLISH
"Link:\n\
\n\
    %s\n\
\n\
To:\n\
\n\
    %s\n",
#endif
#ifdef PROG_LANGUAGE_SPANISH
"El eslabn:\n\
\n\
    %s\n\
\n\
A:\n\
\n\
    %s\n",
#endif
#ifdef PROG_LANGUAGE_FRENCH
"Le lien:\n\
\n\
    %s\n\
\n\
A:\n\
\n\
    %s\n",
#endif
		src_path, tar_path
	    );
	else
	    buf = g_strdup_printf(
#ifdef PROG_LANGUAGE_ENGLISH
"Link %i objects to:\n\
\n\
    %s\n",
#endif
#ifdef PROG_LANGUAGE_SPANISH
"El eslabn %i se opone a:\n\
\n\
    %s\n",
#endif
#ifdef PROG_LANGUAGE_FRENCH
"Le lien %i s'oppose :\n\
\n\
    %s\n",
#endif
		total_objects, tar_path
	    );

        CDialogSetTransientFor(toplevel);
        response = CDialogGetResponseIconData(
            title, buf, NULL,
            (guint8 **)icon_link2_32x32_xpm,
            CDIALOG_BTNFLAG_YES | CDIALOG_BTNFLAG_NO,
            CDIALOG_BTNFLAG_YES
        );
        CDialogSetTransientFor(NULL);

	g_free(buf);	/* Deallocate message. */

	return(response);
}

/*
 *      Confirm move.
 *
 *      Returns one of CDIALOG_RESPONSE_*.
 *      Confirmation will always be CDIALOG_RESPONSE_YES if configuration
 *      specifies no confirmation for this operation.
 */
gint EDVConfirmMove(
        edv_core_struct *core_ptr, GtkWidget *toplevel,
        const gchar *src_path, gint total_objects,
        const gchar *tar_path
)
{
        gint response;
        const gchar *title;
        gchar *buf;


        if(core_ptr == NULL)
            return(CDIALOG_RESPONSE_NOT_AVAILABLE);

        if(!EDVCFGItemListGetValueI(
            core_ptr->cfg_list, EDV_CFG_PARM_CONFIRM_MOVE
        ))
            return(CDIALOG_RESPONSE_YES);

        /* Set up confirmation title and message. */

#ifdef PROG_LANGUAGE_ENGLISH
        title = "Confirm Move";
#endif
#ifdef PROG_LANGUAGE_SPANISH
        title = "Confirme Mueva";
#endif
#ifdef PROG_LANGUAGE_FRENCH
        title = "Confirmer le Mouvement";
#endif

        if((total_objects == 1) && (src_path != NULL))
            buf = g_strdup_printf(
#ifdef PROG_LANGUAGE_ENGLISH
"Move:\n\
\n\
    %s\n\
\n\
To:\n\
\n\
    %s\n",
#endif
#ifdef PROG_LANGUAGE_SPANISH
"Mueva:\n\
\n\
    %s\n\
\n\
A:\n\
\n\
    %s\n",
#endif
#ifdef PROG_LANGUAGE_FRENCH
"Le mouvement:\n\
\n\
    %s\n\
\n\
A:\n\
\n\
    %s\n",
#endif
                src_path, tar_path
            );
        else
            buf = g_strdup_printf(
#ifdef PROG_LANGUAGE_ENGLISH
"Move %i objects to:\n\
\n\
    %s\n",
#endif
#ifdef PROG_LANGUAGE_SPANISH
"Mueva %i objetos a:\n\
\n\
    %s\n",
#endif
#ifdef PROG_LANGUAGE_FRENCH
"Le mouvement %i s'oppose :\n\
\n\
    %s\n",
#endif
                total_objects, tar_path
            );

        CDialogSetTransientFor(toplevel);
        response = CDialogGetResponseIconData(
            title, buf, NULL,
            (guint8 **)icon_move_file_32x32_xpm,
            CDIALOG_BTNFLAG_YES | CDIALOG_BTNFLAG_NO,
            CDIALOG_BTNFLAG_YES
        );
        CDialogSetTransientFor(NULL);

        g_free(buf);    /* Deallocate message. */

        return(response);
}

/*
 *      Confirm copy.
 *
 *      Returns one of CDIALOG_RESPONSE_*.
 *      Confirmation will always be CDIALOG_RESPONSE_YES if configuration
 *      specifies no confirmation for this operation.
 */
gint EDVConfirmCopy(
        edv_core_struct *core_ptr, GtkWidget *toplevel,
        const gchar *src_path, gint total_objects,
        const gchar *tar_path
)
{
        gint response;
        const gchar *title;
        gchar *buf;


        if(core_ptr == NULL)
            return(CDIALOG_RESPONSE_NOT_AVAILABLE);

        if(!EDVCFGItemListGetValueI(
            core_ptr->cfg_list, EDV_CFG_PARM_CONFIRM_COPY
        ))
            return(CDIALOG_RESPONSE_YES);

        /* Set up confirmation title and message. */

#ifdef PROG_LANGUAGE_ENGLISH
        title = "Confirm Copy";
#endif
#ifdef PROG_LANGUAGE_SPANISH
        title = "Confirme Copia";
#endif
#ifdef PROG_LANGUAGE_FRENCH
        title = "Confirmer la Copie";
#endif

        if((total_objects == 1) && (src_path != NULL))
            buf = g_strdup_printf(
#ifdef PROG_LANGUAGE_ENGLISH
"Copy:\n\
\n\
    %s\n\
\n\
To:\n\
\n\
    %s\n",
#endif
#ifdef PROG_LANGUAGE_SPANISH
"La copia:\n\
\n\
    %s\n\
\n\
A:\n\
\n\
    %s\n",
#endif
#ifdef PROG_LANGUAGE_FRENCH
"Copie:\n\
\n\
    %s\n\
\n\
A:\n\
\n\
    %s\n",
#endif
                src_path, tar_path
            );
        else
            buf = g_strdup_printf(
#ifdef PROG_LANGUAGE_ENGLISH
"Copy %i objects to:\n\
\n\
    %s\n",
#endif
#ifdef PROG_LANGUAGE_SPANISH
"La copia %i se opone a:\n\
\n\
    %s\n",
#endif
#ifdef PROG_LANGUAGE_FRENCH
"La copie %i s'oppose :\n\
\n\
    %s\n",
#endif
                total_objects, tar_path
            );

        CDialogSetTransientFor(toplevel);
        response = CDialogGetResponseIconData(
            title, buf, NULL,
            (guint8 **)icon_copy_file_32x32_xpm,
            CDIALOG_BTNFLAG_YES | CDIALOG_BTNFLAG_NO,
            CDIALOG_BTNFLAG_YES
        );
        CDialogSetTransientFor(NULL);

        g_free(buf);    /* Deallocate message. */

        return(response);
}

/*
 *      Confirm delete.
 *
 *      Returns one of CDIALOG_RESPONSE_*.
 *      Confirmation will always be CDIALOG_RESPONSE_YES if configuration
 *      specifies no confirmation for this operation.
 */
gint EDVConfirmDelete(
        edv_core_struct *core_ptr, GtkWidget *toplevel,
        const gchar *src_path, gint total_objects
)
{
        gint response;
        const gchar *title;
        gchar *buf;
	gint delete_method;
	guint8 **icon_data = (guint8 **)icon_trash_32x32_xpm;

        if(core_ptr == NULL)
            return(CDIALOG_RESPONSE_NOT_AVAILABLE);

        if(!EDVCFGItemListGetValueI(
            core_ptr->cfg_list, EDV_CFG_PARM_CONFIRM_DELETE
        ))
            return(CDIALOG_RESPONSE_YES);

	delete_method = EDVCFGItemListGetValueI(
            core_ptr->cfg_list, EDV_CFG_PARM_DELETE_METHOD
        );


        /* Set up confirmation title and message. */

#ifdef PROG_LANGUAGE_ENGLISH
        title = "Confirm Delete";
#endif
#ifdef PROG_LANGUAGE_SPANISH
        title = "Confirme Borre";
#endif
#ifdef PROG_LANGUAGE_FRENCH
        title = "Confirmer Effacer";
#endif

        if((total_objects == 1) && (src_path != NULL))
	{
	    if(ISLPATHDIR(src_path))
		buf = g_strdup_printf(
#ifdef PROG_LANGUAGE_ENGLISH
"Delete directory:\n\
\n\
    %s\n\
\n\
Including any contents within that directory.\n",
#endif
#ifdef PROG_LANGUAGE_SPANISH
"Borre gua:\n\
\n\
    %s\n\
\n\
Inclusive cualquier contenido dentro de esa gua.\n",
#endif
#ifdef PROG_LANGUAGE_FRENCH
"Effacer l'annuaire:\n\
\n\
    %s\n\
\n\
y compris n'importe quels contenus dans cet annuaire.\n",
#endif
		    src_path
		);
	    else
		buf = g_strdup_printf(
#ifdef PROG_LANGUAGE_ENGLISH
"Delete:\n\
\n\
    %s\n",
#endif
#ifdef PROG_LANGUAGE_SPANISH
"Borre:\n\
\n\
    %s\n",
#endif
#ifdef PROG_LANGUAGE_FRENCH
"Effacer:\n\
\n\
    %s\n",
#endif
		    src_path
		);
	}
        else
            buf = g_strdup_printf(
#ifdef PROG_LANGUAGE_ENGLISH
"Delete %i selected objects?\n",
#endif
#ifdef PROG_LANGUAGE_SPANISH
"Borra %i objetos escogidos?\n",
#endif
#ifdef PROG_LANGUAGE_FRENCH
"Efface %i des objets choisis?\n",
#endif
                total_objects
            );

	/* Append an empty line and one more line indicating that this
	 * is a permanent delete if the delete_method is set to
	 * EDV_DELETE_METHOD_PURGE.
	 */
	if(delete_method == EDV_DELETE_METHOD_PURGE)
	{
	    buf = strcatalloc(
		buf,
#ifdef PROG_LANGUAGE_ENGLISH
"\nYou will not be able to recover the object(s)\n\
once they have been deleted!\n"
#endif
#ifdef PROG_LANGUAGE_SPANISH
"\nUsted no ser capaz de recuperar el object(s)\n\
una vez ellos han sido borrados!\n"
#endif
#ifdef PROG_LANGUAGE_FRENCH
"\nVous ne pourrez pas retrouver le object(s)\n\
une fois ils ont t effacs!\n"
#endif
	    );
	    icon_data = (guint8 **)icon_purge_32x32_xpm;
	}

        CDialogSetTransientFor(toplevel);
        response = CDialogGetResponseIconData(
            title, buf, NULL, icon_data,
            CDIALOG_BTNFLAG_YES | CDIALOG_BTNFLAG_NO,
            CDIALOG_BTNFLAG_YES
        );
        CDialogSetTransientFor(NULL);

        g_free(buf);    /* Deallocate message. */

        return(response);
}

/*
 *      Confirm adding of disk object(s) to an archive.
 *
 *      Returns one of CDIALOG_RESPONSE_*.
 *      Confirmation will always be CDIALOG_RESPONSE_YES if configuration
 *      specifies no confirmation for this operation.
 */
gint EDVConfirmArchiveAdd(
        edv_core_struct *core_ptr, GtkWidget *toplevel,
        const gchar *src_path, gint total_objects,
        const gchar *arch_path
)
{
        gint response;
        const gchar *title;
        gchar *buf;


        if(core_ptr == NULL)
            return(CDIALOG_RESPONSE_NOT_AVAILABLE);

        if(!EDVCFGItemListGetValueI(
            core_ptr->cfg_list, EDV_CFG_PARM_CONFIRM_ARCHIVE_ADD
        ))
            return(CDIALOG_RESPONSE_YES);

        /* Set up confirmation title and message. */

#ifdef PROG_LANGUAGE_ENGLISH
        title = "Confirm Add";
#endif
#ifdef PROG_LANGUAGE_SPANISH
        title = "Confirme Agregue";
#endif
#ifdef PROG_LANGUAGE_FRENCH
        title = "Confirmer Ajouter";
#endif

        if((total_objects == 1) && (src_path != NULL))
            buf = g_strdup_printf(
#ifdef PROG_LANGUAGE_ENGLISH
"Add:\n\
\n\
    %s\n\
\n\
To archive:\n\
\n\
    %s\n",
#endif
#ifdef PROG_LANGUAGE_SPANISH
"Agregue:\n\
\n\
    %s\n\
\n\
Al Archivo:\n\
\n\
    %s\n",
#endif
#ifdef PROG_LANGUAGE_FRENCH
"Ajouter:\n\
\n\
    %s\n\
\n\
A l'Archive:\n\
\n\
    %s\n",
#endif
                src_path, arch_path
            );
        else
            buf = g_strdup_printf(
#ifdef PROG_LANGUAGE_ENGLISH
"Add %i objects to archive:\n\
\n\
    %s\n",
#endif
#ifdef PROG_LANGUAGE_SPANISH
"Agregue %i objetos al archivo:\n\
\n\
    %s\n",
#endif
#ifdef PROG_LANGUAGE_FRENCH
"Ajouter %i des objets  l'archive:\n\
\n\
    %s\n",
#endif
                total_objects, arch_path
            );

        CDialogSetTransientFor(toplevel);
        response = CDialogGetResponseIconData(
            title, buf, NULL,
            (guint8 **)icon_archive_add_32x32_xpm,
            CDIALOG_BTNFLAG_YES | CDIALOG_BTNFLAG_NO,
            CDIALOG_BTNFLAG_YES
        );
        CDialogSetTransientFor(NULL);

        g_free(buf);    /* Deallocate message. */

        return(response);
}

/*
 *      Confirm extracting of archive object(s) to from an archive
 *	to the target disk object path.
 *
 *      Returns one of CDIALOG_RESPONSE_*.
 *      Confirmation will always be CDIALOG_RESPONSE_YES if configuration
 *      specifies no confirmation for this operation.
 */
gint EDVConfirmArchiveExtract(
        edv_core_struct *core_ptr, GtkWidget *toplevel,
        const gchar *src_path, gint total_objects,
        const gchar *tar_path
)
{
        gint response;
        const gchar *title;
        gchar *buf;


        if(core_ptr == NULL)
            return(CDIALOG_RESPONSE_NOT_AVAILABLE);

        if(!EDVCFGItemListGetValueI(
            core_ptr->cfg_list, EDV_CFG_PARM_CONFIRM_ARCHIVE_EXTRACT
        ))
            return(CDIALOG_RESPONSE_YES);

        /* Set up confirmation title and message. */

#ifdef PROG_LANGUAGE_ENGLISH
        title = "Confirm Extract";
#endif
#ifdef PROG_LANGUAGE_SPANISH
        title = "Confirme el Extracto";
#endif
#ifdef PROG_LANGUAGE_FRENCH
        title = "Confirmer l'Extrait";
#endif

        if((total_objects == 1) && (src_path != NULL))
            buf = g_strdup_printf(
#ifdef PROG_LANGUAGE_ENGLISH
"Extract:\n\
\n\
    %s\n\
\n\
To:\n\
\n\
    %s\n",
#endif
#ifdef PROG_LANGUAGE_SPANISH
"El extracto:\n\
\n\
    %s\n\
\n\
A:\n\
\n\
    %s\n",
#endif
#ifdef PROG_LANGUAGE_FRENCH
"Extrait:\n\
\n\
    %s\n\
\n\
A:\n\
\n\
    %s\n",
#endif
                src_path, tar_path
            );
        else
            buf = g_strdup_printf(
#ifdef PROG_LANGUAGE_ENGLISH
"Extract %i objects to:\n\
\n\
    %s\n",
#endif
#ifdef PROG_LANGUAGE_SPANISH
"El extracto %i los objetos a:\n\
\n\
    %s\n",
#endif
#ifdef PROG_LANGUAGE_FRENCH
"Extrait %i les objets :\n\
\n\
    %s\n",
#endif
                total_objects, tar_path
            );

        CDialogSetTransientFor(toplevel);
        response = CDialogGetResponseIconData(
            title, buf, NULL,
            (guint8 **)icon_archive_extract_32x32_xpm,
            CDIALOG_BTNFLAG_YES | CDIALOG_BTNFLAG_NO,
            CDIALOG_BTNFLAG_YES
        );
        CDialogSetTransientFor(NULL);

        g_free(buf);    /* Deallocate message. */

        return(response);
}

/*
 *      Confirm delete archive object.
 *
 *      Returns one of CDIALOG_RESPONSE_*.
 *      Confirmation will always be CDIALOG_RESPONSE_YES if configuration
 *      specifies no confirmation for this operation.
 */
gint EDVConfirmArchiveDelete(
        edv_core_struct *core_ptr, GtkWidget *toplevel,
        const gchar *src_path, gint total_objects
)
{
        gint response;
        const gchar *title;
        gchar *buf;


        if(core_ptr == NULL)
            return(CDIALOG_RESPONSE_NOT_AVAILABLE);

        if(!EDVCFGItemListGetValueI(
            core_ptr->cfg_list, EDV_CFG_PARM_CONFIRM_ARCHIVE_DELETE
        ))
            return(CDIALOG_RESPONSE_YES);

        /* Set up confirmation title and message. */

#ifdef PROG_LANGUAGE_ENGLISH
        title = "Confirm Delete";
#endif
#ifdef PROG_LANGUAGE_SPANISH
        title = "Confirme Borre";
#endif
#ifdef PROG_LANGUAGE_FRENCH
        title = "Confirmer Effacer";
#endif

        if((total_objects == 1) && (src_path != NULL))
	    buf = g_strdup_printf(
#ifdef PROG_LANGUAGE_ENGLISH
"Delete:\n\
\n\
    %s\n\
\n\
This archive object cannot be recovered once it\n\
has been deleted!",
#endif
#ifdef PROG_LANGUAGE_SPANISH
"Borre:\n\
\n\
    %s\n\
\n\
Este objeto del archivo no se puede recuperar una\n\
vez se ha borrado!",
#endif
#ifdef PROG_LANGUAGE_FRENCH
"Effacer:\n\
\n\
    %s\n\
\n\
Cet objet d'archive ne peut pas tre retrouv une\n\
fois il a t effac!",
#endif
		src_path
            );
        else
            buf = g_strdup_printf(
#ifdef PROG_LANGUAGE_ENGLISH
"Delete %i selected objects?\n\
\n\
These archive objects cannot be recovered once\n\
they have been deleted!",
#endif
#ifdef PROG_LANGUAGE_SPANISH
"Borra %i objetos escogidos?\n\
\n\
Estos objetos del archivo no se pueden recuperar\n\
una vez ellos han sido borrados!",
#endif
#ifdef PROG_LANGUAGE_FRENCH
"Efface %i des objets choisis?\n\
\n\
Ces objets d'archive ne peuvent pas tre\n\
retrouvs une fois ils ont t effacs!",
#endif
                total_objects
            );

        CDialogSetTransientFor(toplevel);
        response = CDialogGetResponseIconData(
            title, buf, NULL,
            (guint8 **)icon_purge_32x32_xpm,
            CDIALOG_BTNFLAG_YES | CDIALOG_BTNFLAG_NO,
            CDIALOG_BTNFLAG_YES
        );
        CDialogSetTransientFor(NULL);

        g_free(buf);    /* Deallocate message. */

        return(response);
}


/*
 *      Confirm recovering of a recycled object.
 *
 *	Note that src_path is actually a string containing the recycled
 *	object index number.
 *
 *      Returns one of CDIALOG_RESPONSE_*.
 *      Confirmation will always be CDIALOG_RESPONSE_YES if configuration
 *      specifies no confirmation for this operation.
 */
gint EDVConfirmRecover(
        edv_core_struct *core_ptr, GtkWidget *toplevel,
        const gchar *src_path,		/* String containing index number. */
	gint total_objects,
        const gchar *tar_path
)
{
        gint response;
        const gchar *title;
        gchar *buf;
	const gchar *recycled_index_file;


        if(core_ptr == NULL)
            return(CDIALOG_RESPONSE_NOT_AVAILABLE);

        if(!EDVCFGItemListGetValueI(
            core_ptr->cfg_list, EDV_CFG_PARM_CONFIRM_RECOVER
        ))
            return(CDIALOG_RESPONSE_YES);


	/* Get path to recycled objects index file. */
	recycled_index_file = EDVCFGItemListGetValueS(
	    core_ptr->cfg_list, EDV_CFG_PARM_FILE_RECYCLED_INDEX
	);


        /* Set up confirmation title and message. */

#ifdef PROG_LANGUAGE_ENGLISH
        title = "Confirm Recover";
#endif
#ifdef PROG_LANGUAGE_SPANISH
        title = "Confirme Recupere";
#endif
#ifdef PROG_LANGUAGE_FRENCH
        title = "Confirmer Retrouver";
#endif

        if((total_objects == 1) && (src_path != NULL))
	{
	    edv_recbin_object_struct *rec_obj = EDVRecBinObjectStat(
		recycled_index_file, atoi(src_path)
	    );
	    if((rec_obj != NULL) ? (rec_obj->name != NULL) : FALSE)
		buf = g_strdup_printf(
#ifdef PROG_LANGUAGE_ENGLISH
"Recover:\n\
\n\
    %s\n\
\n\
To:\n\
\n\
    %s\n",
#endif
#ifdef PROG_LANGUAGE_SPANISH
"Recupere:\n\
\n\
    %s\n\
\n\
A:\n\
\n\
    %s\n",
#endif
#ifdef PROG_LANGUAGE_FRENCH
"Retrouver:\n\
\n\
    %s\n\
\n\
A:\n\
\n\
    %s\n",
#endif
		     rec_obj->name, tar_path
		);
	    else
                buf = g_strdup_printf(
#ifdef PROG_LANGUAGE_ENGLISH
"Recover 1 object to:\n\
\n\
    %s\n",
#endif
#ifdef PROG_LANGUAGE_SPANISH
"Recupere 1 objeto a:\n\
\n\
    %s\n",
#endif
#ifdef PROG_LANGUAGE_FRENCH
"Retrouver 1 objet :\n\
\n\
    %s\n",
#endif
                    tar_path
                );

	    EDVRecBinObjectDelete(rec_obj);	/* Delete recycled object stats. */
	}
        else
        {
            buf = g_strdup_printf(
#ifdef PROG_LANGUAGE_ENGLISH
"Recover %i objects to:\n\
\n\
    %s\n",
#endif
#ifdef PROG_LANGUAGE_SPANISH
"Recupere %i objetos a:\n\
\n\
    %s\n",
#endif
#ifdef PROG_LANGUAGE_FRENCH
"etrouver %i des objets :\n\
\n\
    %s\n",
#endif
		total_objects, tar_path
	    );
	}

        CDialogSetTransientFor(toplevel);
        response = CDialogGetResponseIconData(
            title, buf, NULL,
            (guint8 **)icon_recover_32x32_xpm,
            CDIALOG_BTNFLAG_YES | CDIALOG_BTNFLAG_NO,
            CDIALOG_BTNFLAG_YES
        );
        CDialogSetTransientFor(NULL);

        g_free(buf);    /* Deallocate message. */

        return(response);
}

/*
 *      Confirm purge of a recycled object.
 *
 *      Note that src_path is actually a string containing the recycled
 *      object index number.
 *
 *      Returns one of CDIALOG_RESPONSE_*.
 *      Confirmation will always be CDIALOG_RESPONSE_YES if configuration
 *      specifies no confirmation for this operation.
 */
gint EDVConfirmPurge(
        edv_core_struct *core_ptr, GtkWidget *toplevel,
        const gchar *src_path, gint total_objects
)
{
        gint response;
        const gchar *title;
        gchar *buf;
        const gchar *recycled_index_file;


        if(core_ptr == NULL)
            return(CDIALOG_RESPONSE_NOT_AVAILABLE);

        if(!EDVCFGItemListGetValueI(
            core_ptr->cfg_list, EDV_CFG_PARM_CONFIRM_PURGE
        ))
            return(CDIALOG_RESPONSE_YES);


        /* Get path to recycled objects index file. */
        recycled_index_file = EDVCFGItemListGetValueS(
            core_ptr->cfg_list, EDV_CFG_PARM_FILE_RECYCLED_INDEX
        );


        /* Set up confirmation title and message. */

#ifdef PROG_LANGUAGE_ENGLISH
        title = "Confirm Purge";
#endif
#ifdef PROG_LANGUAGE_SPANISH
        title = "Confirme Purge";
#endif
#ifdef PROG_LANGUAGE_FRENCH
        title = "Confirmer la Purge";
#endif

        if((total_objects == 1) && (src_path != NULL))
        {
            edv_recbin_object_struct *rec_obj = EDVRecBinObjectStat(
                recycled_index_file, atoi(src_path)
            );
            if((rec_obj != NULL) ? (rec_obj->name != NULL) : FALSE)
                buf = g_strdup_printf(
#ifdef PROG_LANGUAGE_ENGLISH
"Purge recycled object `%s'?\n\
\n\
You will not be able to recover this object\n\
once it has been purged!",
#endif
#ifdef PROG_LANGUAGE_SPANISH
"El objeto de recycled de Purge `%s'?\n\
\n\
Usted no ser capaz de recuperar este objeto\n\
una vez se ha purged!",
#endif
#ifdef PROG_LANGUAGE_FRENCH
"Est-ce que la purge a recycl l'objet `%s'?\n\
\n\
Vous ne pourrez pas retrouver cet objet une\n\
fois il a t purg!",
#endif
                    rec_obj->name
                );
            else
                buf = g_strdup_printf(
#ifdef PROG_LANGUAGE_ENGLISH
"Purge 1 recycled object?\n\
\n\
You will not be able to recover this object\n\
once it has been purged!"
#endif
#ifdef PROG_LANGUAGE_SPANISH
"Purge 1 objeto de recycled?\n\
\n\
Usted no ser capaz de recuperar este objeto\n\
una vez se ha purged!"
#endif
#ifdef PROG_LANGUAGE_FRENCH
"Purge 1 objet recycl?\n\
\n\
Vous ne pourrez pas retrouver cet objet une\n\
fois il a t purg!"
#endif
		);

            EDVRecBinObjectDelete(rec_obj);     /* Delete recycled object stats. */
	}
        else
	{
            buf = g_strdup_printf(
#ifdef PROG_LANGUAGE_ENGLISH
"Purge %i recycled objects?\n\
\n\
You will not be able to recover these objects\n\
once they have been purged!",
#endif
#ifdef PROG_LANGUAGE_SPANISH
"Recycled de Purge %i opone?\n\
\n\
Usted no ser capaz de recuperar estos objetos\n\
una vez ellos han sido purged!",
#endif
#ifdef PROG_LANGUAGE_FRENCH
"Est-ce que la purge a recycl %i objets?\n\
\n\
Vous ne pourrez pas retrouver ces objets\n\
une fois ils ont t purgs!",
#endif
                total_objects
            );
	}

        CDialogSetTransientFor(toplevel);
        response = CDialogGetResponseIconData(
            title, buf, NULL,
            (guint8 **)icon_purge_32x32_xpm,
            CDIALOG_BTNFLAG_YES | CDIALOG_BTNFLAG_NO,
            CDIALOG_BTNFLAG_YES
        );
        CDialogSetTransientFor(NULL);

        g_free(buf);    /* Deallocate message. */

        return(response);
}


/*
 *	Calls appropriate confirmation based on the given GTK DND 
 *	information regarding the target type and the drag action.
 */
gint EDVConfirmDND(
        edv_core_struct *core_ptr,
	gint gdk_action, guint info,
        GtkWidget *toplevel,
        edv_dnd_object_struct **dnd_object, gint total_dnd_objects,
        const gchar *tar_path
)
{
	gint total_objects = total_dnd_objects;
        const gchar *src_path = NULL;


	/* Must have the DND target name, target path, and a positive
	 * number of DND objects.
	 */
        if((tar_path == NULL) || (total_dnd_objects <= 0))
            return(CDIALOG_RESPONSE_NOT_AVAILABLE);

        /* If exactly one source object then get source path of that
         * object. Otherwise leave src_path as NULL.
         */
        if(total_dnd_objects == 1)
        {
            edv_dnd_object_struct *dnd_obj = dnd_object[0];
            if(dnd_obj != NULL)
                src_path = dnd_obj->full_path;
        }


        /* Handle by drag context target type. */

        /* Disk object? */
        if((info == EDV_DND_TYPE_INFO_TEXT_PLAIN) ||
           (info == EDV_DND_TYPE_INFO_TEXT_URI_LIST) ||
           (info == EDV_DND_TYPE_INFO_STRING)
	)
        {
            switch(gdk_action)
            {
              case GDK_ACTION_LINK:
		return(EDVConfirmLink(
		    core_ptr, toplevel,
		    src_path, total_objects, tar_path
		));
		break;
              case GDK_ACTION_MOVE:
                return(EDVConfirmMove(
                    core_ptr, toplevel,
                    src_path, total_objects, tar_path
                ));
                break;
              case GDK_ACTION_COPY:
                return(EDVConfirmCopy(
                    core_ptr, toplevel,
                    src_path, total_objects, tar_path
                ));
                break;
	    }
	}
        /* Recycled object? */
	else if(info == EDV_DND_TYPE_INFO_RECYCLED_OBJECT)
        {
            switch(gdk_action)
            {
              case GDK_ACTION_LINK:
              case GDK_ACTION_MOVE:
              case GDK_ACTION_COPY:
                return(EDVConfirmRecover(
                    core_ptr, toplevel,
                    src_path, total_objects, tar_path
                ));
                break;
	    }
	}
        /* Archive object? */
        else if(info == EDV_DND_TYPE_INFO_ARCHIVE_OBJECT)
        {
            switch(gdk_action)
            {
              case GDK_ACTION_LINK:
              case GDK_ACTION_MOVE:
              case GDK_ACTION_COPY:
		/* The first object is considered the archive itself, and
		 * all subsequent objects are archive objects within the
		 * archive.
		 */
		if(total_dnd_objects == 2)
		{
		    edv_dnd_object_struct *dnd_obj = dnd_object[1];
		    if(dnd_obj != NULL)
			src_path = dnd_obj->full_path;
		}
                return(EDVConfirmArchiveExtract(
                    core_ptr, toplevel,
                    src_path, total_objects - 1, tar_path
                ));
                break;
            }
        }

	/* Unsupported DND target type or drag action, return response not
	 * available just to be on the safe side so that the operation
	 * does not continue.
	 */
	return(CDIALOG_RESPONSE_NOT_AVAILABLE);
}
