#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <time.h>
#include <glib.h>

#include "edvtypes.h"
#include "edvdate.h"

#include "config.h"


gchar *EDVDateFormatString(
	gulong t,		/* In systime seconds. */
	const gchar *format,	/* Can be NULL to just use ctime(). */
	gint relativity
);



/*
 *	Returns a statically allocated string containing the time
 *	specified by t in systime seconds formatted with the given
 *	string format (which can be NULL to imply use ctime() instead)
 *	and relative to the given relativitiy.
 *
 *	If relativity is EDV_DATE_RELATIVITY_CURRENT then format will be
 *	completely ignored.
 *
 *	The returned pointer must not be deallocated, this function
 *	never returns NULL.
 */
gchar *EDVDateFormatString(
        gulong t,               /* In systime seconds. */
        const gchar *format,    /* Can be NULL to just use ctime(). */
        gint relativity
)
{


	/* Handle by relativity. */
	switch(relativity)
	{
	  case EDV_DATE_RELATIVITY_CURRENT:
	    if(1)
	    {
		gulong ct, dt = (gulong)time(NULL) - t;
                static gchar buf[80];


		/* Less than one minute? */
		if(dt < (1 * 60))
		{
		    ct = (dt / 1);
		    sprintf(
			buf, "%ld second%s ago",
			ct, ((ct == 1) ? "" : "s")
		    );
		}
		/* Less than one hour? */
		else if(dt < (60 * 60))
                {
                    ct = (dt / 60);
		    sprintf(
                        buf, "%ld minute%s ago",
                        ct, ((ct == 1) ? "" : "s")
                    );
		}
		/* Less than one day? */
                else if(dt < (24 * 60 * 60))
                {
                    ct = (dt / 60 / 60);
                    sprintf(
                        buf, "%ld hour%s ago",
                        ct, ((ct == 1) ? "" : "s")
                    );
                }
		/* Less than one week? */
                else if(dt < (7 * 24 * 60 * 60))
                {
                    ct = (dt / 60 / 60 / 24);
                    sprintf(
                        buf, "%ld day%s ago",
                        ct, ((ct == 1) ? "" : "s")
                    );
		}
                /* Less than one month (30 days)? */
                else if(dt < (30 * 24 * 60 * 60))
                {
                    ct = (dt / 60 / 60 / 24 / 7);
                    sprintf(
                        buf, "%ld week%s ago",
                        ct, ((ct == 1) ? "" : "s")
                    );
                }
		/* Less than 6 months ago? */
                else if(dt < (6 * 30 * 24 * 60 * 60))
                {
                    ct = (dt / 60 / 60 / 24 / 7 / 30);
                    sprintf(
                        buf, "%ld month%s ago",
                        ct, ((ct == 1) ? "" : "s")
                    );
                }
		else
		{
		    /* All else revert to using ctime(). */
		    time_t t2 = (time_t)t;
		    const gchar *cstrptr;


		    cstrptr = ctime(&t2);
		    strncpy(
			buf,
			(cstrptr != NULL) ? cstrptr : "",
			80
		    );
		    buf[80 - 1] = '\0';
		}

		return(buf);
	    }
	    break;

	  default:	/* EDV_DATE_RELATIVITY_ABSOLUTE */
	    if(format == NULL)
	    {
		/* No format string given so assume to use ctime(). */
		time_t t2 = (time_t)t;
		return((gchar *)ctime(&t2));
	    }
	    else
	    {
		time_t t2 = (time_t)t;
		static gchar buf[1024];
		const struct tm *tm_ptr;


		*buf = '\0';
		tm_ptr = localtime(&t2);

		if(tm_ptr != NULL)
		    strftime(
			buf, 1024,
			format, tm_ptr
		    );
		return(buf);
	    }
	    break;
	}
}
