#include <glib.h>

#include "edvhistory.h"


edv_history_struct *EDVHistoryNew(void);
void EDVHistoryDelete(edv_history_struct *h);

void EDVHistoryDiskObjectSet(
        edv_history_struct *h,
        gint type,
        gulong time_start, gulong time_end,
	gint status,
        const gchar *src_path, const gchar *tar_path,
        const gchar *comments
);

edv_history_struct *EDVHistoryListAppendNew(
        edv_history_struct ***list, gint *total,
        gint *n, gint max
);


/*
 *	Allocates a new history item.
 */
edv_history_struct *EDVHistoryNew(void)
{
	edv_history_struct *h = (edv_history_struct *)g_malloc0(
	    sizeof(edv_history_struct)
	);
	if(h == NULL)
	    return(h);

	h->type = EDV_HISTORY_NONE;

	h->time_start = 0;
	h->time_end = 0;

	h->status = 0;

	h->src_path = NULL;
	h->tar_path = NULL;

	h->comments = NULL;

	return(h);
}

/*
 *	Deallocates the given history item and all its resources.
 */
void EDVHistoryDelete(edv_history_struct *h)
{
	if(h == NULL)
            return;

	g_free(h->src_path);
	g_free(h->tar_path);

	g_free(h->comments);

	g_free(h);
}


/*
 *	Sets the given values to the given history item.
 *
 *	The given type must be one of EDV_HISTORY_DISK_OBJECT_*.
 */
void EDVHistoryDiskObjectSet(
        edv_history_struct *h,
        gint type,
        gulong time_start, gulong time_end,
	gint status,
        const gchar *src_path, const gchar *tar_path,
	const gchar *comments
)
{
	if(h == NULL)
	    return;

	h->type = type;

	h->time_start = time_start;
	h->time_end = time_end;

	h->status = status;

	g_free(h->src_path);
	h->src_path = (src_path != NULL) ? g_strdup(src_path) : NULL;

	g_free(h->tar_path);
        h->tar_path = (tar_path != NULL) ? g_strdup(tar_path) : NULL;

	g_free(h->comments);
	h->comments = (comments != NULL) ? g_strdup(comments) : NULL;
}

/*
 *	Appends a new history item to the given list, reallocating the
 *	given list and increasing the total as needed.
 *
 *	Returns the pointer to the new history item structure and sets
 *	*n to the new index or -1 on error.
 *
 *	If max is positive and appending the new item would cause the
 *	maximum to be exceeded then the oldest (index 0) item will be
 *	deleted and the item pointers in the list will be shifted.
 *
 *	Can return NULL on error.
 */
edv_history_struct *EDVHistoryListAppendNew(
        edv_history_struct ***list, gint *total,
        gint *n, gint max
)
{
	gint i;
	edv_history_struct *h;


	if(n != NULL)
	    *n = -1;

	if((list == NULL) || (total == NULL))
	    return(NULL);

	/* Sanitize total. */
	if(*total < 0)
	    *total = 0;

	/* Allocate one more pointer and append a new item. */
	i = *total;
	*total = i + 1;
	*list = (edv_history_struct **)g_realloc(
	    *list,
	    (*total) * sizeof(edv_history_struct *)
	);
	if(*list == NULL)
	{
	    *total = 0;
	    return(NULL);
	}

	/* Allocate a new item at the highest index. */
	(*list)[i] = h = EDVHistoryNew();
	if(h == NULL)
	    return(NULL);


	/* Check if total has exceeded maximum. */
	if(max > 0)
	{
	    while(*total > max)
	    {
		/* Delete oldest (index 0) item. */
		EDVHistoryDelete((*list)[0]);

		/* Reduce total. */
		i = *total;
		*total = i - 1;

		/* Shift pointers. */
		for(i = 0; i < *total; i++)
		    (*list)[i] = (*list)[i + 1];

		/* Reallocate pointers. */
		*list = (edv_history_struct **)g_realloc(
		    *list,
		    (*total) * sizeof(edv_history_struct *)
		);
		if(*list == NULL)
		{
		    *total = 0;
	    	    return(NULL);
		}
	    }
	}

	/* Update index return. */
	i = *total;
	if(n != NULL)
	    *n = i - 1;

	return(h);
}
