#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <errno.h>
#include <limits.h>
#include <unistd.h>
#include <glib.h>

#include "../include/string.h"
#include "../include/fio.h"
#include "../include/disk.h"

#include "edvtypes.h"
#include "edvobj.h"
#include "edvrecbin.h"
#include "edvrecbinfio.h"
#include "edvutils.h"
#include "edvutilsgtk.h"
#include "config.h"



const gchar *EDVRecBinFIOGetError(void);

/* Recycled index file functions. */
gint EDVRecBinFIOTotalItems(const gchar *filename);
guint *EDVRecBinFIOGetIndexList(
        const gchar *filename, gint *total
);

static void EDVRecBinIndexWriteObjectIterate(
        FILE *fp, guint index, edv_recbin_object_struct *obj
);

edv_recbin_index_struct *EDVRecBinIndexOpen(
        const gchar *filename
);
gint EDVRecBinIndexNext(edv_recbin_index_struct *rbi_ptr);
void EDVRecBinIndexClose(edv_recbin_index_struct *rbi_ptr);

guint EDVRecBinIndexAdd(
        const gchar *filename, edv_recbin_object_struct *obj
);
gint EDVRecBinIndexRemove(
	const gchar *filename, gint index
);

/* Recycled object reference functions. */
static gint EDVRecBinObjectGetFromIndexFileIterate(
	FILE *fp,
        edv_recbin_object_struct *obj
);
static gint EDVRecBinObjectGetFromIndexFile(
        const gchar *filename,          /* recycled.ini */
        guint index,
        edv_recbin_object_struct *obj
);
edv_recbin_object_struct *EDVRecBinObjectStat(
        const gchar *filename,          /* recycled.ini */
        guint index                     /* Index of object stats to load. */
);

/* Deleting and recovering disk objects functions. */
gint EDVRecBinDiskObjectDelete(
	const gchar *index_filename,	/* recycled.ini */
	guint index,			/* Put into recycled index file under this index. */
	const gchar *path,		/* Object to delete. */
	gint (*progress_cb)(gpointer, gulong, gulong),
	gpointer client_data
);
gint EDVRecBinDiskObjectRecover(
        const gchar *index_filename,    /* recycled.ini */
        guint index,			/* Recycled object to recover. */
        const gchar *path,              /* Alternate recovery dir if not NULL. */
        gint (*progress_cb)(gpointer, gulong, gulong),
        gpointer client_data
);
gint EDVRecBinDiskObjectPurge(
        const gchar *index_filename,    /* recycled.ini */
        guint index,                    /* Recycled object to recover. */
        gint (*progress_cb)(gpointer, gulong, gulong),
        gpointer client_data
);
gint EDVRecBinDiskObjectPurgeAll(
        const gchar *index_filename,    /* recycled.ini */
        gint (*progress_cb)(gpointer, gulong, gulong),
        gpointer client_data
);


static gchar *last_error = NULL;


/*
 *	Returns a statically allocated string specifying the the last 
 *	error or NULL if there wasn't any.
 */
const gchar *EDVRecBinFIOGetError(void)
{
	return(last_error);
}

/*
 *	Quickly gets a directory listing of the recycled objects
 *	directory specified by the recycled objects index file filename
 *	then returns the number of recycled objects.
 */
gint EDVRecBinFIOTotalItems(const gchar *filename)
{
	gint total;
	gchar *parent_path;


	if(filename == NULL)
	    return(0);

        /* Get parent directory of the recycled objects index file. */
        parent_path = GetParentDir(filename);
        parent_path = (parent_path != NULL) ? g_strdup(parent_path) : NULL;

	/* Get total number of entries in the recycled objects directory. */
	total = NUMDIRCONTENTS(parent_path);

	/* Subtract the given recycled objects index file from the total
	 * count.
	 */
	total--;
	if(total < 0)
	    total = 0;

	/* Deallocate copy of recycled objects directory. */
	g_free(parent_path);

	return(total);
}

/*
 *	Returns a dynamically allocated list of indexes loaded from the
 *	recycled objects index file specified by filename.
 *
 *	The calling function must deallocate the returned list of recycled
 *	object indexes.
 */
guint *EDVRecBinFIOGetIndexList(
        const gchar *filename, gint *total
)
{
	guint *index = NULL;
	FILE *fp;


	if((filename == NULL) || (total == NULL))
	    return(index);

	/* Reset total. */
	*total = 0;

	/* Open recycled objects index file. */
	fp = FOpen(filename, "rb");
	if(fp == NULL)
	    return(index);

	/* Begin reading recycled objects index file. */
        while(1)
        {

            /* Seek to next start of recycled object block parameter. */
	    if(FSeekToParm(
		fp,
		"BeginRecycledObject",
		EDV_CFG_COMMENT_CHAR, EDV_CFG_DELIMINATOR_CHAR
	    ))
		break;

            if(1)
            {
		gint i;
                gint value[1];


		/* Get value of start of recycled object block parameter
		 * which is the index of the recycled object.
		 */
                FGetValuesI(fp, value, 1);

		/* Allocate more pointers. */
		i = *total;
		*total = i + 1;
		index = (guint *)g_realloc(index, (*total) * sizeof(guint));
		if(index == NULL)
		{
		    *total = 0;
		    break;
		}
		else
		{
		    index[i] = (guint)value[0];
		}
	    }
	}

	/* Close recycled objects index file. */
	FClose(fp);
	fp = NULL;

	return(index);
}


/*
 *	Writes one recycled object block of the specifed object to the
 *	recycled index file opened for writing specified by fp.
 */
static void EDVRecBinIndexWriteObjectIterate(
	FILE *fp, guint index, edv_recbin_object_struct *obj
)
{
	if((fp == NULL) || (obj == NULL))
	    return;

	fprintf(
	    fp,
	    "BeginRecycledObject = %i\n",
	    index
	);
	if(obj->name != NULL)
            fprintf(
                fp,
                "    Name = %s\n",
		obj->name
	    );
        if(obj->original_path != NULL)
            fprintf(
                fp,
                "    OriginalPath = %s\n",
                obj->original_path
            );
        fprintf(
            fp,
            "    DateDeleted = %ld\n",
            obj->date_deleted
        );
        fprintf(
            fp,
            "    Type = %i\n",
            obj->type
        );
        fprintf(
            fp,
            "    Permissions = %i\n",
            obj->permissions
        );
        fprintf(
            fp,
            "    AccessTime = %ld\n",
            obj->access_time
        );
        fprintf(
            fp,
            "    ModifyTime = %ld\n",
            obj->modify_time
        );
        fprintf(
            fp,
            "    ChangeTime = %ld\n",
            obj->change_time
        );
        fprintf(
            fp,
            "    OwnerID = %i\n",
            obj->owner_id
        );
        fprintf(
            fp,
            "    GroupID = %i\n",
            obj->group_id
        );
        fprintf(
            fp,
            "    Size = %ld\n",
            obj->size
        );




        fprintf(
            fp,
            "EndRecycledObject\n"
        );
}

/*
 *	Opens the recycled index file specified by filename and returns
 *	a handle that will be passed to subsequent calls to
 *	EDVRecBinIndexNext() and EDVRecBinIndexClose().
 *
 *	EDVRecBinIndexNext() must be called immediatly after calling
 *	this function to obtain the first recycled object.
 *
 *	Can return NULL on error.
 */
edv_recbin_index_struct *EDVRecBinIndexOpen(
        const gchar *filename
)
{
	FILE *fp;
	edv_recbin_index_struct *rbi_ptr;


	if(filename == NULL)
	    return(NULL);

	/* Open the recycled index file. */
	fp = FOpen(filename, "rb");
	if(fp == NULL)
	    return(NULL);

	/* Allocate a new recycled index handle structure. */
	rbi_ptr = (edv_recbin_index_struct *)g_malloc0(
	    sizeof(edv_recbin_index_struct)
	);
	if(rbi_ptr == NULL)
	{
	    FClose(fp);
	    return(rbi_ptr);
	}

	/* Update values. */
	rbi_ptr->fp = fp;
	rbi_ptr->index = 0;
	rbi_ptr->obj = EDVRecBinObjectNew();

	return(rbi_ptr);
}

/*
 *	Updates the given recbin index handle's index to the next recycled
 *	object index found in the recycled index file.
 *
 *	Returns non-zero on error or end of file.
 */
gint EDVRecBinIndexNext(edv_recbin_index_struct *rbi_ptr)
{
	gint status;
	FILE *fp;
	edv_recbin_object_struct *obj;
	gchar *parm_str = NULL;


	if(rbi_ptr == NULL)
	    return(-1);

	fp = rbi_ptr->fp;
	obj = rbi_ptr->obj;
	if((fp == NULL) || (obj == NULL))
	    return(-1);

        /* Begin reading recycled index file. */
	status = 0;
        while(1)
        {
            /* Read next parameter. */
            parm_str = FSeekNextParm(
                fp, parm_str,
                EDV_CFG_COMMENT_CHAR,
                EDV_CFG_DELIMINATOR_CHAR
            );
            if(parm_str == NULL)
	    {
		status = -1;
                break;
	    }

            /* Begin handling by parameter. */

            /* Start of a recycled object block? */
            if(!strcasecmp(parm_str, "BeginRecycledObject"))
            {
                gint value[1];

                FGetValuesI(fp, value, 1);

		/* Update current index. */
		rbi_ptr->index = (guint)value[0];

		/* Read all subsequent parameters from the current
		 * fp position loading them to the recycled object
		 * structure. Stopping just after the next end of
		 * object block structure.
		 */
		obj->index = (guint)value[0];
		EDVRecBinObjectGetFromIndexFileIterate(fp, obj);

		/* Stop reading once this object has been loaded. */
		break;
            }
            else
            {
                /* Other parameter, skip. */
                FSeekNextLine(fp);
            }
        }

        /* Deallocate parameter string. */
        g_free(parm_str);
        parm_str = NULL;

	return(status);
}

/*
 *	Closes the recycled index file and deallocates the recbin index
 *	handle.
 */
void EDVRecBinIndexClose(edv_recbin_index_struct *rbi_ptr)
{
	if(rbi_ptr == NULL)
	    return;

	/* Close recycled index file. */
	if(rbi_ptr->fp != NULL)
	{
	    FClose(rbi_ptr->fp);
	    rbi_ptr->fp = NULL;
	}

	/* Delete context recycled object structure. */
	EDVRecBinObjectDelete(rbi_ptr->obj);
	rbi_ptr->obj = NULL;

	/* Deallocate structure itself. */
	g_free(rbi_ptr);
}


/*
 *	Adds the values of the specified recycled object to the recycled
 *	index file. The given recycled object structure will not be
 *	modified.
 *
 *	Returns the index of the new entry or 0 on error.
 */
guint EDVRecBinIndexAdd(
        const gchar *filename, edv_recbin_object_struct *obj
)
{
	gint i, total;
	guint *index, cur_index, new_index = 0;
        FILE *fp;
        gchar *parent_path;


        if(filename == NULL)
            return(0);


	/* Get list of indexes in the recycled objects index file. */
	index = EDVRecBinFIOGetIndexList(filename, &total);

	/* Iterate through index list, looking for an index that is not
	 * in the list so that we can use it. The new_index is initially
	 * set to 0, if an available index is found then new_index will be
	 * set to that index value.
	 */
	for(cur_index = 1; cur_index != 0; cur_index++)
	{
	    /* Iterate through index list, checking if cur_index matches
	     * any index in the list and breaking the loop prematurly if
	     * there is a match.
	     */
	    for(i = 0; i < total; i++)
	    {
		if(cur_index == index[i])
		    break;
	    }
	    /* Was cur_index not in the list? */
	    if(i >= total)
	    {
		new_index = cur_index;
		break;
	    }
	}

	/* Deallocate index list. */
	g_free(index);
	index = NULL;
	total = 0;


	/* No more indexes available? */
	if(new_index == 0)
	    return(0);


	/* Begin adding a new recycled object entry to the recycled
	 * objects index file.
	 */

        /* Get parent directory and create it as needed. */
        parent_path = GetParentDir(filename);
        parent_path = (parent_path != NULL) ? g_strdup(parent_path) : NULL;
        if(parent_path != NULL)
        {
            rmkdir(parent_path, S_IRUSR | S_IWUSR | S_IXUSR);
            g_free(parent_path);
            parent_path = NULL;
        }

        /* Open recycled index file for write append. */
        fp = FOpen(filename, "ab");

	/* Write given recycled object structure to recycled index file. */
	EDVRecBinIndexWriteObjectIterate(
	    fp, new_index, obj
	);

	/* Close recycled index file. */
	FClose(fp);
	fp = NULL;


	return(new_index);
}

/*
 *	Removes the entry in the given recycled index file.
 *
 *	Returns the number of occurances removed.
 */
gint EDVRecBinIndexRemove(
        const gchar *filename, gint index
)
{
	FILE *fp;
	gchar *parent_path;
	gchar *out_file, *in_file;
	edv_recbin_index_struct *rbi_ptr;
	gint objects_removed = 0;


	if(filename == NULL)
	    return(objects_removed);


        /* Get parent directory and create it as needed. */
        parent_path = GetParentDir(filename);
        parent_path = (parent_path != NULL) ? g_strdup(parent_path) : NULL;
        if(parent_path != NULL)
        {
            rmkdir(parent_path, S_IRUSR | S_IWUSR | S_IXUSR);
            g_free(parent_path);
            parent_path = NULL;
        }


	/* Make coppies of input and output recycled index file names. */
	in_file = g_strdup(filename);
	out_file = g_strdup(filename);
	out_file = strcatalloc(out_file, "_");

	/* Open output file for writing. */
	fp = FOpen(out_file, "wb");

	/* Open input file. */
	rbi_ptr = EDVRecBinIndexOpen(in_file);
	/* Iterate through all indexes. */
	while(!EDVRecBinIndexNext(rbi_ptr))
	{
	    if(rbi_ptr->index != index)
		EDVRecBinIndexWriteObjectIterate(
		    fp, rbi_ptr->index, rbi_ptr->obj
		);
	    else
		objects_removed++;
	}
	EDVRecBinIndexClose(rbi_ptr);
	rbi_ptr = NULL;


	/* Close output file. */
	FClose(fp);
	fp = NULL;


	/* Remove input file and rename output file to input file. */
	unlink(in_file);
	rename(out_file, in_file);


	/* Deallocate coppied recycled index file names. */
	g_free(out_file);
	g_free(in_file);

	return(objects_removed);
}


/*
 *	Reads all subsequent parameters from the recycled index file
 *	specified by the opened fp and loads them to the given recycled
 *	object structure.
 *
 *	The fp will be positioned just after the next end of object
 *	block.
 *
 *	Inputs assumed valid.
 */
gint EDVRecBinObjectGetFromIndexFileIterate(
        FILE *fp,
        edv_recbin_object_struct *obj
)
{
	gint status = 0;
	gchar *parm_str = NULL;


        /* Begin reading file. */
        while(1)
        {
            /* Read next parameter. */
            parm_str = FSeekNextParm(
                fp, parm_str,
                EDV_CFG_COMMENT_CHAR,
                EDV_CFG_DELIMINATOR_CHAR
            );
            if(parm_str == NULL)
	    {
		status = -1;
                break;
	    }

            /* Begin handling by parameter. */

            /* Name? */
            else if(!strcasecmp(parm_str, "Name"))
            {
                g_free(obj->name);
                obj->name = FGetString(fp);
            }
            /* Original path? */
            else if(!strcasecmp(parm_str, "OriginalPath"))
            {
                g_free(obj->original_path);
                obj->original_path = FGetString(fp);
            }
            /* Date deleted? */
            else if(!strcasecmp(parm_str, "DateDeleted"))
            {
                gchar *value = FGetString(fp);
                if(value != NULL)
                {
                    obj->date_deleted = (gulong)atol(value);
                    g_free(value);
                    value = NULL;
                }
            }
	    /* Type. */
            else if(!strcasecmp(parm_str, "Type"))
            {
                gint value[1];

                FGetValuesI(fp, value, 1);
		obj->type = value[0];
	    }
            /* Permissions. */
            else if(!strcasecmp(parm_str, "Permissions"))
            {
                gint value[1];

                FGetValuesI(fp, value, 1);
                obj->permissions = (guint)value[0];
            }
            /* Access time. */
            else if(!strcasecmp(parm_str, "AccessTime"))
            {
                gchar *value = FGetString(fp);
                if(value != NULL)
                {
                    obj->access_time = (gulong)atol(value);
                    g_free(value);
                    value = NULL;
                }
            }
            /* Modify time. */
            else if(!strcasecmp(parm_str, "ModifyTime"))
            {
                gchar *value = FGetString(fp);
                if(value != NULL)
                {
                    obj->modify_time = (gulong)atol(value);
                    g_free(value);
                    value = NULL;
                }
            }
            /* Change time. */
            else if(!strcasecmp(parm_str, "ChangeTime"))
            {
                gchar *value = FGetString(fp);
                if(value != NULL)
                {
                    obj->change_time = (gulong)atol(value);
                    g_free(value);
                    value = NULL;
                }
            }
            /* Owner ID. */
            else if(!strcasecmp(parm_str, "OwnerID"))
            {
                gint value[1];

                FGetValuesI(fp, value, 1);
                obj->owner_id = value[0];
            }
            /* Group ID. */
            else if(!strcasecmp(parm_str, "GroupID"))
            {
                gint value[1];

                FGetValuesI(fp, value, 1);
                obj->group_id = value[0];
            }
            /* Size. */
            else if(!strcasecmp(parm_str, "Size"))
            {
                gchar *value = FGetString(fp);
                if(value != NULL)
                {
                    obj->size = (gulong)atol(value);
                    g_free(value);
                    value = NULL;
                }
            }

            /* End of a recycled object block? */
            else if(!strcasecmp(parm_str, "EndRecycledObject"))
            {
                /* Skip value, seek to next line. */
                FSeekNextLine(fp);

		/* Stop reading once end of object block is encountered. */
		break;
            }
            else
            {
                /* Other parameter, skip. */
                FSeekNextLine(fp);
            }
	}

        /* Deallocate parameter string. */
        g_free(parm_str);
        parm_str = NULL;

	return(status);
}


/*
 *	Loads original values from the specified recycled index file
 *	to the recycled object structure.
 *
 *	Returns non-zero if the recycled object specified by index does
 *	not exist or error occured.
 */
static gint EDVRecBinObjectGetFromIndexFile(
	const gchar *filename,          /* recycled.ini */
        guint index,
	edv_recbin_object_struct *obj
)
{
	gint status;
	FILE *fp;
	gchar *parm_str = NULL;


	if((filename == NULL) || (obj == NULL))
	    return(-1);

	/* Open recycled index file. */
	fp = FOpen(filename, "rb");
	if(fp == NULL)
	    return(-1);

        /* Begin reading recycled index file. */
	status = -1;
        while(1)
        {
            /* Read next parameter. */
            parm_str = FSeekNextParm(
                fp, parm_str,
                EDV_CFG_COMMENT_CHAR,
                EDV_CFG_DELIMINATOR_CHAR
            );
            if(parm_str == NULL)
                break;

            /* Begin handling by parameter. */

            /* Start of a recycled object block? */
            if(!strcasecmp(parm_str, "BeginRecycledObject"))
            {
                gint value[1];


                FGetValuesI(fp, value, 1);

		/* Index value matches the given index? */
		if((guint)value[0] == index)
		{
		    /* Read all subsequent parameters from the current
		     * fp position loading them to the recycled object
		     * structure. Stopping just after the next end of
		     * object block structure.
 		     */
                    obj->index = value[0];
		    EDVRecBinObjectGetFromIndexFileIterate(fp, obj);

		    /* Mark that we matched this object by its index. */
		    status = 0;

		    /* Stop reading once this object has been loaded. */
		    break;
		}
	    }
	    else
	    {
		/* Other parameter, skip. */
		FSeekNextLine(fp);
	    }
	}

	/* Deallocate parameter string. */
	g_free(parm_str);
	parm_str = NULL;

	/* Close recycled index file. */
	FClose(fp);
	fp = NULL;

	return(status);
}


/*
 *      Returns a new recbin object structure containing the information
 *      from the recycled object specified by index from the recycled index
 *      file filename.
 */
edv_recbin_object_struct *EDVRecBinObjectStat(
        const gchar *filename,          /* recycled.ini */
        guint index                     /* Index of object stats to load. */
)
{
	const gchar *cstrptr;
	gchar *recycled_dir = NULL;
	gchar *recycled_file = NULL;
	edv_recbin_object_struct *obj_rtn, *obj = NULL;
	gchar numstr[80];
	struct stat lstat_buf;


	if(filename == NULL)
	    return(NULL);

#define DO_FREE_LOCALS	\
{ \
 g_free(recycled_dir); \
 recycled_dir = NULL; \
 \
 g_free(recycled_file); \
 recycled_file = NULL; \
 \
 EDVRecBinObjectDelete(obj); \
 obj = NULL; \
}

	/* Get parent directory of the recycled index file path, this will
	 * be the recycled objects directory.
	 */
	recycled_dir = EDVRecBinGetDirectoryFromIndexPath(filename);
	if(recycled_dir == NULL)
	{
	    DO_FREE_LOCALS
	    return(NULL);
	}

	/* Formulate full path to recycled object from the given index
	 * number (which will be the file name of the recycled object)
	 * and the recycled_dir as the prefix.
	 */
	sprintf(numstr, "%i", index);
	cstrptr = PrefixPaths(recycled_dir, numstr);
	recycled_file = (cstrptr != NULL) ? g_strdup(cstrptr) : NULL;
	if(recycled_file == NULL)
	{
            DO_FREE_LOCALS
            return(NULL);
        }

	/* Attempt to obtain local stats of the recycled object. */
	if(lstat(recycled_file, &lstat_buf))
        {
            DO_FREE_LOCALS
            return(NULL);
        }

	/* Create a new recycled object structure and set its values to
	 * the ones found here.
	 */
	obj = EDVRecBinObjectNew();
	if(obj == NULL)
	{
            DO_FREE_LOCALS
            return(NULL);
        }
	/* Get size first from stat just in case, a second size value
	 * will be obtained from the recycled index file.
	 */
	obj->size = lstat_buf.st_size;


	/* Open recycled index file and load original values for the
	 * recycled object.
	 */
	if(EDVRecBinObjectGetFromIndexFile(filename, index, obj))
	{
	    /* Error loading object from recycled index file. */
	    DO_FREE_LOCALS
	    return(NULL);
	}

	/* Set pointer to new object structure to obj_rtn and reset obj
	 * to NULL so it does not get deleted.
	 */
	obj_rtn = obj;
	obj = NULL;

	DO_FREE_LOCALS
#undef DO_FREE_LOCALS

	return(obj_rtn);
}


/*
 *	Puts the object specified by path to the recycled objects 
 *	directory under the index specified by index. If a recycled object
 *	who's name matches the given index already exists then it will be
 *	overwritten.
 *
 *	The index file will not be updated.
 *
 *      Returns the following error codes:
 *
 *      0       Success
 *      -1      General error
 *      -2      Ambiguous or permission denied.
 *      -3      Systems error.
 *      -4      Progress callback responded with cancel.
 */
gint EDVRecBinDiskObjectDelete(
        const gchar *index_filename,    /* recycled.ini */
        guint index,                    /* Put into recycled index file under this index. */
        const gchar *path,              /* Object to delete. */
        gint (*progress_cb)(gpointer, gulong, gulong),
        gpointer client_data
)
{
	gint status;
	const gchar *cstrptr;
	gchar *in_path = NULL, *out_path = NULL;
	gchar *recycled_dir = NULL;
	gchar num_str[80];
	struct stat in_lstat_buf;


        /* Reset last error message. */
        last_error = NULL;


	if((index_filename == NULL) || (path == NULL))
	{
	    last_error = "Bad input value";
	    return(-1);
	}

#define DO_FREE_LOCALS  \
{ \
 g_free(in_path); \
 in_path = NULL; \
\
 g_free(out_path); \
 out_path = NULL; \
\
 g_free(recycled_dir); \
 recycled_dir = NULL; \
}


        /* Get parent directory of the recycled index file path, this will
         * be the recycled objects directory.
         */
        recycled_dir = EDVRecBinGetDirectoryFromIndexPath(index_filename);
        if(recycled_dir == NULL)
        {
            DO_FREE_LOCALS
            last_error =
"Unable to obtain recycled objects directory from recycled index file path";
            return(-2);
        }

        /* Create recycled objects directory as needed. */
	rmkdir(recycled_dir, S_IRUSR | S_IWUSR | S_IXUSR);


	/* Generate input object full path. */
	in_path = g_strdup(path);
	if(in_path == NULL)
        {
            DO_FREE_LOCALS
            last_error =
"Unable to generate path for object to delete";
            return(-2);
        }

	/* Generate output object full path. */
	sprintf(num_str, "%i", index);
	cstrptr = PrefixPaths(recycled_dir, num_str);
	out_path = (cstrptr != NULL) ? g_strdup(cstrptr) : NULL;
	if(out_path == NULL)
	{
            DO_FREE_LOCALS
            last_error =
"Unable to generate recycled object path";
            return(-2);
	}


	/* Make sure input object path exists. */
	if(lstat(in_path, &in_lstat_buf))
	{
            DO_FREE_LOCALS
            last_error = "Unable to stat object to delete";
            return(-2);
        }


	status = 0;


	/* Copy in_path to out_path by in_path's type. */

	/* Regular file? */
	if(S_ISREG(in_lstat_buf.st_mode))
	{
	    gint c;
	    gulong bc, bp;
	    gulong file_size = in_lstat_buf.st_size;
	    FILE	*in_fp = FOpen(in_path, "rb"),
			*out_fp = FOpen(out_path, "wb");


	    /* Successfully opened files? */
	    if(in_fp == NULL)
	    {
		DO_FREE_LOCALS
                FClose(in_fp);
                FClose(out_fp);
		last_error = "Unable to open file for reading";
		return(-1);
	    }
            if(out_fp == NULL)
            {
                DO_FREE_LOCALS
                FClose(in_fp);
                FClose(out_fp);
                last_error = "Unable to open recycled file for writing";
                return(-1);
            }

	    /* Report initial progress? */
            if((progress_cb != NULL) && !status)
	    {
		if(progress_cb(client_data, 0, file_size))
		    status = -4;
	    }

	    /* Begin copying. */
	    bc = bp = 0;		/* Reset byte count and position. */
	    c = fgetc(in_fp);
	    while((c != EOF) && !status)
	    {
		if(fputc(c, out_fp) == EOF)
		{
		    /* Update write error code. */
/*		    write_error_code = ferror(tar_fp); */
		    break;
		}

		/* Increment byte count and position. */
		bc++;
		bp++;

		/* Get next character. */
		c = fgetc(in_fp);

		/* Time to update progress? */
		if(bc >= 10000)
		{
		    if(progress_cb != NULL)
		    {
			if(progress_cb(client_data, bp, file_size))
			{
			    status = -4;
			    break;
			}
		    }

		    /* Reset byte count. */
		    bc = 0;
		}
	    }

	    /* Close files. */
	    FClose(in_fp);
	    in_fp = NULL;
	    FClose(out_fp);
	    out_fp = NULL;

            /* Report completed progress? */
            if((progress_cb != NULL) && !status)
            {
                if(progress_cb(client_data, file_size, file_size))
                    status = -4;
            }

	    /* Remove original file specified by in_path if user did not
	     * abort and there were no errors.
	     */
	    if(!status)
	    {
		if(unlink(in_path))
		{
		    DO_FREE_LOCALS
		    switch(errno)
		    {
		      case EFAULT:
			last_error = "Segmentation fault";
                        break;
                      case EACCES:
                        last_error =
 "Object's permissions or location does not permit it to be deleted";
                        break;
                      case EPERM:
                        last_error =
 "You do not own that object in the tempory files location";
                        break;
                      case ENAMETOOLONG:
                        last_error = "Object's path name is too long";
                        break;
                      case ENOENT:
                        last_error =
 "A compoent of the object's path does not exist";
                        break;
                      case ENOTDIR:
                        last_error =
 "A compoent of the object's path is not a directory";
                        break;
                      case EISDIR:
                        last_error =
 "Object's path reffers to a directory object";
                        break;
                      case ENOMEM:
                        last_error = "System is out of memory";
                        break;
                      case EROFS:
                        last_error =
 "The object exists on a read-only filesystem";
                        break;
                      case ELOOP:
                        last_error =
 "Too many symbolic links encountered in the object's path";
                        break;
                      case EIO:
                        last_error = "An input/output error occured";
                        break;
		      default:
			last_error = "Unable to delete file";
			break;
		    }
		    return(-1);
		}
	    }
	}
	/* Directory? */
	else if(S_ISDIR(in_lstat_buf.st_mode))
        {
            /* Report initial progress? */
            if((progress_cb != NULL) && !status)
            {
                if(progress_cb(client_data, 0, 1))
                    status = -4;
            }

	    if(!status)
	    {
		FILE *out_fp;

		/* Attempt to remove directory. */
		if(rmdir(in_path))
		{
		    switch(errno)
		    {
		      case ENOTEMPTY:
			DO_FREE_LOCALS
			last_error = "Directory is not empty";
			return(-2);
			break;

		      default:
			DO_FREE_LOCALS
			last_error = "Unable to remove directory";
			return(-1);
			break;
		    }
		}

		/* Write output file, but do not put any bytes in it. */
		out_fp = FOpen(out_path, "wb");
		/* Close output file immediatly. */
		FClose(out_fp);
		out_fp = NULL;
	    }

            /* Report completed progress? */
            if((progress_cb != NULL) && !status)
            {
                if(progress_cb(client_data, 1, 1))
                    status = -4;
            }
	}
	/* Link? */
	else if(S_ISLNK(in_lstat_buf.st_mode))
        {
            /* Report initial progress? */
            if((progress_cb != NULL) && !status)
            {
                if(progress_cb(client_data, 0, 1))
                    status = -4;
            }

	    if(!status)
	    {
		FILE *out_fp;
		gint bytes_read;
                gchar buf[PATH_MAX + NAME_MAX + 1];


		/* Read link destination value. */
		bytes_read = readlink(
		    in_path, buf, PATH_MAX + NAME_MAX
		);
		if(bytes_read < 0)
		{
		    DO_FREE_LOCALS
		    last_error = "Unable to read link";
		    return(-1);
		}
		if(bytes_read < (PATH_MAX + NAME_MAX))
		    buf[bytes_read] = '\0';
		else
		    buf[PATH_MAX + NAME_MAX] = '\0';

		/* Write output file containing the link's destination 
		 * value.
		 */
		out_fp = FOpen(out_path, "wb");
		if(out_fp == NULL)
		{
                    DO_FREE_LOCALS
                    last_error = "Unable open recycled file for writing";
                    return(-1);
		}

		fputs(buf, out_fp);

		FClose(out_fp);
		out_fp = NULL;
	    }

            /* Report completed progress? */
            if((progress_cb != NULL) && !status)
            {
                if(progress_cb(client_data, 1, 1))
                    status = -4;
            }

            /* Remove original file specified by in_path if user did not
             * abort and there were no errors.
             */
            if(!status)
            {
                if(unlink(in_path))
                {
                    DO_FREE_LOCALS
                    last_error = "Unable to delete link";
                    return(-1);
                }
            }
	}
        /* FIFO pipe? */
        else if(S_ISFIFO(in_lstat_buf.st_mode))
        {
            /* Report initial progress? */
            if((progress_cb != NULL) && !status)
            {
                if(progress_cb(client_data, 0, 1))
                    status = -4;
            }

            if(!status)
            {
                FILE *out_fp;


                /* Write an empty output file for the FIFO pipe. */
                out_fp = FOpen(out_path, "wb");
                if(out_fp == NULL)
                {
                    DO_FREE_LOCALS
                    last_error = "Unable open recycled file for writing";
                    return(-1);
                }

                /* Do not put anything in the output file. */

                FClose(out_fp);
                out_fp = NULL;
            }

            /* Report completed progress? */
            if((progress_cb != NULL) && !status)
            {
                if(progress_cb(client_data, 1, 1))
                    status = -4;
            }

            /* Remove original file specified by in_path if user did not
             * abort and there were no errors.
             */
            if(!status)
            {
                if(unlink(in_path))
                {
                    DO_FREE_LOCALS
                    last_error = "Unable to delete FIFO pipe";
                    return(-1);
                }
            }
	}
        /* Block device? */
        else if(S_ISBLK(in_lstat_buf.st_mode))
	{
            /* Report initial progress? */
            if((progress_cb != NULL) && !status)
            {
                if(progress_cb(client_data, 0, 1))
                    status = -4;
            }

            if(!status)
            {
                FILE *out_fp;
                gint major, minor;


                /* Get block device's major and minor numbers. */
		EDVGetDeviceNumbers(in_lstat_buf.st_rdev, &major, &minor);

                /* Write output file containing the major and minor
		 * numbers which should total 8 bytes (4 bytes each).
                 */
                out_fp = FOpen(out_path, "wb");
                if(out_fp == NULL)
                {
                    DO_FREE_LOCALS
                    last_error = "Unable open recycled file for writing";
                    return(-1);
                }

		fwrite(&major, sizeof(gint), 1, out_fp);
                fwrite(&minor, sizeof(gint), 1, out_fp);

                FClose(out_fp);
                out_fp = NULL;
            }

            /* Report completed progress? */
            if((progress_cb != NULL) && !status)
            {
                if(progress_cb(client_data, 1, 1))
                    status = -4;
            }

            /* Remove original file specified by in_path if user did not
             * abort and there were no errors.
             */
            if(!status)
            {
                if(unlink(in_path))
                {
                    DO_FREE_LOCALS
                    last_error = "Unable to delete block device";
                    return(-1);
                }
            }
	}
        /* Character device? */
        else if(S_ISCHR(in_lstat_buf.st_mode))
        {
            /* Report initial progress? */
            if((progress_cb != NULL) && !status)
            {
                if(progress_cb(client_data, 0, 1))
                    status = -4;
            }

            if(!status)
            {
                FILE *out_fp;
                gint major, minor;


                /* Get block device's major and minor numbers. */
                EDVGetDeviceNumbers(in_lstat_buf.st_rdev, &major, &minor);

                /* Write output file containing the major and minor
                 * numbers which should total 8 bytes (4 bytes each).
                 */
                out_fp = FOpen(out_path, "wb");
                if(out_fp == NULL)
                {
                    DO_FREE_LOCALS
                    last_error = "Unable open recycled file for writing";
                    return(-1);
                }

                fwrite(&major, sizeof(gint), 1, out_fp);
                fwrite(&minor, sizeof(gint), 1, out_fp);

                FClose(out_fp);
                out_fp = NULL;
            }

            /* Report completed progress? */
            if((progress_cb != NULL) && !status)
            {
                if(progress_cb(client_data, 1, 1))
                    status = -4;
            }

            /* Remove original file specified by in_path if user did not
             * abort and there were no errors.
             */
            if(!status)
            {
                if(unlink(in_path))
                {
                    DO_FREE_LOCALS
                    last_error = "Unable to delete character device";
                    return(-1);
                }
            }
        }
        /* Socket? */
        else if(S_ISSOCK(in_lstat_buf.st_mode))
        {
            /* Report initial progress? */
            if((progress_cb != NULL) && !status)
            {
                if(progress_cb(client_data, 0, 1))
                    status = -4;
            }

            if(!status)
            {
                FILE *out_fp;


                /* Write an empty output file for the socket. */
                out_fp = FOpen(out_path, "wb");
                if(out_fp == NULL)
                {
                    DO_FREE_LOCALS
                    last_error = "Unable open recycled file for writing";
                    return(-1);
                }

		/* Do not put anything in the output file. */

                FClose(out_fp);
                out_fp = NULL;
            }

            /* Report completed progress? */
            if((progress_cb != NULL) && !status)
            {
                if(progress_cb(client_data, 1, 1))
                    status = -4;
            }

            /* Remove original file specified by in_path if user did not
             * abort and there were no errors.
             */
            if(!status)
            {
                if(unlink(in_path))
                {
                    DO_FREE_LOCALS
                    last_error = "Unable to delete socket";
                    return(-1);
                }
            }
        }
	/* Unsupported type? */
	else
	{
            DO_FREE_LOCALS
            last_error = "Unsupported object type";
            return(-2);
	}

        DO_FREE_LOCALS
#undef DO_FREE_LOCALS	

	return(status);
}

/*
 *	Recovers the object specified by index.
 *
 *	If the given path is NULL then the original location will be used,
 *	otherwise the path will specify a alternate location to recover 
 *	to.
 *
 *	The index file will not be updated.
 *
 *	Returns the following error codes:
 *
 *	0	Success
 *	-1	General error
 *	-2	Ambiguous or permission denied.
 *	-3	Systems error.
 *	-4	Progress callback responded with cancel.
 */
gint EDVRecBinDiskObjectRecover(
        const gchar *index_filename,    /* recycled.ini */
        guint index,                    /* Recycled object to recover. */
        const gchar *path,              /* Alternate recovery dir if not NULL. */
        gint (*progress_cb)(gpointer, gulong, gulong),
        gpointer client_data
)
{
        gint status;
        const gchar *cstrptr;
	gbool restore_permissions, restore_ownership;
	edv_recbin_object_struct *obj = NULL;
        gchar *in_path = NULL, *out_path = NULL;
        gchar *recycled_dir = NULL;
        gchar num_str[80];
	struct stat in_lstat_buf, out_lstat_buf;


	/* Reset last error message. */
	last_error = NULL;


        if(index_filename == NULL)
        {
            last_error = "Bad input value";
            return(-1);
        }

#define DO_FREE_LOCALS  \
{ \
 g_free(in_path); \
 in_path = NULL; \
\
 g_free(out_path); \
 out_path = NULL; \
\
 g_free(recycled_dir); \
 recycled_dir = NULL; \
\
 EDVRecBinObjectDelete(obj); \
 obj = NULL; \
}

	/* Get recycled object stats. */
	obj = EDVRecBinObjectStat(index_filename, index);
	if(obj == NULL)
	{
	    DO_FREE_LOCALS
            last_error = "Unable to stat recycled object by the given index";
            return(-1);
	}

        /* Get parent directory of the recycled index file path, this will
         * be the recycled objects directory.
         */
        recycled_dir = EDVRecBinGetDirectoryFromIndexPath(index_filename);
        if(recycled_dir == NULL)
        {
            DO_FREE_LOCALS
            last_error =
"Unable to obtain recycled objects directory from recycled index file path";
            return(-2);
        }

        /* Generate input object full path. */
        sprintf(num_str, "%i", index);
        cstrptr = PrefixPaths(recycled_dir, num_str);
        in_path = (cstrptr != NULL) ? g_strdup(cstrptr) : NULL;
        if(in_path == NULL)
        {
            DO_FREE_LOCALS
            last_error =
"Unable to generate recycled object path";
            return(-2);
        }

        /* Generate output object full path. */
	if(path == NULL)
	{
	    /* Given path is NULL, implying that we use the original path
	     * to generate the output path.
	     */
	    cstrptr = PrefixPaths(
		obj->original_path, obj->name
	    );
	    if(cstrptr != NULL)
		out_path = g_strdup(cstrptr);
	}
	else
	{
	    out_path = g_strdup(path);
	}
	if(out_path == NULL)
	{
            DO_FREE_LOCALS
            last_error =
"Unable to generate recovered object path";
            return(-2);
        }


        /* Make sure input object path exists and output object path
         * does not.
         */
        if(lstat(in_path, &in_lstat_buf))
        {
            DO_FREE_LOCALS
            last_error = "Unable to stat recycled object path";
            return(-2);
        }
        if(!lstat(out_path, &out_lstat_buf))
        {
            DO_FREE_LOCALS
            last_error = "Recovered object already exists";
            return(-2);
        }


	status = 0;
	restore_permissions = FALSE;
	restore_ownership = FALSE;


        /* Copy in_path to out_path by object type. */

	/* Regular file? */
	if(obj->type == EDV_OBJECT_TYPE_FILE)
	{
            gint c;
            gulong bc, bp;
            gulong file_size = in_lstat_buf.st_size;
            FILE        *in_fp = FOpen(in_path, "rb"),
                        *out_fp = FOpen(out_path, "wb");


            /* Successfully opened files? */
            if(in_fp == NULL)
            {
                DO_FREE_LOCALS
                FClose(in_fp);
                FClose(out_fp);
                last_error = "Unable to open recycled file for reading";
                return(-1);
            }
            if(out_fp == NULL)
            {
                DO_FREE_LOCALS
                FClose(in_fp);
                FClose(out_fp);
                last_error = "Unable to open recovered file for writing";
                return(-1);
            }

            /* Report initial progress? */
            if((progress_cb != NULL) && !status)
            {
                if(progress_cb(client_data, 0, file_size))
                    status = -4;
            }

            /* Begin copying. */
            bc = bp = 0;                /* Reset byte count and position. */
            c = fgetc(in_fp);
            while((c != EOF) && !status)
            {
                if(fputc(c, out_fp) == EOF)
                {
                    /* Update write error code. */
/*                  write_error_code = ferror(tar_fp); */
                    break;
                }

                /* Increment byte count and position. */
                bc++;
                bp++;

                /* Get next character. */
                c = fgetc(in_fp);

                /* Time to update progress? */
                if(bc >= 10000)
                {
                    if(progress_cb != NULL)
                    {
                        if(progress_cb(client_data, bp, file_size))
                        {
                            status = -4;
                            break;
                        }
                    }

                    /* Reset byte count. */
                    bc = 0;
                }
            }

            /* Close files. */
            FClose(in_fp);
            in_fp = NULL;
            FClose(out_fp);
            out_fp = NULL;

            /* Report completed progress? */
            if((progress_cb != NULL) && !status)
            {
                if(progress_cb(client_data, file_size, file_size))
                    status = -4;
            }

            /* Remove original file specified by in_path if user did not
             * abort and there were no errors.
             */
            if(!status)
            {
                if(unlink(in_path))
                {
                    DO_FREE_LOCALS
                    last_error = "Unable to remove recycled object";
                    return(-1);
                }
            }

	    restore_permissions = TRUE;
	    restore_ownership = TRUE;
        }
        /* Directory? */
        else if(obj->type == EDV_OBJECT_TYPE_DIRECTORY)
        {
            /* Report initial progress? */
            if((progress_cb != NULL) && !status)
            {
                if(progress_cb(client_data, 0, 1))
                    status = -4;
            }

	    if(!status)
	    {
		/* Create recovered directory. */
		if(mkdir(out_path, 0))
		{
		    DO_FREE_LOCALS
                    last_error = "Unable to create recovered directory";
                    return(-1);
		}
	    }

            /* Report completed progress? */
            if((progress_cb != NULL) && !status)
            {
                if(progress_cb(client_data, 1, 1))
                    status = -4;
            }

            /* Remove original file specified by in_path if user did not
             * abort and there were no errors.
             */
            if(!status)
            {
                if(unlink(in_path))
                {
                    DO_FREE_LOCALS
                    last_error = "Unable to remove recycled object";
                    return(-1);
                }
            }

            restore_permissions = TRUE;
            restore_ownership = TRUE;
	}
        /* Link? */
        else if(obj->type == EDV_OBJECT_TYPE_LINK)
        {
            /* Report initial progress? */
            if((progress_cb != NULL) && !status)
            {
                if(progress_cb(client_data, 0, 1))
                    status = -4;
            }

            if(!status)
            {
                FILE *in_fp;
                gint bytes_read;
                gchar buf[PATH_MAX + NAME_MAX + 1];


                /* Open input file for reading. */
		in_fp = FOpen(in_path, "rb");
		if(in_fp == NULL)
		{
                    DO_FREE_LOCALS
                    last_error = "Unable to open recycled file for reading";
                    return(-1);
		}

		/* Read input file as link destinationvalue. */
		bytes_read = fread(
		    buf, sizeof(gchar), PATH_MAX + NAME_MAX, in_fp
		);
                if(bytes_read < 0)
                {
                    DO_FREE_LOCALS
		    FClose(in_fp);
                    last_error = "Unable to read recycled file";
                    return(-1);
                }
                if(bytes_read < (PATH_MAX + NAME_MAX))
                    buf[bytes_read] = '\0';
                else
                    buf[PATH_MAX + NAME_MAX] = '\0';

		/* Close input file. */
		FClose(in_fp);
		in_fp = NULL;


		/* Create link containing the destination value read from
		 * the input file.
		 */
		if(symlink(buf, out_path))
		{
                    DO_FREE_LOCALS
                    last_error = "Unable to create recovered link";
                    return(-1);
		}
	    }

            /* Report completed progress? */
            if((progress_cb != NULL) && !status)
            {
                if(progress_cb(client_data, 1, 1))
                    status = -4;
            }

            /* Remove original file specified by in_path if user did not
             * abort and there were no errors.
             */
            if(!status)
            {
                if(unlink(in_path))
                {
                    DO_FREE_LOCALS
                    last_error = "Unable to remove recycled object";
                    return(-1);
                }
            }

            restore_permissions = FALSE;	/* Do not restore permissions for link. */
            restore_ownership = TRUE;
        }
        /* FIFO pipe? */
        else if(obj->type == EDV_OBJECT_TYPE_FIFO)
        {
            /* Report initial progress? */
            if((progress_cb != NULL) && !status)
            {
                if(progress_cb(client_data, 0, 1))
                    status = -4;
            }

            if(!status)
            {
                mode_t m = S_IFIFO | S_IRUSR | S_IWUSR;

                /* Create FIFO pipe. */
                if(mkfifo(out_path, m))
                {
                    DO_FREE_LOCALS
                    last_error = "Unable to create recovered FIFO pipe";
                    return(-1);
                }
            }

            /* Report completed progress? */
            if((progress_cb != NULL) && !status)
            {
                if(progress_cb(client_data, 1, 1))
                    status = -4;
            }

            /* Remove original file specified by in_path if user did not
             * abort and there were no errors.
             */
            if(!status)
            {
                if(unlink(in_path))
                {
                    DO_FREE_LOCALS
                    last_error = "Unable to remove recycled object";
                    return(-1);
                }
            }

            restore_permissions = TRUE;
            restore_ownership = TRUE;
	}
        /* Block device? */
        else if(obj->type == EDV_OBJECT_TYPE_DEVICE_BLOCK)
        {
            /* Report initial progress? */
            if((progress_cb != NULL) && !status)
            {
                if(progress_cb(client_data, 0, 1))
                    status = -4;
            }

            if(!status)
            {
                FILE *in_fp;
                gint major, minor;
                mode_t m = S_IFBLK | S_IRUSR | S_IWUSR;
		dev_t rdev;


                /* Open input file for reading. */
                in_fp = FOpen(in_path, "rb");
                if(in_fp == NULL)
                {
                    DO_FREE_LOCALS
                    last_error = "Unable to open recycled file for reading";
                    return(-1);
		}

		fread(&major, sizeof(gint), 1, in_fp);
                fread(&minor, sizeof(gint), 1, in_fp);

                /* Close input file. */
                FClose(in_fp);
                in_fp = NULL;


		/* Format dev to contain the major and minor numbers. */
		rdev = EDVFormatDeviceNumbers(major, minor);

                /* Block device with the major and minor numbers taken
		 * from the input file.
		 */
                if(mknod(out_path, m, rdev))
                {
                    DO_FREE_LOCALS
                    last_error = "Unable to create recovered block device";
                    return(-1);
                }
            }

            /* Report completed progress? */
            if((progress_cb != NULL) && !status)
            {
                if(progress_cb(client_data, 1, 1))
                    status = -4;
            }

            /* Remove original file specified by in_path if user did not
             * abort and there were no errors.
             */
            if(!status)
            {
                if(unlink(in_path))
                {
                    DO_FREE_LOCALS
                    last_error = "Unable to remove recycled object";
                    return(-1);
                }
            }

            restore_permissions = TRUE;
            restore_ownership = TRUE;
        }
        /* Character device? */
        else if(obj->type == EDV_OBJECT_TYPE_DEVICE_CHARACTER)
        {
            /* Report initial progress? */
            if((progress_cb != NULL) && !status)
            {
                if(progress_cb(client_data, 0, 1))
                    status = -4;
            }

            if(!status)
            {
                FILE *in_fp;
                gint major, minor;
                mode_t m = S_IFCHR | S_IRUSR | S_IWUSR;
                dev_t rdev;


                /* Open input file for reading. */
                in_fp = FOpen(in_path, "rb");
                if(in_fp == NULL)
                {
                    DO_FREE_LOCALS
                    last_error = "Unable to open recycled file for reading";
                    return(-1);
                }

                fread(&major, sizeof(gint), 1, in_fp);
                fread(&minor, sizeof(gint), 1, in_fp);

                /* Close input file. */
                FClose(in_fp);
                in_fp = NULL;


                /* Format dev to contain the major and minor numbers. */
                rdev = EDVFormatDeviceNumbers(major, minor);

                /* Character device with the major and minor numbers taken
                 * from the input file.
                 */
                if(mknod(out_path, m, rdev))
                {
                    DO_FREE_LOCALS
                    last_error = "Unable to create recovered character device";
                    return(-1);
                }
            }

            /* Report completed progress? */
            if((progress_cb != NULL) && !status)
            {
                if(progress_cb(client_data, 1, 1))
                    status = -4;
            }

            /* Remove original file specified by in_path if user did not
             * abort and there were no errors.
             */
            if(!status)
            {
                if(unlink(in_path))
                {
                    DO_FREE_LOCALS
                    last_error = "Unable to remove recycled object";
                    return(-1);
                }
            }

            restore_permissions = TRUE;
            restore_ownership = TRUE;
        }
        /* Socket? */
        else if(obj->type == EDV_OBJECT_TYPE_SOCKET)
	{
	    /* Don't know how to recreate socket objects, so just create
	     * an empty file.
	     */
	    FILE *out_fp = FOpen(out_path, "wb");


            /* Report initial progress? */
            if((progress_cb != NULL) && !status)
            {
                if(progress_cb(client_data, 0, 1))
                    status = -4;
            }

	    /* Unable to open output file for writing? */
	    if(out_fp == NULL)
	    {
                DO_FREE_LOCALS
                last_error = "Unable to open recovered file for writing";
                return(-1);
            }

	    /* Do not write anything. */

	    FClose(out_fp);
	    out_fp = NULL;


            /* Report completed progress? */
            if((progress_cb != NULL) && !status)
            {
                if(progress_cb(client_data, 1, 1))
                    status = -4;
            }

            /* Remove original file specified by in_path if user did not
             * abort and there were no errors.
             */
            if(!status)
            {
                if(unlink(in_path))
                {
                    DO_FREE_LOCALS
                    last_error = "Unable to remove recycled object";
                    return(-1);
                }
            }

            restore_permissions = TRUE;
            restore_ownership = TRUE;
	}
        /* Unsupported type? */
        else
        {
            DO_FREE_LOCALS
            last_error = "Unsupported object type";
            return(-2);
        }


	/* Restore ownership? */
	if(restore_ownership)
	{
	    lchown(out_path, obj->owner_id, obj->group_id);
	}

	/* Restore permissions. */
	if(restore_permissions)
	{
	    mode_t m = 0;

	    if(obj->permissions & EDV_PERMISSION_UEXECUTE)
		m |= S_IXUSR;
            if(obj->permissions & EDV_PERMISSION_UREAD)
                m |= S_IRUSR;
            if(obj->permissions & EDV_PERMISSION_UWRITE)
                m |= S_IWUSR;
            if(obj->permissions & EDV_PERMISSION_GEXECUTE)
                m |= S_IXGRP;
            if(obj->permissions & EDV_PERMISSION_GREAD)
                m |= S_IRGRP;
            if(obj->permissions & EDV_PERMISSION_GWRITE)
                m |= S_IWGRP;
            if(obj->permissions & EDV_PERMISSION_AEXECUTE)
                m |= S_IXOTH;
            if(obj->permissions & EDV_PERMISSION_AREAD)
                m |= S_IROTH;
            if(obj->permissions & EDV_PERMISSION_AWRITE)
                m |= S_IWOTH;
            if(obj->permissions & EDV_PERMISSION_SETUID)
                m |= S_ISUID;
            if(obj->permissions & EDV_PERMISSION_SETGID)
                m |= S_ISGID;
            if(obj->permissions & EDV_PERMISSION_STICKY)
                m |= S_ISVTX;

	    chmod(out_path, m);
	}


        DO_FREE_LOCALS
#undef DO_FREE_LOCALS

        return(status);
}

/*
 *      Purges the recycled object specified by index.
 *
 *      The index file will not be updated.
 *
 *      Returns the following error codes:
 *
 *      0       Success
 *      -1      General error
 *      -2      Ambiguous or permission denied.
 *      -3      Systems error.
 *      -4      Progress callback responded with cancel.
 */
gint EDVRecBinDiskObjectPurge(
        const gchar *index_filename,    /* recycled.ini */
        guint index,                    /* Recycled object to purge. */
        gint (*progress_cb)(gpointer, gulong, gulong),
        gpointer client_data
)
{
        gint status;
        const gchar *cstrptr;
        gchar *in_path = NULL;
        gchar *recycled_dir = NULL;
        gchar num_str[80];


        /* Reset last error message. */
        last_error = NULL;


        if(index_filename == NULL)
        {
            last_error = "Bad input value";
            return(-1);
        }

#define DO_FREE_LOCALS  \
{ \
 g_free(in_path); \
 in_path = NULL; \
\
 g_free(recycled_dir); \
 recycled_dir = NULL; \
}

        /* Get parent directory of the recycled index file path, this will
         * be the recycled objects directory.
         */
        recycled_dir = EDVRecBinGetDirectoryFromIndexPath(index_filename);
        if(recycled_dir == NULL)
        {
            DO_FREE_LOCALS
            last_error =
"Unable to obtain recycled objects directory from recycled index file path";
            return(-2);
        }

        /* Create recycled objects directory as needed. */
        rmkdir(recycled_dir, S_IRUSR | S_IWUSR | S_IXUSR);


        /* Generate input object full path. */
        sprintf(num_str, "%i", index);
        cstrptr = PrefixPaths(recycled_dir, num_str);
        in_path = (cstrptr != NULL) ? g_strdup(cstrptr) : NULL;
        if(in_path == NULL)
        {
            DO_FREE_LOCALS
            last_error =
"Unable to generate recycled object path";
            return(-2);
        }


        status = 0;


	/* Remove in_path, the recycled object. */
	if(1)
	{
            /* Report initial progress? */
            if((progress_cb != NULL) && !status)
            {
                if(progress_cb(client_data, 0, 1))
                    status = -4;
            }

            /* Remove original file specified by in_path if user did not
             * abort and there were no errors.
             */
            if(!status)
            {
		/* Remove recycled object, ignore any errors. */
                unlink(in_path);
            }

            /* Report completed progress? */
            if((progress_cb != NULL) && !status)
            {
                if(progress_cb(client_data, 1, 1))
                    status = -4;
            }
	}

        DO_FREE_LOCALS
#undef DO_FREE_LOCALS

        return(status);
}

/*
 *	Similar to EDVRecBinDiskObjectPurge() except that it purges
 *	all contents in recycled objects directory, including the
 *	recycled objects index file.
 *
 *	Calling this function is more efficient than calling
 *	EDVRecBinDiskObjectPurge() for each recycled object.
 *
 *      Returns the following error codes:
 *
 *      0       Success
 *      -1      General error
 *      -2      Ambiguous or permission denied.
 *      -3      Systems error.
 *      -4      Progress callback responded with cancel.
 */
gint EDVRecBinDiskObjectPurgeAll(
        const gchar *index_filename,    /* recycled.ini */
        gint (*progress_cb)(gpointer, gulong, gulong),
        gpointer client_data
)
{
        gint status, strc;
	gchar **strv;
        gchar *recycled_dir;


        /* Reset last error message. */
        last_error = NULL;


        if(index_filename == NULL)
        {
            last_error = "Bad input value";
            return(-1);
        }

#define DO_FREE_LOCALS  \
{ \
 g_free(recycled_dir); \
 recycled_dir = NULL; \
}

        /* Get parent directory of the recycled index file path, this will
         * be the recycled objects directory.
         */
        recycled_dir = EDVRecBinGetDirectoryFromIndexPath(index_filename);
        if(recycled_dir == NULL)
        {
            DO_FREE_LOCALS
            last_error =
"Unable to obtain recycled objects directory from recycled index file path";
            return(-2);
        }

        /* Create recycled objects directory as needed. */
        rmkdir(recycled_dir, S_IRUSR | S_IWUSR | S_IXUSR);


        status = 0;

	/* Get all contents of the recycled objects directory, which
	 * should include the recycled objects index file.
	 */
	strv = GetDirEntNames2(recycled_dir, &strc);
	if(strv != NULL)
	{
	    gint i;
	    const gchar *cstrptr, *cstrptr2;


	    /* Iterate through each entry in the recycled objects
	     * directory and remove it.
	     */
	    for(i = 0; i < strc; i++)
	    {
#define DO_FREE_CONTINUE	\
{ \
 g_free(strv[i]); \
 strv[i] = NULL; \
\
 continue; \
}
		cstrptr = strv[i];
		if(cstrptr == NULL)
		    DO_FREE_CONTINUE

		/* If there was an error or user aborted then just
		 * deallocate this directory entry name and all subsequent
		 * ones.
		 */
		if(status)
		    DO_FREE_CONTINUE

		/* Skip special directory notations. */
		if(!strcmp(cstrptr, "..") ||
                   !strcmp(cstrptr, ".")
		)
		    DO_FREE_CONTINUE

		/* Generate full path to current directory entry. */
		cstrptr2 = PrefixPaths(recycled_dir, cstrptr);
		if(cstrptr2 == NULL)
		    DO_FREE_CONTINUE

		/* Report progress? */
		if(progress_cb != NULL)
		{
		    if(progress_cb(client_data, i, strc))
		    {
			status = -4;
			DO_FREE_CONTINUE
		    }
		}

		/* Remove object specified by the full path of the
		 * directory entry name.
		 */
                if(unlink(cstrptr2))
                {
                    last_error = "Unable to remove recycled object";
		    status = -1;
		    DO_FREE_CONTINUE
                }

		DO_FREE_CONTINUE
#undef DO_FREE_CONTINUE
	    }

	    /* Deallocate directory entry names pointer array, each name
	     * in the array has already been deallocated.
	     */
	    g_free(strv);
	    strv = NULL;

	    /* Report final progress? */
	    if((progress_cb != NULL) && !status)
		progress_cb(client_data, strc, strc);
	}

        DO_FREE_LOCALS
#undef DO_FREE_LOCALS

        return(status);
}
